//+--------------------------------------------------------------+
//| Simple Static Template MT4.mq4                               |
//| Copyright 2026, Centropolis Corp.                            |
//| https://www.mql5.com/en/users/w.hudson                       |
//+--------------------------------------------------------------+
#property copyright "Copyright 2026, Centropolis Corp."
#property link      "https://www.mql5.com/en/users/w.hudson"
#property version   "1.00"
#property strict

//+--------------------------------------------------------------+
//| Main robot constants                                         |
//+--------------------------------------------------------------+
#define EANICK "Simple Template"        // Robot name displayed in the graphical user interface
#define GLOBALVARPREFIX "HelpVar"       // Prefix for global variables, prevents conflicts when running multiple robot instances
#define EABARS 990                      // Maximum number of historical bars for analysis in the internal virtual chart
#define MINUTES_TRY 2                   // Maximum time in minutes for attempting to open/close positions from the start of a new bar

//+--------------------------------------------------------------+
//| Robot input parameters                                       |
//+--------------------------------------------------------------+
//+----------------- "Volumes" group -----------------------+
input group "Volumes";
sinput bool bInitLotControl=false;      // Automatic lot calculation
sinput double LotE=0.01;                // Fixed lot size
sinput double DepositE=1000.0;          // Deposit for lot calculation

//+----------------- "Waiting Out Losses Mode" group -----------------------+
input group "Waiting Out Losses Mode";
input bool bInitSitE=false;             // Loss linearization mode (sit out losses)
input int MinutesHoldE=144000;          // Position holding time in minutes

//+----------------- "Martingale" group -----------------------+
input group "Martingale";
input bool bInitMartinE=false;          // Martingale system
input int MaxMartinLotMultiplierStepsE=5; // Maximum Martingale steps

//+----------------- "Repurchase" group -----------------------+
input group "Repurchase";
input bool bInitRepurchaseE=true;       // Repurchase system
input double MinPercenPriceStepE=0.25;  // First price step for repurchase
input double NextStepMultiplierE=1.5;   // Repurchase step multiplier

//+----------------- "Other" group -----------------------+
input group "Other";
sinput int MagicE = 15464;              // Magic number
input int SLE=0;                        // Stop Loss in points
input int TPE=0;                        // Take Profit in points
input double ComissionPerLotE=0.0;      // Commission per lot
input const int SpreadEE=5000;          // Maximum spread in points
sinput string CommentI="Simple Template"; // Comment for orders

//+----------------- "Fighting The Obsolescence Of Settings" group -----------------------+
input group "Fighting The Obsolescence Of Settings";
sinput uint DaysToFuture=365;           // Number of days into the future

//+----------------- "Direction Variables" group -----------------------+
input group "Direction Variables";
input bool TBE=true;                    // Allow opening BUY positions
input bool TSE=true;                    // Allow opening SELL positions

//+----------------- "Strategy Variables" group -----------------------+
input group "Strategy Variables";
input int BarsChainForOpenE = 5;        // Concurrent bars for opening a position
input int BarsChainForCloseE = 5;       // Concurrent bars for closing a position

//+--------------------------------------------------------------+
//| Global variables                                             |
//+--------------------------------------------------------------+
int HistoryDaysLoadI=365;               // Number of history days to load
int SlippageMaxOpen=100;                // Maximum slippage when opening positions in points
int SlippageMaxClose=100;               // Maximum slippage when closing positions in points

//+--------------------------------------------------------------+
//| Check if magic number belongs to our robot                   |
//| Parameters:                                                  |
//| magic1 - magic number to check                               |
//| Returns: true if magic belongs to our robot                  |
//+--------------------------------------------------------------+
bool bOurMagic(long magic1)
   {
   if ( (int)magic1 == BotO.MagicF ) return true;
   return false;                            // return false if magic does not belong to our robot
   }  
   
//+--------------------------------------------------------------+
//| Check possibility of additional position opening             |
//| Used for the Repurchase system                               |
//| Parameters:                                                  |
//| CurrentPrice - current price                                 |
//| bDirection - direction (true=BUY, false=SELL)                |
//| Returns: true if additional position can be opened           |
//+--------------------------------------------------------------+
bool AddtionalOpenPrice(double CurrentPrice,bool bDirection)
   {
   bool ord;                            // flag for successful order selection for analysis
   // first determine if there are open positions for the same instrument, if so determine the extreme open price
   double BorderPriceBUY = -1.0;       // exact (lowest) open price of BUY positions, -1.0 means no positions
   double BorderPriceSELL = -1.0;      // exact (highest) open price of SELL positions, -1.0 means no positions
   double PriceSymbolASK = SymbolInfoDouble(_Symbol,SYMBOL_ASK);  // current ASK price for buying
   double PriceSymbolBID = SymbolInfoDouble(_Symbol,SYMBOL_BID);  // current BID price for selling
   
   int PositionsBuy=0;                  // counter of open BUY positions for current symbol
   int PositionsSell=0;                 // counter of open SELL positions for current symbol
   
   // determine the extreme price to count from
   for ( int i=0; i<OrdersTotal(); i++ )
      {
      ord=OrderSelect(i, SELECT_BY_POS, MODE_TRADES);  
      long cMagic = OrderMagicNumber();          
      if ( ord && cMagic == MagicE )
         {
         string SymbolNotFiltered = OrderSymbol();  
         if (_Symbol == SymbolNotFiltered)
            {
            if (OrderType() == OP_BUY)
               {
               double PriceOpenPosition = OrderOpenPrice();
               if (BorderPriceBUY == -1.0 || PriceOpenPosition < BorderPriceBUY)
                  {
                  BorderPriceBUY = PriceOpenPosition;
                  PositionsBuy++;
                  }
               }                  
            else if (OrderType() == OP_SELL)
               {
               double PriceOpenPosition = OrderOpenPrice();
               if (BorderPriceSELL == -1.0 || PriceOpenPosition > BorderPriceSELL)
                  {
                  BorderPriceSELL = PriceOpenPosition;
                  PositionsSell++;
                  }                  
               }               
            }
         }
      }
   
   // after determining the extreme price, determine if a position can be opened right now
   if ( bDirection && (BorderPriceBUY == -1.0 || (BorderPriceBUY > PriceSymbolASK && PositionsBuy > 0 && PositionsSell == 0 && BorderPriceBUY-PriceSymbolASK >= MathPow(NextStepMultiplierE,double(PositionsBuy-1))*BorderPriceBUY*MinPercenPriceStepE/100.0 )) )
      {
      return true;                          // return true if condition for additional BUY opening is met
      }
   else if ( !bDirection && (BorderPriceSELL == -1.0 || (BorderPriceSELL < PriceSymbolBID && PositionsSell > 0 && PositionsBuy == 0 && PriceSymbolBID-BorderPriceSELL >= MathPow(NextStepMultiplierE,double(PositionsSell-1))*BorderPriceSELL*MinPercenPriceStepE/100.0 )) )
      {
      return true;                          // return true if condition for additional SELL opening is met
      }
   
   return false;                        // return false if conditions for additional opening are not met
   }

//+--------------------------------------------------------------+
//| Close positions to profit for repurchase system              |
//| Parameters:                                                  |
//| bDirection - direction (true=BUY, false=SELL)                |
//+--------------------------------------------------------------+
void CloseToProfit(bool bDirection)
   {
   bool ord;                            // flag for successful order selection for analysis
   bool order;                          // result of order close operation
   int Tickets[];                       // dynamic array of order tickets to close in profit
   int TicketsTotal=0;                  // total number of orders suitable for closing
   int TicketNumCurrent=0;              // current index when filling tickets array
   int ticket;                          // temporary variable to store current order ticket
   
   double SummProfit=0.0;               // total profit of all orders in specified direction including swaps and commissions
   
   // calculating total profit by orders
   for ( int i=0; i<OrdersTotal(); i++ )
      {
      ord=OrderSelect(i, SELECT_BY_POS, MODE_TRADES);
      if (ord && OrderSymbol() == _Symbol && OrderMagicNumber() == MagicE )
         {
         if (bDirection && OrderType() == OP_BUY)
            {
            TicketsTotal=TicketsTotal+1;
            SummProfit+=OrderProfit()+OrderSwap() - OrderLots()*ComissionPerLotE;   
            }
         if (!bDirection && OrderType() == OP_SELL)
            {
            TicketsTotal=TicketsTotal+1;
            SummProfit+=OrderProfit()+OrderSwap() - OrderLots()*ComissionPerLotE;
            }         
         }
      }
      
   if (SummProfit > 0.0)
      {
      ArrayResize(Tickets,TicketsTotal);
      SummProfit=0.0;
      for ( int i=0; i<OrdersTotal(); i++ )
         {
         ord=OrderSelect(i, SELECT_BY_POS, MODE_TRADES);
         ticket=OrderTicket();                  
         if (ord && OrderSymbol() == _Symbol && OrderMagicNumber() == MagicE && TicketNumCurrent < TicketsTotal )
            {
            if ( (bDirection && OrderType() == OP_BUY) || (!bDirection && OrderType() == OP_SELL) )
               {
               Tickets[TicketNumCurrent]=ticket;
               TicketNumCurrent=TicketNumCurrent+1;
               }
            }
         }      
      }
   if (ArraySize(Tickets) > 0)
      {
      for ( int i=0; i<ArraySize(Tickets); i++ )
         {
         ord=OrderSelect( Tickets[i], SELECT_BY_TICKET, MODE_TRADES );
         if (ord && Tickets[i] > 0 && OrderType() == OP_BUY )
            {
            order=OrderClose(Tickets[i],OrderLots(),Bid,MathAbs(SlippageMaxClose),Red);
            }
         else if (ord && Tickets[i] > 0 && OrderType() == OP_SELL )
            {
            order=OrderClose(Tickets[i],OrderLots(),Ask,MathAbs(SlippageMaxClose),Green);
            }       
         }      
      }   
   }

//+--------------------------------------------------------------+
//| Max balance control                                          |
//+--------------------------------------------------------------+
double LastMaxBalance;              // maximum found balance for the last year to calculate drawdown
datetime LastTimeChecked;           // time of last max balance check for optimization
datetime LastTimeBorder;            // time of boundary deal, up to which history analysis is already performed

//+--------------------------------------------------------------+
//| Search for new balance highs to control drawdown             |
//| Analyzes deal history and determines maximum balance         |
//| for calculating permissible drawdown                         |
//+--------------------------------------------------------------+
void UpdateMaxBalance()
   {
   bool ord;                         // flag for successful order selection from history
   double TempSummDelta=0.0;         // accumulated profit delta to restore historical balance at the moment of each deal
   datetime LastTimeBorderTemp=LastTimeBorder; // temporary variable to store time of the newest processed deal in current analysis cycle
   datetime BorderTime=TimeCurrent()-HistoryDaysLoadI*86400; // boundary time for deal history analysis
   
   // Iterate through deal history from new to old
   for ( int i=OrdersHistoryTotal()-1; i>=0; i-- ) 
      {
      ord = OrderSelect(i, SELECT_BY_POS,MODE_HISTORY);
      
      // Consider only closing deals of our robots in specified time range
      if ( ord && OrderCloseTime() > BorderTime && bOurMagic(OrderMagicNumber()) ) 
         {
         datetime DealTime = OrderCloseTime();
         
         // If we haven't considered this deal yet
         if (DealTime >= LastTimeChecked && DealTime >= LastTimeBorder )
            {
            // Subtract deal profit to restore balance at the moment before the deal
            TempSummDelta-=OrderProfit();
            double AB = AccountInfoDouble(ACCOUNT_BALANCE);            
            double ConstructedBalance = TempSummDelta+AccountInfoDouble(ACCOUNT_BALANCE);
            
            // If greater balance extremum found
            if (ConstructedBalance > LastMaxBalance)
               {
               LastMaxBalance=ConstructedBalance;
               }
               
            // Protection against abnormal values - allow balance excess max by 20%
            if (AB > 0.0 && ConstructedBalance > AB && (ConstructedBalance - AB) / AB > 0.2 )
               {
               LastMaxBalance = AB * 1.2;  // Limit max excess to 20%
               break;
               }               
            if (DealTime > LastTimeBorderTemp) LastTimeBorderTemp=DealTime;
            }
         else
            {
            break;  // Stop analysis, reached already processed deals
            } 
         }
      else if (OrderCloseTime() < BorderTime) break;  // Stop analysis, went out of time frame
      }
      
   // If max balance not defined, use current balance as starting one
   if (LastMaxBalance == 0) LastMaxBalance=AccountInfoDouble(ACCOUNT_BALANCE);
   
   // Update boundary time of processed deals
   LastTimeBorder=LastTimeBorderTemp;
   }
//

//+--------------------------------------------------------------+
//| Creation of the single chart object                          |
//| Initializes Chart object with arrays for storing             |
//| historical data (OHLC) of current symbol                     |
//+--------------------------------------------------------------+
void CreateChart()
   {
   Chart::TCN=EABARS;              // set number of bars for analysis
   ChartO = new Chart();           // create new chart object

   ChartO.lastcopied=0;            // reset copied bars counter
   
   // Set size for historical data storage arrays
   ArrayResize(ChartO.CloseI,Chart::TCN+2);    // close prices array
   ArrayResize(ChartO.OpenI,Chart::TCN+2);     // open prices array
   ArrayResize(ChartO.HighI,Chart::TCN+2);     // high prices array
   ArrayResize(ChartO.LowI,Chart::TCN+2);      // low prices array
   ArrayResize(ChartO.TimeI,Chart::TCN+2);     // bar times array

   ChartO.CurrentSymbol = _Symbol;             // set current symbol
   ChartO.Timeframe = PERIOD_CURRENT;          // set current period
   ChartO.GetTimeSeries();                     // load time series
   }

//+--------------------------------------------------------------+
//| Creation of trading robot instance                           |
//| Initializes BotInstance object for trading management        |
//+--------------------------------------------------------------+
void CreateInstance()
   {
   BotO = new BotInstance();       // create new trading robot instance
   }

//+--------------------------------------------------------------+
//| Main robot tick processing function                          |
//| Controls trading logic execution on each tick                |
//| Processes new bars and performs trading operations           |
//+--------------------------------------------------------------+
void BotTick()
   {
   // Check data readiness and trading session
   if ( ChartO.lastcopied >= Chart::TCN+1 && ChartO.ChartPoint > 0.0 && BotO.bInTradeSession() )
      {
      BotO.bNewBarV=BotO.bNewBar();  // check for new bar appearance
      } 
   else BotO.bNewBarV=false;         // if data not ready, reset new bar flag

   datetime tc=TimeCurrent();        // get current time
   
   // Execute trading logic on new bar appearance or by timer
   if ( (BotO.NewBarType == "START" && BotO.bNewBarV) || (BotO.NewBarType == "LATE" && tc - BotO.PreviousLateTick >= 20))
      {
      BotO.PreviousLateTick=tc;      // update last tick time
      
      // Recalculate trading signals only on new bar
      if (BotO.PreviousTimeCalc != ChartO.TimeI[1])
         {
         BotO.CalculateForTrade();   // calculate trading signals
         BotO.PreviousTimeCalc=ChartO.TimeI[1]; // remember calculation time
         } 
         
      BotO.bOpened=BotO.Opened();    // check for open positions     
                
      if (!BotO.bOpened) BotO.InstanceTick(true);  // first processing for closing positions
      BotO.InstanceTick(false);      // second processing for opening positions             
      }
   } 
   
//+-----------------------------------------------------------------+
//| Chart class - graphical data management                         |
//| Responsible for loading, storing and updating historical price  |
//| data (OHLC) and bar times for trading signal analysis           |
//+-----------------------------------------------------------------+
class Chart
   {
   public:
   datetime TimeI[];                    // bar times array (indexed as time series)
   double CloseI[];                     // bar close prices array
   double OpenI[];                      // bar open prices array
   double HighI[];                      // bar high prices array
   double LowI[];                       // bar low prices array
   string BasicName;                    // basic name that was in substring
   string BasicSymbol;                  // basic instrument extracted from substring
   string BasicPeriod;                  // basic chart period
   double ChartPoint;                   // point size of current chart for calculations
   double ChartAsk;                     // current chart Ask price
   double ChartBid;                     // current chart Bid price
  
   datetime tTimeI[];                   // auxiliary array for controlling new bar appearance
   
   static int TCN;                      // total number of bars for analysis (static class variable)
   
   string CurrentSymbol;                // corrected symbol for trading
   ENUM_TIMEFRAMES Timeframe;           // chart period (timeframe)
   int copied;                          // number of copied data during last update
   int lastcopied;                      // last number of successfully copied data
   datetime LastCloseTime;              // time of last processed bar
   MqlTick LastTick;                    // structure with last tick data
   
   // Variables for storing trading sessions in minutes from start of day
   int MONDAY_MinuteEquivalentFrom;     // Monday trading session start time in minutes
   int TUESDAY_MinuteEquivalentFrom;    // Tuesday trading session start time in minutes
   int WEDNESDAY_MinuteEquivalentFrom;  // Wednesday trading session start time in minutes
   int THURSDAY_MinuteEquivalentFrom;   // Thursday trading session start time in minutes
   int FRIDAY_MinuteEquivalentFrom;     // Friday trading session start time in minutes
   int SATURDAY_MinuteEquivalentFrom;   // Saturday trading session start time in minutes
   int SUNDAY_MinuteEquivalentFrom;     // Sunday trading session start time in minutes
   
   int MONDAY_MinuteEquivalentTo;       // Monday trading session end time in minutes
   int TUESDAY_MinuteEquivalentTo;      // Tuesday trading session end time in minutes
   int WEDNESDAY_MinuteEquivalentTo;    // Wednesday trading session end time in minutes
   int THURSDAY_MinuteEquivalentTo;     // Thursday trading session end time in minutes
   int FRIDAY_MinuteEquivalentTo;       // Friday trading session end time in minutes
   int SATURDAY_MinuteEquivalentTo;     // Saturday trading session end time in minutes
   int SUNDAY_MinuteEquivalentTo;       // Sunday trading session end time in minutes
   
   //+------------------------------------------------------------------+
   //| Time series correction for tester                                |
   //| Fixes issues with time series in MQL4 tester                     |
   //| Determines trading sessions for different days of week           |
   //+------------------------------------------------------------------+
   void GetTimeSeries()
      {
      // Declare variables for recording session start and end 
      datetime date_from_reserve=0;        // reserve session start time 
      datetime date_to_reserve=0;          // reserve session end time    
      
      // Variables for recording trading session start and end in datetime format
      datetime date_from_MONDAY;           // Monday trading session start time
      datetime date_to_MONDAY;             // Monday trading session end time
      datetime date_from_TUESDAY;          // Tuesday trading session start time
      datetime date_to_TUESDAY;            // Tuesday trading session end time
      datetime date_from_WEDNESDAY;        // Wednesday trading session start time
      datetime date_to_WEDNESDAY;          // Wednesday trading session end time
      datetime date_from_THURSDAY;         // Thursday trading session start time
      datetime date_to_THURSDAY;           // Thursday trading session end time
      datetime date_from_FRIDAY;           // Friday trading session start time
      datetime date_to_FRIDAY;             // Friday trading session end time
      datetime date_from_SATURDAY;         // Saturday trading session start time
      datetime date_to_SATURDAY;           // Saturday trading session end time
      datetime date_from_SUNDAY;           // Sunday trading session start time
      datetime date_to_SUNDAY;             // Sunday trading session end time
      
      // MqlDateTime structures for working with trading session time components
      MqlDateTime date_from_MONDAY0;       // Monday trading session start time structure
      MqlDateTime date_to_MONDAY0;         // Monday trading session end time structure
      MqlDateTime date_from_TUESDAY0;      // Tuesday trading session start time structure
      MqlDateTime date_to_TUESDAY0;        // Tuesday trading session end time structure
      MqlDateTime date_from_WEDNESDAY0;    // Wednesday trading session start time structure
      MqlDateTime date_to_WEDNESDAY0;      // Wednesday trading session end time structure
      MqlDateTime date_from_THURSDAY0;     // Thursday trading session start time structure
      MqlDateTime date_to_THURSDAY0;       // Thursday trading session end time structure
      MqlDateTime date_from_FRIDAY0;       // Friday trading session start time structure
      MqlDateTime date_to_FRIDAY0;         // Friday trading session end time structure
      MqlDateTime date_from_SATURDAY0;     // Saturday trading session start time structure
      MqlDateTime date_to_SATURDAY0;       // Saturday trading session end time structure
      MqlDateTime date_from_SUNDAY0;       // Sunday trading session start time structure
      MqlDateTime date_to_SUNDAY0;         // Sunday trading session end time structure
      
      // Get trading session information for each day of week
      bool bMONDAY = SymbolInfoSessionTrade(CurrentSymbol,MONDAY,0,date_from_MONDAY,date_to_MONDAY);
      bool bTUESDAY = SymbolInfoSessionTrade(CurrentSymbol,TUESDAY,0,date_from_TUESDAY,date_to_TUESDAY);
      bool bWEDNESDAY = SymbolInfoSessionTrade(CurrentSymbol,WEDNESDAY,0,date_from_WEDNESDAY,date_to_WEDNESDAY);
      bool bTHURSDAY = SymbolInfoSessionTrade(CurrentSymbol,THURSDAY,0,date_from_THURSDAY,date_to_THURSDAY);
      bool bFRIDAY = SymbolInfoSessionTrade(CurrentSymbol,FRIDAY,0,date_from_FRIDAY,date_to_FRIDAY);
      bool bSATURDAY = SymbolInfoSessionTrade(CurrentSymbol,SATURDAY,0,date_from_SATURDAY,date_to_SATURDAY);
      bool bSUNDAY = SymbolInfoSessionTrade(CurrentSymbol,SUNDAY,0,date_from_SUNDAY,date_to_SUNDAY);
         
      // Convert datetime to MqlDateTime structures for convenient work with time components
      TimeToStruct(date_from_MONDAY,date_from_MONDAY0); 
      TimeToStruct(date_from_TUESDAY,date_from_TUESDAY0);
      TimeToStruct(date_from_WEDNESDAY,date_from_WEDNESDAY0);
      TimeToStruct(date_from_THURSDAY,date_from_THURSDAY0);
      TimeToStruct(date_from_FRIDAY,date_from_FRIDAY0);
      TimeToStruct(date_from_SATURDAY,date_from_SATURDAY0);
      TimeToStruct(date_from_SUNDAY,date_from_SUNDAY0);
      TimeToStruct(date_to_MONDAY,date_to_MONDAY0);
      TimeToStruct(date_to_TUESDAY,date_to_TUESDAY0);
      TimeToStruct(date_to_WEDNESDAY,date_to_WEDNESDAY0);
      TimeToStruct(date_to_THURSDAY,date_to_THURSDAY0);
      TimeToStruct(date_to_FRIDAY,date_to_FRIDAY0);
      TimeToStruct(date_to_SATURDAY,date_to_SATURDAY0);
      TimeToStruct(date_to_SUNDAY,date_to_SUNDAY0);   
       
      // Calculate minute equivalents of trading session start times
      MONDAY_MinuteEquivalentFrom = date_from_MONDAY0.hour*60 + date_from_MONDAY0.min;
      TUESDAY_MinuteEquivalentFrom = date_from_TUESDAY0.hour*60 + date_from_TUESDAY0.min;
      WEDNESDAY_MinuteEquivalentFrom = date_from_WEDNESDAY0.hour*60 + date_from_WEDNESDAY0.min;
      THURSDAY_MinuteEquivalentFrom = date_from_THURSDAY0.hour*60 + date_from_THURSDAY0.min;
      FRIDAY_MinuteEquivalentFrom = date_from_FRIDAY0.hour*60 + date_from_FRIDAY0.min;
      SATURDAY_MinuteEquivalentFrom = date_from_SATURDAY0.hour*60 + date_from_SATURDAY0.min;
      SUNDAY_MinuteEquivalentFrom = date_from_SUNDAY0.hour*60 + date_from_SUNDAY0.min;
      
      // Calculate minute equivalents of trading session end times
      MONDAY_MinuteEquivalentTo = date_to_MONDAY0.hour*60 + date_to_MONDAY0.min;
      TUESDAY_MinuteEquivalentTo = date_to_TUESDAY0.hour*60 + date_to_TUESDAY0.min;
      WEDNESDAY_MinuteEquivalentTo = date_to_WEDNESDAY0.hour*60 + date_to_WEDNESDAY0.min;
      THURSDAY_MinuteEquivalentTo = date_to_THURSDAY0.hour*60 + date_to_THURSDAY0.min;
      FRIDAY_MinuteEquivalentTo = date_to_FRIDAY0.hour*60 + date_to_FRIDAY0.min;
      SATURDAY_MinuteEquivalentTo = date_to_SATURDAY0.hour*60 + date_to_SATURDAY0.min;
      SUNDAY_MinuteEquivalentTo = date_to_SUNDAY0.hour*60 + date_to_SUNDAY0.min;
      
      // Correction for cases 0:00 - 24:00 and when start time equals end time
      
      // Monday processing
      if (bMONDAY)
         {
         // Case of round-the-clock trading (0:00 - 0:00 next day)
         if (MONDAY_MinuteEquivalentFrom == 0 && MONDAY_MinuteEquivalentTo == 0 && date_from_MONDAY != date_to_MONDAY)
            {
            MONDAY_MinuteEquivalentTo=60*23+59;  // Set end to 23:59
            }
         // Case when start time equals end time (but different days)
         else if (MONDAY_MinuteEquivalentFrom == MONDAY_MinuteEquivalentTo && date_from_MONDAY != date_to_MONDAY)
            {
            if (MONDAY_MinuteEquivalentFrom < 60*23+59)
               {
               MONDAY_MinuteEquivalentTo = MONDAY_MinuteEquivalentFrom+1;  // Add 1 minute
               }
            else MONDAY_MinuteEquivalentTo = 0;  // Move to start of next day
            }
         // Case when start and end are on same day (no trading)
         else if (date_from_MONDAY == date_to_MONDAY)
            {
            MONDAY_MinuteEquivalentFrom = 0;
            MONDAY_MinuteEquivalentTo = 0;
            }
         }
      else
         {
         // If trading session not defined, set zero values
         MONDAY_MinuteEquivalentFrom = 0;
         MONDAY_MinuteEquivalentTo = 0;
         }
         
      // Tuesday processing
      if (bTUESDAY)
         {
         if (TUESDAY_MinuteEquivalentFrom == 0 && TUESDAY_MinuteEquivalentTo == 0 && date_from_TUESDAY != date_to_TUESDAY)
            {
            TUESDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (TUESDAY_MinuteEquivalentFrom == TUESDAY_MinuteEquivalentTo && date_from_TUESDAY != date_to_TUESDAY)
            {
            if (TUESDAY_MinuteEquivalentFrom < 60*23+59)
               {
               TUESDAY_MinuteEquivalentTo = TUESDAY_MinuteEquivalentFrom+1;
               }
            else TUESDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_TUESDAY == date_to_TUESDAY)
            {
            TUESDAY_MinuteEquivalentFrom = 0;
            TUESDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         TUESDAY_MinuteEquivalentFrom = 0;
         TUESDAY_MinuteEquivalentTo = 0;
         }
         
      // Wednesday processing
      if (bWEDNESDAY)
         {
         if (WEDNESDAY_MinuteEquivalentFrom == 0 && WEDNESDAY_MinuteEquivalentTo == 0 && date_from_WEDNESDAY != date_to_WEDNESDAY)
            {
            WEDNESDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (WEDNESDAY_MinuteEquivalentFrom == WEDNESDAY_MinuteEquivalentTo && date_from_WEDNESDAY != date_to_WEDNESDAY)
            {
            if (WEDNESDAY_MinuteEquivalentFrom < 60*23+59)
               {
               WEDNESDAY_MinuteEquivalentTo = WEDNESDAY_MinuteEquivalentFrom+1;
               }
            else WEDNESDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_WEDNESDAY == date_to_WEDNESDAY)
            {
            WEDNESDAY_MinuteEquivalentFrom = 0;
            WEDNESDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         WEDNESDAY_MinuteEquivalentFrom = 0;
         WEDNESDAY_MinuteEquivalentTo = 0;
         }
         
      // Thursday processing
      if (bTHURSDAY)
         {
         if (THURSDAY_MinuteEquivalentFrom == 0 && THURSDAY_MinuteEquivalentTo == 0 && date_from_THURSDAY != date_to_THURSDAY)
            {
            THURSDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (THURSDAY_MinuteEquivalentFrom == THURSDAY_MinuteEquivalentTo && date_from_THURSDAY != date_to_THURSDAY)
            {
            if (THURSDAY_MinuteEquivalentFrom < 60*23+59)
               {
               THURSDAY_MinuteEquivalentTo = THURSDAY_MinuteEquivalentFrom+1;
               }
            else THURSDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_THURSDAY == date_to_THURSDAY)
            {
            THURSDAY_MinuteEquivalentFrom = 0;
            THURSDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         THURSDAY_MinuteEquivalentFrom = 0;
         THURSDAY_MinuteEquivalentTo = 0;
         }      
      if (bFRIDAY)
         {
         if (FRIDAY_MinuteEquivalentFrom == 0 && FRIDAY_MinuteEquivalentTo == 0 && date_from_FRIDAY != date_to_FRIDAY)
            {
            FRIDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (FRIDAY_MinuteEquivalentFrom == FRIDAY_MinuteEquivalentTo && date_from_FRIDAY != date_to_FRIDAY)
            {
            if (FRIDAY_MinuteEquivalentFrom < 60*23+59)
               {
               FRIDAY_MinuteEquivalentTo = FRIDAY_MinuteEquivalentFrom+1;
               }
            else FRIDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_FRIDAY == date_to_FRIDAY)
            {
            FRIDAY_MinuteEquivalentFrom = 0;
            FRIDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         FRIDAY_MinuteEquivalentFrom = 0;
         FRIDAY_MinuteEquivalentTo = 0;
         }      
      if (bSATURDAY)
         {
         if (SATURDAY_MinuteEquivalentFrom == 0 && SATURDAY_MinuteEquivalentTo == 0 && date_from_SATURDAY != date_to_SATURDAY)
            {
            SATURDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (SATURDAY_MinuteEquivalentFrom == SATURDAY_MinuteEquivalentTo && date_from_SATURDAY != date_to_SATURDAY)
            {
            if (SATURDAY_MinuteEquivalentFrom < 60*23+59)
               {
               SATURDAY_MinuteEquivalentTo = SATURDAY_MinuteEquivalentFrom+1;
               }
            else SATURDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_SATURDAY == date_to_SATURDAY)
            {
            SATURDAY_MinuteEquivalentFrom = 0;
            SATURDAY_MinuteEquivalentTo = 0;
            }                  
         }
      else
         {
         SATURDAY_MinuteEquivalentFrom = 0;
         SATURDAY_MinuteEquivalentTo = 0;
         }      
      if (bSUNDAY)
         {
         if (SUNDAY_MinuteEquivalentFrom == 0 && SUNDAY_MinuteEquivalentTo == 0 && date_from_SUNDAY != date_to_SUNDAY)
            {
            SUNDAY_MinuteEquivalentTo=60*23+59;
            }
         else if (SUNDAY_MinuteEquivalentFrom == SUNDAY_MinuteEquivalentTo && date_from_SUNDAY != date_to_SUNDAY)
            {
            if (SUNDAY_MinuteEquivalentFrom < 60*23+59)
               {
               SUNDAY_MinuteEquivalentTo = SUNDAY_MinuteEquivalentFrom+1;
               }
            else SUNDAY_MinuteEquivalentTo = 0;
            }
         else if (date_from_SUNDAY == date_to_SUNDAY)
            {
            SUNDAY_MinuteEquivalentFrom = 0;
            SUNDAY_MinuteEquivalentTo = 0;
            }                  
         }     
      else
         {
         SUNDAY_MinuteEquivalentFrom = 0;
         SUNDAY_MinuteEquivalentTo = 0;
         }
      }

   //+------------------------------------------------------------------+
   //| Chart tick processing                                            |
   //| Updates historical data when new bars appear                     |
   //| Copies OHLC data and updates current prices                      |
   //+------------------------------------------------------------------+
   void ChartTick()
      {
      SymbolInfoTick(CurrentSymbol,LastTick);   // get last tick by symbol
      
      // Check for new bar appearance
      ArraySetAsSeries(tTimeI,false);           // set direct indexing for copying
      copied=CopyTime(CurrentSymbol,Timeframe,0,2,tTimeI); // copy time of last 2 bars
      ArraySetAsSeries(tTimeI,true);            // return reverse indexing (as in MT4)
      
      // If new bar appeared, update all historical data
      if ( copied == 2 && tTimeI[1] > LastCloseTime )
         {
         // Temporarily set direct indexing for data copying
         ArraySetAsSeries(CloseI,false);        // close prices array                      
         ArraySetAsSeries(OpenI,false);         // open prices array                         
         ArraySetAsSeries(HighI,false);         // high prices array                      
         ArraySetAsSeries(LowI,false);          // low prices array                            
         ArraySetAsSeries(TimeI,false);         // bar times array                                                          
         
         // Copy historical OHLC data
         lastcopied=CopyClose(CurrentSymbol,Timeframe,0,Chart::TCN+2,CloseI); // close prices
         lastcopied=CopyOpen(CurrentSymbol,Timeframe,0,Chart::TCN+2,OpenI);   // open prices
         lastcopied=CopyHigh(CurrentSymbol,Timeframe,0,Chart::TCN+2,HighI);   // high prices
         lastcopied=CopyLow(CurrentSymbol,Timeframe,0,Chart::TCN+2,LowI);     // low prices
         lastcopied=CopyTime(CurrentSymbol,Timeframe,0,Chart::TCN+2,TimeI);   // bar times
         
         // Return reverse indexing (as in MT4: [0] - current bar, [1] - previous)
         ArraySetAsSeries(CloseI,true);         // close prices array
         ArraySetAsSeries(OpenI,true);          // open prices array
         ArraySetAsSeries(HighI,true);          // high prices array                      
         ArraySetAsSeries(LowI,true);           // low prices array
         ArraySetAsSeries(TimeI,true);          // bar times array       
         
         LastCloseTime=tTimeI[1];               // remember time of last processed bar
         }
         
      // Update current prices and point size
      ChartBid=LastTick.bid;                    // current Bid price
      ChartAsk=LastTick.ask;                    // current Ask price
      ChartPoint=SymbolInfoDouble(CurrentSymbol,SYMBOL_POINT); // point size for calculations
      }
   };
//+--------------------------------------------------------------+
//| Static variable of Chart class                               |
//+--------------------------------------------------------------+
int Chart::TCN = 0;                     // number of bars for analysis (static variable)
Chart *ChartO;                          // global pointer to chart object

//+--------------------------------------------------------------+
//| BotInstance class - trading robot instance                   |
//| Manages trading logic, signals and deal execution            |
//| Contains strategy parameters and market analysis methods     |
//+--------------------------------------------------------------+
class BotInstance
   {
   public:
   //+------------------------------------------------------------------+
   //| Individual parameters of specific robot                          |
   //+------------------------------------------------------------------+
   bool bInitBuy;                       // permission to initialize buys for this robot instance
   bool bInitSell;                      // permission to initialize sells for this robot instance
   
   // Additional variables for counter-trend strategy
   int BarsChainForOpen;                // contiguous bars in one direction to open counter-trend position
   int BarsChainForClose;               // contiguous bars in one direction to close counter-trend position
   
   //+------------------------------------------------------------------+
   //| General parameters for specific robot                            |
   //+------------------------------------------------------------------+
   bool bInitSpreadControl;             // enable spread control before opening positions
   int SpreadE;                         // maximum allowable spread for opening positions
   int SpreadCloseE;                    // maximum allowable spread for closing positions (equals SpreadE)
   int MagicF;                          // magic number to identify orders of this robot
   string CurrentSymbol;                // trading symbol for this robot instance

   datetime DatetimeStart;              // start date from settings (symbolizes date from which we count time into future)

   int TradeDirection;                  // trading signal direction: 0-no signal, 1-buy, 2-sell
   int TradeDirectionClose;             // close signal direction: 0-no signal, 1-close buys, 2-close sells
   
   bool bOpened;                        // flag for existence of open positions for this robot
   
   bool bNewBarV;                       // auxiliary variable signaling that new bar arrived
   
   //+------------------------------------------------------------------+
   //| Constructor of BotInstance class                                 |
   //| Initializes all robot parameters with default values             |
   //+------------------------------------------------------------------+
   BotInstance()
      {
      bOpened=false;                    // initially positions are not open

      SpreadE=SpreadEE;                 // set max spread from global settings
      MagicF=MagicE;                    // set magic number from global settings
      RestartParams();                  // initialize other parameters
      }

   //+------------------------------------------------------------------+
   //| Main trading signal calculation function                         |
   //| Analyzes market situation and determines trading direction       |
   //+------------------------------------------------------------------+
   //| Main function for robot trading logic                            |
   //+------------------------------------------------------------------+
   void CalculateForTrade()
      {
      // Correct tone for signal designation (price movement prediction)
      
      // For opening new positions:
      // In other places of code:
      //   BotO.TradeDirection - 1 - is buy
      //   BotO.TradeDirection - 2 - is sell
      //   BotO.TradeDirection - 0 - no signal
      // Locally:
      //   TradeDirection - 1 - is buy
      //   TradeDirection - 2 - is sell
      //   TradeDirection - 0 - no signal
      
      // For closing hanging positions:
      // In other places of code:
      //   BotO.TradeDirectionClose - 1 - is buy
      //   BotO.TradeDirectionClose - 2 - is sell
      //   BotO.TradeDirectionClose - 0 - no signal
      // Locally:
      //   TradeDirectionClose - 1 - is buy
      //   TradeDirectionClose - 2 - is sell
      //   TradeDirectionClose - 0 - no signal
      
      // ChartO - access to chart data on which virtual robot is running
      
      // Example of calculating these values considering adopted strategy
      // Counter-trend strategy: look for sequence of bars in one direction,
      // then open position in opposite direction (expecting bounce/retracement)
      
      // First determine direction of the freshest closed bar (index [1])
      int firstTradeDirection = 0;      
      if ( ChartO.OpenI[1] > ChartO.CloseI[1] ) firstTradeDirection = 1; // bearish bar (close lower than open)
      if ( ChartO.CloseI[1] > ChartO.OpenI[1] ) firstTradeDirection = 2; // bullish bar (close higher than open)
      
      // Based on direction of this freshest bar, look for previous bars
      // indicating same direction and determine if there is signal for opening
      // or closing and in which direction
      // BarsChainForOpen - number of concurrent bars for open signal
      // BarsChainForClose - number of concurrent bars for close signal
              
      bool Aborted=false; // flag for aborting bar chain search
      if (firstTradeDirection == 1)
         {
         // Signal determination for opening long position (counter-trend to bearish trend)
         for ( int i=2; i<=BarsChainForOpen; i++ )
            {
            // Check that all previous bars are also bearish
            if (ChartO.OpenI[i] <= ChartO.CloseI[i])
               {
               Aborted=true; // found bullish bar - abort search
               break;
               }
            }
         if ( !Aborted ) TradeDirection = 1; // open long
         else TradeDirection = 0;
         
         Aborted=false;
         // Signal determination for closing short positions
         for ( int i=2; i<=BarsChainForClose; i++ )
            {
            // Check that all previous bars are also bearish
            if (ChartO.OpenI[i] <= ChartO.CloseI[i])
               {
               Aborted=true; // found bullish bar - abort search
               break;
               }            
            }
         if ( !Aborted ) TradeDirectionClose = 1; // close short
         else TradeDirectionClose = 0;            
         }
      else if (firstTradeDirection == 2)
         {
         // Signal determination for opening short position (counter-trend to bullish trend)
         for ( int i=2; i<=BarsChainForOpen; i++ )
            {
            // Check that all previous bars are also bullish
            if (ChartO.CloseI[i] <= ChartO.OpenI[i])
               {
               Aborted=true; // found bearish bar - abort search
               break;
               }
            }
         if ( !Aborted ) TradeDirection = 2; // open short
         else TradeDirection = 0;
         
         Aborted=false;
         // Signal determination for closing long positions
         for ( int i=2; i<=BarsChainForClose; i++ )
            {
            // Check that all previous bars are also bullish
            if (ChartO.CloseI[i] <= ChartO.OpenI[i])
               {
               Aborted=true; // found bearish bar - abort search
               break;
               }            
            }
         if ( !Aborted ) TradeDirectionClose = 2; // close long
         else TradeDirectionClose = 0;            
         }
      else
         {
         TradeDirection = 0;        // no signal for opening new positions
         TradeDirectionClose = 0;   // no signal for closing existing positions
         }                   
      }

   //+------------------------------------------------------------------+
   //| Method for processing position closing logic                     |
   //| Closes positions based on TradeDirectionClose signals            |
   //+------------------------------------------------------------------+
   void TradeClose()
      {
      // Close positions depending on received signal
      if (TradeDirectionClose==1) CloseType(OP_SELL,MagicF);     // close SELL positions on signal 1
      else if (TradeDirectionClose==2) CloseType(OP_BUY,MagicF); // close BUY positions on signal 2
      }

   //+------------------------------------------------------------------+
   //| Method for processing position opening logic                     |
   //| Opens positions based on TradeDirection signals                  |
   //+------------------------------------------------------------------+
   void Trade()
      {
      // Open positions depending on received signal and license check
      if (TradeDirection==1 && Days() <= DaysToFuture) BuyF(SLE,TPE);      // open BUY on signal 1 and valid license
      else if (TradeDirection==2 && Days() <= DaysToFuture) SellF(SLE,TPE); // open SELL on signal 2 and valid license
      }

   //+------------------------------------------------------------------+
   //| Robot parameters preparation and reset                           |
   //| Initializes parameters when instance is created                  |
   //+------------------------------------------------------------------+
   void RestartParams()
      {
      DatetimeStart=__DATETIME__;              // set start compilation date as reference point

      bInitSpreadControl=true;                 // enable spread control by default
      SpreadCloseE = SpreadEE;                 // set max spread for closing equal to spread for opening
      MagicF=MagicE;                           // copy magic number from global settings
      CurrentSymbol=ChartO.CurrentSymbol;      // set current trading symbol
      bInitBuy = TBE;                          // copy buy permission from global settings
      bInitSell = TSE;                         // copy sell permission from global settings
      
      // Add variable assignment for counter-trend strategy
      BarsChainForOpen = BarsChainForOpenE;    // number of bars for opening positions
      BarsChainForClose = BarsChainForCloseE;  // number of bars for closing positions           
      }

   //+------------------------------------------------------------------+
   //| Check existence of positions opened on current bar               |
   //| Returns: true if there are positions opened on current bar       |
   //+------------------------------------------------------------------+
   bool Opened()
      {
      bool ord;                                // flag for successful order selection
      
      // Iterate through all open positions
      for ( int i=0; i<OrdersTotal(); i++ )
         {
         ord=OrderSelect( i, SELECT_BY_POS, MODE_TRADES );
         
         // Check if position belongs to our robot and symbol
         if ( ord && OrderSymbol() == CurrentSymbol && OrderMagicNumber() == MagicF )
            {
            // If position opened on current bar, return true
            if ( OrderOpenTime() >= ChartO.TimeI[0] ) return true;
            }
         }
      return false;                            // no positions found on current bar
      }

   //+------------------------------------------------------------------+
   //| Check if within allowed trading window                           |
   //| Analyzes current time and trading sessions                       |
   //| Returns: true if trading is allowed at current time              |
   //+------------------------------------------------------------------+
   bool bInTradeSession()
      {
         MqlDateTime ME;                        // structure for parsing time into components
         TimeToStruct(ChartO.TimeI[0],ME);      // convert current bar time to structure
         int MinuteEqui = ME.hour*60+ME.min;    // calculate minute equivalent of current time

         // Check Monday trading session
         if (ME.day_of_week == MONDAY)
         {
             // Normal session (start earlier than end on same day)
             if (ChartO.MONDAY_MinuteEquivalentFrom < ChartO.MONDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.MONDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.MONDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             // Session across midnight (start later than end, transition to next day)
             else if (ChartO.MONDAY_MinuteEquivalentFrom > ChartO.MONDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.MONDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.MONDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         // Check Tuesday trading session
         else if (ME.day_of_week == TUESDAY)
         {
             if (ChartO.TUESDAY_MinuteEquivalentFrom < ChartO.TUESDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.TUESDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.TUESDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.TUESDAY_MinuteEquivalentFrom > ChartO.TUESDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.TUESDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.TUESDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         // Check Wednesday trading session
         else if (ME.day_of_week == WEDNESDAY)
         {
             if (ChartO.WEDNESDAY_MinuteEquivalentFrom < ChartO.WEDNESDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.WEDNESDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.WEDNESDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.WEDNESDAY_MinuteEquivalentFrom > ChartO.WEDNESDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.WEDNESDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.WEDNESDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == THURSDAY)
         {
             if (ChartO.THURSDAY_MinuteEquivalentFrom < ChartO.THURSDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.THURSDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.THURSDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.THURSDAY_MinuteEquivalentFrom > ChartO.THURSDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.THURSDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.THURSDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == FRIDAY)
         {
             if (ChartO.FRIDAY_MinuteEquivalentFrom < ChartO.FRIDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.FRIDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.FRIDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.FRIDAY_MinuteEquivalentFrom > ChartO.FRIDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.FRIDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.FRIDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == SATURDAY)
         {
             if (ChartO.SATURDAY_MinuteEquivalentFrom < ChartO.SATURDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.SATURDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.SATURDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.SATURDAY_MinuteEquivalentFrom > ChartO.SATURDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.SATURDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.SATURDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else if (ME.day_of_week == SUNDAY)
         {
             if (ChartO.SUNDAY_MinuteEquivalentFrom < ChartO.SUNDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.SUNDAY_MinuteEquivalentFrom && MinuteEqui <= ChartO.SUNDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
             else if (ChartO.SUNDAY_MinuteEquivalentFrom > ChartO.SUNDAY_MinuteEquivalentTo)
             {
                 if (MinuteEqui >= ChartO.SUNDAY_MinuteEquivalentFrom || MinuteEqui <= ChartO.SUNDAY_MinuteEquivalentTo) return true;
                 else return false;
             }
         }
         else
         {
             return false;
         }
         return false;
      }

   //+------------------------------------------------------------------+
   //| Internal robot tick                                              |
   //| Controls sequence of trading operations                          |
   //| Parameters:                                                      |
   //|   bFirst - first processing flag (true=close, false=open)        |
   //+------------------------------------------------------------------+
   void InstanceTick(bool bFirst)
      {
      if ((bTester && bNewBarV) || !bTester)// if new bar appeared
         {
         if ( bFirst )                      // first processing - close positions
            {
            TradeClose();                   // execute position closing logic
            } 
         if ( !bFirst )                     // second processing - open positions
            {
            Trade();                        // execute position opening logic        
            }
         }
      }
      
   public:
   datetime Time0;                          // time of last processed bar
   string NewBarType;                       // new bar type ("START" or "LATE")
   datetime PreviousLateTick;               // time of previous late tick
   datetime PreviousTimeCalc;               // time of previous trading signal calculation

   //+------------------------------------------------------------------+
   //| New bar detection                                                |
   //| Analyzes new bar appearance and determines processing type       |
   //| Returns: true if new bar appeared                                |
   //+------------------------------------------------------------------+
   bool bNewBar()
      {
      // Check new bar appearance at the beginning of its formation (first 20 seconds)
      if (ChartO.LastTick.time > 0 && ChartO.TimeI[1] > 0 && ChartO.TimeI[1] != Time0 && ChartO.ChartPoint != 0.0 
      && ChartO.LastTick.time - ChartO.TimeI[0] <= 20 && ChartO.LastTick.time - ChartO.TimeI[0] >= 0 )
         {
         NewBarType = "START";              // processing type - bar start (optimal time)
         Time0=ChartO.TimeI[1];             // remember time of processed bar
         return true;         
         }
      // Check new bar appearance at any other time (late processing)
      else if ( ChartO.LastTick.time > 0 && ChartO.TimeI[1] > 0 && ChartO.TimeI[1] != Time0 && ChartO.ChartPoint != 0.0 )
         {
         NewBarType = "LATE";               // processing type - late (middle or end of bar)
         Time0=ChartO.TimeI[1];             // remember time of processed bar
         return true;
         }
      else return false;                    // new bar not detected
      }

   //+------------------------------------------------------------------+
   //| Auxiliary functions for working with time and orders             |
   //+------------------------------------------------------------------+

   //+------------------------------------------------------------------+
   //| Hour correction                                                  |
   //| Parameters: hour0 - initial hour value                           |
   //| Returns: correct hour value (1-23) or 0 on error                 |
   //+------------------------------------------------------------------+
   int HourCorrect(int hour0)
   {
   int rez=0;                               // correction result
   if ( hour0 < 24 && hour0 > 0 )          // check allowable hour range
      {
      rez=hour0;                            // return initial value if correct
      }
   return rez;                              // return 0 if value incorrect
   }

   //+------------------------------------------------------------------+
   //| Minute correction                                                |
   //| Parameters: minute0 - initial minute value                       |
   //| Returns: correct minute value (1-59) or 0 on error               |
   //+------------------------------------------------------------------+
   int MinuteCorrect(int minute0)
      {
      int rez=0;                            // correction result
      if ( minute0 < 60 && minute0 > 0 )    // check allowable minute range
         {
         rez=minute0;                       // return initial value if correct
         }
      return rez;                           // return 0 if value incorrect     
      }

   //+------------------------------------------------------------------+
   //| Count total number of robot orders                               |
   //| Returns: number of open positions of this robot                  |
   //+------------------------------------------------------------------+
   int OrdersG()
      {
      bool ord;                             // flag for successful order selection
      int OrdersG0=0;                       // order counter
      
      // Iterate through all open positions
      for ( int i=0; i<OrdersTotal(); i++ )
         {
         ord=OrderSelect( i, SELECT_BY_POS, MODE_TRADES );
         
         // Count only orders of our robot and symbol
         if ( ord && OrderSymbol() == CurrentSymbol && OrderMagicNumber() == MagicF )
            {
            OrdersG0++;                     // increment counter
            }
         }
      return OrdersG0;                      // return total number of orders
      }

   //+------------------------------------------------------------------+
   //| Check absence of opposite positions                              |
   //| Parameters: direction - direction (true=SELL, false=BUY)         |
   //| Returns: true if no opposite positions                           |
   //+------------------------------------------------------------------+
   bool OrdersG(bool direction)
      {
      bool ord;                             // flag for successful order selection
      
      // Iterate through all open positions
      for ( int i=0; i<OrdersTotal(); i++ )
         {
         ord=OrderSelect( i, SELECT_BY_POS, MODE_TRADES );
         
         // Check existence of opposite positions
         if ( ord && OrderSymbol() == CurrentSymbol && OrderMagicNumber() == MagicF && 
              ((!direction && OrderType() == OP_BUY) || (direction && OrderType() == OP_SELL)) )
            {
            return false;                   // opposite position found
            }
         }
      return true;                          // no opposite positions found
      }

   //+------------------------------------------------------------------+
   //| Structure for storing Stop Loss and Take Profit levels           |
   //+------------------------------------------------------------------+
   struct StopsPoints
      {
      int StopLossPoints;                   // Stop Loss level in points from position open price
      int TakeProfitPoints;                 // Take Profit level in points from position open price
      };

   //+------------------------------------------------------------------+
   //| Trading functions block                                          |
   //+------------------------------------------------------------------+

   //+------------------------------------------------------------------+
   //| Open BUY positions function                                      |
   //| Parameters:                                                      |
   //|   SL0 - Stop Loss size in points                                 |
   //|   TP0 - Take Profit size in points                               |
   //+------------------------------------------------------------------+
   void BuyF(double SL0,double TP0)
      {
      double DtA;                           // time difference for position opening frequency control
      double SLTemp0=MathAbs(SL0);          // absolute Stop Loss value
      double TPTemp0=MathAbs(TP0);          // absolute Take Profit value

      // Check time of last position opening via global variable
      DtA=double(TimeCurrent())-GlobalVariableGet(GLOBALVARPREFIX+IntegerToString(MagicF));
      if ( DtA > 0 || DtA < 0 )             // if variable exists
         {
         if ( ChartO.LastTick.bid > 0.0 && ChartO.OpenI[0] > 0.0 && ((NewBarType == "LATE" && ChartO.OpenI[0] >= ChartO.LastTick.bid) || NewBarType == "START") 
         && bInitBuy && (DtA > 0 || DtA < 0) && ( (!bInitRepurchaseE && OrdersG() == 0) || (bInitRepurchaseE && OrdersG(true) && AddtionalOpenPrice(ChartO.LastTick.ask,true))) )
            {
            CheckForOpen(true,MagicF,OP_BUY,ChartO.ChartBid,ChartO.ChartAsk,MathAbs(SlippageMaxOpen),ChartO.ChartBid,int(SLTemp0),int(TPTemp0),LotE,CurrentSymbol,0,CommentI,bInitLotControl,bInitSpreadControl,SpreadE);
            }
         }
      }

   //+------------------------------------------------------------------+
   //| Open SELL positions function                                     |
   //| Parameters:                                                      |
   //|   SL0 - Stop Loss size in points                                 |
   //|   TP0 - Take Profit size in points                               |
   //+------------------------------------------------------------------+
   void SellF(double SL0,double TP0)
      {
      double DtA;                           // time difference for position opening frequency control
      double SLTemp0=MathAbs(SL0);          // absolute Stop Loss value
      double TPTemp0=MathAbs(TP0);          // absolute Take Profit value

      // Check time of last position opening via global variable
      DtA=double(TimeCurrent())-GlobalVariableGet(GLOBALVARPREFIX+IntegerToString(MagicF));
      if ( DtA > 0 || DtA < 0 )             // if variable exists
         {
         // Check all conditions for opening SELL position
         if ( ChartO.LastTick.bid > 0.0 && ChartO.OpenI[0] > 0.0 && 
              ((NewBarType == "LATE" && ChartO.LastTick.bid >= ChartO.OpenI[0]) || NewBarType == "START") 
              && bInitSell && (DtA > 0 || DtA < 0) && 
              ( (!bInitRepurchaseE && OrdersG() == 0) || (bInitRepurchaseE && OrdersG(false) && AddtionalOpenPrice(ChartO.LastTick.bid,false))) )
            {
            // Call function to check and open order
            CheckForOpen(true,MagicF,OP_SELL,ChartO.ChartBid,ChartO.ChartAsk,MathAbs(SlippageMaxOpen),ChartO.ChartBid,int(SLTemp0),int(TPTemp0),LotE,CurrentSymbol,0,CommentI,bInitLotControl,bInitSpreadControl,SpreadE);
            }         
         }
      }

   public:
   //+------------------------------------------------------------------+
   //| Calculate number of days since trading start                     |
   //| Returns: number of days since DatetimeStart                      |
   //+------------------------------------------------------------------+
   double Days()
      {
      // If time has passed since start, return number of days
      if ( double(ChartO.TimeI[0] - DatetimeStart) > 0 ) return double(ChartO.TimeI[0] - DatetimeStart)/86400.0;
      else return 0.0;                      // otherwise return 0
      }
   private:

   //+------------------------------------------------------------------+
   //| Check sufficiency of funds for position opening                  |
   //| Parameters:                                                      |
   //|   symb - trading symbol                                          |
   //|   lots - lot size                                                |
   //|   type - operation type (OP_BUY or OP_SELL)                      |
   //| Returns: true if funds are sufficient                            |
   //+------------------------------------------------------------------+
   bool CheckMoneyForTrade(string symb, double lots,int type)
      {
      double free_margin=AccountFreeMarginCheck(symb,type,lots); // check free margin after position opening
      
      // If not enough money
      if(free_margin<0)
        {
         string oper=(type==OP_BUY)? "Buy":"Sell"; // determine operation type for message
         Print("Not enough money for ", oper," ",lots, " ", symb, " Error code=",GetLastError()); // print error message
         return(false);                     // return false - insufficient funds
        }
      // Check passed successfully
      return(true);                         // return true - funds sufficient
      }

   //+------------------------------------------------------------------+
   //| Universal function for order check and opening                   |
   //| Performs all necessary checks before opening position            |
   //+------------------------------------------------------------------+
   void CheckForOpen(bool Condition0,int Magic0,int OrdType,double PriceBid,double PriceAsk,int Slippage0,double PriceClose0,int SL0,int TP0,double Lot0,string Symbol0,datetime Expiration0,string Comment0,bool bLotControl0,bool bSpreadControl0,int Spread0)
      {
      bool ord;                             // flag for successful order opening on Condition0
      double LotTemp=Lot0;                  // temporary variable for lot size
      double SpreadLocal=MarketInfo(CurrentSymbol,MODE_SPREAD); // current spread
      double LotAntiError=MarketInfo(CurrentSymbol,MODE_MINLOT); // minimum lot size
                                                                
      // Check spread if spread control enabled
      if ( (SpreadLocal <= Spread0 && bSpreadControl0 == true ) || ( bSpreadControl0 == false )  )                    
         {   
         if ( bLotControl0 == false )       // if autolot disabled
            {
            LotTemp=Lot0;
            }
         else if ( bLotControl0 == true )
            {
            UpdateMaxBalance();//first update max balance info
            LotTemp=Lot0*(LastMaxBalance/DepositE);
            }
         if (bInitMartinE)
            {
            LotTemp=CalcMartinLot(LotTemp);
            }
         LotAntiError=GetLotAniError(LotTemp);
   
         if ( LotAntiError > 0 && CheckMoneyForTrade(ChartO.CurrentSymbol,LotAntiError,OP_SELL) && (OrdType == OP_SELL || OrdType == OP_SELLLIMIT || OrdType == OP_SELLSTOP) )
            {
            ord=OrderSend( Symbol0 , OrdType, LotAntiError, PriceBid, Slippage0,SL0 != 0 ? ChartO.ChartBid+CorrectLevels(SL0)*ChartO.ChartPoint: 0,TP0 != 0 ? ChartO.ChartBid-CorrectLevels(TP0)*ChartO.ChartPoint: 0, Comment0, Magic0, Expiration0, Red);
            GlobalVariableSet(GLOBALVARPREFIX+IntegerToString(MagicF),double(TimeCurrent()) ); 
            }
         if ( LotAntiError > 0 && CheckMoneyForTrade(ChartO.CurrentSymbol,LotAntiError,OP_BUY) && (OrdType == OP_BUY || OrdType == OP_BUYLIMIT || OrdType == OP_BUYSTOP)  )
            {
            ord=OrderSend( Symbol0 , OrdType, LotAntiError, PriceAsk, Slippage0,SL0 != 0 ? ChartO.ChartAsk-CorrectLevels(SL0)*ChartO.ChartPoint: 0, TP0 != 0 ? ChartO.ChartAsk+CorrectLevels(TP0)*ChartO.ChartPoint: 0, Comment0, Magic0, Expiration0, Green);
            GlobalVariableSet(GLOBALVARPREFIX+IntegerToString(MagicF),double(TimeCurrent()) ); 
            }
         }         
      }

   //calculate martin-lot enhancement relative to base   
   bool bFirstMaxCalculation;
   datetime LastMaxBalanceTime;             // time of last max balance for martingale
   datetime OptimizationBorderTime;         // optimization boundary time for history analysis

   //+------------------------------------------------------------------+
   //| Calculate lot size for martingale system                         |
   //| Parameters:                                                      |
   //|   BasicLot - base lot size                                       |
   //| Returns: corrected lot size considering martingale               |
   //+------------------------------------------------------------------+
   double CalcMartinLot(double BasicLot) 
      { 
      // Martingale lot calculation based on losing trades history analysis
      // MaxMartinLotMultiplierStepsE - maximum number of lot increase steps
      
      bool ord;                             // flag for successful order selection from history
      bool bFirst=false;                    // flag for first found deal
      bool bSecond=false;                   // flag for finding max loss
      double TempSummProfit=0.0;            // accumulated loss sum
      double LastMaxProfit=0.0;             // maximum found loss
      datetime FirstDate=0;                 // time of first deal in analyzed period
      
      datetime BorderTime=TimeCurrent()-365*86400; // analysis boundary - 1 year ago
      
      // Iterate through deal history from new to old to search for last max loss
      for ( int i=OrdersHistoryTotal()-1; i>=0; i-- )
         {
         ord=OrderSelect(i, SELECT_BY_POS,MODE_HISTORY);
         if ( OrderCloseTime() < BorderTime ) break; // go beyond analyzed period
         
         // Analyze only deals of our robot and symbol
         if ( OrderMagicNumber() == MagicE && OrderSymbol() == ChartO.CurrentSymbol )
            {
            if ( OrderCloseTime() >= OptimizationBorderTime) // if deal is in analyzed period
               {
               double TempProfit=-(OrderProfit() + OrderCommission() + OrderSwap()); // determine current deal loss (invert sign)
               TempSummProfit+=TempProfit;      // accumulate total loss
               
               if (!bFirst)                     // remember time of first deal
                  {
                  FirstDate=OrderCloseTime()+1;
                  bFirst=true;
                  }
                  
               // If found larger loss, update max
               if (TempSummProfit > LastMaxProfit)
                  {
                  LastMaxProfit = TempSummProfit;
                  LastMaxBalanceTime=OrderCloseTime(); // remember time of max loss
                  bSecond=true;
                  }
               }
            else break;                         // went beyond optimization boundaries
            }
         }
      if (!bSecond) LastMaxBalanceTime=FirstDate; // if max not found, use time of first deal
      if (LastMaxBalanceTime > OptimizationBorderTime) OptimizationBorderTime=LastMaxBalanceTime;
      
      
      double TempSummLots=0.0;//total lot of all losing positions
      int summs=0;//how many losing positions already found
      TempSummProfit=0.0;
      double MiddleProfit=0.0;
      //now remains to calculate sum of lots from negative positions
      
      for ( int i=OrdersHistoryTotal()-1; i>=0; i-- )//calculate average loss size for all deals per 1 lot
      {
      ord=OrderSelect(i, SELECT_BY_POS,MODE_HISTORY);
      if ( OrderCloseTime() < BorderTime ) break;
      if ( OrderMagicNumber() == MagicE && OrderSymbol() == ChartO.CurrentSymbol )
         {
         if (OrderCloseTime() < LastMaxBalanceTime) break;//interruption if went into already calculated part of history
         if (summs >= MaxMartinLotMultiplierStepsE) break;//interruption if exceeding number of allowed losses
         
         double TempProfit=OrderProfit() + OrderCommission() + OrderSwap();//determine profit of current deal
         if (TempProfit < 0 )
            {
            double TempVolume=OrderLots();//determine volume of current deal
            TempSummProfit+=MathAbs(TempProfit/TempVolume);
            summs++;
            }
         }
      }   
      
      //finish calculating average loss per 1 lot, if there was more than 1 position
      if (summs>1)
         {
         MiddleProfit = TempSummProfit/summs;
         }
      //
      
      summs=0;//reset counter
      //now using this average loss size we will add weights to each loss lot
      for ( int i=OrdersHistoryTotal()-1; i>=0; i-- )//calculate sum of lots of negative positions (considering limits on their max quantity)
         {
         ord=OrderSelect(i, SELECT_BY_POS,MODE_HISTORY);
         if ( OrderCloseTime() < BorderTime ) break;
         if ( OrderMagicNumber() == MagicE && OrderSymbol() == ChartO.CurrentSymbol )
            {
            if (OrderCloseTime() < LastMaxBalanceTime) break;//interruption if went into already calculated part of history
            if (summs >= MaxMartinLotMultiplierStepsE) break;//interruption if exceeding number of allowed losses
            
            double TempProfit=OrderProfit() + OrderCommission() + OrderSwap();//determine profit of current deal
            if (TempProfit < 0 )
               {
               double TempVolume=OrderLots();//determine volume of current deal
               if (MiddleProfit > 0.0)
                  {             
                  double Multiplier = MathAbs(TempProfit/TempVolume)/MiddleProfit;
                  TempSummLots += TempVolume * Multiplier;
                  summs++;
                  } 
               else
                  {
                  TempSummLots += TempVolume;
                  summs++;                  
                  }
               }
            }
         }
      
      //decision making     
      return BasicLot + TempSummLots;  
      }

   //+------------------------------------------------------------------+
   //| Check and lot size correction considering limits                 |
   //| Corrects lot size according to min/max                           |
   //| value and lot step, checks margin sufficiency                    |
   //| Parameters:                                                      |
   //|   InputLot - initial lot size for check                          |
   //| Returns: corrected lot or -1.0 if insufficient margin            |
   //+------------------------------------------------------------------+
   double GetLotAniError(double InputLot)
      {
      double Free = AccountFreeMargin();
      double margin = MarketInfo(CurrentSymbol,MODE_MARGINREQUIRED);
      double minLot = MarketInfo(CurrentSymbol,MODE_MINLOT);
      double Max_Lot = MarketInfo(CurrentSymbol,MODE_MAXLOT);
      double Step = MarketInfo(CurrentSymbol,MODE_LOTSTEP);
      double Lot13;
      int LotCorrection;
      LotCorrection=int(MathFloor(InputLot/Step));
      Lot13=LotCorrection*Step;   
      if(Lot13<=minLot) Lot13 = minLot;
      if(Lot13>=Max_Lot) Lot13 = Max_Lot;
      
      if( Lot13*margin>=Free ) Lot13=-1.0;
   
      return Lot13;
      }
   
   //+------------------------------------------------------------------+
   //| Check if current time falls into specified interval              |
   //| Supports both server and local computer time                     |
   //| Parameters:                                                      |
   //|   HourStart0 - interval start hour                               |
   //|   MinuteStart0 - interval start minute                           |
   //|   HourEnd0 - interval end hour                                   |
   //|   MinuteEnd0 - interval end minute                               |
   //|   bInitPCTime0 - use local time (true/false)                     |
   //| Returns: true if current time falls into interval                |
   //+------------------------------------------------------------------+
   bool bTimeInterval(int HourStart0,int MinuteStart0,int HourEnd0,int MinuteEnd0,bool bInitPCTime0)
      {
      bool rez=false;
      
      if ( (Hour() >= HourStart0 && Hour() <= HourEnd0 && Minute() >= MinuteStart0 && Minute() <= MinuteEnd0 && bInitPCTime0==false) || ( TimeHour(TimeLocal()) >= HourStart0 && TimeHour(TimeLocal()) <= HourEnd0 && TimeMinute(TimeLocal()) >= MinuteStart0 && TimeMinute(TimeLocal()) <= MinuteEnd0 && bInitPCTime0==true) )
         {
         rez=true;
         }
         
      return rez;
      } 
   
   //+------------------------------------------------------------------+
   //| Stop Loss and Take Profit levels correction                      |
   //| Checks and corrects levels considering minimum distance          |
   //| from current price according to broker requirements              |
   //| Parameters:                                                      |
   //|   level0 - initial level in points                               |
   //| Returns: corrected level or 0 if incorrect                       |
   //+------------------------------------------------------------------+
   int CorrectLevels(int level0)
      {
      int rez;
      int ZeroLevel0=int(MathAbs(MarketInfo(CurrentSymbol,MODE_STOPLEVEL))+MathAbs(MarketInfo(CurrentSymbol,MODE_SPREAD))+1);
   
      if ( MathAbs(level0) > ZeroLevel0 )       // if level greater than minimum
         {
         rez=int(MathAbs(level0));              // return absolute level value
         }
      else
         {
         rez=ZeroLevel0;                        // otherwise return minimum level
         }   
      return rez;
      }

   //+------------------------------------------------------------------+
   //| Check position holding time expiration                           |
   //| Parameters:                                                      |
   //|   TimeStart - position holding start time                        |
   //| Returns: true if holding time expired                            |
   //+------------------------------------------------------------------+
   bool bTimeIsOut(int TimeStart)
      {
      // Check holding time excess in minutes
      if ( double(int(TimeCurrent()) - TimeStart)/60.0 > double(MinutesHoldE)  ) return true;
      else return false;
      }

   //+------------------------------------------------------------------+
   //| Close all orders of specified type                               |
   //| Parameters:                                                      |
   //|   OT - order type for closing (OP_BUY or OP_SELL)                |
   //|   magic0 - magic number for filtering orders                     |
   //+------------------------------------------------------------------+
   void CloseType(ENUM_ORDER_TYPE OT,int magic0)
      {
      bool order;                           // order close operation result
      bool ord;                             // flag for successful order selection
      int Tickets[];                        // array of order tickets for closing
      double Lotsx[];                       // array of order lot sizes
      bool Typex[];                         // array of order types
      string SymbolX[];                     // array of order symbols
      int TicketsTotal=0;                   // total number of orders for closing
      int TicketNumCurrent=0;               // current index for filling arrays
      MqlTick TickS;                        // structure for getting current prices
      
      // Check spread if spread control enabled
      if (  !bInitSpreadControl || (bInitSpreadControl && MarketInfo(CurrentSymbol,MODE_SPREAD) <= SpreadCloseE)  )
         {
         // If repurchase system not used or loss linearization enabled
         if (!bInitRepurchaseE || bInitSitE)
            {
            // First pass - count number of orders for closing
            for ( int i=0; i<OrdersTotal(); i++ )
               {
               ord=OrderSelect( i, SELECT_BY_POS, MODE_TRADES );
               int tempticket=OrderTicket();         
               if (ord && tempticket > 0 && OrderMagicNumber() == magic0 && OrderSymbol() == CurrentSymbol
               && ( !bInitSitE || ( bInitSitE && ( (OrderProfit()+OrderSwap() - OrderLots()*ComissionPerLotE) > 0.0 || bTimeIsOut(int(OrderOpenTime())) ) ) ) )            
                  {
                  TicketsTotal++;
                  }
               }
            
            if ( TicketsTotal > 0 )
               {
               ArrayResize(Tickets,TicketsTotal);
               ArrayResize(Lotsx,TicketsTotal);
               ArrayResize(Typex,TicketsTotal);
               ArrayResize(SymbolX,TicketsTotal);
                        
               for ( int i=0; i<OrdersTotal(); i++ )
                   {
                   ord=OrderSelect( i, SELECT_BY_POS, MODE_TRADES );
                                        
                   if ( ord && OrderMagicNumber() == magic0 && TicketNumCurrent < TicketsTotal && OrderSymbol() == CurrentSymbol
                   && ( !bInitSitE || ( bInitSitE && ( (OrderProfit()+OrderSwap() - OrderLots()*ComissionPerLotE) > 0.0 || bTimeIsOut(int(OrderOpenTime())) ) ) ) )
                       {
                       int tempticket=OrderTicket();
                       if ( tempticket > 0 )
                          {
                          Tickets[TicketNumCurrent]=tempticket;
                          if ( OrderType() == OP_BUY )
                              {
                              Typex[TicketNumCurrent]=true;
                              }
                          if ( OrderType() == OP_SELL )
                              {
                              Typex[TicketNumCurrent]=false;
                              }                           
                          Lotsx[TicketNumCurrent]=OrderLots();
                          SymbolX[TicketNumCurrent]=OrderSymbol();            
                          TicketNumCurrent=TicketNumCurrent+1;                        
                          }
                       }
                   }
                     
               for ( int i=0; i<TicketsTotal; i++ )
                   {
                   SymbolInfoTick(SymbolX[i],TickS);
                   ord=OrderSelect( Tickets[i], SELECT_BY_TICKET, MODE_TRADES );
                   if ( ord && Tickets[i] > 0 && Typex[i] == true && OT == OP_BUY 
                   && ChartO.LastTick.bid > 0.0 && ChartO.OpenI[0] > 0.0 && ((NewBarType == "LATE" && ChartO.LastTick.bid >= ChartO.OpenI[0]) || NewBarType == "START") )
                       {
                       order=OrderClose(Tickets[i],Lotsx[i],TickS.bid,MathAbs(SlippageMaxClose),Green);
                       }
                   if ( ord && Tickets[i] > 0 && Typex[i] == false && OT == OP_SELL 
                   && ChartO.LastTick.bid > 0.0 && ChartO.OpenI[0] > 0.0 && ((NewBarType == "LATE" && ChartO.OpenI[0] >= ChartO.LastTick.bid) || NewBarType == "START") )
                       {
                       order=OrderClose(Tickets[i],Lotsx[i],TickS.ask,MathAbs(SlippageMaxClose),Red);
                       }                    
                   }
               }            
            }
         else
            {
            if (OT == OP_BUY) CloseToProfit(true); 
            if (OT == OP_SELL) CloseToProfit(false);
            }
         }   
      }
   };

BotInstance *BotO;

//+--------------------------------------------------------------+
//| Expert initialization function                               |
//+--------------------------------------------------------------+
bool bFirstTimer = false;
//+--------------------------------------------------------------+
//| Global variables for expert management                       |
//+--------------------------------------------------------------+
bool bFirstTick = false;                // flag of the first tick for initialization in timer-less mode
bool bTimerCreated=false;               // flag of successful timer creation

//+--------------------------------------------------------------+
//| Expert initialization function                               |
//| Executed when expert is launched on chart                    |
//| Creates timer for real-time operation                        |
//+--------------------------------------------------------------+
int OnInit()
  {
   bTester=MQLInfoInteger(MQL_TESTER);  // determine if we are working in strategy tester
   
   // Attempt to create timer (up to 5 attempts)
   for (int i = 0; i < 5; i++)
      {
      if (!bTester)
         {
         bTimerCreated = EventSetTimer(1);     // create timer with 1 second interval for real time
         }
      else bTimerCreated=false;              // timer is not needed in tester
      
      // If timer was not created, try again
      if (!bTimerCreated)
         {
         EventKillTimer();                   // remove timer just in case it remained         
         Sleep(100);                         // short pause before next attempt
         }
      else break;                            // timer created successfully, exit loop
      }
   return(INIT_SUCCEEDED);                   // return successful initialization code
  }

//+--------------------------------------------------------------+
//| Expert deinitialization function                             |
//| Executed when expert is removed from chart                   |
//| Clears resources and stops timer                             |
//+--------------------------------------------------------------+
void OnDeinit(const int reason)
  {
     bTimerCreated=false;                    // reset timer flag
     bFirstTimer = false;                    // reset first timer flag
     bFirstTick = false;                     // reset first tick flag
     DeInit();                               // execute user deinitialization
     EventKillTimer();                       // stop timer
  }

//+----------------------------------------------------------------+
//| Timer event handler                                            |
//| Executed every second in real time                             |
//| Initializes robot on first call, then simulates ticks          |
//+----------------------------------------------------------------+
void OnTimer()
   {
   if (bTimerCreated)
      {
      if (!bFirstTimer)
         {
         Init();                             // initialize robot on first timer call
         bFirstTimer=true;                   // set first timer flag
         } 
      else Simulated();                      // execute main robot logic
      }
   }

//+--------------------------------------------------------------+
//| Tick event handler                                           |
//| Executed on every price change                               |
//| Used in tester or when timer is not working                  |
//+--------------------------------------------------------------+
void OnTick()
   {
   if (!bTimerCreated)                       // if timer is not created (e.g. in tester)
      {
      if (!bFirstTick)
         {
         Init();                             // initialize robot on first tick
         bFirstTick=true;                    // set first tick flag
         }       
      else Simulated();                      // execute main robot logic
      }
   }

//+--------------------------------------------------------------+
//| Variables for adaptation                                     |
//+--------------------------------------------------------------+
datetime LastAdapt=0;                       // time of last parameter readaptation

//+--------------------------------------------------------------+
//| Tick or timer simulation                                     |
//| Main function for executing trading logic                    |
//| Updates chart data, executes trading operations              |
//+--------------------------------------------------------------+
void Simulated()
   {
   ChartO.ChartTick();                      // generate tick on virtual chart (update data)
   BotTick();                               // generate tick on virtual bot (execute trading logic)
   UpdateStatus();                          // update user interface status
   }

int Cor=0;                                  // auxiliary variable for correction
bool bTester = false;                       // flag for working in strategy tester

//+--------------------------------------------------------------+
//| Trading robot initialization                                 |
//| Creates all necessary objects and interface                  |
//| Called on first tick or timer                                |
//+--------------------------------------------------------------+
void Init()
   {
   DeleteSimpleInterface();                 // remove previous interface if it existed
   
   CreateChart();                           // create single chart object for data analysis
   CreateInstance();                        // create single trading robot instance
      
   CreateSimpleInterface();                 // create user interface
   UpdateStatus();                          // update interface status with current data
   }

//+--------------------------------------------------------------+
//| Trading robot deinitialization                               |
//| Clears memory and removes interface                          |
//| Called when removing expert from chart                       |
//+--------------------------------------------------------------+
void DeInit()
   {
   DeleteSimpleInterface();                 // remove user interface
   
   // Clear dynamic memory
   delete ChartO;                           // delete chart object
   delete BotO;                             // delete trading robot object
   }

//+--------------------------------------------------------------+
//| Lot balancer - correction coefficient by timeframe           |
//| Parameters:                                                  |
//| tf - timeframe for coefficient calculation                   |
//| Returns: average movement correction coefficient             |
//+--------------------------------------------------------------+
double PeriodK(ENUM_TIMEFRAMES tf)
   {
   double ktemp;                            // temporary variable to store coefficient
   
   // Determine coefficient depending on timeframe
   switch(tf)
      {
      case  PERIOD_H1:                      // hourly timeframe
          ktemp = MathSqrt(1.0/60.0);       // coefficient for H1
          break;
      case  PERIOD_H4:                      // 4-hour timeframe
          ktemp = MathSqrt(1.0/240.0);      // coefficient for H4
          break;
      case PERIOD_M1:                       // minute timeframe
          ktemp = 1.0;                      // base coefficient for M1
          break;
      case PERIOD_M5:                       // 5-minute timeframe
          ktemp = MathSqrt(1.0/5.0);        // coefficient for M5
          break;
      case PERIOD_M15:                      // 15-minute timeframe
          ktemp = MathSqrt(1.0/15.0);       // coefficient for M15
          break;
      case PERIOD_M30:                      // 30-minute timeframe
          ktemp = MathSqrt(1.0/30.0);       // coefficient for M30
          break;
      default: ktemp = 0;                   // for undefined timeframes coefficient equals zero
      }
   return ktemp;
   }

//+--------------------------------------------------------------+
//| Interface section                                            |
//+--------------------------------------------------------------+
bool bInterface=true; // flag for enabling graphical interface

//+--------------------------------------------------------------+
//| Functions for creating graphical interface elements          |
//+--------------------------------------------------------------+

//+--------------------------------------------------------------+
//| Creating rectangular frame for interface                     |
//| Parameters:                                                  |
//| chart_ID - chart ID                                          |
//| name - label object name                                     |
//| sub_window - subwindow number                                |
//| x, y - anchor coordinates                                    |
//| width, height - rectangle dimensions                         |
//| corner - anchor corner                                       |
//| clr - background color                                       |
//| style - border style                                         |
//| line_width - border thickness                                |
//| back - background object                                     |
//| selection - possibility of selection                         |
//| hidden - hidden object                                       |
//| z_order - display order                                      |
//| Returns: true on successful creation                         |
//+--------------------------------------------------------------+
bool RectLabelCreate(
const long chart_ID=0, // chart ID 
const string name="RectLabel", // label name 
const int sub_window=0, // subwindow number 
const int x=0, // X axis coordinate 
const int y=0, // Y axis coordinate 
const int width=50, // width 
const int height=18, // height 
const color back_clr=C'236,233,216', // background color 
const ENUM_BORDER_TYPE border=BORDER_SUNKEN, // border type const 
ENUM_BASE_CORNER corner=CORNER_LEFT_UPPER, // anchor chart corner 
const color clr=clrRed, // flat border color 
const ENUM_LINE_STYLE style=STYLE_SOLID, // flat border style 
const int line_width=1, // flat border thickness 
const bool back=false, // on background const 
bool selection=false, // select for moving
const bool hidden=true, // hidden in object list 
const long z_order=0) // mouse click priority 
{ 
//--- reset error value 
ResetLastError(); 
//--- create rectangular label 
if(!ObjectCreate(chart_ID,name,OBJ_RECTANGLE_LABEL,sub_window,0,0)) 
   { 
   Print(__FUNCTION__, ": failed to create rectangular label! Error code = ",GetLastError()); 
   return(false); 
   } 
//--- set label coordinates 
ObjectSetInteger(chart_ID,name,OBJPROP_XDISTANCE,x); 
ObjectSetInteger(chart_ID,name,OBJPROP_YDISTANCE,y); 
//--- set label dimensions 
ObjectSetInteger(chart_ID,name,OBJPROP_XSIZE,width); 
ObjectSetInteger(chart_ID,name,OBJPROP_YSIZE,height); 
//--- set background color 
ObjectSetInteger(chart_ID,name,OBJPROP_BGCOLOR,back_clr); 
//--- set border type 
ObjectSetInteger(chart_ID,name,OBJPROP_BORDER_TYPE,border); 
//--- set chart corner relative to which point coordinates will be determined 
ObjectSetInteger(chart_ID,name,OBJPROP_CORNER,corner); 
//--- set flat border color (in Flat mode) 
ObjectSetInteger(chart_ID,name,OBJPROP_COLOR,clr); 
//--- set flat border line style 
ObjectSetInteger(chart_ID,name,OBJPROP_STYLE,style); 
//--- set flat border thickness 
ObjectSetInteger(chart_ID,name,OBJPROP_WIDTH,line_width); 
//--- display on foreground (false) or background (true) 
ObjectSetInteger(chart_ID,name,OBJPROP_BACK,back); 
//--- enable (true) or disable (false) label moving mode with mouse 
ObjectSetInteger(chart_ID,name,OBJPROP_SELECTABLE,selection); 
ObjectSetInteger(chart_ID,name,OBJPROP_SELECTED,selection); 
//--- hide (true) or show (false) graphical object name in object list 
ObjectSetInteger(chart_ID,name,OBJPROP_HIDDEN,hidden); 
//--- set priority for receiving mouse click event on chart 
ObjectSetInteger(chart_ID,name,OBJPROP_ZORDER,z_order); 
//--- successful execution 
return(true); 
}

//create text
bool LabelCreate(const long              chart_ID=0,               // chart ID
                 const string            name="Label",             // label name
                 const int               sub_window=0,             // subwindow number
                 const int               x=0,                      // X axis coordinate
                 const int               y=0,                      // Y axis coordinate
                 const ENUM_BASE_CORNER  corner=CORNER_LEFT_UPPER, // anchor chart corner
                 const string            text="Label",             // text
                 const string            font="Arial",             // font
                 const int               font_size=10,             // font size
                 const color             clr=clrRed,               // color
                 const double            angle=0.0,                // text angle
                 const ENUM_ANCHOR_POINT anchor=ANCHOR_LEFT_UPPER, // anchor method
                 const bool              back=false,               // on background
                 const bool              selection=false,          // select for moving
                 const bool              hidden=true,              // hidden in object list
                 const long              z_order=0)                // mouse click priority
  {
//--- reset error value
   ResetLastError();
//--- create text label
   if(!ObjectCreate(chart_ID,name,OBJ_LABEL,sub_window,0,0))
     {
      Print(__FUNCTION__,
            ": failed to create text label! Error code = ",GetLastError());
      return(false);
     }
//--- set label coordinates
   ObjectSetInteger(chart_ID,name,OBJPROP_XDISTANCE,x);
   ObjectSetInteger(chart_ID,name,OBJPROP_YDISTANCE,y);
//--- set chart corner relative to which point coordinates will be determined
   ObjectSetInteger(chart_ID,name,OBJPROP_CORNER,corner);
//--- set text
   ObjectSetString(chart_ID,name,OBJPROP_TEXT,text);
//--- set text font
   ObjectSetString(chart_ID,name,OBJPROP_FONT,font);
//--- set font size
   ObjectSetInteger(chart_ID,name,OBJPROP_FONTSIZE,font_size);
//--- set text angle
   ObjectSetDouble(chart_ID,name,OBJPROP_ANGLE,angle);
//--- set anchor method
   ObjectSetInteger(chart_ID,name,OBJPROP_ANCHOR,anchor);
//--- set color
   ObjectSetInteger(chart_ID,name,OBJPROP_COLOR,clr);
//--- display on foreground (false) or background (true)
   ObjectSetInteger(chart_ID,name,OBJPROP_BACK,back);
//--- enable (true) or disable (false) label moving mode with mouse
   ObjectSetInteger(chart_ID,name,OBJPROP_SELECTABLE,selection);
   ObjectSetInteger(chart_ID,name,OBJPROP_SELECTED,selection);
//--- hide (true) or show (false) graphical object name in object list
   ObjectSetInteger(chart_ID,name,OBJPROP_HIDDEN,hidden);
//--- set priority for receiving mouse click event on chart
   ObjectSetInteger(chart_ID,name,OBJPROP_ZORDER,z_order);
//--- successful execution
   return(true);
  }

//+-----------------------------------------------------------------+
//| Array of user interface object names                            |
//| Contains unique names of all interface graphical elements       |
//| Used for creating and managing elements on chart                |
//+-----------------------------------------------------------------+
string OwnObjectNames[] = {
"AlgoWealthX-BorderPanel",      // rectangular interface border frame (RectLabel)
"AlgoWealthX-MainPanel",        // main interface panel (RectLabel)
"AlgoWealthX-EAName",           // text label with expert name (Label)
"AlgoWealthX-Balance",          // text label with account balance (Label)
"AlgoWealthX-Equity",           // text label with account equity (Label)
"AlgoWealthX-Leverage",         // text label with leverage (Label)
"AlgoWealthX-Broker",           // text label with broker name (Label)
"AlgoWealthX-DaysRemaining",    // text label with remaining work days (Label)
"AlgoWealthX-StartDate",        // text label with start date (Label)
"AlgoWealthX-EndDate",          // text label with end date (Label)
"AlgoWealthX-Magic"             // text label with expert magic number (Label)
};

//+--------------------------------------------------------------+
//| Calculation of days since trading start                      |
//| Calculates number of days passed since DatetimeStart         |
//| Returns: number of days as a fractional number               |
//+--------------------------------------------------------------+
double Days()
   {
   datetime ClosestTime=0;                  // nearest time for calculation (initialization with zero)
   
   // Determine start time - use robot start time if it is greater than zero
   if (BotO.DatetimeStart > ClosestTime) ClosestTime = BotO.DatetimeStart;
         
   // Calculate difference between current time and start time
   if ( double(ChartO.TimeI[0] - ClosestTime) > 0 ) return double(ChartO.TimeI[0] - ClosestTime)/86400.0; // return number of days (86400 seconds in a day)
   else return 0.0;                         // if time has not passed or is incorrect, return 0
   }

//+--------------------------------------------------------------+
//| Variables for status update                                  |
//+--------------------------------------------------------------+
datetime LastRead;                          // time of last settings reading from file

//+--------------------------------------------------------------+
//| User interface state update                                  |
//| Displays current account information and robot state         |
//| Called on every tick to actualize data                       |
//+--------------------------------------------------------------+
void UpdateStatus()
   {
   string TempText="EA Name : ";            // form string with expert name
   TempText+=EANICK;                        // add robot name from constant
   ObjectSetString(0,OwnObjectNames[2],OBJPROP_TEXT,TempText); // display name in interface
   
   TempText="Balance : ";                   // form string with balance
   TempText+=DoubleToString(NormalizeDouble(AccountInfoDouble(ACCOUNT_BALANCE),2),2); // add current balance
   ObjectSetString(0,OwnObjectNames[3],OBJPROP_TEXT,TempText); // display balance in interface
   
   TempText="Equity : ";                    // form string with equity
   TempText+=DoubleToString(NormalizeDouble(AccountInfoDouble(ACCOUNT_EQUITY),2),2); // add current equity
   ObjectSetString(0,OwnObjectNames[4],OBJPROP_TEXT,TempText); // display equity in interface
   
   TempText="Leverage : 1/";                // form string with leverage
   TempText+=DoubleToString(AccountInfoInteger(ACCOUNT_LEVERAGE),0); // add leverage size
   ObjectSetString(0,OwnObjectNames[5],OBJPROP_TEXT,TempText); // display leverage in interface
   
   TempText="Broker : ";                    // form string with broker name
   TempText+=AccountInfoString(ACCOUNT_COMPANY); // add broker company name
   ObjectSetString(0,OwnObjectNames[6],OBJPROP_TEXT,TempText); // display broker in interface
   
   // Display remaining robot work days
   TempText="Days remaining : ";            // form string with remaining days
   if ( double(DaysToFuture)-Days() > 0 ) TempText+=DoubleToString(NormalizeDouble(double(DaysToFuture)-Days(),2),2); // if days remained
   else TempText+="0, trading disabled!";  // if time expired, trading disabled
   ObjectSetString(0,OwnObjectNames[7],OBJPROP_TEXT,TempText); // display remaining days in interface
   TempText="Optimization finished : ";    // form string with optimization finish date
   MqlDateTime StartDate;                   // structure for parsing compilation date
   TimeToStruct(__DATETIME__,StartDate);    // convert compilation date to structure
   TempText+=IntegerToString(StartDate.day)+"."+IntegerToString(StartDate.mon)+"."+IntegerToString(StartDate.year); // add date
   ObjectSetString(0,OwnObjectNames[8],OBJPROP_TEXT,TempText); // display optimization date in interface
   
   TempText="Will trade until : ";          // form string with trading end date
   MqlDateTime EndDate;                     // structure for parsing end date
   TimeToStruct(__DATETIME__+datetime(DaysToFuture*86400),EndDate); // calculate end date
   TempText+=IntegerToString(EndDate.day)+"."+IntegerToString(EndDate.mon)+"."+IntegerToString(EndDate.year); // add date
   ObjectSetString(0,OwnObjectNames[9],OBJPROP_TEXT,TempText); // display end date in interface
   
   TempText="Magic : ";                     // form string with magic number
   TempText+=IntegerToString(MagicE);       // add magic number value
   ObjectSetString(0,OwnObjectNames[10],OBJPROP_TEXT,TempText); // display magic number in interface
   }

//+--------------------------------------------------------------+
//| Calculation of corrected font size for different DPI         |
//| Parameters:                                                  |
//| BasicFont - base font size                                   |
//| Returns: corrected font size (6-72)                          |
//+--------------------------------------------------------------+
int CalcFontCorrected(int BasicFont)
   {
   int EtalonDPI=100;                       // reference DPI value for calculation
   double DPI = TerminalInfoInteger(TERMINAL_SCREEN_DPI); // get current screen DPI
   int FirstCorrection = int(MathFloor(double(EtalonDPI)/DPI * BasicFont)); // calculate corrected size
   
   if (FirstCorrection < 6)                 // if size is less than minimum
      {
         return 6;                          // return minimum font size
      }
   else if (FirstCorrection > 72)           // if size is greater than maximum
      {
         return 72;                         // return maximum font size
      }
   else
      {
      return FirstCorrection;               // return corrected size
      }      
   }

//+--------------------------------------------------------------+
//| Creation of simple information interface                     |
//| Creates rectangular frame and text labels with information   |
//| about account state and robot parameters                     |
//+--------------------------------------------------------------+
void CreateSimpleInterface()
   {
   ENUM_BASE_CORNER corner=CORNER_LEFT_UPPER; // anchor all elements to upper left corner
   int x=5;                                 // X axis offset
   int y=15;                                // Y axis offset
   int Width = 280;                         // interface width
   int Height = 195;                        // interface height
   int Border = 5;                          // border size
   
   // Creation of interface graphical elements
   RectLabelCreate(0,OwnObjectNames[0],0,x,y,Width+2*Border,Height+2*Border,clrLightCyan,BORDER_RAISED,corner,clrOrchid,STYLE_SOLID,3,false,false,true,0); // outer frame
   RectLabelCreate(0,OwnObjectNames[1],0,x+Border,y+Border,Width,Height,clrDarkSlateBlue,BORDER_RAISED,corner,clrOrchid,STYLE_SOLID,3,false,false,true,0); // main panel
   
   // Creation of text elements with information
   LabelCreate(0,OwnObjectNames[2],0,x+Border+2,y+8+Border,corner,"EA name : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0); // expert name
   LabelCreate(0,OwnObjectNames[3],0,x+Border+2,y+17+Border+20*1,corner,"Balance : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT); // balance
   LabelCreate(0,OwnObjectNames[4],0,x+Border+2,y+17+Border+20*2,corner,"Equity : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT); // equity
   LabelCreate(0,OwnObjectNames[5],0,x+Border+2,y+17+Border+20*3,corner,"Leverage : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT); // leverage
   LabelCreate(0,OwnObjectNames[6],0,x+Border+2,y+17+Border+20*4,corner,"Broker : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT); // broker
   LabelCreate(0,OwnObjectNames[7],0,x+Border+2,y+17+Border+20*5,corner,"DaysRemaining : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT); // remaining days          
   LabelCreate(0,OwnObjectNames[8],0,x+Border+2,y+17+Border+20*6,corner,"StartDate : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT); // start date
   LabelCreate(0,OwnObjectNames[9],0,x+Border+2,y+17+Border+20*7,corner,"EndDate : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT); // end date
   LabelCreate(0,OwnObjectNames[10],0,x+Border+2,y+17+Border+20*8,corner,"Magic : ","Arial",CalcFontCorrected(11),clrLightCyan,0.0,ANCHOR_LEFT); // magic number
   }

//+--------------------------------------------------------------+
//| Deletion of all user interface elements                      |
//| Clears chart of all created graphical objects                |
//+--------------------------------------------------------------+
void DeleteSimpleInterface()
   {
   ObjectsDeleteAll(0,"AlgoWealthX");       // remove all objects with prefix "AlgoWealthX"
   }

