
import pandas as pd
import wbdata
import MetaTrader5 as mt5
from catboost import CatBoostRegressor
from sklearn.model_selection import train_test_split
from sklearn.metrics import mean_squared_error
import warnings
import matplotlib.pyplot as plt
import seaborn as sns
from datetime import datetime, timedelta

# Отключение предупреждений
warnings.filterwarnings("ignore", category=UserWarning, module="wbdata")

# Загрузка данных с Всемирного банка
indicators = {
    'NY.GDP.MKTP.KD.ZG': 'GDP growth',  # ВВП рост
    'FP.CPI.TOTL.ZG': 'Inflation',       # Инфляция
    'FR.INR.RINR': 'Real interest rate', # Реальная процентная ставка
    'NE.EXP.GNFS.ZS': 'Exports',         # Экспорт товаров и услуг (% от ВВП)
    'NE.IMP.GNFS.ZS': 'Imports',         # Импорт товаров и услуг (% от ВВП)
    'BN.CAB.XOKA.GD.ZS': 'Current account balance', # Баланс текущего счета (% от ВВП)
    'GC.DOD.TOTL.GD.ZS': 'Government debt', # Государственный долг (% от ВВП)
    'SL.UEM.TOTL.ZS': 'Unemployment rate', # Уровень безработицы (% от трудоспособного населения)
    'NY.GNP.PCAP.CD': 'GNI per capita',   # ВВП на душу населения (текущие доллары США)
    'NY.GDP.PCAP.KD.ZG': 'GDP per capita growth', # Рост ВВП на душу населения (постоянные доллары США 2010 года)
    'NE.RSB.GNFS.ZS': 'Reserves in months of imports', # Резервы в месяцах импорта
    'NY.GDP.DEFL.KD.ZG': 'GDP deflator', # Дефлятор ВВП (постоянные доллары США 2010 года)
    'NY.GDP.PCAP.KD': 'GDP per capita (constant 2015 US$)', # ВВП на душу населения (постоянные доллары США 2015 года)
    'NY.GDP.PCAP.PP.CD': 'GDP per capita, PPP (current international $)', # ВВП на душу населения, ППС (текущие международные доллары)
    'NY.GDP.PCAP.PP.KD': 'GDP per capita, PPP (constant 2017 international $)', # ВВП на душу населения, ППС (постоянные международные доллары 2017 года)
    'NY.GDP.PCAP.CN': 'GDP per capita (current LCU)', # ВВП на душу населения (текущие национальные валюты)
    'NY.GDP.PCAP.KN': 'GDP per capita (constant LCU)', # ВВП на душу населения (постоянные национальные валюты)
    'NY.GDP.PCAP.CD': 'GDP per capita (current US$)', # ВВП на душу населения (текущие доллары США)
    'NY.GDP.PCAP.KD': 'GDP per capita (constant 2010 US$)', # ВВП на душу населения (постоянные доллары США 2010 года)
    'NY.GDP.PCAP.KD.ZG': 'GDP per capita growth (annual %)', # Рост ВВП на душу населения (годовой %)
    'NY.GDP.PCAP.KN.ZG': 'GDP per capita growth (constant LCU)', # Рост ВВП на душу населения (постоянные национальные валюты)
}

# Получение данных для каждого индикатора отдельно
data_frames = []
for indicator, name in indicators.items():
    try:
        data_frame = wbdata.get_dataframe({indicator: name}, country='all')
        data_frames.append(data_frame)
    except Exception as e:
        print(f"Error fetching data for indicator '{indicator}': {e}")

# Объединение данных в один DataFrame
if data_frames:
    data = pd.concat(data_frames, axis=1)
    
    # Вывод информации о доступных индикаторах и их данных
    print("Available indicators and their data:")
    print(data.columns)
    print(data.head())

    # Сохранение данных в CSV
    data.to_csv('economic_data.csv', index=True)

    # Вывод статистики
    print("Economic Data Statistics:")
    print(data.describe())
else:
    print("No data was fetched. Please check your internet connection and the availability of the World Bank API.")
    data = pd.DataFrame()  # Create an empty DataFrame to avoid errors in the rest of the script

# Загрузка данных с MetaTrader5
if not mt5.initialize():
    print("initialize() failed")
    mt5.shutdown()

# Получение всех валютных пар
symbols = mt5.symbols_get()
symbol_names = [symbol.name for symbol in symbols]

# Загрузка исторических данных для каждой валютной пары
historical_data = {}
for symbol in symbol_names:
    rates = mt5.copy_rates_from_pos(symbol, mt5.TIMEFRAME_D1, 0, 1000)
    df = pd.DataFrame(rates)
    df['time'] = pd.to_datetime(df['time'], unit='s')
    df.set_index('time', inplace=True)
    historical_data[symbol] = df

# Подготовка данных для прогнозирования
def prepare_data(symbol_data, economic_data):
    data = symbol_data.copy()
    data['close_diff'] = data['close'].diff()
    data['close_corr'] = data['close'].rolling(window=30).corr(data['close'].shift(1))

    for indicator in indicators.keys():
        if indicator in economic_data.columns:
            # Используем метод forward fill для заполнения пропущенных значений
            data[indicator] = economic_data[indicator].ffill()
        else:
            print(f"Warning: Data for indicator '{indicator}' is not available.")

    data.dropna(inplace=True)
    return data

# Подготовка данных для всех валютных пар
prepared_data = {}
for symbol, df in historical_data.items():
    prepared_data[symbol] = prepare_data(df, data)

# Обновленная функция forecast
def forecast(symbol, symbol_data):
    if len(symbol_data) < 50:  # Проверяем, достаточно ли данных для прогнозирования
        print(f"Not enough data for {symbol}. Skipping forecast.")
        return None, None

    X = symbol_data.drop(columns=['close'])
    y = symbol_data['close']

    X_train, X_test, y_train, y_test = train_test_split(X, y, test_size=0.2, shuffle=False)

    if len(X_train) == 0 or len(X_test) == 0:
        print(f"Not enough data for {symbol} after splitting. Skipping forecast.")
        return None, None

    model = CatBoostRegressor(iterations=1000, learning_rate=0.1, depth=8, loss_function='RMSE', verbose=100)
    model.fit(X_train, y_train, verbose=False)

    predictions = model.predict(X_test)
    mse = mean_squared_error(y_test, predictions)
    print(f"Mean Squared Error for {symbol}: {mse}")

    # Анализ важности признаков
    feature_importance = model.feature_importances_
    feature_names = X.columns
    importance_df = pd.DataFrame({'feature': feature_names, 'importance': feature_importance})
    importance_df = importance_df.sort_values('importance', ascending=False)
    print(f"Top 10 Feature Importance for {symbol}:")
    print(importance_df.head(10))

    # Прогнозирование на месяц вперед
    future_data = symbol_data.tail(30).copy()
    future_predictions = model.predict(future_data.drop(columns=['close']))

    return future_predictions, importance_df

# Прогнозирование для всех валютных пар
forecasts = {}
feature_importances = {}
for symbol, df in prepared_data.items():
    try:
        forecast_result, importance_df = forecast(symbol, df)
        if forecast_result is not None and importance_df is not None:
            forecasts[symbol] = forecast_result
            feature_importances[symbol] = importance_df
    except Exception as e:
        print(f"Error forecasting for {symbol}: {e}")

# Визуализация прогнозов
def visualize_forecast(symbol, forecast):
    plt.figure(figsize=(12, 6))
    plt.plot(range(len(forecast)), forecast, label='Forecast')
    plt.title(f'Forecast for {symbol}')
    plt.xlabel('Days')
    plt.ylabel('Price')
    plt.legend()
    plt.savefig(f'{symbol}_forecast.png')
    plt.close()

# Визуализация важности признаков
def visualize_feature_importance(symbol, importance_df):
    if importance_df is None or importance_df.empty:
        print(f"No feature importance data available for {symbol}")
        return
    
    plt.figure(figsize=(12, 6))
    sns.barplot(x='importance', y='feature', data=importance_df.head(10))
    plt.title(f'Top 10 Feature Importance for {symbol}')
    plt.xlabel('Importance')
    plt.ylabel('Feature')
    plt.tight_layout()
    plt.savefig(f'{symbol}_feature_importance.png')
    plt.close()

# Создание визуализаций
for symbol, forecast in forecasts.items():
    visualize_forecast(symbol, forecast)
    if symbol in feature_importances:
        visualize_feature_importance(symbol, feature_importances[symbol])

# Интерпретация результатов
def interpret_results(symbol, forecast, importance_df):
    if forecast is None or importance_df is None:
        return f"Insufficient data for interpretation of {symbol}"

    trend = "upward" if forecast[-1] > forecast[0] else "downward"
    volatility = "high" if forecast.std() / forecast.mean() > 0.1 else "low"
    top_feature = importance_df.iloc[0]['feature']
    
    interpretation = f"""
    Interpretation for {symbol}:
    1. Price Trend: The forecast shows a {trend} trend for the next 30 days.
    2. Volatility: The predicted price movement shows {volatility} volatility.
    3. Key Influencing Factor: The most important feature for this forecast is '{top_feature}'.
    4. Economic Implications:
       - If GDP growth is a top factor, it suggests strong economic performance is influencing the currency.
       - High importance of inflation rate might indicate monetary policy changes are affecting the currency.
       - If trade balance factors are crucial, international trade dynamics are likely driving currency movements.
    5. Trading Implications:
       - {trend.capitalize()} trend suggests potential for {'long' if trend == 'upward' else 'short'} positions.
       - {'Consider using tight stop-losses due to high volatility.' if volatility == 'high' else 'Lower volatility might allow for wider stop-losses.'}
    6. Risk Assessment:
       - Always consider the model's limitations and potential for unexpected market events.
       - Past performance doesn't guarantee future results.
    """
    return interpretation

# Generate and print interpretations
for symbol in forecasts.keys():
    interpretation = interpret_results(symbol, forecasts[symbol], feature_importances.get(symbol))
    print(interpretation)
    
    # Save interpretation to a text file
    with open(f'{symbol}_interpretation.txt', 'w') as f:
        f.write(interpretation)

# Закрытие соединения с MetaTrader5
mt5.shutdown()

print("Analysis complete. Check the generated PNG files for visualizations and TXT files for interpretations.")
