
import pandas as pd
import wbdata
import MetaTrader5 as mt5
from catboost import CatBoostRegressor
from sklearn.model_selection import train_test_split
from sklearn.metrics import mean_squared_error
import warnings

# Отключение предупреждений
warnings.filterwarnings("ignore", category=UserWarning, module="wbdata")

# Загрузка данных с Всемирного банка
indicators = {
    'NY.GDP.MKTP.KD.ZG': 'GDP growth',  # ВВП рост
    'FP.CPI.TOTL.ZG': 'Inflation',       # Инфляция
    'FR.INR.RINR': 'Real interest rate', # Реальная процентная ставка
    'NE.EXP.GNFS.ZS': 'Exports',         # Экспорт товаров и услуг (% от ВВП)
    'NE.IMP.GNFS.ZS': 'Imports',         # Импорт товаров и услуг (% от ВВП)
    'BN.CAB.XOKA.GD.ZS': 'Current account balance', # Баланс текущего счета (% от ВВП)
    'GC.DOD.TOTL.GD.ZS': 'Government debt', # Государственный долг (% от ВВП)
    'SL.UEM.TOTL.ZS': 'Unemployment rate', # Уровень безработицы (% от трудоспособного населения)
    'NY.GNP.PCAP.CD': 'GNI per capita',   # ВВП на душу населения (текущие доллары США)
    'NY.GDP.PCAP.KD.ZG': 'GDP per capita growth', # Рост ВВП на душу населения (постоянные доллары США 2010 года)
    'NE.RSB.GNFS.ZS': 'Reserves in months of imports', # Резервы в месяцах импорта
    'NY.GDP.DEFL.KD.ZG': 'GDP deflator', # Дефлятор ВВП (постоянные доллары США 2010 года)
    'NY.GDP.PCAP.KD': 'GDP per capita (constant 2015 US$)', # ВВП на душу населения (постоянные доллары США 2015 года)
    'NY.GDP.PCAP.PP.CD': 'GDP per capita, PPP (current international $)', # ВВП на душу населения, ППС (текущие международные доллары)
    'NY.GDP.PCAP.PP.KD': 'GDP per capita, PPP (constant 2017 international $)', # ВВП на душу населения, ППС (постоянные международные доллары 2017 года)
    'NY.GDP.PCAP.CN': 'GDP per capita (current LCU)', # ВВП на душу населения (текущие национальные валюты)
    'NY.GDP.PCAP.KN': 'GDP per capita (constant LCU)', # ВВП на душу населения (постоянные национальные валюты)
    'NY.GDP.PCAP.CD': 'GDP per capita (current US$)', # ВВП на душу населения (текущие доллары США)
    'NY.GDP.PCAP.KD': 'GDP per capita (constant 2010 US$)', # ВВП на душу населения (постоянные доллары США 2010 года)
    'NY.GDP.PCAP.KD.ZG': 'GDP per capita growth (annual %)', # Рост ВВП на душу населения (годовой %)
    'NY.GDP.PCAP.KN.ZG': 'GDP per capita growth (constant LCU)', # Рост ВВП на душу населения (постоянные национальные валюты)
}

# Получение данных для каждого индикатора отдельно
data_frames = []
for indicator in indicators.keys():
    try:
        data_frame = wbdata.get_dataframe({indicator: indicators[indicator]}, country='all')
        data_frames.append(data_frame)
    except Exception as e:
        print(f"Error fetching data for indicator '{indicator}': {e}")

# Объединение данных в один DataFrame
data = pd.concat(data_frames, axis=1)

# Вывод информации о доступных индикаторах и их данных
print("Available indicators and their data:")
print(data.columns)
print(data.head())

# Сохранение данных в CSV
data.to_csv('economic_data.csv', index=True)

# Вывод статистики
print("Economic Data Statistics:")
print(data.describe())

# Загрузка данных с MetaTrader5
if not mt5.initialize():
    print("initialize() failed")
    mt5.shutdown()

# Получение всех валютных пар
symbols = mt5.symbols_get()
symbol_names = [symbol.name for symbol in symbols]

# Загрузка исторических данных для каждой валютной пары
historical_data = {}
for symbol in symbol_names:
    rates = mt5.copy_rates_from_pos(symbol, mt5.TIMEFRAME_D1, 0, 1000)
    df = pd.DataFrame(rates)
    df['time'] = pd.to_datetime(df['time'], unit='s')
    df.set_index('time', inplace=True)
    historical_data[symbol] = df

# Подготовка данных для прогнозирования
def prepare_data(symbol_data, economic_data):
    data = symbol_data.copy()
    data['close_diff'] = data['close'].diff()
    data['close_corr'] = data['close'].rolling(window=30).corr(data['close'].shift(1))

    for indicator in indicators.keys():
        if indicator in economic_data.columns:
            data[indicator] = economic_data[indicator]
        else:
            print(f"Warning: Data for indicator '{indicator}' is not available.")

    data.dropna(inplace=True)
    return data

# Подготовка данных для всех валютных пар
prepared_data = {}
for symbol, df in historical_data.items():
    prepared_data[symbol] = prepare_data(df, data)

# Прогнозирование с помощью CatBoost
def forecast(symbol_data):
    X = symbol_data.drop(columns=['close'])
    y = symbol_data['close']

    X_train, X_test, y_train, y_test = train_test_split(X, y, test_size=0.2, shuffle=False)

    model = CatBoostRegressor(iterations=1000, learning_rate=0.1, depth=8, loss_function='RMSE', verbose=100)
    model.fit(X_train, y_train, verbose=False)

    predictions = model.predict(X_test)
    mse = mean_squared_error(y_test, predictions)
    print(f"Mean Squared Error for {symbol}: {mse}")

    # Прогнозирование на месяц вперед
    future_data = symbol_data.tail(30).copy()
    if len(predictions) >= 30:
        future_data['close'] = predictions[-30:]
    else:
        future_data['close'] = predictions

    future_predictions = model.predict(future_data.drop(columns=['close']))

    return future_predictions

# Прогнозирование для всех валютных пар
forecasts = {}
for symbol, df in prepared_data.items():
    try:
        forecasts[symbol] = forecast(df)
    except Exception as e:
        print(f"Error forecasting for {symbol}: {e}")

# Закрытие соединения с MetaTrader5
mt5.shutdown()

# Вывод прогнозов
for symbol, forecast in forecasts.items():
    print(f"Forecast for {symbol}: {forecast}")

# Сохранение прогнозов в CSV
forecasts_df = pd.DataFrame(forecasts)
forecasts_df.to_csv('forecasts.csv', index=True)
