//+------------------------------------------------------------------+
//|                                                  UK100 Gilts.mq5 |
//|                                        Gamuchirai Zororo Ndawana |
//|                          https://www.mql5.com/en/gamuchiraindawa |
//+------------------------------------------------------------------+
#property copyright "Gamuchirai Zororo Ndawana"
#property link      "https://www.mql5.com/en/gamuchiraindawa"
#property version   "1.00"

//+------------------------------------------------------------------+
//|Libraries we need                                                 |
//+------------------------------------------------------------------+
#include <Trade/Trade.mqh>             //Trade class
CTrade Trade;                          //Initialize the class

//+------------------------------------------------------------------+
//| Inputs                                                           |
//+------------------------------------------------------------------+
input int fetch = 20;                  //How much data should we fetch?
input int look_ahead = 20;             //How far into the future should we forecast?

//+------------------------------------------------------------------+
//| Global vairables                                                 |
//+------------------------------------------------------------------+
matrix coefficients = matrix::Zeros(1,9);
vector mean_values = vector::Zeros(8);
vector std_values = vector::Zeros(8);
vector intercept = vector::Ones(fetch);
matrix input_matrix = matrix::Zeros(9,fetch);
matrix gilts_data,uk100_data,target;
double willr_buffer[],rsi_buffer[];
int willr_handler,rsi_handler;
double forecast,bid,ask;
int model_forecast = 0;
int state = 0;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//--- Fetch the training data
   fetch_training_data();

//--- Scale the training data
   scale_training_data();

//--- Calculate the coefficients
   calculate_coefficient_values();

//--- Setup the indicators
   rsi_handler = iRSI("UK100",PERIOD_CURRENT,20,PRICE_CLOSE);
   willr_handler = iWPR("UK100",PERIOD_CURRENT,30);

//--- Validate the technical indicators
   if((rsi_handler == INVALID_HANDLE) || (willr_handler == INVALID_HANDLE))
     {
      Comment("Failed to load indicators. ",GetLastError());
      return(INIT_FAILED);
     }

//--- Everything wen't well
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//--- Free up the resoruces we don't need
   IndicatorRelease(willr_handler);
   IndicatorRelease(rsi_handler);
   ExpertRemove();

  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//--- Fetch updated market data
   update_market_data();

//--- Fetch a forecast from our model
   fetch_forecast();

//--- Check for a position we can open
   if(PositionsTotal() == 0)
     {
      if(model_forecast == 1)
        {
         check_bullish_sentiment();
        }

      else
         if(model_forecast == -1)
           {
            check_bearish_sentiment();
           }
     }

//--- Check for a reversal
   else
      if(PositionsTotal() > 0)
        {
         if(model_forecast != state)
           {
            Alert("Reversal detected by our AI system! Closing all positions now.");
            Trade.PositionClose("UK100");
           }
        }
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| This function will check if we have oppurtunities to buy         |
//+------------------------------------------------------------------+
void check_bullish_sentiment(void)
  {
   if((willr_buffer[0] > -20) && (rsi_buffer[0] > 70))
     {
      Trade.Buy(0.2,"UK100",ask,ask-5,ask+5,"UK100 Gilts AI");
      state = 1;
     }
  }

//+------------------------------------------------------------------+
//| This function will check if we have oppurtunities to sell        |
//+------------------------------------------------------------------+
void check_bearish_sentiment(void)
  {
   if((willr_buffer[0] < -80) && (rsi_buffer[0] <370))
     {
      Trade.Sell(0.2,"UK100",ask,ask-5,ask+5,"UK100 Gilts AI");
      state = -1;
     }
  }


//+------------------------------------------------------------------+
//| This function will fetch current market data                     |
//+------------------------------------------------------------------+
void update_market_data(void)
  {
//--- Market prices
   bid = SymbolInfoDouble("UK100",SYMBOL_BID);
   ask = SymbolInfoDouble("UK100",SYMBOL_ASK);

//--- Technical indicators
   CopyBuffer(rsi_handler,0,0,1,rsi_buffer);
   CopyBuffer(willr_handler,0,0,1,willr_buffer);
  }

//+------------------------------------------------------------------+
//| Fetch a forecast from our model                                  |
//+------------------------------------------------------------------+
void fetch_forecast(void)
  {
//--- Now we can obtain a forecast from our model
   gilts_data.CopyRates("UKGB_Z4",PERIOD_CURRENT,COPY_RATES_OHLC,0,1);
   uk100_data.CopyRates("UK100",PERIOD_CURRENT,COPY_RATES_OHLC,0,1);

//--- Scale our inputs
   gilts_data[0,0] = ((gilts_data[0,0] - mean_values[1]) / std_values[1]);
   gilts_data[1,0] = ((gilts_data[1,0] - mean_values[2]) / std_values[2]);
   gilts_data[2,0] = ((gilts_data[2,0] - mean_values[3]) / std_values[3]);
   gilts_data[3,0] = ((gilts_data[3,0] - mean_values[4]) / std_values[4]);
   uk100_data[0,0] = ((uk100_data[0,0] - mean_values[5]) / std_values[5]);
   uk100_data[1,0] = ((uk100_data[1,0] - mean_values[6]) / std_values[6]);
   uk100_data[2,0] = ((uk100_data[2,0] - mean_values[7]) / std_values[7]);
   uk100_data[3,0] = ((uk100_data[3,0] - mean_values[8]) / std_values[8]);

   Print("Normalized inputs: ");
   Print(gilts_data);
   Print(uk100_data);

//--- Calculate the model's prediction
   forecast = (
                 (1 * coefficients[0,0]) +
                 (gilts_data[0,0] * coefficients[0,1]) +
                 (gilts_data[1,0] * coefficients[0,2]) +
                 (gilts_data[2,0] * coefficients[0,3]) +
                 (gilts_data[3,0] * coefficients[0,4]) +
                 (uk100_data[0,0] * coefficients[0,5]) +
                 (gilts_data[1,0] * coefficients[0,6]) +
                 (gilts_data[2,0] * coefficients[0,7]) +
                 (gilts_data[3,0] * coefficients[0,8])
              );

//--- Store the model's prediction
   if(forecast < iClose("UK100",PERIOD_CURRENT,0))
     {
      model_forecast = -1;
     }

   if(forecast > iClose("UK100",PERIOD_CURRENT,0))
     {
      model_forecast = 1;
     }

//--- Give the user feedback
   Comment("Model forecast: ",forecast);
  }

//+------------------------------------------------------------------+
//| Calculate coefficient values                                     |
//+------------------------------------------------------------------+
void calculate_coefficient_values(void)
  {
//--- Now we can calculate our coefficient values
   coefficients = target.MatMul(input_matrix.PInv());

   Print("Coefficient values");
   Print(coefficients);
  }

//+------------------------------------------------------------------+
//| Let us scale and standardize the training data                   |
//+------------------------------------------------------------------+
void scale_training_data(void)
  {
//--- Normalizing and scaling our input data
   for(int i = 0; i < 8; i++)
     {
      //--- Extract the vector
      vector temp = input_matrix.Row(i + 1);
      //--- Scale the data
      temp = ((temp - mean_values[i+1]) / std_values[i+1]);
      //--- Write the data back
      input_matrix.Row(temp,i+1);
     }

//--- Finished normalizing the data
   Print("Finished normalizing the data.");
   Print(input_matrix);
  }

//+------------------------------------------------------------------+
//| Let us fetch our training data                                   |
//+------------------------------------------------------------------+
void fetch_training_data(void)
  {
//--- First we will fetch the market data
   gilts_data.CopyRates("UKGB_Z4",PERIOD_CURRENT,COPY_RATES_OHLC,1+look_ahead,fetch);
   uk100_data.CopyRates("UK100",PERIOD_CURRENT,COPY_RATES_OHLC,1+look_ahead,fetch);
   target.CopyRates("UK100",PERIOD_CURRENT,COPY_RATES_CLOSE,1,fetch);

//--- Fill in the input matrix
   input_matrix.Row(intercept,0);
   input_matrix.Row(gilts_data.Row(0),1);
   input_matrix.Row(gilts_data.Row(1),2);
   input_matrix.Row(gilts_data.Row(2),3);
   input_matrix.Row(gilts_data.Row(3),4);
   input_matrix.Row(uk100_data.Row(0),5);
   input_matrix.Row(uk100_data.Row(1),6);
   input_matrix.Row(uk100_data.Row(2),7);
   input_matrix.Row(uk100_data.Row(3),8);

//--- Display the data fetched
   Print("Input matrix: ");
   Print("Rows: ",input_matrix.Rows()," Columns: ",input_matrix.Cols());
   Print(input_matrix);

   Print("Target: ");
   Print("Rows: ",target.Rows()," Columns: ",target.Cols());
   Print(target);

   Print("UK100: ");
   Print("Rows: ",uk100_data.Rows()," Columns: ",uk100_data.Cols());
   Print(uk100_data);

   Print("GILTS: ");
   Print("Rows: ",gilts_data.Rows()," Columns: ",gilts_data.Cols());
   Print(gilts_data);

//--- Calculate the scaling values
   mean_values = input_matrix.Mean(1);
   std_values = input_matrix.Std(1);

   Print("Mean values: ");
   Print(mean_values);

   Print("Std values: ");
   Print(std_values);
  }
//+------------------------------------------------------------------+
