//+------------------------------------------------------------------+
//|                                                       Filter.mqh |
//|                                  Copyright 2023, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#include<Math\Alglib\fasttransforms.mqh>
//+------------------------------------------------------------------+
//|  CFilter - class implementing select filters in the freq domain  |
//+------------------------------------------------------------------+
class CFilter
  {
   int               m_length;                   //length of original series
   int               m_padded_len;                //modified length of series
   int               m_half_padded_len;           //half of modded series
   double            m_slope, m_intercept;     //slope and intercept of trend in series
   double            m_buffer[];               //general internal buffer
   bool              m_initialized;              //initilization flag
   complex           m_dft[];                 //general complex buffer

public:
                     CFilter(double &series[],uint min_padding=0, bool detrend=true);
                    ~CFilter(void);

   void              Lowpass(double freq,double width,double &out[],bool add_trend);
   void              Highpass(double freq,double width, double &out[]);
   void              Bandpass(double freq,double width, double &out[]);
   void              Qmf(double freq,double width, complex &out[]);
   bool              IsInitialized(void) { return m_initialized; }
  };
//+------------------------------------------------------------------+
//|constructor                                                       |
//+------------------------------------------------------------------+
CFilter::CFilter(double &series[],uint min_padding=0,bool detrend=true)
  {
//---local variables
   m_initialized=false;
   int i;
   int npts = ArraySize(series);
   int pad = (int)MathAbs(min_padding);
//--- check size of series
   if(npts<=0)
     {
      Print("Input array is empty");
      return ;
     }
//---
   m_length = npts ;
   for(m_padded_len=2 ; m_padded_len<INT_MAX ; m_padded_len*=2)
     {
      if(m_padded_len >= npts+pad)
         break ;
     }
//---
   if(m_padded_len<npts+pad)
     {
      Print("Warning, calculated length of modified series is too long");
      return;
     }
//---
   m_half_padded_len = m_padded_len / 2;
//---
   ArrayResize(m_buffer,m_padded_len);
//---
   if(m_padded_len > npts)            // Any padding needed?
     {

      if(detrend)
        {
         m_intercept = series[0] ;
         m_slope = (series[npts-1] - series[0]) / (npts-1) ;
        }
      else
        {
         m_intercept = m_slope = 0.0 ;
         for(i=0 ; i<npts ; i++)
            m_intercept += series[i] ;
         m_intercept /= npts ;
        }

      for(i=0 ; i<npts ; i++)
        {
         m_buffer[i]=series[i] - m_intercept - m_slope * i ;
        }
      for(i=npts ; i<m_padded_len ; i++)
        {
         m_buffer[i]=0.0;
        }
     }

   else
     {
      ArrayCopy(m_buffer,series);
      m_intercept = m_slope = 0.0 ;
     }
//---Compute the Fourier transform of the padded serie
   CFastFourierTransform::FFTR1D(m_buffer,int(m_padded_len),m_dft);
//---
   m_initialized = true;

  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CFilter::~CFilter(void)
  {
   ArrayFree(m_buffer);
//---
   ArrayFree(m_dft);
  }
//+--------------------------------------------------------------------+
//|Filters series in frequency domain and returns output in time domain|
//+--------------------------------------------------------------------+
void CFilter::Lowpass(double freq,double width,double &out[],bool add_trend)
  {
//---
   int i ;
   double f, dist, wt ;
   complex dft_temp[];
   ArrayCopy(dft_temp,m_dft);
//---
   for(i=0 ; i<=m_half_padded_len ; i++)
     {
      f = (double) i / (double) m_padded_len ;  // This frequency
      if(f <= freq)                  // Flat to here
         wt = 1.0 ;
      else
        {
         dist = (f - freq) / width ;
         wt = exp(-dist * dist) ;
        }
      dft_temp[i].real*=wt;
      dft_temp[i].imag*=wt;
     }
//---
   double temp[];
//---
   CFastFourierTransform::FFTR1DInv(dft_temp,m_padded_len,temp);
//---
   ArrayResize(out, m_length);
//---
   for(int i = 0; i<m_length; i++)
      out[i]=(add_trend)?temp[i] + m_intercept + m_slope*i:temp[i];

  }
//+------------------------------------------------------------------+
//|Filters series in freq domain and returns time domain output      |
//+------------------------------------------------------------------+
void CFilter::Highpass(double freq,double width,double &out[])
  {
//---
   int i ;
   double f, dist, wt ;
   complex dft_temp[];
   ArrayCopy(dft_temp,m_dft);
//---
   for(i=0 ; i<=m_half_padded_len ; i++)
     {
      f = (double) i / (double) m_padded_len ;  // This frequency
      if(f >= freq)                  // Flat to here
         wt = 1.0 ;
      else
        {
         dist = (f - freq) / width ;
         wt = exp(-dist * dist) ;
        }
      dft_temp[i].real*=wt;
      dft_temp[i].imag*=wt;
     }
//---
   double temp[];
//---
   CFastFourierTransform::FFTR1DInv(dft_temp,m_padded_len,temp);
//---
   ArrayCopy(out,temp,0,0,m_length);

  }
//+----------------------------------------------------------------------------+
//|Filters series in the freq domain and outputs filtered series in time domain|
//+----------------------------------------------------------------------------+
void CFilter::Bandpass(double freq,double width,double &out[])
  {
//---
   int i ;
   double f, dist, wt ;
   complex dft_temp[];
   ArrayCopy(dft_temp,m_dft);
//---
   for(i=0 ; i<=m_half_padded_len ; i++)
     {
      f = (double) i / (double) m_padded_len ;  // This frequency
      dist = (f - freq) / width ;
      wt = exp(-dist * dist) ;
      dft_temp[i].real *= wt ;
      dft_temp[i].imag *= wt ;
     }
//---
   double temp[];
//---
   CFastFourierTransform::FFTR1DInv(dft_temp,m_padded_len,temp);
//---
   ArrayCopy(out,temp,0,0,m_length);
  }

//+------------------------------------------------------------------+
//| Implements Quadrature Mirror Filter, output is complex           |
//+------------------------------------------------------------------+
void CFilter::Qmf(double freq,double width,complex &out[])
  {
//---
   int i ;
   double f, dist, wt ;
   complex dft_temp[];
   ArrayCopy(dft_temp,m_dft);
//---
   for(i=1 ; i<m_half_padded_len ; i++)
     {
      f = (double) i / (double) m_padded_len ;    // This frequency
      dist = (f - freq) / width ;
      wt = exp(-dist * dist) ;
      dft_temp[i].real *= wt ;
      dft_temp[i].imag *= wt ;
      dft_temp[m_padded_len-i].real = dft_temp[m_padded_len-i].imag = 0.0 ;  // Causes QMF outputs
     }

//---
   dft_temp[0].real = 0.0 ;
   dist = (0.5 - freq) / width ;
   dft_temp[m_half_padded_len].real = 0.5 * dft_temp[m_half_padded_len].imag * exp(-dist * dist) ;
//---
   dft_temp[0].imag = dft_temp[m_half_padded_len].imag = 0.0 ;     // By definition of real transform
//---
   CFastFourierTransform::FFTC1DInv(dft_temp,m_padded_len);
   ArrayResize(out,m_length);
//---
   for(i=0 ; i<m_length ; i++)
     {
      out[i].real = dft_temp[i].real/double(m_half_padded_len) ;
      out[i].imag = dft_temp[i].imag/double(m_half_padded_len) ;
     }
  }
//+------------------------------------------------------------------+
