//+------------------------------------------------------------------+
//|                                                     SignalCT.mqh |
//|                   Copyright 2009-2013, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
#include <ct_16.mqh>
#define __PHI 0.61803398874989
#define __INPUTS 4
#define __OUTPUTS 1
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals based on 'Category Theory' Functors (part 3a)      |
//| Type=SignalAdvanced                                              |
//| Name=CategoryTheory                                              |
//| ShortName=CT                                                     |
//| Class=CSignalCT                                                  |
//| Page=signal_ct                                                   |
//| Parameter=Currency,string,USD,Currency (EUR implies Germany)     |
//| Parameter=Objects,bool,true,Use Objects to Objects               |
//| Parameter=Decay,double,0.001,Multi Layer Perceptron Decay        |
//| Parameter=Restarts,int,2,Multi Layer Perceptron Restarts         |
//| Parameter=TrainingStop,datetime,D'2023.01.01',Training Stop      |
//| Parameter=TestingStop,datetime,D'2023.08.01',Testing Stop        |
//| Parameter=Hidden,int,7,Hidden Layer Size                         |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalCT.                                                 |
//| Appointment: Class of generator of trade signals based on        |
//|                     'Category Theory'                            |
//|               Functors (part 3a).                                |
//| Derives from class CExpertTrailing.                              |
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Custom MLP read and write implementations whose soure is not     |
//| shared. Reader can customise their own to their strategy.        |
//+------------------------------------------------------------------+
#import "ct_16.ex5"
  bool Write(string File,double TrainingProfit,CMultilayerPerceptron &MLP);
  bool Read(string File,double TrainingProfit,CMultilayerPerceptron &MLP);
#import


//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalCT : public CExpertSignal
  {
protected:
   //--- adjusted parameters
   double m_step;                // trailing step
   string m_currency;            // currency morph
   bool m_objects;               // objects
   double m_decay;               // decay
   int m_restarts;               // restarts
   datetime m_training_stop;     // training stop
   datetime m_testing_stop;      // testing stop
   int m_hidden;                 // hidden

public:
   //--- methods of setting adjustable parameters
   
   
   
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);
   //---
                     CSignalCT(void);
                    ~CSignalCT(void);
   //--- methods of initialization of protected data
   void                 Step(double value)               { m_step=value;            }
   void                 Currency(string value)           { m_currency=value;        }
   void                 Objects(bool value)              { m_objects=value;         }
   void                 Decay(double value)              { m_decay=value;           }
   void                 Restarts(int value)              { m_restarts=value;        }
   void                 TrainingStop(datetime value)     { m_training_stop=value;   }
   void                 TestingStop(datetime value)      { m_testing_stop=value;    }
   void                 Hidden(int value)                { m_hidden=value;          }

protected:

      void  SetSP500(int Index); 
      
      double GetOutput(datetime Date);
      
      void  SetEconomic(datetime Date);
      
      CElement<string> _e;
      
      CMorphism<string,string> _m;
      
      CCategory _c_economic,_c_sp500;
      
      CDomain<string> _d_economic,_d_sp500;
      
      int m_testing_points;
      int m_training_points;
      double m_training_profit;
      double m_training_deposit;
      CMultilayerPerceptron _MLP;
      bool Train(CMultilayerPerceptron &MLP);
      void TrainingLoad(datetime Date,CMatrixDouble &XY,int &TrainingPoints,int &TestingPoints);

public:
      bool WritePerceptron(double TrainingProfit,CMultilayerPerceptron &MLP);
      bool ReadPerceptron(double &TrainingProfit,CMultilayerPerceptron &MLP);
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSignalCT::CSignalCT(void) :  m_step(0.5),
                              m_currency("USD"),
                              m_objects(true),
                              m_decay(0.001),
                              m_restarts(2),
                              m_training_stop(D'2023.01.01'),
                              m_testing_stop(D'2023.08.01'),
                              m_hidden(7)
  {
//--- initialization of protected data
      m_used_series=USE_SERIES_TIME+USE_SERIES_SPREAD+USE_SERIES_HIGH+USE_SERIES_LOW+USE_SERIES_CLOSE;
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSignalCT::~CSignalCT(void)
  {
  }
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalCT::ValidationSettings(void)
  {
//--- validation settings of additional filters
   if(!CExpertSignal::ValidationSettings())
      return(false);

   if(m_period!=PERIOD_MN1){
      printf(__FUNCSIG__+" Timeframe should be daily! ");
      return(false);}

   if(m_currency!=m_account.Currency()){
      printf(__FUNCSIG__+" Currency should match account! ");
      return(false);}

   if(!Train(_MLP)){
      printf(__FUNCSIG__+" failed to train perceptron! ");
      return(false);}else{
      //printf(__FUNCSIG__+" initialised perceptron with: "+IntegerToString(m_training_points)+" training points, and: "+IntegerToString(m_testing_points)+" testing points");
      }
      
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalCT::InitIndicators(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- 
   m_training_deposit=m_account.Balance();
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalCT::LongCondition(void)
  {
      int result=0;
      
      m_time.Refresh(-1);
      
      double _output=GetOutput(m_time.GetData(0));
      
      result=int(round(100.0*_output));
      
      if(result<0)
      {
         result=0;
      }
      
      return(result);
  }
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalCT::ShortCondition(void)
  {
      int result=0;
      
      m_time.Refresh(-1);
      
      double _output=GetOutput(m_time.GetData(0));
      
      result=int(round(100.0*_output));
      
      if(result>0)
      {
         result=0;
      }
      
      return(-1*result);
  }
//+------------------------------------------------------------------+
//|   Get Output value, forecast for next change in price bar range. |
//+------------------------------------------------------------------+
double CSignalCT::GetOutput(datetime Date)
   {
      if(Date>=D'2023.07.01')
      {
         printf(__FUNCSIG__+" log profit: "+DoubleToString(m_training_profit)+", account profit: "+DoubleToString(m_account.Profit())+", equity: "+DoubleToString(m_account.Equity())+", deposit: "+DoubleToString(m_training_deposit));
         
         if(m_training_profit<m_account.Equity()-m_training_deposit)
         {
            printf(__FUNCSIG__+" perceptron write... ");
            m_training_profit=m_account.Equity()-m_training_deposit;
            
            WritePerceptron(m_training_profit,_MLP);
         }
      }
      
      SetEconomic(Date);
      SetSP500(0);
      
      double _output=0.0;
      
      string _value="";
      
      double _x_inputs[],_y_inputs[];
      ArrayResize(_x_inputs,__INPUTS);
      ArrayResize(_y_inputs,__OUTPUTS);
   
      ArrayInitialize(_x_inputs,0.0);
      ArrayInitialize(_y_inputs,0.0);
      
      //printf(__FUNCSIG__+" on: "+TimeToString(Date));
      
      _value="";_e.Let();_e.Cardinality(1);
      _d_economic.Get(0,_e);_e.Get(0,_value);
      _x_inputs[0]=StringToDouble(_value);//printf(__FUNCSIG__+" val 1: "+_value);
      
      _value="";_e.Let();_e.Cardinality(1);
      _d_economic.Get(1,_e);_e.Get(0,_value);
      _x_inputs[1]=StringToDouble(_value);//printf(__FUNCSIG__+" val 2: "+_value);
      
      _value="";_e.Let();_e.Cardinality(1);
      _d_economic.Get(2,_e);_e.Get(0,_value);
      _x_inputs[2]=StringToDouble(_value);//printf(__FUNCSIG__+" val 3: "+_value);
      
      _value="";_e.Let();_e.Cardinality(1);
      _d_economic.Get(3,_e);_e.Get(0,_value);
      _x_inputs[3]=StringToDouble(_value);//printf(__FUNCSIG__+" val 4: "+_value);
      
      //forward feed?...
      CMLPBase _base;
      _base.MLPProcess(_MLP,_x_inputs,_y_inputs);
      
      _output=_y_inputs[0];
      
      //printf(__FUNCSIG__+" output is: "+DoubleToString(_output));
               
      return(_output);
   }
//+------------------------------------------------------------------+
//|   Get Economic data from file defined in input.                  | 
//|   Load it into Economic category.                                |
//+------------------------------------------------------------------+
void CSignalCT::SetEconomic(datetime Date)
   {
      ResetLastError();
      string _file="_s_"+m_currency+"_"+m_symbol.Name()+"_"+EnumToString(m_period)+"_"+string(m_objects)+".csv";
      int _handle=FileOpen(_file,FILE_SHARE_READ|FILE_ANSI|FILE_COMMON,"\n",CP_ACP);
      
      _d_economic.Let(); _d_economic.Cardinality(4);
               
      if(_handle!=INVALID_HANDLE)
      {
         string _line="";
         int _line_length=0;
         
         while(!FileIsLineEnding(_handle))
         {
            //--- find out how many characters are used for writing the line
            _line_length=FileReadInteger(_handle,INT_VALUE);
            //--- read the line
            _line=FileReadString(_handle,_line_length);
            
            string _values[];
            ushort _separator=StringGetCharacter(",",0);
            if(StringSplit(_line,_separator,_values)==6)
            {
               datetime _date=StringToTime(_values[0]);
               
               if(_date==Date)
               {
                  _e.Let();_e.Cardinality(1);_e.Set(0,_values[1]);
                  _d_economic.Set(0,_e);
                  
                  _e.Let();_e.Cardinality(1);_e.Set(0,_values[2]);
                  _d_economic.Set(1,_e);
                  
                  _e.Let();_e.Cardinality(1);_e.Set(0,_values[3]);
                  _d_economic.Set(2,_e);
                  
                  _e.Let();_e.Cardinality(1);_e.Set(0,_values[4]);
                  _d_economic.Set(3,_e);
                  
                  _c_economic.Domains(_c_economic.Domains()+1); _c_economic.SetDomain(_c_economic.Domains()-1,_d_economic);
                  
                  break;
               }
            }
         }
         
         FileClose(_handle);
      }
      else
      {
         printf(__FUNCSIG__+" failed to load file. Err: "+IntegerToString(GetLastError()));
      }
   }
//+------------------------------------------------------------------+
//|   Get NASDAQ-100 data from symbol (NDX100 for this broker).      | 
//|   Load it into NASDAQ category.                                  |
//+------------------------------------------------------------------+
void CSignalCT::SetSP500(int Index)
   {
      m_high.Refresh(-1);
      m_low.Refresh(-1);
      
      double _value=(m_high.GetData(Index+StartIndex()+m_high.MaxIndex(Index,_c_economic.Homomorphisms()))-m_low.GetData(Index+StartIndex()+m_low.MinIndex(Index,_c_economic.Homomorphisms())))/m_symbol.Point();
      
      _e.Let();_e.Cardinality(1);_e.Set(0,DoubleToString(_value));
      _d_sp500.Cardinality(1);_d_sp500.Set(0,_e);
      
      _c_sp500.SetDomain(_c_sp500.Domains(),_d_sp500);  
   }
//+------------------------------------------------------------------+
//|   Function to train Perceptron.                                  |
//+------------------------------------------------------------------+
bool CSignalCT::Train(CMultilayerPerceptron &MLP)
   {
      CMLPBase _base;
      CMLPTrain _train;
      
      if(!ReadPerceptron(m_training_profit,MLP))
      {
         _base.MLPCreate1(__INPUTS,m_hidden,__OUTPUTS,MLP);
         m_training_profit=0.0;
      }
      else
      {
         printf(__FUNCSIG__+" read perceptron, with profit: "+DoubleToString(m_training_profit));
      }
      
      int _info=0;
      CMatrixDouble _xy;
      CMLPReport _report;
      TrainingLoad(m_training_stop,_xy,m_training_points,m_testing_points);
      //
      if(m_training_points>0)
      {
         _train.MLPTrainLM(MLP,_xy,m_training_points,m_decay,m_restarts,_info,_report);
         
         if(_info>0){ return(true); }
      } 
      
      return(false);
   }
//+------------------------------------------------------------------+
//|   Function Get Training Points and Initialize Training Matrix.   |
//+------------------------------------------------------------------+
void CSignalCT::TrainingLoad(datetime Date,CMatrixDouble &XY,int &TrainingPoints,int &TestingPoints)
   {
      TrainingPoints=0;
      TestingPoints=0;
      
      ResetLastError();
      string _file="_s_"+m_currency+"_"+m_symbol.Name()+"_"+EnumToString(m_period)+"_"+string(m_objects)+".csv";
      int _handle=FileOpen(_file,FILE_SHARE_READ|FILE_ANSI|FILE_COMMON,"\n",CP_ACP);
      
      if(_handle!=INVALID_HANDLE)
      {
         string _line="";
         int _line_length=0;
         
         while(!FileIsLineEnding(_handle))
         {
            //--- find out how many characters are used for writing the line
            _line_length=FileReadInteger(_handle,INT_VALUE);
            //--- read the line
            _line=FileReadString(_handle,_line_length);
            
            string _values[];
            ushort _separator=StringGetCharacter(",",0);
            if(StringSplit(_line,_separator,_values)==6)
            {
               datetime _date=StringToTime(_values[0]);
               
               _d_economic.Let(); _d_economic.Cardinality(4);
               
               //printf(__FUNCSIG__+" initializing for: "+TimeToString(Date)+" at: "+TimeToString(_date));
               
               if(_date<Date)
               {
                  TrainingPoints++;
                  //
                  XY.Resize(TrainingPoints,__INPUTS+__OUTPUTS);
                  
                  for(int i=0;i<__INPUTS;i++)
                  {
                     XY[TrainingPoints-1].Set(i,StringToDouble(_values[i+1]));
                  }
                  //
                  XY[TrainingPoints-1].Set(__INPUTS,StringToDouble(_values[__INPUTS+1]));
               }
               else
               {
                  TestingPoints++;
               }
            }
         }
         
         FileClose(_handle);
      }
      else
      {
         printf(__FUNCSIG__+" failed to load file. Err: "+IntegerToString(GetLastError()));
      }
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalCT::WritePerceptron(double TrainingProfit,CMultilayerPerceptron &MLP)
   {
      string _file="_s_perceptron_"+m_currency+"_"+m_symbol.Name()+"_"+EnumToString(m_period)+"_"+IntegerToString(m_hidden)+"_"+string(m_objects)+".bin";
      
      return(Write(_file,TrainingProfit,MLP));
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CSignalCT::ReadPerceptron(double &TrainingProfit,CMultilayerPerceptron &MLP)
   {
      string _file="_s_perceptron_"+m_currency+"_"+m_symbol.Name()+"_"+EnumToString(m_period)+"_"+IntegerToString(m_hidden)+"_"+string(m_objects)+".bin";
      
      return(Read(_file,TrainingProfit,MLP));
   }
//+------------------------------------------------------------------+
