//+------------------------------------------------------------------+
//|                                                   TrailingCT.mqh |
//|                   Copyright 2009-2013, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Math\Stat\Math.mqh>
#include <Expert\ExpertTrailing.mqh>
#include <ct_12.mqh>
#define __PHI 0.61803398874989
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Trailing Stop based on 'Category Theory' orders (case 2)   |
//| Type=Trailing                                                    |
//| Name=CategoryTheory                                              |
//| ShortName=CT                                                     |
//| Class=CTrailingCT                                                |
//| Page=trailing_ct                                                 |
//| Parameter=Step,double,0.5,Trailing Step                          |
//| Parameter=LookbackOperation,int,0, Lookback Operation            |
//| Parameter=LookbackIdentity,int,0, Lookback Identity              |
//| Parameter=TimeframeOperation,int,0, Timeframe Operation          |
//| Parameter=TimeframeIdentity,int,0, Timeframe Identity            |
//| Parameter=AppliedpriceOperation,int,0, Appliedprice Operation    |
//| Parameter=AppliedpriceIdentity,int,0, Appliedprice Identity      |
//| Parameter=IndicatorOperation,int,0, Indicator Operation          |
//| Parameter=IndicatorIdentity,int,0, Indicator Identity            |
//| Parameter=TradeDecisionOperation,int,0, TradeDecision Operation  |
//| Parameter=TradeDecisionIdentity,int,0, TradeDecision Identity    |
//| Parameter=DefaultLookback,int,12, Default Lookback               |
//| Parameter=DefaultAppliedprice,int,2, Default Appliedprice        |
//| Parameter=DefaultIndicator,double,50.0, Default Indicator        |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CTrailingCT.                                               |
//| Appointment: Class traling stops with 'Category Theory'          |
//|               orders (case 2).                                   |
//| Derives from class CExpertTrailing.                              |
//+------------------------------------------------------------------+
int                        __LOOKBACKS[8] = {1,2,3,4,5,6,7,8};
ENUM_TIMEFRAMES            __TIMEFRAMES[8] = {PERIOD_H1,PERIOD_H2,PERIOD_H3,PERIOD_H4,PERIOD_H6,PERIOD_H8,PERIOD_H12,PERIOD_D1};
ENUM_APPLIED_PRICE         __APPLIEDPRICES[4] = { PRICE_MEDIAN, PRICE_TYPICAL, PRICE_OPEN, PRICE_CLOSE };
string                     __INDICATORS[2] = { "RSI", "BOLLINGER_BANDS" };
string                     __DECISIONS[2] = { "TREND", "RANGE" };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CTrailingCT : public CExpertTrailing
  {
protected:
   //--- adjusted parameters
   double m_step;                // trailing step
   
   CMonoid<double>               m_lookback;
   EOperations                   m_lookback_operation;
   int                           m_lookback_identity;
   
   CMonoid<double>               m_timeframe;
   EOperations                   m_timeframe_operation;
   int                           m_timeframe_identity;
   
   CMonoid<double>               m_appliedprice;
   EOperations                   m_appliedprice_operation;
   int                           m_appliedprice_identity;
   
   CMonoid<double>               m_indicator;
   EOperations                   m_indicator_operation;
   int                           m_indicator_identity;
   
   CMonoid<double>               m_trade_decision;
   EOperations                   m_decision_operation;
   int                           m_decision_identity;
   
   //
   int                           m_long_index;
   int                           m_short_index;
   
   int                           m_default_lookback;
   int                           m_default_appliedprice;
   double                        m_default_indicator;
   //
   
   CMonoidAction<double,double>  m_lookback_act;
   CMonoidAction<double,double>  m_timeframe_act;
   CMonoidAction<double,double>  m_appliedprice_act;
   
   bool                          m_correlation_inverted;
   
   int                           m_source_size;       // Source Size

public:
   //--- methods of setting adjustable parameters
   
   
   
   //--- method of verification of settings
   virtual bool         ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool         InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual bool         CheckTrailingStopLong(CPositionInfo *position,double &sl,double &tp);
   virtual bool         CheckTrailingStopShort(CPositionInfo *position,double &sl,double &tp);
   //---
                        CTrailingCT(void);
                        ~CTrailingCT(void);
   //--- methods of initialization of protected data
   void                 Step(double value)                  { m_step=value;                                 }
   
   void                 LookbackOperation(int value)        { m_lookback_operation=(EOperations)value;      }
   void                 LookbackIdentity(int value)         { m_lookback_identity=value;                    }
   
   void                 TimeframeOperation(int value)       { m_timeframe_operation=(EOperations)value;     }
   void                 TimeframeIdentity(int value)        { m_timeframe_identity=value;                   }
   
   void                 AppliedpriceOperation(int value)    { m_appliedprice_operation=(EOperations)value;  }
   void                 AppliedpriceIdentity(int value)     { m_appliedprice_identity=value;                }
   
   void                 IndicatorOperation(int value)       { m_indicator_operation=(EOperations)value;     }
   void                 IndicatorIdentity(int value)        { m_indicator_identity=value;                   }
   
   void                 TradeDecisionOperation(int value)   { m_decision_operation=(EOperations)value;      }
   void                 TradeDecisionIdentity(int value)    { m_decision_identity=value;                    }
   
   void                 CorrelationInverted(bool value)     { m_correlation_inverted=value;                 }
   
   void                 LongIndex(int value)                { m_long_index=value;                           }
   void                 ShortIndex(int value)               { m_short_index=value;                          }
   
   void                 DefaultLookback(int value)          { m_default_lookback=value;                     }
   void                 DefaultAppliedprice(int value)      { m_default_appliedprice=value;                 }
   void                 DefaultIndicator(double value)      { m_default_indicator=value;                    }

protected:
   
   CElement<double>     m_element;

   ENUM_TIMEFRAMES      GetTimeframe(CMonoid<double> &M,ENUM_TIMEFRAMES &T[]);
   int                  GetLookback(CMonoid<double> &M,int &L[],ENUM_TIMEFRAMES Timeframe);
   ENUM_APPLIED_PRICE   GetAppliedprice(CMonoid<double> &M,ENUM_APPLIED_PRICE &A[],int Lookback,ENUM_TIMEFRAMES Timeframe);
   double               GetIndicator(int Lookback,ENUM_TIMEFRAMES Timeframe,ENUM_APPLIED_PRICE Appliedprice);
   
   int                  GetTradeDecision(ENUM_TIMEFRAMES Timeframe,int Lookback,ENUM_APPLIED_PRICE Appliedprice,double Indicator);

   int                  GetLookback(int Lookback,int &L[]){ int _l=-1; for(int i=0;i<8;i++){ if(L[i]==Lookback){ _l=i; break; }}  return(_l); }
   int                  GetTimeframe(ENUM_TIMEFRAMES Timeframe,ENUM_TIMEFRAMES &T[]){ int _t=-1; for(int i=0;i<8;i++){ if(T[i]==Timeframe){ _t=i; break; }}  return(_t); }
   int                  GetAppliedprice(ENUM_APPLIED_PRICE Appliedprice,ENUM_APPLIED_PRICE &A[]){ int _a=-1; for(int i=0;i<4;i++){ if(A[i]==Appliedprice){ _a=i; break; }}  return(_a); }
   
   void                 Operate(CMonoid<double> &M,EOperations &O,int &OutputIndex);
   
   bool                 Ordering(int Lookback);
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CTrailingCT::CTrailingCT(void)
  {
//--- initialization of protected data
      m_used_series=USE_SERIES_TIME+USE_SERIES_SPREAD+USE_SERIES_HIGH+USE_SERIES_LOW+USE_SERIES_CLOSE;
      
      m_lookback.Cardinality(8);
      m_lookback.Operation(m_lookback_operation);
      m_lookback.Identity(m_lookback_identity);
      
      m_timeframe.Cardinality(8);
      m_timeframe.Operation(m_timeframe_operation);
      m_timeframe.Identity(m_timeframe_identity);
      
      m_appliedprice.Cardinality(4);
      m_appliedprice.Operation(m_appliedprice_operation);
      m_appliedprice.Identity(m_appliedprice_identity);
      
      m_indicator.Cardinality(2);
      m_indicator.Operation(m_indicator_operation);
      m_indicator.Identity(m_indicator_identity);
      
      m_trade_decision.Cardinality(2);
      m_trade_decision.Operation(m_decision_operation);
      m_trade_decision.Identity(m_decision_identity);
      
      m_step=0.5;
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CTrailingCT::~CTrailingCT(void)
  {
  }
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CTrailingCT::ValidationSettings(void)
  {
//--- validation settings of additional filters
   if(!CExpertTrailing::ValidationSettings())
      return(false);

   if(m_period!=PERIOD_H1){
      printf(__FUNCSIG__+" Timeframe should be one-hour! ");
      return(false);}
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CTrailingCT::InitIndicators(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertTrailing::InitIndicators(indicators))
      return(false);
//--- 
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Checking trailing stop and/or profit for long position.          |
//+------------------------------------------------------------------+
bool CTrailingCT::CheckTrailingStopLong(CPositionInfo *position,double &sl,double &tp)
  {
//--- check
      if(position==NULL)
         return(false);
      
      m_high.Refresh(-1);
      m_low.Refresh(-1);
      
      int _x=StartIndex();
      
      //partial ordering
      if(Ordering(12))// assigning index
      {
         ENUM_TIMEFRAMES _timeframe=GetTimeframe(m_timeframe,__TIMEFRAMES);
         
         int _lookback=m_default_lookback;
         ENUM_APPLIED_PRICE _appliedprice=__APPLIEDPRICES[m_default_appliedprice];
         double _indicator=m_default_indicator;
         
         if(m_long_index==0)
         {
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,_lookback,_timeframe);
            _indicator=GetIndicator(_lookback,_timeframe,_appliedprice);
         }
         else if(m_long_index==1)
         {
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,__LOOKBACKS[m_default_lookback],_timeframe);
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
            _indicator=GetIndicator(_lookback,_timeframe,_appliedprice);
         }
         else if(m_long_index==2)
         {
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,__LOOKBACKS[m_default_lookback],_timeframe);
            _indicator=GetIndicator(m_default_lookback,_timeframe,_appliedprice);
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
         }
         else if(m_long_index==3)
         {
            _indicator=GetIndicator(__LOOKBACKS[m_default_lookback],_timeframe,__APPLIEDPRICES[m_default_appliedprice]);
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,m_default_lookback,_timeframe);
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
         }
         else if(m_long_index==4)
         {
            _indicator=GetIndicator(__LOOKBACKS[m_default_lookback],_timeframe,__APPLIEDPRICES[m_default_appliedprice]);
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,_lookback,_timeframe);
         }
         else if(m_long_index==5)
         {
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
            _indicator=GetIndicator(_lookback,_timeframe,__APPLIEDPRICES[m_default_appliedprice]);
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,_lookback,_timeframe);
         }
         //
         int _trade_decision=GetTradeDecision(_timeframe,_lookback,_appliedprice,_indicator);
         
         double _type=_trade_decision/100.0;
      
         double _atr=fmax(2.0*m_spread.GetData(_x)*m_symbol.Point(),m_high.GetData(_x)-m_low.GetData(_x))*(_type);
         
         double _sl=m_low.GetData(_x)-(m_step*_atr);
         
         double level =NormalizeDouble(m_symbol.Bid()-m_symbol.StopsLevel()*m_symbol.Point(),m_symbol.Digits());
         double new_sl=NormalizeDouble(_sl,m_symbol.Digits());
         double pos_sl=position.StopLoss();
         double base  =(pos_sl==0.0) ? position.PriceOpen() : pos_sl;
         
         sl=EMPTY_VALUE;
         tp=EMPTY_VALUE;
         if(new_sl>base && new_sl<level)
            sl=new_sl;
      }
      
//---
   return(sl!=EMPTY_VALUE);
  }
//+------------------------------------------------------------------+
//| Checking trailing stop and/or profit for short position.         |
//+------------------------------------------------------------------+
bool CTrailingCT::CheckTrailingStopShort(CPositionInfo *position,double &sl,double &tp)
  {
//--- check
      if(position==NULL)
         return(false);
      
      m_high.Refresh(-1);
      m_low.Refresh(-1);
      
      int _x=StartIndex();
      
      //partial ordering
      if(Ordering(12))
      {
         ENUM_TIMEFRAMES _timeframe=GetTimeframe(m_timeframe,__TIMEFRAMES);
         
         int _lookback=m_default_lookback;
         ENUM_APPLIED_PRICE _appliedprice=__APPLIEDPRICES[m_default_appliedprice];
         double _indicator=m_default_indicator;
         
         if(m_short_index==0)
         {
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,_lookback,_timeframe);
            _indicator=GetIndicator(_lookback,_timeframe,_appliedprice);
         }
         else if(m_short_index==1)
         {
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,__LOOKBACKS[m_default_lookback],_timeframe);
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
            _indicator=GetIndicator(_lookback,_timeframe,_appliedprice);
         }
         else if(m_short_index==2)
         {
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,__LOOKBACKS[m_default_lookback],_timeframe);
            _indicator=GetIndicator(m_default_lookback,_timeframe,_appliedprice);
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
         }
         else if(m_short_index==3)
         {
            _indicator=GetIndicator(__LOOKBACKS[m_default_lookback],_timeframe,__APPLIEDPRICES[m_default_appliedprice]);
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,m_default_lookback,_timeframe);
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
         }
         else if(m_short_index==4)
         {
            _indicator=GetIndicator(__LOOKBACKS[m_default_lookback],_timeframe,__APPLIEDPRICES[m_default_appliedprice]);
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,_lookback,_timeframe);
         }
         else if(m_short_index==5)
         {
            _lookback=GetLookback(m_lookback,__LOOKBACKS,_timeframe);
            _indicator=GetIndicator(_lookback,_timeframe,__APPLIEDPRICES[m_default_appliedprice]);
            _appliedprice=GetAppliedprice(m_appliedprice,__APPLIEDPRICES,_lookback,_timeframe);
         }
         //
         int _trade_decision=GetTradeDecision(_timeframe,_lookback,_appliedprice,_indicator);
         
         double _type=_trade_decision/100.0;
      
         double _atr=fmax(2.0*m_spread.GetData(_x)*m_symbol.Point(),m_high.GetData(_x)-m_low.GetData(_x))*(_type);
         
         double _sl=m_high.GetData(_x)+(m_step*_atr);
         
         double level =NormalizeDouble(m_symbol.Ask()+m_symbol.StopsLevel()*m_symbol.Point(),m_symbol.Digits());
         double new_sl=NormalizeDouble(_sl,m_symbol.Digits());
         double pos_sl=position.StopLoss();
         double base  =(pos_sl==0.0) ? position.PriceOpen() : pos_sl;
         
         sl=EMPTY_VALUE;
         tp=EMPTY_VALUE;
         if(new_sl<base && new_sl>level)
            sl=new_sl;
      }
      
//---
      return(sl!=EMPTY_VALUE);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CTrailingCT::Operate(CMonoid<double> &M,EOperations &O,int &OutputIndex)
   {
      OutputIndex=-1;
      //
      double _values[];
      ArrayResize(_values,M.Cardinality());ArrayInitialize(_values,0.0);
      //
      for(int i=0;i<M.Cardinality();i++)
      {
         m_element.Let();
         if(M.Get(i,m_element))
         {
            if(!m_element.Get(0,_values[i]))
            {
               printf(__FUNCSIG__+" Failed to get double for 1 at: "+IntegerToString(i+1));
            }
         }
         else{ printf(__FUNCSIG__+" Failed to get element for 1 at: "+IntegerToString(i+1)); }
      }
      
      //
      
      if(O==OP_LEAST)
      {
         OutputIndex=0;
         double _least=_values[0];
         for(int i=0;i<M.Cardinality();i++)
         {
            if(_least>_values[i]){ _least=_values[i]; OutputIndex=i; }
         }
      }
      else if(O==OP_MOST)
      {
         OutputIndex=0;
         double _most=_values[0];
         for(int i=0;i<M.Cardinality();i++)
         {
            if(_most<_values[i]){ _most=_values[i]; OutputIndex=i; }
         }
      }
      else if(O==OP_CLOSEST)
      {
         double _mean=0.0;
         for(int i=0;i<M.Cardinality();i++)
         {
            _mean+=_values[i];
         }
         _mean/=M.Cardinality();
         
         OutputIndex=0;
         double _closest=fabs(_values[0]-_mean);
         for(int i=0;i<M.Cardinality();i++)
         {
            if(_closest>fabs(_values[i]-_mean)){ _closest=fabs(_values[i]-_mean); OutputIndex=i; }
         }
      }
      else if(O==OP_FURTHEST)
      {
         double _mean=0.0;
         for(int i=0;i<M.Cardinality();i++)
         {
            _mean+=_values[i];
         }
         _mean/=M.Cardinality();
         
         OutputIndex=0;
         double _furthest=fabs(_values[0]-_mean);
         for(int i=0;i<M.Cardinality();i++)
         {
            if(_furthest<fabs(_values[i]-_mean)){ _furthest=fabs(_values[i]-_mean); OutputIndex=i; }
         }
      }
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
ENUM_TIMEFRAMES CTrailingCT::GetTimeframe(CMonoid<double> &M, ENUM_TIMEFRAMES &T[])
   {
      for(int i=0;i<M.Cardinality();i++)
      {
         ResetLastError();
         double _value=0.0;
         double _buffer[];ArrayResize(_buffer,3);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
         if(CopyClose(m_symbol.Name(),T[i],0,3,_buffer)>=3)
         {
            _value=fabs(_buffer[0]-_buffer[1])/(fabs(_buffer[0]-_buffer[1])+fabs(_buffer[1]-_buffer[2]));
         }
         else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(T[i])+" close prices. err: "+IntegerToString(GetLastError())); }
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!M.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for lookback. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      int _i_out=-1;
      //
      Operate(M,m_timeframe_operation,_i_out);
      
      if(_i_out==-1){ return(INVALID_HANDLE); }
      
      return(T[_i_out]);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CTrailingCT::GetLookback(CMonoid<double> &M,int &L[],ENUM_TIMEFRAMES Timeframe)
   {
      m_close.Refresh(-1);
      
      int _x=StartIndex();
      
      for(int i=0;i<M.Cardinality();i++)
      {
         int _period=(L[i]*PeriodSeconds(Timeframe))/PeriodSeconds(m_period);
         double _value=fabs(m_close.GetData(_x)-m_close.GetData(_x+_period))/(fabs(m_close.GetData(_x)-m_close.GetData(_x+_period))+fabs(m_close.GetData(_x+_period)-m_close.GetData(_x+_period+_period)));
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!M.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for lookback. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
   
      int _i_out=-1;
      //
      Operate(M,m_lookback_operation,_i_out);
      
      if(_i_out==-1){ return(4); }
      
      return(4*L[_i_out]);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
ENUM_APPLIED_PRICE CTrailingCT::GetAppliedprice(CMonoid<double> &M,ENUM_APPLIED_PRICE &A[],int Lookback,ENUM_TIMEFRAMES Timeframe)
   {
      for(int i=0;i<M.Cardinality();i++)
      {
         ResetLastError();
         double _value=0.0;
         if(A[i]==PRICE_CLOSE)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            if(CopyClose(m_symbol.Name(),Timeframe,0,Lookback,_buffer)>=Lookback)
            {
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" close prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(A[i]==PRICE_OPEN)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            if(CopyOpen(m_symbol.Name(),Timeframe,0,Lookback,_buffer)>=Lookback)
            {
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" open prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(A[i]==PRICE_MEDIAN)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            double _h[];ArrayResize(_h,Lookback);ArrayInitialize(_h,0.0);
            double _l[];ArrayResize(_l,Lookback);ArrayInitialize(_l,0.0);
            if(CopyHigh(m_symbol.Name(),Timeframe,0,Lookback,_h)>=Lookback && CopyLow(m_symbol.Name(),Timeframe,0,Lookback,_l)>=Lookback)
            {
               for(int ii=0;ii<Lookback;ii++)
               {
                  _buffer[ii]=0.5*(_h[ii]+_l[ii]);
               }
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(A[i]==PRICE_TYPICAL)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            double _h[];ArrayResize(_h,Lookback);ArrayInitialize(_h,0.0);
            double _l[];ArrayResize(_l,Lookback);ArrayInitialize(_l,0.0);
            double _c[];ArrayResize(_c,Lookback);ArrayInitialize(_c,0.0);
            if(CopyHigh(m_symbol.Name(),Timeframe,0,Lookback,_h)>=Lookback && CopyLow(m_symbol.Name(),Timeframe,0,Lookback,_l)>=Lookback && CopyClose(m_symbol.Name(),Timeframe,0,Lookback,_c)>=Lookback)
            {
               for(int ii=0;ii<Lookback;ii++)
               {
                  _buffer[ii]=(1.0/3.0)*(_h[ii]+_l[ii]+_c[ii]);
               }
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(A[i]==PRICE_WEIGHTED)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            double _h[];ArrayResize(_h,Lookback);ArrayInitialize(_h,0.0);
            double _l[];ArrayResize(_l,Lookback);ArrayInitialize(_l,0.0);
            double _c[];ArrayResize(_c,Lookback);ArrayInitialize(_c,0.0);
            if(CopyHigh(m_symbol.Name(),Timeframe,0,Lookback,_h)>=Lookback && CopyLow(m_symbol.Name(),Timeframe,0,Lookback,_l)>=Lookback && CopyClose(m_symbol.Name(),Timeframe,0,Lookback,_c)>=Lookback)
            {
               for(int ii=0;ii<Lookback;ii++)
               {
                  _buffer[ii]=(1.0/4.0)*(_h[ii]+_l[ii]+_c[ii]+_c[ii]);
               }
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
         }
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!M.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for applied price. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      int _i_out=-1;
      //
      Operate(M,m_appliedprice_operation,_i_out);
      
      if(_i_out==-1){ return(4); }
      
      return(A[_i_out]);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CTrailingCT::GetIndicator(int Lookback,ENUM_TIMEFRAMES Timeframe,ENUM_APPLIED_PRICE Appliedprice)
   {
      m_spread.Refresh(-1);
      
      int _x=StartIndex();
      
      double _indicator[2];
      ArrayInitialize(_indicator,0.0);
      
      for(int i=0;i<2;i++)
      {
         ResetLastError();
         double _value=0.0;
         //
         if(__INDICATORS[i]=="RSI")
         {
            double _spread=(m_symbol.Point()*m_spread.GetData(_x));
            double _current_gain=_spread,_current_loss=_spread;
            double _gain=_spread,_loss=_spread;
            double _buffer[];ArrayResize(_buffer,fmax(2,Lookback));ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            if(CopyClose(m_symbol.Name(),Timeframe,0,fmax(2,Lookback),_buffer)>=fmax(2,Lookback))
            {
               if(_buffer[0]>_buffer[1]){ _current_gain+=_buffer[0]-_buffer[1]; }
               else if(_buffer[1]>_buffer[0]){ _current_loss+=_buffer[1]-_buffer[0]; }
               
               for(int ii=1;ii<Lookback-1;ii++)
               {
                  if(_buffer[ii]>_buffer[ii+1]){ _gain+=_buffer[ii]-_buffer[ii+1]; }
                  else if(_buffer[ii+1]>_buffer[ii]){ _loss+=_buffer[ii+1]-_buffer[ii]; }
               }
               
               _gain/=(Lookback-1);
               _loss/=(Lookback-1);
               
               _gain+=_current_gain;
               _loss+=_current_loss;
               
               _value=(_gain/_loss);
               _value+=1.0;
               _value=100.0/_value;
               _value=100.0-_value;
               
               _indicator[0]=_value;
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" close prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(__INDICATORS[i]=="BOLLINGER_BANDS")
         {
            double _base=0.0,_upper=0.0,_lower=0.0;
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            if(CopyClose(m_symbol.Name(),Timeframe,0,Lookback,_buffer)>=Lookback)
            {
               for(int ii=0;ii<Lookback;ii++)
               {
                  _base+=_buffer[ii];
               }
               
               _base/=(Lookback);
               
               double _deviation=MathStandardDeviation(_buffer);
               
               _upper+=(_base+(2.0*_deviation));
               _lower+=(_base-(2.0*_deviation));
               
               _value=((1.0+((_buffer[0]-_base)/(fabs(_buffer[0]-_base)+(_upper-_lower))))*50.0);
               
               _indicator[1]=_value;
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" close prices. err: "+IntegerToString(GetLastError())); }
         }
         
         //
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!m_indicator.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for indicator. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      int _i_out=-1;
      Operate(m_indicator,m_indicator_operation,_i_out);
      
      return(_indicator[_i_out]);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CTrailingCT::GetTradeDecision(ENUM_TIMEFRAMES Timeframe,int Lookback,ENUM_APPLIED_PRICE Appliedprice,double Indicator)
   {
      m_close.Refresh(-1);
      
      int _x=StartIndex();
      
      for(int i=0;i<2;i++)
      {
         ResetLastError();
         double _value=0.0;
         //
         int _max_index=-1,_min_index=-1;
         double _max=m_close.MaxValue(_x,Lookback,_max_index);
         double _min=m_close.MinValue(_x,Lookback,_min_index);
         
         if(__DECISIONS[i]=="TREND")
         {
            int _trends=0;
            if(Appliedprice==PRICE_CLOSE)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               if(CopyClose(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_buffer)>=_x+Lookback+1)
               {
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
            }
            else if(Appliedprice==PRICE_OPEN)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               if(CopyOpen(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_buffer)>=_x+Lookback+1)
               {
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
            }
            else if(Appliedprice==PRICE_MEDIAN)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               double _h[];ArrayResize(_h,_x+Lookback+1);ArrayInitialize(_h,0.0);
               double _l[];ArrayResize(_l,_x+Lookback+1);ArrayInitialize(_l,0.0);
               if(CopyHigh(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_h)>=_x+Lookback+1 && CopyLow(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_l)>=_x+Lookback+1)
               {
                  for(int ii=0;ii<Lookback+1;ii++)
                  {
                     _buffer[ii]=0.5*(_h[ii]+_l[ii]);
                  }
                  
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
               else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
            }
            else if(Appliedprice==PRICE_TYPICAL)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               double _h[];ArrayResize(_h,_x+Lookback+1);ArrayInitialize(_h,0.0);
               double _l[];ArrayResize(_l,_x+Lookback+1);ArrayInitialize(_l,0.0);
               double _c[];ArrayResize(_c,_x+Lookback+1);ArrayInitialize(_c,0.0);
               if(CopyHigh(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_h)>=_x+Lookback+1 && CopyLow(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_l)>=_x+Lookback+1 && CopyClose(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_c)>=_x+Lookback+1)
               {
                  for(int ii=0;ii<Lookback+1;ii++)
                  {
                     _buffer[ii]=(1.0/3.0)*(_h[ii]+_l[ii]+_c[ii]);
                  }
                  
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
               else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
            }
            else if(Appliedprice==PRICE_WEIGHTED)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               double _h[];ArrayResize(_h,_x+Lookback+1);ArrayInitialize(_h,0.0);
               double _l[];ArrayResize(_l,_x+Lookback+1);ArrayInitialize(_l,0.0);
               double _c[];ArrayResize(_c,_x+Lookback+1);ArrayInitialize(_c,0.0);
               if(CopyHigh(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_h)>=_x+Lookback+1 && CopyLow(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_l)>=_x+Lookback+1 && CopyClose(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_c)>=_x+Lookback+1)
               {
                  for(int ii=0;ii<Lookback+1;ii++)
                  {
                     _buffer[ii]=(1.0/4.0)*(_h[ii]+_l[ii]+_c[ii]+_c[ii]);
                  }
                  
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _trends++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
               else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
            }
            
            if(_trends>0){ _value/=_trends; }
            if(_max-_min>0.0){ _value/=(_max-_min); }
         }
         else if(__DECISIONS[i]=="RANGE")
         {
            int _ranges=0;
            if(Appliedprice==PRICE_CLOSE)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               if(CopyClose(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_buffer)>=_x+Lookback+1)
               {
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
            }
            else if(Appliedprice==PRICE_OPEN)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               if(CopyOpen(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_buffer)>=_x+Lookback+1)
               {
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
            }
            else if(Appliedprice==PRICE_MEDIAN)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               double _h[];ArrayResize(_h,_x+Lookback+1);ArrayInitialize(_h,0.0);
               double _l[];ArrayResize(_l,_x+Lookback+1);ArrayInitialize(_l,0.0);
               if(CopyHigh(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_h)>=_x+Lookback+1 && CopyLow(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_l)>=_x+Lookback+1)
               {
                  for(int ii=0;ii<Lookback+1;ii++)
                  {
                     _buffer[ii]=0.5*(_h[ii]+_l[ii]);
                  }
                  
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
               else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
            }
            else if(Appliedprice==PRICE_TYPICAL)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               double _h[];ArrayResize(_h,_x+Lookback+1);ArrayInitialize(_h,0.0);
               double _l[];ArrayResize(_l,_x+Lookback+1);ArrayInitialize(_l,0.0);
               double _c[];ArrayResize(_c,_x+Lookback+1);ArrayInitialize(_c,0.0);
               if(CopyHigh(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_h)>=_x+Lookback+1 && CopyLow(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_l)>=_x+Lookback+1 && CopyClose(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_c)>=_x+Lookback+1)
               {
                  for(int ii=0;ii<Lookback+1;ii++)
                  {
                     _buffer[ii]=(1.0/3.0)*(_h[ii]+_l[ii]+_c[ii]);
                  }
                  
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
               else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
            }
            else if(Appliedprice==PRICE_WEIGHTED)
            {
               double _buffer[];ArrayResize(_buffer,_x+Lookback+1);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
               double _h[];ArrayResize(_h,_x+Lookback+1);ArrayInitialize(_h,0.0);
               double _l[];ArrayResize(_l,_x+Lookback+1);ArrayInitialize(_l,0.0);
               double _c[];ArrayResize(_c,_x+Lookback+1);ArrayInitialize(_c,0.0);
               if(CopyHigh(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_h)>=_x+Lookback+1 && CopyLow(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_l)>=_x+Lookback+1 && CopyClose(m_symbol.Name(),Timeframe,0,_x+Lookback+1,_c)>=_x+Lookback+1)
               {
                  for(int ii=0;ii<Lookback+1;ii++)
                  {
                     _buffer[ii]=(1.0/4.0)*(_h[ii]+_l[ii]+_c[ii]+_c[ii]);
                  }
                  
                  if(_buffer[_x]>_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]<_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii]-_buffer[ii+1]);
                        }
                     }
                  }
                  else if(_buffer[_x]<_buffer[_x+Lookback])
                  {
                     for(int ii=_x;ii<_x+Lookback;ii++)
                     {
                        if(_buffer[ii]>_buffer[ii+1])
                        {
                           _ranges++;
                           _value+=(_buffer[ii+1]-_buffer[ii]);
                        }
                     }
                  }
               }
               else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
            }
            
            if(_ranges>0){ _value/=_ranges; }
            if(_max-_min>0.0){ _value/=(_max-_min); }
         }
         
         //
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!m_trade_decision.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for decision. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      int _i_out=-1;
      Operate(m_trade_decision,m_decision_operation,_i_out);
      
      double _trade_decision=0.0;
      
      if(_i_out==0)
      {
         if(m_close.GetData(_x)>m_close.GetData(_x+Lookback))
         {
            _trade_decision=Indicator;
         }
         else if(m_close.GetData(_x)<m_close.GetData(_x+Lookback))
         {
            _trade_decision=100.0-Indicator;
         }
      }
      else if(_i_out==1)
      {
         if(m_close.GetData(_x)>m_close.GetData(_x+Lookback))
         {
            _trade_decision=100.0-Indicator;
         }
         else if(m_close.GetData(_x)<m_close.GetData(_x+Lookback))
         {
            _trade_decision=Indicator;
         }
      }
      
      return(int(round(_trade_decision)));
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CTrailingCT::Ordering(int Lookback)
   {
      m_low.Refresh(-1);
      m_high.Refresh(-1);
      m_close.Refresh(-1);
      
      double _c[],_l[],_h[];
      
      ArrayResize(_c,Lookback);ArrayInitialize(_c,0.0);
      ArrayResize(_l,Lookback);ArrayInitialize(_l,0.0);
      ArrayResize(_h,Lookback);ArrayInitialize(_h,0.0);
      
      for(int i=0;i<Lookback;i++)
      {
         _c[i]=m_close.GetData(i);
         _l[i]=m_low.GetData(Lookback+i);
         _h[i]=m_high.GetData(Lookback+i);
      }
      
      double _r_h=0.0,_r_l=0.0;
      
      if(MathCorrelationSpearman(_c,_l,_r_l) && MathCorrelationSpearman(_c,_h,_r_h))
      {
         if(_r_l>=__PHI){ LongIndex(5); }
         else if(_r_l>=1.0-__PHI){ LongIndex(4); }
         else if(_r_l>=0.0){ LongIndex(3); }
         else if(_r_l>=(-1.0+__PHI)){ LongIndex(2); }
         else if(_r_l>=(-__PHI)){ LongIndex(1); }
         else{ LongIndex(0);}
         
         if(_r_h>=__PHI){ ShortIndex(5); }
         else if(_r_h>=1.0-__PHI){ ShortIndex(4); }
         else if(_r_h>=0.0){ ShortIndex(3); }
         else if(_r_h>=(-1.0+__PHI)){ ShortIndex(2); }
         else if(_r_h>=(-__PHI)){ ShortIndex(1); }
         else{ ShortIndex(0);}
         
         return(true);
      }
      
      return(false);
   }
//+------------------------------------------------------------------+
