﻿#include "EADetector.mqh"

//--- Input Parameters
input int TestMode = 5;  // Test Mode (1-6)
input int Range1   = 1;  // Parameter 1
input int Range2   = 1;  // Parameter 2

//+------------------------------------------------------------------+
//| Test Mode 1: Simplest Usage (Global Function)                    |
//+------------------------------------------------------------------+
int TestMode1_Simple() {
  ::Print("========== Test Mode 1: Simplest Usage (Global Function) ==========");
  ::Print("Function: CheckDuplicate_OnePerSymbolPeriod()");
  ::Print("Action: Deny + Alert Box (OK button)");

  // Ultra-simple: One line of code for detection
  if (!CheckDuplicate_OnePerSymbolPeriod()) {
    return INIT_FAILED;
  }

  ::Print("Detection passed. Execution allowed.");
  return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Test Mode 2: One EA per Terminal (Global Function)               |
//+------------------------------------------------------------------+
int TestMode2_OnePerTerminal() {
  ::Print("========== Test Mode 2: Global Limit - One EA per Terminal ==========");
  ::Print("Function: CheckDuplicate_OnePerTerminal()");
  ::Print("Action: Deny + Alert Box (OK button)");

  if (!CheckDuplicate_OnePerTerminal()) {
    return INIT_FAILED;
  }

  ::Print("Detection passed. Execution allowed.");
  return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Test Mode 3: One EA per Symbol (Global Function)                 |
//+------------------------------------------------------------------+
int TestMode3_OnePerSymbol() {
  ::Print("========== Test Mode 3: Per Symbol Limit (Global Function) ==========");
  ::Print("Function: CheckDuplicate_OnePerSymbol()");
  ::Print("Action: Deny + Alert Box (OK button)");

  if (!CheckDuplicate_OnePerSymbol()) {
    return INIT_FAILED;
  }

  ::Print("Detection passed. Execution allowed.");
  return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Test Mode 4: Ask User to Continue (Global Function)              |
//+------------------------------------------------------------------+
int TestMode4_AskUser() {
  ::Print("========== Test Mode 4: Ask User Decision (Global Function) ==========");
  ::Print("Function: CheckDuplicate_AskUser()");
  ::Print("Action: User Prompt (Yes/No buttons)");

  if (!CheckDuplicate_AskUser()) {
    return INIT_FAILED;
  }

  ::Print("User chose to continue execution.");
  return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Test Mode 5: Unique Parameters Only (Global Function)            |
//+------------------------------------------------------------------+
int TestMode5_UniqueParams() {
  ::Print("========== Test Mode 5: Unique Parameters Only (Global Function) ==========");
  ::Print("Function: CheckDuplicate_UniqueParams()");
  ::Print("Action: Deny + Alert Box (OK button)");

  if (!CheckDuplicate_UniqueParams()) {
    return INIT_FAILED;
  }

  ::Print("Detection passed. Execution allowed.");
  return INIT_SUCCEEDED;
}

//+------------------------------------------------------------------+
//| Test Mode 6: Test All Advanced Features                          |
//+------------------------------------------------------------------+
int TestMode6_AllFunctions() {
  ::Print("========== Test Mode 6: Comprehensive Function Test ==========");

  SDuplicateDetectorConfig config;
  config.policy = POLICY_ONE_PER_SYMBOL_PERIOD;

  // Test IsRunning (Quick Check)
  bool isDuplicate = CEADuplicateDetector::IsRunning(config.policy);
  ::Print("IsRunning() result: ", isDuplicate ? "Duplicate Detected" : "No Duplicate Found");

  if (!isDuplicate) {
    ::Print("No duplicate EAs detected. Proceeding.");
    return INIT_SUCCEEDED;
  }

  // Get all duplicate chart IDs
  long duplicateCharts[];
  CEADuplicateDetector::GetDuplicateCharts(duplicateCharts, config.policy);
  ::Print("Found ", ::ArraySize(duplicateCharts), " duplicate EA instance(s):");

  for (int i = 0; i < ::ArraySize(duplicateCharts); i++) {
    ::Print("  Chart ", i + 1, ": ID=", duplicateCharts[i],
           " Symbol=", ::ChartSymbol(duplicateCharts[i]),
           " Period=", ::EnumToString(::ChartPeriod(duplicateCharts[i])));
  }

  // Test "Ask User" mode manually via config
  config.action = ACTION_ASK_USER;
  config.msgContent = "Duplicate EA detected:\n{info}\n\nDo you want to continue running?";

  SDuplicateDetectResult result = CEADuplicateDetector::Detect(config);

  ::Print("User Decision: ", result.userDecision == INIT_SUCCEEDED ? "Continue" : "Stop");

  return result.userDecision;
}

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit() {
  ::Print("");
  ::Print("======================================================");
  ::Print("EA Duplicate Detector Test (Simplified)");
  ::Print("Current Test Mode: ", TestMode);
  ::Print("Range1: ", Range1);
  ::Print("Range2: ", Range2);
  ::Print("======================================================");

  int result = INIT_SUCCEEDED;

  // Route to different tests based on TestMode
  switch(TestMode) {
    case 1:
      result = TestMode1_Simple();
      break;

    case 2:
      result = TestMode2_OnePerTerminal();
      break;

    case 3:
      result = TestMode3_OnePerSymbol();
      break;

    case 4:
      result = TestMode4_AskUser();
      break;

    case 5:
      result = TestMode5_UniqueParams();
      break;

    case 6:
      result = TestMode6_AllFunctions();
      break;

    default:
      ::Print("Invalid Test Mode: ", TestMode);
      ::Print("Please use a value between 1 and 6.");
      result = INIT_SUCCEEDED;
  }

  return result;
}
