/**
Optimal F Library, version 1.00

Copyright 2018, Anthony J. Garot
MQL5 Profile: https://www.mql5.com/en/users/tonegarot
My website: http://www.garot.com/trading/
*/

#ifndef OPTIMAL_F_MQH
#define OPTIMAL_F_MQH

#include <Arrays/ArrayDouble.mqh>

// global constants
const double	GEOM_MEAN_DEFAULT_VALUE=0.30;		// Return value if not enough trades
const int		GEOM_MEAN_MIN_TRADES=0;				// Make 0 for Ralph Vince's example to work
const int		DEBUG = 1;							// 0=quiet; 1=DEBUG mode

class COptimalF
{

private:

	double			m_largestLoss;					// Used for optimal F calculation
	CArrayDouble	*m_profitLoss;					// Holds all profit & loss amounts

public:

	// Constructor
	void COptimalF()
	{
		m_profitLoss=new CArrayDouble;
		ResetTradeCounts();
	}
	void ~COptimalF(void)
	{
		delete(m_profitLoss);						// destroy what you create
	}

	// Use for initializing the object
	// Also in unit tests
	void ResetTradeCounts(void)
	{
		m_largestLoss=0.0;
		if( ! m_profitLoss.Shutdown())
		{
			PrintFormat("%s(): CArrayDouble.Shutdown() error", __FUNCTION__);
		}
	}

	// Adds in a profit (or loss) after a trade is complete
	void AddProfitTrade(const double amount,const double swap=0.0)
	{
		m_profitLoss.Add(amount+swap);

		// Collect the worst loss (for Optimal f calculation)
		// m_largestLoss should be a negative #
		// Be careful with exactly $0.00 trades. Some consider this a loss; others don't.
		if(amount<0 && amount<m_largestLoss)
		{
			if ( DEBUG ) PrintFormat("m_largestLoss from [$%.2f] to [$%.2f]",m_largestLoss,amount);
			m_largestLoss=amount;
		}
	}

	// This calculates the Optimal f by Geometric Mean.
	// See "The Mathematics of Money Management" by Ralph Vince equation (1.11)
	// Adapted from Python code at:
	// http://www.seertrading.com/an-optimal-f-money-management-position-sizing-strategy-by-ralph-vince/
	double GeometricMeanOptimalF(void)
	{
		double TWR;
		double BestTWR=1.0;
		double f;
		int i;
		double optimalF=-1.0;			// initialize to an impossible value
		bool bestFound = false;

		// How many trades do we have so far?
		int totalTrades=m_profitLoss.Total();

		// Cannot do calcuation if we don't have a largest loss
		// or don't meet our minimum trades requirement.
		if(m_largestLoss==0.0 || totalTrades<GEOM_MEAN_MIN_TRADES)
		{
			return GEOM_MEAN_DEFAULT_VALUE;
		}

		// Loop using hundredths granularity.
		for(f=0.00; f<1.00; f+=0.01)
		{
			// Terminal Wealth Relative = return on your stake as a multiple.
			// A TWR of 10.55 means you would have made 10.55 times your original stake, or 955% profit.
			TWR=1.0;		// Reset to unity.

			// Loop through all historical trades
			for(i=0; i<totalTrades; i++)
			{
				// Need the profit or loss of every single trade.
				// These are stored in a CArrayDouble.
				const double tradeProfitLoss=m_profitLoss.At(i);

				// Holding period.
				// An HPR is 1 plus the gain or loss as a percentage.
				// A trade that profitted 10% --> HPR = 1.10.
				// A trade that lost 10%		--> HPR = 0.90.
				double HPR=1.0+(f *(-tradeProfitLoss/m_largestLoss));
				TWR*=HPR;
			}

			// Find the best f value (so far . . . )
			if(BestTWR<TWR)
			{
				bestFound=true;
				BestTWR = TWR;		// Hold the best
				optimalF = f;		// This is the best found through the iterations
				if ( DEBUG ) PrintFormat("(best) TWR [%.2f] f [%.2f]",TWR,f);
			}
			else
			{
				if(bestFound)
				{
					// Since the distribution increases, then descreases,
					// if we've already found the best value and are no
					// no longer switching out, we can leave the loop early.
					break;
				}
				if ( DEBUG ) PrintFormat(". . TWR [%.2f] f [%.2f]",TWR,f);
			}
		}

		// If for any reason we didn't get a valid optimalF, return the default.
		if(optimalF<0)
		{
			if ( DEBUG ) PrintFormat("Geometric Mean not primed. Using default value [%.2f]",GEOM_MEAN_DEFAULT_VALUE);
			optimalF=GEOM_MEAN_DEFAULT_VALUE;
		}

		if ( DEBUG ) PrintFormat(">> optimalF [%.2f] <<",optimalF);
		return optimalF;
	}
};

#endif // OPTIMAL_F_MQH
