//+------------------------------------------------------------------+
//|                                              InspectorDialog.mqh |
//|                                    Copyright (c) 2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                           https://www.mql5.com/ru/articles/7795/ |
//+------------------------------------------------------------------+

#include <ControlsPlus/Dialog.mqh>
#include <ControlsPlus/Button.mqh>
#include <ControlsPlus/Edit.mqh>
#include <ControlsPlus/CheckBox.mqh>
#include <ControlsPlus/Label.mqh>
#include <Layouts/LayoutDefines.mqh>
#include <Layouts/Box.mqh>
#include <Layouts/ComboBoxResizable.mqh>
#include <Layouts/CheckGroupResizable.mqh>
#include <Layouts/SpinEditResizable.mqh>
#include <Layouts/ComboBoxWebColors.mqh>
#include <Layouts/AppDialogResizable.mqh>
#include <Layouts/LayoutStdLib.mqh>
#include <Layouts/AutoPtr.mqh>
#include <Layouts/LayoutMonitors.mqh>
#include <Layouts/Sort.mqh>

// TODO:
//       detect default colors (for specific control type) and don't assign them explicitly
//       existing solution does not work reliably for compound controls/containers
//       because background belongs for subsidiary object and its color is assigned in Create method, not in constructor
// PRB:  CLabel doesn't have border to highlight user selection

//+------------------------------------------------------------------+
//| defines                                                          |
//+------------------------------------------------------------------+
#define BUTTON_WIDTH (100)
#define BUTTON_HEIGHT (20)

#define DEFAULT_MARGIN 0

#define ROW_HEIGHT (22)

#define ON_LAYOUT_SELECTION 88
#define ON_LAYOUT_MARGINS   89

//+-----------------------------------------------------------------------+
//|                           G L O B A L S                               |
//+-----------------------------------------------------------------------+

// user-friendly names
const string _types[] = {"Box H", "Box V", "Button", "Edit", "Label", "SpinEdit", "DatePicker", "ComboBox", "ListView", "CheckGroup", "RadioGroup"};
// actual class names
const string _rttis[] = {"CBoxH", "CBoxV", "CButton", "CEdit", "CLabel", "SpinEditResizable", "CDatePicker", "ComboBoxResizable", "ListViewResizable", "CheckGroupResizable", "RadioGroupResizable"};
// fast access codes/indices
      enum   _TYPES     {_BoxH,   _BoxV,   _Button,  _Edit,  _Label,  _SpinEdit,  _DatePicker,  _ComboBox,  _ListView,  _CheckGroup,  _RadioGroup};


//+-----------------------------------------------------------------------+
//| Custom Styler                                                         |
//+-----------------------------------------------------------------------+

class MyLayoutStyleable: public StdLayoutStyleable
{
  public:
    virtual void apply(CWnd *control, const STYLER_PHASE phase) override
    {
      if(phase == STYLE_PHASE_BEFORE_INIT)
      {
        CBox *box = dynamic_cast<CBox *>(control);
        if(box != NULL)
        {
          box.Margins(DEFAULT_MARGIN);
          box.HorizontalAlign(HORIZONTAL_ALIGN_CENTER_NOSIDES);
        }
      }
      else
      {
        CEdit *edit = dynamic_cast<CEdit *>(control);
        if(edit && edit.ReadOnly())
        {
          edit.ColorBackground(CONTROLS_CHECKGROUP_COLOR_BG);
        }
      }
    }
};


struct Properties
{
  string name;
  int type;
  int width;
  int height;
  int style;
  string text;
  color clr;
  int align;
  ushort margins[4];
};


class PropertySet
{
  public:
    StdValue<string> name;
    StdValue<int> type;
    StdValue<int> width;
    StdValue<int> height;
    StdValue<int> style; // VERTICAL_ALIGN / HORIZONTAL_ALIGN / ENUM_ALIGN_MODE
    StdValue<string> text;
    StdValue<color> clr;
    StdValue<int> align; // ENUM_WND_ALIGN_FLAGS + WND_ALIGN_CONTENT
    StdValue<ushort> margins[4];
    
    Properties flatten() const
    {
      Properties p = {0};
      string s = ~name;
      StringReplace(s, " ", "");
      p.name = s;
      p.type = ~type;
      p.width = ~width;
      p.height = ~height;
      p.style = ~style;
      p.text = ~text;
      p.clr = ~clr;
      p.align = ~align;
      p.margins[0] = ~margins[0];
      p.margins[1] = ~margins[1];
      p.margins[2] = ~margins[2];
      p.margins[3] = ~margins[3];
      return p;
    }
};


//+-----------------------------------------------------------------------+
//| Custom Cache with indexing support                                    |
//+-----------------------------------------------------------------------+

class InspectorDialog;

class MyStdLayoutCache: public StdLayoutCache
{
  protected:
    MyLayoutStyleable styler;
    InspectorDialog *parent;
    
    // fast access
    int index[];
    int start;
    
  public:
    MyStdLayoutCache(InspectorDialog *owner): parent(owner)
    {
    }

    virtual StdLayoutStyleable *getStyler() const override
    {
      return (StdLayoutStyleable *)&styler;
    }
    
    virtual CWnd *get(const long m) override
    {
      if(m < 0 && ArraySize(index) > 0)
      {
        int offset = (int)(-m - start);
        if(offset >= 0 && offset < ArraySize(index))
        {
          return StdLayoutCache::get(index[offset]);
        }
      }
      
      return StdLayoutCache::get(m);
    }
    
    void buildIndex()
    {
      start = parent.GetInstanceId();
      int stop = 0;
      for(int i = 0; i < cacheSize(); i++)
      {
        int id = (int)get(i).Id();
        if(id > stop) stop = id;
      }
      
      ArrayResize(index, stop - start + 1);
      ArrayInitialize(index, -1);
      for(int i = 0; i < cacheSize(); i++)
      {
        CWnd *wnd = get(i);
        index[(int)(wnd.Id() - start)] = i;
      }
    }
    
    virtual bool onEvent(const int event, CWnd *control) override
    {
      if(control)
      {
        if(event == ON_CHANGE)
        {
          Print("Change ", control.Name());
        }
        else if(event == ON_CLICK)
        {
          Print("Click ", control.Name());
          TestModeButton *button  = dynamic_cast<TestModeButton *>(control);
          if(button)
          {
            Print("Test mode: ", button.Pressed());
            designer.setTestMode(button.Pressed());
          }
        }
      }
      
      if(event == ON_LAYOUT_MARGINS)
      {
        for(int i = 0; i < cacheSize(); i++)
        {
          SpinEditPropertyShort *spin = dynamic_cast<SpinEditPropertyShort *>(get(i));
          if(spin)
          {
            spin.onEvent(ON_LAYOUT_MARGINS, NULL);
          }
        }
      }
      return false;
    }
};


//+-----------------------------------------------------------------------+
//| Application specific state monitors for controls                      |
//+-----------------------------------------------------------------------+

class ApplyButtonStateMonitor: public EnableStateMonitor
{
  // what's required to detect Apply button state
  const int NAME;
  const int TYPE;
  
  public:
    ApplyButtonStateMonitor(StdValue<string> *n, StdValue<int> *t): NAME(0), TYPE(1)
    {
      ArrayResize(sources, 2);
      sources[NAME] = n;
      sources[TYPE] = t;
    }

    virtual bool isEnabled(void) override
    {
      StdValue<string> *name = sources[NAME];
      StdValue<int> *type = sources[TYPE];
      return StringLen(~name) > 0 && ~type != -1 && ~name != "Client";
    }
};

class EnableStateMonitorOnType: public EnableStateMonitor
{
  protected:
    const int TYPE;
  public:
    EnableStateMonitorOnType(StdValue<int> *t): TYPE(0)
    {
      ArrayResize(sources, 1);
      sources[TYPE] = t;
    }
};

class StyleComboStateMonitor: public EnableStateMonitorOnType
{
  public:
    StyleComboStateMonitor(StdValue<int> *t): EnableStateMonitorOnType(t) {}

    virtual bool isEnabled(void) override
    {
      StdValue<int> *type = sources[TYPE];
      const int value = ~type;
      return value == _BoxH || value == _BoxV || value == _Edit;
    }
};

class ColorComboStateMonitor: public EnableStateMonitorOnType
{
  public:
    ColorComboStateMonitor(StdValue<int> *t): EnableStateMonitorOnType(t) {}

    virtual bool isEnabled(void) override
    {
      StdValue<int> *type = sources[TYPE];
      const int value = ~type;
      const bool enabled = value == _BoxH || value == _BoxV || value == _Button || value == _Edit;
      ComboBoxResizable *combo = dynamic_cast<ComboBoxResizable *>(control);
      if(combo && !enabled)
      {
        combo.Select(-1);
      }
      return enabled;
    }
};

class CaptionStateMonitor: public EnableStateMonitorOnType
{
  public:
    CaptionStateMonitor(StdValue<int> *t): EnableStateMonitorOnType(t) {}

    virtual bool isEnabled(void) override
    {
      StdValue<int> *type = sources[TYPE];
      const int value = ~type;
      const bool enabled = value == _Button || value == _Edit || value == _Label;
      CEdit *edit = dynamic_cast<CEdit *>(control);
      if(edit)
      {
        edit.ReadOnly(!enabled);
        edit.ColorBackground(enabled ? CONTROLS_EDIT_COLOR_BG : CONTROLS_DIALOG_COLOR_BG);
      }
      return enabled;
    }
};


//+-----------------------------------------------------------------------+
//| Controls to properties binding 1 to 1                                 |
//+-----------------------------------------------------------------------+

class DefaultWidth
{
  public:
    const static int size;
};

const static int DefaultWidth::size = BUTTON_WIDTH;

class DefaultHeight
{
  public:
    const static int size;
};

const static int DefaultHeight::size = BUTTON_HEIGHT;

class SpinEditProperty: public NotifiableProperty<SpinEditResizable,int>
{
  public:
    virtual int value() override
    {
      return Value();
    }
};

template<typename T>
class SpinEditPropertySize: public SpinEditProperty
{
  public:
    SpinEditPropertySize()
    {
      RTTI;
    }
    virtual bool onEvent(const int event, void *parent) override
    {
      SpinEditProperty::onEvent(event, parent);
      if(event == ON_CLICK)
      {
        if((TerminalInfoInteger(TERMINAL_KEYSTATE_SHIFT) & 0x80000000) != 0)
        {
          property = (int)T::size; // defaults BUTTON_WIDTH/BUTTON_HEIGHT;
          SpinEditResizable *ptr = (SpinEditResizable *)property.backlink();
          ptr.Value(~property);
        }
      }
      return true;
    };
};

class SpinEditPropertyShort: public NotifiableProperty<SpinEditResizable,ushort>
{
  public:
    SpinEditPropertyShort()
    {
      RTTI;
    }
    
    virtual ushort value() override
    {
      return (ushort)Value();
    }

    virtual bool onEvent(const int event, void *parent) override
    {
      static int defaultMargin = 0;
      
      NotifiableProperty<SpinEditResizable,ushort>::onEvent(event, parent);
      if(event == ON_CLICK)
      {
        const bool shift = (TerminalInfoInteger(TERMINAL_KEYSTATE_SHIFT) & 0x80000000) != 0;
        const bool control = (TerminalInfoInteger(TERMINAL_KEYSTATE_CONTROL) & 0x80000000) != 0;
        if(shift || control)
        {
          if(control) defaultMargin = ~property;
          else defaultMargin = 0;
          EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_MARGINS, 0, 0.0, NULL);
        }
      }
      else if(event == ON_LAYOUT_MARGINS)
      {
        SpinEditResizable *ptr = (SpinEditResizable *)property.backlink();
        ptr.Value(defaultMargin); // reset
      }
      return true;
    };
    
};

class EditProperty: public NotifiableProperty<CEdit,string>
{
  public:
    virtual string value() override
    {
      return Text();
    }
};

class ComboBoxProperty: public NotifiableProperty<ComboBoxResizable,int>
{
  public:
    virtual int value() override
    {
      return (int)Value();
    }
};

class ComboBoxColorProperty: public NotifiableProperty<ComboBoxWebColors,color>
{
  public:
    virtual color value() override
    {
      return (color)Value();
    }
};

class ApplyButton: public Notifiable<CButton>
{
  public:
    virtual bool onEvent(const int event, void *parent) override
    {
      if(event == ON_CLICK)
      {
        Properties p = inspector.getProperties().flatten();
        designer.inject(p);
        ChartRedraw();
        return true;
      }
      return false;
    };
};

class ExportButton: public Notifiable<CButton>
{
  public:
    virtual bool onEvent(const int event, void *parent) override
    {
      if(event == ON_CLICK)
      {
        designer.Export();
        return true;
      }
      return false;
    };
};

class AlignCheckGroupProperty: public NotifiableProperty<CheckGroupResizable,int>
{
  public:
    virtual int value() override
    {
      return (int)Value();
    }
    
    virtual bool onEvent(const int event, void *parent) override
    {
      if((TerminalInfoInteger(TERMINAL_KEYSTATE_SHIFT) & 0x80000000) != 0)
      {
        if(Value() > ~property) // new is larger than old, something is switched on
        {
          for(int i = 0; i < 4; i++)
          {
            int value = (1 << i);
            Check(i, value);
          }
        }
        else // switched off
        {
          for(int i = 0; i < 4; i++)
          {
            Check(i, 0);
          }
        }
      }
      return NotifiableProperty<CheckGroupResizable,int>::onEvent(event, parent);
    };
};

class TestModeButton: public CButton
{
};

//+-----------------------------------------------------------------------+
//| Main dialog window with controls                                      |
//+-----------------------------------------------------------------------+

class InspectorDialog: public AppDialogResizable
{
  private:
    CBox *m_main;
    ComboBoxResizable *selector;

    MyStdLayoutCache *cache;
    ApplyButtonStateMonitor *applyMonitor;
    StyleComboStateMonitor *styleMonitor;
    ColorComboStateMonitor *colorMonitor;
    CaptionStateMonitor *captionMonitor;

    PropertySet props;
    
  public:
    InspectorDialog(void);
    ~InspectorDialog(void);

    // general event handler
    virtual bool OnEvent(const int id, const long &lparam, const double &dparam, const string &sparam) override;
    virtual bool Run(void) override;

    bool CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2);
    
    MyStdLayoutCache *getCache(void) const
    {
      return cache;
    }
    
    PropertySet *getProperties(void) const
    {
      return (PropertySet *)&props;
    }

    static int GetTypeByRTTI(const string rtti);
    static color DefaultBgColor(const _TYPES t);

  protected:
    virtual void SelfAdjustment(const bool restore = false) override;

    bool OnRefresh(const long &lparam, const double &dparam, const string &sparam);
    bool OnRemoteSelection();
    bool OnBrowserSelection();
    
    void UpdateSelector(StdLayoutCache *remote, CWnd *ptr = NULL);

    template<typename T>
    static color GetBgColor();
};


//+------------------------------------------------------------------+
//| Event handling                                                   |
//+------------------------------------------------------------------+

EVENT_MAP_BEGIN(InspectorDialog)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_END_EDIT, cache, EditProperty)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CHANGE, cache, SpinEditProperty)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CLICK, cache, SpinEditProperty)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CHANGE, cache, SpinEditPropertyShort)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CLICK, cache, SpinEditPropertyShort) // request margins reset
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CHANGE, cache, ComboBoxProperty)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CHANGE, cache, ComboBoxColorProperty)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CHANGE, cache, AlignCheckGroupProperty)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CLICK, cache, ApplyButton)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CLICK, cache, ExportButton)
  ON_EVENT_LAYOUT_ARRAY(ON_CLICK, cache) // default (stub)
  ON_EVENT_LAYOUT_ARRAY(ON_LAYOUT_MARGINS, cache) // broadcast (reset all margins to 0 or equal values)
  ON_EVENT(ON_CHANGE, selector, OnBrowserSelection)
  
  ON_EXTERNAL_EVENT(ON_LAYOUT_REFRESH, OnRefresh)
  ON_NO_ID_EVENT(ON_LAYOUT_SELECTION, OnRemoteSelection)
EVENT_MAP_END(AppDialogResizable)

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
InspectorDialog::InspectorDialog(void)
{
  RTTI;
  cache = new MyStdLayoutCache(&this);
  applyMonitor = new ApplyButtonStateMonitor(&props.name, &props.type);
  styleMonitor = new StyleComboStateMonitor(&props.type);
  colorMonitor = new ColorComboStateMonitor(&props.type);
  captionMonitor = new CaptionStateMonitor(&props.type);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
InspectorDialog::~InspectorDialog(void)
{
  delete captionMonitor;
  delete colorMonitor;
  delete styleMonitor;
  delete applyMonitor;
  delete cache;
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool InspectorDialog::CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2)
{
  StdLayoutBase::setCache(cache); // assign the cache object to store implicit objects
  {
    _layout<InspectorDialog> dialog(this, name, x1, y1, x2, y2);
    
    // ------------------
    
    {
      _layout<CBoxV> clientArea("main", ClientAreaWidth(), ClientAreaHeight());
      m_main = clientArea.get();
      clientArea <= WND_ALIGN_CLIENT;
      clientArea <= clrYellow;
      clientArea <= VERTICAL_ALIGN_STACK;

      {
        _layout<CBoxH> row("SelectorRow", ClientAreaWidth(), ROW_HEIGHT, (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH));
        row <= clrBlack;

        {
          _layout<ComboBoxResizable> combo("SelectorCombo", BUTTON_WIDTH, BUTTON_HEIGHT);
          selector = combo.get();

          _layout<CEdit> label("SelectorLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Selector:");
          label <= true <= ALIGN_RIGHT;
        }
      }

      {
        _layout<CBoxH> row("NameRow", ClientAreaWidth(), ROW_HEIGHT, (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH));
        row <= clrDarkOrchid;

        {
          // "\0" is a hack because editbox always starts with a space (non-zero length string)
          _layout<EditProperty> edit("NameEdit", BUTTON_WIDTH, BUTTON_HEIGHT, "\0");
          edit.attach(&props.name);
          _layout<CEdit> label("NameLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Name:");
          label <= true <= ALIGN_RIGHT;
        }
      }

      {
        _layout<CBoxH> row("TypeRow", ClientAreaWidth(), ROW_HEIGHT);
        row <= clrBlue <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          ArrayItemGenerator<ComboBoxProperty,string> gen(_types);
          _layout<ComboBoxProperty> combo("TypeCombo", BUTTON_WIDTH, BUTTON_HEIGHT);
          combo <= gen <= 0;
          combo.attach(&props.type);
          _layout<CEdit> label("TypeLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Type:");
          label <= true <= ALIGN_RIGHT;
        }
      }

      {
        _layout<CBoxH> row("WidthRow", ClientAreaWidth(), ROW_HEIGHT);
        row <= clrDodgerBlue <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          _layout<SpinEditPropertySize<DefaultWidth>> edit1("WidthSpinEdit", BUTTON_WIDTH, BUTTON_HEIGHT);
          edit1["min"] <= 10;
          edit1["max"] <= 500;
          edit1["value"] <= BUTTON_WIDTH;
          edit1.attach(&props.width);
          
          _layout<CEdit> label1("WidthLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Width:");
          label1 <= true <= ALIGN_RIGHT;
        }
      }

      {
        _layout<CBoxH> row("HeightRow", ClientAreaWidth(), ROW_HEIGHT);
        row <= clrDodgerBlue <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          _layout<SpinEditPropertySize<DefaultHeight>> edit2("HeightEdit", BUTTON_WIDTH, BUTTON_HEIGHT);
          edit2["min"] <= 10;
          edit2["max"] <= 500;
          edit2["value"] <= BUTTON_HEIGHT;
          edit2.attach(&props.height);

          _layout<CEdit> label2("HeightLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Height:");
          label2 <= true <= ALIGN_RIGHT;
        }
      }

      {
        _layout<CBoxH> row("StyleRow", ClientAreaWidth(), ROW_HEIGHT);
        row <= clrLightGreen <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          string styles[] = {"center", "justify", "left/top", "right/bottom", "stack"};
          ArrayItemGenerator<ComboBoxProperty,string> gen(styles);
          _layout<ComboBoxProperty> combo("StyleCombo", BUTTON_WIDTH, BUTTON_HEIGHT);
          combo["enable"] <= gen <= 0;
          combo.attach(&props.style);
          styleMonitor.attach(combo.get());

          _layout<CEdit> label("StyleLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Style:");
          label <= true <= ALIGN_RIGHT;
        }
      }

      {
        _layout<CBoxH> row("TextRow", ClientAreaWidth(), ROW_HEIGHT);
        row <= clrGold <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          _layout<EditProperty> edit("TextEdit", BUTTON_WIDTH, BUTTON_HEIGHT, "\0");
          edit <= true; // read-only from beginning (for BoxH)
          edit.attach(&props.text);
          captionMonitor.attach(edit.get());

          _layout<CEdit> label("TextLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Text:");
          label <= true <= ALIGN_RIGHT;
        }
      }

      {
        _layout<CBoxH> row("ColorsRow", ClientAreaWidth(), ROW_HEIGHT);
        row <= clrSalmon <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          _layout<ComboBoxColorProperty> colors("ColorCombo", BUTTON_WIDTH, BUTTON_HEIGHT);
          colors.attach(&props.clr);
          colorMonitor.attach(colors.get());
          
          _layout<CEdit> label("ColorLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Color:");
          label <= true <= ALIGN_RIGHT;
        }
      }

      {
        _layout<CBoxH> row("AlignmentTitleRow", ClientAreaWidth(), ROW_HEIGHT);
        row <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          _layout<CEdit> label2("MarginsTitleLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Margins");
          label2 <= true;
          _layout<CEdit> label1("AlignmentTitleLabel", BUTTON_WIDTH, BUTTON_HEIGHT, "Alignment");
          label1 <= true;
        }
      }

      {
        _layout<CBoxH> row("AlignmentRow", ClientAreaWidth(), BUTTON_HEIGHT * 5);
        row <= clrDarkKhaki <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH);

        {
          {
            _layout<CBoxV> left("CheckBoxesColumn", BUTTON_WIDTH, BUTTON_HEIGHT * 5);
            {
              _layout<AlignCheckGroupProperty> sides("AlignSides", BUTTON_WIDTH, BUTTON_HEIGHT * 5 + 5);
              string a[] = {"Left", "Top", "Right", "Bottom", "Fill"};
              ArrayItemGenerator<AlignCheckGroupProperty,string> gen(a, true);
              sides <= gen <= WND_ALIGN_WIDTH;
              sides.attach(&props.align);
            }
          }
          {
            _layout<CBoxV> right("MarginsColumn", BUTTON_WIDTH, BUTTON_HEIGHT * 5);
            {
              _layout<TestModeButton> common("TestMode", BUTTON_WIDTH, BUTTON_HEIGHT, WND_ALIGN_WIDTH);
              common <= true <= "Test mode";

              _layout<SpinEditPropertyShort> spins("Margins", 4);
              spins["height"] <= CONTROLS_LIST_ITEM_HEIGHT <= WND_ALIGN_WIDTH;
              spins["min"] <= 0;
              spins["max"] <= 500;
              spins["value"] <= 1 <= 0; // spinedit does not show value which equals to default 0, if not altered and returned back
              
              spins.attach(props.margins);
            }
          }
        }
      }

      {
        _layout<CBoxH> row("ActionRow", ClientAreaWidth(), ROW_HEIGHT, (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_WIDTH));
        row <= clrCyan;

        {
          _layout<ExportButton> button2("Export", BUTTON_WIDTH, BUTTON_HEIGHT);
          _layout<ApplyButton> button1("Apply", BUTTON_WIDTH, BUTTON_HEIGHT);
          button1["enable"] <= false;
          applyMonitor.attach(button1.get());
        }
      }

    }
    
    // ----------------------
  }

  SelfAdjustment();
  EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_REFRESH, 0, 0.0, m_instance_id);

  return true;
}

void InspectorDialog::SelfAdjustment(const bool restore = false)
{
  m_main.Pack();
  Rebound(Rect());
}

bool InspectorDialog::OnRefresh(const long &lparam, const double &dparam, const string &sparam)
{
  if(sparam == m_instance_id)
  {
    m_main.Pack();
    Rebound(Rect());
    return true;
  }

  // refresh comes from the form (something was added)
  UpdateSelector(designer.getCache());
  
  return false;
}

bool InspectorDialog::Run(void)
{
  bool result = AppDialogResizable::Run();
  if(result)
  {
    cache.buildIndex();
  }
  return result;
}

static int InspectorDialog::GetTypeByRTTI(const string rtti)
{
  for(int i = 0; i < ArraySize(_rttis); i++)
  {
    if(StringFind(rtti, _rttis[i]) >= 0) return i;
  }
  return -1;
}

template<typename T>
static color InspectorDialog::GetBgColor()
{
  AutoPtr<T> auto(new T());
  return (~auto).ColorBackground();
}

static color InspectorDialog::DefaultBgColor(const _TYPES t)
{
  switch(t)
  {
    // CWndObj's
    case _Button:
      return GetBgColor<CButton>();
    case _Edit:
      return GetBgColor<CEdit>();
    case _Label:
      return GetBgColor<CLabel>();               // user color doesn't work, but color is saved in the object and reported correctly
    
    // CWndContainer's - no ColorBackground      // user color doesn't work
    //case _SpinEdit:
    //  return GetBgColor<SpinEditResizable>();
    //case _DatePicker:
    //  return GetBgColor<CDatePicker>();
    //case _ComboBox:
    //  return GetBgColor<ComboBoxResizable>();
    
    // CWndClient's (problem: background color is assigned after construction)
    case _BoxH:
    case _BoxV:
      return GetBgColor<CBox>();
    case _ListView:
      return GetBgColor<ListViewResizable>(); // actual: white, reported: none; user color doesn't work
    case _CheckGroup:
      return GetBgColor<CheckGroupResizable>(); // actual: 3*247, reported: none; user color changes only panel, items left unchanged
    case _RadioGroup:
      return GetBgColor<RadioGroupResizable>();
  }
  return clrNONE;
}


void InspectorDialog::UpdateSelector(StdLayoutCache *remote, CWnd *ptr = NULL)
{
  selector.ItemsClear();

  string names[];
  ArrayResize(names, remote.cacheSize());
  
  int k = 0;
  for(int i = 0; i < remote.cacheSize(); i++)
  {
    CWnd *wnd = remote.get(i);
    int type = GetTypeByRTTI(wnd._rtti);
    if(type > -1)
    {
      names[k++] = StringSubstr(wnd.Name(), 5) + " " + _types[type] + ShortToString((ushort)(i + 1));
    }
  }
  ArrayResize(names, k);
  DefaultCompare<string> cmp;
  SORT::Sort(names, cmp);

  for(int i = 0; i < k; i++)
  {
    ushort index = StringGetCharacter(names[i], StringLen(names[i]) - 1);
    selector.AddItem(StringSubstr(names[i], 0, StringLen(names[i]) - 1), index - 1);
    // CListView is patched to use default values -1 instead of 0
    // 0 is valid value here, but standard CListView replaces it with total number of items
    // another way how this can be solved is by excessive call of ItemUpdate
    // selector.ItemUpdate(i, StringSubstr(names[i], 0, StringLen(names[i]) - 1), index - 1);
  }

  /*
  for(int i = 0; i < remote.cacheSize(); i++)
  {
    CWnd *wnd = remote.get(i);
    int type = GetTypeByRTTI(wnd._rtti);
    if(type > -1)
    {
      selector.AddItem(StringSubstr(wnd.Name(), 5) + " " + _types[type], i);
    }
  }
  */
  
  if(ptr)
  {
    selector.SelectByValue(remote.indexOf(ptr));
  }
}

bool InspectorDialog::OnBrowserSelection()
{
  DefaultStdLayoutCache *remote = designer.getCache();
  if(remote.getSelected() != remote.get(selector.Value()))
  {
    remote.select(remote.get(selector.Value()));
  }
  return true;
}

// All changes must be reflected in properties/fields by this handler
// because they normally filled by user clicks/changes (standard events),
// but here they updated internally by object selection

bool InspectorDialog::OnRemoteSelection()
{
  DefaultStdLayoutCache *remote = designer.getCache();
  CWnd *ptr = remote.getSelected();

  UpdateSelector(remote, ptr);
  
  if(ptr)
  {
    /*
      string name;
      int type;
      int width;
      int height;
      int style;
      string text;
      color clr;
      int align;
      ushort margins[4];
    */
    string purename = StringSubstr(ptr.Name(), 5); // cut instance id prefix
    CWndObj *x = dynamic_cast<CWndObj *>(props.name.backlink());
    if(x) x.Text(purename);
    props.name = purename;
    
    int t = -1;
    ComboBoxResizable *types = dynamic_cast<ComboBoxResizable *>(props.type.backlink());
    if(types)
    {
      t = GetTypeByRTTI(ptr._rtti);
      types.Select(t);
      props.type = t;
    }
    
    // width and height
    SpinEditResizable *w = dynamic_cast<SpinEditResizable *>(props.width.backlink());
    w.Value(ptr.Width());
    props.width = ptr.Width();

    SpinEditResizable *h = dynamic_cast<SpinEditResizable *>(props.height.backlink());
    h.Value(ptr.Height());
    props.height = ptr.Height();

    // specific accessors
    CWndObj *source = dynamic_cast<CWndObj *>(ptr);
    CWndClient *client = dynamic_cast<CWndClient *>(ptr);
    
    // text
    CWndObj *y = dynamic_cast<CWndObj *>(props.text.backlink());
   
    if(y)
    {
      if(source)
      {
        string s = source.Text();
        props.text = s;
        if(StringLen(s) == 0) y.Text("\0");
        else y.Text(s);
      }
      else
      {
        y.Text("\0");
        props.text = "";
      }
    }
    
    // style
    ComboBoxResizable *style = dynamic_cast<ComboBoxResizable *>(props.style.backlink());
    CEdit *edit = dynamic_cast<CEdit *>(ptr);
    if(edit)
    {
      ENUM_ALIGN_MODE am = edit.TextAlign();
      int value = 0;
      switch(am)
      {
        case ALIGN_LEFT:
          value = 2;
          break;
        case ALIGN_CENTER:
          value = 0;
          break;
        case ALIGN_RIGHT:
          value = 3;
          break;
      }
      style.Select(value);
      props.style = value;
    }
    else
    {
      CBox *box = dynamic_cast<CBox *>(ptr);
      if(box)
      {
        if(box.LayoutStyle() == LAYOUT_STYLE_VERTICAL)
        {
          style.Select(box.VerticalAlign());
          props.style = (int)box.VerticalAlign();
        }
        else
        if(box.LayoutStyle() == LAYOUT_STYLE_HORIZONTAL)
        {
          style.Select(box.HorizontalAlign());
          props.style = (int)box.HorizontalAlign();
        }
      }
    }
    
    // alignment
    CheckGroupResizable *checks = dynamic_cast<CheckGroupResizable *>(props.align.backlink());
    if(checks)
    {
      int a = ptr.Alignment(); // ENUM_WND_ALIGN_FLAGS
      if((a & WND_ALIGN_CONTENT) != 0)
      {
        a -= 128;
        a |= 16;
      }
      
      props.align = a;

      PackedRect r(ptr.Margins());
      
      // checks.Value(a); doesn't work (changes not translated to checkboxes)
      for(int i = 0; i < 5; i++)
      {
        int value = (1 << i) & a;
        checks.Check(i, value);
        
        if(i < 4)
        {
          SpinEditResizable *spin = dynamic_cast<SpinEditResizable *>(props.margins[i].backlink());
          spin.Value(r.parts[i]);
          props.margins[i] = (ushort)r.parts[i];
        }
      }
    }
    
    // color
    props.clr = clrNONE;
    color clr = clrNONE;
    if(source) clr = source.ColorBackground();
    else if(client) clr = client.ColorBackground();

    ComboBoxResizable *colors = dynamic_cast<ComboBoxResizable *>(props.clr.backlink());
    // Print(EnumToString((_TYPES)t), " default: ", DefaultBgColor((_TYPES)t), " custom: ", clr);
    
    if(clr != clrNONE && clr != DefaultBgColor((_TYPES)t))
    {
      if(colors.SelectByValue(clr))
      {
        props.clr = clr;
      }
      else
      {
        colors.Select(-1);
        props.clr = clrNONE;
      }
    }
    else
    {
      colors.Select(-1);
      props.clr = clrNONE;
    }
  }
  
  return true;
}
