//+------------------------------------------------------------------+
//|                                                 DesignerForm.mqh |
//|                                    Copyright (c) 2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                           https://www.mql5.com/ru/articles/7795/ |
//+------------------------------------------------------------------+

#include <ControlsPlus/Dialog.mqh>
#include <ControlsPlus/Button.mqh>
#include <ControlsPlus/Edit.mqh>
#include <ControlsPlus/CheckBox.mqh>
#include <Layouts/LayoutDefines.mqh>
#include <Layouts/Box.mqh>
#include <Layouts/ComboBoxResizable.mqh>
#include <Layouts/CheckGroupResizable.mqh>
#include <Layouts/RadioGroupResizable.mqh>
#include <Layouts/ListViewResizable.mqh>
#include <Layouts/SpinEditResizable.mqh>
#include <Layouts/AppDialogResizable.mqh>
#include <Layouts/LayoutStdLib.mqh>
#include <Layouts/LayoutConverters.mqh>
#include <Layouts/LayoutExporter.mqh>
#include <Layouts/AutoPtr.mqh>
#include <Layouts/TraceObjects.mqh>

#ifdef DEFAULT_MARGIN
#undef DEFAULT_MARGIN
#endif

//+------------------------------------------------------------------+
//| defines                                                          |
//+------------------------------------------------------------------+
#define DEFAULT_MARGIN 0
#define DUMMY_ITEM_NUMBER 11

class DesignerForm;

class DefaultLayoutStyleable: public StdLayoutStyleable
{
  public:
    virtual void apply(CWnd *control, const STYLER_PHASE phase) override
    {
      if(phase == STYLE_PHASE_BEFORE_INIT)
      {
        control.Margins(DEFAULT_MARGIN);
      }
    }
};

class DefaultStdLayoutCache: public StdLayoutCache
{
  protected:
    DefaultLayoutStyleable styler;
    DesignerForm *parent;
    CWnd *selected;
    
    bool highlight(CWnd *control, const color clr)
    {
      CWndObj *obj = dynamic_cast<CWndObj *>(control);
      if(obj != NULL)
      {
        obj.ColorBorder(clr);
        return true;
      }
      else
      {
        CWndClient *client = dynamic_cast<CWndClient *>(control);
        if(client != NULL)
        {
          client.ColorBorder(clr);
          return true;
        }
        else // last chance: compound control
        {
          CWndContainer *container = dynamic_cast<CWndContainer *>(control);
          if(container)
          {
            for(int i = 0; i < container.ControlsTotal(); i++)
            {
              CWndObj *child = dynamic_cast<CWndObj *>(container.Control(i));
              if(child != NULL)
              {
                child.ColorBorder(clr);
              }
            }
          }
        }
      }
      return false;
    }

  public:
    DefaultStdLayoutCache(DesignerForm *owner): parent(owner) {}
    
    void select(CWnd *element)
    {
      if(parent.isTestMode()) return;
      if(selected)
      {
        highlight(selected, CONTROLS_BUTTON_COLOR_BORDER);
      }
      selected = element;
      if(selected)
      {
        Print("Auto: ", selected.Name(), " -> ", selected._rtti);
        highlight(selected, clrRed);
        EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_SELECTION, 0, 0.0, NULL);
      }
    }
    
    CWnd *getSelected(void) const
    {
      if(!find(selected))
      {
        CWnd *result = findParent(selected);
        if(result) return result;
      }
    
      return selected;
    }

    virtual StdLayoutStyleable *getStyler() const override
    {
      return (StdLayoutStyleable *)&styler;
    }

    virtual bool onEvent(const int event, CWnd *control) override
    {
      if(parent.isTestMode()) return false;
      
      if(control != NULL)
      {
        highlight(selected, CONTROLS_BUTTON_COLOR_BORDER);

        CWnd *element = control;
        if(!find(element)) // this is an auxiliary object, not a compound control
        {
          element = findParent(element); // get actual GUI element
        }
        
        if(element == NULL)
        {
          Print("Can't find GUI element for ", control._rtti + " / " + control.Name());
          return true;
        }
        
        if(selected == element)
        {
          if(MessageBox("Delete " + element._rtti + " / " + element.Name() + "?", "Confirm", MB_OKCANCEL) == IDOK)
          {
            CWndContainer *container;
            container = dynamic_cast<CWndContainer *>(findParent(element));
            if(container)
            {
              revoke(element); // deep remove of all references (with subtree) from cache
              container.Delete(element); // delete all subtree of wnd-objects
              
              CBox *box = dynamic_cast<CBox *>(container);
              if(box) box.Pack();
            }
            else
            {
              Print("Can't find parent container");
            }
            selected = NULL;
            EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_SELECTION, 0, 0.0, NULL);
            return true;
          }
        }
        selected = element; // control;

        const bool b = highlight(selected, clrRed);
        Print(control.Name(), " -> ", element._rtti, " / ", element.Name(), " / ", element.Id());
        EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_SELECTION, 0, 0.0, NULL);

        return true;
      }
      return false;
    }
};


//+-----------------------------------------------------------------------+
//| Main dialog window with controls                                      |
//+-----------------------------------------------------------------------+
class DesignerForm: public AppDialogResizable
{
  private:
    DefaultStdLayoutCache *cache;
    CWndClient *client;
    CBox *pMain;
    bool testMode;
    
    template<typename T>
    T *applyProperties(T *ptr, const Properties &props)
    {
      ptr <= LayoutConverters::boxAlignBits2enum(props.align);

      static const string sides[4] = {"left", "top", "right", "bottom"}; // NB. properties hardcoded, keep in sync
      for(int i = 0; i < 4; i++)
      {
        ptr[sides[i]] <= (int)props.margins[i];
      }
      
      if(StringLen(props.text))
      {
        ptr <= props.text;
      }
      else
      {
        ptr <= props.name;
      }
      
      if(props.clr != clrNONE) // can make transparent controls if allow clrNONE
      {
        ptr["background"] <= props.clr;
      }
      
      return ptr;
    }

    template<typename T,typename C>
    T *populate(T *ptr)
    {
      StdGroupItemGenerator<C> gen(DUMMY_ITEM_NUMBER, "Item");
      ptr <= gen;
      return ptr;
    }

    StdLayoutBase *getPtr(const Properties &props)
    {
      switch(props.type)
      {
        case _BoxH:
          {
            _layout<CBoxH> *temp = applyProperties(new _layout<CBoxH>(props.name, props.width, props.height), props);
            temp <= (HORIZONTAL_ALIGN)props.style;
            return temp;
          }
        case _BoxV:
          {
            _layout<CBoxV> *temp = applyProperties(new _layout<CBoxV>(props.name, props.width, props.height), props);
            temp <= (VERTICAL_ALIGN)props.style;
            return temp;
          }
        case _Button:
          return applyProperties(new _layout<CButton>(props.name, props.width, props.height), props);
        case _Edit:
          {
            _layout<CEdit> *temp = applyProperties(new _layout<CEdit>(props.name, props.width, props.height), props);
            temp <= (ENUM_ALIGN_MODE)LayoutConverters::style2textAlign(props.style);
            return temp;
          }
        case _Label:
          return applyProperties(new _layout<CLabel>(props.name, props.width, props.height), props);
        case _SpinEdit:
          {
            _layout<SpinEditResizable> *temp = applyProperties(new _layout<SpinEditResizable>(props.name, props.width, props.height), props);
            temp["min"] <= 0;
            temp["max"] <= DUMMY_ITEM_NUMBER;
            temp["value"] <= 1 <= 0;
            return temp;
          }
        case _DatePicker:
          return applyProperties(new _layout<CDatePicker>(props.name, props.width, props.height, TimeCurrent()), props);
        case _ComboBox:
          return populate<_layout<ComboBoxResizable>,ComboBoxResizable>(applyProperties(new _layout<ComboBoxResizable>(props.name, props.width, props.height), props));
        case _ListView:
          return populate<_layout<ListViewResizable>,ListViewResizable>(applyProperties(new _layout<ListViewResizable>(props.name, props.width, props.height), props));
        case _CheckGroup:
          return populate<_layout<CheckGroupResizable>,CheckGroupResizable>(applyProperties(new _layout<CheckGroupResizable>(props.name, props.width, props.height), props));
        case _RadioGroup:
          return populate<_layout<RadioGroupResizable>,RadioGroupResizable>(applyProperties(new _layout<RadioGroupResizable>(props.name, props.width, props.height), props));
      }
          
      return NULL;
    }
    
    bool isSameType(_TYPES t, CWnd *ptr)
    {
      string s = EnumToString(t);
      string r[];
      if(StringSplit(s, '_', r) != 2) return false;
      return StringFind(ptr._rtti, r[1]) != -1;
    }
    
    void update(CWnd *ptr, const Properties &props)
    {
      ptr.Width(props.width);
      ptr.Height(props.height);
      ptr.Alignment(LayoutConverters::boxAlignBits2enum(props.align));
      ptr.Margins(props.margins[0], props.margins[1], props.margins[2], props.margins[3]);
      CWndObj *obj = dynamic_cast<CWndObj *>(ptr);
      if(obj)
      {
        obj.Text(props.text);
        if(props.clr != clrNONE) // can make transparent controls if allow clrNONE
        {
          obj.ColorBackground(props.clr);
        }
        else
        {
          color clr = InspectorDialog::DefaultBgColor(_TYPES(props.type));
          if(clr != clrNONE)
          {
            obj.ColorBackground(clr);
          }
        }
      }
      else
      {
        CWndClient *clnt = dynamic_cast<CWndClient *>(ptr);
        if(clnt)
        {
          if(props.clr != clrNONE)
          {
            clnt.ColorBackground(props.clr);
          }
          else
          {
            color clr = InspectorDialog::DefaultBgColor(_TYPES(props.type));
            clnt.ColorBackground(clr);
          }
        }
      }
      
      CBoxH *boxh = dynamic_cast<CBoxH *>(ptr);
      if(boxh)
      {
        boxh.HorizontalAlign((HORIZONTAL_ALIGN)props.style);
        boxh.Pack();
        return;
      }
      CBoxV *boxv = dynamic_cast<CBoxV *>(ptr);
      if(boxv)
      {
        boxv.VerticalAlign((VERTICAL_ALIGN)props.style);
        boxv.Pack();
        return;
      }
      CEdit *edit = dynamic_cast<CEdit *>(ptr);
      if(edit)
      {
        edit.TextAlign(LayoutConverters::style2textAlign(props.style));
        return;
      }
    }
    
    
  public:
    DesignerForm(void);
    ~DesignerForm(void);

    bool CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2);
    
    // general event handler
    virtual bool OnEvent(const int id, const long &lparam, const double &dparam, const string &sparam);

    DefaultStdLayoutCache *getCache(void) const
    {
      return cache;
    }
    
    void inject(Properties &props)
    {
      CWnd *ptr = cache.get(props.name);
      if(ptr != NULL)
      {
        if(!isSameType(_TYPES(props.type), ptr))
        {
          Print("The name ", props.name, " is already used: ", ptr._rtti);
        }
        else
        {
          CWnd *sel = cache.getSelected();
          if(ptr == sel)
          {
            if(cache.indexOf(ptr) == 0) // client area
            {
              for(int i = 0; i < 4; i++) // keep default margins where unspecified
              {
                if((props.align & (1 << i)) == 0)
                {
                  props.margins[i] = DEFAULT_MARGIN;
                }
              }
              props.align |= WND_ALIGN_CLIENT;
            }
            update(ptr, props);
            Rebound(Rect());
            EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_REFRESH, 0, 0.0, m_instance_id);
          }
          else
          {
            Print("Inspector name: ", props.name, " does not correspond to selected: ", (sel ? sel.Name() : "(null)"));
          }
        }
        return;
      }
      
      CBox *box = dynamic_cast<CBox *>(cache.getSelected());
      
      if(box == NULL) box = cache.findParent(cache.getSelected());
      
      if(box)
      {
        CWnd *added;
        StdLayoutBase::setCache(cache);
        {
          _layout<CBox> injectionPanel(box, box.Name());
        
          {
            AutoPtr<StdLayoutBase> base(getPtr(props));
            added = (~base).get();
            added.Id(rand() + ((long)rand() << 32));
          }
        }
        box.Pack();
        cache.select(added);
      }
      else // this is top-level client area
      {
        if(cache.cacheSize() == 0 && (props.type != _BoxH && props.type != _BoxV))
        {
          Alert("First element must be container, given: ", EnumToString((_TYPES)props.type));
          return;
        }
        
        if(cache.cacheSize() > 0)
        {
          ptr = cache.getSelected();
          Alert("Please select a box (container) to inject element into it, selected: ", (ptr != NULL ? ptr._rtti : "(null)"));
          return;
        }
        
        if(props.align != WND_ALIGN_CLIENT)
        {
          props.align |= WND_ALIGN_CLIENT;
          ArrayFill(props.margins, 0, 4, DEFAULT_MARGIN);
        }
        
        CBox *added;
        StdLayoutBase::setCache(cache);
        {
          _layout<DesignerForm> dialog(this, Name());

          {
            AutoPtr<StdLayoutBase> base(getPtr(props));
            added = (~base).get();
            added.Id(rand() + ((long)rand() << 32));
          }
        }
        added.Pack();
        cache.select(added);
      }
      
      EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_REFRESH, 0, 0.0, m_instance_id);
    }
    
    void setTestMode(const bool enable)
    {
      cache.select(NULL);
      testMode = enable;
    }
    
    bool isTestMode(void) const
    {
      return testMode;
    }
    
    void Import(const string filename)
    {
      LayoutExporter exporter(&this, designer.getCache());
      int w, h;
      if(!exporter.openFileBinary(filename, w, h))
      {
        return;
      }
      
      Width(w);
      Height(h);
      
      RubbArray<StdLayoutBase *> stack;

      CWnd *added;
      CBox *box = NULL;
      StdLayoutBase::setCache(cache);
      {
        _layout<DesignerForm> dialog(this, Name());
        
        StdLayoutBase stub;
        
        while(exporter.hasNextBinary())
        {
          int d = exporter.readDelimiterBinary();
          Properties props;
          if(d == '{' || d == '[')
          {
            exporter.readNextBinary(props);
            StdLayoutBase *ptr = getPtr(props);
            if(props.type != -1)
            {
              if(ptr)
              {
                added = ptr.get();
                added.Id(rand() + ((long)rand() << 32));
                stack << ptr;
              
                if(box == NULL) box = added;
              }
              else
              {
                Print("Can't create ", props.name);
                stack << &stub;
              }
            }
            else
            {
              Print("Unknown type for ", props.name);
              stack << &stub;
            }
          }
          else if(d == '}' || d == ']')
          {
            StdLayoutBase *ptr = stack.pop();
            if(ptr != &stub)
            {
              delete ptr;
            }
          }
        }
      }
      box.Show();
      box.Pack();
      
      exporter.closeFileBinary();
      EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_REFRESH, 0, 0.0, m_instance_id);
    }

    void Export(void);

  protected:
    CBox *GetMainContainer(void);
    virtual void SelfAdjustment(const bool restore = false) override;
    bool OnRefresh(const long &lparam, const double &dparam, const string &sparam);
    void PrintCache(void);
    void OnClickClient(void)
    {
      if(!testMode)
      {
        cache.select(client);
      }
    }
};


//+------------------------------------------------------------------+
//| Event handling                                                   |
//+------------------------------------------------------------------+

EVENT_MAP_BEGIN(DesignerForm)
  ON_EVENT_LAYOUT_ARRAY(ON_CLICK, cache)
  ON_EVENT(ON_CLICK, client, OnClickClient)
  ON_EXTERNAL_EVENT(ON_LAYOUT_REFRESH, OnRefresh)
EVENT_MAP_END(AppDialogResizable)

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
DesignerForm::DesignerForm(void)
{
  RTTI;
  pMain = NULL;
  cache = new DefaultStdLayoutCache(&this);
  testMode = false;
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
DesignerForm::~DesignerForm(void)
{
  delete cache;
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool DesignerForm::CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2)
{
  StdLayoutBase::setCache(cache);
  {
    _layout<DesignerForm> dialog(this, name, x1, y1, x2, y2);
    
    // -------------------------------------
    // this is here all GUI stuff is landing
    // -------------------------------------
  }

  EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_REFRESH, 0, 0.0, m_instance_id);

  return true;
}

void DesignerForm::Export(void)
{
  if(cache.cacheSize() == 0)
  {
    Import("layout");
  }
  else
  {
    LayoutExporter exporter(&this, cache);
    exporter.saveToFile("layout" + (string)(long)TimeLocal());
  }
}

void DesignerForm::PrintCache(void)
{
  cache.print();
}

CBox *DesignerForm::GetMainContainer(void)
{
  for(int i = 0; i < ControlsTotal(); i++)
  {
    client = dynamic_cast<CWndClient *>(Control(i));
    if(client != NULL)
    {
      for(int j = 0; j < client.ControlsTotal(); j++)
      {
        CBox *box = dynamic_cast<CBox *>(client.Control(j));
        if(box != NULL)
        {
          return box;
        }
      }
      break;
    }
  }
  return NULL;
}

void DesignerForm::SelfAdjustment(const bool restore = false)
{
  if(pMain == NULL)
  {
    pMain = GetMainContainer();
    if(pMain)
    {
      Print("Main container: ", pMain._rtti, " ", pMain.Name());
    }
  }
  
  if(pMain)
  {
    pMain.Pack();
    Rebound(Rect());
  }
}

bool DesignerForm::OnRefresh(const long &lparam, const double &dparam, const string &sparam)
{
  if(sparam == m_instance_id)
  {
    SelfAdjustment();

    /*
    // convenient debug utils
    if((TerminalInfoInteger(TERMINAL_KEYSTATE_CAPSLOCK) & 1) != 0)
    {
      //TraceObjects();
      PrintCache();
      TraverseWindows(&this);
    }
    */
    
    return true;
  }
  return false;
}
