//+------------------------------------------------------------------+
//|                                                  DynamicForm.mqh |
//|                                    Copyright (c) 2020, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                           https://www.mql5.com/ru/articles/7734/ |
//+------------------------------------------------------------------+

#include <ControlsPlus/Dialog.mqh>
#include <ControlsPlus/Button.mqh>
#include <ControlsPlus/Edit.mqh>
#include <ControlsPlus/CheckBox.mqh>
#include <Layouts/LayoutDefines.mqh>
#include <Layouts/Box.mqh>
#include <Layouts/ComboBoxResizable.mqh>
#include <Layouts/RadioGroupResizable.mqh>
#include <Layouts/AppDialogResizable.mqh>
#include <Layouts/LayoutStdLib.mqh>
#include <Layouts/LayoutExporter.mqh>
#include <Layouts/AutoPtr.mqh>

//+------------------------------------------------------------------+
//| defines                                                          |
//+------------------------------------------------------------------+
// buttons
#define BUTTON_WIDTH (100)
#define BUTTON_HEIGHT (20)
// editbox
#define EDIT_HEIGHT (20)
// groups
#define GROUP_WIDTH (150)
#define LIST_HEIGHT (179)
#define RADIO_HEIGHT (56)
#define CHECK_HEIGHT (63)
#define DEFAULT_MARGIN 5


class DynamicForm;

class MyLayoutStyleable: public StdLayoutStyleable
{
  public:
    virtual void apply(CWnd *control, const STYLER_PHASE phase) override
    {
      if(phase == STYLE_PHASE_BEFORE_INIT)
      {
        control.Margins(DEFAULT_MARGIN);
      }
    }
};

class MyStdLayoutCache: public StdLayoutCache
{
  protected:
    MyLayoutStyleable styler;
    DynamicForm *parent;
    CWnd *selected;
    
    bool highlight(CWnd *control, const color clr)
    {
      CWndObj *obj = dynamic_cast<CWndObj *>(control);
      if(obj != NULL)
      {
        obj.ColorBorder(clr);
        return true;
      }
      else
      {
        CWndClient *client = dynamic_cast<CWndClient *>(control);
        if(client != NULL)
        {
          client.ColorBorder(clr);
          return true;
        }
      }
      return false;
    }

  public:
    MyStdLayoutCache(DynamicForm *owner): parent(owner) {}

    virtual StdLayoutStyleable *getStyler() const override
    {
      return (StdLayoutStyleable *)&styler;
    }

    virtual bool onEvent(const int event, CWnd *control) override
    {
      if(control != NULL)
      {
        highlight(selected, CONTROLS_BUTTON_COLOR_BORDER);

        CWnd *element = control;
        if(!find(element)) // this is an auxiliary object, not a compound control
        {
          element = findParent(control); // get actual GUI element
        }
        
        if(element == NULL)
        {
          Print("Can't find GUI element for ", control._rtti + " / " + control.Name());
          return true;
        }
        
        if(selected == control)
        {
          if(MessageBox("Delete " + element._rtti + " / " + element.Name() + "?", "Confirm", MB_OKCANCEL) == IDOK)
          {
            CWndContainer *container;
            container = dynamic_cast<CWndContainer *>(findParent(element));
            if(container)
            {
              revoke(element); // deep remove of all references (with subtree) from cache
              container.Delete(element); // delete all subtree of wnd-objects
              
              CBox *box = dynamic_cast<CBox *>(container);
              if(box) box.Pack();
            }
            selected = NULL;
            return true;
          }
        }
        selected = control;

        const bool b = highlight(selected, clrRed);
        Print(control.Name(), " -> ", element._rtti, " / ", element.Name(), " / ", b);

        return true;
      }
      return false;
    }
};


class NotifiableButton: public Notifiable<CButton>
{
    static int count;
    
    StdLayoutBase *getPtr(const int value)
    {
      switch(value)
      {
        case 0:
          return new _layout<CButton>("More" + (string)count++, BUTTON_WIDTH, BUTTON_HEIGHT);
        case 1:
          return new _layout<CCheckBox>("More" + (string)count++, BUTTON_WIDTH, BUTTON_HEIGHT);
        case 2:
          return new _layout<CEdit>("More" + (string)count++, BUTTON_WIDTH, BUTTON_HEIGHT);
      }
      return NULL;
    }
    
  public:
    virtual bool onEvent(const int event, void *anything) override
    {
      DynamicForm *parent = dynamic_cast<DynamicForm *>(anything);
      MyStdLayoutCache *cache = parent.getCache();
      StdLayoutBase::setCache(cache);
      CBox *box = cache.get("column1");
      if(box != NULL)
      {
        // put target box to the stack by retrieving it from the cache
        _layout<CBox> injectionPanel(box, box.Name());
        
        {
          CRadioGroup *selector = cache.get("selector");
          if(selector != NULL)
          {
            const int value = (int)selector.Value();
            if(value != -1)
            {
              AutoPtr<StdLayoutBase> base(getPtr(value));
              (~base).get().Id(rand() + (rand() << 32));
            }
          }
        }
        box.Pack();
      }
      
      return true;
    }
};


static int NotifiableButton::count = 0;

//+-----------------------------------------------------------------------+
//| Main dialog window with controls                                      |
//+-----------------------------------------------------------------------+
class DynamicForm: public AppDialogResizable
{
  private:
    CBox *m_main;
    CButton m_button3; // this is intentionally made as automatic variable, so the button object is undeletable from chart ;-)
    
    MyStdLayoutCache *cache;
    
  public:
    DynamicForm(void);
    ~DynamicForm(void);

    bool CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2);
    
    // general event handler
    virtual bool OnEvent(const int id, const long &lparam, const double &dparam, const string &sparam);

    MyStdLayoutCache *getCache(void) const
    {
      return cache;
    }

  protected:
    // specific handlers
    void OnClickButton3(void);
    
    void PrintCache();

    virtual void SelfAdjustment(const bool restore = false) override;
    bool OnRefresh();
};


//+------------------------------------------------------------------+
//| Event handling                                                   |
//+------------------------------------------------------------------+

EVENT_MAP_BEGIN(DynamicForm)
  ON_EVENT(ON_CLICK, m_button3, OnClickButton3)
  ON_EVENT_LAYOUT_CTRL_DLG(ON_CLICK, cache, NotifiableButton)
  ON_EVENT_LAYOUT_ARRAY(ON_CLICK, cache)
  ON_NO_ID_EVENT(ON_LAYOUT_REFRESH, OnRefresh) // "panic" message
EVENT_MAP_END(AppDialogResizable)

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
DynamicForm::DynamicForm(void)
{
  RTTI;
  cache = new MyStdLayoutCache(&this);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
DynamicForm::~DynamicForm(void)
{
  delete cache;
}

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool DynamicForm::CreateLayout(const long chart, const string name, const int subwin, const int x1, const int y1, const int x2, const int y2)
{
  StdLayoutBase::setCache(cache); // assign the cache object to store implicit objects
  {
    _layout<DynamicForm> dialog(this, name, x1, y1, x2, y2);
    
    // ------------------
    
    {
      // example of implicit object in the cache
      _layout<CBoxV> clientArea("main", ClientAreaWidth(), ClientAreaHeight());
      m_main = clientArea.get();
      clientArea <= WND_ALIGN_CLIENT <= PackedRect(10, 10, 10, 10);
      clientArea["background"] <= clrYellow <= VERTICAL_ALIGN_TOP;

      {
        _layout<CBoxH> buttonRow("buttonrow", ClientAreaWidth(), BUTTON_HEIGHT * 5);
        buttonRow <= 5.0 <= (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_TOP|WND_ALIGN_WIDTH);
        buttonRow["background"] <= clrCyan;

        {
          // these 2 buttons will be rendered in reverse order (destruction order)
          _layout<CButton> button3(m_button3, "Export", BUTTON_WIDTH, BUTTON_HEIGHT);
          _layout<NotifiableButton> button2("Inject", BUTTON_WIDTH, BUTTON_HEIGHT);
        }
      }
      
      {
        _layout<CBoxH> buttonRow("buttonrow2", ClientAreaWidth(), ClientAreaHeight(),
          (ENUM_WND_ALIGN_FLAGS)(WND_ALIGN_CONTENT|WND_ALIGN_CLIENT));
        buttonRow["top"] <= BUTTON_HEIGHT * 5;

        {
          {
            _layout<CBoxV> column("column1", GROUP_WIDTH, 100, WND_ALIGN_HEIGHT);
            //column <= 10.0f; // float means box padding (as opposed to margins/alignment of controls); don't mix both
            column <= clrGray;
            {
              // dynamically created controls will be injected here
            }
          }

          {
            _layout<CBoxH> column("column2", GROUP_WIDTH, 100, WND_ALIGN_HEIGHT);
          
            _layout<RadioGroupResizable> selector("selector", GROUP_WIDTH, CHECK_HEIGHT);
            selector <= WND_ALIGN_HEIGHT;
            string types[3] = {"Button", "CheckBox", "Edit"};
            ArrayItemGenerator<RadioGroupResizable,string> ctrls(types);
            selector <= ctrls;
          }
        }
      }
    }

    // ----------------------
  }

  SelfAdjustment();
  EventChartCustom(CONTROLS_SELF_MESSAGE, ON_LAYOUT_REFRESH, 0, 0.0, NULL);

  return true;
}


void DynamicForm::PrintCache()
{
  /*
  for(int i = 0; i < cache.cacheSize(); i++)
  {
    CWnd *wnd = cache.get(i);
    Print(wnd._rtti, " / ", wnd.Name());
  }
  */
  LayoutExporter exporter(&this);
  Print((string)exporter.saveToFile(cache, "layout.txt") + " elements saved to file");
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void DynamicForm::OnClickButton3(void)
{
  PrintCache();
}

void DynamicForm::SelfAdjustment(const bool restore = false)
{
  m_main.Pack();
  Rebound(Rect());
}

bool DynamicForm::OnRefresh()
{
  SelfAdjustment();
  return true;
}
