//+------------------------------------------------------------------+
//|                                                    ClrObject.mqh |
//|                                             Copyright GF1D, 2010 |
//|                                             garf1eldhome@mail.ru |
//+------------------------------------------------------------------+
#property copyright "GF1D, 2010"
#property link      "garf1eldhome@mail.ru"

#include "..\..\AdoErrors.mqh"

//--------------------------------------------------------------------
#import "AdoSuite.dll"
   long CreateManagedObject(const string, const string, string&, string&);
   void DestroyManagedObject(const long, string&, string&);
#import

//--------------------------------------------------------------------
/// \brief  \~english Represents CLR Object.
///
/// \~english Includes neccessary methods for creating and disposing managed objects, exception handling. Abstract class
class CClrObject
{
private:
// variables
   bool _IsCreated, _IsAssigned;
   long _ClrHandle;
   string _MqlTypeName;

protected:
   
// properties
   
   /// \brief  \~english Gets type string of inherited class
   const string MqlTypeName() { return _MqlTypeName; }
   /// \brief  \~english Sets type string of inherited class
   void MqlTypeName(const string value) { _MqlTypeName = value; }
   
// methods
   
   /// \brief  \~english Creates CLR object
   /// \~english \param  asmName   short assembly name: System, System.Data etc
   /// \~english \param  typeName  full type name eg System.String, System.Data.DataTable etc
   void CreateClrObject(const string asmName, const string typeName);   
   /// \brief  \~english Destroys CLR object. Called automatically in desctructor, so dont call it explictly!
   void DestroyClrObject();
   
// events

   /// \brief  \~english Called before object is being created. Virtual
   /// \~english \param isCanceling  bool variable, passed by a reference. If set value to false, then object creation will be suppressed
   /// \~english \param creating    when true indicates that object is creating, otherwise object is assigning using CClrObject::Assign
   virtual void OnObjectCreating(bool& isCanceling, bool creating = true)   {}
   /// \brief  \~english Called after CLR object was created
   virtual void OnObjectCreated()                        {}
   /// \brief  \~english Called before object is being destroyed. Virtual
   virtual void OnObjectDestroying()                     {}
   /// \brief  \~english Called after CLR object was destroyed
   virtual void OnObjectDestroyed()                      {}
   
   /// \brief  \~english Called when an exception occurs. Virtual
   /// \~english \param method    method name where the exception was thrown
   /// \~english \param type      exception type. Usually one of .NET types
   /// \~english \param message   exception message. Describes error details
   /// \~english \param mqlErr    appropriate mql error equivalent. ADOERR_FIRST by default
  virtual void OnClrException(const string method, const string type, const string message, const ushort mqlErr);
   
public: 
   /// \brief  \~english constructor
   CClrObject() { _MqlTypeName = "CClrObject"; }
   /// \brief  \~english destructor
   ~CClrObject() { DestroyClrObject(); }
   
// properties
   
   /// \brief  \~english Returns pointer for GCHandle, catching the object
   const long ClrHandle() { return _ClrHandle; }
   /// \brief  \~english Indicates whether object was assigned
   const bool IsAssigned() { return _IsAssigned; }
   /// \brief  \~english Indicates whether object was created
   const bool IsCreated() { return _IsCreated; }
   
// methods

   /// \brief  \~english Assigns this object to an existing CLR object
   /// \~english \param handle       pointer to GCHanlde with object
   /// \~english \param autoDestroy  Indicates whether CLR object has to be destroyed with appropriate lrObject, falso - if object must be in memory. Default is false.
   void Assign(const long handle, const bool autoDestroy); 
};

//--------------------------------------------------------------------
void CClrObject::CreateClrObject(const string asmName, const string typeName)
{
   bool isCanceling = false;
   
   OnObjectCreating(isCanceling, true);
   
   if (isCanceling) return;
   
   string exType = "", exMsg = "";
   StringInit(exType, 64);
   StringInit(exMsg, 256);

   _ClrHandle = CreateManagedObject(asmName, typeName, exType, exMsg);
   
   if (exType != "") 
   {
      _IsCreated = false;
      OnClrException("CreateClrObject", exType, exMsg);
   }
   else _IsCreated = true;
   _IsAssigned = false;
   
   OnObjectCreated();

}

//--------------------------------------------------------------------
CClrObject::DestroyClrObject(void)
{
   if (!_IsCreated) return;

   OnObjectDestroying();
   
   string exType = "", exMsg = "";
   StringInit(exType, 64);
   StringInit(exMsg, 256);
 
   DestroyManagedObject(_ClrHandle, exType, exMsg);
   
   _IsCreated = false;
   
   if (exType != "") 
      OnClrException("DestroyClrObject", exType, exMsg);
   
   OnObjectDestroyed();
}

//--------------------------------------------------------------------
CClrObject::Assign(const long handle, const bool autoDestroy = false)
{
   bool isCanceling = false;
   OnObjectCreating(isCanceling, false);
   
   if (isCanceling) return;
    
   _ClrHandle = handle;
   _IsCreated = autoDestroy;
   _IsAssigned = true;
  
   OnObjectCreated();
}

//--------------------------------------------------------------------
CClrObject::OnClrException(const string method, const string type, const string message, const ushort mqlErr = ADOERR_FIRST)
{
   Alert("Method ", _MqlTypeName, "::", method, " made an exception of type ", type, ":\r\n", message);
   SetUserError(mqlErr);
}
