//|------------------------------------------------------------------+
//|                                               CSOMDisplayRGB.mqh |
//|                                    Copyright (c) 2018, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                           https://www.mql5.com/ru/articles/5473/ |
//|------------------------------------------------------------------+

#include <CSOM/CSOMDisplay.mqh>

class CSOMDisplayRGB: public CSOMDisplay
{
  protected:
    int indexRGB[];
    bool enabled;
    bool showrgb;

    void CalculateRGB(const int ind, const int count, double &sum, int &col) const;
  
  public:
    void EnableRGB(const int &vector[]);
    void DisableRGB() { enabled = false; };
    virtual void RenderOutput() override;
    virtual string GetNodeAsString(const int node_index, const int plane) const override;
    virtual void Reset() override;
};

void CSOMDisplayRGB::Reset()
{
  CSOMDisplay::Reset();
  enabled = false;
  showrgb = false;
}

void CSOMDisplayRGB::EnableRGB(const int &vector[])
{
  if(!m_allocated) return;

  enabled = true;
  
  int n = ArraySize(vector);
  
  if(n == 0)
  {
    n = m_dimension;
  }
  else
  {
    n = MathMin(m_dimension, n);
  }
  
  ArrayResize(indexRGB, n);
  ArrayInitialize(indexRGB, 0);

  uint count = 0;
  for(int i = 0; i < n; i++)
  {
    if(i < ArraySize(vector) && vector[i] != 0)
    {
      indexRGB[i] = vector[i] > 0 ? i + 1 : -(i + 1);
      count++;
    }
  }
  
  if(count == 0)
  {
    for(int i = 0; i < n; i++)
    {
      indexRGB[i] = i + 1;
    }
  }
  
  m_min[DIM_OUTPUT] = DBL_MAX;
  m_max[DIM_OUTPUT] = -DBL_MAX;
  for(int i = 0; i < ArraySize(m_node); i++)
  {
    for(int j = 0; j < n; j++)
    {
      double w = 0;
      if(indexRGB[j] > 0)
      {
        w = m_node[i].GetWeight((int)indexRGB[j] - 1);
      }
      else if(indexRGB[j] < 0)
      {
        w = -m_node[i].GetWeight(-(int)indexRGB[j] - 1);
      }
      if(w < m_min[DIM_OUTPUT]) m_min[DIM_OUTPUT] = w;
      if(w > m_max[DIM_OUTPUT]) m_max[DIM_OUTPUT] = w;
    }
  }
}

void CSOMDisplayRGB::CalculateRGB(const int ind, const int count, double &sum, int &col) const
{
  static double w[];
  ArrayResize(w, ArraySize(indexRGB));
  
  double range = m_max[DIM_OUTPUT] - m_min[DIM_OUTPUT];
  double wc[3] = {0};
  int nc = 0;
  
  sum = 0;

  for(int p = 0; p < ArraySize(indexRGB); p++)
  {
    w[p] = 0;
    if(indexRGB[p] != 0)
    {
      w[p] = (indexRGB[p] > 0) ? m_node[ind].GetWeight((int)indexRGB[p] - 1) : -m_node[ind].GetWeight(-(int)indexRGB[p] - 1);
      double x = MathMin(1, MathMax((w[p] - m_min[DIM_OUTPUT]) / range, 0));
      sum += x * x;
      if(nc < 3)
      {
        wc[nc++] = w[p];
      }
    }
  }
  sum /= count;

  if(count == 3)
  {
    col = RGB256((int)MathMax(255 * (wc[0] - m_min[DIM_OUTPUT]) / range, 0), (int)MathMax(255 * (wc[1] - m_min[DIM_OUTPUT]) / range, 0), (int)MathMax(255 * (wc[2] - m_min[DIM_OUTPUT]) / range, 0));
  }
  else
  {
    int hsv = (int)(255 * MathSqrt(sum));
    col = RGB256(hsv, hsv, hsv);
  }
}

void CSOMDisplayRGB::RenderOutput()
{
  if(!enabled)
  {
    CSOMDisplay::RenderOutput();
    showrgb = false;
    return;
  }

  const int k = DIM_OUTPUT;
  
  if(m_max[k] <= 0) return;

  int ind = 0;
  
  color backclr = (color)ChartGetInteger(ChartID(), CHART_COLOR_BACKGROUND);
  m_bmp[k].Erase(backclr);
  
  int count = 0;
  for(int p = 0; p < ArraySize(indexRGB); p++)
  {
    if(indexRGB[p] != 0) count++;
  }
  
  if(count == 0) return; // unexpected
  
  double maxsum = 0;
  int maxi = -1, maxj = -1;
  int maxind = -1;
  
  for(int i = 0; i < m_xcells; i++)
  {
    for(int j = 0; j < m_ycells; j++)
    {
      double sum = 0;
      int col = 0;

      CalculateRGB(ind, count, sum, col);
      
      RenderCell(k, col, ind, (j % 2 == 0), m_node[ind].IsSelected());

      if(sum > maxsum)
      {
        maxsum = sum;
        maxi = i;
        maxj = j;
        maxind = ind;
      }

      ind++;
    }
  }
  
  if(maxind > -1)
  {
    Print("Brightest point [", maxi, ",", maxj, "]");
    double code[];
    m_node[maxind].GetCodeVector(code);
    Denormalize(code);
    ArrayPrint(code);
    ShowNode(maxind, count == 3 ? "RGB" : "BRIGHT", DIM_OUTPUT);
  }
  
  showrgb = true;
  
  RenderTitle(k, m_bmpw + (m_hexCells ? m_cellwidth / 2 : 0));
}

string CSOMDisplayRGB::GetNodeAsString(const int index, const int plane) const
{
  if(!showrgb || plane != DIM_OUTPUT)
  {
    return CSOMDisplay::GetNodeAsString(index, plane);
  }

  int count = 0;
  for(int p = 0; p < ArraySize(indexRGB); p++)
  {
    if(indexRGB[p] != 0) count++;
  }
  
  if(count == 0) return CSOMDisplay::GetNodeAsString(index, plane);

  double sum = 0;
  int col = 0;
  
  CalculateRGB(index, count, sum, col);
  
  if(count == 3)
  {
    return StringFormat("0x%06X: r=%d, g=%d, b=%d", col, col & 0xFF, ((col >> 8) & 0xFF), ((col >> 16) & 0xFF));
  }
  else
  {
    return StringFormat("0x%06X: r=%d, g=%d, b=%d", col, col & 0xFF, ((col >> 8) & 0xFF), ((col >> 16) & 0xFF));
  }
  return "n/a";
}
