//|------------------------------------------------------------------+
//|                                                  CSOMDisplay.mqh |
//|                                    Copyright (c) 2018, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                           https://www.mql5.com/ru/articles/5472/ |
//|                           https://www.mql5.com/ru/articles/5473/ |
//|                                                            v.1.1 |
//|------------------------------------------------------------------+

#include <CSOM/CSOM.mqh>

#include <Canvas/Canvas.mqh>


#define LEGEND_HEIGHT 15

enum ColorSchemes
{
  Black_White = 0,
  Blue_Green_Red = 1,
  Red_Black_Green = 2,
  Red_White_Green = 3
};

class CSOMDisplay: public CSOM
{
  protected:
    // display settings
    uchar Palette[768];
    int m_bmpw, m_bmph; // BMP-image sizes
    int m_cellwidth;
    int m_cellheight;
    CCanvas m_bmp[];    // BMP-image objects (use helper class)
    ColorSchemes m_clrSchema; // color gradient
    int m_maxpict;      // number of images in a row
    bool m_bShowBorder;
    bool m_bShowTitle;
  
    int m_x0, m_y0;     // display origin
    int m_firstImageBar;
    
  protected:
    virtual void RenderCell(const int img, const int col, const int node_index, const bool cell_even, const bool selected = false);
    virtual void InverseCell(const int img, const int node_index, const bool cell_even);
    virtual void ShowPatternHelper(double &vector[], const string name = NULL, const int plane = -1) const;
    virtual void Inverse(const int node_index, const bool cell_even);
    virtual void RenderTitle(const int m, const int w);
    virtual void RenderOutput();
    virtual void HideChart();
    virtual string GetNodeAsString(const int node_index, const int plane) const;
    
  public:
    CSOMDisplay(): CSOM() {};
    
    virtual bool Init(const int xc, const int yc, const int bmpw, const int bmph, const int maxpict = 4, const ColorSchemes clrSchema = Blue_Green_Red, const bool bhex = true, const bool bborder = false, const bool btitle = true);
    virtual bool DisplayInit(const int bmpw, const int bmph, const int maxpict = 4, const ColorSchemes clrSchema = Blue_Green_Red, const bool bborder = false, const bool btitle = true);
    virtual void Reframe(const int xincrement, const int yincrement) override;
    virtual void Reset() override;
    
    // Display
    virtual void Render(const bool inProgress = false);
    virtual void ShowBMP(const bool saveToFile = false);
    virtual void HideBMP();
    virtual void ShowAllPatterns(uint start = 0, uint end = 0) const;
    virtual void ShowAllNodes() const;
    virtual void ShowPattern(const double &vector[], const string name = NULL, const int plane = -1) const;
    virtual void ShowNode(const int ind, const string name = NULL, const int plane = -1) const;
    virtual void ShowClusters(int limit = -1) const;
    virtual void ProgressUpdate() override;

    void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam);

    static void ColToRGB(int col, int &r, int &g, int &b); // extract RGB components from color value
    static int RGB256(int r, int g, int b) { return(r + 256 * g + 65536 * b); } // construct color from RGB components

  protected:
    void Blend(uchar c1, uchar c2, uchar r1, uchar g1, uchar b1, uchar r2, uchar g2, uchar b2); // create gradient palette
    int GetPalColor(uchar index); // get color value by palette index
    
    void CalcPixelCoordinates(const int x1, const int y1, int &xc, int &yc) const;
    int GetNodeIndex(const string name, const long x, const long y, const long dx, const long dy, int &px, int &py) const;
    void Rewind();
};


void CSOMDisplay::Reset()
{
  CSOM::Reset();
  HideBMP();
}

bool CSOMDisplay::DisplayInit(const int bmpw, const int bmph, const int maxpict = 4, const ColorSchemes clrSchema = Blue_Green_Red, const bool bborder = false, const bool btitle = true)
{
  m_x0 = 10;
  m_y0 = 25; // space for comments

  m_bShowBorder = bborder;
  m_bShowTitle = btitle;
  m_maxpict = maxpict;
  m_clrSchema = clrSchema;
  m_bmpw = bmpw;
  m_bmph = bmph;

  m_cellwidth = m_bmpw / m_xcells;
  m_cellheight = m_bmph / m_ycells;

  HideChart();
  
  ChartSetInteger(ChartID(), CHART_SCALEFIX, true);
  int bars = (int)ChartGetInteger(0, CHART_WIDTH_IN_BARS);
  int pixels = (int)ChartGetInteger(0, CHART_WIDTH_IN_PIXELS);
  double pixelsPerBar = 1.0 * pixels / bars;
  
  datetime bar0time = TimeCurrent() / PeriodSeconds() * PeriodSeconds();
  int needbars = (int)((m_bmpw / pixelsPerBar * 1.1) * (m_dimension + EXTRA_DIMENSIONS + 1)) + 10 /*extra*/;
  datetime times[];
  
  if(maxpict == 0)
  {
    ChartSetDouble(ChartID(), CHART_FIXED_MAX, 1);
    ChartSetDouble(ChartID(), CHART_FIXED_MIN, 0);
#ifdef SOM_VERBOSE
    Print("pixelsPerBar=", pixelsPerBar, " ", bars, " ", pixels, " barsPerImage=", m_bmpw / pixelsPerBar);
#endif
    if(CopyTime(_Symbol, PERIOD_CURRENT, 0, needbars, times) != needbars)
    {
      Print("No required info for ", needbars, " bars times");
    }
  }
  else
  {
    ChartSetDouble(ChartID(), CHART_FIXED_MAX, 1);
    ChartSetDouble(ChartID(), CHART_FIXED_MIN, 1);
  }
  
  double price = 1.0;
  
  ArrayResize(m_bmp, m_dimension + EXTRA_DIMENSIONS); // allocate bitmaps
  for(int i = 0; i < m_dimension + EXTRA_DIMENSIONS; i++)
  {
    string name = /*m_sID +*/ "SOM_" + (string)i + "_" + m_titles[i];
    if(maxpict > 0)
    {
      int pos1 = i / m_maxpict;
      int pos2 = i % m_maxpict;
      if(!m_hexCells)
        m_bmp[i].CreateBitmapLabel(name, m_x0 + pos2 * (m_bmpw + m_cellwidth), m_y0 + pos1 * (LEGEND_HEIGHT + m_bmph + m_cellheight), m_bmpw, m_bmph + LEGEND_HEIGHT);
      else
        m_bmp[i].CreateBitmapLabel(name, m_x0 + pos2 * (m_bmpw + m_cellwidth), m_y0 + pos1 * (LEGEND_HEIGHT + m_bmph + m_cellheight), m_bmpw + m_cellwidth / 2, m_bmph + m_cellheight / 2 + LEGEND_HEIGHT);
    }
    else
    {
      int barno = needbars - (int)((m_bmpw / pixelsPerBar * 1.1) * (m_dimension + EXTRA_DIMENSIONS - i));
      if(i == m_dimension + EXTRA_DIMENSIONS - 1)
      {
        m_firstImageBar = barno;
      }
#ifdef SOM_VERBOSE
      Print(i, " ", barno, " ", (string)times[barno]);
#endif
      if(!m_hexCells)
        m_bmp[i].CreateBitmap(name, times[barno], price, m_bmpw, m_bmph + LEGEND_HEIGHT);
      else
        m_bmp[i].CreateBitmap(name, times[barno], price, m_bmpw + m_cellwidth / 2, m_bmph + m_cellheight / 2 + LEGEND_HEIGHT);
    }
    ObjectSetInteger(0, name, OBJPROP_ZORDER, i + 1);
  }

  switch(m_clrSchema)
  {
    case Black_White: // Black to White gradient
      Blend(0, 255, 0, 0, 0, 255, 255, 255); 
      break;

    case Blue_Green_Red: // Blue-green-red
      Blend(0, 1*64-1, 98, 98, 255, 98, 255, 255);
      Blend(1*64, 2*64-1, 98, 255, 255, 98, 255, 98);
      Blend(2*64, 3*64-1, 98, 255, 98, 255, 255, 98);
      Blend(3*64, 4*64-1, 255, 255, 98, 255, 98, 98);
      break;

    case Red_Black_Green: // Red-black-green
      Blend(0, 127, 155, 0, 0, 0, 0, 0);
      Blend(127, 255, 0, 0, 0, 0, 155, 0);
      break;
      
    case Red_White_Green: // Red-white-green
      Blend(0, 127, 155, 0, 0, 255, 255, 255);
      Blend(127, 255, 255, 255, 255, 0, 155, 0);
      break;
      
    default:
      Blend(0, 255, 0, 0, 0, 255, 255, 255);
  }

  return true;
}

bool CSOMDisplay::Init(const int xc, const int yc, const int bmpw, const int bmph, const int maxpict = 4, const ColorSchemes clrSchema = Blue_Green_Red, const bool bhex = true, const bool bborder = false, const bool btitle = true)
{
  m_firstImageBar = -1;
  bool result = CSOM::Init(xc, yc, bhex);
  if(result)
  {
    DisplayInit(bmpw, bmph, maxpict, clrSchema, bborder, btitle);
  }
  return result;
}

void CSOMDisplay::Reframe(const int xincrement, const int yincrement)
{
  CSOM::Reframe(xincrement, yincrement);
  m_cellwidth = m_bmpw / m_xcells;
  m_cellheight = m_bmph / m_ycells;
}

void CSOMDisplay::Render(const bool inProgress = false)
{
  if(!m_allocated) return;

  int ind = 0;
  
  color backclr = (color)ChartGetInteger(ChartID(), CHART_COLOR_BACKGROUND);
  
  for(int k = 0; k < m_dimension + EXTRA_DIMENSIONS; k++)
  {
    m_bmp[k].Erase(backclr);
  }

  for(int i = 0; i < m_xcells; i++)
  {
    for(int j = 0; j < m_ycells; j++)
    {
      for(int k = 0; k < m_dimension; k++)
      {
        double range = (m_max[k] - m_min[k]);
        //double value = (m_node[ind].GetWeight(k) - m_min[k]) / range; // alternative
        double value = m_node[ind].GetWeight(k) * m_sigma[k] + m_mean[k];
        
        char avg = (char)(255 * (value - m_min[k]) / range);
        int col = (int)GetPalColor(avg);
        RenderCell(k, col, ind, (j % 2 == 0), m_node[ind].IsSelected());
      }
      
      if(!inProgress)
      {
        // TODO: refactoring required
        
        { int k = DIM_HITCOUNT;
          if(m_max[k] > 0)
          {
            char avg = (char)(255 * (m_node[ind].GetHitsCount() - m_min[k]) / m_max[k]);
            int col = (int)GetPalColor(avg);
            RenderCell(k, col, ind, (j % 2 == 0), m_node[ind].IsSelected());
        } }
        
        { int k = DIM_UMATRIX;
          if(m_max[k] > 0)
          {
            char avg = (char)(255 * (m_node[ind].GetDistance() - m_min[k]) / m_max[k]);
            int col = (int)GetPalColor(avg);
            RenderCell(k, col, ind, (j % 2 == 0), m_node[ind].IsSelected());
        } }
  
        { int k = DIM_NODEMSE;
          if(m_max[k] > 0)
          {
            char avg = (char)(255 * (m_node[ind].GetMSE() - m_min[k]) / (m_max[k] - m_min[k]));
            int col = (int)GetPalColor(m_node[ind].GetMSE() > 0 ? avg : (uchar)0);
            RenderCell(k, col, ind, (j % 2 == 0), m_node[ind].IsSelected());
        } }

        { int k = DIM_OUTPUT;
          if(m_max[k] > 0)
          {
            char avg = (char)(255 * (m_node[ind].GetOutput() - m_min[k]) / (m_max[k] - m_min[k]));
            int col = (int)GetPalColor(avg);
            RenderCell(k, col, ind, (j % 2 == 0), m_node[ind].IsSelected());
        } }
        
        { int k = DIM_CLUSTERS;
          static color clusterColors[10] = {clrRed, clrGreen, clrBlue, clrYellow, clrMagenta, clrCyan, clrGray, clrOrange, clrSpringGreen, clrDarkGoldenrod};
          int cluster = m_node[ind].GetCluster();
          if(cluster != -1)
          {
            if(cluster < 10)
            {
              RenderCell(k, clusterColors[cluster], ind, (j % 2 == 0), m_node[ind].IsSelected());
            }
            else
            {
              color clr = (color)RGB256(255 / 2 * (cluster % 3), 255 / 4 * (cluster % 5), 255 / 6 * (cluster % 7));
              RenderCell(k, clr, ind, (j % 2 == 0), m_node[ind].IsSelected());
            }
        } }
      }
      
      ind++;
    }
    
  }

  // color gradient on the axis with values
  int w = m_bmpw + (m_hexCells ? m_cellwidth / 2 : 0);
  for(int m = 0; m < m_dimension + EXTRA_DIMENSIONS; m++)
  {
    RenderTitle(m, w);
  }
}

void CSOMDisplay::RenderTitle(const int m, const int w)
{
  for(int k = 0; k < LEGEND_HEIGHT; k++)
  {
    for(int l = 0; l < w; l++)
    {
      uchar colind = (uchar)MathRound(l * 255.0 / w);
      uint col = ColorToARGB(GetPalColor(colind));
      m_bmp[m].PixelSet(l, k, col);
    }
  }
  
  m_bmp[m].FontSet("Lucida Sans Unicode", 15);//Microsoft Sans Serif Small Fonts
  
  // display the range of values
  m_bmp[m].TextOut(0, 0, DoubleToString(m_min[m], 2), clrWhite);
  m_bmp[m].TextOut(1, 1, DoubleToString(m_min[m], 2), clrBlack);
  m_bmp[m].TextOut(m_bmpw, 0, DoubleToString(m_max[m], 2), clrWhite, TA_RIGHT);
  m_bmp[m].TextOut(m_bmpw + 1, 1, DoubleToString(m_max[m], 2), clrBlack, TA_RIGHT);

  m_bmp[m].FontSet("Impact", 15);
  
  string title = m_titles[m];
  
  int extent = m_bmp[m].TextWidth(title);
  if(extent > m_bmpw / 2)
  {
    int len = StringLen(title);
    // allowed width / pixels per char -> number of chars to keep
    int fit = m_bmpw / 2 / (extent / len);
    title = StringSubstr(title, 0, fit) + ShortToString(0x2026);
  }
  
  m_bmp[m].TextOut(m_bmpw / 2, 0, title, clrBlack, TA_CENTER);
  
  // default font
  m_bmp[m].FontSet("Lucida Sans Unicode", (int)(m_cellheight / 3.5));
}

void CSOMDisplay::RenderOutput()
{
  const int k = DIM_OUTPUT;
  
  if(m_max[k] <= 0) return;

  int ind = 0;
  
  color backclr = (color)ChartGetInteger(ChartID(), CHART_COLOR_BACKGROUND);
  m_bmp[k].Erase(backclr);
  
  for(int i = 0; i < m_xcells; i++)
  {
    for(int j = 0; j < m_ycells; j++)
    {
      char avg = (char)(255 * (m_node[ind].GetOutput() - m_min[k]) / (m_max[k] - m_min[k]));
      int col = (int)GetPalColor(avg);
      RenderCell(k, col, ind, (j % 2 == 0), m_node[ind].IsSelected());
        
      ind++;
    }
  }
  
  int w = m_bmpw + (m_hexCells ? m_cellwidth / 2 : 0);
  RenderTitle(k, w);
}


void CSOMDisplay::RenderCell(const int img, const int col, const int node_index, const bool cell_even, const bool selected = false)
{
  int x1, y1, x2, y2;
  x1 = m_node[node_index].GetX() * m_cellwidth;
  y1 = m_node[node_index].GetY() * m_cellheight;
  x2 = x1 + m_cellwidth;
  y2 = y1 + m_cellheight;
  
  uint fclr = ColorToARGB(selected ? ~col : col);

  if(!m_hexCells) // square cells
  {
    y1 = LEGEND_HEIGHT + y1;
    y2 = LEGEND_HEIGHT + y2;
    m_bmp[img].FillRectangle(x1, y1, x2, y2, fclr);
    if(m_bShowBorder)
    {
      m_bmp[img].Rectangle(x1, y1, x2, y2, ColorToARGB(clrBlack));
    }
  }
  else // hexagonal cells
  {
    int x_size = (int)MathAbs(x2 - x1);
    int y_size = (int)MathAbs(y2 - y1);
  
    y1 = LEGEND_HEIGHT + y1 + y_size / 4;
    y2 = LEGEND_HEIGHT + y2 + y_size / 4;
    if(cell_even)
    {
      x1 = x1 + x_size / 2;
      x2 = x2 + x_size / 2;
    }

    m_bmp[img].FillRectangle(x1, y1 + y_size / 4, x2, y2 - y_size / 4, fclr);
    m_bmp[img].FillTriangle(x1, y1 + y_size / 4, x1 + x_size / 2, y1 - y_size / 4, x2, y1 + y_size / 4, fclr);
    m_bmp[img].FillTriangle(x1, y2 - y_size / 4, x2, y2 - y_size / 4, x1 + x_size / 2, y2 + y_size / 4, fclr);

    if(m_bShowBorder)
    {
      uint bcol = ColorToARGB(clrBlack);
      m_bmp[img].Line(x1, y1 + y_size / 4, x1 + x_size / 2, y1 - y_size / 4, bcol);
      m_bmp[img].Line(x1 + x_size / 2, y1 - y_size / 4, x2, y1 + y_size / 4, bcol);

      m_bmp[img].Line(x1, y2 - y_size / 4, x1 + x_size / 2, y2 + y_size / 4, bcol);
      m_bmp[img].Line(x2, y2 - y_size / 4, x2 - x_size / 2, y2 + y_size / 4, bcol);

      m_bmp[img].Line(x1, y1 + y_size / 4, x1, y2 - y_size / 4, bcol);
      m_bmp[img].Line(x2, y1 + y_size / 4, x2, y2 - y_size / 4, bcol);
    }
  }
}

void CSOMDisplay::InverseCell(const int img, const int node_index, const bool cell_even)
{
  int x1, y1, x2, y2;
  x1 = m_node[node_index].GetX() * m_cellwidth;
  y1 = m_node[node_index].GetY() * m_cellheight;
  x2 = x1 + m_cellwidth;
  y2 = y1 + m_cellheight;
  
  if(!m_hexCells) // square cells
  {
    y1 = LEGEND_HEIGHT + y1;
    y2 = LEGEND_HEIGHT + y2;
  }
  else // hexagonal cells
  {
    int x_size = (int)MathAbs(x2 - x1);
    int y_size = (int)MathAbs(y2 - y1);
  
    y1 = LEGEND_HEIGHT + y1 + y_size / 4;
    y2 = LEGEND_HEIGHT + y2 + y_size / 4;
    if(cell_even)
    {
      x1 = x1 + x_size / 2;
      x2 = x2 + x_size / 2;
    }
    
    y1 += y_size / 4;
    y2 -= y_size / 4;

    for(int i = x1; i < x2; i++)
    {
      for(int j = y1 - 1; j > y1 - y_size / 2; j--)
      {
        if(y1 - 1 - j < MathMin(i - x1, x2 - i - 2))
        {
          m_bmp[img].PixelSet(i, j, ~m_bmp[img].PixelGet(i, j));
        }
      }

      for(int j = y2; j < y2 + y_size / 2; j++)
      {
        if(j - y2 < MathMin(i - x1 + 1, x2 - i - 1))
        {
          m_bmp[img].PixelSet(i, j, ~m_bmp[img].PixelGet(i, j));
        }
      }
    }
  }
  
  for(int i = x1; i < x2; i++)
  {
    for(int j = y1; j < y2; j++)
    {
      m_bmp[img].PixelSet(i, j, ~m_bmp[img].PixelGet(i, j));
    }
  }
  
}

void CSOMDisplay::ShowBMP(const bool saveToFile = false)
{
  int cellwidth = m_hexCells ? m_bmpw / m_xcells / 2 : 0;
  int cellheight = m_hexCells ? m_bmph / m_ycells / 2 : 0;
  for(int i = 0; i < m_dimension + EXTRA_DIMENSIONS; i++)
  {
    m_bmp[i].Update();
    if(saveToFile) // save BMP-image to file
    {
      string name = string(i) + "_" + m_titles[i];
      ResourceSave(m_bmp[i].ResourceName(), m_sID + "\\" + name + ".bmp");
    }
  }
}

void CSOMDisplay::ProgressUpdate()
{
  Render(true);
  ShowBMP(false);
  ChartRedraw();
}

void CSOMDisplay::HideBMP()
{
  for(int i = 0; i < ArraySize(m_bmp); i++)
  {
    m_bmp[i].Destroy();
  }
  ArrayResize(m_bmp, 0);
}

void CSOMDisplay::CalcPixelCoordinates(const int x1, const int y1, int &xc, int &yc) const
{
  int y_size = m_hexCells ? m_cellheight / 2 : 0;
  xc = x1 * m_cellwidth + m_cellwidth / 2;
  yc = y1 * m_cellheight + LEGEND_HEIGHT + (m_cellheight + y_size) / 2;
  if(m_hexCells && (y1 % 2 == 0)) xc += m_cellwidth / 2;
}


void CSOMDisplay::ShowPatternHelper(double &vector[], const string name = NULL, const int plane = -1) const
{
  // vector is mutated inside GetBestMatchingIndex due to (optional) normalization
  int index = GetBestMatchingIndex(vector);
  if(index > -1)
  {
    ShowNode(index, name, plane);
  }
}

void CSOMDisplay::ShowPattern(const double &vector[], const string name = NULL, const int plane = -1) const
{
  double data[];
  ArrayCopy(data, vector);
  ShowPatternHelper(data, name, plane);
}

void CSOMDisplay::ShowNode(const int ind, const string name = NULL, const int plane = -1) const
{
  if(m_bShowTitle || name != NULL)
  {
    bool oddy = ((ind % m_ycells) % 2) == 1;
    
    int avgcolor = 0;
    switch(m_clrSchema)
    {
      case 0: // Black to White gradient
        avgcolor = clrCyan;
        break;
  
      case 1: // Blue-green-red
        avgcolor = clrDimGray;
        break;
  
      case 2: // Red-black-green
        avgcolor = clrLightBlue;
        break;
        
      case 3: // Red-white-green
        avgcolor = clrBlack;
        break;
        
      default:
        avgcolor = clrOrange;
    }
    
    int x = m_node[ind].GetX();
    int y = m_node[ind].GetY();
    int xc, yc;
    CalcPixelCoordinates(x, y, xc, yc);

    if(plane == -1)
    {
      for(int i = 0; i < m_dimension; i++)
      {
        if(name != NULL)
        {
          m_bmp[i].TextOut(xc, yc, name, avgcolor, TA_CENTER|TA_VCENTER);
        }
        else
        {
          // denormalized value of code-vector
          double value = m_node[ind].GetWeight(i) * m_sigma[i] + m_mean[i];
          m_bmp[i].TextOut(xc, yc - m_cellheight / 4, DoubleToString(value, 4), avgcolor, TA_CENTER|TA_VCENTER);
          // mean[count]
          m_bmp[i].TextOut(xc, yc, DoubleToString(m_node[ind].GetHitsMean(i), 3) + "[" + (string)m_node[ind].GetHitsCount() + "]", avgcolor, TA_CENTER|TA_VCENTER);
          // sigma
          m_bmp[i].TextOut(xc, yc + m_cellheight / 4, DoubleToString(m_node[ind].GetHitsDeviation(i), 4), avgcolor, TA_CENTER|TA_VCENTER);
        }
      }
      
      if(m_node[ind].GetHitsCount() > 0)
      {
        m_bmp[DIM_HITCOUNT].TextOut(xc, yc, GetNodeAsString(ind, DIM_HITCOUNT), avgcolor, TA_CENTER|TA_VCENTER);
        // debug
        //m_bmp[DIM_HITCOUNT].TextOut(xc, yc, (string)(x) + "," + (string)(y) + "[" + (string)ind + "]", avgcolor, TA_CENTER|TA_VCENTER);
      }
      if(m_node[ind].GetDistance() > 0)
      {
        m_bmp[DIM_UMATRIX].TextOut(xc, yc, GetNodeAsString(ind, DIM_UMATRIX), avgcolor, TA_CENTER|TA_VCENTER);
      }
      if(m_node[ind].GetMSE() > 0)
      {
        m_bmp[DIM_NODEMSE].TextOut(xc, yc, GetNodeAsString(ind, DIM_NODEMSE), avgcolor, TA_CENTER|TA_VCENTER);
      }
      m_bmp[DIM_OUTPUT].TextOut(xc, yc, GetNodeAsString(ind, DIM_OUTPUT), avgcolor, TA_CENTER|TA_VCENTER);
    }
    else
    {
      string family;
      int size;
      uint flags, angle;
      m_bmp[plane].FontGet(family, size, flags, angle);
      m_bmp[plane].FontSet("Lucida Sans Unicode", (int)(m_cellheight / 2));
      
      m_bmp[plane].TextOut(xc, yc, name != NULL ? name : "*", clrWhite, TA_CENTER|TA_VCENTER);
      m_bmp[plane].TextOut(xc + 1, yc + 1, name != NULL ? name : "*", clrBlack, TA_CENTER|TA_VCENTER);
      
      m_bmp[plane].FontSet(family, size, flags, angle);
    }
  }
}

void CSOMDisplay::ShowAllPatterns(uint start = 0, uint end = 0) const
{
  if(!m_allocated) return;
  
  double data[];
  if(start >= (uint)m_nSet) start = 0;
  if(end == 0 || end >= (uint)m_nSet) end = m_nSet;
  for(uint i = start; i < end; i++)
  {
    ArrayCopy(data, m_set, 0, m_dimension * i, m_dimension);
    ShowPattern(data, m_set_titles[i]);
    ShowPatternHelper(data, m_set_titles[i], DIM_CLUSTERS);
  }
}

void CSOMDisplay::ShowAllNodes() const
{
  if(!m_allocated) return;

  for(int i = 0; i < m_xcells * m_ycells; i++)
  {
    ShowNode(i);
  }
}

void CSOMDisplay::ShowClusters(int limit = -1) const
{
  if(!m_allocated) return;

  double data[];
  int nclusters = ArraySize(m_clusters) / m_dimension;
  Print("Clusters [", nclusters, "]:");
  ArrayPrint(m_titles, 0, NULL, 0, m_dimension);
  if(limit == -1) limit = 5;
  
  int nlabels = ArraySize(m_labels);
  
  for(int i = 0; i < nclusters; i++)
  {
    ArrayCopy(data, m_clusters, 0, m_dimension * i, m_dimension);
    int index = GetBestMatchingIndexNormalized(data);
    ShowNode(index, i < nlabels && m_labels[i] != NULL ? m_labels[i] : "C" + (string)i, DIM_CLUSTERS);
    if(i < limit)
    {
      Denormalize(data);
      Print("N", i, " [", m_node[index].GetX(), ",", m_node[index].GetY(), "] ", i < nlabels && m_labels[i] != NULL ? m_labels[i] : "");
      ArrayPrint(data);
    }
  }
}


void CSOMDisplay::Blend(uchar c1, uchar c2, uchar r1, uchar g1, uchar b1, uchar r2, uchar g2, uchar b2)
{
  int n = c2 - c1;
  for(int i = 0; i <= n; i++)
  {
    if(c1 + i + 2 < ArraySize(Palette))
    {
      Palette[3 * (c1 + i)] = (uchar)MathRound(1 * (r1 * (n - i) + r2 * i) * 1.0 / n);
      Palette[3 * (c1 + i) + 1] = (uchar)MathRound(1 * (g1 * (n - i) + g2 * i) * 1.0 / n);
      Palette[3 * (c1 + i) + 2] = (uchar)MathRound(1 * (b1 * (n - i) + b2 * i) * 1.0 / n);
    }
  }
}

int CSOMDisplay::GetPalColor(uchar index)
{
  int ind = index;
  if(ind <= 0) ind = 0;
  uchar r = Palette[3 * ind];
  uchar g = Palette[3 * ind + 1];
  uchar b = Palette[3 * ind + 2];
  return RGB256(r, g, b);
}

void CSOMDisplay::ColToRGB(int col,int &r,int &g,int &b)
{
  r = col & 0xFF;
  g = (col >> 8) & 0xFF;
  b = (col >> 16) & 0xFF;
}

void CSOMDisplay::HideChart()
{
  // this does not suit because it hides comment as well
  // ChartSetInteger(ChartID(), CHART_SHOW, false);
  
  ChartNavigate(ChartID(), CHART_END, 0);
  
  ChartSetInteger(ChartID(), CHART_SHOW_OHLC, false);
  ChartSetInteger(ChartID(), CHART_SHOW_ONE_CLICK, false);
  ChartSetInteger(ChartID(), CHART_SHOW_BID_LINE, false);
  ChartSetInteger(ChartID(), CHART_SHOW_ASK_LINE, false);
  ChartSetInteger(ChartID(), CHART_SHOW_LAST_LINE, false);
  ChartSetInteger(ChartID(), CHART_SHOW_PERIOD_SEP, false);
  ChartSetInteger(ChartID(), CHART_SHOW_GRID, false);
  ChartSetInteger(ChartID(), CHART_SHOW_VOLUMES, CHART_VOLUME_HIDE);
  ChartSetInteger(ChartID(), CHART_SHOW_OBJECT_DESCR, true);
  ChartSetInteger(ChartID(), CHART_SHOW_TRADE_LEVELS, false);
  ChartSetInteger(ChartID(), CHART_SHOW_DATE_SCALE, false);
  ChartSetInteger(ChartID(), CHART_SHOW_PRICE_SCALE, false);
  ChartSetInteger(ChartID(), CHART_FOREGROUND, false);
  ChartSetInteger(ChartID(), CHART_SHIFT, false);
  ChartSetInteger(ChartID(), CHART_MODE, CHART_LINE);
  // ChartSetInteger(ChartID(), CHART_SCALE, 0);

  color clrBack = (color)ChartGetInteger(ChartID(), CHART_COLOR_BACKGROUND);
  // ChartSetInteger(ChartID(), CHART_COLOR_FOREGROUND, clrWhite);
  ChartSetInteger(ChartID(), CHART_COLOR_CHART_LINE, clrBack);
  ChartSetInteger(ChartID(), CHART_COLOR_ASK, clrBack);
  ChartSetInteger(ChartID(), CHART_COLOR_BID, clrBack);

  ChartSetString(ChartID(), CHART_COMMENT, "");
  ChartRedraw(ChartID());
}

void CSOMDisplay::Rewind()
{
  if(m_firstImageBar > -1)
  {
    int visible = (int)ChartGetInteger(0, CHART_FIRST_VISIBLE_BAR);
    if(visible > m_firstImageBar)
    {
      if(!ChartNavigate(0, CHART_CURRENT_POS, visible - m_firstImageBar))
      {
        Print(GetLastError());
      }
    }
  }
}

int CSOMDisplay::GetNodeIndex(const string name, const long x, const long y, const long dx, const long dy, int &px, int &py) const
{
  long x0, y0;
  if(m_maxpict == 0)
  {
    datetime dt = (datetime)ObjectGetInteger(0, name, OBJPROP_TIME);
    double price = 1.0;
    int _x, _y;
    ChartTimePriceToXY(0, 0, dt, price, _x, _y);
    x0 = _x;
    y0 = _y;
  }
  else
  {
    x0 = (long)ObjectGetInteger(0, name, OBJPROP_XDISTANCE);
    y0 = (long)ObjectGetInteger(0, name, OBJPROP_YDISTANCE) + LEGEND_HEIGHT;
  }
  if(m_hexCells)
  {
    x0 += m_cellwidth / 4;
    y0 += m_cellheight / 4;
  }
  if(x >= x0 && y >= y0 && x <= x0 + dx && y <= y0 + dy)
  {
    px = (int)(x - x0) / m_cellwidth;
    py = (int)(y - y0) / m_cellheight;
    if(px >= 0 && px < m_xcells && py >= 0 && py < m_ycells)
    {
      return px * m_ycells + py;
    }
  }
  return -1;
}

void CSOMDisplay::OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
{
  static bool dragging = false;
  
  if(id == CHARTEVENT_MOUSE_MOVE)
  {
    uint mask = (uint)sparam;
    if(mask == 0 && dragging)
    {
      Rewind();
      dragging = (mask != 0);
      return;
    }
    dragging = (mask != 0);

    // show tooltip with information about node under mouse cursor
    
    long x = (long)lparam;
    long y = (long)dparam;
    long dx = m_bmpw + (m_hexCells ? m_cellwidth / 2 : 0);
    long dy = m_bmph + LEGEND_HEIGHT + (m_hexCells ? m_cellheight / 2 : 0);

    ENUM_OBJECT type = (m_maxpict == 0 ? OBJ_BITMAP : OBJ_BITMAP_LABEL);

    for(int i = 0; i < ObjectsTotal(0, 0, type); i++)
    {
      string name = ObjectName(0, i, 0, type);
      
      int px, py;
      int index = GetNodeIndex(name, x, y, dx, dy, px, py);
      if(index > -1)
      {
        int plane = (int)ObjectGetInteger(0, name, OBJPROP_ZORDER) - 1;
        
        string coor = name + " (" + (string)px + "," + (string)py + ")\n";
        
        if(plane < m_dimension)
        {
          string value = DoubleToString(m_node[index].GetWeight(plane) * m_sigma[plane] + m_mean[plane], 4) + "\n"
          + DoubleToString(m_node[index].GetHitsMean(plane), 3) + ShortToString(0x00B1) + DoubleToString(m_node[index].GetHitsDeviation(plane), 4) + "\n"
          + "[" + (string)m_node[index].GetHitsCount() + "]";
          ObjectSetString(0, name, OBJPROP_TOOLTIP, coor + value);
        }
        else
        {
          ObjectSetString(0, name, OBJPROP_TOOLTIP, coor + GetNodeAsString(index, plane));
        }
      }
    }
  }
  else if(id == CHARTEVENT_CHART_CHANGE && !dragging)
  {
    Rewind();
  }
  else if(id == CHARTEVENT_OBJECT_CLICK && !dragging)
  {
    ENUM_OBJECT type = (m_maxpict == 0 ? OBJ_BITMAP : OBJ_BITMAP_LABEL);
    if(ObjectGetInteger(0, sparam, OBJPROP_TYPE) == type)
    {
      long x = (long)lparam;
      long y = (long)dparam;
      long dx = m_bmpw + (m_hexCells ? m_cellwidth / 2 : 0);
      long dy = m_bmph + LEGEND_HEIGHT + (m_hexCells ? m_cellheight / 2 : 0);
      
      int px, py;
      int index = GetNodeIndex(sparam, x, y, dx, dy, px, py);
      if(index > -1)
      {
        int plane = (int)ObjectGetInteger(0, sparam, OBJPROP_ZORDER) - 1;
        if(plane == DIM_CLUSTERS)
        {
          int cluster = m_node[index].GetCluster();
          for(int i = 0; i < ArraySize(m_node); i++)
          {
            if(m_node[i].GetCluster() == cluster)
            {
              m_node[i].Select();
              Inverse(i, ((i % m_ycells) % 2) == 0);
            }
          }
        }
        else
        {
          m_node[index].Select();
          Inverse(index, ((index % m_ycells) % 2) == 0);
          
          double vector[];
          m_node[index].GetCodeVector(vector);
          CalculateOutput(vector);
          RenderOutput();
        }
        ShowBMP();
        ChartRedraw();
      }
    }
  }
}

void CSOMDisplay::Inverse(const int node_index, const bool cell_even)
{
  for(int m = 0; m < m_dimension + EXTRA_DIMENSIONS; m++)
  {
    InverseCell(m, node_index, cell_even); 
  }
}

string CSOMDisplay::GetNodeAsString(const int index, const int plane) const
{
  if(plane == DIM_HITCOUNT) return "[" + (string)m_node[index].GetHitsCount() + "]";
  else if(plane == DIM_UMATRIX) return "<" + DoubleToString(m_node[index].GetDistance(), 3) + ">";
  else if(plane == DIM_NODEMSE) return "<" + DoubleToString(m_node[index].GetMSE(), 3) + ">";
  else if(plane == DIM_CLUSTERS) return "Cluster " + (string)m_node[index].GetCluster() + (m_node[index].GetCluster() < ArraySize(m_labels) && m_labels[m_node[index].GetCluster()] != NULL ? "\n" + m_labels[m_node[index].GetCluster()] : "");
  else if(plane == DIM_OUTPUT) return DoubleToString(m_node[index].GetOutput(), 3);
  else if(plane < m_dimension)
  {
    return (string)m_node[index].GetWeight(plane);
  }
  return (string)index;
}
