//|------------------------------------------------------------------+
//|                                                 SOM-Forecast.mq5 |
//|                                    Copyright (c) 2018, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                           https://www.mql5.com/ru/articles/5473/ |
//|------------------------------------------------------------------+
#property copyright   "Copyright (c) 2018, Marketeer"
#property link        "https://www.mql5.com/en/users/marketeer"
#property version     "1.0"
#property description "Kohonen map neural network test bed and showcase with GUI and 1 step ahead prediction"

// #define SOM_VERBOSE
#define SOM_OUTLIERS_SOFT

#include <CSOM/CSOMDisplay.mqh>

input string _1 = ""; // Network Structure and Data Settings
input string DataFileName = ""; // ·    DataFileName (*.csv)
input string NetFileName = ""; // ·    NetFileName (*.som)
input int CellsX = 10; // ·    CellsX
input int CellsY = 10; // ·    CellsY
input bool HexagonalCell = true; // ·    HexagonalCell
input bool UseNormalization = true; // ·    UseNormalization
input int EpochNumber = 100; // ·    EpochNumber
input int ValidationSetPercent = 0; // ·    ValidationSetPercent
input int ClusterNumber = -1; // ·    ClusterNumber
input bool ForecastMode = false; // ·    ForecastMode
input bool AllowDuplicatedLabels = false; // ·    AllowDuplicatedLabels
input int ReframeNumber = 0; // ·    ReframeNumber
input int ReframeStep = 2; // ·    ReframeStep

input string _2 = ""; // Visualization
input int ImageW = 500; // ·    ImageW
input int ImageH = 500; // ·    ImageH
input int MaxPictures = 0; // ·    MaxPictures
input bool ShowBorders = false; // ·    ShowBorders
input bool ShowTitles = true; // ·    ShowTitles
input ColorSchemes ColorScheme = Blue_Green_Red; // ·    ColorScheme
input bool ShowProgress = true; // ·    ShowProgress

input string _3 = ""; // Options
input int RandomSeed = 0; // ·    RandomSeed
input bool SaveImages = false; // ·    SaveImages


CSOMDisplay KohonenMap;

int OnInit()
{
  if(ValidationSetPercent > 50)
  {
    Print("Use ValidationSetPercent less then 50%");
    return INIT_PARAMETERS_INCORRECT;
  }
  KohonenMap.Reset();
  ChartSetInteger(0, CHART_EVENT_MOUSE_MOVE, true);
  EventSetMillisecondTimer(1);
  return INIT_SUCCEEDED;
}

void OnDeinit(const int reason)
{
  KohonenMap.Reset();
  Comment("");
}

void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
{
  KohonenMap.OnChartEvent(id, lparam, dparam, sparam);
}

void OnTimer()
{
  EventKillTimer();
  
  MathSrand(RandomSeed);
  
  bool hasOneTestPattern = false;

  if(NetFileName != "")
  {
    if(!KohonenMap.Load(NetFileName)) return;
    KohonenMap.DisplayInit(ImageW, ImageH, MaxPictures, ColorScheme, ShowBorders, ShowTitles);

    Comment("Map ", NetFileName, " is loaded; size: ", KohonenMap.GetWidth(), "*", KohonenMap.GetHeight(), "; features: ", KohonenMap.GetFeatureCount());

    if(DataFileName != "")
    {
      if(!KohonenMap.LoadPatterns(DataFileName))
      {
        Print("Data loading error, file: ", DataFileName);
        return;
      }
      
      if(ForecastMode)
      {
        int m = KohonenMap.GetFeatureCount();
        KohonenMap.SetFeatureMask(m - 1, 0);
        
        int n = KohonenMap.GetDataCount();
        double vector[];
        double forecast[];
        double future;
        int correct = 0;
        double error = 0;
        double variance = 0;
        for(int i = 0; i < n; i++)
        {
          KohonenMap.GetPattern(i, vector);
          future = vector[m - 1]; // preserve future
          vector[m - 1] = 0;      // make future unknown for the net (it's not used anyway due to the mask)
          KohonenMap.GetBestMatchingFeatures(vector, forecast);
          
          if(future * forecast[m - 1] > 0) // check if the directions match
          {
            correct++;
          }
          
          error += (future - forecast[m - 1]) * (future - forecast[m - 1]);
          variance += future * future;
        }

        string result;
        StringConcatenate(result, "Correct forecasts: ", correct, " out of ", n, " => ", DoubleToString(correct * 100.0 / n, 2), "%, error => ", error / variance);
        Print(result);
        Comment(result);
      }
    }
    else
    {
      // TODO: generate test examples
    }
  }
  else // a net file is not provided, so training is assumed
  {
    if(DataFileName == "")
    {
      // TODO: generate demo vectors with predictable values
    }
    else // a data file is provided
    {
      if(!KohonenMap.LoadPatterns(DataFileName))
      {
        Print("Data loading error, file: ", DataFileName);
        return;
      }
    }

    KohonenMap.Init(CellsX, CellsY, ImageW, ImageH, MaxPictures, ColorScheme, HexagonalCell, ShowBorders, ShowTitles);

    if(ForecastMode)
    {
      KohonenMap.SetFeatureMask(KohonenMap.GetFeatureCount() - 1, 0);
    }

    if(ValidationSetPercent > 0)
    {
      KohonenMap.SetValidationSection((int)(KohonenMap.GetDataCount() * (1.0 - ValidationSetPercent / 100.0)));
    }
    
    if(ReframeNumber > 0)
    {
      KohonenMap.TrainAndReframe(EpochNumber, UseNormalization, ShowProgress, ReframeNumber, ReframeStep, ReframeStep);
    }
    else
    {
      KohonenMap.Train(EpochNumber, UseNormalization, ShowProgress);
    }

    KohonenMap.CalculateDistances(); // build U-Matrix
    
    if(ClusterNumber > 1)
    {
      KohonenMap.Clusterize(ClusterNumber);
    }
    else
    {
      KohonenMap.Clusterize(); // use U-Matrix & QE
    }

  }

  if(!hasOneTestPattern)
  {
    // default output is calculated without feature mask
    KohonenMap.SetFeatureMask(0, 0);
    double vector[];
    ArrayResize(vector, KohonenMap.GetFeatureCount());
    ArrayInitialize(vector, 0);
    KohonenMap.CalculateOutput(vector);
  }

  if(ClusterNumber != 0)
  {
    SetClusterLabels();
  }

  // draw maps into internal BMP buffers
  KohonenMap.Render();

   // draw labels in cells in BMP buffers
  if(hasOneTestPattern)
    KohonenMap.ShowAllPatterns();
  else
    KohonenMap.ShowAllNodes();

  if(ClusterNumber != 0)
  {
    SetClusterLabels();
    KohonenMap.ShowClusters(ClusterNumber); // mark clusters
  }
  
  // display files as bitmap images on chart, optionally save into files
  KohonenMap.ShowBMP(SaveImages);
  
  if(NetFileName == "")
  {
    KohonenMap.Save(KohonenMap.GetID());
  }
}

void SetClusterLabels()
{
  const int nclusters = KohonenMap.GetClusterCount();

  double min, max, best;
  
  double bests[][3]; // [][0 - value; 1 - feature index; 2 - direction]
  ArrayResize(bests, nclusters);
  ArrayInitialize(bests, 0);
  
  int n = KohonenMap.GetFeatureCount();
  for(int i = 0; i < n; i++)
  {
    int direction = 0;
    KohonenMap.GetFeatureBounds(i, min, max);
    if(max - min > 0)
    {
      best = 0;
      double center[];
      for(int j = nclusters - 1; j >= 0; j--)
      {
        KohonenMap.GetCluster(j, center);
        double value = MathMin(MathMax((center[i] - min) / (max - min), 0), 1);
        
        if(value > 0.5)
        {
          direction = +1;
        }
        else
        {
          direction = -1;
          value = 1 - value;
        }
        
        if(value > bests[j][0])
        {
          bests[j][0] = value;
          bests[j][1] = i;
          bests[j][2] = direction;
        }
      }
    }
  }

  // Remove duplicated labels or allow multiple clusters with the same feature label
  // (can be helpful to describe secondary choices)
  if(!AllowDuplicatedLabels)
  {
    double features[][2]; // [][0 - value, 1 - cluster index]
    ArrayResize(features, n);
    ArrayInitialize(features, -1);
    
    for(int j = 0; j < nclusters; j++)
    {
      for(int k = j + 1; k < nclusters; k++)
      {
        if(bests[j][1] == bests[k][1] && bests[j][2] == bests[k][2])
        {
          ReplaceCluster(k, j);
        }
      }
    }
  }
  
  for(int j = 0; j < nclusters; j++)
  {
    if(bests[j][0] > 0)
    {
      KohonenMap.SetLabel(j, (bests[j][2] > 0 ? "+" : "-") + KohonenMap.GetFeatureTitle((int)bests[j][1]));
    }
  }
}

void ReplaceCluster(const int k, const int j)
{
  int n = KohonenMap.GetSize();
  for(int i = 0; i < n; i++)
  {
    CSOMNode *node = KohonenMap.GetNode(i);
    if(node.GetCluster() == k)
    {
      node.SetCluster(j);
    }
  }
}