//|------------------------------------------------------------------+
//|                                             SOM-Explorer-RGB.mq5 |
//|                                    Copyright (c) 2018, Marketeer |
//|                          https://www.mql5.com/en/users/marketeer |
//|                           https://www.mql5.com/ru/articles/5472/ |
//|                           https://www.mql5.com/ru/articles/5473/ |
//|------------------------------------------------------------------+
#property copyright   "Copyright (c) 2018, Marketeer"
#property link        "https://www.mql5.com/en/users/marketeer"
#property version     "1.1"
#property description "Kohonen map neural network test bed and showcase with GUI"
#property description "Includes color-based analysis (RGB and grayscale)"

// #define SOM_VERBOSE

#include <CSOM/CSOMDisplayRGB.mqh>

input string _1 = ""; // Network Structure and Data Settings
input string DataFileName = ""; // ·    DataFileName (*.csv, *.set)
input string NetFileName = ""; // ·    NetFileName (*.som)
input int CellsX = 10; // ·    CellsX
input int CellsY = 10; // ·    CellsY
input bool HexagonalCell = true; // ·    HexagonalCell
input bool UseNormalization = true; // ·    UseNormalization
input int EpochNumber = 100; // ·    EpochNumber
input int ValidationSetPercent = 0; // ·    ValidationSetPercent
input int ClusterNumber = -1; // ·    ClusterNumber
input string FeatureMask = ""; // ·    FeatureMask ([1|0]...)
input bool ApplyFeatureMaskAfterTraining = false; // ·    FeatureMaskAfterTraining

input string _2 = ""; // Visualization
input int ImageW = 500; // ·    ImageW
input int ImageH = 500; // ·    ImageH
input int MaxPictures = 0; // ·    MaxPictures
input bool ShowBorders = false; // ·    ShowBorders
input bool ShowTitles = true; // ·    ShowTitles
input ColorSchemes ColorScheme = Blue_Green_Red; // ·    ColorScheme
input bool ShowProgress = true; // ·    ShowProgress

input string _3 = ""; // Options
input int RandomSeed = 0; // ·    RandomSeed
input bool SaveImages = false; // ·    SaveImages
input string FeatureDirection = ""; // ·    FeatureLabellingDirection ([+|-]...)
input bool AllowDuplicatedLabels = false; // ·    AllowDuplicatedLabels
input int SaveClusterAsSettings = -1; // ·    SaveClusterAsSettings
input bool ShowRGBOutput = false; // ·    ShowRGBOutput


CSOMDisplayRGB KohonenMap;

void OnInit()
{
  KohonenMap.Reset();
  ChartSetInteger(0, CHART_EVENT_MOUSE_MOVE, true);
  EventSetMillisecondTimer(1);
}

void OnDeinit(const int reason)
{
  KohonenMap.Reset();
  Comment("");
}

void OnChartEvent(const int id, const long &lparam, const double &dparam, const string &sparam)
{
  KohonenMap.OnChartEvent(id, lparam, dparam, sparam);
}

void OnTimer()
{
  EventKillTimer();
  
  MathSrand(RandomSeed);
  
  bool hasOneTestPattern = false;

  ulong mask = 0;
  if(FeatureMask != "")
  {
    int n = MathMin(StringLen(FeatureMask), 64);
    for(int i = 0; i < n; i++)
    {
      mask |= (StringGetCharacter(FeatureMask, i) == '1' ? 1 : 0) << i;
    }
  }
  
  if(NetFileName != "")
  {
    if(!KohonenMap.Load(NetFileName)) return;
    KohonenMap.DisplayInit(ImageW, ImageH, MaxPictures, ColorScheme, ShowBorders, ShowTitles);

    Comment("Map ", NetFileName, " is loaded; size: ", KohonenMap.GetWidth(), "*", KohonenMap.GetHeight(), "; features: ", KohonenMap.GetFeatureCount());

    if(DataFileName != "")
    {
      // process .set file as a special case of pattern
      if(StringFind(DataFileName, ".set") == StringLen(DataFileName) - 4)
      {
        double v[];
        if(LoadSettings(DataFileName, v))
        {
          KohonenMap.AddPattern(v, "SETTINGS");
          ArrayPrint(v);
          double y[];
          CSOMNode *node = KohonenMap.GetBestMatchingFeatures(v, y);
          Print("Matched Node Output (", node.GetX(), ",", node.GetY(),
            "); Hits:", node.GetHitsCount(), "; Error:", node.GetMSE(),
            "; Cluster N", node.GetCluster(), ":");
          ArrayPrint(y);
          KohonenMap.CalculateOutput(v, true);
          hasOneTestPattern = true;
        }
      }
      else
      if(!KohonenMap.LoadPatterns(DataFileName))
      {
        Print("Data loading error, file: ", DataFileName);

        // generate a random test vector
        int n = KohonenMap.GetFeatureCount();
        double min, max;
        double v[];
        ArrayResize(v, n);
        for(int i = 0; i < n; i++)
        {
          KohonenMap.GetFeatureBounds(i, min, max);
          v[i] = (max - min) * rand() / 32767 + min;
        }
        KohonenMap.AddPattern(v, "RANDOM");
        Print("Random Input:");
        ArrayPrint(v);
        double y[];
        CSOMNode *node = KohonenMap.GetBestMatchingFeatures(v, y);
        Print("Matched Node Output (", node.GetX(), ",", node.GetY(),
          "); Hits:", node.GetHitsCount(), "; Error:", node.GetMSE(),
          "; Cluster N", node.GetCluster(), ":");
        ArrayPrint(y);
        KohonenMap.CalculateOutput(v, true);
        hasOneTestPattern = true;
      }
    }
  }
  else // a net file is not provided, so training is assumed
  {
    if(DataFileName == "")
    {
      // generate 3-d demo vectors with unscaled values {[0,+1000], [0,+1], [-1,+1]}
      // feed them to the net to compare results with and without normalization
      // NB. titles should be valid filenames for BMP
      string titles[] = {"R1000", "R1", "R2"};
      KohonenMap.AssignFeatureTitles(titles);
      double x[3];
      for(int i = 0; i < 1000; i++)
      {
        x[0] = 1000.0 * rand() / 32767;
        x[1] = 1.0 * rand() / 32767;
        x[2] = -2.0 * rand() / 32767 + 1.0;
        /*
        if(rand() > 32000)
        {
          x[rand() / 10922] *= 10; // emulate an outlier
        }
        */
        KohonenMap.AddPattern(x, StringFormat("%f %f %f", x[0], x[1], x[2]));
      }
    }
    else // a data file is provided
    {
      if(!KohonenMap.LoadPatterns(DataFileName))
      {
        Print("Data loading error, file: ", DataFileName);
        return;
      }
    }

    KohonenMap.Init(CellsX, CellsY, ImageW, ImageH, MaxPictures, ColorScheme, HexagonalCell, ShowBorders, ShowTitles);
    
    if(ValidationSetPercent > 0 && ValidationSetPercent < 50)
    {
      KohonenMap.SetValidationSection((int)(KohonenMap.GetDataCount() * (1.0 - ValidationSetPercent / 100.0)));
    }
    
    if(!ApplyFeatureMaskAfterTraining) KohonenMap.SetFeatureMask(0, mask);

    KohonenMap.Train(EpochNumber, UseNormalization, ShowProgress);

    if(ApplyFeatureMaskAfterTraining) KohonenMap.SetFeatureMask(0, mask);
    
    KohonenMap.CalculateDistances(); // build U-Matrix
    
    if(ClusterNumber > 1)
    {
      KohonenMap.Clusterize(ClusterNumber);
    }
    else
    {
      KohonenMap.Clusterize(); // use U-Matrix & QE
    }

  }

  if(!hasOneTestPattern && !(ShowRGBOutput && StringLen(FeatureDirection) > 0))
  {
    // default output is calculated without feature mask
    KohonenMap.SetFeatureMask(0, 0);
    double vector[];
    ArrayResize(vector, KohonenMap.GetFeatureCount());
    ArrayInitialize(vector, 0);
    KohonenMap.CalculateOutput(vector);
  }

  // draw maps into internal BMP buffers
  KohonenMap.Render();

   // draw labels in cells in BMP buffers
  if(hasOneTestPattern)
    KohonenMap.ShowAllPatterns();
  else
    KohonenMap.ShowAllNodes();

  if(ClusterNumber != 0)
  {
    if(mask != 0)
    {
      SetClusterLabels(mask);
    }
    KohonenMap.ShowClusters(ClusterNumber); // mark clusters
    if(SaveClusterAsSettings > -1)
    {
      SaveSettings(KohonenMap.GetID() + "Cluster" + (string)SaveClusterAsSettings + ".set", SaveClusterAsSettings);
    }
  }

  if(ShowRGBOutput && StringLen(FeatureDirection) > 0)
  {
    int rgbmask[];
    ArrayResize(rgbmask, StringLen(FeatureDirection));
    for(int i = 0; i < StringLen(FeatureDirection); i++)
    {
      rgbmask[i] = -(StringGetCharacter(FeatureDirection, i) - ',');
    }
    KohonenMap.EnableRGB(rgbmask);
    KohonenMap.RenderOutput();
    KohonenMap.DisableRGB();
  }
  
  // display files as bitmap images on chart, optionally save into files
  KohonenMap.ShowBMP(SaveImages);
  
  if(NetFileName == "")
  {
    KohonenMap.Save(KohonenMap.GetID());
  }
}

bool LoadSettings(const string filename, double &v[])
{
  int h = FileOpen(filename, FILE_READ | FILE_TXT);
  if(h == INVALID_HANDLE)
  {
    Print("FileOpen error ", filename, " : ",GetLastError());
    return false;
  }
  
  int n = KohonenMap.GetFeatureCount();
  ArrayResize(v, n);
  ArrayInitialize(v, EMPTY_VALUE);
  int count = 0;

  while(!FileIsEnding(h))
  {
    string line = FileReadString(h);
    if(StringFind(line, ";") == 0) continue;
    string name2value[];
    if(StringSplit(line, '=', name2value) != 2) continue;
    int index = KohonenMap.FindFeature(name2value[0]);
    if(index != -1)
    {
      string values[];
      if(StringSplit(name2value[1], '|', values) > 0)
      {
        v[index] = StringToDouble(values[0]);
        count++;
      }
    }
  }
  
  Print("Settings loaded: ", filename, "; features found: ", count);
  
  ulong mask = 0;
  for(int i = 0; i < n; i++)
  {
    if(v[i] != EMPTY_VALUE)
    {
      mask |= (1 << i);
    }
    else
    {
      v[i] = 0;
    }
  }
  if(mask != 0)
  {
    KohonenMap.SetFeatureMask(0, mask);
  }
 
  FileClose(h);
  return count > 0;
}

bool SaveSettings(const string filename, const int cluster = 0)
{
  if(cluster < 0 || cluster >= KohonenMap.GetClusterCount()) return false;

  int h = FileOpen(filename, FILE_WRITE | FILE_TXT);
  if(h == INVALID_HANDLE)
  {
    Print("FileOpen for writing error ", filename, " : ",GetLastError());
    return false;
  }
  
  double center[];
  KohonenMap.GetCluster(cluster, center);
  
  FileWrite(h, "; This file was created by SOM-explorer\n;");
  
  int n = KohonenMap.GetFeatureCount();
  for(int i = 0; i < n; i++)
  {
    FileWrite(h, KohonenMap.GetFeatureTitle(i), "=", (float)center[i], "||", (float)center[i], "||", (float)center[i], "||", (float)(2 * center[i]), "||N");
  }
  
  FileClose(h);
  Print("Settings stub is saved: ", filename);
  return true;
}

void SetClusterLabels(const ulong mask)
{
  if(StringLen(FeatureDirection) == 0) return;
  
  const int nclusters = KohonenMap.GetClusterCount();

  double min, max, best;
  
  double bests[][2]; // [][0 - value; 1 - feature index]
  ArrayResize(bests, nclusters);
  ArrayInitialize(bests, 0);
  
  int n = KohonenMap.GetFeatureCount();
  for(int i = 0; i < n; i++)
  {
    if((mask & (1 << i)) != 0)
    {
      int direction = 0;
      if(i < StringLen(FeatureDirection))
      {
        KohonenMap.GetFeatureBounds(i, min, max);
        if(max - min > 0)
        {
          direction = -(StringGetCharacter(FeatureDirection, i) - ',');
          best = 0;
          double center[];
          for(int j = nclusters - 1; j >= 0; j--)
          {
            KohonenMap.GetCluster(j, center);
            double value = MathMin(MathMax((center[i] - min) / (max - min), 0), 1);
            if(direction == -1) value = 1 - value;
            if(value > bests[j][0])
            {
              bests[j][0] = value;
              bests[j][1] = i;
            }
          }
        }
      }
    }
  }

  // Remove duplicated labels or allow multiple clusters with the same feature label
  // (can be helpful to describe secondary choices)
  if(!AllowDuplicatedLabels)
  {
    double features[][2]; // [][0 - value, 1 - cluster index]
    ArrayResize(features, n);
    ArrayInitialize(features, -1);
    
    for(int j = 0; j < nclusters; j++)
    {
      if(bests[j][0] > features[(int)bests[j][1]][0])
      {
        if(features[(int)bests[j][1]][1] > -1)
        {
          bests[(int)features[(int)bests[j][1]][1]][0] = 0;
        }
        features[(int)bests[j][1]][0] = bests[j][0];
        features[(int)bests[j][1]][1] = j;
      }
      else
      {
        bests[j][0] = 0;
      }
    }
  }
  
  for(int j = 0; j < nclusters; j++)
  {
    if(bests[j][0] > 0)
    {
      KohonenMap.SetLabel(j, KohonenMap.GetFeatureTitle((int)bests[j][1]));
    }
  }
}
