//+------------------------------------------------------------------+
//|                                              WriteDataToFile.mq5 |
//|                                  Blog: http://tol64.blogspot.com |
//|            Copyright 2012, https://login.mql5.com/en/users/tol64 |
//+------------------------------------------------------------------+
#property copyright   "Copyright 2012, tol64"
#property link        "https://login.mql5.com/en/users/tol64"
#property description "email: hello.tol64@gmail.com"
#property version     "1.00"
#property script_show_inputs
//---
#define nmf __FUNCTION__+": " // Macro expansion of the function name before sending the message to the journal
//---
// Macro expansion with further action selection
#define BC if(curr_mwatch==CURRENT) { break; } if(curr_mwatch==MARKETWATCH || curr_mwatch==ALL_LIST_SYMBOLS) { continue; }
//---
#define TRM_DP TerminalInfoString(TERMINAL_DATA_PATH) // Folder for storing the terminal data
//---
//_________________________________
// HEADER_FORMATS_ENUMERATION
enum FORMAT_HEADERS
  {
   NSDT_5 = 0, // "Date" "Time" "Open" "High" "Low" "Close" "Volume"
   NSDT_6 = 1  // Date,Open,High,Low,Close,Volume
  };
//---
//___________________________
// ENUMERATION_OF_DATA_FORMATS
enum FORMAT_DATETIME
  {
   SEP_POINT1 = 0, // dd.mm.yyyy hh:mm
   SEP_POINT2 = 1, // dd.mm.yyyy, hh:mm
   SEP_SLASH1 = 2, // dd/mm/yyyy hh:mm
   SEP_SLASH2 = 3  // dd/mm/yyyy, hh:mm
  };
//---
//____________________________
// ENUMERATION_OF_FILING_MODES
enum CURRENT_MARKETWATCH
  {
   CURRENT          = 0, // ONLY CURRENT SYMBOLS
   MARKETWATCH      = 1, // MARKETWATCH SYMBOLS
   ALL_LIST_SYMBOLS = 2  // ALL LIST SYMBOLS
  };
//---
//____________________________________________________________________
//+------------------------------------------------------------------+
//| EXTERNAL_PARAMETERS                                              |
//+------------------------------------------------------------------+
input datetime            start_date     = D'01.01.2011'; // Start Date
input datetime            end_date       = D'18.09.2012'; // End Date
input FORMAT_HEADERS      format_headers = NSDT_5;        // Format Headers
input FORMAT_DATETIME     format_date    = SEP_POINT2;    // Format Datetime
input CURRENT_MARKETWATCH curr_mwatch    = CURRENT;       // Mode Write Symbols
input bool                clear_mwatch   = true;          // Clear Market Watch
input bool                show_progress  = true;          // Show Progress (%)
//---
//____________________________________________________________________
//+------------------------------------------------------------------+
//| GLOBAL_VARIABLES_AND_ARRAYS                                      |
//+------------------------------------------------------------------+
MqlRates rates[]; // Array for copying data
//---
string symbols[]; // Symbol array
//---
// Array of graphic object names
string arr_nmobj[22]=
  {
   "fon","hd01",
   "nm01","nm02","nm03","nm04","nm05","nm06","nm07","nm08","nm09","nm10",
   "nm11","nm12","nm13","nm14","nm15","nm16","nm17","nm18","nm19","nm20"
  };
//---
// Array of displayed text containing graphic objects
string arr_txtobj[21];
//---
string path="";         // File path
int cnt_symb=0;         // Number of symbols
int sz_arr_symb=0;      // Symbol array size
int bars=0;             // Number of bars according to the specified TF
int copied_bars=0;      // Number of bars copied for writing
double pgs_pcnt=0;      // Writing progress
int hFl=INVALID_HANDLE; // File handle
//---
string   // Variables for data formatting
sdt="",  // Date line
dd="",   // Day
mm="",   // Month
yyyy="", // Year
tm="",   // Time
sep="";  // Separator
//---
int max_bars=0; // Maximum number of bars in the terminal settings
//---
datetime
first_date=0,        // First available data in a specified period
first_termnl_date=0, // First available data in the terminal's database
first_server_date=0, // First available data in the server's database
check_start_date=0;  // Checked correct date value
//---
//____________________________________________________________________
//+------------------------------------------------------------------+
//| SCRIPT >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> |
//+------------------------------------------------------------------+
void OnStart()
  {
// If user-defined parameters are incorrect,
// error message is shown and the program is closed
   if(ValidationParameters()) { return; }
//---
   max_bars=TerminalInfoInteger(TERMINAL_MAXBARS); // Receive available number of bars in the window
//---
   GetSymbolsToArray(); // Filling symbol array with names
   sz_arr_symb=ArraySize(symbols); // Receive symbol array size
//---
   SetSeparateForFormatDate(); // Set a separator for date format
//---
// Revise all symbols and write their data to file
   for(int s=0; s<=sz_arr_symb-1; s++)
     {
      copied_bars=0; // Reset copied bars variable to zero for writing
      pgs_pcnt=0.0; // Reset variable of the symbol data writing progress
      //---
      InfoTable(s); ChartRedraw();
      //---
      // Receive current symbol data
      int res=GetDataCurrentSymbol(s);
      //---
      if(res==0) { BC } // If zero, break the loop or start the next iteration
      //---
      if(res==2) // Program operation interrupted by user
        {
         DelAllScriptObjects(); // Deleted objects created by the script from the chart
         //---
         Print("------\nUser deleted the script!"); break;
        }
      //---
      // Receive the path for creating the file and create directories for them
      // If the string is empty, break the loop or start the next iteration
      if((path=CheckCreateGetPath(s))=="") { BC }
      //---
      WriteDataToFile(s); // Write data to file
     }
//---
// Delete symbols from Market Watch window if necessary
   DelSymbolsFromMarketWatch();
//---
// Delete objects created by the script from the chart
   Sleep(1000); DelAllScriptObjects();
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CHECKING_CORRECTNESS_OF_PARAMETERS                               |
//+------------------------------------------------------------------+
bool ValidationParameters()
  {
   if(start_date>=end_date)
     {
      MessageBox("The start date should be earlier than the ending one!\n\n"
                 "Application cannot continue. Please retry.",
                 //---
                 "Parameter error!",MB_ICONERROR);
      //---
      return(true);
     }
//---
   if(format_headers==NSDT_5 && 
      (format_date==SEP_POINT1 || format_date==SEP_SLASH1))
     {
      MessageBox("For the headers of the following format:\n\n"
                 "\"Date\" ""\"Time\" ""\"Open\" ""\"High\" ""\"Low\" ""\"Close\" ""\"Volume\"\n\n"
                 "Date/time format can be selected out of two versions:\n\n"
                 "dd.mm.yyyy, hh:mm\n"
                 "dd/mm/yyyy, hh:mm\n\n"
                 "Application cannot continue. Please retry.",
                 //---
                 "Header and date/time formats do not match!",MB_ICONERROR);
      //---
      return(true);
     }
//---
   if(format_headers==NSDT_6 && 
      (format_date==SEP_POINT2 || format_date==SEP_SLASH2))
     {
      MessageBox("For the headers of the following format:\n\n"
                 "Date,Open,High,Low,Close,Volume\n\n"
                 "Date/time format can be selected out of two versions:\n\n"
                 "dd.mm.yyyy hh:mm\n"
                 "dd/mm/yyyy hh:mm\n\n"
                 "Application cannot continue. Please retry.",
                 //---
                 "Header and date/time formats do not match!",MB_ICONERROR);
      //---
      return(true);
     }
//---
   return(false);
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| DELETE_SYMBOLS_FROM_MARKET_WATCH_WINDOW                          |
//+------------------------------------------------------------------+
void DelSymbolsFromMarketWatch()
  {
   if(clear_mwatch) // If "extra" characters should be deleted from market Watch window
     {
      for(int s=0; s<=sz_arr_symb-1; s++)
        {
         if(s==0) { Print("------"); }
         //---
         if(!SymbolSelect(symbols[s],false))
           { Print("Chart with "+symbols[s]+" symbol is open. Symbol will stay in Market Watch window."); }
        }
     }
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| FILLING_SYMBOL_ARRAY_WITH_NAMES                                  |
//+------------------------------------------------------------------+
void GetSymbolsToArray()
  {
// If only the current symbol data is required
   if(curr_mwatch==CURRENT)
     { ArrayResize(symbols,1); symbols[0]=_Symbol; }
//---
// If data on all symbols  from Market Watch window or
// or the entire symbol list is required
   if(curr_mwatch==MARKETWATCH || curr_mwatch==ALL_LIST_SYMBOLS)
     {
      // Receive the number of symbols in Market Watch window
      cnt_symb=SymbolsTotal(MWatchOrAllList());
      //---
      for(int i=0; i<=cnt_symb-1; i++)
        {
         string nm_symb="";
         //---
         ArrayResize(symbols,i+1); // Increase the array size by one once again
         //---
         // Receive a name of a symbol from Market Watch window
         nm_symb=SymbolName(i,MWatchOrAllList());
         symbols[i]=nm_symb; // Put the symbol name into the array
        }
     }
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| POINTER_TO_MARKET_WATCH_WINDOW_OR_TO_COMMON_LIST                 |
//+------------------------------------------------------------------+
bool MWatchOrAllList()
  {
   if(curr_mwatch==MARKETWATCH) { return(true); }
   if(curr_mwatch==ALL_LIST_SYMBOLS) { return(false); }
//---
   return(true);
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| DEFINING_SEPARATOR_FOR_DATE_FORMAT                               |
//+------------------------------------------------------------------+
void SetSeparateForFormatDate()
  {
   switch(format_date)
     {
      case SEP_POINT1 : case SEP_POINT2 : sep="."; break; // Full point as a separator
      case SEP_SLASH1 : case SEP_SLASH2 : sep="/"; break; // Slash as a separator
     }
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| RECEIVE_SYMBOL_DATA                                              |
//+------------------------------------------------------------------+
int GetDataCurrentSymbol(int s)
  {
   Print("------\n"+IS(s+1)+" >>>"); // Save a symbol number in the journal
//---
// Check and download the necessary amount of requested data
   int res=CheckLoadHistory(s,_Period);
//---
   InfoTable(s); ChartRedraw(); // Update the data in the data table
//---
   switch(res)
     {
      case -1 : Print("Unknown symbol "+symbols[s]+" (code: -1)!");                         return(0);
      case -2 :
         Print("Number of requested bars exceeds the maximum number that can be displayed on a chart (code: -2)!...\n"
               "...The available amount of data will be used for writing.");                break;
      //---
      case -3 : Print("Execution interrupted by user (code: -3)!");                         return(2);
      case -4 : Print("Download failed (code: -4)!");                                       return(0);
      case  0 : Print("All symbol data downloaded (code: 0).");                             break;
      case  1 : Print("Time series data is sufficient (code: 1).");                         break;
      case  2 : Print("Time series created based on existing terminal data (code: 2).");    break;
      //---
      default : Print("Execution result is not defined!");
     }
//---
// Copy data to the array
   if(CopyRates(symbols[s],_Period,check_start_date,end_date,rates)<=0)
     { Print("Error when copying symbol data "+symbols[s]+" - ",ErrorDesc(Error())+""); return(0); }
   else
     {
      copied_bars=ArraySize(rates); // Receive array size
      //---
      Print("Symbol: ",symbols[s],"; Timeframe: ",gStrTF(_Period),"; Copied bars: ",copied_bars);
     }
//---
   return(1); // Return 1, if all is well
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CHECK_AND_DOWNLOAD_NECESSARY_AMOUNT_OF_REQUESTED_DATA            |
//+------------------------------------------------------------------+
int CheckLoadHistory(int s,ENUM_TIMEFRAMES period)
  {
   datetime times[100]; // Array for gradual data downloading
//---
// Check and add a symbol in Market Watch window if it is not there
   if(!SymbolInfoInteger(symbols[s],SYMBOL_SELECT))
     {
      if(Error()==ERR_MARKET_UNKNOWN_SYMBOL)
        { return(-1); } // Returns -1 in case of "Unknown symbol" error
      //---
      SymbolSelect(symbols[s],true); // Add symbol in Market Watch window
     }
//---
// Check already existing amount of history
   SeriesInfoInteger(symbols[s],period,SERIES_FIRSTDATE,first_date); // Request first history date
//---
   if(first_date>0 && first_date<=start_date) { check_start_date=start_date; return(1); } // Return 1 if there is enough history for operation
//---
// If this stage is reached, there is not enough history
// Find out the initial date for which the minute data is available
   if(SeriesInfoInteger(symbols[s],PERIOD_M1,SERIES_TERMINAL_FIRSTDATE,first_termnl_date))
     {
      // If the data is present
      if(first_termnl_date>0)
        {
         // Forced generation of a time series by copying the time of the specified time frame's first bar
         CopyTime(symbols[s],period,first_termnl_date+PeriodSeconds(period),1,times);
         //---
         // Second check. Request first history date.
         if(SeriesInfoInteger(symbols[s],period,SERIES_FIRSTDATE,first_date))
           {
            // If the time series has been made of existing terminal data and their amount is sufficient at the moment...
            if(first_date>0 && first_date<=start_date)
              { check_start_date=start_date; return(2); } // ...return 2
           }
        }
     }
//________________________________________________________________
// If this stage is reached, the amount of data is less than requested
// An attempt will be made to download data from the server
//---
   int cnt_try=0,tl_try=2;
//---
// Request the very first data date from the server
   while(!SeriesInfoInteger(symbols[s],PERIOD_M1,SERIES_SERVER_FIRSTDATE,first_server_date))
     {
      if(IsStopped()) { return(-3); } // Execution interrupted by user
      //---
      cnt_try++; // Increase the counter
      //---
      Print(nmf,"Please wait. The program is trying to download symbol history ["+symbols[s]+"] (attempt: "+IS(cnt_try)+"/"+IS(tl_try)+")...");
      //---
      // There will be two attempts in total
      if(cnt_try==2) { break; } Sleep(1000);
     }
//---
   Print("The very first data date on the server: ",first_server_date);
//---
// If the server has more data that requested
   if(first_server_date>start_date)
     { check_start_date=first_server_date; } // ...download it (save the time of the first bar)
//---
// If the amount of existing data in the terminal database exceeds the amount of data on the server
   if(first_date>0 && first_date<first_server_date) // Display the appropriate message
     {
      Print("Attention! Amount of data in the terminal database exceeds the amount of data on the server.\n",
            "The very first data date on the server: ",first_server_date,".\n",
            "The very first data date in the terminal database: ",first_termnl_date);
     }
//---
// Gradual download of 100 bars up to the initial date
   cnt_try=0; // Reset the attempt counter
//---
// The loop will be executed till it is stopped by a user...
   while(!IsStopped()) // ...or the counter is overloaded in the loop body
     {
      // Check till the history is synchronized
      while(!SeriesInfoInteger(symbols[s],period,SERIES_SYNCHRONIZED))
        {
         if(IsStopped()) { return(-3); } // Execution interrupted by user
         //---
         bool flg_msg=false;
         if(!flg_msg) { Print(nmf,"Please wait. Synchronizing history..."); flg_msg=true; }
         Sleep(1000);
        }
      //---
      // Receive the number of bars for the specified time frame
      bars=Bars(symbols[s],period);
      //---
      if(bars>0) // If there is some result...
        {
         // ...if the number of the bars for the specified period exceeds ...
         // the number specified in "Max number of bars in chart" terminal option...
         if(bars>=max_bars)
           {
            // Correct the date, starting from which the data is copied
            datetime time_fb[];
            //---
            // Receive the first available date and save it
            CopyTime(symbols[s],period,bars,1,time_fb);
            check_start_date=time_fb[0];
            //---
            return(-2); // ... return -2
           }
         //---
         // Request the very first date for the specified period
         if(SeriesInfoInteger(symbols[s],period,SERIES_FIRSTDATE,first_date))
           {
            // If the date is less or equal to the specified initial date in the script settings...
            if(first_date>0 && first_date<=check_start_date) { return(0); } // ...return 0
           }
        }
      //---
      // If this stage is reached, there is not enough data
      // Request the new batch of 100 bars of history from the trade server from the last available bar having bars index
      int copied=CopyTime(symbols[s],period,bars,100,times);
      //---
      if(copied>0) // If request is successful
        {
         // Check the data
         // If the time of the first available bar is now less or equal to the specified initial one...
         if(times[0]<=check_start_date) { return(0); } // ...return 0
         //---
         // If the number of (bars+copied amount) exceeds or is equal to...
         // ...the amount specified in "Max number of bars in chart" terminal option...
         if(bars+copied>=max_bars)
           {
            // Correct the date, starting from which the data is copied
            datetime time_fb[];
            CopyTime(symbols[s],period,max_bars,1,time_fb);
            check_start_date=time_fb[0];
            //---
            return(-2); // ...return -2
           }
         //---
         cnt_try=0;
        }
      else // If request failed...
        {
         // ...increase the counter of failed attempts
         cnt_try++;
         //---
         // If there were 2 attempts already...
         if(cnt_try>=1) { return(-4); } // ...return -4
         //---
         Print(nmf,"Please wait. Downloading data..."); Sleep(1000);
        }
     }
//---
// Execution interrupted by user
   return(-3);
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| WRITE_DATA_TO_FILE                                               |
//+------------------------------------------------------------------+
void WriteDataToFile(int s)
  {
// Number of decimal places in the symbol price
   int dgt=(int)SymbolInfoInteger(symbols[s],SYMBOL_DIGITS);
//---
   string nm_fl=path+symbols[s]+"_"+gStrTF(_Period)+".csv"; // File name
//---
// Receive file handle for writing
   hFl=FileOpen(nm_fl,FILE_WRITE|FILE_CSV|FILE_ANSI,',');
//---
   if(hFl>0) // If the handle is received
     {
      // Write the headers
      if(format_headers==NSDT_5)
        { FileWrite(hFl,"\"Date\" ""\"Time\" ""\"Open\" ""\"High\" ""\"Low\" ""\"Close\" ""\"Volume\""); }
      //---
      if(format_headers==NSDT_6)
        { FileWrite(hFl,"Date","Open","High","Low","Close","Volume"); }
      //---
      // Write the data
      for(int i=0; i<=copied_bars-1; i++)
        {
         if(IsStopped()) // If program execution interrupted by a user
           {
            DelAllScriptObjects(); // Deleted objects created by the script from the chart
            //---
            Print("------\nUser deleted the script!"); break;
           }
         //---
         sdt=TSdm(rates[i].time); // Bar open time
         //---
         // Divide the date by year, month and time
         yyyy=StringSubstr(sdt,0,4);
         mm=StringSubstr(sdt,5,2);
         dd=StringSubstr(sdt,8,2);
         tm=StringSubstr(sdt,11);
         //---
         string sep_dt_tm=""; // Separator of Date and Time columns
         //---
         // Join the data with the separator in the necessary order
         if(format_date==SEP_POINT1 || format_date==SEP_SLASH1) { sep_dt_tm=" "; }
         if(format_date==SEP_POINT2 || format_date==SEP_SLASH2) { sep_dt_tm=","; }
         //---
         // Join everything in one line
         StringConcatenate(sdt,dd,sep,mm,sep,yyyy,sep_dt_tm,tm);
         //---
         FileWrite(hFl,
                   sdt,// Date-time
                   DS_dgt(rates[i].open,dgt),      // Open price
                   DS_dgt(rates[i].high,dgt),      // High price
                   DS_dgt(rates[i].low,dgt),       // Low price
                   DS_dgt(rates[i].close,dgt),     // Close price
                   IS((int)rates[i].tick_volume)); // Tick volume price
         //---
         // Update writing progress value for the current symbol
         pgs_pcnt=((double)(i+1)/copied_bars)*100;
         //---
         // Update data in the table
         InfoTable(s); if(show_progress) { ChartRedraw(); }
        }
      //---
      FileClose(hFl); // Close the file
     }
   else { Print("Error when creating/opening file!"); }
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CHECK_DIRECTORY_AND_CREATE_NECESSARY_DATA_FOLDERS                |
//+------------------------------------------------------------------+
string CheckCreateGetPath(int s)
  {
   int i=1;
   long search=-1;
   string ffname="",lpath="";
   string file="*.csv",folder="*";
   string
   root="DATA_OHLC\\",// Root data folder
   fSmb=symbols[s]+"\\",// Symbol name
   fTF=gStrTF(_Period)+"\\"; // Symbol time frame
//---
   bool flgROOT=false,flgSYMBOL=false;
//---
//+------------------------------------------------------------------+
//| SEARCHING_FOR_DATA_OHLC_ROOT_FOLDER                              |
//+------------------------------------------------------------------+
   lpath=folder;
   search=FileFindFirst(lpath,ffname); // Set search handle in Metatrader 5\MQL5\Files
//---
   Print("Directory: ",TRM_DP+"\\MQL5\\Files\\");
//---
// Set the flag if the first folder is a root one
   if(ffname==root)
     { flgROOT=true; Print("Root folder "+root+" present"); }
//---
   if(search!=INVALID_HANDLE) // If search handle received
     {
      if(!flgROOT) // If the first folder is not a root one
        {
         // Sort out all files searching for the root folder
         while(FileFindNext(search,ffname))
           {
            if(IsStopped()) // Execution interrupted by user
              {
               // Delete objects created by the script from the chart
               DelAllScriptObjects();
               //---
               Print("------\nUser deleted the script!"); return("");
              }
            //---
            if(ffname==root) // Set the flag if found
              { flgROOT=true; Print("Root folder "+root+" present"); break; }
           }
        }
      //---
      FileFindClose(search); search=-1; // Close root folder search handle
     }
   else { Print("Error when receiving the search handle or directory "+TRM_DP+" is empty: ",ErrorDesc(Error())); }
//---
//+------------------------------------------------------------------+
//| SEARCHING_SYMBOL_FOLDER                                          |
//+------------------------------------------------------------------+
   lpath=root+folder;
//---
// Set search handle in the root folder ..\Files\DATA OHLC\
   search=FileFindFirst(lpath,ffname);
//---
// Set the flag if the first folder of the current symbol
   if(ffname==fSmb) { flgSYMBOL=true; Print("Symbol folder "+fSmb+" present"); }
//---
   if(search!=INVALID_HANDLE) // If search handle received
     {
      if(!flgSYMBOL) // If the first folder is not of the current symbol
        {
         // Sort out all the files in the root folder searching the symbol folder
         while(FileFindNext(search,ffname))
           {
            if(IsStopped()) // Execution interrupted by user
              {
               // Delete objects created by the script from the chart
               DelAllScriptObjects();
               //---
               Print("------\nUser deleted the script!"); return("");
              }
            //---
            if(ffname==fSmb) // Set the flag if found
              { flgSYMBOL=true; Print("Symbol folder "+fSmb+" present"); break; }
           }
        }
      //---
      FileFindClose(search); search=-1; // Close symbol folder search handle
     }
   else { Print("Error when receiving search handle or the directory "+path+" is empty"); }
//---
//+------------------------------------------------------------------+
//| CREATE_NECESSARY_DIRECTORIES_ACCORDING_TO_CHECK_RESULTS          |
//+------------------------------------------------------------------+
   if(!flgROOT) // If there is no DATA_OHLC... root folder
     {
      if(FolderCreate("DATA_OHLC")) // ...we should create it
        { Print("root folder created ..\DATA_OHLC\\"); }
      else
        { Print("Error when creating DATA_OHLC: ",ErrorDesc(Error())); return(""); }
     }
//---
   if(!flgSYMBOL) // If there is no folder of the symbol, the values of which should be received...
     {
      if(FolderCreate(root+symbols[s])) // ...we should create it
        {
         Print("..\DATA_OHLC\\ symbol folder created"+fSmb+"");
         //---
         return(root+symbols[s]+"\\"); // Return the path for creating the file for writing
        }
      else
        { Print("Error when creating ..\DATA_OHLC\\ symbol folder"+fSmb+"\: ",ErrorDesc(Error())); return(""); }
     }
//---
   if(flgROOT && flgSYMBOL)
     {
      return(root+symbols[s]+"\\"); // Return the path for creating the file for writing
     }
//---
   return("");
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| INFORMATION_PANEL                                                |
//|------------------------------------------------------------------+
void InfoTable(int s)
  {
   int fnt_sz=8;            // Font size
   string fnt="Calibri";    // Header font
   color clr=clrWhiteSmoke; // Color
//---
   int xH=300;
   int height_pnl=0;
   int yV1=1,yV2=12,xV1=165,xV2=335,xV3=1;
//---
   string sf="",stf="",ssf="";
   bool flg_sf=false,flg_stf=false,flg_ssf=false;
//---
   if(show_progress) { height_pnl=138; } else { height_pnl=126; }
//---
   flg_sf=SeriesInfoInteger(symbols[s],_Period,SERIES_FIRSTDATE,first_date);
   flg_stf=SeriesInfoInteger(symbols[s],PERIOD_M1,SERIES_TERMINAL_FIRSTDATE,first_termnl_date);
   flg_ssf=SeriesInfoInteger(symbols[s],PERIOD_M1,SERIES_SERVER_FIRSTDATE,first_server_date);
//---
   if(flg_sf) { sf=TSdm(first_date); } else { sf="?"; }
   if(flg_stf) { stf=TSdm(first_termnl_date); } else { stf="?"; }
   if(flg_ssf) { ssf=TSdm(first_server_date); } else { ssf="?"; }
//---
   if(cnt_symb==0) { cnt_symb=1; }
//---
   int anchor1=ANCHOR_LEFT_UPPER,anchor2=ANCHOR_RIGHT_UPPER,corner=CORNER_LEFT_UPPER;
//---
   string path_symbol=SymbolInfoString(symbols[s],SYMBOL_PATH);
   path_symbol=StringSubstr(path_symbol,0,StringLen(path_symbol)-StringLen(symbols[s]));
//---
   arr_txtobj[0]="INFO TABLE";
   arr_txtobj[1]="Symbol (current / total) : ";
   arr_txtobj[2]=""+symbols[s]+" ("+IS(s+1)+"/"+IS(cnt_symb)+")";
   arr_txtobj[3]="Path Symbol : ";
   arr_txtobj[4]=path_symbol;
   arr_txtobj[5]="Timeframe : ";
   arr_txtobj[6]=gStrTF(_Period);
   arr_txtobj[7]="Input Start Date : ";
   arr_txtobj[8]=TSdm(start_date);
   arr_txtobj[9]="First Date (H1) : ";
   arr_txtobj[10]=sf;
   arr_txtobj[11]="First Terminal Date (M1) : ";
   arr_txtobj[12]=stf;
   arr_txtobj[13]="First Server Date (M1) : ";
   arr_txtobj[14]=ssf;
   arr_txtobj[15]="Max. Bars In Options Terminal : ";
   arr_txtobj[16]=IS(max_bars);
   arr_txtobj[17]="Copied Bars : ";
   arr_txtobj[18]=IS(copied_bars);
   arr_txtobj[19]="Progress Value Current Symbol : ";
   arr_txtobj[20]=DS(pgs_pcnt,2)+"%";
//---
   Create_Edit(0,0,arr_nmobj[0],"",corner,fnt,fnt_sz,clrDimGray,clrDimGray,345,height_pnl,xV3,yV1,2,C'15,15,15');
//---
   Create_Edit(0,0,arr_nmobj[1],arr_txtobj[0],corner,fnt,8,clrWhite,C'64,0,0',345,12,xV3,yV1,2,clrFireBrick);
//---
   Create_Label(0,arr_nmobj[2],arr_txtobj[1],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2,0);
   Create_Label(0,arr_nmobj[3],arr_txtobj[2],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2,0);
//---
   Create_Label(0,arr_nmobj[4],arr_txtobj[3],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*2,0);
   Create_Label(0,arr_nmobj[5],arr_txtobj[4],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*2,0);
//---
   Create_Label(0,arr_nmobj[6],arr_txtobj[5],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*3,0);
   Create_Label(0,arr_nmobj[7],arr_txtobj[6],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*3,0);
//---
   Create_Label(0,arr_nmobj[8],arr_txtobj[7],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*4,0);
   Create_Label(0,arr_nmobj[9],arr_txtobj[8],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*4,0);
//---
   Create_Label(0,arr_nmobj[10],arr_txtobj[9],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*5,0);
   Create_Label(0,arr_nmobj[11],arr_txtobj[10],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*5,0);
//---
   Create_Label(0,arr_nmobj[12],arr_txtobj[11],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*6,0);
   Create_Label(0,arr_nmobj[13],arr_txtobj[12],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*6,0);
//---
   Create_Label(0,arr_nmobj[14],arr_txtobj[13],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*7,0);
   Create_Label(0,arr_nmobj[15],arr_txtobj[14],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*7,0);
//---
   Create_Label(0,arr_nmobj[16],arr_txtobj[15],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*8,0);
   Create_Label(0,arr_nmobj[17],arr_txtobj[16],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*8,0);
//---
   Create_Label(0,arr_nmobj[18],arr_txtobj[17],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*9,0);
   Create_Label(0,arr_nmobj[19],arr_txtobj[18],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*9,0);
//---
   if(show_progress)
     {
      Create_Label(0,arr_nmobj[20],arr_txtobj[19],anchor2,corner,fnt,fnt_sz,clr,xV1,yV1+yV2*10,0);
      Create_Label(0,arr_nmobj[21],arr_txtobj[20],anchor2,corner,fnt,fnt_sz,clr,xV2,yV1+yV2*10,0);
     }
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CREATING_LABEL_OBJECT                                            |
//+------------------------------------------------------------------+
void Create_Label(long   chrt_id,   // chart id
                  string lable_nm,  // object name
                  string rename,    // displayed name
                  long   anchor,    // anchor point
                  long   corner,    // attachment corner
                  string font_bsc,  // font
                  int    font_size, // font size
                  color  font_clr,  // font color
                  int    x_dist,    // X scale coordinate
                  int    y_dist,    // Y scale coordinate
                  long   zorder)    // priority
  {
   if(ObjectCreate(chrt_id,lable_nm,OBJ_LABEL,0,0,0)) // creating object
     {
      ObjectSetString(chrt_id,lable_nm,OBJPROP_TEXT,rename);          // set name
      ObjectSetString(chrt_id,lable_nm,OBJPROP_FONT,font_bsc);        // set font
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_COLOR,font_clr);      // set font color
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_ANCHOR,anchor);       // set anchor point
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_CORNER,corner);       // set attachment corner
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_FONTSIZE,font_size);  // set font size
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_XDISTANCE,x_dist);    // set X coordinates
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_YDISTANCE,y_dist);    // set Y coordinates
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_SELECTABLE,false);    // unable to highlight the object, if FALSE
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_ZORDER,zorder);       // Higher/lower priority
      ObjectSetString(chrt_id,lable_nm,OBJPROP_TOOLTIP,"\n");         // no tooltip, if "\n"
     }
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CREATING_EDIT_OBJECT                                             |
//+------------------------------------------------------------------+
void Create_Edit(long   chrt_id,       // chart id
                 int    nmb_win,       // window (subwindow) index
                 string lable_nm,      // object name
                 string text,          // displayed text
                 long   corner,        // attachment corner
                 string font_bsc,      // font
                 int    font_size,     // font size
                 color  font_clr,      // font color
                 color  font_clr_brd,  // font color
                 int    xsize,         // width
                 int    ysize,         // height
                 int    x_dist,        // X scale coordinate
                 int    y_dist,        // Y scale coordinate
                 long   zorder,        // priority
                 color  clr)           // background color
  {
   if(ObjectCreate(chrt_id,lable_nm,OBJ_EDIT,nmb_win,0,0)) // creating object
     {
      ObjectSetString(chrt_id,lable_nm,OBJPROP_TEXT,text);                     // set name
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_CORNER,corner);                // set attachment corner
      ObjectSetString(chrt_id,lable_nm,OBJPROP_FONT,font_bsc);                 // set font
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_ALIGN,ALIGN_CENTER);           // center alignment
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_FONTSIZE,font_size);           // set font size
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_COLOR,font_clr);               // font color
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_BORDER_COLOR,font_clr_brd);    // background color
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_BGCOLOR,clr);                  // background color
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_XSIZE,xsize);                  // width
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_YSIZE,ysize);                  // height
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_XDISTANCE,x_dist);             // set X coordinate
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_YDISTANCE,y_dist);             // set Y coordinate
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_SELECTABLE,false);             // unable to highlight the object, if FALSE
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_ZORDER,zorder);                // Higher/lower priority
      ObjectSetInteger(chrt_id,lable_nm,OBJPROP_READONLY,true);                // Read only
      ObjectSetString(chrt_id,lable_nm,OBJPROP_TOOLTIP,"\n");                  // no tooltip if "\n"
     }
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| DELETE_ALL_GRAPHICAL_OBJECTS_CREATED_BY_THE_SCRIPT               |
//+------------------------------------------------------------------+
void DelAllScriptObjects()
  {
// Receive the size of graphical object names array
   int sz_arr1=ArraySize(arr_nmobj);
//---
// Delete all objects
   for(int i=0; i<sz_arr1; i++)
     { DelObjbyName(arr_nmobj[i]);  }
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| DELETE_OBJECTS_BY_NAME                                           |
//+------------------------------------------------------------------+
int DelObjbyName(string Name)
  {
   int nm_obj=0;
   bool res=false;
//---
   nm_obj=ObjectFind(ChartID(),Name);
//---
   if(nm_obj>=0)
     {
      res=ObjectDelete(ChartID(),Name);
      //---
      if(!res) { Print("Object deletion error: - "+ErrorDesc(Error())+""); return(false); }
     }
//---
   return(res);
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CONVERSION_FROM_DATETIME_TO_STRING_IN_TIME_DATE_FORMAT           |
//+------------------------------------------------------------------+
string TSd(datetime aValue)
  {
   return(TimeToString(aValue,TIME_DATE));
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CONVERSION_FROM_DATETIME_TO_STRING_                              |
//| IN_TIME_DATE|TIME_MINUTES_FORMAT                                 |
//+------------------------------------------------------------------+
string TSdm(datetime aValue)
  {
   return(TimeToString(aValue,TIME_DATE|TIME_MINUTES));
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CONVERSION_FROM_INTEGER_TO_STRING                                |
//+------------------------------------------------------------------+
string IS(int aValue)
  {
   return(IntegerToString(aValue));
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CONVERSION_FROM_DOUBLE_TO_STRING                                 |
//+------------------------------------------------------------------+
string DS(double aValue,int amount)
  {
   return(DoubleToString(aValue,amount));
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| CONVERSION_FROM_DOUBLE_TO_STRING_WITH_DIGITS_CHARACTERS          |
//+------------------------------------------------------------------+
string DS_dgt(double aValue,int digit)
  {
   return(DoubleToString(aValue,digit));
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| RETURNS_TIMEFRAME_LINE                                           |
//+------------------------------------------------------------------+
string gStrTF(ENUM_TIMEFRAMES TF)
  {
   string str="";
//---
   if(TF==PERIOD_CURRENT) TF=Period();
//---
   switch(TF)
     {
      case PERIOD_M1  : str="M1";  break;
      case PERIOD_M2  : str="M2";  break;
      case PERIOD_M3  : str="M3";  break;
      case PERIOD_M4  : str="M4";  break;
      case PERIOD_M5  : str="M5";  break;
      case PERIOD_M6  : str="M6";  break;
      case PERIOD_M10 : str="M10"; break;
      case PERIOD_M12 : str="M12"; break;
      case PERIOD_M15 : str="M15"; break;
      case PERIOD_M20 : str="M20"; break;
      case PERIOD_M30 : str="M30"; break;
      case PERIOD_H1  : str="H1";  break;
      case PERIOD_H2  : str="H2";  break;
      case PERIOD_H3  : str="H3";  break;
      case PERIOD_H4  : str="H4";  break;
      case PERIOD_H6  : str="H6";  break;
      case PERIOD_H8  : str="H8";  break;
      case PERIOD_H12 : str="H12"; break;
      case PERIOD_D1  : str="D1";  break;
      case PERIOD_W1  : str="W1";  break;
      case PERIOD_MN1 : str="MN1"; break;
      //---
      default : str="Unknown time frame!";
     }
//---
   return(str);
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| RECEIVE_THE_LAST_DETECTED_ERROR                                  |
//+------------------------------------------------------------------+
int Error()
  {
   return(GetLastError());
  }
//____________________________________________________________________
//+------------------------------------------------------------------+
//| RETURN_ERROR_DESCRIPTION                                         |
//+------------------------------------------------------------------+
string ErrorDesc(int error_code)
  {
   string error_string;
//----
   switch(error_code)
     {
      //--- Trade server return codes
      case 10004: error_string=""+IS(Error())+": Requote";                                                         break;
      case 10006: error_string=""+IS(Error())+": Request rejected";                                                break;
      case 10007: error_string=""+IS(Error())+": Request canceled by trader";                                      break;
      case 10008: error_string=""+IS(Error())+": Order placed";                                                    break;
      case 10009: error_string=""+IS(Error())+": Order executed";                                                  break;
      case 10010: error_string=""+IS(Error())+": Order executed partially";                                        break;
      case 10011: error_string=""+IS(Error())+": Request processing error";                                        break;
      case 10012: error_string=""+IS(Error())+": Request canceled due to time limit";                              break;
      case 10013: error_string=""+IS(Error())+": Invalid request";                                                 break;
      case 10014: error_string=""+IS(Error())+": Incorrect volume in request";                                     break;
      case 10015: error_string=""+IS(Error())+": Incorrect price in request";                                      break;
      case 10016: error_string=""+IS(Error())+":  Incorrect stops in request";                                     break;
      case 10017: error_string=""+IS(Error())+": Trading disabled";                                                break;
      case 10018: error_string=""+IS(Error())+": Market closed";                                                   break;
      case 10019: error_string=""+IS(Error())+": Insufficient funds";                                              break;
      case 10020: error_string=""+IS(Error())+": Prices have changed";                                             break;
      case 10021: error_string=""+IS(Error())+": No quotes for request processing";                                break;
      case 10022: error_string=""+IS(Error())+": Invalid order expiration date in request";                        break;
      case 10023: error_string=""+IS(Error())+": Order state changed";                                             break;
      case 10024: error_string=""+IS(Error())+": Too frequent requests";                                           break;
      case 10025: error_string=""+IS(Error())+": No changes in request";                                           break;
      case 10026: error_string=""+IS(Error())+": Autotrading disabled by trader";                                  break;
      case 10027: error_string=""+IS(Error())+": Autotrading disabled by the client terminal";                     break;
      case 10028: error_string=""+IS(Error())+": Request blocked for processing";                                  break;
      case 10029: error_string=""+IS(Error())+": Order or position frozen";                                        break;
      case 10030: error_string=""+IS(Error())+": Specified type of order execution for the balance is not supported";    break;
      case 10031: error_string=""+IS(Error())+": No connection with trade server";                                 break;
      case 10032: error_string=""+IS(Error())+": Operation is allowed only for real accounts";                     break;
      case 10033: error_string=""+IS(Error())+": Pending orders have reached the limit";                           break;
      case 10034: error_string=""+IS(Error())+": Volume of orders and positions for this symbol has reached the limit";  break;

      //--- Execution time errors

      case 0:  // Operation completed successfully
      case 4001: error_string=""+IS(Error())+": Unexpected internal error";                                                                                                           break;
      case 4002: error_string=""+IS(Error())+": Invalid parameter during the internal call of the client terminal function";                                                          break;
      case 4003: error_string=""+IS(Error())+": Invalid parameter during the call of the system function";                                                                            break;
      case 4004: error_string=""+IS(Error())+": Not enough memory for system function execution";                                                                                     break;
      case 4005: error_string=""+IS(Error())+": Structure contains the objects of lines and/or dynamic arrays and/or structures with such an objects and/or classes";                 break;
      case 4006: error_string=""+IS(Error())+": Invalid array type or size or corrupted object of the dynamic array";                                                                 break;
      case 4007: error_string=""+IS(Error())+": Not enough memory for array redistribution or an attempt to change the size of the static array";                                     break;
      case 4008: error_string=""+IS(Error())+": Not enough memory for line redistribution";                                                                                           break;
      case 4009: error_string=""+IS(Error())+": Uninitialized line";                                                                                                                  break;
      case 4010: error_string=""+IS(Error())+": Invalid date and/or time";                                                                                                            break;
      case 4011: error_string=""+IS(Error())+": Requested array size exceeds 2 GB";                                                                                                   break;
      case 4012: error_string=""+IS(Error())+": Invalid pointer";                                                                                                                     break;
      case 4013: error_string=""+IS(Error())+": Invalid pointer type";                                                                                                                break;
      case 4014: error_string=""+IS(Error())+": System function not allowed for calling";                                                                                             break;
      //-- Charts
      case 4101: error_string=""+IS(Error())+": Invalid chart ID";                                                                                                                    break;
      case 4102: error_string=""+IS(Error())+": Chart not responding";                                                                                                                break;
      case 4103: error_string=""+IS(Error())+": Chart not found";                                                                                                                     break;
      case 4104: error_string=""+IS(Error())+": Chart has no Expert Advisor able to process the event";                                                                               break;
      case 4105: error_string=""+IS(Error())+": Error when opening chart";                                                                                                            break;
      case 4106: error_string=""+IS(Error())+": Error when changing chart symbol or period";                                                                                          break;
      case 4107: error_string=""+IS(Error())+": Invalid timer parameter";                                                                                                             break;
      case 4108: error_string=""+IS(Error())+": Error when creating a timer";                                                                                                         break;
      case 4109: error_string=""+IS(Error())+": Invalid chart property ID";                                                                                                           break;
      case 4110: error_string=""+IS(Error())+": Error when creating a screenshot";                                                                                                    break;
      case 4111: error_string=""+IS(Error())+": Error when navigating the chart";                                                                                                     break;
      case 4112: error_string=""+IS(Error())+": Error when applying the template";                                                                                                    break;
      case 4113: error_string=""+IS(Error())+": Subwindow containing specified indicator not found";                                                                                  break;
      case 4114: error_string=""+IS(Error())+": Error when adding indicator to chart";                                                                                                break;
      case 4115: error_string=""+IS(Error())+": Error when removing indicator from the chart";                                                                                        break;
      case 4116: error_string=""+IS(Error())+": Indicator not found on the specified chart";                                                                                          break;
      //-- Graphical objects
      case 4201: error_string=""+IS(Error())+": Error when working with graphical object";                                                                                            break;
      case 4202: error_string=""+IS(Error())+": Graphical object not found";                                                                                                          break;
      case 4203: error_string=""+IS(Error())+": Invalid graphical object property ID";                                                                                                break;
      case 4204: error_string=""+IS(Error())+": Unable to receive the date corresponding to the value";                                                                               break;
      case 4205: error_string=""+IS(Error())+": Unable to receive the value corresponding to the date";                                                                               break;
      //-- MarketInfo
      case 4301: error_string=""+IS(Error())+": Unknown symbol ";                                                                                                                     break;
      case 4302: error_string=""+IS(Error())+": Symbol not selected in MarketWatch";                                                                                                  break;
      case 4303: error_string=""+IS(Error())+": Invalid symbol property ID";                                                                                                          break;
      case 4304: error_string=""+IS(Error())+": Last tick time unknown (no ticks)";                                                                                                   break;
      //-- Access to history
      case 4401: error_string=""+IS(Error())+": Requested history not found!";                                                                                                        break;
      case 4402: error_string=""+IS(Error())+": Invalid history property ID";                                                                                                         break;
      //-- Global_Variables
      case 4501: error_string=""+IS(Error())+": Global variable of the client terminal not found";                                                                                    break;
      case 4502: error_string=""+IS(Error())+": Global variable of the client terminal with the same name already exists";                                                            break;
      case 4510: error_string=""+IS(Error())+": Failed to send email";                                                                                                                break;
      case 4511: error_string=""+IS(Error())+": Failed to play sound";                                                                                                                break;
      case 4512: error_string=""+IS(Error())+": Invalid program property ID";                                                                                                         break;
      case 4513: error_string=""+IS(Error())+": Invalid terminal property ID";                                                                                                        break;
      case 4514: error_string=""+IS(Error())+": Failed to send file via ftp";                                                                                                         break;
      //-- Custom indicator buffers
      case 4601: error_string=""+IS(Error())+": Not enough memory for distribution of indicator buffers";                                                                             break;
      case 4602: error_string=""+IS(Error())+": Invalid index of the custom indicator buffer";                                                                                        break;
      //-- Custom indicator properties
      case 4603: error_string=""+IS(Error())+": Invalid custom indicator property ID";                                                                                                break;
      //-- Account
      case 4701: error_string=""+IS(Error())+": Invalid account property ID";                                                                                                         break;
      case 4751: error_string=""+IS(Error())+": Invalid trading property ID";                                                                                                         break;
      case 4752: error_string=""+IS(Error())+": Trading disabled for the Expert Advisor";                                                                                             break;
      case 4753: error_string=""+IS(Error())+": Position not found";                                                                                                                  break;
      case 4754: error_string=""+IS(Error())+": Order not found";                                                                                                                     break;
      case 4755: error_string=""+IS(Error())+": Deal not found";                                                                                                                      break;
      case 4756: error_string=""+IS(Error())+": Failed to send a trade request";                                                                                                      break;
      //-- Indicators
      case 4801: error_string=""+IS(Error())+": Unknown symbol";                                                                                                                      break;
      case 4802: error_string=""+IS(Error())+": Indicator cannot be created";                                                                                                         break;
      case 4803: error_string=""+IS(Error())+": Not enough memory for adding the indicator";                                                                                          break;
      case 4804: error_string=""+IS(Error())+": Indicator cannot be applied to another indicator";                                                                                    break;
      case 4805: error_string=""+IS(Error())+": Error when adding the indicator";                                                                                                     break;
      case 4806: error_string=""+IS(Error())+": Requested data not found";                                                                                                            break;
      case 4807: error_string=""+IS(Error())+": Invalid indicator handle";                                                                                                            break;
      case 4808: error_string=""+IS(Error())+": Invalid number of parameters when creating an indicator";                                                                             break;
      case 4809: error_string=""+IS(Error())+": No parameters when creating the indicator";                                                                                           break;
      case 4810: error_string=""+IS(Error())+": Custom indicator name should be the first parameter in the array";                                                                    break;
      case 4811: error_string=""+IS(Error())+": Invalid parameter type in the array when creating an indicator";                                                                      break;
      case 4812: error_string=""+IS(Error())+": Invalid index of the requested indicator buffer";                                                                                     break;
      //-- Depth of Market
      case 4901: error_string=""+IS(Error())+": Unable to add Depth of Market";                                                                                                       break;
      case 4902: error_string=""+IS(Error())+": Unable to remove Depth of Market";                                                                                                    break;
      case 4903: error_string=""+IS(Error())+": Unable to receive Depth of Market data";                                                                                              break;
      case 4904: error_string=""+IS(Error())+": Error when subscribing to new Depth of Market data";                                                                                  break;
      //-- File operations
      case 5001: error_string=""+IS(Error())+": Unable to open more than 64 files simultaneously";                                                                                    break;
      case 5002: error_string=""+IS(Error())+": Invalid file name";                                                                                                                   break;
      case 5003: error_string=""+IS(Error())+": File name is too long";                                                                                                               break;
      case 5004: error_string=""+IS(Error())+": Error when opening the file";                                                                                                         break;
      case 5005: error_string=""+IS(Error())+": Not enough memory for reading cache";                                                                                                 break;
      case 5006: error_string=""+IS(Error())+": Error when deleting the file";                                                                                                        break;
      case 5007: error_string=""+IS(Error())+": File with such handle has already been closed or has not been opened at all";                                                         break;
      case 5008: error_string=""+IS(Error())+": Invalid file handle";                                                                                                                 break;
      case 5009: error_string=""+IS(Error())+": File should be open for writing";                                                                                                     break;
      case 5010: error_string=""+IS(Error())+": File should be open for reading";                                                                                                     break;
      case 5011: error_string=""+IS(Error())+": File should be open as a binary one";                                                                                                 break;
      case 5012: error_string=""+IS(Error())+": File should be open as a text one";                                                                                                   break;
      case 5013: error_string=""+IS(Error())+": File should be open as a text or CSV one";                                                                                            break;
      case 5014: error_string=""+IS(Error())+": File should be open as CSV one";                                                                                                      break;
      case 5015: error_string=""+IS(Error())+": Error when reading the file";                                                                                                         break;
      case 5016: error_string=""+IS(Error())+": String size should be specified as the file is opened as a binary one";                                                               break;
      case 5017: error_string=""+IS(Error())+": Text file should be provided for string arrays, while binary one should be provided for the rest ones";                               break;
      case 5018: error_string=""+IS(Error())+": It is a directory, not a file";                                                                                                       break;
      case 5019: error_string=""+IS(Error())+": File does not exist";                                                                                                                 break;
      case 5020: error_string=""+IS(Error())+": File cannot be overwritten";                                                                                                          break;
      case 5021: error_string=""+IS(Error())+": Invalid directory name";                                                                                                              break;
      case 5022: error_string=""+IS(Error())+": Directory does not exist";                                                                                                            break;
      case 5023: error_string=""+IS(Error())+": This is a file, not a directory";                                                                                                     break;
      case 5024: error_string=""+IS(Error())+": Directory cannot be deleted";                                                                                                         break;
      case 5025: error_string=""+IS(Error())+": Unable to clear the directory (one or several files may be blocked and deletion failed)";                                             break;
      //-- Conversion of strings
      case 5030: error_string=""+IS(Error())+": No data in string";                                                                                                                   break;
      case 5031: error_string=""+IS(Error())+": Invalid data in string";                                                                                                              break;
      case 5032: error_string=""+IS(Error())+": Invalid time in string";                                                                                                              break;
      case 5033: error_string=""+IS(Error())+": Error when transforming string to date";                                                                                              break;
      case 5034: error_string=""+IS(Error())+": Not enough memory for string";                                                                                                        break;
      case 5035: error_string=""+IS(Error())+": String length is less than expected";                                                                                                 break;
      case 5036: error_string=""+IS(Error())+": Too big number exceeding ULONG_MAX";                                                                                                  break;
      case 5037: error_string=""+IS(Error())+": Invalid format string";                                                                                                               break;
      case 5038: error_string=""+IS(Error())+": Number of format specifiers exceeds number of parameters";                                                                            break;
      case 5039: error_string=""+IS(Error())+": Number of parameters exceeds number of format specifiers";                                                                            break;
      case 5040: error_string=""+IS(Error())+": Corrupted string type parameter";                                                                                                     break;
      case 5041: error_string=""+IS(Error())+": Position is outside the string";                                                                                                      break;
      case 5042: error_string=""+IS(Error())+": 0 added at the end of the string, meaningless operation";                                                                             break;
      case 5043: error_string=""+IS(Error())+": Unknown data type when converting to string";                                                                                         break;
      case 5044: error_string=""+IS(Error())+": Corrupted string object";                                                                                                             break;
      //-- Working with arrays
      case 5050: error_string=""+IS(Error())+": Copying incompatible arrays. String array should be copied only to string one, while numeric array - to numeric one";                 break;
      case 5051: error_string=""+IS(Error())+": Receiving array declared as AS_SERIES and it has insufficient size";                                                                  break;
      case 5052: error_string=""+IS(Error())+": Too small array, starting position is outside the array";                                                                             break;
      case 5053: error_string=""+IS(Error())+": Zero length array";                                                                                                                   break;
      case 5054: error_string=""+IS(Error())+": There should be a numeric array";                                                                                                     break;
      case 5055: error_string=""+IS(Error())+": There should be a one-dimensional array";                                                                                             break;
      case 5056: error_string=""+IS(Error())+": Unable to use time series";                                                                                                           break;
      case 5057: error_string=""+IS(Error())+": There should be a double type array";                                                                                                 break;
      case 5058: error_string=""+IS(Error())+": There should be a float type array";                                                                                                  break;
      case 5059: error_string=""+IS(Error())+": There should be a long type array";                                                                                                   break;
      case 5060: error_string=""+IS(Error())+": There should be an int type array";                                                                                                   break;
      case 5061: error_string=""+IS(Error())+": There should be a short type array";                                                                                                  break;
      case 5062: error_string=""+IS(Error())+": There should be a char type array";                                                                                                   break;
      //-- Custom errors

      default: error_string="Error is not defined!";
     }
//----
   return(error_string);
  }
//+------------------------------------------------------------------+
