//+------------------------------------------------------------------+
//|                                      Hilhorst-Schehr-problem.mq5 |
//| H. J. Hilhorst and G. Schehr                                     |
//| A note on q-Gaussians and non-Gaussians in statistical mechanics |
//| J.Stat.Mech.(2007)P06003, http://arxiv.org/pdf/0705.0600v2.pdf   |
//|                        Copyright 2012, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2012, MetaQuotes Software Corp."
#property link      "http://www.mql5.com"
#property version   "1.00"
//+------------------------------------------------------------------+
//| erf                                                              |
//+------------------------------------------------------------------+
double erf(double x)
  {
   double r=x;
   double z=x;
   int n=0;
   do
     {
      n++;
      z=-z*x*x*(2*n-1)/((2*n+1)*n*1.0);
      r+=z;
     }
   while(MathAbs(z)>1E-10 && n<100);
   r=2*r/MathSqrt(M_PI);
   return(r);
  }
const
double coef_a[4] = { 0.886226899, -1.645349621,  0.914624893, -0.140543331};
double coef_b[4] = {-2.118377725,  1.442710462, -0.329097515,  0.012229801};
double coef_c[4] = {-1.970840454, -1.624906493,  3.429567803,  1.641345311};
double coef_d[2] = { 3.543889200,  1.637067800};
//+------------------------------------------------------------------+
//| erfinv                                                           |
//+------------------------------------------------------------------+
double erfinv(const double y)
  {
   if(y<-1.0 || y>1.0)
     {
      Print("erfinv(y) argument out of range");
      return(0);
     }
   double y0=0.7;
   double x,z;
   if(MathAbs(y)==1.0) { x=-y*MathLog(DBL_EPSILON); }
   else
   if(y<-y0)
     {
      z = MathSqrt(-MathLog((1.0+y)/2.0));
      x = -(((coef_c[3]*z+coef_c[2])*z+coef_c[1])*z+coef_c[0])/((coef_d[1]*z+coef_d[0])*z+1.0);
     }
   else
     {
      if(y<y0)
        {
         z = y*y;
         x = y*(((coef_a[3]*z+coef_a[2])*z+coef_a[1])*z+coef_a[0])/((((coef_b[3]*z+coef_b[3])*z+coef_b[1])*z+coef_b[0])*z+1.0);
        }
      else
        {
         z = MathSqrt(-MathLog((1.0-y)/2.0));
         x = (((coef_c[3]*z+coef_c[2])*z+coef_c[1])*z+coef_c[0])/((coef_d[1]*z+coef_d[0])*z+1.0);
        };
      x = x - (erf(x) - y) / (2.0/MathSqrt(M_PI) * MathExp(-x*x));
      x = x - (erf(x) - y) / (2.0/MathSqrt(M_PI) * MathExp(-x*x));
     };
   return(x);
  }
//+------------------------------------------------------------------+
//| TestFunction                                                     |
//+------------------------------------------------------------------+
double TestFunction(double u,double rho)
  {
   double c1=MathSqrt((2-rho)/rho);
   double arg=erfinv(2*u);
   double c2=-2*(1-rho)/rho;
   return(c1*MathExp(c2*arg*arg));
  }
//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
//--- number of function points
   int n=100;
   double x[];
   double y[];
   ArrayResize(x,n);
   ArrayResize(y,n);
//--- range: from -1/2 to 1/2
   double x1=-0.5;
   double x2=0.5;
   double delta=(x2-x1)/(n-1);
//--- generate data
   for(int i=0; i<n; i++)
     {
      x[i]=x1+i*delta;
      y[i]=TestFunction(x[i],0.7);
     }
//--- save to file
   int filehandle=FileOpen("test-data.csv",FILE_WRITE|FILE_CSV|FILE_ANSI,'\r');
   if(filehandle==INVALID_HANDLE)
     {
      Alert("Error in open of file "); return;
     }
   for(int i=0; i<n; i++)
     {
      string s="";
      s+=DoubleToString(x[i],8)+";";
      s+=DoubleToString(y[i],8)+";";
      s+="\r";
      FileWriteString(filehandle,s);
     }
   FileClose(filehandle);
  }
//+------------------------------------------------------------------+
