//+------------------------------------------------------------------+
//|                                             EC_Example3_Test.mq5 |
//|                                  Hilhorst-Schehr problem example |
//|                        Copyright 2012, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2012, MetaQuotes Software Corp."
#property link      "http://www.mql5.com"
#property version   "1.00"
#include <CMatrix.mqh>
//+------------------------------------------------------------------+
//| CECCalculator                                                    |
//+------------------------------------------------------------------+
class CECCalculator
  {
protected:
   int               m_size;
   //--- x[i]
   double            m_x[];
   //--- y[i]
   double            m_y[];
   //--- matrix
   CMatrix           m_matrix;
   //--- Y[i]
   double            m_ec_y[];
   //--- eigencoordinates X1[i],X2[i],X3[i],X4[i]
   double            m_ec_x1[];
   double            m_ec_x2[];
   double            m_ec_x3[];
   double            m_ec_x4[];
   //--- coefficients C1,C2,C3,C4
   double            m_ec_coefs[];
   //--- plot function f1=Y-C2*X2-C3*X3-C4*X4
   double            m_f1[];
   //--- plot function f2=Y-C1*X1-C3*X3-C4*X4
   double            m_f2[];
   //--- plot function f3=Y-C1*X1-C2*X2-C4*X4
   double            m_f3[];
   //--- plot function f4=Y-C1*X1-C2*X2-C3*X3
   double            m_f4[];
private:
   //--- function
   double            P2(double x,double B,double a,double theta,double x0);
   //--- integration
   double            Integrate(double &x[],double &y[],int ind);
   //--- calculation of Y
   void              CalcY(double &y[]);
   //--- calculation of X1
   void              CalcX1(double &x1[]);
   //--- calculation of X2
   void              CalcX2(double &x2[]);
   //--- calculation of X3
   void              CalcX3(double &x3[]);
   //--- calculation of X4
   void              CalcX4(double &x4[]);
   //--- calculation of correlator
   double            Correlator(int ind1,int ind2);
public:
   //--- generates a test function x[i],y[i]
   void              GenerateData(int size,double x1,double x2,double B,double a,double theta,double x0);
   //--- loads data from file
   bool              LoadData(string filename);
   //--- saves data to file
   bool              SaveData(string filename);
   //--- saves the results
   void              SaveResults(string filename);
   //--- calculates eigencoordinates
   void              CalcEigenCoordinates();
   //--- calculates the coefficients of linear expansion
   void              CalcEigenCoefficients();
   //--- calculates parameters
   void              CalculateParameters();
   //--- calculates plot functions Y1, Y2, Y3, Y4
   void              CalculatePlotFunctions();
  };
//+------------------------------------------------------------------+
//| P1(x)                                                            |
//+------------------------------------------------------------------+
double CECCalculator::P2(double x,double B,double a,double theta,double x0)
  {
   return(B*MathPow(1+a*(x-x0)*(x-x0),-theta));
  }
//+------------------------------------------------------------------+
//| GenerateData                                                     |
//+------------------------------------------------------------------+
void CECCalculator::GenerateData(int size,double x1,double x2,double B,double a,double theta,double x0)
  {
   if(size<=0) return;
   if(x1>=x2) return;
   m_size=size;
   ArrayResize(m_x,m_size);
   ArrayResize(m_y,m_size);
   double delta=(x2-x1)/(m_size-1);
//---
   for(int i=0; i<m_size; i++)
     {
      m_x[i]=x1+i*delta;
      m_y[i]=P2(m_x[i],B,a,theta,x0);
     }
  };
//+------------------------------------------------------------------+
//| LoadData                                                         |
//+------------------------------------------------------------------+
bool CECCalculator::LoadData(string filename)
  {
   int filehandle=FileOpen(filename,FILE_CSV|FILE_READ|FILE_ANSI,'\r');
   if(filehandle==INVALID_HANDLE)
     {
      Alert("Error in open of file ",filename,", error",GetLastError());
      return(false);
     }
   m_size=0;
   while(!FileIsEnding(filehandle))
     {
      string str=FileReadString(filehandle);
      if(str!="")
        {
         string astr[];
         StringSplit(str,';',astr);
         if(ArraySize(astr)==2)
           {
            ArrayResize(m_x,m_size+1);
            ArrayResize(m_y,m_size+1);
            m_x[m_size]=StringToDouble(astr[0]);
            m_y[m_size]=StringToDouble(astr[1]);
            m_size++;
           }
         else
           {
            m_size=0;
            FileClose(filehandle);
            return(false);
           }
        }
     }
   FileClose(filehandle);
   return(true);
  }
//+------------------------------------------------------------------+
//| SaveData                                                         |
//+------------------------------------------------------------------+
bool CECCalculator::SaveData(string filename)
  {
   if(m_size==0) return(false);
   if(ArraySize(m_x)!=ArraySize(m_y)) return(false);
   if(ArraySize(m_x)==0) return(false);
   int filehandle=FileOpen(filename,FILE_WRITE|FILE_CSV|FILE_ANSI);
   if(filehandle==INVALID_HANDLE)
     {
      Alert("Error in open of file ",filename,", error",GetLastError());
      return(false);
     }
   for(int i=0; i<ArraySize(m_x); i++)
     {
      string s=DoubleToString(m_x[i],8)+";";
      s+=DoubleToString(m_y[i],8)+";";
      s+="\r";
      FileWriteString(filehandle,s);
     }
   FileClose(filehandle);
   return(true);
  }
//+------------------------------------------------------------------+
//| Integrate                                                        |
//+------------------------------------------------------------------+
double CECCalculator::Integrate(double &x[],double &y[],int ind)
  {
   double sum=0;
   for(int i=0; i<ind-1; i++) sum+=(x[i+1]-x[i])*(y[i+1]+y[i])*0.5;
   return(sum);
  }
//+------------------------------------------------------------------+
//| CalcY                                                            |
//+------------------------------------------------------------------+
void CECCalculator::CalcY(double &y[])
  {
   if(m_size==0) return;
   ArrayResize(y,m_size);
   for(int i=0; i<m_size; i++) y[i]=m_y[i]-m_y[0];
  };
//+------------------------------------------------------------------+
//| CalcX1                                                           |
//+------------------------------------------------------------------+
void CECCalculator::CalcX1(double &x1[])
  {
   if(m_size==0) return;
   ArrayResize(x1,m_size);
//--- X1=(x^2)*P2(x)+(xm)^2*P2(xm)
   for(int i=0; i<m_size; i++) x1[i]=(m_x[i]*m_x[i])*m_y[i]+(m_x[0]*m_x[0])*m_y[0];
  }
//+------------------------------------------------------------------+
//| CalcX2                                                           |
//+------------------------------------------------------------------+
void CECCalculator::CalcX2(double &x2[])
  {
   if(m_size==0) return;
   ArrayResize(x2,m_size);
//--- X2=(x)*P2(x)-(xm)*P2(xm)
   for(int i=0; i<m_size; i++) x2[i]=m_x[i]*m_y[i]-m_x[0]*m_y[0];
  }
//+------------------------------------------------------------------+
//| CalcX3                                                           |
//+------------------------------------------------------------------+
void CECCalculator::CalcX3(double &x3[])
  {
   if(m_size==0) return;
   double tmp[];
   ArrayResize(tmp,m_size);
   for(int i=0; i<m_size; i++) tmp[i]=m_x[i]*m_y[i];
//--- X3=Integrate(X*P2(x)) 
   ArrayResize(x3,m_size);
   for(int i=0; i<m_size; i++) x3[i]=Integrate(m_x,tmp,i);
  }
//+------------------------------------------------------------------+
//| CalcX4                                                           |
//+------------------------------------------------------------------+
void CECCalculator::CalcX4(double &x4[])
  {
   if(m_size==0) return;
//--- X4=Integrate(P2(x))
   ArrayResize(x4,m_size);
   for(int i=0; i<m_size; i++) x4[i]=Integrate(m_x,m_y,i);
  }
//+------------------------------------------------------------------+
//| CalcEigenCoordinates                                             |
//+------------------------------------------------------------------+
void CECCalculator::CalcEigenCoordinates()
  {
   CalcY(m_ec_y);
   CalcX1(m_ec_x1);
   CalcX2(m_ec_x2);
   CalcX3(m_ec_x3);
   CalcX4(m_ec_x4);
  }
//+------------------------------------------------------------------+
//| Correlator                                                       |
//+------------------------------------------------------------------+
double CECCalculator::Correlator(int ind1,int ind2)
  {
   if(m_size==0) return(0);
   if(ind1<=0 || ind1>5) return(0);
   if(ind2<=0 || ind2>5) return(0);
//---
   double arr1[];
   double arr2[];
   ArrayResize(arr1,m_size);
   ArrayResize(arr2,m_size);
//---
   switch(ind1)
     {
      case 1: ArrayCopy(arr1,m_ec_x1,0,0,WHOLE_ARRAY); break;
      case 2: ArrayCopy(arr1,m_ec_x2,0,0,WHOLE_ARRAY); break;
      case 3: ArrayCopy(arr1,m_ec_x3,0,0,WHOLE_ARRAY); break;
      case 4: ArrayCopy(arr1,m_ec_x4,0,0,WHOLE_ARRAY); break;
      case 5: ArrayCopy(arr1,m_ec_y,0,0,WHOLE_ARRAY); break;
     }
   switch(ind2)
     {
      case 1: ArrayCopy(arr2,m_ec_x1,0,0,WHOLE_ARRAY); break;
      case 2: ArrayCopy(arr2,m_ec_x2,0,0,WHOLE_ARRAY); break;
      case 3: ArrayCopy(arr2,m_ec_x3,0,0,WHOLE_ARRAY); break;
      case 4: ArrayCopy(arr2,m_ec_x4,0,0,WHOLE_ARRAY); break;
      case 5: ArrayCopy(arr2,m_ec_y,0,0,WHOLE_ARRAY); break;
     }
//---
   double sum=0;
   for(int i=0; i<m_size; i++) { sum+=arr1[i]*arr2[i];  }
   sum=sum/m_size;
   return(sum);
  };
//+------------------------------------------------------------------+
//| CalcEigenCoefficients                                            |
//+------------------------------------------------------------------+
void CECCalculator::CalcEigenCoefficients()
  {
//--- set matrix size 4x5
   m_matrix.SetSize(4,5);
//--- calculate correlation matrix  
   for(int i=4; i>=1; i--)
     {
      string s="";
      for(int j=1; j<=5; j++)
        {
         double corr=Correlator(i,j);
         m_matrix.Set(i,j,corr);
         s=s+" "+DoubleToString(m_matrix.Get(i,j));
        }
      Print(i," ",s);
     }
//--- solve linear equations
   m_matrix.GaussSolve(m_ec_coefs);
//--- show coefficients  
   for(int i=ArraySize(m_ec_coefs)-1; i>=0; i--) Print("C",i+1,"=",m_ec_coefs[i]);
  };
//+------------------------------------------------------------------+
//| CalculateParameters a,theta                                      |
//+------------------------------------------------------------------+
void CECCalculator::CalculateParameters()
  {
   if(ArraySize(m_ec_coefs)==0) {Print("Coefficients are not calculated!"); return;}
   double c1=m_ec_coefs[0];
   double c2=m_ec_coefs[1];
   double c3=m_ec_coefs[2];
   double c4=m_ec_coefs[3];

//--- calculate x0
//--- x0=-C2/2C1
   double x01=-c2/(2*c1);
   Print("1: x0=",x01);

//--- x0=-C4/C3
   double x02=-c4/c3;
   Print("2: x0=",x02);

//--- a=-c1/(1+c1*x0^2)
   double a1=-c1/(1+c1*x01*x01);
   Print("1: a=",a1);

//--- a=c2/(x0*(2-c2*x0))
   double a2=c2/(x02*(2-c2*x02));
   Print("2: a=",a2);

//--- theta=1-(1+a*x0^2)*c3/2*a
   double theta1=1-(1+a1*x01*x01)*c3/(2*a1);
   Print("1: theta=",theta1);

//--- theta=1+(1+a*x0^2)*c4/2*a*x0
   double theta2=1+(1+a2*x02*x02)*c4/(2*a2*x02);
   Print("2: theta=",theta2);
  };
//+------------------------------------------------------------------+
//| CalculatePlotFunctions                                           |
//| f1=Y-C2*X2-C3*X3-C4*X4                                           |
//| f2=Y-C1*X1-C3*X3-C4*X4                                           |
//| f3=Y-C1*X1-C2*X2-C4*X4                                           |
//| f4=Y-C1*X1-C2*X2-C3*X3                                           |
//+------------------------------------------------------------------+
void CECCalculator::CalculatePlotFunctions()
  {
   if(ArraySize(m_ec_coefs)==0) {Print("Coefficients are not calculated!"); return;}
//---
   ArrayResize(m_f1,m_size);
   ArrayResize(m_f2,m_size);
   ArrayResize(m_f3,m_size);
   ArrayResize(m_f4,m_size);
//---
   for(int i=0; i<m_size; i++)
     {
      //--- plot function f1=Y-C2*X2-C3*X3-C4*X4
      m_f1[i]=m_ec_y[i]-m_ec_coefs[1]*m_ec_x2[i]-m_ec_coefs[2]*m_ec_x3[i]-m_ec_coefs[3]*m_ec_x4[i];
      //--- plot function f2=Y-C1*X1-C3*X3-C4*X4
      m_f2[i]=m_ec_y[i]-m_ec_coefs[0]*m_ec_x1[i]-m_ec_coefs[2]*m_ec_x3[i]-m_ec_coefs[3]*m_ec_x4[i];
      //--- plot function f3=Y-C1*X1-C2*X2-C4*X4
      m_f3[i]=m_ec_y[i]-m_ec_coefs[0]*m_ec_x1[i]-m_ec_coefs[1]*m_ec_x2[i]-m_ec_coefs[3]*m_ec_x4[i];
      //--- plot function f3=Y-C1*X1-C2*X2-C3*X3
      m_f4[i]=m_ec_y[i]-m_ec_coefs[0]*m_ec_x1[i]-m_ec_coefs[1]*m_ec_x2[i]-m_ec_coefs[2]*m_ec_x3[i];
     }
  }
//+------------------------------------------------------------------+
//| SaveResults                                                      |
//+------------------------------------------------------------------+
void CECCalculator::SaveResults(string filename)
  {
   if(m_size==0) return;
   int filehandle=FileOpen(filename,FILE_WRITE|FILE_CSV|FILE_ANSI);
   if(filehandle==INVALID_HANDLE)
     {
      Alert("Error in open of file ",filename," for writing, error",GetLastError());
      return;
     }
   for(int i=0; i<m_size; i++)
     {
      string s=DoubleToString(m_x[i],8)+";";
      s+=DoubleToString(m_y[i],8)+";";
      s+=DoubleToString(m_ec_y[i],8)+";";
      s+=DoubleToString(m_ec_x1[i],8)+";";
      s+=DoubleToString(m_f1[i],8)+";";
      s+=DoubleToString(m_ec_x2[i],8)+";";
      s+=DoubleToString(m_f2[i],8)+";";
      s+=DoubleToString(m_ec_x3[i],8)+";";
      s+=DoubleToString(m_f3[i],8)+";";
      s+=DoubleToString(m_ec_x4[i],8)+";";
      s+=DoubleToString(m_f4[i],8)+";";
      s+="\r";
      FileWriteString(filehandle,s);
     }
   FileClose(filehandle);
  }
//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
   CECCalculator ec;
//ec.GenerateData(100,0.25,15.25,1,0.5,2,1);
//   ec.SaveData("test-data2.csv");
////--- load data   
   ec.LoadData("test-data.csv");
//--- calculate eigencoordinates
   ec.CalcEigenCoordinates();
//--- calculate coefficients
   ec.CalcEigenCoefficients();
//--- calculate parameters
   ec.CalculateParameters();
//--- calculate plot functions
   ec.CalculatePlotFunctions();
//--- save results to file
   ec.SaveResults("ex3-test-results.csv");
  }
//+------------------------------------------------------------------+
