//+------------------------------------------------------------------+
//|                                                      FibVWAP.mq5 |
//|                               Copyright 2025, Christian Benjamin |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, Christian Benjamin."
#property link      "https://www.mql5.com/en/users/lynnchris"
#property version   "1.0"
#property strict

//--- Input parameters
input int               InpNumBars         = 200;           // Number of bars to analyze for Fibonacci retracement
input ENUM_TIMEFRAMES   InpTimeFrame       = PERIOD_H1;     // Timeframe to calculate indicators
input int               InpTimerInterval   = 60;            // Timer interval in seconds for periodic updates
input string            InpPythonURL       = "http://127.0.0.1:5110/getSignal"; // Python endpoint for signal processing
input int               InpHTTPTimeout     = 5000;          // HTTP timeout in milliseconds

//--- Global variables
double g_FibLevels[7];         // Fibonacci retracement ratios array
double g_SwingHigh = 0.0;
double g_SwingLow  = 0.0;
double g_VWAP      = 0.0;
string g_LastSignal = "";

//--- Additional global variables for logging and tracking
datetime g_LastUpdateTime = 0;

//+------------------------------------------------------------------+
//| Helper: Convert bool to string                                   |
//+------------------------------------------------------------------+
string BoolToString(bool val)
  {
   return(val ? "true" : "false");
  }

//+------------------------------------------------------------------+
//| Helper: Convert ushort (character code) to string                |
//+------------------------------------------------------------------+
string CharToStr(ushort ch)
  {
   return(StringFormat("%c", ch));
  }

//+------------------------------------------------------------------+
//| Helper: Convert a string to lower case                           |
//| This custom function avoids the implicit conversion warning.      |
//+------------------------------------------------------------------+
string MyStringToLower(string s)
  {
   string res = "";
   int len = StringLen(s);
   for(int i = 0; i < len; i++)
     {
      ushort ch = s[i];
      // Check if character is uppercase A-Z.
      if(ch >= 'A' && ch <= 'Z')
         ch = ch + 32;
      res += CharToStr(ch);
     }
   return res;
  }

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
   Print("FibVWAP No-Trade EA initializing...");
   if(!EventSetTimer(InpTimerInterval))
     {
      Print("Error: Unable to set timer.");
      return(INIT_FAILED);
     }
   InitializeFibonacciArray();
   g_LastUpdateTime = TimeCurrent();
   Print("FibVWAP No-Trade EA successfully initialized.");
   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   EventKillTimer();
   Print("FibVWAP No-Trade EA deinitialized, reason code: " + IntegerToString(reason));
  }

//+------------------------------------------------------------------+
//| Expert timer function: periodic update                           |
//+------------------------------------------------------------------+
void OnTimer()
  {
   UpdateIndicators();

   string payload = BuildJSONPayload();
   Print("Payload sent to Python: " + payload);

   string signal = SendDataToPython(payload);
   if(signal != "")
     {
      if(signal != g_LastSignal)
        {
         g_LastSignal = signal;
         Print("New signal received: " + signal);
         Comment("ML Signal: " + signal);
         // Draw an arrow on the chart for the new signal.
         DrawSignalArrow(signal);
        }
      else
        {
         Print("Signal unchanged: " + signal);
        }
     }
   else
     {
      Print("Warning: No valid signal received.");
     }

   UpdateChartObjects();
  }

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
   MainProcessingLoop();
  }

//+------------------------------------------------------------------+
//| Update indicators: Fibonacci and VWAP                            |
//+------------------------------------------------------------------+
void UpdateIndicators()
  {
   CalculateFibonacciLevels(InpTimeFrame, InpNumBars, g_SwingHigh, g_SwingLow, g_FibLevels);
   g_VWAP = CalculateVWAP(InpTimeFrame, InpNumBars);
   Print("Updated Indicators: SwingHigh=" + DoubleToString(g_SwingHigh, Digits()) +
         ", SwingLow=" + DoubleToString(g_SwingLow, Digits()) +
         ", VWAP=" + DoubleToString(g_VWAP, Digits()));
  }

//+------------------------------------------------------------------+
//| Initialize Fibonacci levels array                                |
//+------------------------------------------------------------------+
void InitializeFibonacciArray()
  {
   g_FibLevels[0] = 0.000;
   g_FibLevels[1] = 0.236;
   g_FibLevels[2] = 0.382;
   g_FibLevels[3] = 0.500;
   g_FibLevels[4] = 0.618;
   g_FibLevels[5] = 0.786;
   g_FibLevels[6] = 1.000;
  }

//+------------------------------------------------------------------+
//| Calculate Fibonacci retracement levels                           |
//| Note: The array parameter 'levels' is passed by reference         |
//+------------------------------------------------------------------+
void CalculateFibonacciLevels(ENUM_TIMEFRAMES timeframe, int bars, double &swingHigh, double &swingLow, double &levels[])
  {
   int available = Bars(Symbol(), timeframe);
   if(available < bars)
     {
      Print("Not enough bars available for Fibonacci calculation. Requested: " +
            IntegerToString(bars) + " Available: " + IntegerToString(available));
      return;
     }
   swingHigh = -DBL_MAX;
   swingLow = DBL_MAX;
   for(int i = 0; i < bars; i++)
     {
      double highVal = iHigh(Symbol(), timeframe, i);
      double lowVal = iLow(Symbol(), timeframe, i);
      if(highVal > swingHigh)
         swingHigh = highVal;
      if(lowVal < swingLow)
         swingLow = lowVal;
     }
   if(swingHigh <= swingLow)
     {
      Print("Error: Swing High is not greater than Swing Low.");
      return;
     }
   double range = swingHigh - swingLow;
   for(int j = 0; j < 7; j++)
     {
      double levelPrice = swingLow + range * levels[j];
      Print("Fib Level " + IntegerToString(j) + " (" + DoubleToString(levels[j]*100, 0) +
            "%): " + DoubleToString(levelPrice, Digits()));
     }
  }

//+------------------------------------------------------------------+
//| Calculate VWAP                                                   |
//+------------------------------------------------------------------+
double CalculateVWAP(ENUM_TIMEFRAMES timeframe, int bars)
  {
   int available = Bars(Symbol(), timeframe);
   if(available < bars)
     {
      Print("Not enough bars available for VWAP calculation. Requested: " +
            IntegerToString(bars) + " Available: " + IntegerToString(available));
      return(0.0);
     }
   double cumTPV = 0.0;
   double cumVolume = 0.0;
   for(int i = 0; i < bars; i++)
     {
      double highVal = iHigh(Symbol(), timeframe, i);
      double lowVal = iLow(Symbol(), timeframe, i);
      double closeVal = iClose(Symbol(), timeframe, i);
      // Explicitly cast iVolume (type long) to double.
      double volume = (double)iVolume(Symbol(), timeframe, i);
      double typicalPrice = (highVal + lowVal + closeVal) / 3.0;
      cumTPV += typicalPrice * volume;
      cumVolume += volume;
     }
   if(cumVolume == 0)
     {
      Print("Error: Cumulative volume is zero for VWAP calculation.");
      return(0.0);
     }
   return(cumTPV / cumVolume);
  }

//+------------------------------------------------------------------+
//| Build JSON payload for Python signal processing                  |
//+------------------------------------------------------------------+
string BuildJSONPayload()
  {
   string jsonPayload = "{";
   jsonPayload += "\"symbol\":\"" + Symbol() + "\",";
   jsonPayload += "\"timeframe\":\"" + EnumToString(InpTimeFrame) + "\",";
   jsonPayload += "\"swingHigh\":" + DoubleToString(g_SwingHigh, Digits()) + ",";
   jsonPayload += "\"swingLow\":" + DoubleToString(g_SwingLow, Digits()) + ",";
   jsonPayload += "\"vwap\":" + DoubleToString(g_VWAP, Digits()) + ",";

// Send raw Fibonacci ratios instead of computed prices.
   jsonPayload += "\"fibLevels\":[";
   for(int i = 0; i < 7; i++)
     {
      jsonPayload += DoubleToString(g_FibLevels[i], 3);
      if(i < 6)
         jsonPayload += ",";
     }
   jsonPayload += "],";

// Include priceData: last InpNumBars closing prices.
   jsonPayload += "\"priceData\":[";
   int numBars = InpNumBars;
   int available = Bars(Symbol(), InpTimeFrame);
   if(numBars > available)
      numBars = available;
   for(int i = 0; i < numBars; i++)
     {
      double closePrice = iClose(Symbol(), InpTimeFrame, i);
      jsonPayload += DoubleToString(closePrice, Digits());
      if(i < numBars - 1)
         jsonPayload += ",";
     }
   jsonPayload += "]";

   jsonPayload += "}";
   return(jsonPayload);
  }

//+------------------------------------------------------------------+
//| Send data to Python and get a signal                             |
//+------------------------------------------------------------------+
string SendDataToPython(string payload)
  {
   string headers = "Content-Type: application/json\r\n";
   char postData[];
   StringToCharArray(payload, postData);
   char result[];
   string resultHeaders;
   int resCode = WebRequest("POST", InpPythonURL, headers, InpHTTPTimeout, postData, result, resultHeaders);
   if(resCode == 200)
     {
      string response = CharArrayToString(result);
      Print("HTTP Response: " + response);
      string signal = ParseSignalFromJSON(response);
      return(signal);
     }
   else
     {
      Print("Error: WebRequest returned code " + IntegerToString(resCode) +
            ". Headers: " + resultHeaders);
      return("");
     }
  }

//+------------------------------------------------------------------+
//| Convert char array to string                                     |
//+------------------------------------------------------------------+
string CharArrayToString(const char &arr[])
  {
   string ret = "";
   for(int i = 0; i < ArraySize(arr); i++)
      ret += CharToStr(arr[i]);
   return(ret);
  }

//+------------------------------------------------------------------+
//| Parse the signal from JSON response                              |
//+------------------------------------------------------------------+
string ParseSignalFromJSON(string json)
  {
   int pos = StringFind(json, "\"signal\"");
   if(pos < 0)
     {
      Print("ParseSignalFromJSON: 'signal' key not found.");
      return("");
     }
   pos = StringFind(json, ":", pos);
   if(pos < 0)
     {
      Print("ParseSignalFromJSON: ':' not found after 'signal'.");
      return("");
     }
   int start = StringFind(json, "\"", pos);
   if(start < 0)
     {
      Print("ParseSignalFromJSON: Starting quote not found.");
      return("");
     }
   start++;
   int end = StringFind(json, "\"", start);
   if(end < 0)
     {
      Print("ParseSignalFromJSON: Ending quote not found.");
      return("");
     }
   return(StringSubstr(json, start, end - start));
  }

//+------------------------------------------------------------------+
//| Draw signal arrow on the chart                                   |
//+------------------------------------------------------------------+
void DrawSignalArrow(string signal)
  {
   int arrowCode = 0;
   color arrowColor = clrWhite;
// Use custom lower-case conversion to avoid warnings.
   string lowerSignal = MyStringToLower(signal);
   if(lowerSignal == "buy")
     {
      arrowCode = 233;  // Wingdings code for upward arrow
      arrowColor = clrLime;
     }
   else
      if(lowerSignal == "sell")
        {
         arrowCode = 234;  // Wingdings code for downward arrow
         arrowColor = clrRed;
        }
      else
        {
         return;
        }

// Get the time and price of the current bar.
   datetime timeStamp = iTime(Symbol(), PERIOD_CURRENT, 0);
   double price = iClose(Symbol(), PERIOD_CURRENT, 0);

// Create a unique object name using the current time.
   string arrowName = "SignalArrow_" + IntegerToString((int)TimeCurrent());

   if(!ObjectCreate(0, arrowName, OBJ_ARROW, 0, timeStamp, price))
     {
      Print("Error creating arrow object: " + arrowName);
      return;
     }

   ObjectSetInteger(0, arrowName, OBJPROP_COLOR, arrowColor);
   ObjectSetInteger(0, arrowName, OBJPROP_WIDTH, 2);
   ObjectSetInteger(0, arrowName, OBJPROP_ARROWCODE, arrowCode);

// Adjust arrow position based on signal.
   double pointSize = SymbolInfoDouble(Symbol(), SYMBOL_POINT); // Get the point size
   if(lowerSignal == "buy")
     {
      // Place arrow slightly above the close price (5 ticks above)
      ObjectSetDouble(0, arrowName, OBJPROP_PRICE, price + pointSize * 5);
     }
   else
      if(lowerSignal == "sell")
        {
         // Place arrow slightly below the close price (5 ticks below)
         ObjectSetDouble(0, arrowName, OBJPROP_PRICE, price - pointSize * 5);
        }

   Print("Signal arrow drawn: " + arrowName + " at price " + DoubleToString(price, Digits()));
  }

//+------------------------------------------------------------------+
//| Draw objects on the chart for visual indicator levels            |
//+------------------------------------------------------------------+
void DrawChartObjects()
  {
   string objPrefix = "FibVWAP_";
// Remove previous objects with the given prefix.
   ObjectsDeleteAll(0, objPrefix);
   double range = g_SwingHigh - g_SwingLow;
   for(int i = 0; i < 7; i++)
     {
      double levelPrice = g_SwingLow + range * g_FibLevels[i];
      string name = objPrefix + "FibLevel_" + IntegerToString(i);
      if(!ObjectCreate(0, name, OBJ_HLINE, 0, 0, levelPrice))
         Print("Error creating object: " + name);
      else
        {
         ObjectSetInteger(0, name, OBJPROP_COLOR, clrDeepSkyBlue);
         ObjectSetInteger(0, name, OBJPROP_STYLE, STYLE_DOT);
         ObjectSetString(0, name, OBJPROP_TEXT, "Fib " + DoubleToString(g_FibLevels[i]*100, 0) + "%");
        }
     }
   string vwapName = objPrefix + "VWAP";
   if(!ObjectCreate(0, vwapName, OBJ_HLINE, 0, 0, g_VWAP))
      Print("Error creating VWAP object.");
   else
     {
      ObjectSetInteger(0, vwapName, OBJPROP_COLOR, clrYellow);
      ObjectSetInteger(0, vwapName, OBJPROP_STYLE, STYLE_SOLID);
      ObjectSetString(0, vwapName, OBJPROP_TEXT, "VWAP");
     }
  }

//+------------------------------------------------------------------+
//| Periodically update chart objects                                |
//+------------------------------------------------------------------+
void UpdateChartObjects()
  {
   DrawChartObjects();
  }

//+------------------------------------------------------------------+
//| Extended processing: additional updates and chart redraw         |
//+------------------------------------------------------------------+
void ExtendedProcessing()
  {
   Print("Extended processing executed.");
   UpdateChartObjects();
  }

//+------------------------------------------------------------------+
//| Main processing loop: can be called from OnTick                   |
//+------------------------------------------------------------------+
void MainProcessingLoop()
  {
   datetime currentTime = TimeCurrent();
   if(currentTime - g_LastUpdateTime >= InpTimerInterval)
     {
      UpdateChartObjects();
      g_LastUpdateTime = currentTime;
     }
  }

//+------------------------------------------------------------------+
//| Custom logging function for detailed debug information           |
//+------------------------------------------------------------------+
void LogDebugInfo(string info)
  {
   string logMessage = TimeToString(TimeCurrent(), TIME_DATE|TIME_SECONDS) + " | " + info;
   Print(logMessage);
  }

//+------------------------------------------------------------------+
//| Additional utility: Pause execution (for debugging)              |
//+------------------------------------------------------------------+
void PauseExecution(int seconds)
  {
   datetime endTime = TimeCurrent() + seconds;
   while(TimeCurrent() < endTime)
     {
      Sleep(100);
     }
  }

//+------------------------------------------------------------------+
//| End of EA code                                                   |
//+------------------------------------------------------------------+
