//+------------------------------------------------------------------+
//|                                 GBPUSD BB Breakout Benchmark.mq5 |
//|                                               Gamuchirai Ndawana |
//|                    https://www.mql5.com/en/users/gamuchiraindawa |
//+------------------------------------------------------------------+
#property copyright "Gamuchirai Ndawana"
#property link      "https://www.mql5.com/en/users/gamuchiraindawa"
#property version   "1.00"

//+------------------------------------------------------------------+
//| System constants                                                 |
//+------------------------------------------------------------------+
#define  BB_SHIFT   0                              // Our bollinger band should not be shifted
#define  SYMBOL     "GBPUSD"                       // The intended pair for our trading system
#define  BB_PRICE   PRICE_CLOSE                    // The price our bollinger band should work on
#define  BB_PERIOD  90                             // The period for our bollinger bands
#define  BB_SD      2.0                            // The standard deviation for our bollinger bands
#define  LOT        0.1                            // Our intended lot size
#define  TF    PERIOD_M15                          // Our intended time frame
#define  ATR_MULTIPLE 20                           // ATR Multiple
#define  ATR_PERIOD 14                             // ATR Period
#define  K_PERIOD 12                               // Stochastic K period
#define  D_PERIOD 20                               // Stochastic D period
#define  STO_SMOOTHING 12                          // Stochastic smoothing
#define  LOGISTIC_MODEL_PARAMS 5                   // Total inputs to our logistic model

//+------------------------------------------------------------------+
//| Dependencies                                                     |
//+------------------------------------------------------------------+
#include <Trade\Trade.mqh>
CTrade Trade;

//+------------------------------------------------------------------+
//| User inputs                                                      |
//+------------------------------------------------------------------+
input int fetch = 5;                            // How many historical bars of data should we fetch?
input int look_ahead = 10;                      // How far ahead into the future should we forecast?

//+------------------------------------------------------------------+
//| Global variables                                                 |
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Technical indicators                                             |
//+------------------------------------------------------------------+
int    bb_handler,atr_handler,stoch_handler;
double bb_u[],bb_m[],bb_l[],atr[],stoch[];
double logistic_prediction;
double learning_rate = 5E-3;

vector open_price  = vector::Zeros(fetch);
vector open_price_old = vector::Zeros(fetch);
vector close_price = vector::Zeros(fetch);
vector close_price_old = vector::Zeros(fetch);
vector high_price  = vector::Zeros(fetch);
vector high_price_old = vector::Zeros(fetch);
vector low_price = vector::Zeros(fetch);
vector low_price_old = vector::Zeros(fetch);

vector target      = vector::Zeros(fetch);
vector coef        = vector::Zeros(LOGISTIC_MODEL_PARAMS);
double max_forecast = 0;
double min_forecast = 0;
double baseline_forecast = 0;

//+------------------------------------------------------------------+
//| System variables                                                 |
//+------------------------------------------------------------------+
int state,higher_state;
double o,h,l,c,bid,ask;

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//--- Setup our system
   if(!setup())
      return(INIT_FAILED);
//---
   return(INIT_SUCCEEDED);
  }

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//--- Release resources we no longer need
   release();
  }

//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
//--- Update our system variables
   update();
  }
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Custom functions                                                 |
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Setup our technical indicators and other variables               |
//+------------------------------------------------------------------+
bool setup(void)
  {
//--- Setup our system
   bb_handler = iBands(SYMBOL,TF,BB_PERIOD,BB_SHIFT,BB_SD,BB_PRICE);
   atr_handler = iATR(SYMBOL,TF,ATR_PERIOD);
   stoch_handler = iStochastic(SYMBOL,TF,K_PERIOD,D_PERIOD,STO_SMOOTHING,MODE_EMA,STO_LOWHIGH);
   state = 0;
   higher_state = 0;
   setup_logistic_model();

//--- Validate our system has been setup correctly
   if((bb_handler != INVALID_HANDLE) && (Symbol() == SYMBOL))
      return(true);

//--- Something went wrong!
   return(false);
  }

//+------------------------------------------------------------------+
//| Setup our logistic regression model                              |
//+------------------------------------------------------------------+
void setup_logistic_model(void)
  {
   open_price.CopyRates(SYMBOL,TF,COPY_RATES_OPEN,(fetch + look_ahead),fetch);
   open_price_old.CopyRates(SYMBOL,TF,COPY_RATES_OPEN,(fetch + (look_ahead * 2)),fetch);

   high_price.CopyRates(SYMBOL,TF,COPY_RATES_HIGH,(fetch + look_ahead),fetch);
   high_price_old.CopyRates(SYMBOL,TF,COPY_RATES_HIGH,(fetch + (look_ahead * 2)),fetch);

   low_price.CopyRates(SYMBOL,TF,COPY_RATES_LOW,(fetch + look_ahead),fetch);
   low_price_old.CopyRates(SYMBOL,TF,COPY_RATES_LOW,(fetch + (look_ahead * 2)),fetch);

   close_price.CopyRates(SYMBOL,TF,COPY_RATES_CLOSE,(fetch + look_ahead),fetch);
   close_price_old.CopyRates(SYMBOL,TF,COPY_RATES_CLOSE,(fetch + (look_ahead * 2)),fetch);

   open_price = open_price - open_price_old;
   high_price = high_price - high_price_old;
   low_price = low_price - low_price_old;
   close_price = close_price - close_price_old;

   CopyBuffer(atr_handler,0,0,fetch,atr);

   for(int i = (fetch + look_ahead); i > look_ahead; i--)
     {
      if(iClose(SYMBOL,TF,i) > iClose(SYMBOL,TF,i - look_ahead))
         target[i-look_ahead-1] = 0;
      if(iClose(SYMBOL,TF,i) < iClose(SYMBOL,TF,i - look_ahead))
         target[i-look_ahead-1] = 1;
     }


//Fitting our coefficients
   coef[0] = 0;
   coef[1] = 0;
   coef[2] = 0;
   coef[3] = 0;
   coef[4] = 0;

   for(int i =0; i < fetch; i++)
     {
      double prediction = 1 / (1 + MathExp(-(coef[0] + (coef[1] * open_price[i]) + (coef[2] * high_price[i]) + (coef[3] * low_price[i]) + (coef[4] * close_price[i]))));
      coef[0] = coef[0] + (learning_rate * (target[i] - prediction)) * prediction * (1 - prediction) * 1.0;
      coef[1] = coef[1] + (learning_rate * (target[i] - prediction)) * prediction * (1 - prediction) * open_price[i];
      coef[2] = coef[2] + (learning_rate * (target[i] - prediction)) * prediction * (1 - prediction) * high_price[i];
      coef[3] = coef[3] + (learning_rate * (target[i] - prediction)) * prediction * (1 - prediction) * low_price[i];
      coef[4] = coef[4] + (learning_rate * (target[i] - prediction)) * prediction * (1 - prediction) * close_price[i];
     }

   for(int i =0; i < fetch; i++)
     {
      double prediction = 1 / (1 + MathExp(-(coef[0] + (coef[1] * open_price[i]) + (coef[2] * high_price[i]) + (coef[3] * low_price[i]) + (coef[4] * close_price[i]))));
      if(i == 0)
        {
         max_forecast = prediction;
         min_forecast = prediction;
        }
      max_forecast = (prediction > max_forecast) ? (prediction) : max_forecast;
      min_forecast = (prediction < min_forecast) ? (prediction) : min_forecast;
     }

   baseline_forecast = ((max_forecast + min_forecast) / 2);

   Print(coef);
   Print("Baseline: ",baseline_forecast);
  }

//+------------------------------------------------------------------+
//| Release the resources we no longer need                          |
//+------------------------------------------------------------------+
void release(void)
  {
//--- Free up system resources for our end user
   IndicatorRelease(bb_handler);
   IndicatorRelease(atr_handler);
   IndicatorRelease(stoch_handler);
  }

//+------------------------------------------------------------------+
//| Update our system variables                                      |
//+------------------------------------------------------------------+
void update(void)
  {
   static datetime timestamp;
   datetime current_time = iTime(SYMBOL,TF,0);

   if(timestamp != current_time)
     {
      //--- Update system variables
      timestamp = current_time;
      //--- Update our system
      CopyBuffer(bb_handler,0,1,1,bb_m);
      CopyBuffer(bb_handler,1,1,1,bb_u);
      CopyBuffer(bb_handler,2,1,1,bb_l);
      CopyBuffer(atr_handler,0,1,1,atr);
      CopyBuffer(stoch_handler,0,1,1,stoch);
      Comment("U: ",bb_u[0],"\nM: ",bb_m[0],"\nL: ",bb_l[0]);

      //--- Market prices
      o = iOpen(SYMBOL,TF,0);
      c = iClose(SYMBOL,TF,0);
      h = iHigh(SYMBOL,TF,0);
      l = iLow(SYMBOL,TF,0);
      bid = SymbolInfoDouble(SYMBOL,SYMBOL_BID);
      ask = SymbolInfoDouble(SYMBOL,SYMBOL_ASK);

      //--- Should we reset our system state?
      if(PositionsTotal() == 0)
        {
         state = 0;
         find_setup();
        }
      //--- Manage our open positions
      if(PositionsTotal() > 0)
         manage_setup();
     }
  }

//+------------------------------------------------------------------+
//| Find an oppurtunity to trade                                     |
//+------------------------------------------------------------------+
void find_setup(void)
  {
   double open_input = iOpen(SYMBOL,TF,0)   - iOpen(SYMBOL,TF,look_ahead);
   double close_input = iClose(SYMBOL,TF,0) - iClose(SYMBOL,TF,look_ahead);
   double high_input = iHigh(SYMBOL,TF,0)   - iHigh(SYMBOL,TF,look_ahead);
   double low_input = iLow(SYMBOL,TF,0)     - iLow(SYMBOL,TF,look_ahead);
   double prediction = 1 / (1 + MathExp(-(coef[0] + (coef[1] * open_input) + (coef[2] * high_input) + (coef[3] * low_input) + (coef[4] * close_input))));
   Print("Odds: ",prediction - baseline_forecast);
   
//--- Check if we have breached the bollinger bands
   if((c > bb_u[0]) && (stoch[0] < 50))
     {
      
      Trade.Sell(LOT,SYMBOL,bid);
      state = -1;


      if(((prediction - baseline_forecast) > 0))
        {
         Trade.Sell((LOT * 2),SYMBOL,bid);
         Trade.Sell((LOT * 2),SYMBOL,bid);
         state = -1;
        }

      return;
     }


   if((c < bb_l[0]) && (stoch[0] > 50))
     {
      
      Trade.Buy(LOT,SYMBOL,ask);
      state = 1;
      
      if(((prediction - baseline_forecast) < 0))
        {
         Trade.Buy((LOT * 2),SYMBOL,ask);
         Trade.Buy((LOT * 2),SYMBOL,ask);
         state = 1;
        }

      return;
     }
  }

//+------------------------------------------------------------------+
//| Manage our open positions                                        |
//+------------------------------------------------------------------+
void manage_setup(void)
  {
   if(((c < bb_l[0]) && (state == -1))||((c > bb_u[0]) && (state == 1)))
      Trade.PositionClose(SYMBOL);


//--- Update the stop loss
   for(int i = PositionsTotal() -1; i >= 0; i--)
     {
      string symbol = PositionGetSymbol(i);
      if(_Symbol == symbol)
        {
         double position_size = PositionGetDouble(POSITION_VOLUME);
         double risk_factor = 1;
         if(position_size == (LOT * 2))
            risk_factor = 2;
         double atr_stop = atr[0] * ATR_MULTIPLE * risk_factor;
         ulong ticket = PositionGetInteger(POSITION_TICKET);
         double position_price = PositionGetDouble(POSITION_PRICE_OPEN);
         long type = PositionGetInteger(POSITION_TYPE);
         double current_take_profit = PositionGetDouble(POSITION_TP);
         double current_stop_loss = PositionGetDouble(POSITION_SL);
         if(type == POSITION_TYPE_BUY)
           {
            double atr_stop_loss = (bid - (atr_stop));
            double atr_take_profit = (bid + (atr_stop));

            if((current_stop_loss < atr_stop_loss) || (current_stop_loss == 0))
              {
               Trade.PositionModify(ticket,atr_stop_loss,current_take_profit);
              }
           }
         else
            if(type == POSITION_TYPE_SELL)
              {
               double atr_stop_loss = (ask + (atr_stop));
               double atr_take_profit = (ask - (atr_stop));
               if((current_stop_loss > atr_stop_loss) || (current_stop_loss == 0))
                 {
                  Trade.PositionModify(ticket,atr_stop_loss,current_take_profit);
                 }
              }
        }
     }
  }
//+------------------------------------------------------------------+
