//+------------------------------------------------------------------+
//| Bitcoin Monitoring Expert Advisor                                |
//| Copyright 2024, Clemence Benjamin                                |
//| https://www.mql5.com/en/users/billionaire2024/seller            |
//+------------------------------------------------------------------+
#property copyright "Clemence Benjamin"
#property link      "https://www.mql5.com/en/users/billionaire2024/seller"
#property version   "1.0"
#property description "Bitcoin Monitoring Expert Advisor"

//--- Input parameters
input string IndicatorName = "ema100_monitoring_indicator"; // Name of the custom indicator, be sure to put the name as it is saved.
input bool EnableTerminalAlerts = true;
input bool EnablePushNotifications = true;
input bool EnableTelegramAlerts = false;
input string TelegramBotToken = "YOUR_BOT_TOKEN";         // Replace with your bot token
input string TelegramChatID = "YOUR_CHAT_ID";             // Replace with your chat ID

//--- Indicator handles
int indicatorHandle = INVALID_HANDLE;
int emaHandle = INVALID_HANDLE;

//--- Alert function
void AlertMessage(string message) {
   if (EnableTerminalAlerts) Alert(message);
   if (EnablePushNotifications) SendNotification(message);
   if (EnableTelegramAlerts) SendTelegramMessage(message);
}

//--- Telegram Alerting
void SendTelegramMessage(string message) {
   if (EnableTelegramAlerts) {
      string url = "https://api.telegram.org/bot" + TelegramBotToken + "/sendMessage?chat_id=" + TelegramChatID + "&text=" + message;
      int timeout = 5000;
      ResetLastError();

      char postData[];
      uchar result[];
      string response;

      int res = WebRequest("GET", url, NULL, timeout, postData, result, response);

      if (res != 200) {
         Print("Telegram WebRequest failed. Error: ", GetLastError(), ", HTTP Code: ", res);
      } else {
         Print("Telegram message sent successfully: ", response);
      }
   }
}

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit() {
   Print("Bitcoin Monitoring EA started.");

   // Attach the custom indicator to the chart
   indicatorHandle = iCustom(_Symbol, _Period, IndicatorName);
   if (indicatorHandle == INVALID_HANDLE) {
      Print("Failed to attach indicator: ", IndicatorName, ". Error: ", GetLastError());
      return(INIT_FAILED);
   }

   // Attach built-in EMA 100 to the chart
   emaHandle = iMA(_Symbol, _Period, 100, 0, MODE_EMA, PRICE_CLOSE);
   if (emaHandle == INVALID_HANDLE) {
      Print("Failed to create EMA 100. Error: ", GetLastError());
      return(INIT_FAILED);
   }

   // Add EMA 100 to the terminal chart
   if (!ChartIndicatorAdd(0, 0, emaHandle)) {
      Print("Failed to add EMA 100 to the chart. Error: ", GetLastError());
   }

   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason) {
   Print("Bitcoin Monitoring EA stopped.");
   if (indicatorHandle != INVALID_HANDLE) {
      IndicatorRelease(indicatorHandle);
   }
   if (emaHandle != INVALID_HANDLE) {
      IndicatorRelease(emaHandle);
   }
}

void OnTick() {
   static datetime lastAlertTime = 0; // Prevent repeated alerts for the same signal

   if (indicatorHandle == INVALID_HANDLE || emaHandle == INVALID_HANDLE) return;

   double buffer1[], buffer2[];
   ArraySetAsSeries(buffer1, true);
   ArraySetAsSeries(buffer2, true);

   // Read data from the custom indicator
   if (CopyBuffer(indicatorHandle, 0, 0, 1, buffer1) < 0) {
      Print("Failed to copy Buffer1. Error: ", GetLastError());
      return;
   }
   if (CopyBuffer(indicatorHandle, 1, 0, 1, buffer2) < 0) {
      Print("Failed to copy Buffer2. Error: ", GetLastError());
      return;
   }

   // Check for signals in Buffer1
   if (buffer1[0] != EMPTY_VALUE && TimeCurrent() != lastAlertTime) {
      string message = "Signal detected: Look for EMA 100 bounce (Low). Symbol: " + _Symbol;
      AlertMessage(message);
      lastAlertTime = TimeCurrent();
   }

   // Check for signals in Buffer2
   if (buffer2[0] != EMPTY_VALUE && TimeCurrent() != lastAlertTime) {
      string message = "Signal detected: Look for EMA 100 bounce (High). Symbol: " + _Symbol;
      AlertMessage(message);
      lastAlertTime = TimeCurrent();
   }

   // Debugging EMA 100 value
   double emaValueArray[];
   ArraySetAsSeries(emaValueArray, true); // Ensure it's set as series
   if (CopyBuffer(emaHandle, 0, 0, 1, emaValueArray) > 0) {
      Print("EMA 100 Current Value: ", emaValueArray[0]);
   } else {
      Print("Failed to read EMA 100 buffer. Error: ", GetLastError());
   }
}
