//+------------------------------------------------------------------+
//|                                                  TrailingATR.mqh |
//|                             Copyright 2000-2023, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertTrailing.mqh>
#include <My\Cmlp-.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Trailing Stop based on ATR oscillator Patterns.            |
//| Type=Trailing                                                    |
//| Name=ATR                                                         |
//| Class=CTrailingATR                                               |
//| Page=                                                            |
//| Parameter=PeriodMA,int,14,Period of averaging                    |
//| Parameter=Pattern_0,int,50,Pattern 0                             |
//| Parameter=Pattern_1,int,50,Pattern 1                             |
//| Parameter=Pattern_2,int,50,Pattern 2                             |
//| Parameter=PatternsUsed,int,8,Patterns Used BitMap                |
//| Parameter=StopLevel,int,30,Stop Loss trailing (in ATR multiples) |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CTrailingATR.                                              |
//| Purpose: Class of trailing stops with ATR oscillator Patterns.   |
//|              Derives from class CExpertTrailing.                 |
//+------------------------------------------------------------------+
class CTrailingATR : public CExpertTrailing
{
protected:
   CiATR             m_atr;            // object-oscillator
   CiMA              m_ma;             // object-indicator
   CiRSI             m_rsi;
   CiSAR             m_sar;            // object-indicator

   int               m_pattern_0;      // model 0 "ATR-Based Stop Loss"
   int               m_pattern_1;      // model 1 "ATR Trailing Stop Signal"
   int               m_pattern_2;      // model 2 "ATR as an Exit Signal"
   //
   int               m_patterns_usage;
   
   int               m_stop_level;

   int               m_ma_period;      // the "period of averaging" parameter of the oscillator
   ENUM_APPLIED_PRICE m_ma_applied;    // the "object of averaging" parameter of the oscillator

public:
                     CTrailingATR(void);
                    ~CTrailingATR(void);

   //--- methods of setting adjustable parameters
   void              StopLevel(int stop_level)
   {  m_stop_level = stop_level;
   }
   void              PeriodMA(int value)
   {  m_ma_period = value;
   }
   void              Applied(ENUM_APPLIED_PRICE value)
   {  m_ma_applied = value;
   }
   void              Pattern_0(int value)
   {  m_pattern_0 = value;
   }
   void              Pattern_1(int value)
   {  m_pattern_1 = value;
   }
   void              Pattern_2(int value)
   {  m_pattern_2 = value;
   }
   void              PatternsUsed(int value)
   {  m_patterns_usage = value;
   }
   double            ATR(int ind)
   {                 m_atr.Refresh(-1);
      return(m_atr.Main(ind));
   }
   double            MA(int ind)
   {                 m_ma.Refresh(-1);
      return(m_ma.Main(ind));
   }
   double            SAR(int ind)
   {                 m_sar.Refresh(-1);
      return(m_sar.Main(ind));
   }

   //--- methods of initialization of protected data

   virtual bool      ValidationSettings(void);
protected:
   //--- method of initialization of the oscillator
   bool              InitATR(CIndicators *indicators);
   //---
   virtual bool      CheckTrailingStopLong(CPositionInfo *position, double &sl, double &tp);
   virtual bool      CheckTrailingStopShort(CPositionInfo *position, double &sl, double &tp);

protected:
   //--- methods to check for patterns
   void            IsPattern_0(double &Price, ENUM_POSITION_TYPE T);
   void            IsPattern_1(double &Price, ENUM_POSITION_TYPE T);
   void            IsPattern_2(double &Price, ENUM_POSITION_TYPE T);

};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CTrailingATR::CTrailingATR(void) :    m_pattern_0(50),
   m_ma_applied(PRICE_CLOSE),
   m_pattern_1(50),
   m_pattern_2(50),
   m_ma_period(14),
   m_stop_level(50)

{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE + USE_SERIES_SPREAD + USE_SERIES_TIME;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CTrailingATR::~CTrailingATR(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CTrailingATR::ValidationSettings(void)
{  if(!CExpertTrailing::ValidationSettings())
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Initialize Indicators.                                           |
//+------------------------------------------------------------------+
bool CTrailingATR::InitATR(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- add object to collection
   if(!indicators.Add(GetPointer(m_atr)))
   {  printf(__FUNCTION__ + ": error adding object");
      return(false);
   }
//--- initialize object
   if(!m_atr.Create(m_symbol.Name(), m_period, m_ma_period))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_ma.Create(m_symbol.Name(), m_period, m_ma_period, 0, MODE_SMA, m_ma_applied))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_rsi.Create(m_symbol.Name(), m_period, m_ma_period, m_ma_applied))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_atr.Create(m_symbol.Name(), m_period, m_ma_period))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Checking trailing stop and/or profit for long position.          |
//+------------------------------------------------------------------+
bool CTrailingATR::CheckTrailingStopLong(CPositionInfo *position, double &sl, double &tp)
{
//--- check
   if(position == NULL)
      return(false);
//---
   double result  = 0.0, results = 0.0;
   sl = EMPTY_VALUE;
   tp = EMPTY_VALUE;
//---
   double delta = ((m_symbol.FreezeLevel()+m_symbol.StopsLevel())*m_symbol.Point())+m_symbol.SpreadFloat();
   double pos_sl = position.StopLoss();
   double base  = (pos_sl == 0.0) ? position.PriceOpen() : pos_sl;
   double price = m_symbol.Bid();
//--- if the model 0 is used and "ATR-Based Stop Loss"
   if(((m_patterns_usage & 0x01) != 0))
   {  IsPattern_0(price, POSITION_TYPE_BUY);
      result += m_pattern_0 * price;
      results += m_pattern_0;
   }
//--- if the model 1 is used and "ATR MA Stop Signal"
   if(((m_patterns_usage & 0x02) != 0))
   {  IsPattern_1(price, POSITION_TYPE_BUY);
      result += m_pattern_1 * price;
      results += m_pattern_1;
   }
//--- if the model 2 is used and "ATR SAR Stop Signal"
   if(((m_patterns_usage & 0x04) != 0))
   {  IsPattern_2(price, POSITION_TYPE_BUY);
      result += m_pattern_2 * price;
      results += m_pattern_2;
   }
//---
   if(results > 0)
   {  result /= results;
   }
//---
   if(price - base > delta)
   {  sl = price - delta;
   }
//---
   return(sl != EMPTY_VALUE);
}
//+------------------------------------------------------------------+
//| Checking trailing stop and/or profit for short position.         |
//+------------------------------------------------------------------+
bool CTrailingATR::CheckTrailingStopShort(CPositionInfo *position, double &sl, double &tp)
{
//--- check
   if(position == NULL)
      return(false);
//---
   double result  = 0.0, results = 0.0;
   sl = EMPTY_VALUE;
   tp = EMPTY_VALUE;
//---
   double delta = ((m_symbol.FreezeLevel()+m_symbol.StopsLevel())*m_symbol.Point())+m_symbol.SpreadFloat();
   double pos_sl=position.StopLoss();
   double base  =(pos_sl==0.0) ? position.PriceOpen() : pos_sl;
   double price =m_symbol.Ask();
//--- if the model 0 is used and "ATR-Based Stop Loss"
   if(((m_patterns_usage & 0x01) != 0))
   {  IsPattern_0(price, POSITION_TYPE_SELL);
      result += m_pattern_0 * price;
      results += m_pattern_0;
   }
//--- if the model 1 is used and "ATR MA Stop Signal"
   if(((m_patterns_usage & 0x02) != 0))
   {  IsPattern_1(price, POSITION_TYPE_SELL);
      result += m_pattern_1 * price;
      results += m_pattern_1;
   }
//--- if the model 2 is used and "ATR SAR Stop Signal"
   if(((m_patterns_usage & 0x04) != 0))
   {  IsPattern_2(price, POSITION_TYPE_SELL);
      result += m_pattern_2 * price;
      results += m_pattern_2;
   }
//---
   if(results > 0)
   {  result /= results;
   }
//---
   if(base - price > delta)
   {  sl = price + delta;
   }
//---
   return(sl != EMPTY_VALUE);
}
//+------------------------------------------------------------------+
//| Check for Pattern 0.                                             |
//+------------------------------------------------------------------+
void CTrailingATR::IsPattern_0(double &Price, ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY)
   {  Price -= (m_stop_level * ATR(StartIndex()));
   }
   else if(T == POSITION_TYPE_SELL)
   {  Price += (m_stop_level * ATR(StartIndex()));
   }
}
//+------------------------------------------------------------------+
//| Check for Pattern 1.                                             |
//+------------------------------------------------------------------+
void CTrailingATR::IsPattern_1(double &Price, ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY)
   {  Price = (MA(StartIndex()) - (m_stop_level * ATR(StartIndex())));
   }
   else if(T == POSITION_TYPE_SELL)
   {  Price = (MA(StartIndex()) + (m_stop_level * ATR(StartIndex())));
   }
}
//+------------------------------------------------------------------+
//| Check for Pattern 2.                                             |
//+------------------------------------------------------------------+
void CTrailingATR::IsPattern_2(double &Price, ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && SAR(StartIndex()) < Low(StartIndex()))
   {  Price = (SAR(StartIndex()) - (m_stop_level * ATR(StartIndex())));
   }
   else if(T == POSITION_TYPE_SELL && SAR(StartIndex()) > High(StartIndex()))
   {  Price = (SAR(StartIndex()) + (m_stop_level * ATR(StartIndex())));
   }
}
//+------------------------------------------------------------------+
