//+------------------------------------------------------------------+
//|                                                    SignalATR.mqh |
//|                             Copyright 2000-2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals of ATR oscillator Patterns.                        |
//| Type=SignalAdvanced                                              |
//| Name=ATR                                                         |
//| ShortName=ATR                                                    |
//| Class=CSignalATR                                                 |
//| Page=signal_adx                                                  |
//| Parameter=PeriodMA,int,14,Period of averaging                    |
//| Parameter=ThresholdPoints,int,255,ATR Threshold in Points        |
//| Parameter=Pattern_0,int,50,Pattern 0                             |
//| Parameter=Pattern_1,int,50,Pattern 1                             |
//| Parameter=Pattern_2,int,50,Pattern 2                             |
//| Parameter=Pattern_3,int,50,Pattern 3                             |
//| Parameter=Pattern_4,int,50,Pattern 4                             |
//| Parameter=PatternsUsed,int,255,Patterns Used BitMap              |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalATR.                                                |
//| Purpose: Class of generator of trade signals based on            |
//|          ATR oscillator Patterns.                                |
//| Is derived from the CExpertSignal class.                         |
//+------------------------------------------------------------------+
#define  __COMPRESSION_LIMIT 65
class CSignalATR : public CExpertSignal
{
protected:
   CiATR             m_atr;            // object-oscillator
   CiMA              m_ma;             // object-indicator
   CiRSI             m_rsi;
   //--- adjusted parameters
   int               m_ma_period;      // the "period of averaging" parameter of the oscillator
   ENUM_APPLIED_PRICE m_ma_applied;    // the "object of averaging" parameter of the oscillator
   ENUM_TIMEFRAMES   m_alternate;      // the "alternate time frame" parameter of the oscillator
   //--- "weights" of market models (0-100)
   int               m_pattern_0;      // model 0 "Volatility Breakout Signal"
   int               m_pattern_1;      // model 1 "ATR Threshold for Breakout Confirmation"
   int               m_pattern_2;      // model 2 "ATR as an Exit Signal"
   int               m_pattern_3;      // model 3 "Volatility Contraction Signal"
   int               m_pattern_4;      // model 4 "ATR Channel (ATR Bands)"
   //
   int               m_threshold_points;   //
   
public:
                     CSignalATR(void);
                    ~CSignalATR(void);
   //--- methods of setting adjustable parameters
   void              PeriodMA(int value)
   {  m_ma_period = value;
   }
   void              Applied(ENUM_APPLIED_PRICE value)
   {  m_ma_applied = value;
   }
   void              AlternateFrame(ENUM_TIMEFRAMES value)
   {  m_alternate = value;
   }
   //--- methods of adjusting "weights" of market models
   void              Pattern_0(int value)
   {  m_pattern_0 = value;
   }
   void              Pattern_1(int value)
   {  m_pattern_1 = value;
   }
   void              Pattern_2(int value)
   {  m_pattern_2 = value;
   }
   void              Pattern_3(int value)
   {  m_pattern_3 = value;
   }
   void              Pattern_4(int value)
   {  m_pattern_4 = value;
   }
   void              PatternsUsed(int value)
   {  m_patterns_usage = value;
      PatternsUsage(value);
   }
   void              ThresholdPoints(int value)
   {  m_threshold_points = value;
   }
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the oscillator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   //--- method of initialization of the oscillator
   bool              InitATR(CIndicators *indicators);
   //--- methods of getting data
   double            Base(int ind)
   {                 m_atr.Refresh(-1);
      return(m_atr.Main(ind));
   }
   double            BaseAverage(int ind)
   {                 m_atr.Refresh(-1);
      vector _v;
      _v.CopyIndicatorBuffer(m_atr.Handle(), 0, ind, m_ma_period);
      return(_v.Mean());
   }
   double            ATR(int ind)
   {                 m_atr.Refresh(-1);
      return(m_atr.Main(ind));
   }
   double            ATRAverage(int ind)
   {                 m_atr.Refresh(-1);
      vector _v;
      _v.CopyIndicatorBuffer(m_atr.Handle(), 0, ind, m_ma_period);
      return(_v.Mean());
   }
   double            Close(int ind)
   {                 m_close.Refresh(-1);
      return(m_close.GetData(ind));
   }
   double            High(int ind)
   {                 m_high.Refresh(-1);
      return(m_high.GetData(ind));
   }
   double            Low(int ind)
   {                 m_low.Refresh(-1);
      return(m_low.GetData(ind));
   }
   double            Range(int ind)
   {                 return(High(ind) - Low(ind));
   }
   double            MA(int ind)
   {                 m_ma.Refresh(-1);
      return(m_ma.Main(ind));
   }
   double            RSI(int ind)
   {                 m_rsi.Refresh(-1);
      return(m_rsi.Main(ind));
   }
   int               X()
   {  return(StartIndex());
   }
   //--- methods to check for patterns
   bool              IsPattern_0(ENUM_POSITION_TYPE T);
   bool              IsPattern_1(ENUM_POSITION_TYPE T);
   bool              IsPattern_2(ENUM_POSITION_TYPE T);
   bool              IsPattern_3(ENUM_POSITION_TYPE T);
   bool              IsPattern_4(ENUM_POSITION_TYPE T);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSignalATR::CSignalATR(void) : m_ma_period(14),
   m_ma_applied(PRICE_CLOSE),
   m_alternate(PERIOD_W1),
   m_pattern_0(50),
   m_pattern_1(50),
   m_pattern_2(50),
   m_pattern_3(50),
   m_pattern_4(50),
   m_threshold_points(250)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE;
   PatternsUsage(m_patterns_usage);
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSignalATR::~CSignalATR(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalATR::ValidationSettings(void)
{
//--- validation settings of additional filters
   if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_ma_period <= 0)
   {  printf(__FUNCTION__ + ": period MA must be greater than 0");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalATR::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- create and initialize MA oscillator
   if(!InitATR(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Initialize MA indicators.                                        |
//+------------------------------------------------------------------+
bool CSignalATR::InitATR(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- add object to collection
   if(!indicators.Add(GetPointer(m_atr)))
   {  printf(__FUNCTION__ + ": error adding object");
      return(false);
   }
//--- initialize object
   if(!m_atr.Create(m_symbol.Name(), m_period, m_ma_period))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_ma.Create(m_symbol.Name(), m_period, m_ma_period, 0, MODE_SMA, m_ma_applied))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_rsi.Create(m_symbol.Name(), m_period, m_ma_period, m_ma_applied))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_atr.Create(m_symbol.Name(), m_period, m_ma_period))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalATR::LongCondition(void)
{  int result  = 0, results = 0;
//--- if the model 0 is used and "Volatility Breakout Signal"
   if(((m_patterns_usage & 0x01) != 0) && IsPattern_0(POSITION_TYPE_BUY))
   {  result += m_pattern_0;
      results++;
   }
//--- if the model 1 is used and "ATR Threshold for Breakout Confirmation"
   if(((m_patterns_usage & 0x02) != 0) && IsPattern_1(POSITION_TYPE_BUY))
   {  result += m_pattern_1;
      results++;
   }
//--- if the model 2 is used and "ATR as an Exit Signal"
   if(((m_patterns_usage & 0x04) != 0) && IsPattern_2(POSITION_TYPE_BUY))
   {  result += m_pattern_2;
      results++;
   }
//--- if the model 3 is used and "Volatility Contraction Signal"
   if(((m_patterns_usage & 0x08) != 0) && IsPattern_3(POSITION_TYPE_BUY))
   {  result += m_pattern_3;
      results++;
   }
//--- if the model 4 is used and "ATR Channel (ATR Bands)"
   if(((m_patterns_usage & 0x10) != 0) && IsPattern_4(POSITION_TYPE_BUY))
   {  result += m_pattern_4;
      results++;
   }
//--- return the result
//if(result > 0)printf(__FUNCSIG__+" result is: %i",result);
   if(results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalATR::ShortCondition(void)
{  int result  = 0, results = 0;
//--- if the model 0 is used and "Volatility Breakout Signal"
   if(((m_patterns_usage & 0x01) != 0) && IsPattern_0(POSITION_TYPE_SELL))
   {  result += m_pattern_0;
      results++;
   }
//--- if the model 1 is used and "ATR Threshold for Breakout Confirmation"
   if(((m_patterns_usage & 0x02) != 0) && IsPattern_1(POSITION_TYPE_SELL))
   {  result += m_pattern_1;
      results++;
   }
//--- if the model 2 is used and "ATR as an Exit Signal"
   if(((m_patterns_usage & 0x04) != 0) && IsPattern_2(POSITION_TYPE_SELL))
   {  result += m_pattern_2;
      results++;
   }
//--- if the model 3 is used and "Volatility Contraction Signal"
   if(((m_patterns_usage & 0x08) != 0) && IsPattern_3(POSITION_TYPE_SELL))
   {  result += m_pattern_3;
      results++;
   }
//--- if the model 4 is used and "ATR Channel (ATR Bands)"
   if(((m_patterns_usage & 0x10) != 0) && IsPattern_4(POSITION_TYPE_SELL))
   {  result += m_pattern_4;
      results++;
   }
//--- return the result
//if(result > 0)printf(__FUNCSIG__+" result is: %i",result);
   if(results > 0)
   {  return(int(round(result / results)));
   }
   return(0);
}
//+------------------------------------------------------------------+
//| Check for Pattern 0.                                             |
//+------------------------------------------------------------------+
bool CSignalATR::IsPattern_0(ENUM_POSITION_TYPE T)
{  if(ATR(X()) > ATR(X() + 1) && ATR(X() + 1) > ATR(X() + 2) && ATR(X() + 2) > ATR(X() + 3))
   {  if(T == POSITION_TYPE_BUY && Close(X()) > Close(X() + 1) && Close(X() + 1) > Close(X() + 2) && Close(X() + 2) > Close(X() + 3))
      {  return(true);
      }
      else if(T == POSITION_TYPE_SELL && Close(X()) < Close(X() + 1) && Close(X() + 1) < Close(X() + 2) && Close(X() + 2) < Close(X() + 3))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 1.                                             |
//+------------------------------------------------------------------+
bool CSignalATR::IsPattern_1(ENUM_POSITION_TYPE T)
{  if(ATR(X()) >= m_threshold_points*m_symbol.Point())
   {  if(T == POSITION_TYPE_BUY && Close(X()) > Close(X() + 1) && Close(X() + 1) > Close(X() + 2) && Close(X() + 2) > Close(X() + 3))
      {  return(true);
      }
      else if(T == POSITION_TYPE_SELL && Close(X()) < Close(X() + 1) && Close(X() + 1) < Close(X() + 2) && Close(X() + 2) < Close(X() + 3))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 2.                                             |
//+------------------------------------------------------------------+
bool CSignalATR::IsPattern_2(ENUM_POSITION_TYPE T)
{  if(ATR(X()) >= 2.0 * m_threshold_points*m_symbol.Point())
   {  if(T == POSITION_TYPE_SELL && Close(X()) > Close(X() + 1) && Close(X() + 1) > Close(X() + 2) && Close(X() + 2) > Close(X() + 3))
      {  return(true);
      }
      else if(T == POSITION_TYPE_BUY && Close(X()) < Close(X() + 1) && Close(X() + 1) < Close(X() + 2) && Close(X() + 2) < Close(X() + 3))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 3.                                             |
//+------------------------------------------------------------------+
bool CSignalATR::IsPattern_3(ENUM_POSITION_TYPE T)
{  if(ATR(X()) < ATR(X() + 1) && ATR(X() + 1) < ATR(X() + 2) && ATR(X() + 2) < ATR(X() + 3))
   {  if(T == POSITION_TYPE_SELL && Close(X() + 3) > Close(X() + 4) && Close(X() + 4) > Close(X() + 5) && Close(X() + 5) > Close(X() + 6))
      {  return(true);
      }
      else if(T == POSITION_TYPE_BUY && Close(X() + 3) < Close(X() + 4) && Close(X() + 4) < Close(X() + 5) && Close(X() + 5) < Close(X() + 6))
      {  return(true);
      }
   }
   return(false);
}
//+------------------------------------------------------------------+
//| Check for Pattern 4.                                             |
//+------------------------------------------------------------------+
bool CSignalATR::IsPattern_4(ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && Close(X()) > MA(X()) + 2.0 * ATR(X()))
   {  return(true);
   }
   else if(T == POSITION_TYPE_SELL && Close(X()) < MA(X()) - 2.0 * ATR(X()))
   {  return(true);
   }
   return(false);
}
//+------------------------------------------------------------------+
