//+------------------------------------------------------------------+
//|                                                     MoneyATR.mqh |
//|                             Copyright 2000-2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertMoney.mqh>
#include <Trade\DealInfo.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Trading with optimized trade volume of ATR oscillator Patterns|
//| Type=Money                                                       |
//| Name=ATR                                                         |
//| Class=CMoneyATR                                                  |
//| Page=                                                            |
//| Parameter=DecreaseFactor,double,3.0,Decrease factor              |
//| Parameter=Percent,double,10.0,Percent                            |
//| Parameter=ThresholdPoints,int,255,ATR Threshold in Points        |
//| Parameter=Pattern_0,int,50,Pattern 0                             |
//| Parameter=Pattern_1,int,50,Pattern 1                             |
//| Parameter=Pattern_2,int,50,Pattern 2                             |
//| Parameter=PatternsUsed,int,8,Patterns Used BitMap                |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CMoneyATR.                                                 |
//| Purpose: Class of money management with ATR oscillator Patterns. |
//|              Derives from class CExpertMoney.                    |
//+------------------------------------------------------------------+
class CMoneyATR : public CExpertMoney
{
protected:
   CiATR             m_atr;            // object-oscillator
   CiMA              m_ma;             // object-indicator
   CiRSI             m_rsi;
   CiSAR             m_sar;            // object-indicator

   int               m_pattern_0;      // model 0 "ATR-Based Stop Loss"
   int               m_pattern_1;      // model 1 "ATR MA Stop Signal"
   int               m_pattern_2;      // model 2 "ATR SAR Stop Signal"
   //
   int               m_patterns_usage;

   double            m_decrease_factor;
   
   int               m_stop_level;

   int               m_ma_period;      // the "period of averaging" parameter of the oscillator
   ENUM_APPLIED_PRICE m_ma_applied;    // the "object of averaging" parameter of the oscillator

public:
                     CMoneyATR(void);
                    ~CMoneyATR(void);

   //--- methods of setting adjustable parameters
   void              StopLevel(int stop_level)
   {  m_stop_level = stop_level;
   }
   void              PeriodMA(int value)
   {  m_ma_period = value;
   }
   void              Applied(ENUM_APPLIED_PRICE value)
   {  m_ma_applied = value;
   }
   void              Pattern_0(int value)
   {  m_pattern_0 = value;
   }
   void              Pattern_1(int value)
   {  m_pattern_1 = value;
   }
   void              Pattern_2(int value)
   {  m_pattern_2 = value;
   }
   void              PatternsUsed(int value)
   {  m_patterns_usage = value;
   }
   double            ATR(int ind)
   {                 m_atr.Refresh(-1);
      return(m_atr.Main(ind));
   }
   double            MA(int ind)
   {                 m_ma.Refresh(-1);
      return(m_ma.Main(ind));
   }
   double            SAR(int ind)
   {                 m_sar.Refresh(-1);
      return(m_sar.Main(ind));
   }

   //---
   void              DecreaseFactor(double decrease_factor)
   {  m_decrease_factor = decrease_factor;
   }
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //---
   virtual double    CheckOpenLong(double price, double sl);
   virtual double    CheckOpenShort(double price, double sl);
   
   double            Optimize(double lots);
   
   double            High(int ind)
   {                 m_high.Refresh(-1);
      return(m_high.GetData(ind));
   }
   double            Low(int ind)
   {                 m_low.Refresh(-1);
      return(m_low.GetData(ind));
   }
   double            Range(int ind)
   {                 return(High(ind) - Low(ind));
   }

protected:
   //--- methods to check for patterns
   void            IsPattern_0(double &Price, ENUM_POSITION_TYPE T);
   void            IsPattern_1(double &Price, ENUM_POSITION_TYPE T);
   void            IsPattern_2(double &Price, ENUM_POSITION_TYPE T);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CMoneyATR::CMoneyATR(void) :  m_pattern_0(50),
   m_ma_applied(PRICE_CLOSE),
   m_pattern_1(50),
   m_pattern_2(50),
   m_ma_period(14),
   m_decrease_factor(3.0)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CMoneyATR::~CMoneyATR(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CMoneyATR::ValidationSettings(void)
{  if(!CExpertMoney::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_decrease_factor <= 0.0)
   {  printf(__FUNCTION__ + ": decrease factor must be greater then 0");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CMoneyATR::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- add object to collection
   if(!indicators.Add(GetPointer(m_atr)))
   {  printf(__FUNCTION__ + ": error adding object");
      return(false);
   }
//--- initialize object
   if(!m_atr.Create(m_symbol.Name(), m_period, m_ma_period))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_ma.Create(m_symbol.Name(), m_period, m_ma_period, 0, MODE_SMA, m_ma_applied))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_rsi.Create(m_symbol.Name(), m_period, m_ma_period, m_ma_applied))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
   if(!m_atr.Create(m_symbol.Name(), m_period, m_ma_period))
   {  printf(__FUNCTION__ + ": error initializing object");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Getting lot size for open long position.                         |
//+------------------------------------------------------------------+
double CMoneyATR::CheckOpenLong(double price, double sl)
{  if(m_symbol == NULL)
      return(0.0);
//--- select lot size
   double lot;
   if(price == 0.0)
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_BUY, m_symbol.Ask(), m_percent);
   else
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_BUY, price, m_percent);
//---
   double result  = 0.0, results = 0.0;
   price =m_symbol.Bid();
//--- if the model 0 is used and "ATR-Based Stop Loss"
   if(((m_patterns_usage & 0x01) != 0))
   {  IsPattern_0(price, POSITION_TYPE_BUY);
      result += m_pattern_0 * price;
      results += m_pattern_0;
   }
//--- if the model 1 is used and "ATR MA Stop Signal"
   if(((m_patterns_usage & 0x02) != 0))
   {  IsPattern_1(price, POSITION_TYPE_BUY);
      result += m_pattern_1 * price;
      results += m_pattern_1;
   }
//--- if the model 2 is used and "ATR SAR Stop Signal"
   if(((m_patterns_usage & 0x04) != 0))
   {  IsPattern_2(price, POSITION_TYPE_BUY);
      result += m_pattern_2 * price;
      results += m_pattern_2;
   }
//---
   if(results > 0)
   {  result /= results;
      double _risk = (fabs(m_symbol.Bid()-result)/m_symbol.Point())*(m_symbol.TickSize()/m_symbol.Point())*m_symbol.TickValue();
      _risk /= m_account.FreeMargin();
      _risk *= 100.0;
      double _risk_lots = m_percent/_risk;// where m_percent is also max risk
      lot = fmin(2.0*lot, fmax(_risk_lots, m_symbol.LotsMin()));
   }
//--- return trading volume
   return(Optimize(lot));
}
//+------------------------------------------------------------------+
//| Getting lot size for open short position.                        |
//+------------------------------------------------------------------+
double CMoneyATR::CheckOpenShort(double price, double sl)
{  if(m_symbol == NULL)
      return(0.0);
//--- select lot size
   double lot;
//---
   if(price == 0.0)
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_SELL, m_symbol.Bid(), m_percent);
   else
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_SELL, price, m_percent);
//---
   double result  = 0.0, results = 0.0;
   price =m_symbol.Ask();
//--- if the model 0 is used and "ATR-Based Stop Loss"
   if(((m_patterns_usage & 0x01) != 0))
   {  IsPattern_0(price, POSITION_TYPE_SELL);
      result += m_pattern_0 * price;
      results += m_pattern_0;
   }
//--- if the model 1 is used and "ATR MA Stop Signal"
   if(((m_patterns_usage & 0x02) != 0))
   {  IsPattern_1(price, POSITION_TYPE_SELL);
      result += m_pattern_1 * price;
      results += m_pattern_1;
   }
//--- if the model 2 is used and "ATR SAR Stop Signal"
   if(((m_patterns_usage & 0x04) != 0))
   {  IsPattern_2(price, POSITION_TYPE_SELL);
      result += m_pattern_2 * price;
      results += m_pattern_2;
   }
//---
   if(results > 0)
   {  result /= results;
      double _risk = (fabs(result-m_symbol.Ask())/m_symbol.Point())*(m_symbol.TickSize()/m_symbol.Point())*m_symbol.TickValue();
      _risk /= m_account.FreeMargin();
      _risk *= 100.0;
      double _risk_lots = m_percent/_risk;// where m_percent is also max risk
      lot = fmin(2.0*lot, fmax(_risk_lots, m_symbol.LotsMin()));
   }
//--- return trading volume
   return(Optimize(lot));
}
//+------------------------------------------------------------------+
//| Optimizing lot size for open.                                    |
//+------------------------------------------------------------------+
double CMoneyATR::Optimize(double lots)
{  double lot=lots;
//--- calculate number of losses orders without a break
   if(m_decrease_factor>0)
     {
      //--- select history for access
      HistorySelect(0,TimeCurrent());
      //---
      int       orders=HistoryDealsTotal();  // total history deals
      int       losses=0;                    // number of consequent losing orders
      CDealInfo deal;
      //---
      for(int i=orders-1;i>=0;i--)
        {
         deal.Ticket(HistoryDealGetTicket(i));
         if(deal.Ticket()==0)
           {
            Print("CMoneySizeOptimized::Optimize: HistoryDealGetTicket failed, no trade history");
            break;
           }
         //--- check symbol
         if(deal.Symbol()!=m_symbol.Name())
            continue;
         //--- check profit
         double profit=deal.Profit();
         if(profit>0.0)
            break;
         if(profit<0.0)
            losses++;
        }
      //---
      if(losses>1)
         lot=NormalizeDouble(lot-lot*losses/m_decrease_factor,2);
     }
//--- normalize and check limits
   double stepvol=m_symbol.LotsStep();
   lot=stepvol*NormalizeDouble(lot/stepvol,0);
//---
   double minvol=m_symbol.LotsMin();
   if(lot<minvol)
      lot=minvol;
//---
   double maxvol=m_symbol.LotsMax();
   if(lot>maxvol)
      lot=maxvol;
//---
   return(lot);
}
//+------------------------------------------------------------------+
//| Check for Pattern 0.                                             |
//+------------------------------------------------------------------+
void CMoneyATR::IsPattern_0(double &Price, ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY)
   {  Price -= (m_stop_level * ATR(StartIndex()));
   }
   else if(T == POSITION_TYPE_SELL)
   {  Price += (m_stop_level * ATR(StartIndex()));
   }
}
//+------------------------------------------------------------------+
//| Check for Pattern 1.                                             |
//+------------------------------------------------------------------+
void CMoneyATR::IsPattern_1(double &Price, ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY)
   {  Price = (MA(StartIndex()) - (m_stop_level * ATR(StartIndex())));
   }
   else if(T == POSITION_TYPE_SELL)
   {  Price = (MA(StartIndex()) + (m_stop_level * ATR(StartIndex())));
   }
}
//+------------------------------------------------------------------+
//| Check for Pattern 2.                                             |
//+------------------------------------------------------------------+
void CMoneyATR::IsPattern_2(double &Price, ENUM_POSITION_TYPE T)
{  if(T == POSITION_TYPE_BUY && SAR(StartIndex()) < Low(StartIndex()))
   {  Price = (SAR(StartIndex()) - (m_stop_level * ATR(StartIndex())));
   }
   else if(T == POSITION_TYPE_SELL && SAR(StartIndex()) > High(StartIndex()))
   {  Price = (SAR(StartIndex()) + (m_stop_level * ATR(StartIndex())));
   }
}
//+------------------------------------------------------------------+
