//+------------------------------------------------------------------+
//|                                                      NewsTrading |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                            https://www.mql5.com/en/users/kaaiblo |
//+------------------------------------------------------------------+
#include "CommonVariables.mqh"
#include "DayLightSavings/DaylightSavings_UK.mqh"
#include "DayLightSavings/DaylightSavings_US.mqh"
#include "DayLightSavings/DaylightSavings_AU.mqh"
#include "CandleProperties.mqh"

int       DBMemoryConnection;//In memory database handle

//--- Enumeration for Calendar Importance
enum Calendar_Importance
  {
   Calendar_Importance_None,//NONE
   Calendar_Importance_Low,//LOW
   Calendar_Importance_Moderate,//MODERATE
   Calendar_Importance_High,//HIGH
   Calendar_Importance_All//ALL
  } myImportance;

//--- Enumeration for Calendar Sector
enum Event_Sector
  {
   Event_Sector_None,//NONE
   Event_Sector_Market,//MARKET
   Event_Sector_Gdp,//GDP
   Event_Sector_Jobs,//JOBS
   Event_Sector_Prices,//PRICES
   Event_Sector_Money,//MONEY
   Event_Sector_Trade,//TRADE
   Event_Sector_Government,//GOVERNMENT
   Event_Sector_Business,//BUSINESS
   Event_Sector_Consumer,//CONSUMER
   Event_Sector_Housing,//HOUSING
   Event_Sector_Taxes,//TAXES
   Event_Sector_Holidays,//HOLIDAYS
   Event_Sector_ALL//ALL
  } mySector;

//--- Enumeration for Calendar Event Frequency
enum Event_Frequency
  {
   Event_Frequency_None,//NONE
   Event_Frequency_Week,//WEEK
   Event_Frequency_Month,//MONTH
   Event_Frequency_Quarter,//QUARTER
   Event_Frequency_Year,//YEAR
   Event_Frequency_Day,//DAY
   Event_Frequency_ALL//ALL
  } myFrequency;

//--- Enumeration for Calendar Event type
enum Event_Type
  {
   Event_Type_Event,//EVENT
   Event_Type_Indicator,//INDICATOR
   Event_Type_Holiday,//HOLIDAY
   Event_Type_All//ALL
  } myType;

//--- Enumeration for Calendar Event Currency
enum Event_Currency
  {
   Event_Currency_Symbol,//SYMBOL CURRENCIES
   Event_Currency_Margin,//SYMBOL MARGIN
   Event_Currency_Base,//SYMBOL BASE
   Event_Currency_Profit,//SYMBOL PROFIT
   Event_Currency_ALL,//ALL CURRENCIES
   Event_Currency_NZD_NZ,//NZD -> NZ
   Event_Currency_EUR_EU,//EUR -> EU
   Event_Currency_JPY_JP,//JPY -> JP
   Event_Currency_CAD_CA,//CAD -> CA
   Event_Currency_AUD_AU,//AUD -> AU
   Event_Currency_CNY_CN,//CNY -> CN
   Event_Currency_EUR_IT,//EUR -> IT
   Event_Currency_SGD_SG,//SGD -> SG
   Event_Currency_EUR_DE,//EUR -> DE
   Event_Currency_EUR_FR,//EUR -> FR
   Event_Currency_BRL_BR,//BRL -> BR
   Event_Currency_MXN_MX,//MXN -> MX
   Event_Currency_ZAR_ZA,//ZAR -> ZA
   Event_Currency_HKD_HK,//HKD -> HK
   Event_Currency_INR_IN,//INR -> IN
   Event_Currency_NOK_NO,//NOK -> NO
   Event_Currency_USD_US,//USD -> US
   Event_Currency_GBP_GB,//GBP -> GB
   Event_Currency_CHF_CH,//CHF -> CH
   Event_Currency_KRW_KR,//KRW -> KW
   Event_Currency_EUR_ES,//EUR -> ES
   Event_Currency_SEK_SE,//SEK -> SE
   Event_Currency_ALL_WW//ALL -> WW
  } myCurrency;

//--- Enumeration for News Profiles
enum NewsSelection
  {
   News_Select_Custom_Events,//CUSTOM NEWS EVENTS
   News_Select_Settings//NEWS SETTINGS
  } myNewsSelection;

//--- Structure to store event ids and whether to use these ids
struct CustomEvent
  {
   bool              useEvents;
   string            EventIds[];
  } CEvent1,CEvent2,CEvent3,CEvent4,CEvent5;

//--- Structure variable to store Calendar next Event data
Calendar UpcomingNews;
//+------------------------------------------------------------------+
//|News class                                                        |
//+------------------------------------------------------------------+
class CNews : private CCandleProperties
  {
   //Private Declarations Only accessable by this class/header file
private:

   //-- To keep track of what is in our database
   enum CalendarComponents
     {
      AutoDST_Table,//AutoDST Table
      CalendarAU_View,//View for DST_AU
      CalendarNONE_View,//View for DST_NONE
      CalendarUK_View,//View for DST_UK
      CalendarUS_View,//View for DST_US
      EventInfo_View,//View for Event Information
      Currencies_View,//View for Currencies
      RecentEventInfo_View,//View for Recent Dates For Events
      UpcomingEventInfo_View,//View for Upcoming Dates For Events
      Record_Table,// Record Table
      TimeSchedule_Table,//TimeSchedule Table
      MQL5Calendar_Table,//MQL5Calendar Table
      AutoDST_Trigger,//Table Trigger for AutoDST
      Record_Trigger//Table Trigger for Record
     };

   //-- structure to retrieve all the objects in the database
   struct SQLiteMaster
     {
      string         type;//will store object's type
      string         name;//will store object's name
      string         tbl_name;//will store table name
      int            rootpage;//will store rootpage
      string         sql;//Will store the sql create statement
     } DBContents[];//Array of type SQLiteMaster

   //--  MQL5CalendarContents inherits from SQLiteMaster structure
   struct MQL5CalendarContents:SQLiteMaster
     {
      CalendarComponents  Content;
      string         insert;//Will store the sql insert statement
     } CalendarContents[14],DBMemory;//Array to Store objects in our database

   CTimeManagement   CTime;//TimeManagement Object declaration
   CDaylightSavings_UK  Savings_UK;//DaylightSavings Object for the UK and EU
   CDaylightSavings_US  Savings_US;//DaylightSavings Object for the US
   CDaylightSavings_AU  Savings_AU;//DaylightSavings Object for the AU

   bool              AutoDetectDST(DST_type &dstType);//Function will determine Broker DST
   DST_type          DSTType;//variable of DST_type enumeration declared in the CommonVariables class/header file
   bool              InsertIntoTables(int db,Calendar &Evalues[]);//Function for inserting Economic Data in to a database's table
   void              CreateAutoDST(int db);//Function for creating and inserting Recommend DST for the Broker into a table
   bool              CreateCalendarTable(int db,bool &tableExists);//Function for creating a table in a database
   bool              CreateTimeTable(int db,bool &tableExists);//Function for creating a table in a database
   void              CreateCalendarViews(int db);//Function for creating views in a database
   void              CreateRecordTable(int db);//Creates a table to store the record of when last the Calendar database was updated/created
   string            DropRequest;//Variable for dropping tables in the database

   //-- Function for retrieving the MQL5CalendarContents structure for the enumartion type CalendarComponents
   MQL5CalendarContents CalendarStruct(CalendarComponents Content)
     {
      MQL5CalendarContents Calendar;
      for(uint i=0;i<CalendarContents.Size();i++)
        {
         if(CalendarContents[i].Content==Content)
           {
            return CalendarContents[i];
           }
        }
      return Calendar;
     }

   //--- To Store Calendar DB Data
   struct CalendarData
     {
      int            EventId;//Event Id
      string         Country;//Event Country
      string         EventName;//Event Name
      string         EventType;//Event Type
      string         EventImportance;//Event Importance
      string         EventCurrency;//Event Currency
      string         EventCode;//Event Code
      string         EventSector;//Event Sector
      string         EventForecast;//Event Forecast Value
      string         EventPreval;//Event Previous Value
      string         EventImpact;//Event Impact
      string         EventFrequency;//Event Frequency
      string         DST_UK;//DST UK
      string         DST_US;//DST US
      string         DST_AU;//DST AU
      string         DST_NONE;//DST NONE
     } DB_Data[],DB_Cal;//Structure variables

   //--- Will Retrieve all relevant Calendar data for DB in Memory from DB in Storage
   void              GetCalendar(CalendarData &Data[])
     {
      //--- Open calendar DB in Storage
      int db=DatabaseOpen(NEWS_DATABASE_FILE, DATABASE_OPEN_READWRITE | DATABASE_OPEN_CREATE| DATABASE_OPEN_COMMON);
      if(db==INVALID_HANDLE)//Checks if the database was able to be opened
        {
         //if opening the database failed
         if(!FileIsExist(NEWS_DATABASE_FILE,FILE_COMMON))//Checks if the database Calendar exists in the common folder
           {
            return;//Returns true when the database was failed to be opened and the file doesn't exist in the common folder
           }
        }

      string SqlRequest;

      //--- switch statement for different News Profiles
      switch(myNewsSelection)
        {
         case  News_Select_Custom_Events://CUSTOM NEWS EVENTS
            //--- Get filtered calendar DB data
            SqlRequest = StringFormat("Select MQ.EventId,MQ.Country,MQ.EventName,MQ.EventType,MQ.EventImportance,MQ.EventCurrency,"
                                      "MQ.EventCode,MQ.EventSector,MQ.EventForecast,MQ.EventPreValue,MQ.EventImpact,MQ.EventFrequency,"
                                      "TS.DST_UK,TS.DST_US,TS.DST_AU,TS.DST_NONE from %s MQ "
                                      "Inner Join %s TS on TS.ID=MQ.ID Where %s OR %s OR %s OR %s OR %s;",
                                      CalendarStruct(MQL5Calendar_Table).name,CalendarStruct(TimeSchedule_Table).name,
                                      Request_Events(CEvent1),Request_Events(CEvent2),Request_Events(CEvent3),
                                      Request_Events(CEvent4),Request_Events(CEvent5));
            break;
         case News_Select_Settings://NEWS SETTINGS
            //--- Get filtered calendar DB data
            SqlRequest = StringFormat("Select MQ.EventId,MQ.Country,MQ.EventName,MQ.EventType,MQ.EventImportance,MQ.EventCurrency,"
                                      "MQ.EventCode,MQ.EventSector,MQ.EventForecast,MQ.EventPreValue,MQ.EventImpact,MQ.EventFrequency,"
                                      "TS.DST_UK,TS.DST_US,TS.DST_AU,TS.DST_NONE from %s MQ "
                                      "Inner Join %s TS on TS.ID=MQ.ID "
                                      "Where %s and %s and %s and %s and %s;",
                                      CalendarStruct(MQL5Calendar_Table).name,CalendarStruct(TimeSchedule_Table).name,
                                      Request_Importance(myImportance),Request_Frequency(myFrequency),
                                      Request_Sector(mySector),Request_Type(myType),Request_Currency(myCurrency));
            break;
         default://Unknown
            break;
        }

      //--- Process Sql request
      int Request = DatabasePrepare(db,SqlRequest);
      if(Request==INVALID_HANDLE)
        {
         //--- Print details if request failed.
         Print("DB: ",NEWS_DATABASE_FILE, " request failed with code ", GetLastError());
         Print("SQL");
         Print(SqlRequest);
        }
      else
        {
         //--- Clear data from whole array
         ArrayRemove(Data,0,WHOLE_ARRAY);
         //--- create structure variable to get data from request
         CalendarData data;
         //Assigning values from the sql query into Data structure array
         for(int i=0; DatabaseReadBind(Request,data); i++)
           {
            //--- Resize Data Array
            ArrayResize(Data,i+1,i+2);
            Data[i]  = data;
           }
        }
      DatabaseFinalize(Request);//Finalize request
      //--- Close Calendar database
      DatabaseClose(db);
     }

   //--- Retrieve the AutoDST enumeration data from calendar DB in storage
   DST_type          GetAutoDST()
     {
      string Sch_Dst;
      //--- open the database 'Calendar' in the common folder
      int db=DatabaseOpen(NEWS_DATABASE_FILE, DATABASE_OPEN_READONLY|DATABASE_OPEN_COMMON);

      if(db==INVALID_HANDLE)//Checks if 'Calendar' failed to be opened
        {
         if(!FileIsExist(NEWS_DATABASE_FILE,FILE_COMMON))//Checks if 'Calendar' database exists
           {
            Print("Could not find Database!");
            return DST_NONE;//return default value when failed.
           }
        }

      //--- Sql query to get AutoDST value
      string request_text="SELECT DST FROM 'AutoDST'";
      //--- Process sql request
      int request=DatabasePrepare(db,request_text);
      if(request==INVALID_HANDLE)
        {
         Print("DB: ",NEWS_DATABASE_FILE, " request failed with code ", GetLastError());
         DatabaseClose(db);//Close Database
         return DST_NONE;//return default value when failed.
        }

      //--- Read Sql request output data
      if(DatabaseRead(request))
        {
         //-- Store the first column data into string variable Sch_Dst
         if(!DatabaseColumnText(request,0,Sch_Dst))
           {
            Print("DatabaseRead() failed with code ", GetLastError());
            DatabaseFinalize(request);//Finalize request
            DatabaseClose(db);//Closes the database 'Calendar'
            return DST_NONE;//return default value when failed.
           }
        }
      DatabaseFinalize(request);//Finalize request
      DatabaseClose(db);//Closes the database 'Calendar'
      return (Sch_Dst=="DST_UK")?DST_UK:(Sch_Dst=="DST_US")?DST_US:
             (Sch_Dst=="DST_AU")?DST_AU:DST_NONE;//Returns the enumeration value for each corresponding string
     }

   //--- Retrieve Sql request string for custom event ids
   string            Request_Events(CustomEvent &CEvent)
     {
      //--- Default request string
      string EventReq="MQ.EventId='0'";
      //--- Check if this Custom event should be included in the SQL request
      if(CEvent.useEvents)
        {
         //--- Get request for first event id
         EventReq=StringFormat("(MQ.EventId='%s'",
                               (CEvent.EventIds.Size()>0)?
                               CEvent.EventIds[0]:"0");
         //--- Iterate through remaining event ids and add to the SQL request
         for(uint i=1;i<CEvent.EventIds.Size();i++)
           {
            EventReq+=StringFormat(" OR MQ.EventId='%s'",CEvent.EventIds[i]);
           }
         EventReq+=")";
        }
      //--- Return SQL request for custom event ids
      return EventReq;
     }

   //--- Retrieve Sql request string for calendar event Importance
   string            Request_Importance(Calendar_Importance Importance)
     {
      //--- Constant request prefix string
      const string constant="MQ.EventImportance";
      //--- switch statement for Calendar_Importance enumeration
      switch(Importance)
        {
         case Calendar_Importance_All://String Request for all event Importance
            return constant+"<>'"+EnumToString(myImportance)+"'";
            break;
         default://String Request for any event Importance
            return constant+"='"+EnumToString(IMPORTANCE(myImportance))+"'";
            break;
        }
     }

   //--- Retrieve Sql request string for calendar event Frequency
   string            Request_Frequency(Event_Frequency Frequency)
     {
      //--- Constant request prefix string
      const string constant="MQ.EventFrequency";
      //--- switch statement for Event_Frequency enumeration
      switch(Frequency)
        {
         case Event_Frequency_ALL://String Request for all event frequencies
            return constant+"<>'"+EnumToString(myFrequency)+"'";
            break;
         default://String Request for any event frequency
            return constant+"='"+EnumToString(FREQUENCY(myFrequency))+"'";
            break;
        }
     }

   //--- Retrieve Sql request string for calendar event Sector
   string            Request_Sector(Event_Sector Sector)
     {
      //--- Constant request prefix string
      const string constant="MQ.EventSector";
      //--- switch statement for Event_Sector enumeration
      switch(Sector)
        {
         case Event_Sector_ALL://String Request for all event sectors
            return constant+"<>'"+EnumToString(mySector)+"'";
            break;
         default://String Request for any event sector
            return constant+"='"+EnumToString(SECTOR(mySector))+"'";
            break;
        }
     }

   //--- Retrieve Sql request string for calendar event type
   string            Request_Type(Event_Type Type)
     {
      //--- Constant request prefix string
      const string constant="MQ.EventType";
      //--- switch statement for Event_Type enumeration
      switch(Type)
        {
         case Event_Type_All://String Request for all event types
            return constant+"<>'"+EnumToString(myType)+"'";
            break;
         default://String request for any event type
            return constant+"='"+EnumToString(TYPE(myType))+"'";
            break;
        }
     }

   //--- Retrieve Sql request string for calendar event Currency
   string            Request_Currency(Event_Currency Currency)
     {
      //--- Constant request prefix string and request suffix
      const string constant_prefix="(MQ.EventCurrency",constant_suffix="')";
      //--- switch statement for Event_Currency enumeration
      switch(Currency)
        {
         case Event_Currency_ALL://String Request for all currencies
            return constant_prefix+"<>'"+EnumToString(myCurrency)+constant_suffix;
            break;
         case Event_Currency_Symbol://String Request for all symbol currencies
            return constant_prefix+"='"+CSymbol.CurrencyBase()+"' or MQ.EventCurrency='"+
                   CSymbol.CurrencyMargin()+"' or MQ.EventCurrency='"+CSymbol.CurrencyProfit()+constant_suffix;
            break;
         case Event_Currency_Margin://String Request for Margin currency
            return constant_prefix+"='"+CSymbol.CurrencyMargin()+constant_suffix;
            break;
         case Event_Currency_Base://String Request for Base currency
            return constant_prefix+"='"+CSymbol.CurrencyBase()+constant_suffix;
            break;
         case Event_Currency_Profit://String Request for Profit currency
            return constant_prefix+"='"+CSymbol.CurrencyProfit()+constant_suffix;
            break;
         case Event_Currency_NZD_NZ://String Request for NZD currency
            return constant_prefix+"='NZD' and MQ.EventCode='NZ"+constant_suffix;
            break;
         case Event_Currency_EUR_EU://String Request for EUR currency and EU code
            return constant_prefix+"='EUR' and MQ.EventCode='EU"+constant_suffix;
            break;
         case Event_Currency_JPY_JP://String Request for JPY currency
            return constant_prefix+"='JPY' and MQ.EventCode='JP"+constant_suffix;
            break;
         case Event_Currency_CAD_CA://String Request for CAD currency
            return constant_prefix+"='CAD' and MQ.EventCode='CA"+constant_suffix;
            break;
         case Event_Currency_AUD_AU://String Request for AUD currency
            return constant_prefix+"='AUD' and MQ.EventCode='AU"+constant_suffix;
            break;
         case Event_Currency_CNY_CN://String Request for CNY currency
            return constant_prefix+"='CNY' and MQ.EventCode='CN"+constant_suffix;
            break;
         case Event_Currency_EUR_IT://String Request for EUR currency and IT code
            return constant_prefix+"='EUR' and MQ.EventCode='IT"+constant_suffix;
            break;
         case Event_Currency_SGD_SG://String Request for SGD currency
            return constant_prefix+"='SGD' and MQ.EventCode='SG"+constant_suffix;
            break;
         case Event_Currency_EUR_DE://String Request for EUR currency and DE code
            return constant_prefix+"='EUR' and MQ.EventCode='DE"+constant_suffix;
            break;
         case Event_Currency_EUR_FR://String Request for EUR currency and FR code
            return constant_prefix+"='EUR' and MQ.EventCode='FR"+constant_suffix;
            break;
         case Event_Currency_BRL_BR://String Request for BRL currency
            return constant_prefix+"='BRL' and MQ.EventCode='BR"+constant_suffix;
            break;
         case Event_Currency_MXN_MX://String Request for MXN currency
            return constant_prefix+"='MXN' and MQ.EventCode='MX"+constant_suffix;
            break;
         case Event_Currency_ZAR_ZA://String Request for ZAR currency
            return constant_prefix+"='ZAR' and MQ.EventCode='ZA"+constant_suffix;
            break;
         case Event_Currency_HKD_HK://String Request for HKD currency
            return constant_prefix+"='HKD' and MQ.EventCode='HK"+constant_suffix;
            break;
         case Event_Currency_INR_IN://String Request for INR currency
            return constant_prefix+"='INR' and MQ.EventCode='IN"+constant_suffix;
            break;
         case Event_Currency_NOK_NO://String Request for NOK currency
            return constant_prefix+"='NOK' and MQ.EventCode='NO"+constant_suffix;
            break;
         case Event_Currency_USD_US://String Request for USD currency
            return constant_prefix+"='USD' and MQ.EventCode='US"+constant_suffix;
            break;
         case Event_Currency_GBP_GB://String Request for GBP currency
            return constant_prefix+"='GBP' and MQ.EventCode='GB"+constant_suffix;
            break;
         case Event_Currency_CHF_CH://String Request for CHF currency
            return constant_prefix+"='CHF' and MQ.EventCode='CH"+constant_suffix;
            break;
         case Event_Currency_KRW_KR://String Request for KRW currency
            return constant_prefix+"='KRW' and MQ.EventCode='KR"+constant_suffix;
            break;
         case Event_Currency_EUR_ES://String Request for EUR currency and ES code
            return constant_prefix+"='EUR' and MQ.EventCode='ES"+constant_suffix;
            break;
         case Event_Currency_SEK_SE://String Request for SEK currency
            return constant_prefix+"='SEK' and MQ.EventCode='SE"+constant_suffix;
            break;
         case Event_Currency_ALL_WW://String Request for ALL currency
            return constant_prefix+"='ALL' and MQ.EventCode='WW"+constant_suffix;
            break;
         default://String Request for no currencies
            return constant_prefix+"='"+constant_suffix;
            break;
        }
     }

   //Public declarations accessable via a class's Object
public:
                     CNews(void);//Constructor
                    ~CNews(void);//Destructor
   void              CreateEconomicDatabase();//Creates the Calendar database for a specific Broker
   datetime          GetLatestNewsDate();//Gets the latest/newest date in the Calendar database
   void              EconomicDetails(Calendar &NewsTime[],datetime date_from=0,datetime date_to=0);//Gets values from the MQL5 economic Calendar
   void              EconomicDetailsMemory(Calendar &NewsTime[],datetime date,bool ImpactRequired);//Gets values from the MQL5 DB Calendar in Memory
   void              CreateEconomicDatabaseMemory();//Create calendar database in memory
   void              EconomicNextEvent();//Will update UpcomingNews structure variable with the next event data
   bool              UpdateRecords();//Checks if the main Calendar database needs an update or not
   //--- Checks if a news event is occurring and modifies the parameters passed by reference
   bool              isEvent(uint SecondsPreEvent,string &Name,string &Importance,string &Code);

   //--- Convert Importance string into Calendar Event Importance Enumeration
   ENUM_CALENDAR_EVENT_IMPORTANCE IMPORTANCE(string Importance)
     {
      //--- Calendar Importance is High
      if(Importance==EnumToString(CALENDAR_IMPORTANCE_HIGH))
        {
         return CALENDAR_IMPORTANCE_HIGH;
        }
      else
         //--- Calendar Importance is Moderate
         if(Importance==EnumToString(CALENDAR_IMPORTANCE_MODERATE))
           {
            return CALENDAR_IMPORTANCE_MODERATE;
           }
         else
            //--- Calendar Importance is Low
            if(Importance==EnumToString(CALENDAR_IMPORTANCE_LOW))
              {
               return CALENDAR_IMPORTANCE_LOW;
              }
            else
               //--- Calendar Importance is None
              {
               return CALENDAR_IMPORTANCE_NONE;
              }
     }

   //--- Convert Calendar_Importance Enumeration into Calendar Event Importance Enumeration
   ENUM_CALENDAR_EVENT_IMPORTANCE IMPORTANCE(Calendar_Importance Importance)
     {
      //--- switch statement for Calendar_Importance enumeration
      switch(Importance)
        {
         case Calendar_Importance_None://None
            return CALENDAR_IMPORTANCE_NONE;
            break;
         case Calendar_Importance_Low://Low
            return CALENDAR_IMPORTANCE_LOW;
            break;
         case Calendar_Importance_Moderate://Moderate
            return CALENDAR_IMPORTANCE_MODERATE;
            break;
         case Calendar_Importance_High://High
            return CALENDAR_IMPORTANCE_HIGH;
            break;
         default://None
            return CALENDAR_IMPORTANCE_NONE;
            break;
        }
     }

   //--- Convert Calendar Event Importance Enumeration into string Importance Rating
   string            GetImportance(ENUM_CALENDAR_EVENT_IMPORTANCE Importance)
     {
      //--- switch statement for ENUM_CALENDAR_EVENT_IMPORTANCE enumeration
      switch(Importance)
        {
         case  CALENDAR_IMPORTANCE_HIGH://High
            return "HIGH";
            break;
         case CALENDAR_IMPORTANCE_MODERATE://Moderate
            return "MODERATE";
            break;
         case CALENDAR_IMPORTANCE_LOW://Low
            return "LOW";
            break;
         default://None
            return "NONE";
            break;
        }
     }

   //--- Retrieve color for each Calendar Event Importance Enumeration
   color             GetImportance_color(ENUM_CALENDAR_EVENT_IMPORTANCE Importance)
     {
      //--- switch statement for ENUM_CALENDAR_EVENT_IMPORTANCE enumeration
      switch(Importance)
        {
         case CALENDAR_IMPORTANCE_HIGH://High
            return clrRed;
            break;
         case CALENDAR_IMPORTANCE_MODERATE://Moderate
            return clrOrange;
            break;
         case CALENDAR_IMPORTANCE_LOW://Low
            return clrBlue;
            break;
         default://None
            return (isLightMode)?clrBlack:clrWhite;
            break;
        }
     }

   //--- Convert Event_Sector Enumeration into Calendar Event Sector Enumeration
   ENUM_CALENDAR_EVENT_SECTOR SECTOR(Event_Sector Sector)
     {
      //--- switch statement for Event_Sector enumeration
      switch(Sector)
        {
         case Event_Sector_None://NONE
            return CALENDAR_SECTOR_NONE;
            break;
         case Event_Sector_Market://MARKET
            return CALENDAR_SECTOR_MARKET;
            break;
         case Event_Sector_Gdp://GDP
            return CALENDAR_SECTOR_GDP;
            break;
         case Event_Sector_Jobs://JOBS
            return CALENDAR_SECTOR_JOBS;
            break;
         case Event_Sector_Prices://PRICES
            return CALENDAR_SECTOR_PRICES;
            break;
         case Event_Sector_Money://MONEY
            return CALENDAR_SECTOR_MONEY;
            break;
         case Event_Sector_Trade://TRADE
            return CALENDAR_SECTOR_TRADE;
            break;
         case Event_Sector_Government://GOVERNMENT
            return CALENDAR_SECTOR_GOVERNMENT;
            break;
         case Event_Sector_Business://BUSINESS
            return CALENDAR_SECTOR_BUSINESS;
            break;
         case Event_Sector_Consumer://CONSUMER
            return CALENDAR_SECTOR_CONSUMER;
            break;
         case Event_Sector_Housing://HOUSING
            return CALENDAR_SECTOR_HOUSING;
            break;
         case Event_Sector_Taxes://TAXES
            return CALENDAR_SECTOR_TAXES;
            break;
         case Event_Sector_Holidays://HOLIDAYS
            return CALENDAR_SECTOR_HOLIDAYS;
            break;
         default://Unknown
            return CALENDAR_SECTOR_NONE;
            break;
        }
     }

   //--- Convert Event_Frequency Enumeration into Calendar Event Frequency Enumeration
   ENUM_CALENDAR_EVENT_FREQUENCY FREQUENCY(Event_Frequency Frequency)
     {
      //--- switch statement for Event_Frequency enumeration
      switch(Frequency)
        {
         case  Event_Frequency_None://NONE
            return CALENDAR_FREQUENCY_NONE;
            break;
         case Event_Frequency_Day://DAY
            return CALENDAR_FREQUENCY_DAY;
            break;
         case Event_Frequency_Week://WEEK
            return CALENDAR_FREQUENCY_WEEK;
            break;
         case Event_Frequency_Month://MONTH
            return CALENDAR_FREQUENCY_MONTH;
            break;
         case Event_Frequency_Quarter://QUARTER
            return CALENDAR_FREQUENCY_QUARTER;
            break;
         case Event_Frequency_Year://YEAR
            return CALENDAR_FREQUENCY_YEAR;
            break;
         default://Unknown
            return CALENDAR_FREQUENCY_NONE;
            break;
        }
     }

   //--- Convert Event_Type Enumeration into Calendar Event Type Enumeration
   ENUM_CALENDAR_EVENT_TYPE TYPE(Event_Type Type)
     {
      //--- switch statement for Event_Type enumeration
      switch(Type)
        {
         case Event_Type_Event://EVENT
            return CALENDAR_TYPE_EVENT;
            break;
         case Event_Type_Indicator://INDICATOR
            return CALENDAR_TYPE_INDICATOR;
            break;
         case Event_Type_Holiday://HOLIDAY
            return CALENDAR_TYPE_HOLIDAY;
            break;
         default://Unknown
            return CALENDAR_TYPE_EVENT;
            break;
        }
     }

   //--- Convert Impact string into Calendar Event Impact Enumeration
   ENUM_CALENDAR_EVENT_IMPACT IMPACT(string Impact)
     {
      //--- Positive Impact
      if(Impact=="CALENDAR_IMPACT_POSITIVE")
        {
         return CALENDAR_IMPACT_POSITIVE;
        }
      else
         //--- Negative Impact
         if(Impact=="CALENDAR_IMPACT_NEGATIVE")
           {
            return CALENDAR_IMPACT_NEGATIVE;
           }
         else
           {
            //--- Unknown Impact
            return CALENDAR_IMPACT_NA;
           }
     }
  };

//+------------------------------------------------------------------+
//|Constructor                                                       |
//+------------------------------------------------------------------+
CNews::CNews(void):DropRequest("PRAGMA foreign_keys = OFF; "
                                  "PRAGMA secure_delete = ON; "
                                  "Drop %s IF EXISTS '%s'; "
                                  "Vacuum; "
                                  "PRAGMA foreign_keys = ON;")//Sql drop statement
  {
//-- initializing properties for the AutoDST table
   CalendarContents[0].Content = AutoDST_Table;
   CalendarContents[0].name = "AutoDST";
   CalendarContents[0].sql = "CREATE TABLE IF NOT EXISTS AutoDST(DST TEXT NOT NULL DEFAULT 'DST_NONE')STRICT;";
   CalendarContents[0].tbl_name = "AutoDST";
   CalendarContents[0].type = "table";
   CalendarContents[0].insert = "INSERT INTO 'AutoDST'(DST) VALUES ('%s');";

   string views[] = {"AU","NONE","UK","US"};
//-- Sql statement for creating the table views for each DST schedule
   string view_sql = "CREATE VIEW IF NOT EXISTS Calendar_%s "
                     "AS "
                     "SELECT C.Eventid as 'ID',C.Eventname as 'Name',C.Country as 'Country', "
                     "(CASE WHEN Date(REPLACE(T.DST_%s,'.','-'))<R.Date THEN CONCAT(T.DST_%s,' | Yesterday') "
                     "WHEN Date(REPLACE(T.DST_%s,'.','-'))=R.Date THEN CONCAT(T.DST_%s,' | Today') "
                     "WHEN Date(REPLACE(T.DST_%s,'.','-'))>R.Date THEN CONCAT(T.DST_%s,' | Tomorrow') END) as "
                     "'Date',C.EventCurrency as 'Currency',Replace(C.EventImportance,'CALENDAR_IMPORTANCE_','')"
                     " as 'Importance' from MQL5Calendar C,Record R Inner join TimeSchedule T on C.ID=T.ID Where"
                     " DATE(REPLACE(T.DST_%s,'.','-'))>=DATE(R.Date,'-1 day') AND DATE(REPLACE(T.DST_%s,'.','-'))"
                     "<=DATE(R.Date,'+1 day') Order by T.DST_%s Asc;";

//Assign properties for table views for each DST schedule
   for(uint i=1;i<=views.Size();i++)
     {
      CalendarContents[i].Content = (CalendarComponents)i;
      CalendarContents[i].name = StringFormat("Calendar_%s",views[i-1]);
      CalendarContents[i].sql = StringFormat(view_sql,views[i-1],views[i-1],views[i-1],views[i-1],views[i-1]
                                             ,views[i-1],views[i-1],views[i-1],views[i-1],views[i-1]);
      CalendarContents[i].tbl_name = StringFormat("Calendar_%s",views[i-1]);
      CalendarContents[i].type = "view";
     }

//--- initializing properties for the EventInfo view
   CalendarContents[5].Content = EventInfo_View;
   CalendarContents[5].name = "Event Info";
   CalendarContents[5].sql = "CREATE VIEW IF NOT EXISTS 'Event Info' "
                             "AS SELECT DISTINCT MC.EVENTID as 'ID',MC.COUNTRY as 'Country',MC.EVENTNAME as 'Name',"
                             "REPLACE(MC.EVENTTYPE,'CALENDAR_TYPE_','') as 'Type',REPLACE(MC.EVENTSECTOR,'CALENDAR_SECTOR_','') as 'Sector',"
                             "REPLACE(MC.EVENTIMPORTANCE,'CALENDAR_IMPORTANCE_','') as 'Importance',MC.EVENTCURRENCY as 'Currency',"
                             "REPLACE(MC.EVENTFREQUENCY,'CALENDAR_FREQUENCY_','') as 'Frequency',MC.EVENTCODE as 'Code' "
                             "FROM MQL5Calendar MC ORDER BY \"Country\" Asc,"
                             "CASE \"Importance\" WHEN 'HIGH' THEN 1 WHEN 'MODERATE' THEN 2 WHEN 'LOW' THEN 3 ELSE 4 END,\"Sector\" Desc;";
   CalendarContents[5].tbl_name = "Event Info";
   CalendarContents[5].type = "view";

//--- initializing properties for the Currencies view
   CalendarContents[6].Content = Currencies_View;
   CalendarContents[6].name = "Currencies";
   CalendarContents[6].sql = "CREATE VIEW IF NOT EXISTS Currencies AS "
                             "SELECT Distinct EventCurrency as 'Currency',EventCode as 'Code' FROM 'MQL5Calendar';";
   CalendarContents[6].tbl_name = "Currencies";
   CalendarContents[6].type = "view";

//--- initializing properties for the UpcomingEventInfo view
   CalendarContents[7].Content = UpcomingEventInfo_View;
   CalendarContents[7].name = "Upcoming Event Dates";
   CalendarContents[7].sql = "CREATE VIEW IF NOT EXISTS 'Upcoming Event Dates' AS WITH UNIQUE_EVENTS AS(SELECT DISTINCT M.EVENTID as 'E_ID',"
                             "M.COUNTRY as 'Country',M.EVENTNAME as 'Name',M.EVENTCURRENCY as 'Currency' FROM 'MQL5Calendar' M),"
                             "INFO_DATE AS(SELECT E_ID,Country,Name,Currency,(SELECT T.DST_NONE as 'Time' FROM MQL5Calendar M,"
                             "Record R INNER JOIN TIMESCHEDULE T ON T.ID=M.ID WHERE DATE(REPLACE(Time,'.','-'))>R.Date AND "
                             "E_ID=M.EVENTID ORDER BY Time ASC LIMIT 1) as 'Next Event Date' FROM UNIQUE_EVENTS) SELECT E_ID "
                             "as 'ID',Country,Name,Currency,(CASE WHEN \"Next Event Date\" IS NULL THEN 'Unknown' ELSE "
                             "\"Next Event Date\" END) as 'Upcoming Date',(CASE WHEN \"Next Event Date\"<>'Unknown' THEN "
                             "(case cast (strftime('%w', DATE(REPLACE(\"Next Event Date\",'.','-'))) as integer) WHEN 0 THEN"
                             " 'Sunday' WHEN 1 THEN 'Monday' WHEN 2 THEN 'Tuesday' WHEN 3 THEN 'Wednesday' WHEN 4 THEN 'Thursday'"
                             " WHEN 5 THEN 'Friday' ELSE 'Saturday' END)  ELSE 'Unknown' END) as 'Day' FROM INFO_DATE Order BY "
                             "\"Upcoming Date\" ASC;";
   CalendarContents[7].tbl_name = "Upcoming Event Dates";
   CalendarContents[7].type = "view";

//--- initializing properties for the RecentEventInfo view
   CalendarContents[8].Content = RecentEventInfo_View;
   CalendarContents[8].name = "Recent Event Dates";
   CalendarContents[8].sql = "CREATE VIEW IF NOT EXISTS 'Recent Event Dates' AS WITH UNIQUE_EVENTS AS(SELECT DISTINCT M.EVENTID"
                             " as 'E_ID',M.COUNTRY as 'Country',M.EVENTNAME as 'Name',M.EVENTCURRENCY as 'Currency'"
                             "FROM 'MQL5Calendar' M),INFO_DATE AS(SELECT E_ID,Country,Name,Currency,"
                             "(SELECT T.DST_NONE as 'Time' FROM MQL5Calendar M,Record R INNER JOIN TIMESCHEDULE T ON"
                             " T.ID=M.ID WHERE DATE(REPLACE(Time,'.','-'))<=R.Date AND E_ID=M.EVENTID ORDER BY Time DESC"
                             " LIMIT 1) as 'Last Event Date' FROM UNIQUE_EVENTS) SELECT E_ID as 'ID',Country,Name,Currency"
                             ",\"Last Event Date\" as 'Recent Date',(case cast (strftime('%w', DATE(REPLACE(\"Last Event Date\""
                             ",'.','-'))) as integer) WHEN 0 THEN 'Sunday' WHEN 1 THEN 'Monday' WHEN 2 THEN 'Tuesday' WHEN 3 THEN"
                             " 'Wednesday' WHEN 4 THEN 'Thursday' WHEN 5 THEN 'Friday' ELSE 'Saturday' END) as 'Day' FROM INFO_DATE"
                             " Order BY \"Recent Date\" DESC;";
   CalendarContents[8].tbl_name = "Recent Event Dates";
   CalendarContents[8].type = "view";

//-- initializing properties for the Record table
   CalendarContents[9].Content = Record_Table;
   CalendarContents[9].name = "Record";
   CalendarContents[9].sql = "CREATE TABLE IF NOT EXISTS Record(Date TEXT NOT NULL)STRICT;";
   CalendarContents[9].tbl_name="Record";
   CalendarContents[9].type = "table";
   CalendarContents[9].insert = "INSERT INTO 'Record'(Date) VALUES (Date(REPLACE('%s','.','-')));";

//-- initializing properties for the TimeSchedule table
   CalendarContents[10].Content = TimeSchedule_Table;
   CalendarContents[10].name = "TimeSchedule";
   CalendarContents[10].sql = "CREATE TABLE IF NOT EXISTS TimeSchedule(ID INT NOT NULL,DST_UK   TEXT   NOT NULL,DST_US   TEXT   NOT NULL,"
                              "DST_AU   TEXT   NOT NULL,DST_NONE   TEXT   NOT NULL,FOREIGN KEY (ID) REFERENCES MQL5Calendar (ID))STRICT;";
   CalendarContents[10].tbl_name="TimeSchedule";
   CalendarContents[10].type = "table";
   CalendarContents[10].insert = "INSERT INTO 'TimeSchedule'(ID,DST_UK,DST_US,DST_AU,DST_NONE) "
                                 "VALUES (%d,'%s','%s', '%s', '%s');";

//-- initializing properties for the MQL5Calendar table
   CalendarContents[11].Content = MQL5Calendar_Table;
   CalendarContents[11].name = "MQL5Calendar";
   CalendarContents[11].sql = "CREATE TABLE IF NOT EXISTS MQL5Calendar(ID INT NOT NULL,EVENTID  INT   NOT NULL,COUNTRY  TEXT   NOT NULL,"
                              "EVENTNAME   TEXT   NOT NULL,EVENTTYPE   TEXT   NOT NULL,EVENTIMPORTANCE   TEXT   NOT NULL,"
                              "EVENTCURRENCY  TEXT   NOT NULL,EVENTCODE   TEXT   NOT NULL,EVENTSECTOR TEXT   NOT NULL,"
                              "EVENTFORECAST  TEXT   NOT NULL,EVENTPREVALUE  TEXT   NOT NULL,EVENTIMPACT TEXT   NOT NULL,"
                              "EVENTFREQUENCY TEXT   NOT NULL,PRIMARY KEY(ID))STRICT;";
   CalendarContents[11].tbl_name="MQL5Calendar";
   CalendarContents[11].type = "table";
   CalendarContents[11].insert = "INSERT INTO 'MQL5Calendar'(ID,EVENTID,COUNTRY,EVENTNAME,EVENTTYPE,EVENTIMPORTANCE,EVENTCURRENCY,EVENTCODE,"
                                 "EVENTSECTOR,EVENTFORECAST,EVENTPREVALUE,EVENTIMPACT,EVENTFREQUENCY) "
                                 "VALUES (%d,%d,'%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s');";

//-- Sql statement for creating the AutoDST table's trigger
   CalendarContents[12].Content = AutoDST_Trigger;
   CalendarContents[12].name = "OnlyOne_AutoDST";
   CalendarContents[12].sql = "CREATE TRIGGER IF NOT EXISTS OnlyOne_AutoDST "
                              "BEFORE INSERT ON AutoDST "
                              "BEGIN "
                              "Delete from AutoDST; "
                              "END;";
   CalendarContents[12].tbl_name="AutoDST";
   CalendarContents[12].type = "trigger";

//-- Sql statement for creating the Record table's trigger
   CalendarContents[13].Content = Record_Trigger;
   CalendarContents[13].name = "OnlyOne_Record";
   CalendarContents[13].sql = "CREATE TRIGGER IF NOT EXISTS OnlyOne_Record "
                              "BEFORE INSERT ON Record "
                              "BEGIN "
                              "Delete from Record; "
                              "END;";
   CalendarContents[13].tbl_name="Record";
   CalendarContents[13].type = "trigger";

//-- initializing properties for the MQL5Calendar table for DB in System Memory
   DBMemory.Content = MQL5Calendar_Table;
   DBMemory.name = "MQL5Calendar";
   DBMemory.sql = "CREATE TABLE IF NOT EXISTS MQL5Calendar(EVENTID  INT   NOT NULL,COUNTRY  TEXT   NOT NULL,"
                  "EVENTNAME   TEXT   NOT NULL,EVENTTYPE   TEXT   NOT NULL,EVENTIMPORTANCE   TEXT   NOT NULL,"
                  "EVENTCURRENCY  TEXT   NOT NULL,EVENTCODE   TEXT   NOT NULL,EVENTSECTOR TEXT   NOT NULL,"
                  "EVENTFORECAST  TEXT   NOT NULL,EVENTPREVALUE  TEXT   NOT NULL,EVENTIMPACT TEXT   NOT NULL,"
                  "EVENTFREQUENCY TEXT   NOT NULL,DST_UK   TEXT   NOT NULL,DST_US   TEXT   NOT NULL,"
                  "DST_AU   TEXT   NOT NULL,DST_NONE   TEXT   NOT NULL)STRICT;";
   DBMemory.tbl_name="MQL5Calendar";
   DBMemory.type = "table";
   DBMemory.insert = "INSERT INTO 'MQL5Calendar'(EVENTID,COUNTRY,EVENTNAME,EVENTTYPE,EVENTIMPORTANCE,EVENTCURRENCY,EVENTCODE,"
                     "EVENTSECTOR,EVENTFORECAST,EVENTPREVALUE,EVENTIMPACT,EVENTFREQUENCY,DST_UK,DST_US,DST_AU,DST_NONE) "
                     "VALUES (%d,'%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s','%s', '%s', '%s');";
  }

//+------------------------------------------------------------------+
//|Destructor                                                        |
//+------------------------------------------------------------------+
CNews::~CNews(void)
  {
   if(FileIsExist(NEWS_TEXT_FILE,FILE_COMMON))//Check if the news database open text file exists
     {
      FileDelete(NEWS_TEXT_FILE,FILE_COMMON);
     }
   DatabaseClose(DBMemoryConnection);//Close DB in memory
  }

//+------------------------------------------------------------------+
//|Gets values from the MQL5 economic Calendar                       |
//+------------------------------------------------------------------+
void CNews::EconomicDetails(Calendar &NewsTime[],datetime date_from=0,datetime date_to=0)
  {
   int Size=0;//to keep track of the size of the events in the NewsTime array
   MqlCalendarCountry countries[];
   string Country_code="";

   for(int i=0,count=CalendarCountries(countries); i<count; i++)
     {
      MqlCalendarValue values[];
      //-- Set End date
      date_to=(date_to==0)?(datetime)(CTime.MonthsS(6)+iTime(Symbol(),PERIOD_D1,0))
              :date_to;//Set end date
      if(CalendarValueHistory(values,date_from,date_to,countries[i].code))
        {
         for(int x=0; x<(int)ArraySize(values); x++)
           {
            if(values[x].time>=date_from)
              {
               MqlCalendarEvent event;
               ulong event_id=values[x].event_id;//Get the event id
               if(CalendarEventById(event_id,event))
                 {
                  ArrayResize(NewsTime,Size+1,Size+2);//Readjust the size of the array to +1 of the array size
                  StringReplace(event.name,"'","");//Removing or replacing single quotes(') from event name with an empty string
                  NewsTime[Size].CountryName = countries[i].name;//storing the country's name from the specific event
                  NewsTime[Size].EventName = event.name;//storing the event's name
                  NewsTime[Size].EventType = EnumToString(event.type);//storing the event type from (ENUM_CALENDAR_EVENT_TYPE) to a string
                  //-- storing the event importance from (ENUM_CALENDAR_EVENT_IMPORTANCE) to a string
                  NewsTime[Size].EventImportance = EnumToString(event.importance);
                  NewsTime[Size].EventId = event.id;//storing the event id
                  NewsTime[Size].EventDate = TimeToString(values[x].time);//storing normal event time
                  NewsTime[Size].EventCurrency = countries[i].currency;//storing event currency
                  NewsTime[Size].EventCode = countries[i].code;//storing event code
                  NewsTime[Size].EventSector = EnumToString(event.sector);//storing event sector from (ENUM_CALENDAR_EVENT_SECTOR) to a string
                  if(values[x].HasForecastValue())//Checks if the event has a forecast value
                    {
                     NewsTime[Size].EventForecast = (string)values[x].forecast_value;//storing the forecast value into a string
                    }
                  else
                    {
                     NewsTime[Size].EventForecast = "None";//storing 'None' as the forecast value
                    }

                  if(values[x].HasPreviousValue())//Checks if the event has a previous value
                    {
                     NewsTime[Size].EventPreval = (string)values[x].prev_value;//storing the previous value into a string
                    }
                  else
                    {
                     NewsTime[Size].EventPreval = "None";//storing 'None' as the previous value
                    }
                  //-- storing the event impact from (ENUM_CALENDAR_EVENT_IMPACT) to a string
                  NewsTime[Size].EventImpact =  EnumToString(values[x].impact_type);
                  //-- storing the event frequency from (ENUM_CALENDAR_EVENT_FREQUENCY) to a string
                  NewsTime[Size].EventFrequency =  EnumToString(event.frequency);
                  Size++;//incrementing the Calendar array NewsTime
                 }
              }
           }
        }
     }
  }

//+------------------------------------------------------------------+
//|Gets values from the MQL5 DB Calendar in Memory                   |
//+------------------------------------------------------------------+
void CNews::EconomicDetailsMemory(Calendar &NewsTime[],datetime date,bool ImpactRequired)
  {
//--- SQL query to retrieve news data for a certain date
   string request_text;
//--- Check if Event impact is required for retrieving news events
   if(ImpactRequired)
     {
      /*
      Within this request we select all the news events that will occur or have occurred in the current day and the next news event
      after the current day where the previous and forecast values aren't 'None'. For each of these events that are identified as
      having a previous and forecast values, we then get the event impact from the same event that has occurred in the past with
      previous,forecast and event impact values where the previous and forecast values are similar, meaning if the current news
      event's previous value is larger than the forecast value this should be the same for the most recent previous event that
      has an event impact within 24 months from the current event. We then use this event impact as an assumption for the current
      event.
      */
      request_text=StringFormat("WITH DAILY_EVENTS AS(SELECT M.EVENTID as 'E_ID',M.COUNTRY,M.EVENTNAME as 'Name',M.EVENTTYPE as"
                                " 'Type',M.EVENTIMPORTANCE as 'Importance',M.%s as 'Time',M.EVENTCURRENCY as 'Currency',M.EVENTCODE"
                                " as 'Code',M.EVENTSECTOR as 'Sector',M.EVENTFORECAST as 'Forecast',M.EVENTPREVALUE as 'PREVALUE',"
                                "M.EVENTFREQUENCY as 'Freq' FROM %s M WHERE DATE(REPLACE(Time,'.','-'))=DATE(REPLACE('%s','.','-'))"
                                " AND (Forecast<>'None' AND Prevalue<>'None')),DAILY_IMPACT AS(SELECT DE.E_ID,DE.COUNTRY,DE.Name,"
                                "DE.Type,DE.Importance,DE.Time,DE.Currency,DE.Code,DE.Sector,DE.Forecast,DE.Prevalue,DE.Freq,"
                                "MC.EVENTIMPACT as 'IMPACT', RANK() OVER(PARTITION BY DE.E_ID,DE.Time ORDER BY MC.%s DESC)DateOrder"
                                " FROM %s MC INNER JOIN DAILY_EVENTS DE on DE.E_ID=MC.EVENTID WHERE DATE(REPLACE(MC.%s,'.','-'))<"
                                "DATE(REPLACE(DE.Time,'.','-')) AND DATE(REPLACE(MC.%s,'.','-'))>=DATE(REPLACE(DE.Time,'.','-'),"
                                "'-24 months') AND (MC.EVENTFORECAST<>'None' AND MC.EVENTPREVALUE<>'None' AND (CASE WHEN Forecast>"
                                "Prevalue THEN 'more' WHEN Forecast<Prevalue THEN 'less' ELSE 'equal' END)=(CASE WHEN MC.EVENTFORECAST"
                                ">MC.EVENTPREVALUE THEN 'more' WHEN MC.EVENTFORECAST<MC.EVENTPREVALUE THEN 'less' ELSE 'equal' END)) "
                                "ORDER BY MC.%s),DAILY_EVENTS_RECORDS AS(SELECT * FROM DAILY_IMPACT WHERE DateOrder=1 ORDER BY Time"
                                " ASC),NEXT_EVENT AS(SELECT M.EVENTID as 'E_ID',M.COUNTRY,M.EVENTNAME as 'Name',M.EVENTTYPE as 'Type',"
                                "M.EVENTIMPORTANCE as 'Importance',M.%s as 'Time',M.EVENTCURRENCY as 'Currency',M.EVENTCODE as 'Code',"
                                "M.EVENTSECTOR as 'Sector',M.EVENTFORECAST as 'Forecast',M.EVENTPREVALUE as 'PREVALUE',M.EVENTFREQUENCY"
                                " as 'Freq' FROM %s M WHERE DATE(REPLACE(Time,'.','-'))>DATE(REPLACE('%s','.','-'))  AND (Forecast<>"
                                "'None' AND Prevalue<>'None' AND DATE(REPLACE(Time,'.','-'))<=DATE(REPLACE('%s','.','-'),'+60 days'))),"
                                "NEXT_IMPACT AS(SELECT NE.E_ID,NE.COUNTRY,NE.Name,NE.Type,NE.Importance,NE.Time,NE.Currency,NE.Code"
                                ",NE.Sector,NE.Forecast,NE.Prevalue,NE.Freq,MC.EVENTIMPACT as 'IMPACT',RANK() OVER(PARTITION BY "
                                "NE.E_ID,NE.Time ORDER BY MC.%s DESC)DateOrder FROM %s MC INNER JOIN NEXT_EVENT NE on NE.E_ID=MC.EVENTID "
                                "WHERE DATE(REPLACE(MC.%s,'.','-'))<DATE(REPLACE(NE.Time,'.','-')) AND DATE(REPLACE(MC.%s,'.','-'))>="
                                "DATE(REPLACE(NE.Time,'.','-'),'-24 months') AND (MC.EVENTFORECAST<>'None' AND MC.EVENTPREVALUE<>'None'"
                                " AND (CASE WHEN Forecast>Prevalue THEN 'more' WHEN Forecast<Prevalue THEN 'less' ELSE 'equal' END)="
                                "(CASE WHEN MC.EVENTFORECAST>MC.EVENTPREVALUE THEN 'more' WHEN MC.EVENTFORECAST<MC.EVENTPREVALUE THEN "
                                "'less' ELSE 'equal' END)) ORDER BY MC.%s),NEXT_EVENT_RECORD AS(SELECT * FROM NEXT_IMPACT WHERE "
                                "DateOrder=1 ORDER BY Time ASC LIMIT 1),ALL_EVENTS AS(SELECT * FROM NEXT_EVENT_RECORD UNION ALL "
                                "SELECT * FROM DAILY_EVENTS_RECORDS)SELECT E_ID,Country,Name,Type,Importance,Time,Currency,Code,"
                                "Sector,Forecast,Prevalue,Impact,Freq FROM ALL_EVENTS GROUP BY Time ORDER BY Time Asc;",
                                EnumToString(MySchedule),DBMemory.name,TimeToString(date),EnumToString(MySchedule),DBMemory.name,
                                EnumToString(MySchedule),EnumToString(MySchedule),EnumToString(MySchedule),EnumToString(MySchedule)
                                ,DBMemory.name,TimeToString(date),TimeToString(date),EnumToString(MySchedule),DBMemory.name,
                                EnumToString(MySchedule),EnumToString(MySchedule),EnumToString(MySchedule));
     }
   else
     {
      /*
      Within this request we select all the news events that will occur or have occurred in the current day and the next news
      event after the current day
      */
      request_text=StringFormat("WITH DAILY_EVENTS AS(SELECT M.EVENTID as 'E_ID',M.COUNTRY,M.EVENTNAME as 'Name',M.EVENTTYPE as"
                                " 'Type',M.EVENTIMPORTANCE as 'Importance',M.%s as 'Time',M.EVENTCURRENCY as 'Currency',M.EVENTCODE"
                                " as 'Code',M.EVENTSECTOR as 'Sector',M.EVENTFORECAST as 'Forecast',M.EVENTPREVALUE as 'PREVALUE'"
                                ",M.EVENTFREQUENCY as 'Freq',M.EVENTIMPACT as 'Impact' FROM %s M WHERE DATE(REPLACE(Time,'.','-'))"
                                "=DATE(REPLACE('%s','.','-'))),DAILY_EVENTS_RECORDS AS(SELECT * FROM DAILY_EVENTS ORDER BY Time ASC)"
                                ",NEXT_EVENT AS(SELECT M.EVENTID as 'E_ID',M.COUNTRY,M.EVENTNAME as 'Name',M.EVENTTYPE as 'Type',"
                                "M.EVENTIMPORTANCE as 'Importance',M.%s as 'Time',M.EVENTCURRENCY as 'Currency',M.EVENTCODE as "
                                "'Code',M.EVENTSECTOR as 'Sector',M.EVENTFORECAST as 'Forecast',M.EVENTPREVALUE as 'PREVALUE',"
                                "M.EVENTFREQUENCY as 'Freq',M.EVENTIMPACT as 'Impact' FROM %s M WHERE DATE(REPLACE(Time,'.','-'))"
                                ">DATE(REPLACE('%s','.','-')) AND (DATE(REPLACE(Time,'.','-'))<=DATE(REPLACE('%s','.','-'),"
                                "'+60 days'))),NEXT_EVENT_RECORD AS(SELECT * FROM NEXT_EVENT ORDER BY Time ASC LIMIT 1),"
                                "ALL_EVENTS AS(SELECT * FROM NEXT_EVENT_RECORD UNION ALL SELECT * FROM "
                                "DAILY_EVENTS_RECORDS)SELECT * FROM ALL_EVENTS GROUP BY Time ORDER BY Time Asc;",
                                EnumToString(MySchedule),DBMemory.name,TimeToString(date),EnumToString(MySchedule),DBMemory.name,
                                TimeToString(date),TimeToString(date));
     }

   int request=DatabasePrepare(DBMemoryConnection,request_text);//Creates a handle of a request, which can then be executed using DatabaseRead()
   if(request==INVALID_HANDLE)//Checks if the request failed to be completed
     {
      Print("DB: ",NEWS_DATABASE_MEMORY, " request failed with code ", GetLastError());
      PrintFormat(request_text);
     }
//--- Calendar structure variable
   Calendar ReadDB_Data;
//--- Remove any data in the array
   ArrayRemove(NewsTime,0,WHOLE_ARRAY);
   for(int i=0; DatabaseReadBind(request,ReadDB_Data); i++)//Will read all the results from the sql query/request
     {
      //--- Resize array NewsTime
      ArrayResize(NewsTime,i+1,i+2);
      //--- Assign calendar structure values into NewsTime array index
      NewsTime[i]  = ReadDB_Data;
     }
//--- Removes a request created in DatabasePrepare()
   DatabaseFinalize(request);
  }

//+------------------------------------------------------------------+
//|Will update UpcomingNews structure variable with the next news    |
//|event data                                                        |
//+------------------------------------------------------------------+
void CNews::EconomicNextEvent()
  {
//--- Declare unassigned Calendar structure variable Next
   Calendar Next;
//--- assign empty values to Calendar structure variable UpcomingNews
   UpcomingNews = Next;
//--- assign default date
   datetime NextEvent=0;
//--- Iterate through CalendarArray to retrieve news events
   for(uint i=0;i<CalendarArray.Size();i++)
     {
      //--- Check for next earliest news event from CalendarArray
      if((NextEvent==0)||(TimeTradeServer()<datetime(CalendarArray[i].EventDate)
                          &&NextEvent>datetime(CalendarArray[i].EventDate))||(NextEvent<TimeTradeServer()))
        {
         //--- assign values from the CalendarArray
         NextEvent = datetime(CalendarArray[i].EventDate);
         Next = CalendarArray[i];
        }
     }
//--- assign the next news event data into UpcomingNews variable
   UpcomingNews = Next;
  }

//+------------------------------------------------------------------+
//|Checks if News is event is about to occur or is occurring         |
//+------------------------------------------------------------------+
bool CNews::isEvent(uint SecondsPreEvent,string &Name,string &Importance,string &Code)
  {
//--- assign default value
   Name=NULL;
//--- Iterate through CalendarArray
   for(uint i=0;i<CalendarArray.Size();i++)
     {
      //--- Check if news event is within a timespan
      if(CTime.TimeIsInRange(CTime.TimeMinusOffset(datetime(CalendarArray[i].EventDate),SecondsPreEvent),
                             CTime.TimePlusOffset(datetime(CalendarArray[i].EventDate),59)))
        {
         //--- assign appropriate CalendarArray values
         Name=CalendarArray[i].EventName;
         Importance=CalendarArray[i].EventImportance;
         Code=CalendarArray[i].EventCode;
         //--- news event is currently within the timespan
         return true;
        }
     }
//--- no news event is within the current timespan.
   return false;
  }

//+------------------------------------------------------------------+
//|Checks if the main Calendar database needs an update or not       |
//+------------------------------------------------------------------+
bool CNews::UpdateRecords()
  {
//initialize variable to true
   bool perform_update=true;
//--- open/create
//-- try to open database Calendar
   int db=DatabaseOpen(NEWS_DATABASE_FILE, DATABASE_OPEN_READWRITE | DATABASE_OPEN_CREATE| DATABASE_OPEN_COMMON);
   if(db==INVALID_HANDLE)//Checks if the database was able to be opened
     {
      //if opening the database failed
      if(!FileIsExist(NEWS_DATABASE_FILE,FILE_COMMON))//Checks if the database Calendar exists in the common folder
        {
         return perform_update;//Returns true when the database was failed to be opened and the file doesn't exist in the common folder
        }
     }

   int MasterRequest = DatabasePrepare(db,"select * from sqlite_master where type<>'index';");
   if(MasterRequest==INVALID_HANDLE)
     {
      Print("DB: ",NEWS_DATABASE_FILE, " request failed with code ", GetLastError());
      DatabaseFinalize(MasterRequest);//Removes a request created in DatabasePrepare()
      DatabaseClose(db);
      return perform_update;
     }
   else
     {
      SQLiteMaster ReadContents;
      //Assigning values from the sql query into DBContents array
      for(int i=0; DatabaseReadBind(MasterRequest,ReadContents); i++)
        {
         ArrayResize(DBContents,i+1,i+2);
         DBContents[i]  = ReadContents ;
         /*Check if the end of the sql string has a character ';' if not add this character to the string*/
         DBContents[i].sql = (StringFind(ReadContents.sql,";",StringLen(ReadContents.sql)-1)==
                              (StringLen(ReadContents.sql)-1))?ReadContents.sql:ReadContents.sql+";";;
        }

      uint contents_exists = 0;
      for(uint i=0;i<DBContents.Size();i++)
        {
         bool isCalendarContents = false;
         for(uint x=0;x<CalendarContents.Size();x++)
           {
            /*Store Sql query from CalendarContents without string ' IF NOT EXISTS'*/
            string CalendarSql=CalendarContents[x].sql;
            StringReplace(CalendarSql," IF NOT EXISTS","");
            //-- Check if the Db object is in our list
            if(DBContents[i].name==CalendarContents[x].name&&
               (DBContents[i].sql==CalendarSql||
                DBContents[i].sql==CalendarContents[x].sql)&&
               CalendarContents[x].type==DBContents[i].type&&
               CalendarContents[x].tbl_name==DBContents[i].tbl_name)
              {
               contents_exists++;
               isCalendarContents = true;
              }
           }
         if(!isCalendarContents)
           {
            //-- Print DBcontent's name if it does not match with CalendarContents
            PrintFormat("DBContent: %s is not needed!",DBContents[i].name);
            //-- We will drop the table if it is not neccessary
            DatabaseExecute(db,StringFormat(DropRequest,DBContents[i].type,DBContents[i].name));
            Print("Attempting To Clean Database...");
           }
        }
      /*If not all the CalendarContents exist in the Calendar Database before an update */
      if(contents_exists!=CalendarContents.Size())
        {
         DatabaseFinalize(MasterRequest);//Removes a request created in DatabasePrepare()
         DatabaseClose(db);
         return perform_update;
        }
     }

//-- Sql query to determine the lastest or maximum date recorded
   /* If the last recorded date data in the 'Record' table is not equal to the current day, perform an update! */
   string request_text=StringFormat("SELECT Date FROM %s where Date=Date(REPLACE('%s','.','-'));",
                                    CalendarStruct(Record_Table).name,TimeToString(TimeTradeServer()));
   int request=DatabasePrepare(db,request_text);//Creates a handle of a request, which can then be executed using DatabaseRead()
   if(request==INVALID_HANDLE)//Checks if the request failed to be completed
     {
      Print("DB: ",NEWS_DATABASE_FILE, " request failed with code ", GetLastError());
      DatabaseClose(db);
      return perform_update;
     }

   if(DatabaseRead(request))//Will be true if there are results from the sql query/request
     {
      DatabaseFinalize(request);//Removes a request created in DatabasePrepare()
      Calendar TodaysNews[];
      datetime Today = CTime.Time(TimeTradeServer(),0,0,0);
      EconomicDetails(TodaysNews,Today,Today+CTime.DaysS());

      for(uint i=0;i<TodaysNews.Size();i++)
        {
         //--- request to retrieve the news events that match the news events in the TodaysNews array
         request_text=StringFormat("SELECT M.EventID FROM %s M Inner Join %s T on T.ID=M.ID where "
                                   "Replace(T.DST_NONE,'.','-')=Replace('%s','.','-') and M.EventID=%d;",
                                   CalendarStruct(MQL5Calendar_Table).name,CalendarStruct(TimeSchedule_Table).name,
                                   TodaysNews[i].EventDate,TodaysNews[i].EventId);
         request=DatabasePrepare(db,request_text);//Creates a handle of a request, which can then be executed using DatabaseRead()
         if(request==INVALID_HANDLE)//Checks if the request failed to be completed
           {
            Print("DB: ",NEWS_DATABASE_FILE, " request failed with code ", GetLastError());
            DatabaseFinalize(request);
            DatabaseClose(db);
            return perform_update;
           }
         //PrintFormat(request_text);
         if(!DatabaseRead(request))//Will be true if there are results from the sql query/request
           {
            DatabaseFinalize(request);
            DatabaseClose(db);
            return perform_update;
           }
         DatabaseFinalize(request);
        }
      DatabaseClose(db);//Closes the database
      perform_update=false;
      return perform_update;
     }
   else
     {
      DatabaseFinalize(request);//Removes a request created in DatabasePrepare()
      DatabaseClose(db);//Closes the database
      return perform_update;
     }
  }

//+------------------------------------------------------------------+
//|Creates the Calendar database for a specific Broker               |
//+------------------------------------------------------------------+
void CNews::CreateEconomicDatabase()
  {
   if(FileIsExist(NEWS_DATABASE_FILE,FILE_COMMON))//Check if the database exists
     {
      if(!UpdateRecords())//Check if the database is up to date
        {
         return;//will terminate execution of the rest of the code below
        }
     }
   if(FileIsExist(NEWS_TEXT_FILE,FILE_COMMON))//Check if the database is open
     {
      return;//will terminate execution of the rest of the code below
     }

   Calendar Evalues[];//Creating a Calendar array variable
   bool failed=false,tableExists=false;
   int file=INVALID_HANDLE;
//--- open/create the database 'Calendar'
//-- will try to open/create in the common folder
   int db=DatabaseOpen(NEWS_DATABASE_FILE, DATABASE_OPEN_READWRITE | DATABASE_OPEN_CREATE| DATABASE_OPEN_COMMON);
   if(db==INVALID_HANDLE)//Checks if the database 'Calendar' failed to open/create
     {
      Print("DB: ",NEWS_DATABASE_FILE, " open failed with code ", GetLastError());
      return;//will terminate execution of the rest of the code below
     }
   else
     {
      //-- try to create a text file 'NewsDatabaseOpen' in common folder
      file=FileOpen(NEWS_TEXT_FILE,FILE_WRITE|FILE_ANSI|FILE_TXT|FILE_COMMON);
      if(file==INVALID_HANDLE)
        {
         DatabaseClose(db);//Closes the database 'Calendar' if the News text file failed to be created
         return;//will terminate execution of the rest of the code below
        }
     }

   DatabaseTransactionBegin(db);//Starts transaction execution
   Print("Please wait...");

//-- attempt to create the MQL5Calendar and TimeSchedule tables
   if(!CreateCalendarTable(db,tableExists)||!CreateTimeTable(db,tableExists))
     {
      FileClose(file);//Closing the file 'NewsDatabaseOpen.txt'
      FileDelete(NEWS_TEXT_FILE,FILE_COMMON);//Deleting the file 'NewsDatabaseOpen.txt'
      return;//will terminate execution of the rest of the code below
     }

   EconomicDetails(Evalues);//Retrieving the data from the Economic Calendar
   if(tableExists)//Checks if there is an existing table within the Calendar Database
     {
      //if there is an existing table we will notify the user that we are updating the table.
      PrintFormat("Updating %s",NEWS_DATABASE_FILE);
     }
   else
     {
      //if there isn't an existing table we will notify the user that we about to create one
      PrintFormat("Creating %s",NEWS_DATABASE_FILE);
     }

//-- attempt to insert economic event data into the calendar tables
   if(!InsertIntoTables(db,Evalues))
     {
      //-- Will assign true if inserting economic vaules failed in the MQL5Calendar and TimeSchedule tables
      failed=true;
     }

   if(failed)
     {
      //--- roll back all transactions and unlock the database
      DatabaseTransactionRollback(db);
      PrintFormat("%s: DatabaseExecute() failed with code %d", __FUNCTION__, GetLastError());
      FileClose(file);//Close the text file 'NEWS_TEXT_FILE'
      FileDelete(NEWS_TEXT_FILE,FILE_COMMON);//Delete the text file, as we are reverted/rolled-back the database
      ArrayRemove(Evalues,0,WHOLE_ARRAY);//Removes the values in the array
      DatabaseClose(db);//Close the database
      return;
     }
   else
     {
      if(tableExists)
        {
         //Let the user/trader know that the database was updated
         PrintFormat("%s Updated",NEWS_DATABASE_FILE);
        }
      else
        {
         //Let the user/trader know that the database was created
         PrintFormat("%s Created",NEWS_DATABASE_FILE);
        }
      CreateCalendarViews(db);//Will create Calendar views
      CreateRecordTable(db);//Will create the 'Record' table and insert the  current time
      CreateAutoDST(db);//Will create the 'AutoDST' table and insert the broker's DST schedule
      //--- all transactions have been performed successfully - record changes and unlock the database
      DatabaseTransactionCommit(db);
      DatabaseClose(db);//Close the database
     }
   FileClose(file);//Close the text file 'NEWS_TEXT_FILE'
   FileDelete(NEWS_TEXT_FILE,FILE_COMMON);//Delete the text file, as we are about to close the database
   ArrayRemove(Evalues,0,WHOLE_ARRAY);//Removes the values in the array
  }

//+------------------------------------------------------------------+
//|Create calendar database in memory                                |
//+------------------------------------------------------------------+
void CNews::CreateEconomicDatabaseMemory()
  {
//--- Open/create the database in memory
   DBMemoryConnection=DatabaseOpen(NEWS_DATABASE_MEMORY,DATABASE_OPEN_MEMORY);
   if(DBMemoryConnection==INVALID_HANDLE)//Checks if the database failed to open/create
     {
      Print("DB: ",NEWS_DATABASE_MEMORY, " open failed with code ", GetLastError());
      return;//will terminate execution of the rest of the code below
     }
//--- Drop the table if it already exists
   DatabaseExecute(DBMemoryConnection,StringFormat("Drop table IF EXISTS %s",DBMemory.name));
//--- Attempt to create the table
   if(!DatabaseExecute(DBMemoryConnection,DBMemory.sql))
     {
      Print("DB: create the Calendar table failed with code ", GetLastError());
      return;
     }
//--- Check if the table exists
   if(DatabaseTableExists(DBMemoryConnection,DBMemory.tbl_name))
     {
      //--- Get all news data and time from the database in storage
      GetCalendar(DB_Data);
      //--- Insert all the news data and times into the table
      for(uint i=0;i<DB_Data.Size();i++)
        {
         string request_text=StringFormat(DBMemory.insert,DB_Data[i].EventId,DB_Data[i].Country,
                                          DB_Data[i].EventName,DB_Data[i].EventType,DB_Data[i].EventImportance,
                                          DB_Data[i].EventCurrency,DB_Data[i].EventCode,DB_Data[i].EventSector,
                                          DB_Data[i].EventForecast,DB_Data[i].EventPreval,DB_Data[i].EventImpact,
                                          DB_Data[i].EventFrequency,DB_Data[i].DST_UK,DB_Data[i].DST_US,
                                          DB_Data[i].DST_AU,DB_Data[i].DST_NONE);

         if(!DatabaseExecute(DBMemoryConnection, request_text))//Will attempt to run this sql request/query
           {
            //--- failed to run sql request/query
            Print(GetLastError());
            PrintFormat(request_text);
            return;
           }
        }
     }
//--- Remove all data from the array
   ArrayRemove(DB_Data,0,WHOLE_ARRAY);
//--- Assign the DST schedule
   MySchedule = (MQLInfoInteger(MQL_TESTER))?(MyDST==AutoDst_Selection)?GetAutoDST():MySchedule:DST_NONE;
  }
//+------------------------------------------------------------------+
//|Function for creating a table in a database                       |
//+------------------------------------------------------------------+
bool CNews::CreateCalendarTable(int db,bool &tableExists)
  {
//-- Checks if a table 'MQL5Calendar' exists
   if(DatabaseTableExists(db,CalendarStruct(MQL5Calendar_Table).name))
     {
      tableExists=true;//Assigns true to tableExists variable
      //-- Checks if a table 'TimeSchedule' exists in the database 'Calendar'
      if(DatabaseTableExists(db,CalendarStruct(TimeSchedule_Table).name))
        {
         //-- We will drop the table if the table already exists
         if(!DatabaseExecute(db,StringFormat("Drop Table %s",CalendarStruct(TimeSchedule_Table).name)))
           {
            //If the table failed to be dropped/deleted
            PrintFormat("Failed to drop table %s with code %d",CalendarStruct(TimeSchedule_Table).name,GetLastError());
            DatabaseClose(db);//Close the database
            return false;//will terminate execution of the rest of the code below and return false, when the table cannot be dropped
           }
        }
      //--We will drop the table if the table already exists
      if(!DatabaseExecute(db,StringFormat("Drop Table %s",CalendarStruct(MQL5Calendar_Table).name)))
        {
         //If the table failed to be dropped/deleted
         PrintFormat("Failed to drop table %s with code %d",CalendarStruct(MQL5Calendar_Table).name,GetLastError());
         DatabaseClose(db);//Close the database
         return false;//will terminate execution of the rest of the code below and return false, when the table cannot be dropped
        }
     }
//-- If the database table 'MQL5Calendar' doesn't exist
   if(!DatabaseTableExists(db,CalendarStruct(MQL5Calendar_Table).name))
     {
      //--- create the table 'MQL5Calendar'
      if(!DatabaseExecute(db,CalendarStruct(MQL5Calendar_Table).sql))//Checks if the table was successfully created
        {
         Print("DB: create the Calendar table failed with code ", GetLastError());
         DatabaseClose(db);//Close the database
         return false;//Function returns false if creating the table failed
        }
     }
   return true;//Function returns true if creating the table was successful
  }

//+------------------------------------------------------------------+
//|Function for creating a table in a database                       |
//+------------------------------------------------------------------+
bool CNews::CreateTimeTable(int db,bool &tableExists)
  {
//-- If the database table 'TimeSchedule' doesn't exist
   if(!DatabaseTableExists(db,CalendarStruct(TimeSchedule_Table).name))
     {
      //--- create the table 'TimeSchedule'
      if(!DatabaseExecute(db,CalendarStruct(TimeSchedule_Table).sql))//Checks if the table was successfully created
        {
         Print("DB: create the Calendar table failed with code ", GetLastError());
         DatabaseClose(db);//Close the database
         return false;//Function returns false if creating the table failed
        }
     }
   return true;//Function returns true if creating the table was successful
  }

//+------------------------------------------------------------------+
//|Function for creating views in a database                         |
//+------------------------------------------------------------------+
void CNews::CreateCalendarViews(int db)
  {
   for(uint i=1;i<=8;i++)
     {
      if(!DatabaseExecute(db,CalendarStruct((CalendarComponents)i).sql))//Checks if the view was successfully created
        {
         Print("DB: create the Calendar view failed with code ", GetLastError());
         PrintFormat(CalendarStruct((CalendarComponents)i).sql);
        }
     }
  }

//+------------------------------------------------------------------+
//|Function for inserting Economic Data in to a database's table     |
//+------------------------------------------------------------------+
bool CNews::InsertIntoTables(int db,Calendar &Evalues[])
  {
   for(uint i=0; i<Evalues.Size(); i++)//Looping through all the Economic Events
     {
      string request_insert_into_calendar =
         StringFormat(CalendarStruct(MQL5Calendar_Table).insert,
                      i,
                      Evalues[i].EventId,
                      Evalues[i].CountryName,
                      Evalues[i].EventName,
                      Evalues[i].EventType,
                      Evalues[i].EventImportance,
                      Evalues[i].EventCurrency,
                      Evalues[i].EventCode,
                      Evalues[i].EventSector,
                      Evalues[i].EventForecast,
                      Evalues[i].EventPreval,
                      Evalues[i].EventImpact,
                      Evalues[i].EventFrequency);//Inserting all the columns for each event record
      if(DatabaseExecute(db,request_insert_into_calendar))//Check if insert query into calendar was successful
        {
         string request_insert_into_time =
            StringFormat(CalendarStruct(TimeSchedule_Table).insert,
                         i,
                         //-- Economic EventDate adjusted for UK DST(Daylight Savings CTime)
                         Savings_UK.adjustDaylightSavings(StringToTime(Evalues[i].EventDate)),
                         //-- Economic EventDate adjusted for US DST(Daylight Savings CTime)
                         Savings_US.adjustDaylightSavings(StringToTime(Evalues[i].EventDate)),
                         //-- Economic EventDate adjusted for AU DST(Daylight Savings CTime)
                         Savings_AU.adjustDaylightSavings(StringToTime(Evalues[i].EventDate)),
                         Evalues[i].EventDate//normal Economic EventDate
                        );//Inserting all the columns for each event record
         if(!DatabaseExecute(db,request_insert_into_time))
           {
            Print(GetLastError());
            //-- Will print the sql query to check for any errors or possible defaults in the query/request
            Print(request_insert_into_time);
            return false;//Will end the loop and return false, as values failed to be inserted into the table
           }
        }
      else
        {
         Print(GetLastError());
         //-- Will print the sql query to check for any errors or possible defaults in the query/request
         Print(request_insert_into_calendar);
         return false;//Will end the loop and return false, as values failed to be inserted into the table
        }
     }
   return true;//Will return true, all values were inserted into the table successfully
  }

//+------------------------------------------------------------------+
//|Creates a table to store the record of when last the Calendar     |
//|database was updated/created                                      |
//+------------------------------------------------------------------+
void CNews::CreateRecordTable(int db)
  {
   bool failed=false;
   if(!DatabaseTableExists(db,CalendarStruct(Record_Table).name))//Checks if the table 'Record' exists in the databse 'Calendar'
     {
      //--- create the table
      if(!DatabaseExecute(db,CalendarStruct(Record_Table).sql))//Will attempt to create the table 'Record'
        {
         Print("DB: create the Records table failed with code ", GetLastError());
         DatabaseClose(db);//Close the database
         return;//Exits the function if creating the table failed
        }
      else//If Table was created Successfully then Create Trigger
        {
         DatabaseExecute(db,CalendarStruct(Record_Trigger).sql);
        }
     }
   else
     {
      DatabaseExecute(db,CalendarStruct(Record_Trigger).sql);
     }
//Sql query/request to insert the current time into the 'Date' column in the table 'Record'
   string request_text=StringFormat(CalendarStruct(Record_Table).insert,TimeToString(TimeTradeServer()));
   if(!DatabaseExecute(db, request_text))//Will attempt to run this sql request/query
     {
      Print(GetLastError());
      PrintFormat(CalendarStruct(Record_Table).insert,TimeToString(TimeTradeServer()));
      failed=true;//assign true if the request failed
     }
   if(failed)
     {
      //--- roll back all transactions and unlock the database
      DatabaseTransactionRollback(db);
      PrintFormat("%s: DatabaseExecute() failed with code %d", __FUNCTION__, GetLastError());
     }
  }

//+------------------------------------------------------------------+
//|Function for creating and inserting Recommend DST for the Broker  |
//|into a table                                                      |
//+------------------------------------------------------------------+
void CNews::CreateAutoDST(int db)
  {
   bool failed=false;//boolean variable
   if(!AutoDetectDST(DSTType))//Check if AutoDetectDST went through all the right procedures
     {
      return;//will terminate execution of the rest of the code below
     }

   if(!DatabaseTableExists(db,CalendarStruct(AutoDST_Table).name))//Checks if the table 'AutoDST' exists in the databse 'Calendar'
     {
      //--- create the table AutoDST
      if(!DatabaseExecute(db,CalendarStruct(AutoDST_Table).sql))//Will attempt to create the table 'AutoDST'
        {
         Print("DB: create the AutoDST table failed with code ", GetLastError());
         DatabaseClose(db);//Close the database
         return;//Exits the function if creating the table failed
        }
      else//If Table was created Successfully then Create Trigger
        {
         DatabaseExecute(db,CalendarStruct(AutoDST_Trigger).sql);
        }
     }
   else
     {
      //Create trigger if AutoDST table exists
      DatabaseExecute(db,CalendarStruct(AutoDST_Trigger).sql);
     }
//Sql query/request to insert the recommend DST for the Broker using the DSTType variable to determine which string data to insert
   string request_text=StringFormat(CalendarStruct(AutoDST_Table).insert,EnumToString(DSTType));
   if(!DatabaseExecute(db, request_text))//Will attempt to run this sql request/query
     {
      Print(GetLastError());
      PrintFormat(CalendarStruct(AutoDST_Table).insert,EnumToString(DSTType));//Will print the sql query if failed
      failed=true;//assign true if the request failed
     }
   if(failed)
     {
      //--- roll back all transactions and unlock the database
      DatabaseTransactionRollback(db);
      PrintFormat("%s: DatabaseExecute() failed with code %d", __FUNCTION__, GetLastError());
     }
  }

//+------------------------------------------------------------------+
//|Gets the latest/newest date in the Calendar database              |
//+------------------------------------------------------------------+
datetime CNews::GetLatestNewsDate()
  {
//--- open the database 'Calendar' in the common folder
   int db=DatabaseOpen(NEWS_DATABASE_FILE, DATABASE_OPEN_READONLY|DATABASE_OPEN_COMMON);

   if(db==INVALID_HANDLE)//Checks if 'Calendar' failed to be opened
     {
      if(!FileIsExist(NEWS_DATABASE_FILE,FILE_COMMON))//Checks if 'Calendar' database exists
        {
         Print("Could not find Database!");
         return 0;//Will return the earliest date which is 1970.01.01 00:00:00
        }
     }
   string latest_record="1970.01.01";//string variable with the first/earliest possible date in MQL5
//Sql query to determine the lastest or maximum recorded time from which the database was updated.
   string request_text="SELECT REPLACE(Date,'-','.') FROM 'Record'";
   int request=DatabasePrepare(db,request_text);
   if(request==INVALID_HANDLE)
     {
      Print("DB: ",NEWS_DATABASE_FILE, " request failed with code ", GetLastError());
      DatabaseClose(db);//Close Database
      return 0;
     }
   if(DatabaseRead(request))//Will read the one record in the 'Record' table
     {
      //-- Will assign the first column(column 0) value to the variable 'latest_record'
      if(!DatabaseColumnText(request,0,latest_record))
        {
         Print("DatabaseRead() failed with code ", GetLastError());
         DatabaseFinalize(request);//Finalize request
         DatabaseClose(db);//Closes the database 'Calendar'
         return D'1970.01.01';//Will end the for loop and will return the earliest date which is 1970.01.01 00:00:00
        }
     }
   DatabaseFinalize(request);
   DatabaseClose(db);//Closes the database 'Calendar'
   return (datetime)latest_record;//Returns the string latest_record converted to datetime
  }

//+------------------------------------------------------------------+
//|Function will determine Broker DST                                |
//+------------------------------------------------------------------+
bool CNews::AutoDetectDST(DST_type &dstType)
  {
   MqlCalendarValue values[];//Single array of MqlCalendarValue type
   string eventtime[];//Single string array variable to store NFP(Nonfarm Payrolls) dates for the 'United States' from the previous year
//-- Will store the previous year into an integer
   int lastyear = CTime.ReturnYear(CTime.TimeMinusOffset(iTime(Symbol(),PERIOD_CURRENT,0),CTime.YearsS()));
//-- Will store the start date for the previous year
   datetime lastyearstart = StringToTime(StringFormat("%s.01.01 00:00:00",(string)lastyear));
//-- Will store the end date for the previous year
   datetime lastyearend = StringToTime(StringFormat("%s.12.31 23:59:59",(string)lastyear));
//-- Getting last year's calendar values for CountryCode = 'US'
   if(CalendarValueHistory(values,lastyearstart,lastyearend,"US"))
     {
      for(int x=0; x<(int)ArraySize(values); x++)
        {
         if(values[x].event_id==840030016)//Get only NFP Event Dates
           {
            ArrayResize(eventtime,eventtime.Size()+1,eventtime.Size()+2);//Increasing the size of eventtime array by 1
            eventtime[eventtime.Size()-1] = TimeToString(values[x].time);//Storing the dates in an array of type string
           }
        }
     }
//-- datetime variables to store the broker's timezone shift(change)
   datetime ShiftStart=D'1970.01.01 00:00:00',ShiftEnd=D'1970.01.01 00:00:00';
   string   EURUSD="";//String variables declarations for working with EURUSD
   bool     EurusdIsFound=false;//Boolean variables declarations for working with EURUSD
   for(int i=0;i<SymbolsTotal(true);i++)//Will loop through all the Symbols inside the Market Watch
     {
      string SymName = SymbolName(i,true);//Assign the Symbol Name of index 'i' from the list of Symbols inside the Market Watch
      //-- Check if the Symbol outside the Market Watch has a SYMBOL_CURRENCY_BASE of EUR
      //-- and a SYMBOL_CURRENCY_PROFIT of USD, and this Symbol is not a Custom Symbol(Is not from the broker)
      if(((CSymbol.CurrencyBase(SymName)=="EUR"&&CSymbol.CurrencyProfit(SymName)=="USD")||
          (StringFind(SymName,"EUR")>-1&&CSymbol.CurrencyProfit(SymName)=="USD"))&&
         !CSymbol.Custom(SymName))
        {
         EURUSD = SymName;//Assigning the name of the EURUSD Symbol found inside the Market Watch
         EurusdIsFound = true;//EURUSD Symbol was found in the Trading Terminal for your Broker
         break;//Will end the for loop
        }
     }
   if(!EurusdIsFound)//Check if EURUSD Symbol was already Found in the Market Watch
     {
      for(int i=0; i<SymbolsTotal(false); i++)//Will loop through all the available Symbols outside the Market Watch
        {
         string SymName = SymbolName(i,false);//Assign the Symbol Name of index 'i' from the list of Symbols outside the Market Watch
         //-- Check if the Symbol outside the Market Watch has a SYMBOL_CURRENCY_BASE of EUR
         //-- and a SYMBOL_CURRENCY_PROFIT of USD, and this Symbol is not a Custom Symbol(Is not from the broker)
         if(((CSymbol.CurrencyBase(SymName)=="EUR"&&CSymbol.CurrencyProfit(SymName)=="USD")||
             (StringFind(SymName,"EUR")>-1&&CSymbol.CurrencyProfit(SymName)=="USD"))
            &&!CSymbol.Custom(SymName))
           {
            EURUSD = SymName;//Assigning the name of the EURUSD Symbol found outside the Market Watch
            EurusdIsFound = true;//EURUSD Symbol was found in the Trading Terminal for your Broker
            break;//Will end the for loop
           }
        }
     }
   if(!EurusdIsFound)//Check if EURUSD Symbol was Found in the Trading Terminal for your Broker
     {
      Print("Cannot Find EURUSD!");
      Print("Cannot Create Database!");
      Print("Server DST Cannot be Detected!");
      dstType = DST_NONE;//Assigning enumeration value DST_NONE, Broker has no DST(Daylight Savings CTime)
      return false;//Returning False, Broker's DST schedule was not found
     }

   struct DST
     {
      bool           result;
      datetime       date;
     } previousresult,currentresult;

   bool timeIsShifted;//Boolean variable declaration will be used to determine if the broker changes it's timezone
   for(uint i=0;i<eventtime.Size();i++)
     {
      //-- Store the result of if the eventdate is the larger candlestick
      currentresult.result = IsLargerThanPreviousAndNext((datetime)eventtime[i],CTime.HoursS(),EURUSD);
      currentresult.date = (datetime)eventtime[i];//Store the eventdate from eventtime[i]
      //-- Check if there is a difference between the previous result and the current result
      timeIsShifted = ((currentresult.result!=previousresult.result&&i>0)?true:false);

      //-- Check if the Larger candle has shifted from the previous event date to the current event date in eventtime[i] array
      if(timeIsShifted)
        {
         if(ShiftStart==D'1970.01.01 00:00:00')//Check if the ShiftStart variable has not been assigned a relevant value yet
           {
            ShiftStart=currentresult.date;//Store the eventdate for when the timeshift began
           }
         ShiftEnd=previousresult.date;//Store the eventdate timeshift
        }
      previousresult.result = currentresult.result;//Store the previous result of if the eventdate is the larger candlestick
      previousresult.date = currentresult.date;//Store the eventdate from eventtime[i]
     }
//-- Check if the ShiftStart variable has not been assigned a relevant value and the eventdates are more than zero
   if(ShiftStart==D'1970.01.01 00:00:00'&&eventtime.Size()>0)
     {
      Print("Broker ServerTime unchanged!");
      dstType = DST_NONE;//Assigning enumeration value DST_NONE, Broker has no DST(Daylight Savings CTime)
      return true;//Returning True, Broker's DST schedule was found successfully
     }

   datetime DaylightStart,DaylightEnd;//Datetime variables declarations for start and end dates for DaylightSavings
   if(Savings_AU.DaylightSavings(lastyear,DaylightStart,DaylightEnd))
     {
      if(CTime.DateIsInRange(DaylightStart,DaylightEnd,ShiftStart,ShiftEnd))
        {
         Print("Broker ServerTime Adjusted For AU DST");
         dstType = DST_AU;//Assigning enumeration value AU_DST, Broker has AU DST(Daylight Savings CTime)
         return true;//Returning True, Broker's DST schedule was found successfully
        }
     }
   else
     {
      Print("Something went wrong!");
      Print("Cannot Find Daylight-Savings Date For AU");
      PrintFormat("Year: %d Cannot Be Found!",lastyear);
      dstType = DST_NONE;//Assigning enumeration value DST_NONE, Broker has no DST(Daylight Savings CTime)
      return false;//Returning False, Broker's DST schedule was not found
     }

   if(Savings_UK.DaylightSavings(lastyear,DaylightStart,DaylightEnd))
     {
      if(CTime.DateIsInRange(DaylightStart,DaylightEnd,ShiftStart,ShiftEnd))
        {
         Print("Broker ServerTime Adjusted For UK DST");
         dstType = DST_UK;//Assigning enumeration value UK_DST, Broker has UK/EU DST(Daylight Savings CTime)
         return true;//Returning True, Broker's DST schedule was found successfully
        }
     }
   else
     {
      Print("Something went wrong!");
      Print("Cannot Find Daylight-Savings Date For UK");
      PrintFormat("Year: %d Cannot Be Found!",lastyear);
      dstType = DST_NONE;//Assigning enumeration value DST_NONE, Broker has no DST(Daylight Savings CTime)
      return false;//Returning False, Broker's DST schedule was not found
     }

   if(Savings_US.DaylightSavings(lastyear,DaylightStart,DaylightEnd))
     {
      if(CTime.DateIsInRange(DaylightStart,DaylightEnd,ShiftStart,ShiftEnd))
        {
         Print("Broker ServerTime Adjusted For US DST");
         dstType = DST_US;//Assigning enumeration value US_DST, Broker has US DST(Daylight Savings CTime)
         return true;//Returning True, Broker's DST schedule was found successfully
        }
     }
   else
     {
      Print("Something went wrong!");
      Print("Cannot Find Daylight-Savings Date For US");
      PrintFormat("Year: %d Cannot Be Found!",lastyear);
      dstType = DST_NONE;//Assigning enumeration value DST_NONE, Broker has no DST(Daylight Savings CTime)
      return false;//Returning False, Broker's DST schedule was not found
     }
   Print("Cannot Detect Broker ServerTime Configuration!");
   dstType = DST_NONE;//Assigning enumeration value DST_NONE, Broker has no DST(Daylight Savings CTime)
   return false;//Returning False, Broker's DST schedule was not found
  }
//+------------------------------------------------------------------+
