//+------------------------------------------------------------------+
//|                                                          ACS.mq5 |
//|                        Copyright 2024, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright   "2024, MetaQuotes Software Corp."
#property link        "https://www.mql5.com/en/users/lynnchris"
#property description "Script that displays previous day metrics on the current chart and predicts market direction."
#property version     "1.0"

#include <Trade\Trade.mqh>

//+------------------------------------------------------------------+
//| Input parameters                                                 |
//+------------------------------------------------------------------+
input color TableTextColor = clrBlue; // Text color for the table
input int   TableXOffset   = 10;      // X offset for the table
input int   TableYOffset   = 50;      // Y offset for the table
input color SupportColor = clrGreen; // Color for the support line
input color ResistanceColor = clrRed; // Color for the resistance line

//+------------------------------------------------------------------+
//| Global variables                                                 |
//+------------------------------------------------------------------+
double prevDayHigh, prevDayLow, prevDayOpen, prevDayClose, prevDayVolume;
double currentDayVolume;
double keySupport, keyResistance; // Initialized but will be calculated

//+------------------------------------------------------------------+
//| Main function                                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
//--- Get previous day's data
   int prevDayIndex = iBarShift(NULL, PERIOD_D1, iTime(NULL, PERIOD_D1, 1));

   if(prevDayIndex == -1)
     {
      Print("Error retrieving previous day's data.");
      return;
     }

   prevDayHigh    = iHigh(NULL, PERIOD_D1, prevDayIndex);
   prevDayLow     = iLow(NULL, PERIOD_D1, prevDayIndex);
   prevDayOpen    = iOpen(NULL, PERIOD_D1, prevDayIndex);
   prevDayClose   = iClose(NULL, PERIOD_D1, prevDayIndex);
   prevDayVolume  = iVolume(NULL, PERIOD_D1, prevDayIndex);

//--- Get today's volume
   currentDayVolume = iVolume(NULL, PERIOD_D1, 0); // Current day's volume

//--- Calculate key support and resistance
   keySupport = prevDayLow;  // Support level can be set to the previous day's low
   keyResistance = prevDayHigh; // Resistance level can be set to the previous day's high

//--- Manage existing lines (if any)
//ObjectDelete("SupportLine");
//ObjectDelete("ResistanceLine");

//--- Create support line
   if(!ObjectCreate(0, "SupportLine", OBJ_HLINE, 0, 0, keySupport))
     {
      Print("Failed to create SupportLine");
     }
   ObjectSetInteger(0, "SupportLine", OBJPROP_COLOR, SupportColor);
   ObjectSetInteger(0, "SupportLine", OBJPROP_WIDTH, 2); // Set the width of the line

//--- Create resistance line
   if(!ObjectCreate(0, "ResistanceLine", OBJ_HLINE, 0, 0, keyResistance))
     {
      Print("Failed to create ResistanceLine");
     }
   ObjectSetInteger(0, "ResistanceLine", OBJPROP_COLOR, ResistanceColor);
   ObjectSetInteger(0, "ResistanceLine", OBJPROP_WIDTH, 2); // Set the width of the line

//--- Determine the day's nature (Bullish or Bearish)
   string marketNature;
   if(prevDayClose > prevDayOpen)
     {
      marketNature = "Bullish";
     }
   else
      if(prevDayClose < prevDayOpen)
        {
         marketNature = "Bearish";
        }
      else
        {
         marketNature = "Neutral";
        }

//--- Compare volumes and determine market sentiment
   string volumeCommentary;
   if(currentDayVolume > prevDayVolume)
     {
      volumeCommentary = "Current day volume is higher than previous day volume. Bullish sentiment may continue.";
     }
   else
      if(currentDayVolume < prevDayVolume)
        {
         volumeCommentary = "Current day volume is lower than previous day volume. Bearish sentiment may follow.";
        }
      else
        {
         volumeCommentary = "Current day volume is equal to previous day volume. Market sentiment remains uncertain.";
        }

//--- Generate market movement commentary
   string marketCommentary;
   if(marketNature == "Bullish")
     {
      marketCommentary = "The market closed higher yesterday, indicating bullish sentiment. Look for potential continuation patterns.";
     }
   else
      if(marketNature == "Bearish")
        {
         marketCommentary = "The market closed lower yesterday, indicating bearish sentiment. Consider taking positions that align with this trend.";
        }
      else
        {
         marketCommentary = "The market showed neutrality with little change. Watch for potential breakout opportunities.";
        }

//--- Display the information in a table-like format on the chart
   string textOutput = "Previous Day Analytics:\n";
   textOutput += "Open: " + DoubleToString(prevDayOpen, 5) + "\n";
   textOutput += "Close: " + DoubleToString(prevDayClose, 5) + "\n";
   textOutput += "High: " + DoubleToString(prevDayHigh, 5) + "\n";
   textOutput += "Low: " + DoubleToString(prevDayLow, 5) + "\n";
   textOutput += "Volume (Prev Day): " + DoubleToString(prevDayVolume, 0) + "\n";
   textOutput += "Volume (Current Day): " + DoubleToString(currentDayVolume, 0) + "\n";
   textOutput += "Support: " + DoubleToString(keySupport, 5) + "\n";
   textOutput += "Resistance: " + DoubleToString(keyResistance, 5) + "\n";
   textOutput += "\nMarket Nature: " + marketNature + "\n";
   textOutput += volumeCommentary + "\n";
   textOutput += marketCommentary;

// Draw the text output on the chart
   Comment(textOutput);
  }

//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
