//+------------------------------------------------------------------+
//|                                                     SignalHM.mqh |
//|                   Copyright 2009-2024, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals based on Pythogorean HM                            |
//| Type=SignalAdvanced                                              |
//| Name=HM                                                          |
//| ShortName=HM                                                     |
//| Class=CSignalHM                                                  |
//| Page=signal_pm                                                   |
//| Parameter=Fast,int,6,Fast Period                                 |
//| Parameter=Slow,int,30,Slow Period                                |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalHM.                                                 |
//| Purpose: Class of Pythogorean HM.                                |
//|            Derives from class CExpertSignal.                     |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalHM   : public CExpertSignal
{
protected:

   int                           m_fast;                       //
   int                           m_slow;                       //

public:
                     CSignalHM(void);
                    ~CSignalHM(void);

   //--- methods of setting adjustable parameters
   void                          Fast(int value)
   {  m_fast = value;
   }
   void                          Slow(int value)
   {  m_slow = value;
   }

   double            CrossOver(int Index)
   {                 m_close.Refresh(-1);
      return(AM(Index) - m_close.GetData(Index));
   }

   double            Divergence(int Index)
   {                 return((HM_(Index, 2) - HM_(Index + 1, 2)) - (HM(Index, 4) - HM(Index + 1, 4)));
   }

   double            BandsUp(int Index)
   {  vector _bu;
      _bu.CopyRates(m_symbol.Name(), m_period, 2, Index, m_slow);
      return(GM_(Index, 2) + (2.0 * _bu.Std()));
   }

   double            BandsDn(int Index)
   {  vector _bd;
      _bd.CopyRates(m_symbol.Name(), m_period, 4, Index, m_slow);
      return(GM(Index, 4) - (2.0 * _bd.Std()));
   }

   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);

protected:
   //
   double                        AM(int Index, int Mask = 8);
   double                        GM(int Index, int Mask = 8);
   double                        HM(int Index, int Mask = 8);
   double                        GM_(int Index, int Mask = 8);
   double                        HM_(int Index, int Mask = 8);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CSignalHM::CSignalHM(void) :      m_fast(6),
   m_slow(30)

{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE + USE_SERIES_SPREAD + USE_SERIES_TIME;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CSignalHM::~CSignalHM(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalHM::ValidationSettings(void)
{  if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_fast < 3)
   {  printf(__FUNCSIG__ + " periods size should at least be 3! ");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalHM::InitIndicators(CIndicators *indicators)
{
//--- check of pointer is performed in the method of the parent class
//---
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalHM::LongCondition(void)
{  int result = 0;
   m_close.Refresh(-1);
   if(Divergence(StartIndex()+1) > 0.0 && m_close.GetData(StartIndex()) > m_close.GetData(StartIndex()+1))
   {  result = int(round(100.0 * (Divergence(StartIndex()+1)/(fabs(Divergence(StartIndex()))+fabs(Divergence(StartIndex()+1))))));
   }
   return(result);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalHM::ShortCondition(void)
{  int result = 0;
   m_close.Refresh(-1);
   if(Divergence(StartIndex()+1) > 0.0 && m_close.GetData(StartIndex()) < m_close.GetData(StartIndex()+1))
   {  result = int(round(100.0 * (Divergence(StartIndex()+1)/(fabs(Divergence(StartIndex()))+fabs(Divergence(StartIndex()+1))))));
   }
   return(result);
}
//+------------------------------------------------------------------+
//| Arithmetic Mean                                                  |
//+------------------------------------------------------------------+
double CSignalHM::AM(int Index, int Mask = 8)
{  vector _am;
   _am.CopyRates(m_symbol.Name(), m_period, Mask, Index, m_fast);
   return(_am.Mean());
}
//+------------------------------------------------------------------+
//| Geometric Mean                                                   |
//+------------------------------------------------------------------+
double CSignalHM::GM(int Index, int Mask = 8)
{  vector _gm;
   _gm.CopyRates(m_symbol.Name(), m_period, Mask, Index, m_slow);
   return(pow(_gm.Prod(), 1.0 / m_slow));
}
//+------------------------------------------------------------------+
//| Inverse Geometric Mean                                           |
//+------------------------------------------------------------------+
double CSignalHM::GM_(int Index, int Mask = 8)
{  double _am = AM(Index, Mask);
   double _gm = GM(Index, Mask);
   return(_am + (_am - _gm));
}
//+------------------------------------------------------------------+
//| Harmonic Mean                                                    |
//+------------------------------------------------------------------+
double CSignalHM::HM(int Index, int Mask = 8)
{  vector _hm, _hm_i;
   _hm_i.CopyRates(m_symbol.Name(), m_period, Mask, Index, m_slow);
   _hm = (1.0 / _hm_i);
   return(m_slow / _hm.Sum());
}
//+------------------------------------------------------------------+
//| Inverse Harmonic Mean                                            |
//+------------------------------------------------------------------+
double CSignalHM::HM_(int Index, int Mask = 8)
{  double _am = AM(Index, Mask);
   double _hm = HM(Index, Mask);
   return(_am + (_am - _hm));
}
//+------------------------------------------------------------------+
