//+------------------------------------------------------------------+
//|                                                      Network.mqh |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class Cgan
{
protected:
   matrix            weights[];
   vector            biases[];

   vector            inputs;
   vector            hidden_outputs[];
   vector            target;

   int               hidden_layers;
   double            Softplus(double X);
   double            SoftplusDerivative(double X);

public:
   vector            output;

   void              Get(vector &Target)
   {                 target.Init(0);
                     target.Copy(Target);
   }

   void              Set(vector &Inputs)
   {                 inputs.Init(0);
                     inputs.Copy(Inputs);
   }
   
   bool              Get(string File, double &Criteria, datetime &Version);
   bool              Set(string File, double Criteria);

   void              Forward();
   void              Backward(double LearningRate=0.05);
   void              Backward(vector<double> &DiscriminatorOutput, double LearningRate=0.05);
   void              Back(vector<double> &Error, double LearningRate);
   
   void              Cgan(int &Settings[],double InitialWeight,double InitialBias)
   {                 int _size =    ArraySize(Settings);
                     if(_size >= 2 && _size <= USHORT_MAX && Settings[ArrayMinimum(Settings)] > 0 && Settings[ArrayMaximum(Settings)] < USHORT_MAX)
                     {  ArrayResize(weights, _size - 1);
                        ArrayResize(biases, _size - 1);
                        ArrayResize(hidden_outputs, _size - 2);
                        hidden_layers = _size - 2;
                        for(int i = 0; i < _size - 1; i++)
                        {  weights[i].Init(Settings[i + 1], Settings[i]);
                           weights[i].Fill(InitialWeight);
                           biases[i].Init(Settings[i + 1]);
                           biases[i].Fill(InitialBias);
                           if(i < _size - 2)
                           {  hidden_outputs[i].Init(Settings[i + 1]);
                              hidden_outputs[i].Fill(0.0);
                           }
                        }
                        output.Init(Settings[_size - 1]);
                        target.Init(Settings[_size - 1]);
                     }
                     else
                     {  printf(__FUNCSIG__ + " invalid network settings. ");
                        //~Cgan(void);
                     }
   };
   void              ~Cgan(void) { };
};
//+------------------------------------------------------------------+
//| Softplus activation function                                     |
//+------------------------------------------------------------------+
double Cgan::Softplus(double X)
{  return log(1 + exp(X));
}
//+------------------------------------------------------------------+
//| Derivative of the Softplus function                              |
//+------------------------------------------------------------------+
double Cgan::SoftplusDerivative(double X)
{  return exp(X) / (1 + exp(X));
}
//+------------------------------------------------------------------+
//| Get Weights in import                                            |
//+------------------------------------------------------------------+
bool Cgan::Get(string File, double &Criteria, datetime &Version)
{  ResetLastError();
   int _handle = FileOpen(MQLInfoString(MQL_PROGRAM_NAME) + "_" + File + ".bin", FILE_READ | FILE_BIN | FILE_COMMON);
   if(_handle != INVALID_HANDLE)
   {  Criteria = FileReadDouble(_handle);
      Version = FileReadInteger(_handle);
      PrintFormat(__FUNCSIG__ + " for the file: %s, we have the criteria: %.2f, logged on: %s ",
      File,Criteria,TimeToString(Version));
      for(int i = 0; i < int(weights.Size()); i++)
      {  for(int j = 0; j < int(weights[i].Rows()); j++)
         {  for(int k = 0; k < int(weights[i].Cols()); k++)
            {  weights[i][j][k] = FileReadDouble(_handle);
               PrintFormat(__FUNCSIG__ + 
               " weight is: %.2f, for layer %i, at row: %i, and col: %i",
               weights[i][j][k],i,j,k);
            }
         }
      }
      for(int i = 0; i < int(biases.Size()); i++)
      {  for(int j = 0; j < int(biases[i].Size()); j++)
         {  biases[i][j] = FileReadDouble(_handle);
            PrintFormat(__FUNCSIG__ + 
            " %.5f is the bias for layer %i, at row: %i",
            biases[i][j],i,j);
         }
      }
      FileClose(_handle);
   }
   else
   {  printf(__FUNCSIG__ + " failed to create get handle for: " + File + ", err: " + IntegerToString(GetLastError()));
      FileClose(_handle);
      return(false);
   }
   FileClose(_handle);
   return(true);
}
//+------------------------------------------------------------------+
//| Set Weights for export                                           |
//+------------------------------------------------------------------+
bool Cgan::Set(string File, double Criteria)
{  ResetLastError();
   int _handle = FileOpen(MQLInfoString(MQL_PROGRAM_NAME) + "_" + File + ".bin", FILE_WRITE | FILE_BIN | FILE_COMMON);
   if(_handle != INVALID_HANDLE)
   {  FileWriteDouble(_handle, Criteria);
      FileWriteInteger(_handle, int(TimeCurrent()));
      for(int i = 0; i < int(weights.Size()); i++)
      {  for(int j = 0; j < int(weights[i].Rows()); j++)
         {  for(int k = 0; k < int(weights[i].Cols()); k++)
            {  FileWriteDouble(_handle, weights[i][j][k]);
            }
         }
      }
      for(int i = 0; i < int(biases.Size()); i++)
      {  for(int j = 0; j < int(biases[i].Size()); j++)
         {  FileWriteDouble(_handle, biases[i][j]);
         }
      }
      FileClose(_handle);
   }
   else
   {  printf(__FUNCSIG__ + " failed to create set handle for: " + File + ", err: " + IntegerToString(GetLastError()));
      FileClose(_handle);
      return(false);
   }
   FileClose(_handle);
   return(true);
}
//+------------------------------------------------------------------+
//| Forward pass through the neural network                          |
//+------------------------------------------------------------------+
void Cgan::Forward()
{  if(ArraySize(weights) != hidden_layers + 1)
   {  printf(__FUNCSIG__ + " weights matrix array size should be: " + IntegerToString(hidden_layers + 1));
      return;
   }
   if(ArraySize(biases) != hidden_layers + 1)
   {  printf(__FUNCSIG__ + " biases vector array size should be: " + IntegerToString(hidden_layers + 1));
      return;
   }
   if(ArraySize(hidden_outputs) != hidden_layers)
   {  printf(__FUNCSIG__ + " hidden outputs vector array size should be: " + IntegerToString(hidden_layers));
      return;
   }
// Calculate hidden layer values
   for(int h = 0; h < hidden_layers; h++)
   {  for (int i = 0; i < int(weights[h].Rows()); i++)
      {  double _sum = 0.0;
         if(h == 0)
         {  for (int j = 0; j < int(inputs.Size()); j++)
            {  _sum += inputs[j] * weights[h][i][j];
            }
         }
         else
         {  for (int j = 0; j < int(hidden_outputs[h - 1].Size()); j++)
            {  _sum += hidden_outputs[h - 1][j] * weights[h][i][j];
            }
         }
         hidden_outputs[h][i] = Softplus(_sum + biases[h][i]);
      }
   }
// Calculate output
   output.Fill(0.0);
   for (int i = 0; i < int(output.Size()); i++)
   {  double _sum = 0.0;
      for (int j = 0; j < int(weights[hidden_layers].Rows()); j++)
      {  _sum += hidden_outputs[hidden_layers - 1][i] * weights[hidden_layers][j][i];
      }
      output[i] += Softplus(_sum + biases[hidden_layers][i]);
   }
}
//+------------------------------------------------------------------+
//| Backward pass through the neural network to update weights       |
//| and biases using gradient descent                                |
//+------------------------------------------------------------------+
void Cgan::Backward(double LearningRate = 0.05)
{  if(target.Size() != output.Size())
   {  printf(__FUNCSIG__ + " Target & output size should match. ");
      return;
   }
   if(ArraySize(weights) != hidden_layers + 1)
   {  printf(__FUNCSIG__ + " weights matrix array size should be: " + IntegerToString(hidden_layers + 1));
      return;
   }
   if(ArraySize(biases) != hidden_layers + 1)
   {  printf(__FUNCSIG__ + " biases vector array size should be: " + IntegerToString(hidden_layers + 1));
      return;
   }
   if(ArraySize(hidden_outputs) != hidden_layers)
   {  printf(__FUNCSIG__ + " hidden outputs vector array size should be: " + IntegerToString(hidden_layers));
      return;
   }
// Update output layer weights and biases
   vector _output_error = target - output;
   Back(_output_error, LearningRate);
}
//+------------------------------------------------------------------+
//| Backward pass through the neural network to update weights       |
//| and biases using gradient descent                                |
//+------------------------------------------------------------------+
void Cgan::Backward(vector<double> &DiscriminatorOutput, double LearningRate = 0.05)
{  if(target.Size() != output.Size())
   {  printf(__FUNCSIG__ + " Target & output size should match. ");
      return;
   }
   if(ArraySize(weights) != hidden_layers + 1)
   {  printf(__FUNCSIG__ + " weights matrix array size should be: " + IntegerToString(hidden_layers + 1));
      return;
   }
   if(ArraySize(biases) != hidden_layers + 1)
   {  printf(__FUNCSIG__ + " biases vector array size should be: " + IntegerToString(hidden_layers + 1));
      return;
   }
   if(ArraySize(hidden_outputs) != hidden_layers)
   {  printf(__FUNCSIG__ + " hidden outputs vector array size should be: " + IntegerToString(hidden_layers));
      return;
   }
// Update output layer weights and biases
   vector _output_error = -1.0*MathLog(DiscriminatorOutput)*(target - output);//solo modification for GAN
   Back(_output_error, LearningRate);
}
//+------------------------------------------------------------------+
//| Back                                                             |
//+------------------------------------------------------------------+
void Cgan::Back(vector<double> &Error, double LearningRate)
{  vector _output_gradients;
   _output_gradients.Init(output.Size());
   for (int i = 0; i < int(output.Size()); i++)
   {  _output_gradients[i] = Error[i] * SoftplusDerivative(output[i]);
   }
// Calculate output layer gradients
   for (int i = 0; i < int(output.Size()); i++)
   {  for (int j = 0; j < int(weights[hidden_layers].Rows()); j++)
      {  weights[hidden_layers][j][i] += LearningRate * _output_gradients[i] * output[j];
      }
      biases[hidden_layers][i] += LearningRate * _output_gradients[i];
   }
// Calculate hidden layer gradients
   vector _hidden_gradients[];
   ArrayResize(_hidden_gradients, hidden_layers);
   for(int h = hidden_layers - 1; h >= 0;  h--)
   {  _hidden_gradients[h].Init(hidden_outputs[h].Size());
      for (int i = 0; i < int(hidden_outputs[h].Size()); i++)
      {  double _hidden_error = 0.0;
         for (int j = 0; j < int(output.Size()); j++)
         {  _hidden_error += _output_gradients[j] * weights[h][i][j];
         }
         _hidden_gradients[h][i] = _hidden_error * SoftplusDerivative(hidden_outputs[h][i]);
      }
      // Update hidden layer weights and biases
      for (int i = 0; i < int(hidden_outputs[h].Size()); i++)
      {  if(h == 0)
         {  for (int j = 0; j < int(inputs.Size()); j++)
            {  weights[h][i][j] += LearningRate * _hidden_gradients[h][i] * inputs[j];
            }
         }
         else
         {  for (int j = 0; j < int(hidden_outputs[h - 1].Size()); j++)
            {  weights[h][i][j] += LearningRate * _hidden_gradients[h][i] * hidden_outputs[h - 1][j];
            }
         }
         biases[h][i] += LearningRate * _hidden_gradients[h][i];
      }
   }
}
//+------------------------------------------------------------------+
