//+------------------------------------------------------------------+
//|                                                      MoneyCT.mqh |
//|                   Copyright 2009-2017, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
#include <Arrays\ArrayInt.mqh>
#define __PI 245850922/78256779
#define __PHI 0.61803398874989
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals based on DBSCAN classifier                         |
//| Type=SignalAdvanced                                              |
//| Name=CategoryTheory                                              |
//| ShortName=DBSCAN                                                 |
//| Class=CSignalDBSCAN                                              |
//| Page=signal_dbscan                                               |
//| Parameter=KeySize,int,4,Key Size                                 |
//| Parameter=SetSize,int,30,Set Size                                |
//| Parameter=MinPoints,int,2,Min Points                             |
//| Parameter=Epsilon,double,12.5,Epsilon                            |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalDBSCAN.                                             |
//| Purpose: Class of of trade signals based on DBSCAN classifier.   |
//|            Derives from class CExpertSignal.                     |
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalDBSCAN                 : public CExpertSignal
{
protected:

   int                           m_key_size;                            // KeySize
   int                           m_set_size;                            // SetSize
   int                           m_min_points;                          // MinPoints
   double                        m_epsilon;                             // Epsilon

public:
                     CSignalDBSCAN(void);
                    ~CSignalDBSCAN(void);

   //--- method of verification of settings
   virtual bool                  ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool                  InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int                   LongCondition(void);
   virtual int                   ShortCondition(void);
   //---
   void                          KeySize(int value)
   {  m_key_size = value;
   }
   int                           KeySize()
   {  return(m_key_size);
   }
   void                          SetSize(int value)
   {  m_set_size = value;
   }
   void                          MinPoints(int value)
   {  m_min_points = value;
   }
   void                          Epsilon(double value)
   {  m_epsilon = value;
   }

protected:

   double                        GetOutput();

   struct Spoint
   {  vector            key;
      bool              visited;
      int               cluster_id;

                     Spoint()
      {              key.Resize(0);
         visited = false;
         cluster_id = -1;
      };
                    ~Spoint() {};
   };

   struct                        Smodel
   {  Spoint  x[];
      vector         y;

                     Smodel() {};
                    ~Smodel() {};
   };

   Smodel                        m_model;

   void                          DBSCAN(Spoint &SetOfPoints[]);
   bool                          ExpandCluster(Spoint &SetOfPoints[], int Index, int ClusterID);
   void                          RegionQuery(Spoint &P[], int Index, CArrayInt &Neighbours);
   double                        Distance(Spoint &A, Spoint &B);

};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CSignalDBSCAN::CSignalDBSCAN(void)  :   m_set_size(30),
   m_min_points(2),
   m_epsilon(12.50)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_OPEN + USE_SERIES_HIGH + USE_SERIES_LOW + USE_SERIES_CLOSE + USE_SERIES_SPREAD + USE_SERIES_TIME;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CSignalDBSCAN::~CSignalDBSCAN(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalDBSCAN::ValidationSettings(void)
{  if(!CExpertSignal::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_set_size < 2)
   {  printf(__FUNCSIG__ + " set size should at least be 2! ");
      return(false);
   }
   if(m_key_size < 1)
   {  printf(__FUNCSIG__ + " key size should at least be 1! ");
      return(false);
   }
   ArrayResize(m_model.x,m_set_size);
   m_model.y.Resize(m_set_size-1);
   for(int i=0;i<m_set_size;i++)
   { m_model.x[i].key.Resize(KeySize());
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalDBSCAN::InitIndicators(CIndicators *indicators)
{
//--- check pointer
   if(indicators == NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalDBSCAN::LongCondition(void)
{  int result = 0;
   m_high.Refresh(-1);
   m_low.Refresh(-1);
   double _gap = GetOutput();
   int _range_size = 1;//m_set_size;
   double _range = m_high.GetData(m_high.MaxIndex(StartIndex(), StartIndex() + _range_size)) - m_low.GetData(m_low.MinIndex(StartIndex(), StartIndex() + _range_size));
   _gap /= fmax(_range, m_symbol.Point());
   _gap *= 100.0;
   if(_gap > 0.0)
   {  result = int(fmin(100.0, round(_gap)));
   }
//printf(__FUNCSIG__+" gap is: "+DoubleToString(_gap)+", result is: "+IntegerToString(result));
   //result = 0;
   return(result);
}
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalDBSCAN::ShortCondition(void)
{  int result = 0;
   m_high.Refresh(-1);
   m_low.Refresh(-1);
   double _gap = GetOutput();
   int _range_size = 1;//m_set_size;
   double _range = m_high.GetData(m_high.MaxIndex(StartIndex(), StartIndex() + _range_size)) - m_low.GetData(m_low.MinIndex(StartIndex(), StartIndex() + _range_size));
   _gap /= fmax(_range, m_symbol.Point());
   _gap *= 100.0;
   if(_gap < 0.0)
   {  result = int(fmax(-100.0, round(_gap))) * -1;
   }
//printf(__FUNCSIG__+" gap is: "+DoubleToString(_gap)+", result is: "+IntegerToString(result));
   //result = 0;
   return(result);
}
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CSignalDBSCAN::GetOutput()
{  double _output = 0.0;
   ArrayResize(m_model.x, m_set_size);
   for(int i = 0; i < m_set_size; i++)
   {  m_model.x[i].key.Resize(m_key_size);
   }
   m_model.y.Resize(m_set_size - 1);
   m_open.Refresh(-1);
   m_high.Refresh(-1);
   m_low.Refresh(-1);
   m_close.Refresh(-1);
   for(int i = 0; i < m_set_size; i++)
   {  for(int ii = 0; ii < m_key_size; ii++)
      {  if(ii == 0)
         {  m_model.x[i].key[ii] = (m_open.GetData(StartIndex() + i) - m_open.GetData(StartIndex() + ii + i + 1)) / m_symbol.Point();
         }
         else if(ii == 1)
         {  m_model.x[i].key[ii] = (m_high.GetData(StartIndex() + i) - m_high.GetData(StartIndex() + ii + i + 1)) / m_symbol.Point();
         }
         else if(ii == 2)
         {  m_model.x[i].key[ii] = (m_low.GetData(StartIndex() + i) - m_low.GetData(StartIndex() + ii + i + 1)) / m_symbol.Point();
         }
         else if(ii == 3)
         {  m_model.x[i].key[ii] = (m_close.GetData(StartIndex() + i) - m_close.GetData(StartIndex() + ii + i + 1)) / m_symbol.Point();
         }
      }
      if(i > 0) //assign classifier only for data points for which eventual bar range is known
      {  m_model.y[i - 1] = (m_close.GetData(StartIndex() + i - 1) - m_close.GetData(StartIndex() + i)) / m_symbol.Point();
      }
   }
   int _o[];
   DBSCAN(m_model.x);
//ArrayPrint(_o);
   int _output_count = 0;
   for(int i = 1; i < m_set_size; i++)
   {  if(m_model.x[0].cluster_id == m_model.x[i].cluster_id)
      {  _output += (m_model.y[i - 1]);
         _output_count++;
      }
   }
//
   if(_output_count > 0)
   {  _output /= _output_count;
   }
   return(_output);
}
//+------------------------------------------------------------------+
//| Main clustering function                                         |
//+------------------------------------------------------------------+
void CSignalDBSCAN::DBSCAN(Spoint &SetOfPoints[])
{  int _cluster_id = -1;
   int _size = ArraySize(SetOfPoints);
   for(int i = 0; i < _size; i++)
   {  if(SetOfPoints[i].cluster_id == -1)
      {  if(ExpandCluster(SetOfPoints, i, _cluster_id))
         {  _cluster_id++;
            SetOfPoints[i].cluster_id = _cluster_id;
         }
      }
   }
}
//+------------------------------------------------------------------+
//| Function that extends cluster for identified cluster IDs         |
//+------------------------------------------------------------------+
bool CSignalDBSCAN::ExpandCluster(Spoint &SetOfPoints[], int Index, int ClusterID)
{  CArrayInt _seeds;
   RegionQuery(SetOfPoints, Index, _seeds);
   if(_seeds.Total() < m_min_points) // no core point
   {  SetOfPoints[Index].cluster_id = -1;
      return(false);
   }
   else
   {  SetOfPoints[Index].cluster_id = ClusterID;
      for(int ii = 0; ii < _seeds.Total(); ii++)
      {  int _current_p = _seeds[ii];
         CArrayInt _result;
         RegionQuery(SetOfPoints, _current_p, _result);
         if(_result.Total() > m_min_points)
         {  for(int i = 0; i < _result.Total(); i++)
            {  int _result_p = _result[i];
               if(SetOfPoints[_result_p].cluster_id == -1)
               {  SetOfPoints[_result_p].cluster_id = ClusterID;
               }
            }
         }
      }
   }
   return(true);
}
//+------------------------------------------------------------------+
//| Function that returns neighbouring points for an input point &P[]|
//+------------------------------------------------------------------+
void CSignalDBSCAN::RegionQuery(Spoint &P[], int Index, CArrayInt &Neighbours)
{  Neighbours.Resize(0);
   int _size = ArraySize(P);
   for(int i = 0; i < _size; i++)
   {  if(i == Index)
      {  continue;
      }
      else if(Distance(P[i], P[Index]) <= m_epsilon)
      {  Neighbours.Resize(Neighbours.Total() + 1);
         Neighbours.Add(i);
      }
   }
   P[Index].visited = true;
}
//+------------------------------------------------------------------+
//| Function for Euclidean Distance between points                   |
//+------------------------------------------------------------------+
double CSignalDBSCAN::Distance(Spoint &A, Spoint &B)
{  double _d = 0.0;
   for(int i = 0; i < int(fmin(A.key.Size(), B.key.Size())); i++)
   {  _d += pow(A.key[i] - B.key[i], 2.0);
   }
   _d = sqrt(_d);
   return(_d);
}
//+------------------------------------------------------------------+
