//+------------------------------------------------------------------+
//|                                           MoneySizeOptimized.mqh |
//|                             Copyright 2000-2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertMoney.mqh>
#include <Trade\DealInfo.mqh>
#include <my\Cnewton.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Trading with newton's polyomial optimized trade volume     |
//| Type=Money                                                       |
//| Name=SizeOptimized                                               |
//| Class=CMoneySizeOptimized                                        |
//| Page=                                                            |
//| Parameter=DecreaseFactor,double,3.0,Decrease factor              |
//| Parameter=Percent,double,10.0,Percent                            |
//| Parameter=Length,int,5,Length                                    |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CMoneySizeOptimized.                                       |
//| Purpose: Class of money management with newton's polyomial size optimized.          |
//|              Derives from class CExpertMoney.                    |
//+------------------------------------------------------------------+
class CMoneySizeOptimized : public CExpertMoney
{
protected:
   double            m_decrease_factor;

   int               m_length;
   Cnewton           __N;

public:
                     CMoneySizeOptimized(void);
                    ~CMoneySizeOptimized(void);
   //---
   void              DecreaseFactor(double decrease_factor)
   {  m_decrease_factor = decrease_factor;
   }
   void              Length(int value)
   {  m_length = value;
   };
   virtual bool      ValidationSettings(void);
   //---
   virtual double    CheckOpenLong(double price, double sl);
   virtual double    CheckOpenShort(double price, double sl);

protected:
   double            Optimize(double lots);
};
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CMoneySizeOptimized::CMoneySizeOptimized(void) : m_decrease_factor(3.0),
   m_length(5)
{
//--- initialization of protected data
   m_used_series = USE_SERIES_HIGH + USE_SERIES_LOW;
}
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CMoneySizeOptimized::~CMoneySizeOptimized(void)
{
}
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CMoneySizeOptimized::ValidationSettings(void)
{  if(!CExpertMoney::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_decrease_factor <= 0.0)
   {  printf(__FUNCTION__ + ": decrease factor must be greater then 0");
      return(false);
   }
//--- ok
   return(true);
}
//+------------------------------------------------------------------+
//| Getting lot size for open long position.                         |
//+------------------------------------------------------------------+
double CMoneySizeOptimized::CheckOpenLong(double price, double sl)
{  if(m_symbol == NULL)
      return(0.0);
//--- select lot size
   double lot;
   if(price == 0.0)
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_BUY, m_symbol.Ask(), m_percent);
   else
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_BUY, price, m_percent);
//--- return trading volume
   return(Optimize(lot));
}
//+------------------------------------------------------------------+
//| Getting lot size for open short position.                        |
//+------------------------------------------------------------------+
double CMoneySizeOptimized::CheckOpenShort(double price, double sl)
{  if(m_symbol == NULL)
      return(0.0);
//--- select lot size
   double lot;
   if(price == 0.0)
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_SELL, m_symbol.Bid(), m_percent);
   else
      lot = m_account.MaxLotCheck(m_symbol.Name(), ORDER_TYPE_SELL, price, m_percent);
//--- return trading volume
   return(Optimize(lot));
}
//+------------------------------------------------------------------+
//| Optimizing lot size for open.                                    |
//+------------------------------------------------------------------+
double CMoneySizeOptimized::Optimize(double lots)
{  double lot = lots;
//--- 0 factor means no optimization
   if(m_decrease_factor > 0)
   {  m_high.Refresh(-1);
      m_low.Refresh(-1);
      vector _x, _y;
      _x.Init(m_length);
      _y.Init(m_length);
      for(int i = 0; i < m_length; i++)
      {  _x[i] = i;
         _y[i] = (m_high.GetData(StartIndex() + i) - m_low.GetData(StartIndex() + i)) - (m_high.GetData(StartIndex() + i + 1) - m_low.GetData(StartIndex() + i + 1));
      }
      vector _w;
      _w.Init(m_length);
      _w[0] = _y[0];
      __N.Set(_w, _x, _y);
      double _xx = -1.0;
      double _yy = 0.0;
      __N.Get(_w, _x, _xx, _yy);
      //---
      if(_yy > 0.0)
      {  double _range = (m_high.GetData(StartIndex()) - m_low.GetData(StartIndex()));
         _range += (m_decrease_factor*m_symbol.Point());
         _range += _yy;
         lot = NormalizeDouble(lot*(1.0-(_yy/_range)), 2);
      }
   }
//--- normalize and check limits
   double stepvol = m_symbol.LotsStep();
   lot = stepvol * NormalizeDouble(lot / stepvol, 0);
//---
   double minvol = m_symbol.LotsMin();
   if(lot < minvol)
      lot = minvol;
//---
   double maxvol = m_symbol.LotsMax();
   if(lot > maxvol)
      lot = maxvol;
//---
   return(lot);
}
//+------------------------------------------------------------------+
