//+------------------------------------------------------------------+
#property copyright "Daniel Jose"
//+------------------------------------------------------------------+
#include <Canvas\Canvas.mqh>
//+------------------------------------------------------------------+
#define PrintEx(A)  Print(#A, " => ", A)
#define macroRandom (rand() / (double)SHORT_MAX)
#define _SizeLine   300
#define nColuns     2
//+------------------------------------------------------------------+
CCanvas canvas;
//+------------------------------------------------------------------+
double Train[][nColuns] {
                         {-100, -150},
                         { -80,  -50},
                         {  30,   80},
                         { 100,  120},
                        };
//+------------------------------------------------------------------+
const double epsilon = 1e-3;
//+------------------------------------------------------------------+
struct stErr
{
    double  Weight,
            Bias;
};
//+------------------------------------------------------------------+
stErr Cost(const double w, const double b, const uint p1 = 0, const uint p2 = Train.Size() / nColuns)
{
    double x, y, t;
    stErr err;

    ZeroMemory(err);
    for (uint c = p1; c < p2; c++)
    {
        x = Train[c][0];
        y = Train[c][1];
        t = 2 * ((x * w + b) - y);
        err.Weight += (t * x);
        err.Bias += t;
    }

    return err;
}
//+------------------------------------------------------------------+
void PlotText(const int x, const int y, const uchar line, const string sz0)
{
    uint w, h;

    TextGetSize(sz0, w, h);
    canvas.TextOut(x - (w / 2), y + _SizeLine + (line * h) + 5, sz0, ColorToARGB(clrBlack));   
}
//+------------------------------------------------------------------+
inline double CallZoom(void)
{
    double d1 = 0;

    for (uint c = 0; c < Train.Size() / nColuns; c++)
    {
        d1 = MathMax(d1, MathAbs(Train[c][0]));
        d1 = MathMax(d1, MathAbs(Train[c][1]));
    }

    return _SizeLine / d1;
}
//+------------------------------------------------------------------+
void Plot_Train2D(const int x, const int y, const double weight, double bias)
{
    int vx, vy;
    double zoom;
    uint n = Train.Size() / nColuns;

    zoom = CallZoom();
    canvas.LineVertical(x, y - _SizeLine, y + _SizeLine, ColorToARGB(clrRoyalBlue, 255));
    canvas.LineHorizontal(x - _SizeLine, x + _SizeLine, y, ColorToARGB(clrRoyalBlue, 255));
    for (uint c = 0; c < n; c++)
    {
        vx = (int)(Train[c][0] * zoom);
        vy = (int)(Train[c][1] * zoom);
	    canvas.FillCircle(x + vx, y - vy, 5, ColorToARGB(clrRed, 255));
    }
    canvas.LineAA(
                x + (int)(Train[0][0] * zoom), 
                y - (int)((Train[0][0] * weight + bias) * zoom), 
                x + (int)(Train[n - 1][0] * zoom), 
                y - (int)((Train[n - 1][0] * weight + bias) * zoom), 
                ColorToARGB(clrForestGreen));
    PlotText(x, y, 1, StringFormat("Zoom: %.2fx", zoom));
    PlotText(x, y, 2, StringFormat("f(x) = %.4fx %c %.4f", weight, (bias < 0 ? '-' : '+'), MathAbs(bias)));
}
//+------------------------------------------------------------------+
void OnStart()
{
    double weight, bias;
    int    x, y;
    ulong  it0, it1, count;
    stErr  err;

    Print("************************************");
    Print("Stochastic Gradient Descent Neuron...");
    Print("************************************");
    MathSrand(512);
    weight = (double)macroRandom;
    bias = (double)macroRandom;

    it0 = GetTickCount();
    for(count = 0; count < 5; count++)
    {
        err = Cost(weight, bias);
        if ((MathAbs(err.Weight) <= epsilon) && (MathAbs(err.Bias) <= epsilon))
            break;
        weight -= (err.Weight * epsilon);
        bias -= (err.Bias * epsilon);
        PrintFormat("%I64u > w0: %.4f %.4f || b: %.4f %.4f", count, weight, err.Weight, bias, err.Bias);
    }
    it1 = GetTickCount();
    Print("Time: ", (it1 - it0) / 1000.0, " seconds.");
    PrintEx(count);
    PrintEx(weight);
    PrintEx(bias);
    PrintEx(err.Weight);
    PrintEx(err.Bias);
    Print("Press ESC to close....");
    Print("************************************");

	x = (int)ChartGetInteger(0, CHART_WIDTH_IN_PIXELS, 0);
	y = (int)ChartGetInteger(0, CHART_HEIGHT_IN_PIXELS, 0);
	canvas.CreateBitmapLabel("BL", 0, 0, x, y, COLOR_FORMAT_ARGB_NORMALIZE);
	canvas.Erase(ColorToARGB(clrWhite, 255));
    x /= 2;
    y /= 2;
		
    Plot_Train2D(x, y, weight, bias);

	canvas.Update(true);

    while(!IsStopped())
        if (TerminalInfoInteger(TERMINAL_KEYSTATE_ESCAPE) !=0)
            break;

    canvas.Destroy();
}
//+------------------------------------------------------------------+