from sklearn.model_selection import train_test_split
from sklearn.svm import SVC
from sklearn.metrics import accuracy_score
from sklearn.model_selection import cross_val_score
import pandas as pd
import numpy as np
import MetaTrader5 as mt5
import onnx
from skl2onnx import convert_sklearn
from skl2onnx.common.data_types import FloatTensorType
import onnxruntime as rt


if not mt5.initialize():  # This will open MT5 app in your pc
    print("initialize() failed, error code =",mt5.last_error())
    quit()

    
terminal_info = mt5.terminal_info()

data_path = terminal_info.data_path
dataset_path = data_path + "\\MQL5\\Files"

class DualSVMONNX:
    def __init__(self, dataset, c=1.0, kernel='rbf'):
        
        data = pd.read_csv(dataset) # reading a csv file 
        np.random.seed(42)
        
        self.X = data.drop(columns=['target']).astype(np.float32) # dropping the target column from independent variable 
        self.y = data["target"].astype(int) # storing the target variable in its own vector 
        
        self.X = self.X.to_numpy()
        self.y = self.y.to_numpy()            

        # Split the data into training and testing sets
        self.X_train, self.X_test, self.y_train, self.y_test = train_test_split(self.X, self.y, test_size=0.2, random_state=42) 

        self.onnx_model_name = "DualSVMONNX" #our final onnx model file name for saving purposes really
        
        # Create a dual SVM model with a kernel
        
        self.svm_model = SVC(kernel=kernel, C=c)
        

    def fit(self):
        
        self.svm_model.fit(self.X_train, self.y_train) # fitting/training the model
        
        y_preds = self.svm_model.predict(self.X_train)
        
        print("train accuracy = ",accuracy_score(self.y_train, y_preds))
        y_preds = self.svm_model.predict(self.X_test)
        print("test accuracy = ",accuracy_score(self.y_test, y_preds))
        
        
        scores = cross_val_score(self.svm_model, self.X_train, self.y_train, cv=5)
        mean_cv_accuracy = np.mean(scores)

        print(f"\nscores {scores} mean_cv_accuracy {mean_cv_accuracy}")


    def saveONNX(self):                 
        
        initial_type = [('float_input', FloatTensorType(shape=[None, 4]))]  # None means we don't know the rows but we know the columns for sure, Remember !! we have 4 independent variables
        onnx_model = convert_sklearn(self.svm_model, initial_types=initial_type) # Convert the scikit-learn model to ONNX format

        onnx.save_model(onnx_model, dataset_path + f"\\{self.onnx_model_name}.onnx") #saving the onnx model
        
'''
    def loadONNX(self): # Load ONNX and use it 
        
        onnx_model_path = dataset_path + f"\\{self.onnx_model_name}.onnx"

        sess = rt.InferenceSession(onnx_model_path, providers=["CPUExecutionProvider"])
        input_name = sess.get_inputs()[0].name
        label_name = sess.get_outputs()[0].name
        
        input_data = self.X_test[0, :]
        
        # inps = np.float32([0.012,2.0923,3.0021,4.09323])
        # input_data = self.X_test.values.reshape(-1, 4).astype(np.float32)
        
        # Run the model with input data
        label , proba = sess.run(None, {input_name: [input_data]})

        # print("label = ",label, "\nproba =",proba)
'''


svm = DualSVMONNX(dataset_path+"\\DualSVMONNX-data.csv")
svm.fit()
svm.saveONNX()
# svm.loadONNX()