//+------------------------------------------------------------------+
//|                                                   TrailingCT.mqh |
//|                   Copyright 2009-2013, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Math\Stat\Math.mqh>
#include <Expert\ExpertTrailing.mqh>
#include <ct_8.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Trailing Stop based on 'Category Theory' monoids concepts  |
//| Type=Trailing                                                    |
//| Name=CategoryTheory                                              |
//| ShortName=CT                                                     |
//| Class=CTrailingCT                                                |
//| Page=trailing_ct                                                 |
//| Parameter=Step,double,0.5,Trailing Step                          |
//| Parameter=LookbackOperation,int,0,Source Lookback Operation      |
//| Parameter=LookbackIdentity,int,0,Source Lookback Identity        |
//| Parameter=TimeframeOperation,int,0,Source Timeframe Operation    |
//| Parameter=TimeframeIdentity,int,0,Source Timeframe Identity      |
//| Parameter=AppliedpriceOperation,int,0,Source Appliedprice Operation    |
//| Parameter=AppliedpriceIdentity,int,0,Source Appliedprice Identity      |
//| Parameter=IndicatorOperation,int,0,Source Indicator Operation    |
//| Parameter=IndicatorIdentity,int,0,Source Indicator Identity      |
//| Parameter=DecisionOperation,int,0,Source Decision Operation      |
//| Parameter=DecisionIdentity,int,0,Source Decision Identity        |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CTrailingCT.                                               |
//| Appointment: Class traling stops with 'Category Theory'          |
//|               monoids concepts.                                  |
//| Derives from class CExpertTrailing.                              |
//+------------------------------------------------------------------+
int  __LOOKBACKS[8] = {1,2,3,4,5,6,7,8};
ENUM_TIMEFRAMES  __TIMEFRAMES[8] = {PERIOD_H1,PERIOD_H2,PERIOD_H3,PERIOD_H4,PERIOD_H6,PERIOD_H8,PERIOD_H12,PERIOD_D1};
ENUM_APPLIED_PRICE   __APPLIEDPRICES[4] = { PRICE_MEDIAN, PRICE_TYPICAL, PRICE_OPEN, PRICE_CLOSE };
string   __INDICATORS[2] = { "RSI", "BOLLINGER_BANDS" };
string   __DECISIONS[2] = { "TREND", "RANGE" };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CTrailingCT : public CExpertTrailing
  {
protected:
   //--- adjusted parameters
   double m_step;       // trailing step
   
   CMonoid<double>      m_lookback;
   EOperations             m_lookback_operation;
   int                  m_lookback_identity;
   
   CMonoid<double>      m_timeframe;
   EOperations             m_timeframe_operation;
   int                  m_timeframe_identity;
   
   CMonoid<double>      m_appliedprice;
   EOperations             m_appliedprice_operation;
   int                  m_appliedprice_identity;
   
   CMonoid<double>      m_indicator;
   EOperations             m_indicator_operation;
   int                  m_indicator_identity;
   
   CMonoid<double>      m_decision;
   EOperations             m_decision_operation;
   int                  m_decision_identity;
   
   
   int m_source_size;       // Source Size

public:
   //--- methods of setting adjustable parameters
   
   
   
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual bool      CheckTrailingStopLong(CPositionInfo *position,double &sl,double &tp);
   virtual bool      CheckTrailingStopShort(CPositionInfo *position,double &sl,double &tp);
   //---
                     CTrailingCT(void);
                    ~CTrailingCT(void);
   //--- methods of initialization of protected data
   void              Step(double value)                  { m_step=value;   }
   
   void              LookbackOperation(int value)        { m_lookback_operation=(EOperations)value;    }
   void              LookbackIdentity(int value)            { m_lookback_identity=value;        }
   
   void              TimeframeOperation(int value)       { m_timeframe_operation=(EOperations)value;    }
   void              TimeframeIdentity(int value)           { m_timeframe_identity=value;        }
   
   void              AppliedpriceOperation(int value)       { m_appliedprice_operation=(EOperations)value;    }
   void              AppliedpriceIdentity(int value)           { m_appliedprice_identity=value;        }
   
   void              IndicatorOperation(int value)       { m_indicator_operation=(EOperations)value;    }
   void              IndicatorIdentity(int value)           { m_indicator_identity=value;        }
   
   void              DecisionOperation(int value)        { m_decision_operation=(EOperations)value;    }
   void              DecisionIdentity(int value)            { m_decision_identity=value;        }

protected:
   
   CElement<double>     m_element;
   
   void                 Operate_8(CMonoid<double> &M,EOperations &O,double &Values[],int &InputIndices[],int &OutputIndices[]);
   void                 Operate_4(CMonoid<double> &M,EOperations &O,double &Values[],int &InputIndices[],int &OutputIndices[]);
   void                 Operate_2(CMonoid<double> &M,EOperations &O,double &Values[],int &InputIndices[],int &OutputIndices[]);

   int                  GetLookback();
   ENUM_TIMEFRAMES      GetTimeframe();
   ENUM_APPLIED_PRICE   GetAppliedprice(int Lookback,ENUM_TIMEFRAMES Timeframe);
   double               GetIndicator(int Lookback,ENUM_TIMEFRAMES Timeframe,ENUM_APPLIED_PRICE Appliedprice);
   
   int                  GetDecision(int Lookback,double Indicator);

  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CTrailingCT::CTrailingCT(void)
  {
//--- initialization of protected data
      m_used_series=USE_SERIES_TIME+USE_SERIES_SPREAD+USE_SERIES_HIGH+USE_SERIES_LOW+USE_SERIES_CLOSE;
      
      m_lookback.Cardinality(8);
      m_lookback.Operation(m_lookback_operation);
      m_lookback.Identity(m_lookback_identity);
      
      m_timeframe.Cardinality(8);
      m_timeframe.Operation(m_timeframe_operation);
      m_timeframe.Identity(m_timeframe_identity);
      
      m_appliedprice.Cardinality(4);
      m_appliedprice.Operation(m_appliedprice_operation);
      m_appliedprice.Identity(m_appliedprice_identity);
      
      m_indicator.Cardinality(2);
      m_indicator.Operation(m_indicator_operation);
      m_indicator.Identity(m_indicator_identity);
      
      m_decision.Cardinality(2);
      m_decision.Operation(m_decision_operation);
      m_decision.Identity(m_decision_identity);
      
      m_step=0.5;
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CTrailingCT::~CTrailingCT(void)
  {
  }
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CTrailingCT::ValidationSettings(void)
  {
//--- validation settings of additional filters
   if(!CExpertTrailing::ValidationSettings())
      return(false);

   if(m_period!=PERIOD_H1){
      printf(__FUNCSIG__+" Timeframe should be one-hour! ");
      return(false);}
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CTrailingCT::InitIndicators(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertTrailing::InitIndicators(indicators))
      return(false);
//--- 
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Checking trailing stop and/or profit for long position.          |
//+------------------------------------------------------------------+
bool CTrailingCT::CheckTrailingStopLong(CPositionInfo *position,double &sl,double &tp)
  {
//--- check
      if(position==NULL)
         return(false);
      
      m_high.Refresh(-1);
      m_low.Refresh(-1);
      
      int _x=StartIndex();
      
      int _lookback=GetLookback();
      ENUM_TIMEFRAMES _timeframe=GetTimeframe();
      ENUM_APPLIED_PRICE _appliedprice=GetAppliedprice(_lookback,_timeframe);
      double _indicator=GetIndicator(_lookback,_timeframe,_appliedprice);
      int _decision=GetDecision(_lookback,_indicator);
      
      double _type=_decision/100.0;
   
      double _atr=fmax(2.0*m_spread.GetData(_x)*m_symbol.Point(),m_high.GetData(_x)-m_low.GetData(_x))*(_type);
      
      double _sl=m_low.GetData(_x)-(m_step*_atr);
      
      double level =NormalizeDouble(m_symbol.Bid()-m_symbol.StopsLevel()*m_symbol.Point(),m_symbol.Digits());
      double new_sl=NormalizeDouble(_sl,m_symbol.Digits());
      double pos_sl=position.StopLoss();
      double base  =(pos_sl==0.0) ? position.PriceOpen() : pos_sl;
      
      sl=EMPTY_VALUE;
      tp=EMPTY_VALUE;
      if(new_sl>base && new_sl<level)
         sl=new_sl;
//---
   return(sl!=EMPTY_VALUE);
  }
//+------------------------------------------------------------------+
//| Checking trailing stop and/or profit for short position.         |
//+------------------------------------------------------------------+
bool CTrailingCT::CheckTrailingStopShort(CPositionInfo *position,double &sl,double &tp)
  {
//--- check
      if(position==NULL)
         return(false);
      
      m_high.Refresh(-1);
      m_low.Refresh(-1);
      
      int _x=StartIndex();
      
      int _lookback=GetLookback();
      ENUM_TIMEFRAMES _timeframe=GetTimeframe();
      ENUM_APPLIED_PRICE _appliedprice=GetAppliedprice(_lookback,_timeframe);
      double _indicator=GetIndicator(_lookback,_timeframe,_appliedprice);
      int _decision=GetDecision(_lookback,_indicator);
      
      double _type=_decision/100.0;
   
      double _atr=fmax(2.0*m_spread.GetData(_x)*m_symbol.Point(),m_high.GetData(_x)-m_low.GetData(_x))*(_type);
      
      double _sl=m_high.GetData(_x)+(m_step*_atr);
      
      double level =NormalizeDouble(m_symbol.Ask()+m_symbol.StopsLevel()*m_symbol.Point(),m_symbol.Digits());
      double new_sl=NormalizeDouble(_sl,m_symbol.Digits());
      double pos_sl=position.StopLoss();
      double base  =(pos_sl==0.0) ? position.PriceOpen() : pos_sl;
      
      sl=EMPTY_VALUE;
      tp=EMPTY_VALUE;
      if(new_sl<base && new_sl>level)
         sl=new_sl;
//---
      return(sl!=EMPTY_VALUE);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CTrailingCT::Operate_8(CMonoid<double> &M,EOperations &O,double &Values[],int &InputIndices[],int &OutputIndices[])
   {
      for(int i=0;i<8;i++)
      {
         m_element.Let();
         if(m_lookback.Get(i,m_element))
         {
            if(!m_element.Get(0,Values[InputIndices[i]]))
            {
               printf(__FUNCSIG__+" Failed to get double for 1 at: "+IntegerToString(i+1));
            }
         }
         else{ printf(__FUNCSIG__+" Failed to get element for 1 at: "+IntegerToString(i+1)); }
      }
      
      //
      
      if(O==OP_LEAST)
      {
         for(int i=0;i<8;i+=2)
         {
            if(Values[InputIndices[i]]<Values[InputIndices[i+1]]){ OutputIndices[i/2]=i; }
            else if(Values[InputIndices[i]]>Values[InputIndices[i+1]]){ OutputIndices[i/2]=i+1; }
            else { OutputIndices[i/2]=m_lookback.Identity(); }
         }
      }
      else if(O==OP_MOST)
      {
         for(int i=0;i<8;i+=2)
         {
            if(Values[InputIndices[i]]>Values[InputIndices[i+1]]){ OutputIndices[i/2]=i; }
            else if(Values[InputIndices[i]]<Values[InputIndices[i+1]]){ OutputIndices[i/2]=i+1; }
            else { OutputIndices[i/2]=m_lookback.Identity(); }
         }
      }
      else if(O==OP_CLOSEST)
      {
         for(int i=0;i<8;i+=2)
         {
            int _index=-1;
            double _mean=0.5*(Values[InputIndices[i]]+Values[InputIndices[i+1]]),_gap=DBL_MAX;
            for(int ii=0;ii<8;ii++)
            {
               if(_gap>fabs(_mean-Values[InputIndices[ii]])){ _gap=fabs(_mean-Values[InputIndices[ii]]); _index=ii;}
            }
            //
            if(_index==-1){ _index=m_lookback.Identity(); }
            
            OutputIndices[i/2]=_index;
         }
      }
      else if(O==OP_FURTHEST)
      {
         for(int i=0;i<8;i+=2)
         {
            int _index=-1;
            double _mean=0.5*(Values[InputIndices[i]]+Values[InputIndices[i+1]]),_gap=0.0;
            for(int ii=0;ii<8;ii++)
            {
               if(_gap<fabs(_mean-Values[InputIndices[ii]])){ _gap=fabs(_mean-Values[InputIndices[ii]]); _index=ii;}
            }
            //
            if(_index==-1){ _index=m_lookback.Identity(); }
            
            OutputIndices[i/2]=_index;
         }
      }
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CTrailingCT::Operate_4(CMonoid<double> &M,EOperations &O,double &Values[],int &InputIndices[],int &OutputIndices[])
   {
      for(int i=0;i<4;i++)
      {
         m_element.Let();
         if(m_lookback.Get(i,m_element))
         {
            /*printf(__FUNCSIG__+
               " values size: "+IntegerToString(ArraySize(Values))+
               " in indices size: "+IntegerToString(ArraySize(InputIndices))+
               " in indices index: "+IntegerToString(InputIndices[i])
               );*/
               
            if(!m_element.Get(0,Values[InputIndices[i]]))
            {
               printf(__FUNCSIG__+" Failed to get double for 1 at: "+IntegerToString(i+1));
            }
         }
         else{ printf(__FUNCSIG__+" Failed to get element for 1 at: "+IntegerToString(i+1)); }
      }
      
      //
      
      if(O==OP_LEAST)
      {
         for(int i=0;i<4;i+=2)
         {
            if(Values[InputIndices[i]]<Values[InputIndices[i+1]]){ OutputIndices[i/2]=i; }
            else if(Values[InputIndices[i]]>Values[InputIndices[i+1]]){ OutputIndices[i/2]=i+1; }
            else { OutputIndices[i/2]=m_lookback.Identity(); }
         }
      }
      else if(O==OP_MOST)
      {
         for(int i=0;i<4;i+=2)
         {
            if(Values[InputIndices[i]]>Values[InputIndices[i+1]]){ OutputIndices[i/2]=i; }
            else if(Values[InputIndices[i]]<Values[InputIndices[i+1]]){ OutputIndices[i/2]=i+1; }
            else { OutputIndices[i/2]=m_lookback.Identity(); }
         }
      }
      else if(O==OP_CLOSEST)
      {
         for(int i=0;i<4;i+=2)
         {
            int _index=-1;
            double _mean=0.5*(Values[InputIndices[i]]+Values[InputIndices[i+1]]),_gap=DBL_MAX;
            for(int ii=0;ii<4;ii++)
            {
               if(_gap>fabs(_mean-Values[InputIndices[ii]])){ _gap=fabs(_mean-Values[InputIndices[ii]]); _index=ii;}
            }
            //
            if(_index==-1){ _index=m_lookback.Identity(); }
            
            OutputIndices[i/2]=_index;
         }
      }
      else if(O==OP_FURTHEST)
      {
         for(int i=0;i<4;i+=2)
         {
            int _index=-1;
            double _mean=0.5*(Values[InputIndices[i]]+Values[InputIndices[i+1]]),_gap=0.0;
            for(int ii=0;ii<4;ii++)
            {
               if(_gap<fabs(_mean-Values[InputIndices[ii]])){ _gap=fabs(_mean-Values[InputIndices[ii]]); _index=ii;}
            }
            //
            if(_index==-1){ _index=m_lookback.Identity(); }
            
            OutputIndices[i/2]=_index;
         }
      }
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CTrailingCT::Operate_2(CMonoid<double> &M,EOperations &O,double &Values[],int &InputIndices[],int &OutputIndices[])
   {
      for(int i=0;i<2;i++)
      {
         m_element.Let();
         if(m_lookback.Get(i,m_element))
         {
            if(!m_element.Get(0,Values[InputIndices[i]]))
            {
               printf(__FUNCSIG__+" Failed to get double for 1 at: "+IntegerToString(i+1));
            }
         }
         else{ printf(__FUNCSIG__+" Failed to get element for 1 at: "+IntegerToString(i+1)); }
      }
      
      //
      
      if(m_lookback_operation==OP_LEAST)
      {
         for(int i=0;i<2;i+=2)
         {
            if(Values[InputIndices[i]]<Values[InputIndices[i+1]]){ OutputIndices[0]=i; }
            else if(Values[InputIndices[i]]>Values[InputIndices[i+1]]){ OutputIndices[0]=i+1; }
            else { OutputIndices[0]=m_lookback.Identity(); }
         }
      }
      else if(m_lookback_operation==OP_MOST)
      {
         for(int i=0;i<2;i+=2)
         {
            if(Values[InputIndices[i]]>Values[InputIndices[i+1]]){ OutputIndices[0]=i; }
            else if(Values[InputIndices[i]]<Values[InputIndices[i+1]]){ OutputIndices[0]=i+1; }
            else { OutputIndices[0]=m_lookback.Identity(); }
         }
      }
      else if(m_lookback_operation==OP_CLOSEST)
      {
         for(int i=0;i<2;i+=2)
         {
            int _index=-1;
            double _mean=0.5*(Values[InputIndices[i]]+Values[InputIndices[i+1]]),_gap=DBL_MAX;
            for(int ii=0;ii<2;ii++)
            {
               if(_gap>fabs(_mean-Values[InputIndices[ii]])){ _gap=fabs(_mean-Values[InputIndices[ii]]); _index=ii;}
            }
            //
            if(_index==-1){ _index=m_lookback.Identity(); }
            
            OutputIndices[0]=_index;
         }
      }
      else if(m_lookback_operation==OP_FURTHEST)
      {
         for(int i=0;i<2;i+=2)
         {
            int _index=-1;
            double _mean=0.5*(Values[InputIndices[i]]+Values[InputIndices[i+1]]),_gap=0.0;
            for(int ii=0;ii<2;ii++)
            {
               if(_gap<fabs(_mean-Values[InputIndices[ii]])){ _gap=fabs(_mean-Values[InputIndices[ii]]); _index=ii;}
            }
            //
            if(_index==-1){ _index=m_lookback.Identity(); }
            
            OutputIndices[0]=_index;
         }
      }
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CTrailingCT::GetLookback()
   {
      m_close.Refresh(-1);
      
      int _x=StartIndex();
      
      for(int i=0;i<8;i++)
      {
         int _period=(__LOOKBACKS[i]*PeriodSeconds(PERIOD_H4))/PeriodSeconds(m_period);
         double _value=fabs(m_close.GetData(_x)-m_close.GetData(_x+_period))/(fabs(m_close.GetData(_x)-m_close.GetData(_x+_period))+fabs(m_close.GetData(_x+_period)-m_close.GetData(_x+_period+_period)));
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!m_lookback.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for lookback. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      //r of 8
      double _v1[8];ArrayInitialize(_v1,0.0);
      int _i1_in[8];for(int i=0;i<8;i++){ _i1_in[i]=i; }
      int _i1_out[4];ArrayInitialize(_i1_out,-1);
      Operate_8(m_lookback,m_lookback_operation,_v1,_i1_in,_i1_out);
      
      
      //r of 4
      double _v2[8];ArrayInitialize(_v2,0.0);
      int _i2_out[2];ArrayInitialize(_i2_out,-1);
      Operate_4(m_lookback,m_lookback_operation,_v2,_i1_out,_i2_out);
      
      
      //r of 2
      double _v3[8];ArrayInitialize(_v3,0.0);
      int _i3_out[1];ArrayInitialize(_i3_out,-1);
      Operate_2(m_lookback,m_lookback_operation,_v2,_i2_out,_i3_out);
      
      return(4*__LOOKBACKS[_i3_out[0]]);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
ENUM_TIMEFRAMES CTrailingCT::GetTimeframe(void)
   {
      for(int i=0;i<8;i++)
      {
         ResetLastError();
         double _value=0.0;
         double _buffer[];ArrayResize(_buffer,3);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
         if(CopyClose(m_symbol.Name(),__TIMEFRAMES[i],0,3,_buffer)>=3)
         {
            _value=fabs(_buffer[0]-_buffer[1])/(fabs(_buffer[0]-_buffer[1])+fabs(_buffer[1]-_buffer[2]));
         }
         else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(__TIMEFRAMES[i])+" close prices. err: "+IntegerToString(GetLastError())); }
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!m_timeframe.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for lookback. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      //r of 8
      double _v1[8];ArrayInitialize(_v1,0.0);
      int _i1_in[8];for(int i=0;i<8;i++){ _i1_in[i]=i; }
      int _i1_out[4];ArrayInitialize(_i1_out,-1);
      Operate_8(m_timeframe,m_timeframe_operation,_v1,_i1_in,_i1_out);
      
      
      //r of 4
      double _v2[8];ArrayInitialize(_v2,0.0);
      int _i2_out[2];ArrayInitialize(_i2_out,-1);
      Operate_4(m_timeframe,m_timeframe_operation,_v2,_i1_out,_i2_out);
      
      
      //r of 2
      double _v3[8];ArrayInitialize(_v3,0.0);
      int _i3_out[1];ArrayInitialize(_i3_out,-1);
      Operate_2(m_timeframe,m_timeframe_operation,_v2,_i2_out,_i3_out);
      
      return(__TIMEFRAMES[_i3_out[0]]);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
ENUM_APPLIED_PRICE CTrailingCT::GetAppliedprice(int Lookback,ENUM_TIMEFRAMES Timeframe)
   {
      for(int i=0;i<4;i++)
      {
         ResetLastError();
         double _value=0.0;
         if(__APPLIEDPRICES[i]==PRICE_CLOSE)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            if(CopyClose(m_symbol.Name(),Timeframe,0,Lookback,_buffer)>=Lookback)
            {
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" close prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(__APPLIEDPRICES[i]==PRICE_OPEN)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            if(CopyOpen(m_symbol.Name(),Timeframe,0,Lookback,_buffer)>=Lookback)
            {
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" open prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(__APPLIEDPRICES[i]==PRICE_MEDIAN)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            double _h[];ArrayResize(_h,Lookback);ArrayInitialize(_h,0.0);
            double _l[];ArrayResize(_l,Lookback);ArrayInitialize(_l,0.0);
            if(CopyHigh(m_symbol.Name(),Timeframe,0,Lookback,_h)>=Lookback && CopyLow(m_symbol.Name(),Timeframe,0,Lookback,_l)>=Lookback)
            {
               for(int ii=0;ii<Lookback;ii++)
               {
                  _buffer[ii]=0.5*(_h[ii]+_l[ii]);
               }
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(__APPLIEDPRICES[i]==PRICE_TYPICAL)
         {
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            double _h[];ArrayResize(_h,Lookback);ArrayInitialize(_h,0.0);
            double _l[];ArrayResize(_l,Lookback);ArrayInitialize(_l,0.0);
            double _c[];ArrayResize(_c,Lookback);ArrayInitialize(_c,0.0);
            if(CopyHigh(m_symbol.Name(),Timeframe,0,Lookback,_h)>=Lookback && CopyLow(m_symbol.Name(),Timeframe,0,Lookback,_l)>=Lookback && CopyClose(m_symbol.Name(),Timeframe,0,Lookback,_c)>=Lookback)
            {
               for(int ii=0;ii<Lookback;ii++)
               {
                  _buffer[ii]=(1.0/3.0)*(_h[ii]+_l[ii]+_c[ii]);
               }
               _value=MathStandardDeviation(_buffer);
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" high and/or low prices. err: "+IntegerToString(GetLastError())); }
         }
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!m_appliedprice.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for lookback. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      //r of 4
      double _v1[4];ArrayInitialize(_v1,0.0);
      int _i1_in[4];for(int i=0;i<4;i++){ _i1_in[i]=i; }
      int _i1_out[2];ArrayInitialize(_i1_out,-1);
      Operate_4(m_appliedprice,m_appliedprice_operation,_v1,_i1_in,_i1_out);
      
      
      //r of 2
      double _v2[4];ArrayInitialize(_v2,0.0);
      int _i2_out[1];ArrayInitialize(_i2_out,-1);
      Operate_2(m_appliedprice,m_appliedprice_operation,_v2,_i1_out,_i2_out);
      
      return(__APPLIEDPRICES[_i2_out[0]]);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CTrailingCT::GetIndicator(int Lookback,ENUM_TIMEFRAMES Timeframe,ENUM_APPLIED_PRICE Appliedprice)
   {
      m_spread.Refresh(-1);
      
      int _x=StartIndex();
      
      double _indicator[2];
      ArrayInitialize(_indicator,0.0);
      
      for(int i=0;i<2;i++)
      {
         ResetLastError();
         double _value=0.0;
         //
         if(__INDICATORS[i]=="RSI")
         {
            double _spread=(m_symbol.Point()*m_spread.GetData(_x));
            double _current_gain=_spread,_current_loss=_spread;
            double _gain=_spread,_loss=_spread;
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            if(CopyClose(m_symbol.Name(),Timeframe,0,Lookback,_buffer)>=Lookback)
            {
               if(_buffer[0]>_buffer[1]){ _current_gain+=_buffer[0]-_buffer[1]; }
               else if(_buffer[1]>_buffer[0]){ _current_loss+=_buffer[1]-_buffer[0]; }
               
               for(int ii=1;ii<Lookback-1;ii++)
               {
                  if(_buffer[ii]>_buffer[ii+1]){ _gain+=_buffer[ii]-_buffer[ii+1]; }
                  else if(_buffer[ii+1]>_buffer[ii]){ _loss+=_buffer[ii+1]-_buffer[ii]; }
               }
               
               _gain/=(Lookback-1);
               _loss/=(Lookback-1);
               
               _gain+=_current_gain;
               _loss+=_current_loss;
               
               _value=(_gain/_loss);
               _value+=1.0;
               _value=100.0/_value;
               _value=100.0-_value;
               
               _indicator[0]=_value;
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" close prices. err: "+IntegerToString(GetLastError())); }
         }
         else if(__INDICATORS[i]=="BOLLINGER_BANDS")
         {
            double _base=0.0,_upper=0.0,_lower=0.0;
            double _buffer[];ArrayResize(_buffer,Lookback);ArrayInitialize(_buffer,0.0);ArraySetAsSeries(_buffer,true);
            if(CopyClose(m_symbol.Name(),Timeframe,0,Lookback,_buffer)>=Lookback)
            {
               for(int ii=0;ii<Lookback;ii++)
               {
                  _base+=_buffer[ii];
               }
               
               _base/=(Lookback);
               
               double _deviation=MathStandardDeviation(_buffer);
               
               _upper+=(_base+(2.0*_deviation));
               _lower+=(_base-(2.0*_deviation));
               
               _value=((1.0+((_buffer[0]-_base)/(fabs(_buffer[0]-_base)+(_upper-_lower))))*50.0);
               
               _indicator[1]=_value;
            }
            else{ printf(__FUNCSIG__+" Failed to copy: "+EnumToString(Timeframe)+" close prices. err: "+IntegerToString(GetLastError())); }
         }
         
         //
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!m_indicator.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for lookback. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      //r of 2
      double _v1[2];ArrayInitialize(_v1,0.0);
      int _i1_in[2];for(int i=0;i<2;i++){ _i1_in[i]=i; }
      int _i1_out[1];ArrayInitialize(_i1_out,-1);
      Operate_2(m_indicator,m_indicator_operation,_v1,_i1_in,_i1_out);
      
      return(_indicator[_i1_out[0]]);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CTrailingCT::GetDecision(int Lookback,double Indicator)
   {
      m_close.Refresh(-1);
      
      int _x=StartIndex();
      
      for(int i=0;i<2;i++)
      {
         ResetLastError();
         double _value=0.0;
         //
         int _max_index=-1,_min_index=-1;
         double _max=m_close.MaxValue(_x,Lookback,_max_index);
         double _min=m_close.MinValue(_x,Lookback,_min_index);
         
         if(__DECISIONS[i]=="TREND")
         {
            int _trends=0;
            if(m_close.GetData(_x)>m_close.GetData(_x+Lookback))
            {
               for(int ii=_x;ii<_x+Lookback;ii++)
               {
                  if(m_close.GetData(ii)>m_close.GetData(ii+1))
                  {
                     _trends++;
                     _value+=(m_close.GetData(ii)-m_close.GetData(ii+1));
                  }
               }
            }
            else if(m_close.GetData(_x)<m_close.GetData(_x+Lookback))
            {
               for(int ii=_x;ii<_x+Lookback;ii++)
               {
                  if(m_close.GetData(ii)<m_close.GetData(ii+1))
                  {
                     _trends++;
                     _value+=(m_close.GetData(ii+1)-m_close.GetData(ii));
                  }
               }
            }
            
            if(_trends>0){ _value/=_trends; }
            if(_max-_min>0.0){ _value/=(_max-_min); }
         }
         else if(__DECISIONS[i]=="RANGE")
         {
            int _ranges=0;
            if(m_close.GetData(_x)>m_close.GetData(_x+Lookback))
            {
               for(int ii=_x;ii<_x+Lookback;ii++)
               {
                  if(m_close.GetData(ii)<m_close.GetData(ii+1))
                  {
                     _ranges++;
                     _value+=(m_close.GetData(ii+1)-m_close.GetData(ii));
                  }
               }
            }
            else if(m_close.GetData(_x)<m_close.GetData(_x+Lookback))
            {
               for(int ii=_x;ii<_x+Lookback;ii++)
               {
                  if(m_close.GetData(ii)>m_close.GetData(ii+1))
                  {
                     _ranges++;
                     _value+=(m_close.GetData(ii)-m_close.GetData(ii+1));
                  }
               }
            }
            
            if(_ranges>0){ _value/=_ranges; }
            if(_max-_min>0.0){ _value/=(_max-_min); }
         }
         
         //
         
         m_element.Let();
         m_element.Cardinality(1);
         if(m_element.Set(0,_value))
         {
            ResetLastError();
            if(!m_decision.Set(i,m_element,true))
            {
               printf(__FUNCSIG__+" Failed to assign element at index: "+IntegerToString(i)+", for lookback. ERR: "+IntegerToString(GetLastError()));
            }
         }
      }
      
      //r of 2
      double _v1[2];ArrayInitialize(_v1,0.0);
      int _i1_in[2];for(int i=0;i<2;i++){ _i1_in[i]=i; }
      int _i1_out[1];ArrayInitialize(_i1_out,-1);
      Operate_2(m_decision,m_decision_operation,_v1,_i1_in,_i1_out);
      
      double _decision=0.0;
      
      if(_i1_out[0]==0)
      {
         if(m_close.GetData(_x)>m_close.GetData(_x+Lookback))
         {
            _decision=Indicator;
         }
         else if(m_close.GetData(_x)<m_close.GetData(_x+Lookback))
         {
            _decision=100.0-Indicator;
         }
      }
      else if(_i1_out[0]==1)
      {
         if(m_close.GetData(_x)>m_close.GetData(_x+Lookback))
         {
            _decision=100.0-Indicator;
         }
         else if(m_close.GetData(_x)<m_close.GetData(_x+Lookback))
         {
            _decision=Indicator;
         }
      }
      
      return(int(round(_decision)));
   }
//+------------------------------------------------------------------+
