//+------------------------------------------------------------------+
//|                                                     SignalMA.mqh |
//|                   Copyright 2009-2013, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Expert\ExpertSignal.mqh>
#include <Expert\Signal\My\Cct.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Signals crudely based on 'Category Theory r1'              |
//| Type=SignalAdvanced                                              |
//| Name=Category Theory                                             |
//| ShortName=CT                                                     |
//| Class=CSignalCT                                                  |
//| Page=signal_ct                                                   |
//| Parameter=Periods,int,24,Period of calculation                   |
//| Parameter=Applied,ENUM_APPLIED_PRICE,PRICE_CLOSE,Prices series   |
//| Parameter=LongDMK,double,1.0,Long DMK Weight                     |
//| Parameter=ShortDMK,double,1.0,Short DMK Weight                   |
//| Parameter=LongWPR,double,1.0,Long WPR Weight                 |
//| Parameter=ShortWPR,double,1.0,Short WPR Weight               |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CSignalCT.                                                 |
//| Purpose: Class of generator of trade signals based on            |
//|          the 'Category Theory r1' indicator.                     |
//| Is derived from the CExpertSignal class.                         |
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CSignalCT : public CExpertSignal
  {
protected:
   CiDeMarker        m_dmk;            // object-oscillator (first corner)
   CiWPR             m_wpr;            // object-indicator (second corner)
   //--- adjusted parameters
   int               m_periods;        // the "period of calculation" parameter of the oscillator & indicator
   ENUM_APPLIED_PRICE m_applied;       // the "prices series" parameter of the oscillator & indicator
   double            m_longdmk;        // long dmk weight
   double            m_shortdmk;       // short dmk weight
   double            m_longwpr;        // long wpr weight
   double            m_shortwpr;       // short wpr weight

public:
   //--- methods of setting adjustable parameters
   void              Periods(int value)                { m_periods=value;  }
   void              Applied(ENUM_APPLIED_PRICE value) { m_applied=value;  }
   
   void              LongDMK(double value) { m_longdmk=value;  }
   void              ShortDMK(double value) { m_shortdmk=value;  }
   void              LongWPR(double value) { m_longwpr=value;  }
   void              ShortWPR(double value) { m_shortwpr=value;  }
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //--- methods of checking if the market models are formed
   virtual int       LongCondition(void);
   virtual int       ShortCondition(void);
                     CSignalCT(void);
                    ~CSignalCT(void);

protected:
                    
   virtual void      LongMorphism(void);
   virtual void      ShortMorphism(void);
   
   virtual double    Product(ENUM_POSITION_TYPE Position);
   
   NCT::
   CDomain<double>   long_product,short_product;
   //--- method of initialization of the oscillator
   bool              InitDMK(CIndicators *indicators);
   bool              InitWPR(CIndicators *indicators);
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CSignalCT::CSignalCT(void)
  {
//--- initialization of protected data
   m_used_series=USE_SERIES_HIGH+USE_SERIES_LOW+USE_SERIES_CLOSE+USE_SERIES_TIME;
   
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CSignalCT::~CSignalCT(void)
  {
  }
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CSignalCT::ValidationSettings(void)
  {
//--- validation settings of additional filters
   if(!CExpertSignal::ValidationSettings())
      return(false);
//--- 
   if(m_periods<=0)
      return(false);
//--- 
   if(m_longdmk<0.1||m_longdmk>2.0
      ||m_shortdmk<0.1||m_shortdmk>2.0
      ||m_longwpr<0.1||m_longwpr>2.0
      ||m_shortwpr<0.1||m_shortwpr>2.0
      )
      return(false);

//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CSignalCT::InitIndicators(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- initialization of indicators and timeseries of additional filters
   if(!CExpertSignal::InitIndicators(indicators))
      return(false);
//--- create and initialize DMK oscillator
   if(!InitDMK(indicators))
      return(false);
//--- create and initialize WPR oscillator
   if(!InitWPR(indicators))
      return(false);
//--- 
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Initialize DMK oscillators.                                      |
//+------------------------------------------------------------------+
bool CSignalCT::InitDMK(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- add object to collection
   if(!indicators.Add(GetPointer(m_dmk)))
     {
      printf(__FUNCTION__+": error adding DMK object");
      return(false);
     }
//--- initialize object
   if(!m_dmk.Create(m_symbol.Name(),m_period,m_periods))
     {
      printf(__FUNCTION__+": error DMK initializing object");
      return(false);
     }
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Initialize DMK oscillators.                                      |
//+------------------------------------------------------------------+
bool CSignalCT::InitWPR(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- add object to collection
   if(!indicators.Add(GetPointer(m_wpr)))
     {
      printf(__FUNCTION__+": error adding WPR object");
      return(false);
     }
//--- initialize object
   if(!m_wpr.Create(m_symbol.Name(),m_period,m_periods))
     {
      printf(__FUNCTION__+": error initializing WPR object");
      return(false);
     }
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| "Voting" that price will grow.                                   |
//+------------------------------------------------------------------+
int CSignalCT::LongCondition(void)
  {
      int result=0;
      
      //Using Domains Indicator biases (long or short)
      //e.g. an DMK reading of 75 => long-25, short-75 
      //or price at upper WPR => long-0, short-100
      LongMorphism();
      
      result=int(round(Product(POSITION_TYPE_BUY)));
      
      return(result);
  }
//+------------------------------------------------------------------+
//| "Voting" that price will fall.                                   |
//+------------------------------------------------------------------+
int CSignalCT::ShortCondition(void)
  {
      int result=0;
      
      ShortMorphism();
      
      result=int(round(Product(POSITION_TYPE_SELL)));
      
      return(result);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSignalCT::LongMorphism(void)
   {
      int _index=StartIndex();
      
      m_wpr.Refresh(-1);
      m_dmk.Refresh(-1);
      m_close.Refresh(-1);
      
      double _wpr=-1.0*(m_dmk.GetData(0,_index)/100.0);
      double _dmk=(1.0-m_dmk.GetData(0,_index));
      
      NCT::CElement<double> _e;
      _e.Cardinality(2);
      _e.Set(0,_dmk);_e.Set(1,_wpr);
      
      long_product.Cardinality(1);
      long_product.Set(0,_e,true);
   }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CSignalCT::ShortMorphism(void)
   {
      int _index=StartIndex();
      
      m_wpr.Refresh(-1);
      m_dmk.Refresh(-1);
      m_close.Refresh(-1);
      
      double _wpr=-1.0+((m_dmk.GetData(0,_index))/100.0);
      double _dmk=(m_dmk.GetData(0,_index));
      
      NCT::CElement<double> _e;
      _e.Cardinality(2);
      _e.Set(0,_dmk);_e.Set(1,_wpr);
      
      short_product.Cardinality(1);
      short_product.Set(0,_e,true);
   }
//+------------------------------------------------------------------+
//| Morphisms at Product                                             |
//+------------------------------------------------------------------+
double CSignalCT::Product(ENUM_POSITION_TYPE Position)
   {
      double _product=0.0;
      
      NCT::CElement<double> _e;
   
      if(Position==POSITION_TYPE_BUY)
      {
         if(long_product.Cardinality()>=1 && long_product.Get(0,_e))
         {
            _product=100.0*((m_longdmk*_e.Get(0))+(m_longwpr*_e.Get(1)))/(m_longdmk+m_longwpr);
         }
         
         return(_product);
      }
      
      if(short_product.Cardinality()>=1 && short_product.Get(0,_e))
      {
         _product=100.0*((m_shortdmk*_e.Get(0))+(m_shortwpr*_e.Get(1)))/(m_shortdmk+m_shortwpr);
      }
         
      return(_product);
   }
//+------------------------------------------------------------------+
