//+------------------------------------------------------------------+
//|                                         TrailingParabolicSAR.mqh |
//|                   Copyright 2009-2013, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Math\Alglib\dataanalysis.mqh>
#include <Expert\ExpertTrailing.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Trailing Stop based on 'Discriminant Analysis'             |
//| Type=Trailing                                                    |
//| Name=DiscriminantAnalysis                                        |
//| Class=CTrailingDA                                                |
//| Page=                                                            |
//| Parameter=TrailingPoints,int,24,Trailing Data Points             |
//| Parameter=TrailingType,int,1,Trailing Data Type                  |
//| Parameter=TrailingRegulizer,double,0.05,Trailing Regulizer       |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CTrailingDA.                                               |
//| Appointment: Class traling stops with 'Discriminant Analysis'.   |
//| Derives from class CExpertTrailing.                              |
//+------------------------------------------------------------------+
#define     __T_VARS 5
#define     __T_CLASSES 3

#define     __T_BULLISH 2
#define     __T_WHIPSAW 1
#define     __T_BEARISH 0
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CTrailingDA : public CExpertTrailing
  {
   
protected:
   CLDA              LDA;              // object-indicator
   
   //--- adjusted parameters
   int               m_trailing_points;      // the data points
   int               m_trailing_type;        // the data type
   double            m_trailing_regulizer;   // the neutrality definer 

public:
                     CTrailingDA(void);
                    ~CTrailingDA(void);
   //--- methods of setting adjustable parameters
   void              TrailingPoints(int value)                 { m_trailing_points=value;        }
   void              TrailingType(int value)                   { m_trailing_type=value;          }
   void              TrailingRegulizer(double value)           { m_trailing_regulizer=value;     }
   //--- method of verification of settings
   virtual bool      ValidationSettings(void);
   //--- method of creating the indicator and timeseries
   //--- method of creating the indicator and timeseries
   virtual bool      InitIndicators(CIndicators *indicators);
   //---
   virtual bool      CheckTrailingStopLong(CPositionInfo *position,double &sl,double &tp);
   virtual bool      CheckTrailingStopShort(CPositionInfo *position,double &sl,double &tp);
   
   double            m_long_sl;
   double            m_short_sl;

protected:
   
   double            ProcessDA(int Index,ENUM_POSITION_TYPE Type);
   double            Data(int Index,ENUM_POSITION_TYPE Type,bool Variables=true);
   double            Range(int Index){ int _h,_l; return(m_high.MaxValue(StartIndex()+Index,2,_h)-m_low.MinValue(StartIndex()+Index,2,_l)); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CTrailingDA::CTrailingDA(void) : m_trailing_points(24),m_trailing_type(1),m_trailing_regulizer(0.05)

  {
//--- initialization of protected data
   m_used_series=USE_SERIES_OPEN+USE_SERIES_HIGH+USE_SERIES_LOW+USE_SERIES_CLOSE;
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CTrailingDA::~CTrailingDA(void)
  {
  }
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CTrailingDA::ValidationSettings(void)
  {
//--- validation settings of additional filters
   if(!CExpertTrailing::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_trailing_points<3)
     {
      printf(__FUNCTION__+": trailing points must be greater than 2");
      return(false);
     }
   if(m_trailing_type>=4)
     {
      printf(__FUNCTION__+": trailing type must be less than 4");
      return(false);
     }
   if(m_trailing_regulizer<=0.0)
     {
      printf(__FUNCTION__+": trailing regulizer must be greater than 0.0");
      return(false);
     }
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Create indicators.                                               |
//+------------------------------------------------------------------+
bool CTrailingDA::InitIndicators(CIndicators *indicators)
  {
//--- check pointer
   if(indicators==NULL)
      return(false);
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Discriminant Ananlysis method                                    |
//| INPUT PARAMETERS                                                 |
//|     Index   -   int, read index within price buffer.             |
//|                                                                  |
//|     Type    -   position type for determining use of             |
//|                 lows or highs.                                   |
//| OUTPUT                                                           |
//|     double  -   projected ideal stop loss                        |
//+------------------------------------------------------------------+
double CTrailingDA::ProcessDA(int Index,ENUM_POSITION_TYPE Type)
   {
      double _da=0.0;
      
      int _info=0;
      CMatrixDouble _w,_xy,_z;
      _xy.Resize(m_trailing_points,__T_VARS+1);
      
      for(int p=0;p<m_trailing_points;p++)
      {
         for(int v=0;v<__T_VARS;v++)
         {
            _xy[p].Set(v,Data(Index+p+v+1,Type));
         }
         
         _xy[p].Set(__T_VARS,Data(Index+p,Type,false));
      }
      
      LDA.FisherLDAN(_xy,m_trailing_points,__T_VARS,__T_CLASSES,_info,_w);
      
      if(_info>0)
      {
         double _centroids[__T_CLASSES],_unknown_centroid=0.0; ArrayInitialize(_centroids,0.0);
         
         _z.Resize(1,6);
         for(int v=0;v<__T_VARS;v++)
         {
            _z[0].Set(v,Data(Index,Type));
         }
         
         for(int p=0;p<m_trailing_points;p++)
         {
            for(int v=0;v<__T_VARS;v++)
            {
               _centroids[int(_xy[p][5])]+= (_w[0][v]*_xy[p][v]);
            }
         }
         
         // best vector is the first 
         for(int v=0;v<__T_VARS;v++){ _unknown_centroid+= (_w[0][v]*_z[0][v]); }
         
         if(fabs(_centroids[__T_BULLISH]-_unknown_centroid)<fabs(_centroids[__T_BEARISH]-_unknown_centroid) && fabs(_centroids[__T_BULLISH]-_unknown_centroid)<fabs(_centroids[__T_WHIPSAW]-_unknown_centroid))
         {
            _da=(1.0-(fabs(_centroids[__T_BULLISH]-_unknown_centroid)/(fabs(_centroids[__T_BULLISH]-_unknown_centroid)+fabs(_centroids[__T_WHIPSAW]-_unknown_centroid)+fabs(_centroids[__T_BEARISH]-_unknown_centroid))));
         }
         else if(fabs(_centroids[__T_BEARISH]-_unknown_centroid)<fabs(_centroids[__T_BULLISH]-_unknown_centroid) && fabs(_centroids[__T_BEARISH]-_unknown_centroid)<fabs(_centroids[__T_WHIPSAW]-_unknown_centroid))
         {
            _da=-1.0*(1.0-(fabs(_centroids[__T_BEARISH]-_unknown_centroid)/(fabs(_centroids[__T_BULLISH]-_unknown_centroid)+fabs(_centroids[__T_WHIPSAW]-_unknown_centroid)+fabs(_centroids[__T_BEARISH]-_unknown_centroid))));
         }
         
         int _index   =StartIndex();
         double _min_l=Low(_index),_max_l=Low(_index),_min_h=High(_index),_max_h=High(_index);
         
         for(int p=_index;p<m_trailing_points+_index;p++)
         {
            _min_l=fmin(_min_l,Low(p));
            _max_l=fmax(_max_l,Low(p));
            _min_h=fmin(_min_h,High(p));
            _max_h=fmax(_max_h,High(p));
         }
         
         if(Type==POSITION_TYPE_BUY)
         {
            _da*=(_max_l-_min_l);
            _da+=_min_l;
         }
         else if(Type==POSITION_TYPE_SELL)
         {
            _da*=(_max_h-_min_h);
            _da+=_max_h;
         }
      }
      else
      {
         if(_info==-4){ printf(__FUNCSIG__+" internal EVD subroutine hasn't converged "); }
         else if(_info==-2){ printf(__FUNCSIG__+" there is a point with class number outside of [0..NClasses-1] "); }
         else if(_info==-1){ printf(__FUNCSIG__+" incorrect parameters was passed  either: NPoints<0, or NVars<1, or NClasses<2) "); }
      }
      
      return(_da);
   }
//+------------------------------------------------------------------+
//| Data Set method                                                  |
//| INPUT PARAMETERS                                                 |
//|     Index   -   int, read index within price buffer.             |
//|                                                                  |
//|     Type    -   position type for determining use of             |
//|                 lows or highs.                                   |
//|                                                                  |
//|     Variables                                                    |
//|             -   whether data component is variables or .         |
//|                  classifier.                                     |
//| OUTPUT                                                           |
//|     double  -   Data depending on data set type                  |
//|                                                                  |
//| DATA SET TYPES                                                   |
//| 1. Discretized variables.                                        |
//|                                                                  |
//| 2. Normalized variables.                                         |
//|                                                                  |
//| 3. Continuized variables.                                        |
//|                                                                  |
//| 4. Raw data variables.                                           |
//|                                                                  |
//+------------------------------------------------------------------+
double CTrailingDA::Data(int Index,ENUM_POSITION_TYPE Type,bool Variables=true)
   {
      m_low.Refresh(-1);
      m_high.Refresh(-1);
      
      if(Variables)
      {
         if(Type==POSITION_TYPE_BUY)
         {
            if(m_trailing_type==0)
            {
               return(fabs(Low(StartIndex()+Index)-Low(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index)?1.0:((Low(StartIndex()+Index)>Low(StartIndex()+Index+1))?2.0:0.0));
            }
            else if(m_trailing_type==1)
            {
               if(fabs(Low(StartIndex()+Index)-Low(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index))
               {
                  return(0.0);
               }
               return((Low(StartIndex()+Index)-Low(StartIndex()+Index+1))/fmax(m_symbol.Point(),fmax(High(StartIndex()+Index),High(StartIndex()+Index+1))-fmin(Low(StartIndex()+Index),Low(StartIndex()+Index+1))));
            }
            else if(m_trailing_type==2)
            {
               if(fabs(Low(StartIndex()+Index)-Low(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index))
               {
                  return(0.0);
               }
               return(Low(StartIndex()+Index)-Low(StartIndex()+Index+1));
            }
            else if(m_trailing_type==3)
            {
               if(fabs(Low(StartIndex()+Index)-Low(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index))
               {
                  return(Low(StartIndex()+Index+1));
               }
               return(Low(StartIndex()+Index));
            }
         }
         else if(Type==POSITION_TYPE_SELL)
         {
            if(m_trailing_type==0)
            {
               return(fabs(High(StartIndex()+Index)-High(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index)?1.0:((High(StartIndex()+Index)>High(StartIndex()+Index+1))?2.0:0.0));
            }
            else if(m_trailing_type==1)
            {
               if(fabs(High(StartIndex()+Index)-High(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index))
               {
                  return(0.0);
               }
               return((High(StartIndex()+Index)-High(StartIndex()+Index+1))/fmax(m_symbol.Point(),fmax(High(StartIndex()+Index),High(StartIndex()+Index+1))-fmin(Low(StartIndex()+Index),Low(StartIndex()+Index+1))));
            }
            else if(m_trailing_type==2)
            {
               if(fabs(High(StartIndex()+Index)-High(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index))
               {
                  return(0.0);
               }
               return(High(StartIndex()+Index)-High(StartIndex()+Index+1));
            }
            else if(m_trailing_type==3)
            {
               if(fabs(High(StartIndex()+Index)-High(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index))
               {
                  return(High(StartIndex()+Index+1));
               }
               return(High(StartIndex()+Index));
            }
         }
      }
      
      if(Type==POSITION_TYPE_BUY)
      {
         return(fabs(Low(StartIndex()+Index)-Low(StartIndex()+Index+1))<m_trailing_regulizer*(High(StartIndex()+Index+1)-Low(StartIndex()+Index+1))?1.0:((Low(StartIndex()+Index)>Low(StartIndex()+Index+1))?2.0:0.0));
      }
      
      return(fabs(High(StartIndex()+Index)-High(StartIndex()+Index+1))<m_trailing_regulizer*Range(Index)?1.0:((High(StartIndex()+Index)>High(StartIndex()+Index+1))?2.0:0.0));
   }
//+------------------------------------------------------------------+
//| Checking trailing stop and/or profit for long position.          |
//+------------------------------------------------------------------+
bool CTrailingDA::CheckTrailingStopLong(CPositionInfo *position,double &sl,double &tp)
  {
//--- check
   if(position==NULL)
      return(false);
//---
   m_long_sl=ProcessDA(StartIndex(),POSITION_TYPE_BUY);
   
   //printf(__FUNCSIG__+" sl: "+DoubleToString(m_long_sl,m_symbol.Digits()));

   double level =NormalizeDouble(m_symbol.Bid()-(m_symbol.StopsLevel()*m_symbol.Point()),m_symbol.Digits());
   double new_sl=NormalizeDouble(m_long_sl,m_symbol.Digits());
   //printf(__FUNCSIG__+" new sl: "+DoubleToString(new_sl,m_symbol.Digits()));
   double pos_sl=position.StopLoss();
   double base  =(pos_sl==0.0) ? position.PriceOpen() : pos_sl;
   //printf(__FUNCSIG__+" base: "+DoubleToString(base,m_symbol.Digits()));
//---
   sl=EMPTY_VALUE;
   tp=EMPTY_VALUE;
   if(new_sl>base && new_sl<level)
      sl=new_sl;
//---
   return(sl!=EMPTY_VALUE);
  }
//+------------------------------------------------------------------+
//| Checking trailing stop and/or profit for short position.         |
//+------------------------------------------------------------------+
bool CTrailingDA::CheckTrailingStopShort(CPositionInfo *position,double &sl,double &tp)
  {
//--- check
   if(position==NULL)
      return(false);
//---
   m_short_sl=ProcessDA(StartIndex(),POSITION_TYPE_SELL);
   
   //printf(__FUNCSIG__+" sl: "+DoubleToString(m_short_sl,m_symbol.Digits()));

   double level =NormalizeDouble(m_symbol.Ask()+m_symbol.StopsLevel()*m_symbol.Point(),m_symbol.Digits());
   double new_sl=NormalizeDouble(m_short_sl+(m_symbol.Spread()*m_symbol.Point()),m_symbol.Digits());
   //printf(__FUNCSIG__+" new sl: "+DoubleToString(new_sl,m_symbol.Digits()));
   double pos_sl=position.StopLoss();
   double base  =(pos_sl==0.0) ? position.PriceOpen() : pos_sl;
   //printf(__FUNCSIG__+" base: "+DoubleToString(base,m_symbol.Digits()));
//---
   sl=EMPTY_VALUE;
   tp=EMPTY_VALUE;
   if(new_sl<base && new_sl>level)
      sl=new_sl;
//---
   return(sl!=EMPTY_VALUE);
  }
//+------------------------------------------------------------------+
