//+------------------------------------------------------------------+
//|                                                      MoneyDA.mqh |
//|                   Copyright 2009-2017, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include <Math\Alglib\dataanalysis.mqh>
#include <Expert\ExpertMoney.mqh>
#include <Trade\DealInfo.mqh>
// wizard description start
//+------------------------------------------------------------------+
//| Description of the class                                         |
//| Title=Trading with 'Discriminant Analysis' optimized trade volume|
//| Type=Money                                                       |
//| Name=DiscriminantAnalysis                                        |
//| Class=CMoneyDA                                                   |
//| Page=                                                            |
//| Parameter=MoneyPoints,int,5,Money Data Points                    |
//| Parameter=MoneyType,int,1,Money Data Type                        |
//| Parameter=MoneyRegulizer,double,0.05,Money Regulizer             |
//| Parameter=DecreaseFactor,double,3.0,Decrease factor              |
//| Parameter=Percent,double,12.5,Percent                            |
//+------------------------------------------------------------------+
// wizard description end
//+------------------------------------------------------------------+
//| Class CMoneyDA.                                                  |
//| Purpose: Class of money management with 'Discriminant Analysis'  |
//|          optimized trade volume. Derives from class CExpertMoney.|
//+------------------------------------------------------------------+
#define     __M_VARS 5
#define     __M_CLASSES 3

#define     __M_BULLISH 2
#define     __M_WHIPSAW 1
#define     __M_BEARISH 0
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CMoneyDA : public CExpertMoney
  {
protected:
   CLDA              LDA;              // object-indicator
   //--- adjusted parameters
   int               m_money_points;      // the data points
   int               m_money_type;        // the data type
   double            m_money_regulizer;   // the neutrality definer 
   
   double            m_decrease_factor;

public:
                     CMoneyDA(void);
                    ~CMoneyDA(void);
   //---
   void              MoneyPoints(int value)                 { m_money_points=value;        }
   void              MoneyType(int value)                   { m_money_type=value;          }
   void              MoneyRegulizer(double value)           { m_money_regulizer=value;     }
   void              DecreaseFactor(double decrease_factor) { m_decrease_factor=decrease_factor; }
   virtual bool      ValidationSettings(void);
   //---
   virtual double    CheckOpenLong(double price,double sl);
   virtual double    CheckOpenShort(double price,double sl);

protected:
   double            Optimize(double lots);

protected:
   
   double            ProcessDA(int Index);
   double            Data(int Index,bool Variables=true);
   double            Range(int Index){ int _h,_l; return(m_high.MaxValue(StartIndex()+Index,2,_h)-m_low.MinValue(StartIndex()+Index,2,_l)); }
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
void CMoneyDA::CMoneyDA(void) : m_decrease_factor(3.0),m_money_points(5),m_money_regulizer(0.05)
  {
//--- initialization of protected data
   m_used_series=USE_SERIES_OPEN+USE_SERIES_HIGH+USE_SERIES_LOW+USE_SERIES_CLOSE;
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
void CMoneyDA::~CMoneyDA(void)
  {
  }
//+------------------------------------------------------------------+
//| Validation settings protected data.                              |
//+------------------------------------------------------------------+
bool CMoneyDA::ValidationSettings(void)
  {
   if(!CExpertMoney::ValidationSettings())
      return(false);
//--- initial data checks
   if(m_decrease_factor<=0.0)
     {
      printf(__FUNCTION__+": decrease factor must be greater then 0");
      return(false);
     }
     
   if(m_money_points<3)
     {
      printf(__FUNCTION__+": money points must be greater than 2");
      return(false);
     }
   if(m_money_type>=4)
     {
      printf(__FUNCTION__+": money type must be less than 4");
      return(false);
     }
   if(m_money_regulizer<=0.0)
     {
      printf(__FUNCTION__+": money regulizer must be greater than 0.0");
      return(false);
     }
//--- ok
   return(true);
  }
//+------------------------------------------------------------------+
//| Discriminant Ananlysis method                                    |
//| INPUT PARAMETERS                                                 |
//|     Index   -   int, read index within price buffer.             |
//|                                                                  |
//| OUTPUT                                                           |
//|     double  -   projected price bar range                        |
//+------------------------------------------------------------------+
double CMoneyDA::ProcessDA(int Index)
   {
      double _da=0.0;
      
      int _info=0;
      CMatrixDouble _w,_xy,_z;
      _xy.Resize(m_money_points,__M_VARS+1);
      
      for(int p=0;p<m_money_points;p++)
      {
         for(int v=0;v<__M_VARS;v++)
         {
            _xy[p].Set(v,Data(Index+p+v+1));
         }
         
         _xy[p].Set(__M_VARS,Data(Index+p,false));
      }
      
      LDA.FisherLDAN(_xy,m_money_points,__M_VARS,__M_CLASSES,_info,_w);
      
      if(_info>0)
      {
         double _centroids[__M_CLASSES],_unknown_centroid=0.0; ArrayInitialize(_centroids,0.0);
         
         _z.Resize(1,6);
         for(int v=0;v<__M_VARS;v++)
         {
            _z[0].Set(v,Data(Index));
         }
         
         for(int p=0;p<m_money_points;p++)
         {
            for(int v=0;v<__M_VARS;v++)
            {
               _centroids[int(_xy[p][5])]+= (_w[0][v]*_xy[p][v]);
            }
         }
         
         // best vector is the first 
         for(int v=0;v<__M_VARS;v++){ _unknown_centroid+= (_w[0][v]*_z[0][v]); }
         
         if(fabs(_centroids[__M_BULLISH]-_unknown_centroid)<fabs(_centroids[__M_BEARISH]-_unknown_centroid) && fabs(_centroids[__M_BULLISH]-_unknown_centroid)<fabs(_centroids[__M_WHIPSAW]-_unknown_centroid))
         {
            _da=(1.0-(fabs(_centroids[__M_BULLISH]-_unknown_centroid)/(fabs(_centroids[__M_BULLISH]-_unknown_centroid)+fabs(_centroids[__M_WHIPSAW]-_unknown_centroid)+fabs(_centroids[__M_BEARISH]-_unknown_centroid))));
         }
         else if(fabs(_centroids[__M_BEARISH]-_unknown_centroid)<fabs(_centroids[__M_BULLISH]-_unknown_centroid) && fabs(_centroids[__M_BEARISH]-_unknown_centroid)<fabs(_centroids[__M_WHIPSAW]-_unknown_centroid))
         {
            _da=-1.0*(1.0-(fabs(_centroids[__M_BEARISH]-_unknown_centroid)/(fabs(_centroids[__M_BULLISH]-_unknown_centroid)+fabs(_centroids[__M_WHIPSAW]-_unknown_centroid)+fabs(_centroids[__M_BEARISH]-_unknown_centroid))));
         }
      
         int _index   =StartIndex();
         double _min_l=Low(_index),_max_h=High(_index);
         
         for(int p=_index;p<m_money_points+_index;p++)
         {
            _min_l=fmin(_min_l,Low(p));
            _max_h=fmax(_max_h,High(p));
         }
      
         _da*=(_max_h-_min_l);
         _da+=(_max_h-_min_l);
      }
      else
      {
         if(_info==-4){ printf(__FUNCSIG__+" internal EVD subroutine hasn't converged "); }
         else if(_info==-2){ printf(__FUNCSIG__+" there is a point with class number outside of [0..NClasses-1] "); }
         else if(_info==-1){ printf(__FUNCSIG__+" incorrect parameters was passed  either: NPoints<0, or NVars<1, or NClasses<2) "); }
      }
      
      return(_da);
   }
//+------------------------------------------------------------------+
//| Data Set method                                                  |
//| INPUT PARAMETERS                                                 |
//|     Index   -   int, read index within price buffer.             |
//|                                                                  |
//|     Type    -   position type for determining use of             |
//|                 lows or highs.                                   |
//|                                                                  |
//|     Variables                                                    |
//|             -   whether data component is variables or .         |
//|                  classifier.                                     |
//| OUTPUT                                                           |
//|     double  -   Data depending on data set type                  |
//|                                                                  |
//| DATA SET TYPES                                                   |
//| 1. Discretized variables.                                        |
//|                                                                  |
//| 2. Normalized variables.                                         |
//|                                                                  |
//| 3. Continuized variables.                                        |
//|                                                                  |
//| 4. Raw data variables.                                           |
//|                                                                  |
//+------------------------------------------------------------------+
double CMoneyDA::Data(int Index,bool Variables=true)
   {
      m_close.Refresh(-1);
         
      m_low.Refresh(-1);
      m_high.Refresh(-1);
            
      if(Variables)
      {
         if(m_money_type==0)
         {
            return(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_money_regulizer*Range(Index)?1.0:((Close(StartIndex()+Index)>Close(StartIndex()+Index+1))?2.0:0.0));
         }
         else if(m_money_type==1)
         {
            if(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_money_regulizer*Range(Index))
            {
               return(0.0);
            }
            return((Close(StartIndex()+Index)-Close(StartIndex()+Index+1))/fmax(m_symbol.Point(),fmax(High(StartIndex()+Index),High(StartIndex()+Index+1))-fmin(Low(StartIndex()+Index),Low(StartIndex()+Index+1))));
         }
         else if(m_money_type==2)
         {
            if(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_money_regulizer*Range(Index))
            {
               return(0.0);
            }
            return(Close(StartIndex()+Index)-Close(StartIndex()+Index+1));
         }
         else if(m_money_type==3)
         {
            if(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_money_regulizer*Range(Index))
            {
               return(Close(StartIndex()+Index+1));
            }
            return(Close(StartIndex()+Index));
         }
      }
      
      return(fabs(Close(StartIndex()+Index)-Close(StartIndex()+Index+1))<m_money_regulizer*Range(Index)?1.0:((Close(StartIndex()+Index)>Close(StartIndex()+Index+1))?2.0:0.0));
   }
//+------------------------------------------------------------------+
//| Getting lot size for open long position.                         |
//+------------------------------------------------------------------+
double CMoneyDA::CheckOpenLong(double price,double sl)
  {
   double _da=ProcessDA(StartIndex());
   
   if(m_symbol==NULL)
      return(0.0);
   
   sl=m_symbol.Bid()-_da;
   
//--- select lot size
   double _da_1_lot_loss=(_da/m_symbol.TickSize())*m_symbol.TickValue();
   double lot=((m_percent/100.0)*m_account.FreeMargin())/_da_1_lot_loss;
   
//--- calculate margin requirements for 1 lot
   if(m_account.FreeMarginCheck(m_symbol.Name(),ORDER_TYPE_BUY,lot,m_symbol.Ask())<0.0)
     {
      printf(__FUNCSIG__" insufficient margin for sl lot! ");
      lot=m_account.MaxLotCheck(m_symbol.Name(),ORDER_TYPE_BUY,m_symbol.Ask(),m_percent);
     }
   
//--- return trading volume
   return(Optimize(lot));
  }
//+------------------------------------------------------------------+
//| Getting lot size for open short position.                        |
//+------------------------------------------------------------------+
double CMoneyDA::CheckOpenShort(double price,double sl)
  {
   double _da=ProcessDA(StartIndex());
   
   if(m_symbol==NULL)
      return(0.0);
      
   sl=m_symbol.Ask()+_da;
   
//--- select lot size
   double _da_1_lot_loss=(_da/m_symbol.TickSize())*m_symbol.TickValue();
   double lot=((m_percent/100.0)*m_account.FreeMargin())/_da_1_lot_loss;
   
//--- calculate margin requirements for 1 lot
   if(m_account.FreeMarginCheck(m_symbol.Name(),ORDER_TYPE_SELL,lot,m_symbol.Bid())<0.0)
     {
      printf(__FUNCSIG__" insufficient margin for sl lot! ");
      lot=m_account.MaxLotCheck(m_symbol.Name(),ORDER_TYPE_SELL,m_symbol.Bid(),m_percent);
     }
//--- return trading volume
   return(Optimize(lot));
  }
//+------------------------------------------------------------------+
//| Optimizing lot size for open.                                    |
//+------------------------------------------------------------------+
double CMoneyDA::Optimize(double lots)
  {
   double lot=lots;
//--- calculate number of losses orders without a break
   if(m_decrease_factor>0)
     {
      //--- select history for access
      HistorySelect(0,TimeCurrent());
      //---
      int       orders=HistoryDealsTotal();  // total history deals
      int       losses=0;                    // number of consequent losing orders
      CDealInfo deal;
      //---
      for(int i=orders-1;i>=0;i--)
        {
         deal.Ticket(HistoryDealGetTicket(i));
         if(deal.Ticket()==0)
           {
            Print("CMoneyDA::Optimize: HistoryDealGetTicket failed, no trade history");
            break;
           }
         //--- check symbol
         if(deal.Symbol()!=m_symbol.Name())
            continue;
         //--- check profit
         double profit=deal.Profit();
         if(profit>0.0)
            break;
         if(profit<0.0)
            losses++;
        }
      //---
      if(losses>1)
         lot=NormalizeDouble(lot-lot*losses/m_decrease_factor,2);
     }
//--- normalize and check limits
   double stepvol=m_symbol.LotsStep();
   lot=stepvol*NormalizeDouble(lot/stepvol,0);
//---
   double minvol=m_symbol.LotsMin();
   if(lot<minvol)
      lot=minvol;
//---
   double maxvol=m_symbol.LotsMax();
   if(lot>maxvol)
      lot=maxvol;
//---
   return(lot);
  }
//+------------------------------------------------------------------+
