//+------------------------------------------------------------------+
//|                                                       ELN_MA.mq5 |
//|                        Copyright 2023, MetaQuotes Software Corp. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2023, MetaQuotes Software Corp."
#property link      "https://www.mql5.com"
#property version   "1.00"
#resource "\\Indicators\\LogPrices.ex5"
#include<CoordinateDescent.mqh>
#include<ErrorDescription.mqh>
#property script_show_inputs
//--- input parameters
input uint     MA_period_inc=2;        //MA lookback increment
input uint     Num_MA_periods=30;    //Num of lookbacks
input double   Alpha=0.5;
input int      AppliedPrice=PRICE_CLOSE;
input ENUM_MA_METHOD MaMethod=MODE_EMA;
input ENUM_TIMEFRAMES tf=PERIOD_D1;    //time frame
input uint     BarsLookAhead=1;
input uint     Num_Folds = 10;         //Num of Folds for cross validation
input uint     MaximumIterations=1000;
input datetime TrainingSampleStartDate=D'2019.12.31';
input datetime TrainingSampleStopDate=D'2022.12.31';
input datetime TestingSampleStartDate=D'2023.01.02';
input datetime TestingSampleStopDate=D'2023.06.30';
input string   SetSymbol="";
input bool     UseCovarUpdates=true;
input bool     UseFastTest=true;
input bool     UseWarmStart=false;
input int      NumLambdasToTest=50;
input bool     ShowFullOutPut=false;     //print full output to terminal

//+------------------------------------------------------------------+
//|global integer variables                                          |
//+------------------------------------------------------------------+
int size_insample,                 //training set size
    size_outsample,                //testing set size
    size_observations,             //size of of both training and testing sets combined
    size_lambdas,                  //number of lambdas to be tested
    size_predictors,               //number of predictors
    maxperiod,                     //maximum lookback
    price_handle=INVALID_HANDLE,   //log prices indicator handle
    long_ma_handle=INVALID_HANDLE, //long moving average indicator handle
    short_ma_handle=INVALID_HANDLE;//short moving average indicator handle
//+------------------------------------------------------------------+
//|double global variables                                           |
//+------------------------------------------------------------------+

double prices[],                   //array for log transformed prices
       targets[],                  //differenced prices kept here
       predictors_matrix[],               //flat array arranged as matrix of all predictors_matrix ie size_observations by size_predictors
       longma[],                   //long ma indicator values
       Lambdas[],                  //calculated lambdas kept here
       Lambdas_OOS[],              //calculated out of sample lambdas are here
       shortma[],                  //short ma indicator values
       Lambda;                     //initial optimal lambda value
//+------------------------------------------------------------------+
//| Coordinate descent pointer                                       |
//+------------------------------------------------------------------+
CCoordinateDescent *cdmodel;       //coordinate descent pointer
//+------------------------------------------------------------------+
//| Script program start function                                    |
//+------------------------------------------------------------------+
void OnStart()
  {
//get relative shift of is and oos sets
   int teststart,teststop,trainstart,trainstop;
   teststart=iBarShift(SetSymbol!=""?SetSymbol:NULL,tf,TestingSampleStartDate);
   teststop=iBarShift(SetSymbol!=""?SetSymbol:NULL,tf,TestingSampleStopDate);
   trainstart=iBarShift(SetSymbol!=""?SetSymbol:NULL,tf,TrainingSampleStartDate);
   trainstop=iBarShift(SetSymbol!=""?SetSymbol:NULL,tf,TrainingSampleStopDate);
//check for errors from ibarshift calls
   if(teststart<0 || teststop<0 || trainstart<0 || trainstop<0)
     {
      Print(ErrorDescription(GetLastError()));
      return;
     }
//---set the size of the sample sets
   size_observations=(trainstart - teststop) + 1 ;
   size_outsample=(teststart - teststop) + 1;
   size_insample=(trainstart - trainstop) + 1;
   maxperiod=int(Num_MA_periods*MA_period_inc);
   size_insample-=maxperiod;
   size_lambdas=NumLambdasToTest;
   size_predictors=int(Num_MA_periods);
//---check for input errors
   if(size_lambdas<=0 || size_insample<=0 || size_outsample<=0 || size_predictors<=0 || maxperiod<=0 || BarsLookAhead<=0)
     {
      Print("Invalid inputs ");
      return;
     }
//---
   Comment("resizing buffers...");
//---allocate memory
   if(ArrayResize(targets,size_observations)<(int)size_observations ||
      ArrayResize(predictors_matrix,size_observations*size_predictors)<int(size_observations*size_predictors) ||
      ArrayResize(Lambdas,size_lambdas)<(int)size_lambdas ||
      ArrayResize(Lambdas_OOS,size_lambdas)<(int)size_lambdas ||
      ArrayResize(shortma,size_observations)<(int)size_observations ||
      ArrayResize(longma,size_observations)<(int)size_observations ||
      ArrayResize(prices,size_observations+BarsLookAhead)<int(size_observations+BarsLookAhead))
     {
      Print("ArrayResize error ",ErrorDescription(GetLastError()));
      return;
     }

//---
   Comment("getting price predictors_matrix...");
//---set prices handle
   price_handle=iCustom(SetSymbol!=""?SetSymbol:NULL,tf,"::Indicators\\LogPrices.ex5",AppliedPrice);
   if(price_handle==INVALID_HANDLE)
     {
      Print("invalid logprices handle ",ErrorDescription(GetLastError()));
      return;
     }
//---
   Comment("getting indicators...");
//----calculate the full collection of predictors_matrix
   int longmaperiod,shortmaperiod,prevshort,prevlong;
   int k=0;
//---
   prevlong=prevshort=0;
//---
   for(uint iperiod=0; iperiod<Num_MA_periods; iperiod++)
     {
      longmaperiod=(int)(iperiod+1)*int(MA_period_inc);
      shortmaperiod = (longmaperiod>=2)?int(longmaperiod/2):longmaperiod;
      ResetLastError();
      int try=10;
      while(try)
        {
          long_ma_handle=iMA(SetSymbol!=""?SetSymbol:NULL,tf,longmaperiod,0,MaMethod,price_handle);
          short_ma_handle=iMA(SetSymbol!=""?SetSymbol:NULL,tf,shortmaperiod,0,MaMethod,price_handle);
          
          if(long_ma_handle==INVALID_HANDLE || short_ma_handle==INVALID_HANDLE)
             try--;
          else
             break;
        }
      Comment("copying buffers for short ",shortmaperiod," long ",longmaperiod);

     if(CopyBuffer(long_ma_handle,0,teststop,size_observations,longma)<=0 ||
        CopyBuffer(short_ma_handle,0,teststop,size_observations,shortma)<=0)
         {
          Print("error copying to ma buffers  ",GetLastError());
          return;
         }

     for(int i=0 ; i<int(size_observations) ; i++)
          predictors_matrix[i*size_predictors+k] = shortma[i]-longma[i];
     ++k ;

     if(long_ma_handle!=INVALID_HANDLE && short_ma_handle!=INVALID_HANDLE && IndicatorRelease(long_ma_handle) && IndicatorRelease(short_ma_handle))
       {
        long_ma_handle=short_ma_handle=INVALID_HANDLE;
        prevlong=longmaperiod;
        prevshort=shortmaperiod;
       }
    }
//---
   Comment("filling target buffer...");
//---
   ResetLastError();
   if(CopyBuffer(price_handle,0,teststop,size_observations+BarsLookAhead,prices)<int(size_observations+BarsLookAhead))
     {
      Print("error copying to price buffer , ",ErrorDescription(GetLastError()));
      return;
     }
//---
   for(int i=0 ; i<int(size_observations); i++)
      targets[i] = prices[i+BarsLookAhead]-prices[i];
//---
   Comment("optional lambda tuning...");
//---
   if(Alpha<=0)
      Lambda=0;
   else //train
      Lambda=OptimizeLambda(size_insample,size_predictors,(int)Num_Folds,UseCovarUpdates,size_lambdas,Alpha,(int)MaximumIterations,1.e-9,UseFastTest,predictors_matrix,targets,Lambdas,Lambdas_OOS,ShowFullOutPut);
//---
   Comment("coordinate descent engagement...");
//---initialize CD object
   cdmodel=new CCoordinateDescent(size_predictors,size_insample,UseCovarUpdates,0);
//---
   if(cdmodel==NULL)
     {
      Print("error creating Coordinate Descent object ");
      return;
     }
//---set the parameters and data
   cdmodel.SetData(0,size_insample,predictors_matrix,targets);
//---
   Print("optimal lambda ",DoubleToString(Lambda));
//---train the model
   cdmodel.Train(Alpha,Lambda,(int)MaximumIterations,1.e-7,UseFastTest,UseWarmStart);
//---
   Print("explained variance ",cdmodel.GetExplainedVariance());
//---optionally output results of training here
   if(ShowFullOutPut)
     {
      k=0;
      string output;
      for(uint iperiod=0; iperiod<Num_MA_periods; iperiod++)
        {
         longmaperiod=(int)(iperiod+1)*int(MA_period_inc);
         output+=StringFormat("\n%5d ", longmaperiod) ;
         shortmaperiod = (longmaperiod>=2)?int(longmaperiod/2):longmaperiod;
         output+=StringFormat(",%5d ,%.20e ", shortmaperiod,cdmodel.GetBetaAt(k));
         ++k;
        }
      Print(output);
     }
//---
   double sum=0.0;                //cumulated predictions
   double pred;                   //a prediction
   int xptr;
   k=size_observations - (size_insample+maxperiod) - 1;
//---
   Comment("test the model...");
//---do the out of sample test
   for(int i=k ; i<size_observations ; i++)
     {
      xptr = i*size_predictors ;
      pred = 0.0 ;
      for(int ivar=0 ; ivar<int(size_predictors) ; ivar++)
         pred += cdmodel.GetBetaAt(ivar) * (predictors_matrix[xptr+ivar] - cdmodel.GetXmeansAt(ivar)) / cdmodel.GetXscalesAt(ivar) ;
      pred = pred * cdmodel.GetYscale() + cdmodel.GetYmean() ; // Unscale prediction to get it back in original Y domain
      if(pred > 0.0)
         sum += targets[i] ;
      else
         if(pred < 0.0)
            sum -= targets[i] ;
     }
//---
   PrintFormat("OOS total return = %.5lf (%.3lf percent)",sum, 100.0 * (exp(sum) - 1.0)) ;
//---
   delete cdmodel;
//---
   Comment("");
  }
//+------------------------------------------------------------------+
