from __future__ import annotations

import pytest
from logmt5 import LogMT5

# фикстура модуля
logMT5 = LogMT5('TestLogger')
logMT5.sdateLog = '2022-02-16'


def test_fixture():
    assert logMT5.appName == 'TestLogger'
    assert logMT5.sdateLog == '2022-02-16'


@pytest.mark.parametrize("path, sdate", [
    ("d:\\Project\\MQL5\\Logs\\20220216.log", "2022-02-16"),
    ("d:\\tratata.txt", None),
])
def test_getDateLog(path: str, sdate: str | None):
    sdateLog = logMT5.getDateLog(path)
    assert sdateLog == sdate


@pytest.mark.parametrize("line, nameType, levelItem", [
    ('CM 0 14:11:25.690 Cayman (USDJPY,H1) DBG: AnalyserSymbol::Run / USDJPY, H1, 14:11, 19', 'LogItem', 'DBG:'),
    ('CM 0 19:52:04.476 SetTradeRect (USDJPY,H1) SetTradeRect', 'LogItem', 'INF:'),
    ('IP 0 20:17:21.291 TestLogger (USDJPY,H1) INF:', 'NoneType', ''),
    ('number_of_words_less_than_6', 'NoneType', ''),
])
def test_line2item(line: str, nameType: str, levelItem: str):
    logItem = logMT5.line2item(line)
    level = logItem.level if logItem else ''
    assert logItem.__class__.__name__ == nameType
    assert level == levelItem


def test_logItem():
    line = 'CM 0 14:11:25.690 Cayman (USDJPY,H1) DBG: AnalyserSymbol::Run / USDJPY, H1, 14:11, 19'
    item = logMT5.line2item(line)
    assert item.stime == '2022-02-16T14:11:25.690'
    assert item.module == 'Cayman'
    assert item.sender == 'AnalyserSymbol::Run'
    assert item.symbol == 'USDJPY'
    assert item.period == 'H1'
    assert item.level == 'DBG:'
    assert item.message == 'USDJPY, H1, 14:11, 19'


def test_logItemJson():
    line = 'CM 0 14:11:25.690 Cayman (USDJPY,H1) DBG: AnalyserSymbol::Run / USDJPY, H1, 14:11, 19'
    json = (
        '{"@t": "2022-02-16T14:11:25.690Z", '
        '"AppName": "TestLogger", '
        '"Symbol": "USDJPY", '
        '"Period": "H1", '
        '"Module": "Cayman", '
        '"Sender": "AnalyserSymbol::Run", '
        '"Level": "DBG:", '
        '"@l": "Debug", '
        '"Message": "USDJPY, H1, 14:11, 19", '
        '"@m": "USDJPY, H1 / Cayman / AnalyserSymbol::Run / USDJPY, H1, 14:11, 19"}'
    )
    item = logMT5.line2item(line)
    assert item.toJson('TestLogger') == json


@pytest.mark.parametrize("stime, count, json", [
    ('', 14, (
            '{"@t": "2022-02-16T19:52:04.474Z", "AppName": "TestLogger", "Symbol": "USDJPY", '
            '"Period": "H1", "Module": "SetTradeRect", "Sender": "ChartMan::SimplifyNameObj", '
            '"Level": "INF:", "@l": "Information", "Message": "name old=H1 Rectangle 7250 / '
            'new=Rect620D55E4", "@m": "USDJPY, H1 / SetTradeRect / '
            'ChartMan::SimplifyNameObj / name old=H1 Rectangle 7250 / new=Rect620D55E4"}'
    )),
    ('2022-02-16T20:00', 7, (
            '{"@t": "2022-02-16T20:17:21.048Z", "AppName": "TestLogger", "Symbol": "USDJPY", '
            '"Period": "H1", "Module": "TestLogger", "Sender": "OnStart", "Level": "DBG:", '
            '"@l": "Debug", "Message": "debug message", "@m": "USDJPY, H1 / TestLogger / '
            'OnStart / debug message"}'
    )),
])
def test_parseLog(stime: str, count: int, json: str):
    pathLog = 'data/20220216.log'
    listJson = logMT5.parseLog(pathLog, stime)
    assert len(listJson) == count
    assert listJson[0] == json
