from __future__ import annotations

from pathlib import Path
from logitem import *
import re


class LogMT5:
    """MT5 log file handler"""
    appName: str  # имя приложения (владелец сообщений)
    sdateLog: str | None  # дата лог файла (yyyy-mm-dd)

    def __init__(self, appName: str):
        """Class constructor. Seq uses the appName parameter to identify events (messages)
        
        :param appName: application name to which the logs belong.
        

"""
        self.appName = appName
        self.sdateLog = None

    @staticmethod
    def getDateLog(pathLog: str) -> str | None:
        """Define log date from its path
        
        :param pathLog: log file path :return: log date (yyyy-mm-dd) or None
        

"""
        name = Path(pathLog).stem
        sdate = f'{name[:4]}-{name[4:6]}-{name[6:]}' \
            if len(name) == 8 and re.match(r'^[0-9]+$', name) \
            else None
        return sdate

    def line2item(self, line: str) -> LogItem | None:
        """Convert string to log entry (LogItem)
        
        :param line: log file line :return: log file item (LogItem)
        

"""

        # проверяем строку лога на кол-во слов
        # CM 0 19:52:04.476 SetTradeRect (USDJPY,H1) SetTradeRect
        # CM 0 14:11:25.690 Cayman (USDJPY,H1) INF: AnalyserSymbol::Run / USDJPY, H1, 14:11, 19
        # tokens[0]=CM, [1]=0, [2]=14:11:25.690, [3]=Cayman, [4]=(USDJPY,H1)
        # [5]='INF AnalyserSymbol::Run / USDJPY, H1, 14:11, 19'
        tokens = re.split(r'\s', line, maxsplit=5)
        if len(tokens) < 6:
            return None

        # разбираем tokens[4] = '(USDJPY,H1)'
        infoSymbol = re.split(r'[(,)]', tokens[4])
        if len(infoSymbol) != 4:
            return None

        # разбираем tokens[5]='INF: AnalyserSymbol::Run / USDJPY, H1, 14:11, 19'
        # или tokens[5]='SetTradeRect' (если уровня нет, использовать INF: )
        message = tokens[5]
        infoMessage = re.split(r'\s', message)
        level = infoMessage[0] if infoMessage[0] in dictLevel else None
        if level and len(infoMessage) == 1:
            return None  # игнорируем пустое сообщение
        sender = infoMessage[1] if level else infoMessage[0]
        message = message[message.index(sender) + len(sender):]
        # удаляем '/' вначале сообщения, если есть
        message = re.sub('^\s*/', '', message).strip()

        # возвращаем запись лога
        stime = f'{self.sdateLog}T{tokens[2]}'  # дата и время сообщения ISO8601
        return LogItem(stime=stime, module=tokens[3], sender=sender,
                       symbol=infoSymbol[1], period=infoSymbol[2],
                       level='INF:' if not level else level,
                       message=message)

    def parseLog(self, pathLog: str, lastTime: str) -> list:
        """Parse the log file and generate a list of json records. Select only new entries with a time greater than lastTime.
        
        :param pathLog: log file path (../yyyymmdd.log) :param lastTime: time of last entry in Seq from appName :return: list of json entries
        
        
        

"""

        # определяем дату лога
        self.sdateLog = self.getDateLog(pathLog)
        if not self.sdateLog:
            raise ValueError(f'invalid name file {pathLog}')

        # разбираем строки лога
        items = []
        with open(pathLog, encoding='utf-16-le') as flog:
            while line := flog.readline().rstrip():
                item = self.line2item(line)
                if item and item.stime > lastTime:
                    items.append(item.toJson(self.appName))

        print(f'parseLog {len(items)} new messages')
        return items
