//+------------------------------------------------------------------+
//|                                                   Statistica.mqh |
//|                        Copyright 2012, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+

//+------------------------------------------------------------------+
//| Median                                                           |
//+------------------------------------------------------------------+
double Mediana(double &arr[])
  {
   double res=0.0;
   int    size=ArraySize(arr);
//--- check
   if(size<=0)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   if(size%2==0) res=(arr[size/2-1]+arr[size/2])/2;
   else           res=arr[(size+1)/2-1];
//--- returning the result
   return(res);
  }
//+------------------------------------------------------------------+
//| The middle of the 50% interquantile range (bends center)         |
//+------------------------------------------------------------------+
double Mediana50(double &arr[])
  {
   int size=ArraySize(arr);
//--- check
   if(size<=0)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   int m=size/4;
//---
   return((arr[m]+arr[size-m-1])/2);
  }
//+------------------------------------------------------------------+
//| Arithmetical mean of entire sampling                             |
//+------------------------------------------------------------------+
double Average(double &arr[])
  {
   int size=ArraySize(arr);
//--- check
   if(size<=0)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   for(int i=0;i<size;i++)
      sum+=arr[i];
//--- returning the result
   return(sum/size);
  }
//+------------------------------------------------------------------+
//| Arithmetical mean of the 50% interquantile range                 |
//+------------------------------------------------------------------+
double Average50(double &arr[])
  {
   int size=ArraySize(arr);
//--- check
   if(size<=0)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   int    m=size/4;
   int    n=size-m;
//---
   for(int i=m;i<n;i++)
      sum+=arr[i];
//--- returning the result
   return(sum/(size-2*m));
  }
//+------------------------------------------------------------------+
//| Sweep center                                                     |
//+------------------------------------------------------------------+
double SweepCenter(double &arr[])
  {
   int size=ArraySize(arr);
//--- check
   if(size<=0)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   return((arr[0]+arr[size-1])/2);
  }
//+------------------------------------------------------------------+
//| The average value of upper five evaluations                      |
//+------------------------------------------------------------------+
double AverageOfEvaluations(double &arr[])
  {
   double res;
   double array[5];
//--- calculation of five evaluations
   array[0]=Mediana(arr);
   array[1]=Mediana50(arr);
   array[2]=Average(arr);
   array[3]=Average50(arr);
   array[4]=SweepCenter(arr);
//--- check
   if(ArraySort(array))
      res=array[2];
   else
     {
      Print(__FUNCTION__+": array sort error");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return res;
  }
//+------------------------------------------------------------------+
//| Variance                                                         |
//+------------------------------------------------------------------+
double Variance(double &arr[],double mx)
  {
   int size=ArraySize(arr);
//--- check  
   if(size<=1)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   for(int i=0;i<size;i++)
      sum+=MathPow(arr[i]-mx,2);
//---
   return(sum/(size-1));
  }
//+------------------------------------------------------------------+
//| 3rd central moment                                               |
//+------------------------------------------------------------------+
double ThirdCentralMoment(double &arr[],double mx)
  {
   int size=ArraySize(arr);
//--- check
   if(size<=2)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   for(int i=0;i<size;i++)
      sum+=MathPow(arr[i]-mx,3);
//--- returning the result
   return(sum*size/(size-1)/(size-2));
  }
//+------------------------------------------------------------------+
//| 4th central moment                                               |
//+------------------------------------------------------------------+
double FourthCentralMoment(double &arr[],double mx)
  {
   int size=ArraySize(arr);
//--- check
   if(size<=3)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum2=0.0;
   double sum4=0.0;
   for(int i=0;i<size;i++)
     {
      sum2+=MathPow(arr[i]-mx,2);
      sum4+=MathPow(arr[i]-mx,4);
     }
//--- returning the result
   return(((MathPow(size,2)-2*size+3)*sum4-
          (6*size-9)/size*MathPow(sum2,2))/
          (size-1)*(size-2)*(size-3));
  }
//+------------------------------------------------------------------+
//| Custom asymmetry                                                 |
//+------------------------------------------------------------------+
double Asymmetry(double &arr[],double mx,double dx)
  {
//--- check
   if(dx==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//---
   double size=ArraySize(arr);
//--- check
   if(size<=2)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   for(int i=0;i<size;i++)
      sum+=MathPow(arr[i]-mx,3);
//--- returning the result
   return(size/(size-1)/(size-2)/MathPow(dx,1.5)*sum);
  }
//+------------------------------------------------------------------+
//| Excess (another calculation method)                              |
//+------------------------------------------------------------------+
double Excess2(double &arr[],double mx,double dx)
  {
//--- check
   if(dx==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//---
   double size=ArraySize(arr);
//--- check
   if(size<=3)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum2=0.0;
   double sum4=0.0;
   for(int i=0; i<size; i++)
     {
      sum2+=MathPow(arr[i]-mx,2);
      sum4+=MathPow(arr[i]-mx,4);
     }
//--- returning the result
   return((size*(size+1)*sum4-3*sum2*sum2*(size-1))/
          (size-1)/(size-2)/(size-3)/dx/dx);
  }
//+------------------------------------------------------------------+
//| Excess                                                           |
//+------------------------------------------------------------------+
double Excess(double &arr[],double mx,double dx)
  {
//--- check
   if(dx==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//---
   double size=ArraySize(arr);
//--- check
   if(size<4)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   for(int i=0; i<size; i++)
      sum+=MathPow(arr[i]-mx,4);
//--- returning the result
   return(((size *(size+1)*sum/(size-1)/dx/dx)-3*MathPow(size-1,2))/
          (size-2)/(size-3));
  }
//+------------------------------------------------------------------+
//| Euler's gamma function for x>0                                   |
//+------------------------------------------------------------------+
double Gamma(double x)
  {
//--- check
   if(x==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- initialization of variables
   double p=0.0;
   double pp=0.0;
   double qq=0.0;
   double z=1.0;
   double sgngam=1.0;
   int    i=0;
//--- check
   if(x>33.0)
      return(sgngam*GammaStirling(x));
//---
   while(x>=3)
     {
      x--;
      z=z*x;
     }
//---
   while(x<2)
     {
      if(x<0.000000001)
         return(z/((1+0.5772156649015329*x)*x));
      z/=x;
      x++;
     }
//---
   if(x==2)
      return(z);
//---
   x-=2.0;
   pp = 0.000160119522476751861407;
   pp = 0.00119135147006586384913+x*pp;
   pp = 0.0104213797561761569935+x*pp;
   pp = 0.0476367800457137231464+x*pp;
   pp = 0.207448227648435975150+x*pp;
   pp = 0.494214826801497100753+x*pp;
   pp = 0.999999999999999996796+x*pp;
   qq = -0.0000231581873324120129819;
   qq = 0.000539605580493303397842+x*qq;
   qq = -0.00445641913851797240494+x*qq;
   qq = 0.0118139785222060435552+x*qq;
   qq = 0.0358236398605498653373+x*qq;
   qq = -0.234591795718243348568+x*qq;
   qq = 0.0714304917030273074085+x*qq;
   qq = 1.00000000000000000320+x*qq;
//--- check
   if(qq==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(z*pp/qq);
  }
//+--------------------------------------------------------------------+
//| Stirling's approximation for the Euler's gamma function calculation|
//| for x>33                                                           |
//+--------------------------------------------------------------------+
double GammaStirling(double x)
  {
//--- check
   if(x==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- initialization of variables
   double y=0.0;
   double w=0.0;
   double v=0.0;
   double stir=0.0;
//---
   w=1.0/x;
   stir = 0.000787311395793093628397;
   stir = -0.000229549961613378126380+w*stir;
   stir = -0.00268132617805781232825+w*stir;
   stir = 0.00347222221605458667310+w*stir;
   stir = 0.0833333333333482257126+w*stir;
   w=1+w*stir;
//---
   y=MathExp(x);
   if(x>143.01608)
     {
      v = MathPow(x, 0.5*x-0.25);
      y = v*(v/y);
     }
   else
      y=MathPow(x,x-0.5)/y;
//--- returning the result
   return(2.50662827463100050242*y*w);
  }
//+------------------------------------------------------------------+
//| The variance of the sample variance                              |
//+------------------------------------------------------------------+
double VarianceOfSampleVariance(double &arr[],double m4,double dx)
  {
   double size=ArraySize(arr);
//--- check
   if(size<=0)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return((m4-MathPow(dx,2))/size);
  }
//+------------------------------------------------------------------+
//| Variation of the sample standard deviation                       |
//+------------------------------------------------------------------+
double VarianceOfStandartDeviation(double &arr[],double m4,double dx)
  {
//--- check
   if(dx==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(VarianceOfSampleVariance(arr,m4,dx)/dx/4);
  }
//+------------------------------------------------------------------+
//| Sample asymmetry variation                                       |
//+------------------------------------------------------------------+
double VarianceOfAsymmetry(double size)
  {
//--- check
   if(size==-1 || size==-3)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return((6*size-6)/(size+1)/(size+3));
  }
//+------------------------------------------------------------------+
//| Sample excess variation                                          |
//+------------------------------------------------------------------+
double VarianceOfExcess(double size)
  {
//--- check
   if(size==1 || size==-3 || size==-5)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(24*size*(size-2)*(size-3)/MathPow(size-1,2)/(size+3)/(size+5));
  }
//+------------------------------------------------------------------+
//| Variation of a sample mean                                       |
//+------------------------------------------------------------------+
double VarianceOfAverage(double dx,int size)
  {
//--- check
   if(size==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(dx/size);
  }
//+------------------------------------------------------------------+
//| Logarithm  by base                                             |
//+------------------------------------------------------------------+
double Log(double x,double a)
  {
//--- check
   if(x<=0 || a<=0 || a==1)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(MathLog(x)/MathLog(a));
  }
//+------------------------------------------------------------------+
//| Censoring ratio                                                  |
//+------------------------------------------------------------------+
double CensorCoeff(int size,double e)
  {
//--- check
   if(e<1)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(1.55+0.8*Log(1.0*size/10,10)*MathSqrt(e-1));
  }
//+------------------------------------------------------------------+
//| The optimal number of the distribution density histogram columns |
//+------------------------------------------------------------------+
int HistogramLength(double e,int size)
  {
//--- length calculation
   int n=(int)((e/6+0.25)*MathPow(size,0.4));
   if(n%2==0)
      n--;
//--- returning the result
   return(n);
  }
//+------------------------------------------------------------------+
//| Exclusion of misses from the sample                              |
//+------------------------------------------------------------------+
int Resize(double &arr[],double g,double dx,double center)
  {
//--- check
   if(dx<0)
     {
      Print(__FUNCTION__+": the error variable");
      return(-1);
     }
//---
   int size=ArraySize(arr);
//--- check
   if(size==0)
     {
      Print(__FUNCTION__+": array size error");
      return(-1);
     }
//--- initialization of variables
   int    mini=-1;
   int    maxi=-1;
   int    m=0;
   double sig=MathSqrt(dx);
   double gsig=g*sig;
   double min=center-gsig;
   double max=center+gsig;
//--- calculation
   for(int i=size-1;maxi<0;i--)
      if(arr[i]<=max)
         maxi=size-1-i;
   for(int i=0;mini<0;i++)
      if(arr[i]>=min)
         mini=i;
//--- changing the array structure
   m=size-mini-maxi;
   for(int i=0; i<m; i++)
      arr[i]=arr[i+mini];
//--- returning the result
   return(ArrayResize(arr,m));
  }
//+------------------------------------------------------------------+
//| Creating the distribution histogram to *.csv file                |
//| (for viewing in Excel)                                           |
//+------------------------------------------------------------------+
bool Histogram(double &arr[])
  {
   int size=ArraySize(arr);
//--- check
   if(size<=0)
     {
      Print(__FUNCTION__+": array size error");
      return(false);
     }
//--- initialization of variables
   double center=0.0;
   double mx=0.0;
   double xmin=0.0;
   double xmax=0.0;
   double delta=0.0;
   double max=0.0;
   double d=0.0;      // histogram column width
   double dx=0.0;
   double a=0.0;
   double e=0.0;
   double g=0.0;
   double s=0.0;      // unnormalized histogram area (=d*size)
   double histog[];
   int    l=0;        // number of the histogram columns (odd)
   int    num=0;
   int    handle=0;
//---
   ArraySort(arr);
   center=AverageOfEvaluations(arr); // getting the average from the evaluations
   dx=Variance(arr,center);          // variance calculation
   e=Excess2(arr,center,dx);         // excess calculation
   g=CensorCoeff(size,e);            // censoring ratio calculation
   size=Resize(arr,g,dx,center);     // determining the array new size
   mx=Average(arr);                  // average
   dx=Variance(arr,mx);              // variance new value
   a=Asymmetry(arr,mx,dx);           // asymmetry value
   e=Excess2(arr,mx,dx);             // excess new value
   l=HistogramLength(e,size);        // determining the histogram new length
//--- creating a histogram
   max=MathMax(MathAbs(arr[0]-mx),MathAbs(arr[size-1]-mx));
   xmin=mx-max;
   xmax=mx+max;
   d=2*max/l;
   delta=xmax-xmin;
   ArrayResize(histog,l);
   ArrayInitialize(histog,0.0);
   for(int i=0;i<size;i++)
     {
      num=(int)((arr[i]-xmin)/delta*l);
      if(num==l) num--;
      histog[num]++;
     }
//---
   s=d*size;
   for(int i=0;i<l;i++)
      histog[i]/=s;
//---
   if(MathAbs(a)/MathSqrt(VarianceOfAsymmetry(size))<3)
      for(int i=0;i<l/2;i++)
        {
         histog[i]=(histog[i]+histog[l-1-i])/2;
         histog[l-1-i]=histog[i];
        }
//--- writing to file
   handle=FileOpen("histogram.csv",FILE_CSV|FILE_WRITE);
//--- check
   if(handle!=INVALID_HANDLE)
     {
      for(int i=0;i<l;i++)
         FileWrite(handle,histog[i]);
      FileClose(handle);
     }
//--- file open error
   else
     {
      Print(__FUNCTION__+": file open error");
      return(false);
     }
//---
   return true;
  }
//+------------------------------------------------------------------+
//| Custom covariation                                               |
//+------------------------------------------------------------------+
double Cov(double &arrX[],double &arrY[],double mx,double my)
  {
   int sizeX=ArraySize(arrX);
//--- check
   if(sizeX<=1)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//---
   int sizeY=ArraySize(arrY);
//--- check
   if(sizeX!=sizeY)
     {
      Print(__FUNCTION__+" dimensional arrays are not equal");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   for(int i=0;i<sizeX;i++)
      sum+=(arrX[i]-mx)*(arrY[i]-my);
//--- returning the result
   return(sum/(sizeX-1));
  }
//+------------------------------------------------------------------+
//| Custom correlation ratio                                         |
//+------------------------------------------------------------------+
double Corr(double cov,double dx,double dy)
  {
//--- check
   if(dx==0 || dy==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(cov/MathSqrt(dx)/MathSqrt(dy));
  }
//+------------------------------------------------------------------+
//| Custom correlation variance                                      |
//+------------------------------------------------------------------+
double VarianceOfCorr(double p,int size)
  {
//--- check
   if(size==1)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(MathPow(1.0-MathPow(p,2),2)/(size-1));
  }
//+------------------------------------------------------------------+
//| por sequence autocorrelation                                     |
//+------------------------------------------------------------------+
double AutoCorr(double &arr[],double mx,int por)
  {
//--- check
   if(por<0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- initializing
   int    size=ArraySize(arr)-por;
   double sum1=0.0;
   double sum2=0.0;
   double sum3=0.0;
//--- calculation
   for(int i=0; i<size; i++)
     {
      sum1+=(arr[i]-mx)*(arr[i+por]-mx);
      sum2+=MathPow(arr[i]-mx,2);
      sum3+=MathPow(arr[i+por]-mx,2);
     }
//--- check
   if(!sum2 || !sum3)
     {
      Print(__FUNCTION__+": incorrect summ value");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(sum1/MathSqrt(sum2)/MathSqrt(sum3));
  }
//+------------------------------------------------------------------+
//| Autocorrelation function (ACF)                                   |
//+------------------------------------------------------------------+
bool AutoCorrFunc(double &arr[],double mx,int count,double &res[])
  {
   int size=ArraySize(arr);
   count++;
//--- check
   if(count>size)
     {
      Print(__FUNCTION__+": the error variable");
      return(false);
     }
//--- calculation
   double sum1=0.0;
   double sum2=0.0;
   ArrayResize(res,count);
   res[0]=1.0;
   for(int i=1; i<count; i++)
      res[i]=AutoCorr(arr,mx,i);
//---
   return true;
  }
//+------------------------------------------------------------------+
//| a ratio in the y=a*x+b linear regression equation                |
//+------------------------------------------------------------------+
double aCoeff(double p,double sigX,double sigY)
  {
//--- check
   if(sigX==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(p*sigY/sigX);
  }
//+------------------------------------------------------------------+
//| b ratio in the y=a*x+b linear regression equation                |
//+------------------------------------------------------------------+
double bCoeff(double mx,double my,double a)
  {
   return(my-a*mx);
  }
//+------------------------------------------------------------------+
//| Writes the linear regression errors into e[] array               |
//+------------------------------------------------------------------+
bool LineRegresErrors(double &arrX[],double &arrY[],double a,double b,double &e[])
  {
   int sizeX=ArraySize(arrX);
   int sizeY=ArraySize(arrY);
//--- check
   if(sizeX!=sizeY)
     {
      Print(__FUNCTION__+" dimensional arrays are not equal");
      return(false);
     }
//--- record
   ArrayResize(e,sizeX);
   for(int i=0;i<sizeX;i++)
      e[i]=arrY[i]-a*arrX[i]-b;
//---
   return true;
  }
//+------------------------------------------------------------------+
//| Variance of the errors of -deviations from the regression line  |
//| (it is accepted that Me=0)                                       |
//+------------------------------------------------------------------+
double eVariance(double &e[])
  {
   int size=ArraySize(e);
//--- check
   if(size==2)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   for(int i=0;i<size;i++)
      sum+=MathPow(e[i],2);
//--- returning the result
   return(sum/(size-2));
  }
//+------------------------------------------------------------------+
//|  parameter variance in the y=a*x+b linear regression equation   |
//+------------------------------------------------------------------+
double aVariance(int size,double dx,double de)
  {
//--- check
   if(dx==0 || size==1)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(de/dx/(size-1));
  }
//+------------------------------------------------------------------+
//| b parameter variance in the y=a*x+b linear regression equation   |
//+------------------------------------------------------------------+
double bVariance(double &arr[],double dx,double de)
  {
//--- check
   if(dx==0)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//---
   int size=ArraySize(arr);
//--- check
   if(size<=0)
     {
      Print(__FUNCTION__+": array size error");
      return(EMPTY_VALUE);
     }
//--- calculation
   double sum=0.0;
   for(int i=0;i<size;i++)
      sum+=MathPow(arr[i],2);
//--- returning the result
   return(de*(size-1)*sum/dx/size);
  }
//+------------------------------------------------------------------+
//| Determination ratio                                              |
//+------------------------------------------------------------------+
double DeterminationCoeff(int size,double dy,double de)
  {
//--- check
   if(dy==0 || size==1)
     {
      Print(__FUNCTION__+": the error variable");
      return(EMPTY_VALUE);
     }
//--- returning the result
   return(1.0-de*(size-2)/dy/(size-1));
  }
//+------------------------------------------------------------------+
//| Splits an array of arr[n][2] type in two arrays                  |
//+------------------------------------------------------------------+
void ArraySeparate(double &arr[][2],double &arrX[],double &arrY[])
  {
   int size=ArraySize(arr)/2;
//---
   ArrayResize(arrX,size);
   ArrayResize(arrY,size);
//---
   for(int i=0;i<size;i++)
     {
      arrX[i]=arr[i][0];
      arrY[i]=arr[i][1];
     }
  }
//+------------------------------------------------------------------+
//| Joins two arrays into one of arr[n][2] type                      |
//+------------------------------------------------------------------+
void ArrayUnion(double &arrX[],double &arrY[],double &arr[][2])
  {
   int size=ArraySize(arrX);
//---
   ArrayResize(arr,size*2);
   for(int i=0;i<size;i++)
     {
      arr[i][0]=arrX[i];
      arr[i][1]=arrY[i];
     }
  }
//+------------------------------------------------------------------+
//| Writes one-dimensional array to *.csv file                       |
//+------------------------------------------------------------------+
bool WriteArray(double &arr[],string name="")
  {
//--- check
   if(name=="")
      name=TimeToString(TimeLocal(),TIME_DATE|TIME_SECONDS);
//--- opening the file
   int handle=FileOpen(name+".csv",FILE_CSV|FILE_WRITE);
//--- check
   if(handle!=INVALID_HANDLE)
     {
      int size=ArraySize(arr);
      for(int i=0;i<size;i++)
         FileWrite(handle,DoubleToString(arr[i],8));
      FileClose(handle);
     }
//--- file open error
   else
     {
      Print(__FUNCTION__+": file open error");
      return(false);
     }
//---
   return(true);
  }
//+------------------------------------------------------------------+
//| Writes two-dimensional array of arr[n][2] type to *.csv file     |
//+------------------------------------------------------------------+
bool WriteArray2(double &arr[][2],string name="")
  {
//--- check
   if(name=="")
      name=TimeToString(TimeLocal(),TIME_DATE|TIME_SECONDS);
//--- opening the file
   int handle=FileOpen(name+".csv",FILE_CSV|FILE_WRITE);
//--- check
   if(handle!=INVALID_HANDLE)
     {
      int size=ArraySize(arr)/2;
      for(int i=0;i<size;i++)
         FileWrite(handle,DoubleToString(arr[i][0],8),DoubleToString(arr[i][1],8));
      FileClose(handle);
     }
//--- file open error
   else
     {
      Print(__FUNCTION__+": file open error");
      return(false);
     }
//---
   return(true);
  }
//+------------------------------------------------------------------+
