//+------------------------------------------------------------------+
//|                                              Builov_RSI_Pro.mq5 |
//|                              Copyright © 2025, Aleksandr Builov |
//|                        https://www.mql5.com/en/users/nomercysorry/seller |
//+------------------------------------------------------------------+
#property copyright "Copyright © 2025, Aleksandr Builov"
#property link      "https://www.mql5.com/en/users/nomercysorry/seller"
#property version   "1.00"
#property description "RSI Pro Trading System - Advanced RSI-based strategy with enhanced filters"
/*
   Builov RSI Pro Trading Engine v1.0
   Advanced RSI strategy with spread filter and momentum confirmation
*/
#include <Trade\PositionInfo.mqh>
#include <Trade\Trade.mqh>
#include <Trade\SymbolInfo.mqh>  
#include <Trade\AccountInfo.mqh>
#include <Trade\DealInfo.mqh>
#include <Trade\OrderInfo.mqh>
#include <Expert\Money\MoneyFixedMargin.mqh>
CPositionInfo  m_position;                   // trade position object
CTrade         m_trade;                      // trading object
CSymbolInfo    m_symbol;                     // symbol info object
CAccountInfo   m_account;                    // account info wrapper
CDealInfo      m_deal;                       // deals object
COrderInfo     m_order;                      // pending orders object
CMoneyFixedMargin *m_money;
//+------------------------------------------------------------------+
//| Enum Lot or Risk Management                                      |
//+------------------------------------------------------------------+
enum ENUM_LOT_OR_RISK
  {
   lots_min=0, // Minimum Lot Size
   lot=1,      // Fixed Lot Size
   risk=2,     // Risk Percentage per Trade
  };
//--- Input Parameters - Risk Management
input group "=== Risk Management ==="
input ushort   Builov_StopLoss          = 0;           // Stop Loss (pips)
input ushort   Builov_TakeProfit        = 0;           // Take Profit (pips)
input ENUM_LOT_OR_RISK Builov_LotOrRisk = lots_min;    // Money Management Type
input double   Builov_VolumeLotOrRisk   = 3.0;         // Lot Size or Risk Value
//--- Input Parameters - Trailing Stop
input group "=== Trailing Stop ==="
input ushort   Builov_TrailingFrequency = 10;          // Trailing Check Frequency (seconds, <10 = new bar only)
input ushort   Builov_TrailingStop      = 25;          // Trailing Stop Distance (pips)
input ushort   Builov_TrailingStep      = 5;           // Trailing Step (pips)
//--- Input Parameters - RSI Indicator Settings
input group "=== RSI Indicator Settings ==="
input int      Builov_RSI_Period                = 14;             // RSI Period
input double   Builov_RSI_Level_Down            = 35.0;           // RSI Oversold Level
input double   Builov_RSI_Level_Up              = 65.0;           // RSI Overbought Level
input uchar    Builov_CurrentBar                 = 1;              // Current Bar Index
input bool     Builov_UseIndicatorVisualization  = true;           // Show Indicator on Chart
//--- Input Parameters - Trading Filters
input group "=== Trading Filters ==="
input bool     Builov_OnlyOnePosition   = false;       // Allow Only One Position
input bool     Builov_ReverseSignals    = false;       // Reverse Trading Signals
input bool     Builov_CloseOpposite     = false;       // Close Opposite Positions
input ushort   Builov_MaxSpread         = 30;          // Maximum Spread (pips) - NEW FILTER
input bool     Builov_UseMomentumFilter = true;        // Use Momentum Confirmation - NEW FILTER
input double   Builov_MinMomentum       = 0.5;          // Minimum RSI movement for signal confirmation
//--- Input Parameters - System Settings
input group "=== System Settings ==="
input ushort   Builov_SignalsFrequency  = 9;           // Signal Search Frequency (seconds, <10 = new bar only)
input bool     Builov_PrintLog          = false;       // Enable Detailed Logging
input ulong    Builov_Magic             = 395933019;   // Magic Number
//---
ulong  ExtSlippage=10;              // Slippage

double ExtStopLoss      = 0.0;      // Stop Loss      -> double
double ExtTakeProfit    = 0.0;      // Take Profit    -> double
double ExtTrailingStop  = 0.0;      // Trailing Stop  -> double
double ExtTrailingStep  = 0.0;      // Trailing Step  -> double
double ExtMaxSpread      = 0.0;      // Maximum Spread -> double

int    handle_iCustom;              // variable for storing the handle of the RSI indicator 

double   ExtAdjustedPoint;                   // point value adjusted for 3 or 5 points
datetime ExtLastTrailing         = 0;        // "0" -> D'1970.01.01 00:00';
datetime ExtLastSignals          = 0;        // "0" -> D'1970.01.01 00:00';
datetime ExtPrevBars             = 0;        // "0" -> D'1970.01.01 00:00';
//+------------------------------------------------------------------+
//| Structure Need Positions                                         |
//+------------------------------------------------------------------+
struct STRUCT_NEED_POSITION
  {
   ENUM_POSITION_TYPE pos_type;              // position type
   double            volume;                 // position volume (if "0.0" -> the lot is "Money management")
   double            lot_coefficient;        // lot coefficient
   double            stop_loss;              // position stop loss, in pips * ExtAdjustedPoint (if "0.0" -> the ExtStopLoss)
   double            take_profit;            // position take profit, in pips * ExtAdjustedPoint (if "0.0" -> the ExtTakeProfit)
   bool              waiting_transaction;    // waiting transaction, "true" -> it's forbidden to trade, we expect a transaction
   ulong             waiting_order_ticket;   // waiting order ticket, ticket of the expected order
   bool              transaction_confirmed;  // transaction confirmed, "true" -> transaction confirmed
   ulong             magic;                  // magic
   //--- Constructor 
                     STRUCT_NEED_POSITION()
     {
      pos_type                   = WRONG_VALUE;
      volume                     = 0.0;
      lot_coefficient            = 0.0;
      stop_loss                  = 0.0;
      take_profit                = 0.0;
      waiting_transaction        = false;
      waiting_order_ticket       = 0;
      transaction_confirmed      = false;
      magic                      = 0;
     }
  };
STRUCT_NEED_POSITION SNeedPosition[];
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//---
   if(Builov_TrailingStop!=0 && Builov_TrailingStep==0)
     {
      string err_text=(TerminalInfoString(TERMINAL_LANGUAGE)=="Russian")?
                      "Трейлинг невозможен: параметр \"Trailing Step\" равен нулю!":
                      "Trailing is not possible: parameter \"Trailing Step\" is zero!";
      //--- when testing, we will only output to the log about incorrect input parameters
      if(MQLInfoInteger(MQL_TESTER))
        {
         Print(__FUNCTION__,", ERROR: ",err_text);
         return(INIT_FAILED);
        }
      else // if the Expert Advisor is run on the chart, tell the user about the error
        {
         Alert(__FUNCTION__,", ERROR: ",err_text);
         return(INIT_PARAMETERS_INCORRECT);
        }
     }
//---
   if(!m_symbol.Name(Symbol())) // sets symbol name
      return(INIT_FAILED);
   RefreshRates();
//---
   m_trade.SetExpertMagicNumber(Builov_Magic);
   m_trade.SetMarginMode();
   m_trade.SetTypeFillingBySymbol(m_symbol.Name());
   m_trade.SetDeviationInPoints(ExtSlippage);
//--- tuning for 3 or 5 digits
   int digits_adjust=1;
   if(m_symbol.Digits()==3 || m_symbol.Digits()==5)
      digits_adjust=10;
   ExtAdjustedPoint=m_symbol.Point()*digits_adjust;

   ExtStopLoss       = Builov_StopLoss        * ExtAdjustedPoint;
   ExtTakeProfit     = Builov_TakeProfit      * ExtAdjustedPoint;
   ExtTrailingStop   = Builov_TrailingStop    * ExtAdjustedPoint;
   ExtTrailingStep   = Builov_TrailingStep    * ExtAdjustedPoint;
   ExtMaxSpread      = Builov_MaxSpread       * ExtAdjustedPoint;
//--- check the input parameter "Lots"
   string err_text="";
   if(Builov_LotOrRisk==lot)
     {
      if(!CheckVolumeValue(Builov_VolumeLotOrRisk,err_text))
        {
         //--- when testing, we will only output to the log about incorrect input parameters
         if(MQLInfoInteger(MQL_TESTER))
           {
            Print(__FUNCTION__,", ERROR: ",err_text);
            return(INIT_FAILED);
           }
         else // if the Expert Advisor is run on the chart, tell the user about the error
           {
            Alert(__FUNCTION__,", ERROR: ",err_text);
            return(INIT_PARAMETERS_INCORRECT);
           }
        }
     }
   else if(Builov_LotOrRisk==risk)
     {
      if(m_money!=NULL)
         delete m_money;
      m_money=new CMoneyFixedMargin;
      if(m_money!=NULL)
        {
         if(Builov_VolumeLotOrRisk<1 || Builov_VolumeLotOrRisk>100)
           {
            Print("The value for \"Money management\" (",DoubleToString(Builov_VolumeLotOrRisk,2),") -> invalid parameters");
            Print("   parameter must be in the range: from 1.00 to 100.00");
            return(INIT_FAILED);
           }
         if(!m_money.Init(GetPointer(m_symbol),Period(),m_symbol.Point()*digits_adjust))
            return(INIT_FAILED);
         m_money.Percent(Builov_VolumeLotOrRisk);
        }
      else
        {
         Print(__FUNCTION__,", ERROR: Object CMoneyFixedMargin is NULL");
         return(INIT_FAILED);
        }
     }
//--- create handle of the standard RSI indicator
   handle_iCustom=iRSI(m_symbol.Name(),Period(),Builov_RSI_Period,PRICE_CLOSE);
//--- if the handle is not created 
   if(handle_iCustom==INVALID_HANDLE)
     {
      //--- tell about the failure and output the error code 
      PrintFormat("Failed to create handle of the RSI indicator for the symbol %s/%s, error code %d",
                  m_symbol.Name(),
                  EnumToString(Period()),
                  GetLastError());
      //--- the indicator is stopped early 
      return(INIT_FAILED);
     }
//---
   int debug=MQLInfoInteger(MQL_DEBUG);
   int profiler=MQLInfoInteger(MQL_PROFILER);
   int tester=MQLInfoInteger(MQL_TESTER);
   int forward=MQLInfoInteger(MQL_FORWARD);
   int optimization=MQLInfoInteger(MQL_OPTIMIZATION);
   int visual_mode=MQLInfoInteger(MQL_VISUAL_MODE);
//---
   if(Builov_UseIndicatorVisualization)
      if((debug==1 && tester==0) || (debug==0 && tester==0)) // F5 OR Online
        {
         int windows_total=(int)ChartGetInteger(0,CHART_WINDOWS_TOTAL);
         ChartIndicatorAdd(0,windows_total,handle_iCustom);
        }
//---
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//---
   if(m_money!=NULL)
      delete m_money;
//---
   int debug=MQLInfoInteger(MQL_DEBUG);
   int profiler=MQLInfoInteger(MQL_PROFILER);
   int tester=MQLInfoInteger(MQL_TESTER);
   int forward=MQLInfoInteger(MQL_FORWARD);
   int optimization=MQLInfoInteger(MQL_OPTIMIZATION);
   int visual_mode=MQLInfoInteger(MQL_VISUAL_MODE);
//---
   if(Builov_UseIndicatorVisualization)
      if((debug==1 && tester==0) || (debug==0 && tester==0)) // F5 OR Online
        {
         int windows_total=(int)ChartGetInteger(0,CHART_WINDOWS_TOTAL);
         for(int i=windows_total-1;i>=0;i--)
           {
            for(int j=ChartIndicatorsTotal(0,i)-1;j>=0;j--)
               ChartIndicatorDelete(0,i,ChartIndicatorName(0,i,j));
           }
        }
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
   int size_need_position=ArraySize(SNeedPosition);
   if(size_need_position>0)
     {
      for(int i=size_need_position-1;i>=0;i--)
        {
         if(SNeedPosition[i].waiting_transaction)
           {
            if(!SNeedPosition[i].transaction_confirmed)
              {
               Print("transaction_confirmed: ",SNeedPosition[i].transaction_confirmed);   return;
              }
            else if(SNeedPosition[i].transaction_confirmed)
              {
               ArrayRemove(SNeedPosition,i,1);  return;
              }
           }
         if(SNeedPosition[i].pos_type==POSITION_TYPE_BUY)
           {
            if(Builov_CloseOpposite || Builov_OnlyOnePosition)
              {
               int count_buys=0;    double volume_buys=0.0;    double volume_biggest_buys=0.0;
               int count_sells=0;   double volume_sells=0.0;   double volume_biggest_sells=0.0;
               CalculateAllPositions(count_buys,volume_buys,volume_biggest_buys,
                                     count_sells,volume_sells,volume_biggest_sells);
               if(Builov_CloseOpposite)
                 {
                  if(count_sells>0)
                    {
                     double level;
                     if(FreezeStopsLevels(level))
                        ClosePositions(POSITION_TYPE_SELL,level);
                     return;
                    }
                 }
               if(Builov_OnlyOnePosition)
                 {
                  if(count_buys+count_sells==0)
                    {
                     double level;
                     if(FreezeStopsLevels(level))
                       {
                        SNeedPosition[i].waiting_transaction=true;
                        OpenPosition(i,POSITION_TYPE_BUY,
                                     SNeedPosition[i].volume,SNeedPosition[i].lot_coefficient,
                                     SNeedPosition[i].stop_loss,SNeedPosition[i].take_profit,
                                     level,SNeedPosition[i].magic);
                       }
                     return;
                    }
                  else
                     ArrayRemove(SNeedPosition,i,1);
                  return;
                 }
              }
            double level;
            if(FreezeStopsLevels(level))
              {
               SNeedPosition[i].waiting_transaction=true;
               OpenPosition(i,POSITION_TYPE_BUY,
                            SNeedPosition[i].volume,SNeedPosition[i].lot_coefficient,
                            SNeedPosition[i].stop_loss,SNeedPosition[i].take_profit,
                            level,SNeedPosition[i].magic);
              }
            return;
           }
         if(SNeedPosition[i].pos_type==POSITION_TYPE_SELL)
           {
            if(Builov_CloseOpposite || Builov_OnlyOnePosition)
              {
               int count_buys=0;    double volume_buys=0.0;    double volume_biggest_buys=0.0;
               int count_sells=0;   double volume_sells=0.0;   double volume_biggest_sells=0.0;
               CalculateAllPositions(count_buys,volume_buys,volume_biggest_buys,
                                     count_sells,volume_sells,volume_biggest_sells);
               if(Builov_CloseOpposite)
                 {
                  if(count_buys>0)
                    {
                     double level;
                     if(FreezeStopsLevels(level))
                        ClosePositions(POSITION_TYPE_BUY,level);
                     return;
                    }
                 }
               if(Builov_OnlyOnePosition)
                 {
                  if(count_buys+count_sells==0)
                    {
                     double level;
                     if(FreezeStopsLevels(level))
                       {
                        SNeedPosition[i].waiting_transaction=true;
                        OpenPosition(i,POSITION_TYPE_SELL,
                                     SNeedPosition[i].volume,SNeedPosition[i].lot_coefficient,
                                     SNeedPosition[i].stop_loss,SNeedPosition[i].take_profit,
                                     level,SNeedPosition[i].magic);
                       }
                     return;
                    }
                  else
                     ArrayRemove(SNeedPosition,i,1);
                  return;
                 }
              }
            double level;
            if(FreezeStopsLevels(level))
              {
               SNeedPosition[i].waiting_transaction=true;
               OpenPosition(i,POSITION_TYPE_SELL,
                            SNeedPosition[i].volume,SNeedPosition[i].lot_coefficient,
                            SNeedPosition[i].stop_loss,SNeedPosition[i].take_profit,
                            level,SNeedPosition[i].magic);
              }
            return;
           }
        }
     }
//---
   if(Builov_TrailingFrequency>=10) // trailing no more than once every 10 seconds
     {
      datetime time_current=TimeCurrent();
      if(time_current-ExtLastTrailing>10)
        {
         double level;
         if(FreezeStopsLevels(level))
            Trailing(level);
         else
            return;
         ExtLastTrailing=time_current;
        }
     }
   if(Builov_SignalsFrequency>=10) // search for trading signals no more than once every 10 seconds
     {
      datetime time_current=TimeCurrent();
      if(time_current-ExtLastSignals>10)
        {
         //--- search for trading signals
         if(!RefreshRates())
           {
            ExtPrevBars=0; return;
           }
         if(!SearchTradingSignals())
           {
            ExtPrevBars=0; return;
           }
         ExtLastSignals=time_current;
        }
     }
//--- we work only at the time of the birth of new bar
   datetime time_0=iTime(m_symbol.Name(),Period(),0);
   if(time_0==ExtPrevBars)
      return;
   ExtPrevBars=time_0;
   if(Builov_TrailingFrequency<10) // trailing only at the time of the birth of new bar
     {
      double level;
      if(FreezeStopsLevels(level))
         Trailing(level);
     }
   if(Builov_SignalsFrequency<10) // search for trading signals only at the time of the birth of new bar
     {
      if(!RefreshRates())
        {
         ExtPrevBars=0; return;
        }
      //--- search for trading signals
      if(!SearchTradingSignals())
        {
         ExtPrevBars=0; return;
        }
     }
//---
  }
//+------------------------------------------------------------------+
//| TradeTransaction function                                        |
//+------------------------------------------------------------------+
void OnTradeTransaction(const MqlTradeTransaction &trans,
                        const MqlTradeRequest &request,
                        const MqlTradeResult &result)
  {
//--- get transaction type as enumeration value 
   ENUM_TRADE_TRANSACTION_TYPE type=trans.type;
//--- if transaction is result of addition of the transaction in history
   if(type==TRADE_TRANSACTION_DEAL_ADD)
     {
      long     deal_ticket       =0;
      long     deal_order        =0;
      long     deal_time         =0;
      long     deal_time_msc     =0;
      long     deal_type         =-1;
      long     deal_entry        =-1;
      long     deal_magic        =0;
      long     deal_reason       =-1;
      long     deal_position_id  =0;
      double   deal_volume       =0.0;
      double   deal_price        =0.0;
      double   deal_commission   =0.0;
      double   deal_swap         =0.0;
      double   deal_profit       =0.0;
      string   deal_symbol       ="";
      string   deal_comment      ="";
      string   deal_external_id  ="";
      if(HistoryDealSelect(trans.deal))
        {
         deal_ticket       =HistoryDealGetInteger(trans.deal,DEAL_TICKET);
         deal_order        =HistoryDealGetInteger(trans.deal,DEAL_ORDER);
         deal_time         =HistoryDealGetInteger(trans.deal,DEAL_TIME);
         deal_time_msc     =HistoryDealGetInteger(trans.deal,DEAL_TIME_MSC);
         deal_type         =HistoryDealGetInteger(trans.deal,DEAL_TYPE);
         deal_entry        =HistoryDealGetInteger(trans.deal,DEAL_ENTRY);
         deal_magic        =HistoryDealGetInteger(trans.deal,DEAL_MAGIC);
         deal_reason       =HistoryDealGetInteger(trans.deal,DEAL_REASON);
         deal_position_id  =HistoryDealGetInteger(trans.deal,DEAL_POSITION_ID);

         deal_volume       =HistoryDealGetDouble(trans.deal,DEAL_VOLUME);
         deal_price        =HistoryDealGetDouble(trans.deal,DEAL_PRICE);
         deal_commission   =HistoryDealGetDouble(trans.deal,DEAL_COMMISSION);
         deal_swap         =HistoryDealGetDouble(trans.deal,DEAL_SWAP);
         deal_profit       =HistoryDealGetDouble(trans.deal,DEAL_PROFIT);

         deal_symbol       =HistoryDealGetString(trans.deal,DEAL_SYMBOL);
         deal_comment      =HistoryDealGetString(trans.deal,DEAL_COMMENT);
         deal_external_id  =HistoryDealGetString(trans.deal,DEAL_EXTERNAL_ID);
        }
      else
         return;
      ENUM_DEAL_ENTRY enum_deal_entry=(ENUM_DEAL_ENTRY)deal_entry;
      if(deal_symbol==m_symbol.Name() && deal_magic==Builov_Magic)
        {
         if(deal_type==DEAL_TYPE_BUY || deal_type==DEAL_TYPE_SELL)
           {
            int size_need_position=ArraySize(SNeedPosition);
            if(size_need_position>0)
              {
               for(int i=0;i<size_need_position;i++)
                 {
                  if(SNeedPosition[i].waiting_transaction)
                     if(SNeedPosition[i].waiting_order_ticket==deal_order)
                       {
                        Print(__FUNCTION__," Transaction confirmed");
                        SNeedPosition[i].transaction_confirmed=true;
                        break;
                       }
                 }
              }
           }
        }
     }
  }
//+------------------------------------------------------------------+
//| Refreshes the symbol quotes data                                 |
//+------------------------------------------------------------------+
bool RefreshRates(void)
  {
//--- refresh rates
   if(!m_symbol.RefreshRates())
     {
      Print("RefreshRates error");
      return(false);
     }
//--- protection against the return value of "zero"
   if(m_symbol.Ask()==0 || m_symbol.Bid()==0)
      return(false);
//---
   return(true);
  }
//+------------------------------------------------------------------+
//| Check Spread Filter - NEW FUNCTION                               |
//+------------------------------------------------------------------+
bool CheckSpreadFilter(void)
  {
   if(Builov_MaxSpread==0)
      return(true); // Spread filter disabled
   
   double current_spread=(m_symbol.Ask()-m_symbol.Bid())/ExtAdjustedPoint;
   if(current_spread>Builov_MaxSpread)
     {
      if(Builov_PrintLog)
         Print("Spread filter: Current spread (",DoubleToString(current_spread,1),
               " pips) exceeds maximum (",Builov_MaxSpread," pips)");
      return(false);
     }
   return(true);
  }
//+------------------------------------------------------------------+
//| Check Momentum Filter - NEW FUNCTION                            |
//+------------------------------------------------------------------+
bool CheckMomentumFilter(const double rsi_current,const double rsi_previous)
  {
   if(!Builov_UseMomentumFilter)
      return(true); // Momentum filter disabled
   
   // For BUY: RSI should be rising (current > previous)
   // For SELL: RSI should be falling (current < previous)
   // This adds confirmation to the signal
   
   if(rsi_current>Builov_RSI_Level_Down && rsi_previous<Builov_RSI_Level_Down)
     {
      // Potential BUY signal - check if RSI is rising
      if(rsi_current-rsi_previous>=Builov_MinMomentum)
         return(true);
     }
   if(rsi_current<Builov_RSI_Level_Up && rsi_previous>Builov_RSI_Level_Up)
     {
      // Potential SELL signal - check if RSI is falling
      if(rsi_previous-rsi_current>=Builov_MinMomentum)
         return(true);
     }
   
   return(false);
  }
//+------------------------------------------------------------------+
//| Check the correctness of the position volume                     |
//+------------------------------------------------------------------+
bool CheckVolumeValue(double volume,string &error_description)
  {
//--- minimal allowed volume for trade operations
   double min_volume=m_symbol.LotsMin();
   if(volume<min_volume)
     {
      if(TerminalInfoString(TERMINAL_LANGUAGE)=="Russian")
         error_description=StringFormat("Объем меньше минимально допустимого SYMBOL_VOLUME_MIN=%.2f",min_volume);
      else
         error_description=StringFormat("Volume is less than the minimal allowed SYMBOL_VOLUME_MIN=%.2f",min_volume);
      return(false);
     }
//--- maximal allowed volume of trade operations
   double max_volume=m_symbol.LotsMax();
   if(volume>max_volume)
     {
      if(TerminalInfoString(TERMINAL_LANGUAGE)=="Russian")
         error_description=StringFormat("Объем больше максимально допустимого SYMBOL_VOLUME_MAX=%.2f",max_volume);
      else
         error_description=StringFormat("Volume is greater than the maximal allowed SYMBOL_VOLUME_MAX=%.2f",max_volume);
      return(false);
     }
//--- get minimal step of volume changing
   double volume_step=m_symbol.LotsStep();
   int ratio=(int)MathRound(volume/volume_step);
   if(MathAbs(ratio*volume_step-volume)>0.0000001)
     {
      if(TerminalInfoString(TERMINAL_LANGUAGE)=="Russian")
         error_description=StringFormat("Объем не кратен минимальному шагу SYMBOL_VOLUME_STEP=%.2f, ближайший правильный объем %.2f",
                                        volume_step,ratio*volume_step);
      else
         error_description=StringFormat("Volume is not a multiple of the minimal step SYMBOL_VOLUME_STEP=%.2f, the closest correct volume is %.2f",
                                        volume_step,ratio*volume_step);
      return(false);
     }
   error_description="Correct volume value";
   return(true);
  }
//+------------------------------------------------------------------+
//| Lot Check                                                        |
//+------------------------------------------------------------------+
double LotCheck(double lots)
  {
//--- calculate maximum volume
   double volume=NormalizeDouble(lots,2);
   double stepvol=m_symbol.LotsStep();
   if(stepvol>0.0)
      volume=stepvol*MathFloor(volume/stepvol);
//---
   double minvol=m_symbol.LotsMin();
   if(volume<minvol)
      volume=0.0;
//---
   double maxvol=m_symbol.LotsMax();
   if(volume>maxvol)
      volume=maxvol;
   return(volume);
  }
//+------------------------------------------------------------------+
//| Check Freeze and Stops levels                                    |
//+------------------------------------------------------------------+
bool FreezeStopsLevels(double &level)
  {
//--- check Freeze and Stops levels
   if(!RefreshRates() || !m_symbol.Refresh())
      return(false);
//--- FreezeLevel -> for pending order and modification
   double freeze_level=m_symbol.FreezeLevel()*m_symbol.Point();
   if(freeze_level==0.0)
      freeze_level=(m_symbol.Ask()-m_symbol.Bid())*3.0;
//--- StopsLevel -> for TakeProfit and StopLoss
   double stop_level=m_symbol.StopsLevel()*m_symbol.Point();
   if(stop_level==0.0)
      stop_level=(m_symbol.Ask()-m_symbol.Bid())*3.0;

   if(freeze_level<=0.0 || stop_level<=0.0)
      return(false);

   level=(freeze_level>stop_level)?freeze_level:stop_level;

   double spread=m_symbol.Spread()*ExtAdjustedPoint;
   level=(level>spread)?level:spread;
//---
   return(true);
  }
//+------------------------------------------------------------------+
//| Open position                                                    |
//|   double stop_loss                                               |
//|      -> pips * ExtAdjustedPoint (if "0.0" -> the ExtStopLoss)    |
//|   double take_profit                                             |
//|      -> pips * ExtAdjustedPoint (if "0.0" -> the ExtTakeProfit)  |
//+------------------------------------------------------------------+
void OpenPosition(const int index_in_structure,const ENUM_POSITION_TYPE pos_type,
                  const double volume,const double lot_coefficient,
                  const double stop_loss,const double take_profit,
                  const double level,const ulong magic)
  {
//--- buy
   if(pos_type==POSITION_TYPE_BUY)
     {
      double price=m_symbol.Ask();

      double sl=0.0;
      if(stop_loss<=0.0)
        {
         sl=(Builov_StopLoss==0)?0.0:price-ExtStopLoss;
         if(sl!=0.0 && ExtStopLoss<level) // check sl
            sl=price-level;
        }
      else
        {
         sl=(stop_loss==0.0)?0.0:price-stop_loss;
         if(sl!=0.0 && stop_loss<level) // check sl
            sl=price-level;
        }

      double tp=0.0;
      if(take_profit<=0.0)
        {
         tp=(Builov_TakeProfit==0)?0.0:price+ExtTakeProfit;
         if(tp!=0.0 && ExtTakeProfit<level) // check price
            tp=price+level;
        }
      else
        {
         tp=(take_profit==0)?0.0:price+take_profit;
         if(tp!=0.0 && take_profit<level) // check price
            tp=price+level;
        }

      OpenBuy(index_in_structure,volume,lot_coefficient,sl,tp);
     }
//--- sell
   if(pos_type==POSITION_TYPE_SELL)
     {
      double price=m_symbol.Bid();

      double sl=0.0;
      if(stop_loss<=0.0)
        {
         sl=(Builov_StopLoss==0)?0.0:price+ExtStopLoss;
         if(sl!=0.0 && ExtStopLoss<level) // check sl
            sl=price+level;
        }
      else
        {
         sl=(stop_loss==0.0)?0.0:price+stop_loss;
         if(sl!=0.0 && stop_loss<level) // check sl
            sl=price+level;
        }

      double tp=0.0;
      if(take_profit<=0.0)
        {
         tp=(Builov_TakeProfit==0)?0.0:price-ExtTakeProfit;
         if(tp!=0.0 && ExtTakeProfit<level) // check tp
            tp=price-level;
        }
      else
        {
         tp=(take_profit==0)?0.0:price-take_profit;
         if(tp!=0.0 && take_profit<level) // check tp
            tp=price-level;
        }

      OpenSell(index_in_structure,volume,lot_coefficient,sl,tp);
     }
  }
//+------------------------------------------------------------------+
//| Open Buy position                                                |
//+------------------------------------------------------------------+
void OpenBuy(const int index_in_structure,const double volume,const double lot_coefficient,double sl,double tp)
  {
   sl=m_symbol.NormalizePrice(sl);
   tp=m_symbol.NormalizePrice(tp);

   double long_lot=0.0;
   if(volume>0.0)
      long_lot=volume;
   else
     {
      if(Builov_LotOrRisk==risk)
        {
         long_lot=m_money.CheckOpenLong(m_symbol.Ask(),sl);
         if(Builov_PrintLog)
            Print("sl=",DoubleToString(sl,m_symbol.Digits()),
                  ", CheckOpenLong: ",DoubleToString(long_lot,2),
                  ", Balance: ",    DoubleToString(m_account.Balance(),2),
                  ", Equity: ",     DoubleToString(m_account.Equity(),2),
                  ", FreeMargin: ", DoubleToString(m_account.FreeMargin(),2));
         if(long_lot==0.0)
           {
            ArrayRemove(SNeedPosition,index_in_structure,1);
            if(Builov_PrintLog)
               Print(__FUNCTION__,", ERROR: method CheckOpenLong returned the value of 0.0");
            return;
           }
        }
      else if(Builov_LotOrRisk==lot)
         long_lot=Builov_VolumeLotOrRisk;
      else if(Builov_LotOrRisk==lots_min)
         long_lot=m_symbol.LotsMin();
      else
        {
         ArrayRemove(SNeedPosition,index_in_structure,1);
         return;
        }
     }
   if(lot_coefficient>0.0)
     {
      long_lot=LotCheck(long_lot*lot_coefficient);
      if(long_lot==0)
        {
         ArrayRemove(SNeedPosition,index_in_structure,1);
         if(Builov_PrintLog)
            Print(__FUNCTION__,", ERROR: LotCheck returned the 0.0");
         return;
        }
     }

   if(m_symbol.LotsLimit()>0.0)
     {
      int count_buys = 0;  double volume_buys   = 0.0;   double volume_biggest_buys = 0.0;
      int count_sells= 0;  double volume_sells  = 0.0;   double volume_biggest_sells= 0.0;
      CalculateAllPositions(count_buys,volume_buys,volume_biggest_buys,
                            count_sells,volume_sells,volume_biggest_sells);
      if(volume_buys+volume_sells+long_lot>m_symbol.LotsLimit())
        {
         ArrayRemove(SNeedPosition,index_in_structure,1);
         Print("#0 Buy, Volume Buy (",DoubleToString(volume_buys,2),
               ") + Volume Sell (",DoubleToString(volume_sells,2),
               ") + Volume long (",DoubleToString(long_lot,2),
               ") > Lots Limit (",DoubleToString(m_symbol.LotsLimit(),2),")");
         return;
        }
     }
//--- check volume before OrderSend to avoid "not enough money" error (CTrade)
   double free_margin_check= m_account.FreeMarginCheck(m_symbol.Name(),ORDER_TYPE_BUY,long_lot,m_symbol.Ask());
   double margin_check     = m_account.MarginCheck(m_symbol.Name(),ORDER_TYPE_SELL,long_lot,m_symbol.Bid());
   if(free_margin_check>margin_check)
     {
      if(m_trade.Buy(long_lot,m_symbol.Name(),m_symbol.Ask(),sl,tp,"Builov RSI Pro")) // CTrade::Buy -> "true"
        {
         if(m_trade.ResultDeal()==0)
           {
            if(m_trade.ResultRetcode()==10009) // trade order went to the exchange
              {
               SNeedPosition[index_in_structure].waiting_transaction=true;
               SNeedPosition[index_in_structure].waiting_order_ticket=m_trade.ResultOrder();
              }
            else
              {
               SNeedPosition[index_in_structure].waiting_transaction=false;
               if(Builov_PrintLog)
                  Print("#1 Buy -> false. Result Retcode: ",m_trade.ResultRetcode(),
                        ", description of result: ",m_trade.ResultRetcodeDescription());
              }
            if(Builov_PrintLog)
               PrintResultTrade(m_trade,m_symbol);
           }
         else
           {
            if(m_trade.ResultRetcode()==10009)
              {
               SNeedPosition[index_in_structure].waiting_transaction=true;
               SNeedPosition[index_in_structure].waiting_order_ticket=m_trade.ResultOrder();
              }
            else
              {
               SNeedPosition[index_in_structure].waiting_transaction=false;
               if(Builov_PrintLog)
                  Print("#2 Buy -> true. Result Retcode: ",m_trade.ResultRetcode(),
                        ", description of result: ",m_trade.ResultRetcodeDescription());
              }
            if(Builov_PrintLog)
               PrintResultTrade(m_trade,m_symbol);
           }
        }
      else
        {
         SNeedPosition[index_in_structure].waiting_transaction=false;
         if(Builov_PrintLog)
            Print("#3 Buy -> false. Result Retcode: ",m_trade.ResultRetcode(),
                  ", description of result: ",m_trade.ResultRetcodeDescription());
         if(Builov_PrintLog)
            PrintResultTrade(m_trade,m_symbol);
        }
     }
   else
     {
      ArrayRemove(SNeedPosition,index_in_structure,1);
      if(Builov_PrintLog)
         Print(__FUNCTION__,", ERROR: method CAccountInfo::FreeMarginCheck returned the value ",DoubleToString(free_margin_check,2));
      return;
     }
//---
  }
//+------------------------------------------------------------------+
//| Open Sell position                                               |
//+------------------------------------------------------------------+
void OpenSell(const int index_in_structure,const double volume,const double lot_coefficient,double sl,double tp)
  {
   sl=m_symbol.NormalizePrice(sl);
   tp=m_symbol.NormalizePrice(tp);

   double short_lot=0.0;
   if(volume>0.0)
      short_lot=volume;
   else
     {
      if(Builov_LotOrRisk==risk)
        {
         short_lot=m_money.CheckOpenShort(m_symbol.Bid(),sl);
         if(Builov_PrintLog)
            Print("sl=",DoubleToString(sl,m_symbol.Digits()),
                  ", CheckOpenShort: ",DoubleToString(short_lot,2),
                  ", Balance: ",    DoubleToString(m_account.Balance(),2),
                  ", Equity: ",     DoubleToString(m_account.Equity(),2),
                  ", FreeMargin: ", DoubleToString(m_account.FreeMargin(),2));
         if(short_lot==0.0)
           {
            ArrayRemove(SNeedPosition,index_in_structure,1);
            if(Builov_PrintLog)
               Print(__FUNCTION__,", ERROR: method CheckOpenShort returned the value of \"0.0\"");
            return;
           }
        }
      else if(Builov_LotOrRisk==lot)
         short_lot=Builov_VolumeLotOrRisk;
      else if(Builov_LotOrRisk==lots_min)
         short_lot=m_symbol.LotsMin();
      else
        {
         ArrayRemove(SNeedPosition,index_in_structure,1);
         return;
        }
     }
   if(lot_coefficient>0.0)
     {
      short_lot=LotCheck(short_lot*lot_coefficient);
      if(short_lot==0)
        {
         ArrayRemove(SNeedPosition,index_in_structure,1);
         if(Builov_PrintLog)
            Print(__FUNCTION__,", ERROR: LotCheck returned the 0.0");
         return;
        }
     }
   if(m_symbol.LotsLimit()>0.0)
     {
      int count_buys=0;    double volume_buys=0.0;    double volume_biggest_buys=0.0;
      int count_sells=0;   double volume_sells=0.0;   double volume_biggest_sells=0.0;
      CalculateAllPositions(count_buys,volume_buys,volume_biggest_buys,
                            count_sells,volume_sells,volume_biggest_sells);
      if(volume_buys+volume_sells+short_lot>m_symbol.LotsLimit())
        {
         ArrayRemove(SNeedPosition,index_in_structure,1);
         Print("#0 Sell, Volume Buy (",DoubleToString(volume_buys,2),
               ") + Volume Sell (",DoubleToString(volume_sells,2),
               ") + Volume short (",DoubleToString(short_lot,2),
               ") > Lots Limit (",DoubleToString(m_symbol.LotsLimit(),2),")");
         return;
        }
     }
//--- check volume before OrderSend to avoid "not enough money" error (CTrade)
   double free_margin_check= m_account.FreeMarginCheck(m_symbol.Name(),ORDER_TYPE_SELL,short_lot,m_symbol.Bid());
   double margin_check     = m_account.MarginCheck(m_symbol.Name(),ORDER_TYPE_SELL,short_lot,m_symbol.Bid());
   if(free_margin_check>margin_check)
     {
      if(m_trade.Sell(short_lot,m_symbol.Name(),m_symbol.Bid(),sl,tp,"Builov RSI Pro")) // CTrade::Sell -> "true"
        {
         if(m_trade.ResultDeal()==0)
           {
            if(m_trade.ResultRetcode()==10009) // trade order went to the exchange
              {
               SNeedPosition[index_in_structure].waiting_transaction=true;
               SNeedPosition[index_in_structure].waiting_order_ticket=m_trade.ResultOrder();
              }
            else
              {
               SNeedPosition[index_in_structure].waiting_transaction=false;
               if(Builov_PrintLog)
                  Print("#1 Sell -> false. Result Retcode: ",m_trade.ResultRetcode(),
                        ", description of result: ",m_trade.ResultRetcodeDescription());
              }
            if(Builov_PrintLog)
               PrintResultTrade(m_trade,m_symbol);
           }
         else
           {
            if(m_trade.ResultRetcode()==10009)
              {
               SNeedPosition[index_in_structure].waiting_transaction=true;
               SNeedPosition[index_in_structure].waiting_order_ticket=m_trade.ResultOrder();
              }
            else
              {
               SNeedPosition[index_in_structure].waiting_transaction=false;
               if(Builov_PrintLog)
                  Print("#2 Sell -> true. Result Retcode: ",m_trade.ResultRetcode(),
                        ", description of result: ",m_trade.ResultRetcodeDescription());
              }
            if(Builov_PrintLog)
               PrintResultTrade(m_trade,m_symbol);
           }
        }
      else
        {
         SNeedPosition[index_in_structure].waiting_transaction=false;
         if(Builov_PrintLog)
            Print("#3 Sell -> false. Result Retcode: ",m_trade.ResultRetcode(),
                  ", description of result: ",m_trade.ResultRetcodeDescription());
         if(Builov_PrintLog)
            PrintResultTrade(m_trade,m_symbol);
        }
     }
   else
     {
      ArrayRemove(SNeedPosition,index_in_structure,1);
      if(Builov_PrintLog)
         Print(__FUNCTION__,", ERROR: method CAccountInfo::FreeMarginCheck returned the value ",DoubleToString(free_margin_check,2));
      return;
     }
//---
  }
//+------------------------------------------------------------------+
//| Print CTrade result                                              |
//+------------------------------------------------------------------+
void PrintResultTrade(CTrade &trade,CSymbolInfo &symbol)
  {
   Print("File: ",__FILE__,", symbol: ",symbol.Name());
   Print("Code of request result: "+IntegerToString(trade.ResultRetcode()));
   Print("code of request result as a string: "+trade.ResultRetcodeDescription());
   Print("Deal ticket: "+IntegerToString(trade.ResultDeal()));
   Print("Order ticket: "+IntegerToString(trade.ResultOrder()));
   Print("Volume of deal or order: "+DoubleToString(trade.ResultVolume(),2));
   Print("Price, confirmed by broker: "+DoubleToString(trade.ResultPrice(),symbol.Digits()));
   Print("Current bid price: "+DoubleToString(symbol.Bid(),symbol.Digits())+" (the requote): "+DoubleToString(trade.ResultBid(),symbol.Digits()));
   Print("Current ask price: "+DoubleToString(symbol.Ask(),symbol.Digits())+" (the requote): "+DoubleToString(trade.ResultAsk(),symbol.Digits()));
   Print("Broker comment: "+trade.ResultComment());
  }
//+------------------------------------------------------------------+
//| Get value of buffers                                             |
//+------------------------------------------------------------------+
bool iGetArray(const int handle,const int buffer,const int start_pos,const int count,double &arr_buffer[])
  {
   bool result=true;
   if(!ArrayIsDynamic(arr_buffer))
     {
      Print("This a no dynamic array!");
      return(false);
     }
   ArrayFree(arr_buffer);
//--- reset error code 
   ResetLastError();
//--- fill a part of the iBands array with values from the indicator buffer
   int copied=CopyBuffer(handle,buffer,start_pos,count,arr_buffer);
   if(copied!=count)
     {
      //--- if the copying fails, tell the error code 
      PrintFormat("Failed to copy data from the indicator, error code %d",GetLastError());
      //--- quit with zero result - it means that the indicator is considered as not calculated 
      return(false);
     }
   return(result);
  }
//+------------------------------------------------------------------+
//| Trailing                                                         |
//|   Builov_TrailingStop: min distance from price to Stop Loss          |
//+------------------------------------------------------------------+
void Trailing(const double stop_level)
  {
   if(Builov_TrailingStop==0)
      return;
   for(int i=PositionsTotal()-1;i>=0;i--) // returns the number of open positions
      if(m_position.SelectByIndex(i))
         if(m_position.Symbol()==m_symbol.Name() && m_position.Magic()==Builov_Magic)
           {
            if(m_position.PositionType()==POSITION_TYPE_BUY)
              {
               if(m_position.PriceCurrent()-m_position.PriceOpen()>ExtTrailingStop+ExtTrailingStep)
                 {
                  double new_sl=m_symbol.NormalizePrice(m_position.PriceCurrent()-ExtTrailingStop);
                  // Ensure new SL is at least stop_level away from current price
                  if(new_sl<m_position.PriceCurrent()-stop_level)
                     new_sl=m_symbol.NormalizePrice(m_position.PriceCurrent()-stop_level);
                  
                  if(m_position.StopLoss()<new_sl || m_position.StopLoss()==0)
                    {
                     if(m_position.PriceCurrent()-new_sl>=stop_level)
                       {
                        if(!m_trade.PositionModify(m_position.Ticket(),new_sl,m_position.TakeProfit()))
                          {
                           if(Builov_PrintLog)
                              Print("Modify ",m_position.Ticket(),
                                    " Position -> false. Result Retcode: ",m_trade.ResultRetcode(),
                                    ", description of result: ",m_trade.ResultRetcodeDescription());
                          }
                        RefreshRates();
                        m_position.SelectByIndex(i);
                        if(Builov_PrintLog)
                           PrintResultModify(m_trade,m_symbol,m_position);
                       }
                    }
                 }
              }
            else
              {
               if(m_position.PriceOpen()-m_position.PriceCurrent()>ExtTrailingStop+ExtTrailingStep)
                 {
                  double new_sl=m_symbol.NormalizePrice(m_position.PriceCurrent()+ExtTrailingStop);
                  // Ensure new SL is at least stop_level away from current price
                  if(new_sl<m_position.PriceCurrent()+stop_level)
                     new_sl=m_symbol.NormalizePrice(m_position.PriceCurrent()+stop_level);
                  
                  if((m_position.StopLoss()>new_sl || m_position.StopLoss()==0))
                    {
                     double sl_distance=new_sl-m_position.PriceCurrent();
                     if(sl_distance>=stop_level)
                       {
                        if(!m_trade.PositionModify(m_position.Ticket(),new_sl,m_position.TakeProfit()))
                          {
                           if(Builov_PrintLog)
                              Print("Modify ",m_position.Ticket(),
                                    " Position -> false. Result Retcode: ",m_trade.ResultRetcode(),
                                    ", description of result: ",m_trade.ResultRetcodeDescription());
                          }
                        RefreshRates();
                        m_position.SelectByIndex(i);
                        if(Builov_PrintLog)
                           PrintResultModify(m_trade,m_symbol,m_position);
                       }
                    }
                 }
              }

           }
  }
//+------------------------------------------------------------------+
//| Print CTrade result                                              |
//+------------------------------------------------------------------+
void PrintResultModify(CTrade &trade,CSymbolInfo &symbol,CPositionInfo &position)
  {
   Print("File: ",__FILE__,", symbol: ",symbol.Name());
   Print("Code of request result: "+IntegerToString(trade.ResultRetcode()));
   Print("code of request result as a string: "+trade.ResultRetcodeDescription());
   Print("Deal ticket: "+IntegerToString(trade.ResultDeal()));
   Print("Order ticket: "+IntegerToString(trade.ResultOrder()));
   Print("Volume of deal or order: "+DoubleToString(trade.ResultVolume(),2));
   Print("Price, confirmed by broker: "+DoubleToString(trade.ResultPrice(),symbol.Digits()));
   Print("Current bid price: "+DoubleToString(symbol.Bid(),symbol.Digits())+" (the requote): "+DoubleToString(trade.ResultBid(),symbol.Digits()));
   Print("Current ask price: "+DoubleToString(symbol.Ask(),symbol.Digits())+" (the requote): "+DoubleToString(trade.ResultAsk(),symbol.Digits()));
   Print("Broker comment: "+trade.ResultComment());
   Print("Freeze Level: "+DoubleToString(symbol.FreezeLevel(),0),", Stops Level: "+DoubleToString(symbol.StopsLevel(),0));
   Print("Price of position opening: "+DoubleToString(position.PriceOpen(),symbol.Digits()));
   Print("Price of position's Stop Loss: "+DoubleToString(position.StopLoss(),symbol.Digits()));
   Print("Price of position's Take Profit: "+DoubleToString(position.TakeProfit(),symbol.Digits()));
   Print("Current price by position: "+DoubleToString(position.PriceCurrent(),symbol.Digits()));
  }
//+------------------------------------------------------------------+
//| Close positions                                                  |
//+------------------------------------------------------------------+
void ClosePositions(const ENUM_POSITION_TYPE pos_type,const double level)
  {
   for(int i=PositionsTotal()-1;i>=0;i--) // returns the number of current positions
      if(m_position.SelectByIndex(i)) // selects the position by index for further access to its properties
         if(m_position.Symbol()==m_symbol.Name() && m_position.Magic()==Builov_Magic)
            if(m_position.PositionType()==pos_type)
              {
               if(m_position.PositionType()==POSITION_TYPE_BUY)
                  if(MathAbs(m_symbol.Bid()-m_position.PriceOpen())>=level)
                     m_trade.PositionClose(m_position.Ticket()); // close a position by the specified symbol
               if(m_position.PositionType()==POSITION_TYPE_SELL)
                  if(MathAbs(m_symbol.Ask()-m_position.PriceOpen())>=level)
                     m_trade.PositionClose(m_position.Ticket()); // close a position by the specified symbol
              }
  }
//+------------------------------------------------------------------+
//| Calculate all positions                                          |
//+------------------------------------------------------------------+
void CalculateAllPositions(int &count_buys,double &volume_buys,double &volume_biggest_buys,
                           int &count_sells,double &volume_sells,double &volume_biggest_sells)
  {
   count_buys  = 0;  volume_buys   = 0.0; volume_biggest_buys  = 0.0;
   count_sells = 0;  volume_sells  = 0.0; volume_biggest_sells = 0.0;
   for(int i=PositionsTotal()-1;i>=0;i--)
      if(m_position.SelectByIndex(i)) // selects the position by index for further access to its properties
         if(m_position.Symbol()==m_symbol.Name() && m_position.Magic()==Builov_Magic)
           {
            if(m_position.PositionType()==POSITION_TYPE_BUY)
              {
               count_buys++;
               volume_buys+=m_position.Volume();
               if(m_position.Volume()>volume_biggest_buys)
                  volume_biggest_buys=m_position.Volume();
               continue;
              }
            else if(m_position.PositionType()==POSITION_TYPE_SELL)
              {
               count_sells++;
               volume_sells+=m_position.Volume();
               if(m_position.Volume()>volume_biggest_sells)
                  volume_biggest_sells=m_position.Volume();
              }
           }
  }
//+------------------------------------------------------------------+
//| Search trading signals                                           |
//+------------------------------------------------------------------+
bool SearchTradingSignals(void)
  {
   //--- Check spread filter first
   if(!CheckSpreadFilter())
      return(false);
   
   double rsi[];
   ArraySetAsSeries(rsi,true);
   int start_pos=0,count=Builov_CurrentBar+2;
   if(!iGetArray(handle_iCustom,0,start_pos,count,rsi))
      return(false);

   int size_need_position=ArraySize(SNeedPosition);
   
   //--- BUY Signal: RSI crosses above oversold level
   if(rsi[Builov_CurrentBar]>Builov_RSI_Level_Down && rsi[Builov_CurrentBar+1]<Builov_RSI_Level_Down)
     {
      //--- Apply momentum filter if enabled
      if(Builov_UseMomentumFilter)
        {
         if(!CheckMomentumFilter(rsi[Builov_CurrentBar],rsi[Builov_CurrentBar+1]))
           {
            if(Builov_PrintLog)
               Print("BUY signal rejected by momentum filter");
            return(true);
           }
        }
      
      if(!Builov_ReverseSignals)
        {
         ArrayResize(SNeedPosition,size_need_position+1);
         SNeedPosition[size_need_position].pos_type=POSITION_TYPE_BUY;
         SNeedPosition[size_need_position].magic=Builov_Magic;
         return(true);
        }
      else
        {
         ArrayResize(SNeedPosition,size_need_position+1);
         SNeedPosition[size_need_position].pos_type=POSITION_TYPE_SELL;
         SNeedPosition[size_need_position].magic=Builov_Magic;
         return(true);
        }
     }
   
   //--- SELL Signal: RSI crosses below overbought level
   if(rsi[Builov_CurrentBar]<Builov_RSI_Level_Up && rsi[Builov_CurrentBar+1]>Builov_RSI_Level_Up)
     {
      //--- Apply momentum filter if enabled
      if(Builov_UseMomentumFilter)
        {
         if(!CheckMomentumFilter(rsi[Builov_CurrentBar],rsi[Builov_CurrentBar+1]))
           {
            if(Builov_PrintLog)
               Print("SELL signal rejected by momentum filter");
            return(true);
           }
        }
      
      if(!Builov_ReverseSignals)
        {
         ArrayResize(SNeedPosition,size_need_position+1);
         SNeedPosition[size_need_position].pos_type=POSITION_TYPE_SELL;
         SNeedPosition[size_need_position].magic=Builov_Magic;
         return(true);
        }
      else
        {
         ArrayResize(SNeedPosition,size_need_position+1);
         SNeedPosition[size_need_position].pos_type=POSITION_TYPE_BUY;
         SNeedPosition[size_need_position].magic=Builov_Magic;
         return(true);
        }
     }
//---
   return(true);
  }
//+------------------------------------------------------------------+


