//+------------------------------------------------------------------+
//|                                                      FileSet.mqh |
//|                   Copyright 2009-2017, MetaQuotes Software Corp. |
//|                                              http://www.mql5.com |
//+------------------------------------------------------------------+
#include "File.mqh"
#include<Generic\HashMap.mqh>

//+------------------------------------------------------------------+
//| Class CFileSet                                                   |
//| Purpose: Class of operations with set files.                     |
//|          Derives from class CFile.                               |
//+------------------------------------------------------------------+
class CFileSet : public CFile
  {
protected:
   int               m_map_handle;
   CHashMap<string, string> m_map;
public:
                     CFileSet(void);
                    ~CFileSet(void);
   //--- methods for working with files
   int               Open(const string file_name,const int open_flags);
   //--- methods of access to protected data
   int               Count(void) {return m_map.Count();}
   bool              ContainsKey(string key) {return m_map.ContainsKey(key);}
   //--- methods to access data
   template<typename TValue>
   bool              TryGetValue(const string key, TValue &value);
   bool              TryGetValue(const string key, string &value);
private:
   void              Map(void);
   bool              SettingStringParser(const string setting_string, string &key, string &value);
   string            GetInputValueString(string value_string);
  };
//+------------------------------------------------------------------+
//| Constructor                                                      |
//+------------------------------------------------------------------+
CFileSet::CFileSet(void)
  {
  }
//+------------------------------------------------------------------+
//| Destructor                                                       |
//+------------------------------------------------------------------+
CFileSet::~CFileSet(void)
  {
  }
//+------------------------------------------------------------------+
//| Open the set file                                                |
//+------------------------------------------------------------------+
int CFileSet::Open(const string file_name,const int open_flags)
  {
   return(CFile::Open(file_name,open_flags|FILE_TXT));
  }
//+------------------------------------------------------------------+
//| Gets the value associated with the specified key.                |
//+------------------------------------------------------------------+
template<typename TValue>
bool CFileSet::TryGetValue(const string key, TValue &value)
  {
   if(m_handle!=m_map_handle)
      Map();
   string value_string;
   if(m_map.TryGetValue(key,value_string))
     {
      string type = typename(value);
      if(type == "float" || type == "double")
         value = (TValue)StringToDouble(GetInputValueString(value_string));
      else
         if(type == "bool")
            value = (TValue)(GetInputValueString(value_string) == "true");
         else
            value = (TValue)StringToInteger(GetInputValueString(value_string));
      return true;
     }
   return false;
  }
//+------------------------------------------------------------------+
//| Gets the value associated with the specified key.                |
//+------------------------------------------------------------------+
bool CFileSet::TryGetValue(const string key, string &value)
  {
   if(m_handle!=m_map_handle)
      Map();
   return m_map.TryGetValue(key,value);
  }
//+------------------------------------------------------------------+
//| Load the data and map to the CHashMap from the opened file.      |
//+------------------------------------------------------------------+
void CFileSet::Map(void)
  {
   m_map.Clear();
//---
   if(m_handle==INVALID_HANDLE)
      Print("INVALID_HANDLE error while loading the data.");
   while(!CFile::IsEnding())
     {
      string setting_string = FileReadString(m_handle);
      string key, value;
      if(SettingStringParser(setting_string,key,value))
         m_map.Add(key, value);
     }
   m_map_handle = m_handle;
  }
//+------------------------------------------------------------------+
//| Parse given string and return key, value format.                 |
//+------------------------------------------------------------------+
bool CFileSet::SettingStringParser(const string setting_string,string &key,string &value)
  {
   char set_char[];
   int set_char_size;
   set_char_size = StringToCharArray(setting_string,set_char);
//--- if input string is empty
   if(set_char_size<1)
     {
      return false;
     }
//--- if the the input string is the comment
   if(set_char[0]==';')
      return false;
//---
   bool operator_flag = false;
   for(int i=0; i<set_char_size; i++)
     {
      //--- if the '=' operator for key value separator found
      if(set_char[i]=='=')
        {
         operator_flag = true;
         continue;
        }
      if(!operator_flag)
         key+= CharToString(set_char[i]);
      else
         value+= CharToString(set_char[i]);
     }
   return true;
  }
//+------------------------------------------------------------------+
//| Parse given string and return the input value.                   |
//+------------------------------------------------------------------+
string CFileSet::GetInputValueString(string value_string)
  {
   char value_char[];
   int value_char_size;
   value_char_size = StringToCharArray(value_string,value_char);
//--- if input string is empty
   if(value_char_size<1)
      return NULL;
   string parameter;
   bool operator_flag = false;
   for(int i=0; i<value_char_size; i++)
     {
      //--- if the '|' found
      if(value_char[i]=='|')
         return parameter;
      if(!operator_flag)
         parameter+= CharToString(value_char[i]);
     }
   return parameter;
  }
//+------------------------------------------------------------------+
