//+---------------------------------------------------+
//|                                NLD_TP_Include.mq5 |
//|                             http://www.nelodi.com |
//|                     Copyright (c) 2019 NELODI.com |
//+---------------------------------------------------+

#define sigPriceLine    0
#define sigSignalLine   1
#define sigSupportLine  2
#define sigBorderLine   3
#define sigLineShift    4
//+------------------------------------------------------------------+

enum MyPriceType
  {
   ptHighest,    // { > } Highest
   ptWeighted,   // { ~ } Weighted
   ptLowest      // { < } Lowest
  };
//+------------------------------------------------------------------+

int MyTimeFrame=1;    // Period: Frame {1}
int MyMinInterval=1;  // Period: Signal {1}
int MyMaxInterval=20; // Period: Border {20}
double MyOverlap=10;  // Period: Overlap {10}
int MyScan=0;         // Period: Scan [0..7] {0}
int MyIron=0;         // Period: Smoothen {0}

double MySpreadWeight=0; // Spread: Weight {0}
double MySpreadSig=1;    // Spread: On Signal {1}
double MySpreadCld=1;    // Spread: On Border {1}

int MyDelay=0;       // Span: Delay {0}
int MyPriceLine=0;   // Span: Price {0}
int MySignalLine=2;  // Span: Signal {2}
int MyLineShift=4;   // Span: Shift {4}
int MySupportLine=6; // Span: Support {6}
int MyBorderLine=8;  // Span: Border {8}

                     // INPUT parameters
MyPriceType MyHighTop=ptHighest;      // Price: High Top {>}
MyPriceType MyHighCenter=ptWeighted;  // Price: High Center {~}
MyPriceType MyHighBottom=ptLowest;    // Price: High Bottom {<}

MyPriceType MyLowTop=ptHighest;       // Price: Low Top {>}
MyPriceType MyLowCenter=ptWeighted;   // Price: Low Center {~}
MyPriceType MyLowBottom=ptLowest;     // Price: Low Bottom {<}

bool MyPrintTicks=false;

int MySpread=10;

//+---------------------------

datetime ExtPeriodSec;       // Period size in Seconds
datetime ExtIntervalSec;     // Current Interval size in Seconds

datetime ExtMinInterval;  // Min Interval size in Seconds
datetime ExtMaxInterval;  // Max Interval size in Seconds

int minInterval;  // Min Interval size in Periods
int maxInterval;  // Max Interval size in Periods

int ExtMin;
int ExtMax;

int MaxLineShift;
bool MyPush=false;    // Push Forward?

int ExtIdx=0; // Index
int ExtLen=0; // Length

long ExtVolume_Buff[];
long ExtVolumeL_Buff[];
datetime ExtTime_Buff[];
datetime ExtTimeL_Buff[];

double ExtOpen_Buff[];
double ExtClose_Buff[];

double ExtHigh_Buff[];
double ExtLow_Buff[];
double ExtLHigh_Buff[];
double ExtHLow_Buff[];
double ExtWHigh_Buff[];
double ExtWLow_Buff[];

double ExtLine_Buff[][4];

bool needHigh,needLow,needLHigh,needHLow,needWHigh,needWLow;

datetime line_span[5];
int line_shift[5];

datetime lineSpanShift;

double line_value[4];
double lineH_value[4];
double lineL_value[4];

int perLineShift;  // Period LineShift with "Push Forward"

double ExtPip;
double ExtSpread;

datetime ExtStartTime;

datetime ExtLastTF;
datetime ExtLastTL;
int ExtLastZ;
int ExtLastF;
int ExtLastL;

bool ExtReady;
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void MyCalcInit()
  {
   if(MyBorderLine<1) MyBorderLine=1;

   minInterval=MyMinInterval; // Min Interval in Periods
   maxInterval=MyMaxInterval; // Max Interval in Periods

   if(minInterval<1) minInterval=1;
   if(maxInterval<minInterval) maxInterval=minInterval;

   ExtPeriodSec=PeriodSeconds(Period())*MyTimeFrame;

   ExtMinInterval=minInterval*ExtPeriodSec; // Min Interval in Seconds
   ExtMaxInterval=maxInterval*ExtPeriodSec; // Max Interval in Seconds

   ExtIntervalSec=ExtMaxInterval; // Current Interval in Seconds

   if(MyPriceLine>0)
     {
      line_span[sigPriceLine]=ExtMinInterval*MyPriceLine/MyBorderLine;
      line_shift[sigPriceLine]=minInterval*MyPriceLine/MyBorderLine;
     }
   else if(MyPriceLine<0)
     {
      line_span[sigPriceLine]=-ExtPeriodSec*MyPriceLine;
      line_shift[sigPriceLine]=-MyPriceLine;
     }
   else
     {
      line_span[sigPriceLine]=ExtPeriodSec;
      line_shift[sigPriceLine]=1;
     }

   if(MySignalLine>0)
     {
      line_span[sigSignalLine]=ExtMinInterval*MySignalLine/MyBorderLine;
      line_shift[sigSignalLine]=minInterval*MySignalLine/MyBorderLine;
     }
   else if(MySignalLine<0)
     {
      line_span[sigSignalLine]=-ExtPeriodSec*MySignalLine;
      line_shift[sigSignalLine]=-MySignalLine;
     }
   else
     {
      line_span[sigSignalLine]=ExtPeriodSec;
      line_shift[sigSignalLine]=1;
     }

   if(MyLineShift>0)
     {
      line_span[sigLineShift]=(ExtMinInterval+ExtMaxInterval)/2*MyLineShift/MyBorderLine;
      line_shift[sigLineShift]=(minInterval+maxInterval)/2*MyLineShift/MyBorderLine;
     }
   else if(MyLineShift<0)
     {
      line_span[sigLineShift]=-ExtPeriodSec*MyLineShift;
      line_shift[sigLineShift]=-MyLineShift;
     }
   else
     {
      line_span[sigLineShift]=ExtPeriodSec;
      line_shift[sigLineShift]=1;
     }

   if(MySupportLine>0)
     {
      line_span[sigSupportLine]=ExtMaxInterval*MySupportLine/MyBorderLine;
      line_shift[sigSupportLine]=maxInterval*MySupportLine/MyBorderLine;
     }
   else if(MySupportLine<0)
     {
      line_span[sigSupportLine]=-ExtPeriodSec*MySupportLine;
      line_shift[sigSupportLine]=-MySupportLine;
     }
   else
     {
      line_span[sigSupportLine]=ExtPeriodSec;
      line_shift[sigSupportLine]=1;
     }

   line_span[sigBorderLine]=ExtMaxInterval;
   line_shift[sigBorderLine]=maxInterval;

   lineSpanShift=line_span[sigLineShift];

   perLineShift=line_shift[sigLineShift]/2;

   MaxLineShift=line_shift[sigPriceLine];
   for(int k=sigSignalLine;k<=sigBorderLine;k++)
      if(line_shift[k]>MaxLineShift)
         MaxLineShift=line_shift[k];
   MaxLineShift+=line_shift[sigLineShift];

   ExtMax=MaxLineShift*10+10;
   ExtMin=MaxLineShift*2;

   if(ArrayResize(ExtVolume_Buff,ExtMax)<ExtMax) Print("Resize (ExtVolume_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtVolumeL_Buff,ExtMax)<ExtMax) Print("Resize (ExtVolumeL_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtTime_Buff,ExtMax)<ExtMax) Print("Resize (ExtTime_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtTimeL_Buff,ExtMax)<ExtMax) Print("Resize (ExtTimeL_Buff,",ExtMax,") Failed!");

   if(ArrayResize(ExtOpen_Buff,ExtMax)<ExtMax) Print("Resize (ExtOpen_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtClose_Buff,ExtMax)<ExtMax) Print("Resize (ExtClose_Buff,",ExtMax,") Failed!");

   if(ArrayResize(ExtHigh_Buff,ExtMax)<ExtMax) Print("Resize (ExtHigh_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtLow_Buff,ExtMax)<ExtMax) Print("Resize (ExtLow_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtLHigh_Buff,ExtMax)<ExtMax) Print("Resize (ExtLHigh_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtHLow_Buff,ExtMax)<ExtMax) Print("Resize (ExtHLow_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtWHigh_Buff,ExtMax)<ExtMax) Print("Resize (ExtWHigh_Buff,",ExtMax,") Failed!");
   if(ArrayResize(ExtWLow_Buff,ExtMax)<ExtMax) Print("Resize (ExtWLow_Buff,",ExtMax,") Failed!");

   if(ArrayResize(ExtLine_Buff,ExtMax)<ExtMax) Print("Resize (ExtLine_Buff,",ExtMax,") Failed!");

   ExtPip=1;
   for(int i=0;i<_Digits;i++) ExtPip/=10;

   ExtSpread=ExtPip*MySpread;

   needHigh=(MyHighTop==ptHighest) || (MyHighCenter==ptHighest) || (MyHighBottom==ptHighest);
   needLHigh=(MyHighTop==ptLowest) || (MyHighCenter==ptLowest) || (MyHighBottom==ptLowest);
   needWHigh=(MyHighTop==ptWeighted) || (MyHighCenter==ptWeighted) || (MyHighBottom==ptWeighted);

   needLow=(MyLowTop==ptLowest) || (MyLowCenter==ptLowest) || (MyLowBottom==ptLowest);
   needHLow=(MyLowTop==ptHighest) || (MyLowCenter==ptHighest) || (MyLowBottom==ptHighest);
   needWLow=(MyLowTop==ptWeighted) || (MyLowCenter==ptWeighted) || (MyLowBottom==ptWeighted);

   MyCalcReInit(true);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void MyCalcReInit(bool first=false)
  {
//--- Period size in seconds
   ArrayFill(ExtTime_Buff,0,ExtMax,0);
   ArrayFill(ExtTimeL_Buff,0,ExtMax,0);
   ArrayFill(ExtVolume_Buff,0,ExtMax,0);
   ArrayFill(ExtVolumeL_Buff,0,ExtMax,0);

   ArrayFill(ExtOpen_Buff,0,ExtMax,0);
   ArrayFill(ExtClose_Buff,0,ExtMax,0);

   ArrayFill(ExtHigh_Buff,0,ExtMax,0);
   ArrayFill(ExtLHigh_Buff,0,ExtMax,0);
   ArrayFill(ExtWHigh_Buff,0,ExtMax,0);

   ArrayFill(ExtLow_Buff,0,ExtMax,0);
   ArrayFill(ExtHLow_Buff,0,ExtMax,0);
   ArrayFill(ExtWLow_Buff,0,ExtMax,0);

   ArrayFill(ExtLine_Buff,0,ExtMax*4,0);

   ExtIdx=0; // Index
   ExtLen=0; // Length

   ExtLastZ=0;
   ExtLastF=0;
   ExtLastL=0;
   ExtLastTF=0;
   ExtLastTL=0;

   ExtReady=false;
  }
//+------------------------------------------------------------------+

void MyCalc_ReInit()
   {
   Print("ReInit: ",__FUNCTION__,"|",__LINE__," (",ExtLen,"/",ExtMax,")");
   MyCalcReInit();
   }
//+------------------------------------------------------------------+

void MyCalc_Signals(const double sig_T,const double sig_KA,const double sig_KB,const double sig_KC,
                    const double sig_SA,const double sig_SB,const double sig_SC,const double sig_SD,
                    int &openBuySig,int &openSellSig,int &closeBuySig,int &closeSellSig)
  {
   double cTop=MathMax(sig_SC,sig_SD);
   double cBot=MathMin(sig_SC,sig_SD);

   double sig_SAH=MathMax(sig_SA,cBot);
   double sig_SBH=MathMax(sig_SB,cBot);
   double sig_KCH=MathMax(sig_KC,cBot);
   double sig_KBH=MathMax(sig_KB,sig_KCH);
   double sig_KAH=MathMax(sig_KA,sig_KBH);
   double sig_TH=MathMax(sig_T,sig_KAH);

   double sig_SAL=MathMin(sig_SA,cTop);
   double sig_SBL=MathMin(sig_SB,cTop);
   double sig_KCL=MathMin(sig_KC,cTop);
   double sig_KBL=MathMin(sig_KB,sig_KCL);
   double sig_KAL=MathMin(sig_KA,sig_KBL);
   double sig_TL=MathMin(sig_T,sig_KAL);

//--- Open BUY signals
   if(sig_T>=sig_KAH) openBuySig++;
   if(sig_KA>=sig_KBH) openBuySig++;
   if(sig_KB>=sig_KCH) openBuySig++;
   if(sig_KC>=sig_SBH) openBuySig++;
//--- Open SELL signals
   if(sig_T<=sig_KAL) openSellSig++;
   if(sig_KA<=sig_KBL) openSellSig++;
   if(sig_KB<=sig_KCL) openSellSig++;
   if(sig_KC<=sig_SBL) openSellSig++;

//--- Check to close BUY, if we DON'T have a SELL signal yet
   if(openSellSig==0)
     {
      if(sig_T<sig_KAL) closeBuySig++;
      if(sig_KA<sig_KBL) closeBuySig++;
      if(sig_KB<sig_KCL) closeBuySig++;
      if(sig_KC<sig_SBL) closeBuySig++;
     }
//--- Check to close SELL, if we DON'T have a BUY signal yet
   if(openBuySig==0)
     {
      if(sig_T>sig_KAH) closeSellSig++;
      if(sig_KA>sig_KBH) closeSellSig++;
      if(sig_KB>sig_KCH) closeSellSig++;
      if(sig_KC>sig_SBH) closeSellSig++;
     }
  }
//+------------------------------------------------------------------+

void MyCalc_MoveDown()
  {
   if(ExtLen<ExtMax-2 || ExtLen<ExtMin) return;

// ExtIdx = current Index 
// ExtLen = current Length
// ExtMax = max Length
   int k,i,i2,i3;
   i3=ExtLen-ExtMin;

   for(i2=0,i=i3; i<ExtLen; i++,i2++)
     {
      ExtTime_Buff[i2]=ExtTime_Buff[i];
      ExtTimeL_Buff[i2]=ExtTimeL_Buff[i];
      ExtVolume_Buff[i2]=ExtVolume_Buff[i];
      ExtVolumeL_Buff[i2]=ExtVolumeL_Buff[i];

      ExtOpen_Buff[i2]=ExtOpen_Buff[i];
      ExtClose_Buff[i2]=ExtClose_Buff[i];

      ExtHigh_Buff[i2]=ExtHigh_Buff[i];
      ExtLow_Buff[i2]=ExtLow_Buff[i];

      ExtLHigh_Buff[i2]=ExtLHigh_Buff[i];
      ExtHLow_Buff[i2]=ExtHLow_Buff[i];

      ExtWHigh_Buff[i2]=ExtWHigh_Buff[i];
      ExtWLow_Buff[i2]=ExtWLow_Buff[i];

      for(k=0;k<4;k++)
         ExtLine_Buff[i2][k]=ExtLine_Buff[i][k];
     }
   ExtIdx+=i3;
   ExtLen-=i3;
  }
//+------------------------------------------------------------------+

void MyUpdate_Tick(datetime &timeNow,
                   const datetime time,
                   const double ask,
                   const double bid,
                   long tick_volume,
                   const int flags,
                   const bool faster=false)
  {
   bool done=false;
   double price=bid;
   int i=(ExtLen>0?ExtLen-1:0);

   if(ExtLen>0 && time<ExtTimeL_Buff[i])
     {
      Print("ReInit: ",__FUNCTION__,"|",__LINE__," (",ExtLen,"/",ExtMax,")");
      MyCalcReInit();
      i=0;
     }

   MyCalc_MoveDown();

   ExtSpread=ask-bid;
   if(tick_volume==0) tick_volume=1;
   if(faster) timeNow=time; else timeNow=(datetime)MathFloor((double)time/ExtPeriodSec)*ExtPeriodSec;

   if(ExtLen==0 || time>=ExtTime_Buff[i]+ExtPeriodSec)
     {
      i=ExtLen;
      ExtLen++;

      ExtOpen_Buff[i]=price;
      ExtClose_Buff[i]=price;

      ExtLow_Buff[i]=price;
      ExtHLow_Buff[i]=price;
      ExtWLow_Buff[i]=price;

      ExtHigh_Buff[i]=price;
      ExtLHigh_Buff[i]=price;
      ExtWHigh_Buff[i]=price;

      ExtVolume_Buff[i]=tick_volume;
      ExtVolumeL_Buff[i]=tick_volume;

      ExtTime_Buff[i]=(datetime)MathFloor((double)time/ExtPeriodSec)*ExtPeriodSec;
      ExtTimeL_Buff[i]=time;

      if(MyPrintTicks)
        {
         string fst=(faster?"Q":"S");
         if(i>0)
            Print(fst+"Tick ",time," (+",(long)(MathRound(time-ExtTime_Buff[i])),"s) ",ask," | ",bid,"   ",
                  "Prev. O",ExtOpen_Buff[i-1]," C",ExtClose_Buff[i-1]," H",ExtHigh_Buff[i-1]," L",ExtLow_Buff[i-1]);
         else
            Print(fst+"Tick ",time," (+",(long)(MathRound(time-ExtTime_Buff[i])),"s) ",ask," | ",bid);
        }
      ExtLastTF=ExtTime_Buff[i];

      if(i>0 && time<ExtTime_Buff[i]+ExtPeriodSec*MyOverlap) i--; else done=true;
     }

   while(!done)
     {
      if(i==ExtLen-1) ExtClose_Buff[i]=price;

      bool hiUp=false,loDn=false;
      if(price<ExtLow_Buff[i])
        {
         loDn=true;
         ExtLow_Buff[i]=price;
         if(price>ExtHLow_Buff[i])
            ExtHLow_Buff[i]=price;
        }
      if(price>ExtHigh_Buff[i])
        {
         hiUp=true;
         ExtHigh_Buff[i]=price;
         if(price<ExtLHigh_Buff[i])
            ExtLHigh_Buff[i]=price;
        }
      if(time>ExtTimeL_Buff[i])
        {
         long tvol=ExtVolume_Buff[i]+tick_volume;

         if(hiUp) ExtWHigh_Buff[i]=(ExtWHigh_Buff[i]*ExtVolume_Buff[i]+price*tick_volume)/tvol;
         if(loDn) ExtWLow_Buff[i]=(ExtWLow_Buff[i]*ExtVolume_Buff[i]+price*tick_volume)/tvol;

         ExtVolume_Buff[i]=tvol;
         ExtVolumeL_Buff[i]=tick_volume;
        }
      else if(tick_volume>ExtVolumeL_Buff[i])
        {
         long tmp=tick_volume-ExtVolumeL_Buff[i];
         long tvol=ExtVolume_Buff[i]+tmp;

         if(hiUp) ExtWHigh_Buff[i]=(ExtWHigh_Buff[i]*ExtVolume_Buff[i]+price*tmp)/tvol;
         if(loDn) ExtWLow_Buff[i]=(ExtWLow_Buff[i]*ExtVolume_Buff[i]+price*tmp)/tvol;

         ExtVolumeL_Buff[i]=tick_volume;
         ExtVolume_Buff[i]=tvol;
        }
      else
        {
         long tvol=ExtVolume_Buff[i]+1;

         if(hiUp) ExtWHigh_Buff[i]=(ExtWHigh_Buff[i]*ExtVolume_Buff[i]+price)/tvol;
         if(loDn) ExtWLow_Buff[i]=(ExtWLow_Buff[i]*ExtVolume_Buff[i]+price)/tvol;

         ExtVolume_Buff[i]=tvol;
         ExtVolumeL_Buff[i]=1;
        }
      ExtTimeL_Buff[i]=time;

      if(i>0 && time<ExtTime_Buff[i]+ExtPeriodSec*MyOverlap) i--; else done=true;
     }
  }
//+------------------------------------------------------------------+

void MyUpdate_Tick(datetime &timeNow,
                   bool faster=false)
  {
   MqlTick tick;
   if(SymbolInfoTick(Symbol(),tick))
      MyUpdate_Tick(timeNow,tick.time,
                    tick.ask,tick.bid,
                    tick.volume,tick.flags,
                    faster);
  }
//+------------------------------------------------------------------+

void MyUpdate_Period(datetime &timeNow,
                     const datetime time,
                     const double open,
                     const double close,
                     const double high,
                     const double low,
                     long tick_volume,
                     const int spread,
                     const bool faster=false)
  {
   bool done=false;
   int i=(ExtLen>0?ExtLen-1:0);

   if(ExtLen>0 && time<ExtTimeL_Buff[i])
     {
      Print("ReInit: ",__FUNCTION__,"|",__LINE__," (",ExtLen,"/",ExtMax,")");
      MyCalcReInit();
      i=0;
     }

   MyCalc_MoveDown();

   if(tick_volume==0) tick_volume=1;
   if(spread==0) ExtSpread=MySpread*ExtPip; else ExtSpread=spread*ExtPip;
   if(faster) timeNow=time; else timeNow=(datetime)MathFloor((double)time/ExtPeriodSec)*ExtPeriodSec;

   if(ExtLen==0 || time>=ExtTime_Buff[i]+ExtPeriodSec)
     {
      i=ExtLen;
      ExtLen++;

      ExtOpen_Buff[i]=open;
      ExtClose_Buff[i]=close;

      ExtLow_Buff[i]=low;
      ExtHLow_Buff[i]=low;
      ExtWLow_Buff[i]=low;

      ExtHigh_Buff[i]=high;
      ExtLHigh_Buff[i]=high;
      ExtWHigh_Buff[i]=high;

      ExtVolume_Buff[i]=tick_volume;
      ExtVolumeL_Buff[i]=tick_volume;

      ExtTime_Buff[i]=(datetime)MathFloor((double)time/ExtPeriodSec)*ExtPeriodSec;
      ExtTimeL_Buff[i]=time;

      ExtLastTF=ExtTime_Buff[i];

      if(i>0 && time<ExtTime_Buff[i]+ExtPeriodSec*MyOverlap) i--; else done=true;
     }

   while(!done)
     {
      if(i==ExtLen-1) ExtClose_Buff[i]=close;

      bool hiUp=false,loDn=false;
      if(low<ExtLow_Buff[i])
        {
         loDn=true;
         ExtLow_Buff[i]=low;
         if(low>ExtHLow_Buff[i])
            ExtHLow_Buff[i]=low;
        }
      if(high>ExtHigh_Buff[i])
        {
         hiUp=true;
         ExtHigh_Buff[i]=high;
         if(high<ExtLHigh_Buff[i])
            ExtLHigh_Buff[i]=high;
        }

      if(time>ExtTimeL_Buff[i])
        {
         long tvol=ExtVolume_Buff[i]+tick_volume;

         if(hiUp) ExtWHigh_Buff[i]=(ExtWHigh_Buff[i]*ExtVolume_Buff[i]+high*tick_volume)/tvol;
         if(loDn) ExtWLow_Buff[i]=(ExtWLow_Buff[i]*ExtVolume_Buff[i]+low*tick_volume)/tvol;

         ExtVolume_Buff[i]=tvol;
         ExtVolumeL_Buff[i]=tick_volume;
        }
      else if(tick_volume>ExtVolumeL_Buff[i])
        {
         long tmp=tick_volume-ExtVolumeL_Buff[i];
         long tvol=ExtVolume_Buff[i]+tmp;

         if(hiUp) ExtWHigh_Buff[i]=(ExtWHigh_Buff[i]*ExtVolume_Buff[i]+high*tmp)/tvol;
         if(loDn) ExtWLow_Buff[i]=(ExtWLow_Buff[i]*ExtVolume_Buff[i]+low*tmp)/tvol;

         ExtVolumeL_Buff[i]=tick_volume;
         ExtVolume_Buff[i]=tvol;
        }
      else
        {
         long tvol=ExtVolume_Buff[i]+1;

         if(hiUp) ExtWHigh_Buff[i]=(ExtWHigh_Buff[i]*ExtVolume_Buff[i]+high)/tvol;
         if(loDn) ExtWLow_Buff[i]=(ExtWLow_Buff[i]*ExtVolume_Buff[i]+low)/tvol;

         ExtVolume_Buff[i]=tvol;
         ExtVolumeL_Buff[i]=1;
        }
      ExtTimeL_Buff[i]=time;

      if(i>0 && time<ExtTime_Buff[i]+ExtPeriodSec*MyOverlap) i--; else done=true;
     }
  }
//+------------------------------------------------------------------+

void MyUpdate_Rate(datetime &timeNow,
                   MqlRates &rate,
                   const bool faster=false)
  {
   MyUpdate_Period(timeNow,rate.time,
                   rate.open,rate.close,
                   rate.high,rate.low,
                   rate.tick_volume,rate.spread,
                   faster);
  }
//+------------------------------------------------------------------+

int MyUpdate_Rates(datetime &timeNow,
                   int maxCount=0,
                   const bool faster=false)
  {
   if(maxCount<ExtMin) maxCount=ExtMin;
   
   MqlRates rates[];
   int copied=CopyRates(Symbol(),0,0,maxCount,rates);

   if(copied>0)
     {
      for(int i=0; !IsStopped() && i<copied; i++)
         MyUpdate_Rate(timeNow,rates[i],faster);
      return(copied);
     }
   else
      return(0);
  }
//+------------------------------------------------------------------+

bool MyCalc_Period(bool &next,
                   int &pos,
                   int &first,
                   int &last,
                   int &cnt,
                   datetime &timeNow,
                   const datetime &time[],
                   const double &open[],
                   const double &close[],
                   const double &high[],
                   const double &low[],
                   const long &tick_volume[],
                   const int &spread[])
  {
   int rates_total=MathMin(MathMin(
                           MathMin(ArraySize(time),ArraySize(open)),
                           MathMin(ArraySize(close),ArraySize(high))),
                           MathMin(ArraySize(low),ArraySize(tick_volume)));

   if(rates_total<=0) return(false);

   int i,k;

//--- Continue where we left
   datetime timeNext=ExtLastTL;

   pos=ExtLastZ;
   first=ExtLastF;
   last=ExtLastL;
   timeNow=ExtLastTF;
//--- 1st time initialization or Array size changed
   if(ExtIdx+ExtLen>rates_total) // Initialization required
     {
      Print("ReInit: ",__FUNCTION__,"|",__LINE__," (",ExtLen,"/",ExtMax,")");
      MyCalcReInit();
      pos=0;
      first=0;
      last=0;
      timeNow=time[0];
      timeNext=time[0];
     }
   else if(ExtLen>0)
     {
      i=pos-ExtIdx;
      while(first>0 && ExtTime_Buff[i]>timeNow)
        {
         pos--;
         first--;
         last--;
         ExtIdx--;
         timeNow=time[first];
        }
      while(first<rates_total-1 && ExtTime_Buff[i]<time[first])
        {
         pos++;
         first++;
         last++;
         ExtIdx++;
         timeNow=time[first];
        }
     }
//--- Array elements shifted
   if(ExtLen>0 && (timeNow!=time[first] || timeNext!=time[last]))
     {
      while(first>0)
        {
         if(time[first-1]>=timeNow)
           {
            pos--;
            first--;
            last--;
            ExtIdx--;
           }
         else
            break;
        }
      //--- Moving down didn't help? Re-initialize
      if(timeNow!=time[first] || timeNext!=time[last])
        {
         Print("ReInit: ",__FUNCTION__,"|",__LINE__," (",ExtLen,"/",ExtMax,")");
         MyCalcReInit();
         pos=0; first=0; last=0;
         timeNow=time[0];
        }
     }
//--- Moving to the next?
   if(next)
     {
      if(last<rates_total-1)
        {
         //--- Update First/Last period index
         first=last+1;
         last=first;
         timeNow=time[first];
         pos++;
        }
      else
         return(false);
     }
//--- Update "Last" index
   timeNext=timeNow+ExtPeriodSec;
   while(last<rates_total-1)
     {
      if(time[last+1]<timeNext)
        {
         last++;
        }
      else
         break;
     }
//--- Update "Count"
   cnt=last-first+1;

   double open_value,close_value;
   double high_value,low_value;
   double lhigh_value,hlow_value;
   double whigh_value,wlow_value;

   long tick_vol,xvol;

   MyCalc_MoveDown();

   i=pos-ExtIdx; // Index in _Buff[]
   ExtLen=i+1; // Update _Buff[] length
//--- Spread
   if(ArraySize(spread)<first)
      ExtSpread=MySpread*ExtPip;
   else
      ExtSpread=spread[first]*ExtPip;
//--- Time
   timeNow=time[first];
   ExtTime_Buff[i]=timeNow;
//--- Open/Close price
   open_value=open[first];
   close_value=close[last];
   ExtOpen_Buff[i]=open_value;
   ExtClose_Buff[i]=close_value;

//--- Tick Volume
   xvol=tick_volume[last];
   tick_vol=xvol;
   for(k=first;k<last;k++)
      tick_vol+=tick_volume[k];
   ExtVolume_Buff[i]=tick_vol;
//--- Highest, Lowest & Weighted High price
   high_value=high[last];
   lhigh_value=high_value;
   whigh_value=high_value;
   if(needHigh)
      for(k=first;k<last;k++)
         if(high[k]>high_value)
            high_value=high[k];
   if(needLHigh)
      for(k=first;k<last;k++)
         if(high[k]<lhigh_value)
            lhigh_value=high[k];
   if(needWHigh)
     {
      whigh_value=whigh_value*xvol;
      for(k=first;k<last;k++)
         whigh_value=whigh_value+high[k]*tick_volume[k];
      whigh_value=whigh_value/tick_vol;
     }
   ExtHigh_Buff[i]=high_value;
   ExtLHigh_Buff[i]=lhigh_value;
   ExtWHigh_Buff[i]=whigh_value;
//--- Lowest, Highest & Weighted Low price
   low_value=low[last];
   hlow_value=low_value;
   wlow_value=low_value;
   if(needLow)
      for(k=first;k<last;k++)
         if(low[k]<low_value)
            low_value=low[k];
   if(needHLow)
      for(k=first;k<last;k++)
         if(low[k]>hlow_value)
            hlow_value=low[k];
   if(needWLow)
     {
      wlow_value=wlow_value*xvol;
      for(k=first;k<last;k++)
         wlow_value=wlow_value+low[k]*tick_volume[k];
      wlow_value=wlow_value/tick_vol;
     }
   ExtLow_Buff[i]=low_value;
   ExtHLow_Buff[i]=hlow_value;
   ExtWLow_Buff[i]=wlow_value;

   ExtLastZ=pos;
   ExtLastF=first;
   ExtLastL=last;
   ExtLastTF=time[ExtLastF];
   ExtLastTL=time[ExtLastL];

   next=true;
   return(true);
  }
//+------------------------------------------------------------------+

double MyPrice(const MyPriceType typ1,
               const MyPriceType typ2,
               const MyPriceType typ3,
               const double high,
               const double weight,
               const double low)
  {
   double res;
   if(typ1==ptHighest) res=high;
   else if(typ1==ptLowest) res=low;
   else res=weight;
   if(typ2==ptHighest) res=res+high;
   else if(typ2==ptLowest) res=res+low;
   else res=res+weight;
   if(typ3==ptHighest) res=res+high;
   else if(typ3==ptLowest) res=res+low;
   else res=res+weight;
   return(res/3);
  }
//+------------------------------------------------------------------+

void MyHighLow(const datetime timeNext,
               int &k,
               long &tick_vol,
               double &high_value,
               double &lhigh_value,
               double &whigh_value,
               double &low_value,
               double &hlow_value,
               double &wlow_value)
  {
   if(k<=0) return;

   int k2=k;
   if(needHigh)
     {
      k=k2;
      while(k>0 && ExtTime_Buff[k]>timeNext)
        {
         k--;
         if(ExtHigh_Buff[k]>high_value)
            high_value=ExtHigh_Buff[k];
        }
     }
   if(needLow)
     {
      k=k2;
      while(k>0 && ExtTime_Buff[k]>timeNext)
        {
         k--;
         if(ExtLow_Buff[k]<low_value)
            low_value=ExtLow_Buff[k];
        }
     }
   if(needLHigh)
     {
      k=k2;
      while(k>0 && ExtTime_Buff[k]>timeNext)
        {
         k--;
         if(ExtLHigh_Buff[k]<lhigh_value)
            lhigh_value=ExtLHigh_Buff[k];
        }
     }
   if(needHLow)
     {
      k=k2;
      while(k>0 && ExtTime_Buff[k]>timeNext)
        {
         k--;
         if(ExtHLow_Buff[k]>hlow_value)
            hlow_value=ExtHLow_Buff[k];
        }
     }
   if(needWHigh || needWLow)
     {
      k=k2;
      long xvol=tick_vol;
      while(k>0 && ExtTime_Buff[k]>timeNext)
        {
         k--;
         tick_vol+=ExtVolume_Buff[k];
        }
      if(needWHigh)
        {
         k=k2;
         whigh_value=whigh_value*xvol;
         while(k>0 && ExtTime_Buff[k]>timeNext)
           {
            k--;
            whigh_value=whigh_value+ExtWHigh_Buff[k]*ExtVolume_Buff[k];
           }
         whigh_value=whigh_value/tick_vol;
        }
      if(needWLow)
        {
         k=k2;
         wlow_value=wlow_value*xvol;
         while(k>0 && ExtTime_Buff[k]>timeNext)
           {
            k--;
            wlow_value=wlow_value+ExtWLow_Buff[k]*ExtVolume_Buff[k];
           }
         wlow_value=wlow_value/tick_vol;
        }
     }
  }
//+------------------------------------------------------------------+

void MyCalc_Trends(int i,datetime &timeNow)
  {
   int k,lid;
   datetime timeNext;
   double high_value,low_value;
   double lhigh_value,hlow_value;
   double whigh_value,wlow_value;
   long tick_vol;

//--- Separate Periods?
   k=i;
   timeNext=timeNow;
   if(MyDelay>0)
     {
      k-=MyDelay; if(k<0)k=0;
      timeNext-=ExtPeriodSec*MyDelay; // Move "MyDelay" periods back (forward projection)
     }

//--- Get Current Period High/Low prices
   high_value=ExtHigh_Buff[k];
   low_value=ExtLow_Buff[k];

   lhigh_value=ExtLHigh_Buff[k];
   hlow_value=ExtHLow_Buff[k];

   whigh_value=ExtWHigh_Buff[k];
   wlow_value=ExtWLow_Buff[k];

   tick_vol=ExtVolume_Buff[k];

   if(MyScan>0)
     {
      lid=0;
      timeNext=timeNow-line_span[lid];

      MyHighLow(timeNext,k,tick_vol,
                high_value,lhigh_value,whigh_value,
                low_value,hlow_value,wlow_value);

      double lineL=MyPrice(MyLowTop,MyLowCenter,MyLowBottom,hlow_value,wlow_value,low_value);
      double lineH=MyPrice(MyHighTop,MyHighCenter,MyHighBottom,high_value,whigh_value,lhigh_value);
      double lineX=(lineL+lineH)/2;

      lineL_value[lid]=lineL;
      lineH_value[lid]=lineH;
      line_value[lid]=lineX;

      datetime timeLast=timeNow-line_span[3];
      datetime timeX=timeNext;

      double lineL0=lineL;
      double lineH0=lineH;
      double lineX0=lineX;
      double lineMin=lineX;
      double lineMax=lineX;

      bool dirOK=false;

      int zScan=9-MyScan;

      timeNext-=ExtPeriodSec;

      for(lid=1;lid<4;lid++)
        {
         lineL0=lineL;
         lineH0=lineH;
         lineX0=lineX;

         if(zScan>=lid*2+1)
           {
            timeNext=timeNow-line_span[lid];
            //-- Exit from Center ...
            do
              {
               MyHighLow(timeNext,k,tick_vol,
                         high_value,lhigh_value,whigh_value,
                         low_value,hlow_value,wlow_value);
               lineL=MyPrice(MyLowTop,MyLowCenter,MyLowBottom,hlow_value,wlow_value,low_value);
               lineH=MyPrice(MyHighTop,MyHighCenter,MyHighBottom,high_value,whigh_value,lhigh_value);
               lineX=(lineL+lineH)/2;

               dirOK=(lineX>=lineMin && lineX<=lineMax);
               if(dirOK)
                 {
                  lineL0=lineL;
                  lineH0=lineH;
                  lineX0=lineX;
                 }

               timeNext-=ExtPeriodSec;
              }
            while(timeNext>timeLast && dirOK);
           }
         else
           {
            MyHighLow(timeNext,k,tick_vol,
                      high_value,lhigh_value,whigh_value,
                      low_value,hlow_value,wlow_value);
            lineL=MyPrice(MyLowTop,MyLowCenter,MyLowBottom,hlow_value,wlow_value,low_value);
            lineH=MyPrice(MyHighTop,MyHighCenter,MyHighBottom,high_value,whigh_value,lhigh_value);
            lineX=(lineL+lineH)/2;

            dirOK=(lineX>=lineMin && lineX<=lineMax);
            if(dirOK)
              {
               lineL0=lineL;
               lineH0=lineH;
               lineX0=lineX;
              }

            timeNext-=ExtPeriodSec;
           }

         if(lineX<lineMin) lineMin=lineX;
         if(lineX>lineMax) lineMax=lineX;

         if(zScan<=lid*2+1)
           {
            //-- Reverse course ...
            do
              {
               MyHighLow(timeNext,k,tick_vol,
                         high_value,lhigh_value,whigh_value,
                         low_value,hlow_value,wlow_value);
               lineL=MyPrice(MyLowTop,MyLowCenter,MyLowBottom,hlow_value,wlow_value,low_value);
               lineH=MyPrice(MyHighTop,MyHighCenter,MyHighBottom,high_value,whigh_value,lhigh_value);
               lineX=(lineL+lineH)/2;

               dirOK=(lineX<=lineMin || lineX>=lineMax);
               if(dirOK)
                 {
                  lineL0=lineL;
                  lineH0=lineH;
                  lineX0=lineX;
                 }

               if(lineX<lineMin) lineMin=lineX;
               if(lineX>lineMax) lineMax=lineX;

               timeNext-=ExtPeriodSec;
              }
            while(timeNext>timeLast && dirOK);
           }

         if(!dirOK) // Course reversed?
           {
            lineL_value[lid]=(lineL+lineL0)/2;
            lineH_value[lid]=(lineH+lineH0)/2;
            line_value[lid]=(lineX+lineX0)/2;
            timeX=timeNext;
           }
         else
           {
            lineL_value[lid]=(lineL+lineL0*2)/3;
            lineH_value[lid]=(lineH+lineH0*2)/3;
            line_value[lid]=(lineX+lineX0*2)/3;
           }
        }

      lineSpanShift=(timeNow-timeX)*line_span[sigLineShift]/line_span[sigBorderLine];
     }
   else
     {
      for(lid=0;lid<4;lid++)
        {
         timeNext=timeNow-line_span[lid];
         MyHighLow(timeNext,k,tick_vol,
                   high_value,lhigh_value,whigh_value,
                   low_value,hlow_value,wlow_value);
         lineL_value[lid]=MyPrice(MyLowTop,MyLowCenter,MyLowBottom,hlow_value,wlow_value,low_value);
         lineH_value[lid]=MyPrice(MyHighTop,MyHighCenter,MyHighBottom,high_value,whigh_value,lhigh_value);
         line_value[lid]=(lineL_value[lid]+lineH_value[lid])/2;
        }
     }

   if(MyIron==1)
     {
      //--- Average span points in current period
      for(int a=3;a>0;a--)
        {
         lineL_value[a]+=lineL_value[a-1]; lineL_value[a]/=2;
         lineH_value[a]+=lineH_value[a-1]; lineH_value[a]/=2;
         line_value[a]+=line_value[a-1]; line_value[a]/=2;
        }
     }
   else if(MyIron>1)
     {
      //--- Iron-out spikes by averaging "MyIron" periods
      for(lid=1;lid<4;lid++)
        {
         int j=i-(MyIron/(4-lid))+1; if(j<0)j=0;
         int cnt=i-j+1;
         if(cnt>1)
           {
            for(int z=j;z<i;z++)
               line_value[lid]+=ExtLine_Buff[z][lid];
            line_value[lid]/=cnt;
           }
        }
     }

   for(lid=0;lid<4;lid++)
      ExtLine_Buff[i][lid]=line_value[lid];

  }
//+------------------------------------------------------------------+

void MyCalc_Cloud(int k,datetime timeNow,
                  double &sig_T,double &sig_K,double &sig_SA,double &sig_SB,
                  double &sig_T2,double &sig_K2,double &sig_SA2,double &sig_SB2)
  {
   datetime timeNext=timeNow;
//--- Separate periods?
   if(MyDelay>0)
     {
      k-=MyDelay; if(k<0)k=0;
      timeNext-=ExtPeriodSec*MyDelay;
     }

   sig_T=line_value[sigPriceLine];
   sig_K=line_value[sigSignalLine];
   sig_SA=line_value[sigSupportLine];
   sig_SB=line_value[sigBorderLine];

   if(MyPush) // SA, SB, SA2 & SB2 pushed forward by "sigLineShift/2" periods while drawing ...
     {
      timeNext=timeNow-lineSpanShift*1/3;  // = 33%
      while(k>0 && ExtTime_Buff[k]>timeNext) k--;
      sig_T2=(sig_T*2+ExtLine_Buff[k][sigPriceLine])/3;
      sig_K2=(sig_K*2+ExtLine_Buff[k][sigSignalLine])/3;
      sig_SA2=(sig_SA*2+ExtLine_Buff[k][sigSupportLine])/3;
      sig_SB2=(sig_SB*2+ExtLine_Buff[k][sigBorderLine])/3;

      timeNext=timeNow-lineSpanShift*1/2;  // = 50%
      while(k>0 && ExtTime_Buff[k]>timeNext) k--;
      sig_T2+=(sig_T*2+ExtLine_Buff[k][sigPriceLine])/3;
      sig_K2+=(sig_K*2+ExtLine_Buff[k][sigSignalLine])/3;
      sig_SA2+=(sig_SA*2+ExtLine_Buff[k][sigSupportLine])/3;
      sig_SB2+=(sig_SB*2+ExtLine_Buff[k][sigBorderLine])/3;

      sig_T2/=2;
      sig_K2/=2;
      sig_SA2/=2;
      sig_SB2/=2;
     }
   else // Normal drawing. Same as above, but shifted "sigLineShift/2" periods forward ...
     {
      timeNext=timeNow-lineSpanShift*1/3; // = 33%
      while(k>0 && ExtTime_Buff[k]>timeNext) k--;
      sig_T2=(sig_T*2+ExtLine_Buff[k][sigPriceLine])/3;
      sig_K2=(sig_K*2+ExtLine_Buff[k][sigSignalLine])/3;

      timeNext=timeNow-lineSpanShift*1/2; // = 50%
      while(k>0 && ExtTime_Buff[k]>timeNext) k--;
      sig_T2+=(sig_T*2+ExtLine_Buff[k][sigPriceLine])/3;
      sig_K2+=(sig_K*2+ExtLine_Buff[k][sigSignalLine])/3;

      sig_SA=ExtLine_Buff[k][sigSupportLine];
      sig_SB=ExtLine_Buff[k][sigBorderLine];

      timeNext=timeNow-lineSpanShift*5/6; // = 88%
      while(k>0 && ExtTime_Buff[k]>timeNext) k--;
      sig_SA2=(sig_SA*2+ExtLine_Buff[k][sigSupportLine])/3;
      sig_SB2=(sig_SB*2+ExtLine_Buff[k][sigBorderLine])/3;

      timeNext=timeNow-lineSpanShift;
      while(k>0 && ExtTime_Buff[k]>timeNext) k--;
      sig_SA2+=(sig_SA*2+ExtLine_Buff[k][sigSupportLine])/3;
      sig_SB2+=(sig_SB*2+ExtLine_Buff[k][sigBorderLine])/3;

      sig_T2/=2;
      sig_K2/=2;
      sig_SA2/=2;
      sig_SB2/=2;
     }
  }
//+------------------------------------------------------------------+

void MyArrayFill(double &ExtBuffer[],
                 const int first,
                 const int cnt,
                 const double sig)
  {
   if(first<1)
      ArrayFill(ExtBuffer,first,cnt,sig);
   else
     {
      double iPrice1=ExtBuffer[first-1];

      if(sig==0 || iPrice1==0)
        {
         ArrayFill(ExtBuffer,first,cnt,sig);
        }
      else
        {
         double iPrice;
         double iPriceD=sig-iPrice1;

         for(int idx2=0;idx2<cnt;idx2++) // 0..1
           {
            iPrice=iPrice1+iPriceD*(idx2+1)/cnt; // *1/2, *2/2
            ExtBuffer[first+idx2]=iPrice; // first, last
           }
        }
     }
  }
//+------------------------------------------------------------------+  
