//+------------------------------------------------------------------+
//|                                                       alglib.mqh |
//|            Copyright 2003-2022 Sergey Bochkanov (ALGLIB project) |
//|                             Copyright 2012-2023, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
//| Implementation of ALGLIB library in MetaQuotes Language 5        |
//|                                                                  |
//| The features of the library include:                             |
//| - Linear algebra (direct algorithms, EVD, SVD)                   |
//| - Solving systems of linear and non-linear equations             |
//| - Interpolation                                                  |
//| - Optimization                                                   |
//| - FFT (Fast Fourier Transform)                                   |
//| - Numerical integration                                          |
//| - Linear and nonlinear least-squares fitting                     |
//| - Ordinary differential equations                                |
//| - Computation of special functions                               |
//| - Descriptive statistics and hypothesis testing                  |
//| - Data analysis - classification, regression                     |
//| - Implementing linear algebra algorithms, interpolation, etc.    |
//|   in high-precision arithmetic (using MPFR)                      |
//|                                                                  |
//| This file is free software; you can redistribute it and/or       |
//| modify it under the terms of the GNU General Public License as   |
//| published by the Free Software Foundation (www.fsf.org); either  |
//| version 2 of the License, or (at your option) any later version. |
//|                                                                  |
//| This program is distributed in the hope that it will be useful,  |
//| but WITHOUT ANY WARRANTY; without even the implied warranty of   |
//| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the     |
//| GNU General Public License for more details.                     |
//+------------------------------------------------------------------+

#include <Object.mqh>
#include "alglibmisc.mqh"
#include "dataanalysis.mqh"
#include "diffequations.mqh"
#include "delegatefunctions.mqh"
#include "fasttransforms.mqh"
#include "integration.mqh"
#include "interpolation.mqh"

//+------------------------------------------------------------------+
//| The main class, which includes functions for users               |
//+------------------------------------------------------------------+
class CAlglib
  {
public:
   //--- function of package alglibmisc
   //--- high quality random number generator
   static void       HQRndRandomize(CHighQualityRandStateShell &state);
   static void       HQRndSeed(const int s1,const int s2,CHighQualityRandStateShell &state);
   static double     HQRndUniformR(CHighQualityRandStateShell &state);
   static int        HQRndUniformI(CHighQualityRandStateShell &state,const int n);
   static double     HQRndNormal(CHighQualityRandStateShell &state);
   static void       HQRndNormalV(CHighQualityRandStateShell &state,int n,CRowDouble &x);
   static void       HQRndNormalV(CHighQualityRandStateShell &state,int n,vector<double> &x);
   static void       HQRndNormalM(CHighQualityRandStateShell &state,int m,int n,CMatrixDouble &x);
   static void       HQRndNormalM(CHighQualityRandStateShell &state,int m,int n,matrix<double> &x);
   static void       HQRndUnit2(CHighQualityRandStateShell &state,double &x,double &y);
   static void       HQRndNormal2(CHighQualityRandStateShell &state,double &x1,double &x2);
   static double     HQRndExponential(CHighQualityRandStateShell &state,const double lambdav);
   static double     HQRndDiscrete(CHighQualityRandStateShell &state,int n,CRowDouble &x);
   static double     HQRndDiscrete(CHighQualityRandStateShell &state,int n,vector<double> &x);
   static double     HQRndContinuous(CHighQualityRandStateShell &state,int n,CRowDouble &x);
   static double     HQRndContinuous(CHighQualityRandStateShell &state,int n,vector<double> &x);
   //---  build KD-trees
   static void       KDTreeSerialize(CKDTreeShell &obj,string &s_out);
   static void       KDTreeUnserialize(string s_in,CKDTreeShell &obj);
   static void       KDTreeBuild(CMatrixDouble &xy,const int n,const int nx,const int ny,const int normtype,CKDTreeShell &kdt);
   static void       KDTreeBuild(CMatrixDouble &xy,const int nx,const int ny,const int normtype,CKDTreeShell &kdt);
   static void       KDTreeBuildTagged(CMatrixDouble &xy,int &tags[],const int n,const int nx,const int ny,const int normtype,CKDTreeShell &kdt);
   static void       KDTreeBuildTagged(CMatrixDouble &xy,CRowInt &tags,const int n,const int nx,const int ny,const int normtype,CKDTreeShell &kdt);
   static void       KDTreeBuildTagged(CMatrixDouble &xy,int &tags[],const int nx,const int ny,const int normtype,CKDTreeShell &kdt);
   static void       KDTreeBuildTagged(CMatrixDouble &xy,CRowInt &tags,const int nx,const int ny,const int normtype,CKDTreeShell &kdt);
   static void       KDTreeCreateRequestBuffer(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf);
   static int        KDTreeQueryKNN(CKDTreeShell &kdt,double &x[],const int k,const bool selfmatch=true);
   static int        KDTreeQueryKNN(CKDTreeShell &kdt,CRowDouble &x,const int k,const bool selfmatch=true);
   static int        KDTreeQueryKNN(CKDTreeShell &kdt,vector<double> &x,const int k,const bool selfmatch=true);
   static int        KDTreeTsQueryKNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,double &x[],const int k,const bool selfmatch=true);
   static int        KDTreeTsQueryKNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,CRowDouble &x,const int k,const bool selfmatch=true);
   static int        KDTreeTsQueryKNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,vector<double> &x,const int k,const bool selfmatch=true);
   static int        KDTreeQueryRNN(CKDTreeShell &kdt,double &x[],const double r,const bool selfmatch=true);
   static int        KDTreeQueryRNN(CKDTreeShell &kdt,CRowDouble &x,const double r,const bool selfmatch=true);
   static int        KDTreeQueryRNN(CKDTreeShell &kdt,vector<double> &x,const double r,const bool selfmatch=true);
   static int        KDTreeQueryRNNU(CKDTreeShell &kdt,double &x[],const double r,const bool selfmatch=true);
   static int        KDTreeQueryRNNU(CKDTreeShell &kdt,CRowDouble &x,const double r,const bool selfmatch=true);
   static int        KDTreeQueryRNNU(CKDTreeShell &kdt,vector<double> &x,const double r,const bool selfmatch=true);
   static int        KDTreeTsQueryRNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,double &x[],const double r,const bool selfmatch=true);
   static int        KDTreeTsQueryRNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,CRowDouble &x,const double r,const bool selfmatch=true);
   static int        KDTreeTsQueryRNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,vector<double> &x,const double r,const bool selfmatch=true);
   static int        KDTreeTsQueryRNNU(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,double &x[],const double r,const bool selfmatch=true);
   static int        KDTreeTsQueryRNNU(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,CRowDouble &x,const double r,const bool selfmatch=true);
   static int        KDTreeTsQueryRNNU(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,vector<double> &x,const double r,const bool selfmatch=true);
   static int        KDTreeQueryAKNN(CKDTreeShell &kdt,double &x[],const int k,const bool selfmatch=true,const double eps=0);
   static int        KDTreeQueryAKNN(CKDTreeShell &kdt,vector<double> &x,const int k,const bool selfmatch=true,const double eps=0);
   static int        KDTreeQueryAKNN(CKDTreeShell &kdt,CRowDouble &x,const int k,const bool selfmatch=true,const double eps=0);
   static int        KDTreeQueryBox(CKDTreeShell &kdt,double &boxmin[],double &boxmax[]);
   static int        KDTreeQueryBox(CKDTreeShell &kdt,vector<double> &boxmin,vector<double> &boxmax);
   static int        KDTreeQueryBox(CKDTreeShell &kdt,CRowDouble &boxmin,CRowDouble &boxmax);
   static int        KDTreeTsQueryBox(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,double &boxmin[],double &boxmax[]);
   static int        KDTreeTsQueryBox(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,vector<double> &boxmin,vector<double> &boxmax);
   static int        KDTreeTsQueryBox(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,CRowDouble &boxmin,CRowDouble &boxmax);
   static void       KDTreeQueryResultsX(CKDTreeShell &kdt,CMatrixDouble &x);
   static void       KDTreeQueryResultsXY(CKDTreeShell &kdt,CMatrixDouble &xy);
   static void       KDTreeQueryResultsTags(CKDTreeShell &kdt,int &tags[]);
   static void       KDTreeQueryResultsTags(CKDTreeShell &kdt,CRowInt &tags);
   static void       KDTreeQueryResultsDistances(CKDTreeShell &kdt,double &r[]);
   static void       KDTreeQueryResultsDistances(CKDTreeShell &kdt,vector<double> &r);
   static void       KDTreeQueryResultsDistances(CKDTreeShell &kdt,CRowDouble &r);
   static void       KDTreeQueryResultsXI(CKDTreeShell &kdt,CMatrixDouble &x);
   static void       KDTreeQueryResultsXYI(CKDTreeShell &kdt,CMatrixDouble &xy);
   static void       KDTreeQueryResultsTagsI(CKDTreeShell &kdt,int &tags[]);
   static void       KDTreeQueryResultsTagsI(CKDTreeShell &kdt,CRowInt &tags);
   static void       KDTreeQueryResultsDistancesI(CKDTreeShell &kdt,double &r[]);
   static void       KDTreeQueryResultsDistancesI(CKDTreeShell &kdt,vector<double> &r);
   static void       KDTreeQueryResultsDistancesI(CKDTreeShell &kdt,CRowDouble &r);
   //--- functions of package dataanalysis
   //--- data analysis
   static void       DSOptimalSplit2(double &a[],int &c[],const int n,int &info,double &threshold,double &pal,double &pbl,double &par,double &pbr,double &cve);
   static void       DSOptimalSplit2(CRowDouble &a,CRowInt &c,const int n,int &info,double &threshold,double &pal,double &pbl,double &par,double &pbr,double &cve);
   static void       DSOptimalSplit2Fast(double &a[],int &c[],int &tiesbuf[],int &cntbuf[],double &bufr[],int &bufi[],const int n,const int nc,const double alpha,int &info,double &threshold,double &rms,double &cvrms);
   static void       DSOptimalSplit2Fast(CRowDouble &a,CRowInt &c,CRowInt &tiesbuf,CRowInt &cntbuf,CRowDouble &bufr,CRowInt &bufi,const int n,const int nc,const double alpha,int &info,double &threshold,double &rms,double &cvrms);
   //--- decision forest
   static void       DFSerialize(CDecisionForestShell &obj,string &s_out);
   static void       DFUnserialize(const string s_in,CDecisionForestShell &obj);
   static void       DFCreateBuffer(CDecisionForestShell &model,CDecisionForestBuffer &buf);
   static void       DFBuilderCreate(CDecisionForestBuilder &s);
   static void       DFBuilderSetDataset(CDecisionForestBuilder &s,CMatrixDouble &xy,int npoints,int nvars,int nclasses);
   static void       DFBuilderSetRndVars(CDecisionForestBuilder &s,int rndvars);
   static void       DFBuilderSetRndVarsRatio(CDecisionForestBuilder &s,double f);
   static void       DFBuilderSetRndVarsAuto(CDecisionForestBuilder &s);
   static void       DFBuilderSetSubsampleRatio(CDecisionForestBuilder &s,double f);
   static void       DFBuilderSetSeed(CDecisionForestBuilder &s,int seedval);
   static void       DFBuilderSetRDFAlgo(CDecisionForestBuilder &s,int algotype);
   static void       DFBuilderSetRDFSplitStrength(CDecisionForestBuilder &s,int splitstrength);
   static void       DFBuilderSetImportanceTrnGini(CDecisionForestBuilder &s);
   static void       DFBuilderSetImportanceOOBGini(CDecisionForestBuilder &s);
   static void       DFBuilderSetImportancePermutation(CDecisionForestBuilder &s);
   static void       DFBuilderSetImportanceNone(CDecisionForestBuilder &s);
   static double     DFBuilderGetProgress(CDecisionForestBuilder &s);
   static double     DFBuilderPeekProgress(CDecisionForestBuilder &s);
   static void       DFBuilderBuildRandomForest(CDecisionForestBuilder &s,int ntrees,CDecisionForestShell &df,CDFReportShell &rep);
   static double     DFBinaryCompression(CDecisionForestShell &df);
   static void       DFProcess(CDecisionForestShell &df,double &x[],double &y[]);
   static void       DFProcessI(CDecisionForestShell &df,double &x[],double &y[]);
   static double     DFProcess0(CDecisionForestShell &model,double &x[]);
   static double     DFProcess0(CDecisionForestShell &model,CRowDouble &x);
   static int        DFClassify(CDecisionForestShell &model,double &x[]);
   static int        DFClassify(CDecisionForestShell &model,CRowDouble &x);
   static double     DFRelClsError(CDecisionForestShell &df,CMatrixDouble &xy,const int npoints);
   static double     DFAvgCE(CDecisionForestShell &df,CMatrixDouble &xy,const int npoints);
   static double     DFRMSError(CDecisionForestShell &df,CMatrixDouble &xy,const int npoints);
   static double     DFAvgError(CDecisionForestShell &df,CMatrixDouble &xy,const int npoints);
   static double     DFAvgRelError(CDecisionForestShell &df,CMatrixDouble &xy,const int npoints);
   static void       DFBuildRandomDecisionForest(CMatrixDouble &xy,const int npoints,const int nvars,const int nclasses,const int ntrees,const double r,int &info,CDecisionForestShell &df,CDFReportShell &rep);
   static void       DFBuildRandomDecisionForestX1(CMatrixDouble &xy,const int npoints,const int nvars,const int nclasses,const int ntrees,int nrndvars,const double r,int &info,CDecisionForestShell &df,CDFReportShell &rep);
   //--- middle and clusterization
   static void       ClusterizerCreate(CClusterizerState &s);
   static void       ClusterizerSetPoints(CClusterizerState &s,CMatrixDouble &xy,int npoints,int nfeatures,int disttype);
   static void       ClusterizerSetPoints(CClusterizerState &s,CMatrixDouble &xy,int disttype);
   static void       ClusterizerSetDistances(CClusterizerState &s,CMatrixDouble &d,int npoints,bool IsUpper);
   static void       ClusterizerSetDistances(CClusterizerState &s,CMatrixDouble &d,bool IsUpper);
   static void       ClusterizerSetAHCAlgo(CClusterizerState &s,int algo);
   static void       ClusterizerSetKMeansLimits(CClusterizerState &s,int restarts,int maxits);
   static void       ClusterizerSetKMeansInit(CClusterizerState &s,int initalgo);
   static void       ClusterizerSetSeed(CClusterizerState &s,int seed);
   static void       ClusterizerRunAHC(CClusterizerState &s,CAHCReport &rep);
   static void       ClusterizerRunKMeans(CClusterizerState &s,int k,CKmeansReport &rep);
   static void       ClusterizerGetDistances(CMatrixDouble &xy,int npoints,int nfeatures,int disttype,CMatrixDouble &d);
   static void       ClusterizerGetKClusters(CAHCReport &rep,int k,CRowInt &cidx,CRowInt &cz);
   static void       ClusterizerSeparatedByDist(CAHCReport &rep,double r,int &k,CRowInt &cidx,CRowInt &cz);
   static void       ClusterizerSeparatedByCorr(CAHCReport &rep,double r,int &k,CRowInt &cidx,CRowInt &cz);
   static void       KMeansGenerate(CMatrixDouble &xy,const int npoints,const int nvars,const int k,const int restarts,int &info,CMatrixDouble &c,int &xyc[]);
   //--- Fisher LDA functions
   static void       FisherLDA(CMatrixDouble &xy,const int npoints,const int nvars,const int nclasses,int &info,double &w[]);
   static void       FisherLDA(CMatrixDouble &xy,const int npoints,const int nvars,const int nclasses,int &info,CRowDouble &w);
   static void       FisherLDAN(CMatrixDouble &xy,const int npoints,const int nvars,const int nclasses,int &info,CMatrixDouble &w);
   //--- linear regression
   static void       LRBuild(CMatrixDouble &xy,const int npoints,const int nvars,int &info,CLinearModelShell &lm,CLRReportShell &ar);
   static void       LRBuildS(CMatrixDouble &xy,double &s[],const int npoints,const int nvars,int &info,CLinearModelShell &lm,CLRReportShell &ar);
   static void       LRBuildS(CMatrixDouble &xy,CRowDouble &s,const int npoints,const int nvars,int &info,CLinearModelShell &lm,CLRReportShell &ar);
   static void       LRBuildZS(CMatrixDouble &xy,double &s[],const int npoints,const int nvars,int &info,CLinearModelShell &lm,CLRReportShell &ar);
   static void       LRBuildZS(CMatrixDouble &xy,CRowDouble &s,const int npoints,const int nvars,int &info,CLinearModelShell &lm,CLRReportShell &ar);
   static void       LRBuildZ(CMatrixDouble &xy,const int npoints,const int nvars,int &info,CLinearModelShell &lm,CLRReportShell &ar);
   static void       LRUnpack(CLinearModelShell &lm,double &v[],int &nvars);
   static void       LRUnpack(CLinearModelShell &lm,CRowDouble &v,int &nvars);
   static void       LRPack(double &v[],const int nvars,CLinearModelShell &lm);
   static void       LRPack(CRowDouble &v,const int nvars,CLinearModelShell &lm);
   static double     LRProcess(CLinearModelShell &lm,double &x[]);
   static double     LRProcess(CLinearModelShell &lm,CRowDouble &x);
   static double     LRRMSError(CLinearModelShell &lm,CMatrixDouble &xy,const int npoints);
   static double     LRAvgError(CLinearModelShell &lm,CMatrixDouble &xy,const int npoints);
   static double     LRAvgRelError(CLinearModelShell &lm,CMatrixDouble &xy,const int npoints);
   //--- multilayer perceptron
   static void       MLPSerialize(CMultilayerPerceptronShell &obj,string &s_out);
   static void       MLPUnserialize(const string s_in,CMultilayerPerceptronShell &obj);
   static void       MLPCreate0(const int nin,const int nout,CMultilayerPerceptronShell &network);
   static void       MLPCreate1(const int nin,int nhid,const int nout,CMultilayerPerceptronShell &network);
   static void       MLPCreate2(const int nin,const int nhid1,const int nhid2,const int nout,CMultilayerPerceptronShell &network);
   static void       MLPCreateB0(const int nin,const int nout,const double b,const double d,CMultilayerPerceptronShell &network);
   static void       MLPCreateB1(const int nin,int nhid,const int nout,const double b,const double d,CMultilayerPerceptronShell &network);
   static void       MLPCreateB2(const int nin,const int nhid1,const int nhid2,const int nout,const double b,const double d,CMultilayerPerceptronShell &network);
   static void       MLPCreateR0(const int nin,const int nout,double a,const double b,CMultilayerPerceptronShell &network);
   static void       MLPCreateR1(const int nin,int nhid,const int nout,const double a,const double b,CMultilayerPerceptronShell &network);
   static void       MLPCreateR2(const int nin,const int nhid1,const int nhid2,const int nout,const double a,const double b,CMultilayerPerceptronShell &network);
   static void       MLPCreateC0(const int nin,const int nout,CMultilayerPerceptronShell &network);
   static void       MLPCreateC1(const int nin,int nhid,const int nout,CMultilayerPerceptronShell &network);
   static void       MLPCreateC2(const int nin,const int nhid1,const int nhid2,const int nout,CMultilayerPerceptronShell &network);
   static void       MLPRandomize(CMultilayerPerceptronShell &network);
   static void       MLPRandomizeFull(CMultilayerPerceptronShell &network);
   static void       MLPInitPreprocessor(CMultilayerPerceptronShell &network,CMatrixDouble &xy,int ssize);
   static void       MLPProperties(CMultilayerPerceptronShell &network,int &nin,int &nout,int &wcount);
   static int        MLPGetInputsCount(CMultilayerPerceptronShell &network);
   static int        MLPGetOutputsCount(CMultilayerPerceptronShell &network);
   static int        MLPGetWeightsCount(CMultilayerPerceptronShell &network);
   static bool       MLPIsSoftMax(CMultilayerPerceptronShell &network);
   static int        MLPGetLayersCount(CMultilayerPerceptronShell &network);
   static int        MLPGetLayerSize(CMultilayerPerceptronShell &network,const int k);
   static void       MLPGetInputScaling(CMultilayerPerceptronShell &network,const int i,double &mean,double &sigma);
   static void       MLPGetOutputScaling(CMultilayerPerceptronShell &network,const int i,double &mean,double &sigma);
   static void       MLPGetNeuronInfo(CMultilayerPerceptronShell &network,const int k,const int i,int &fkind,double &threshold);
   static double     MLPGetWeight(CMultilayerPerceptronShell &network,const int k0,const int i0,const int k1,const int i1);
   static void       MLPSetInputScaling(CMultilayerPerceptronShell &network,const int i,const double mean,const double sigma);
   static void       MLPSetOutputScaling(CMultilayerPerceptronShell &network,const int i,const double mean,const double sigma);
   static void       MLPSetNeuronInfo(CMultilayerPerceptronShell &network,const int k,const int i,int fkind,double threshold);
   static void       MLPSetWeight(CMultilayerPerceptronShell &network,const int k0,const int i0,const int k1,const int i1,const double w);
   static void       MLPActivationFunction(const double net,const int k,double &f,double &df,double &d2f);
   static void       MLPProcess(CMultilayerPerceptronShell &network,double &x[],double &y[]);
   static void       MLPProcessI(CMultilayerPerceptronShell &network,double &x[],double &y[]);
   static double     MLPError(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize);
   static double     MLPErrorSparse(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int npoints);
   static double     MLPErrorN(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize);
   static int        MLPClsError(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize);
   static double     MLPRelClsError(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints);
   static double     MLPRelClsErrorSparse(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int npoints);
   static double     MLPAvgCE(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints);
   static double     MLPAvgCESparse(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int npoints);
   static double     MLPRMSError(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints);
   static double     MLPRMSErrorSparse(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int npoints);
   static double     MLPAvgError(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints);
   static double     MLPAvgErrorSparse(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int npoints);
   static double     MLPAvgRelError(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints);
   static double     MLPAvgRelErrorSparse(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int npoints);
   static void       MLPGrad(CMultilayerPerceptronShell &network,double &x[],double &desiredy[],double &e,double &grad[]);
   static void       MLPGradN(CMultilayerPerceptronShell &network,double &x[],double &desiredy[],double &e,double &grad[]);
   static void       MLPGradBatch(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize,double &e,double &grad[]);
   static void       MLPGradBatch(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize,double &e,CRowDouble &grad);
   static void       MLPGradBatchSparse(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int ssize,double &e,CRowDouble &grad);
   static void       MLPGradBatchSubset(CMultilayerPerceptronShell &network,CMatrixDouble &xy,int setsize,CRowInt &idx,int subsetsize,double &e,CRowDouble &grad);
   static void       MLPGradBatchSparseSubset(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int setsize,CRowInt &idx,int subsetsize,double &e,CRowDouble &grad);
   static void       MLPGradNBatch(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize,double &e,double &grad[]);
   static void       MLPGradNBatch(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize,double &e,CRowDouble &grad);
   static void       MLPHessianNBatch(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize,double &e,double &grad[],CMatrixDouble &h);
   static void       MLPHessianNBatch(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize,double &e,CRowDouble &grad,CMatrixDouble &h);
   static void       MLPHessianBatch(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize,double &e,double &grad[],CMatrixDouble &h);
   static void       MLPHessianBatch(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int ssize,double &e,CRowDouble &grad,CMatrixDouble &h);
   static void       MLPAllErrorsSubset(CMultilayerPerceptronShell &network,CMatrixDouble &xy,int setsize,CRowInt &subset,int subsetsize,CModelErrors &rep);
   static void       MLPAllErrorsSparseSubset(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int setsize,CRowInt &subset,int subsetsize,CModelErrors &rep);
   static double     MLPErrorSubset(CMultilayerPerceptronShell &network,CMatrixDouble &xy,int setsize,CRowInt &subset,int subsetsize);
   static double     MLPErrorSparseSubset(CMultilayerPerceptronShell &network,CSparseMatrix &xy,int setsize,CRowInt &subset,int subsetsize);
   //--- logit model functions
   static void       MNLTrainH(CMatrixDouble &xy,const int npoints,const int nvars,const int nclasses,int &info,CLogitModelShell &lm,CMNLReportShell &rep);
   static void       MNLProcess(CLogitModelShell &lm,double &x[],double &y[]);
   static void       MNLProcess(CLogitModelShell &lm,CRowDouble &x,CRowDouble &y);
   static void       MNLProcessI(CLogitModelShell &lm,double &x[],double &y[]);
   static void       MNLProcessI(CLogitModelShell &lm,CRowDouble &x,CRowDouble &y);
   static void       MNLUnpack(CLogitModelShell &lm,CMatrixDouble &a,int &nvars,int &nclasses);
   static void       MNLPack(CMatrixDouble &a,const int nvars,const int nclasses,CLogitModelShell &lm);
   static double     MNLAvgCE(CLogitModelShell &lm,CMatrixDouble &xy,const int npoints);
   static double     MNLRelClsError(CLogitModelShell &lm,CMatrixDouble &xy,const int npoints);
   static double     MNLRMSError(CLogitModelShell &lm,CMatrixDouble &xy,const int npoints);
   static double     MNLAvgError(CLogitModelShell &lm,CMatrixDouble &xy,const int npoints);
   static double     MNLAvgRelError(CLogitModelShell &lm,CMatrixDouble &xy,const int ssize);
   static int        MNLClsError(CLogitModelShell &lm,CMatrixDouble &xy,const int npoints);
   //--- Markov chains
   static void       MCPDCreate(const int n,CMCPDStateShell &s);
   static void       MCPDCreateEntry(const int n,const int entrystate,CMCPDStateShell &s);
   static void       MCPDCreateExit(const int n,const int exitstate,CMCPDStateShell &s);
   static void       MCPDCreateEntryExit(const int n,const int entrystate,const int exitstate,CMCPDStateShell &s);
   static void       MCPDAddTrack(CMCPDStateShell &s,CMatrixDouble &xy,const int k);
   static void       MCPDAddTrack(CMCPDStateShell &s,CMatrixDouble &xy);
   static void       MCPDSetEC(CMCPDStateShell &s,CMatrixDouble &ec);
   static void       MCPDAddEC(CMCPDStateShell &s,const int i,const int j,const double c);
   static void       MCPDSetBC(CMCPDStateShell &s,CMatrixDouble &bndl,CMatrixDouble &bndu);
   static void       MCPDAddBC(CMCPDStateShell &s,const int i,const int j,const double bndl,const double bndu);
   static void       MCPDSetLC(CMCPDStateShell &s,CMatrixDouble &c,int &ct[],const int k);
   static void       MCPDSetLC(CMCPDStateShell &s,CMatrixDouble &c,CRowInt &ct,const int k);
   static void       MCPDSetLC(CMCPDStateShell &s,CMatrixDouble &c,int &ct[]);
   static void       MCPDSetLC(CMCPDStateShell &s,CMatrixDouble &c,CRowInt &ct);
   static void       MCPDSetTikhonovRegularizer(CMCPDStateShell &s,const double v);
   static void       MCPDSetPrior(CMCPDStateShell &s,CMatrixDouble &pp);
   static void       MCPDSetPredictionWeights(CMCPDStateShell &s,double &pw[]);
   static void       MCPDSetPredictionWeights(CMCPDStateShell &s,CRowDouble &pw);
   static void       MCPDSolve(CMCPDStateShell &s);
   static void       MCPDResults(CMCPDStateShell &s,CMatrixDouble &p,CMCPDReportShell &rep);
   //--- training neural networks
   static void       MLPTrainLM(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints,const double decay,const int restarts,int &info,CMLPReportShell &rep);
   static void       MLPTrainLBFGS(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints,const double decay,const int restarts,const double wstep,int maxits,int &info,CMLPReportShell &rep);
   static void       MLPTrainES(CMultilayerPerceptronShell &network,CMatrixDouble &trnxy,const int trnsize,CMatrixDouble &valxy,const int valsize,const double decay,const int restarts,int &info,CMLPReportShell &rep);
   static void       MLPKFoldCVLBFGS(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints,const double decay,const int restarts,const double wstep,const int maxits,const int foldscount,int &info,CMLPReportShell &rep,CMLPCVReportShell &cvrep);
   static void       MLPKFoldCVLM(CMultilayerPerceptronShell &network,CMatrixDouble &xy,const int npoints,const double decay,const int restarts,const int foldscount,int &info,CMLPReportShell &rep,CMLPCVReportShell &cvrep);
   static void       MLPCreateTrainer(int nin,int nout,CMLPTrainer &s);
   static void       MLPCreateTrainerCls(int nin,int nclasses,CMLPTrainer &s);
   static void       MLPSetDataset(CMLPTrainer &s,CMatrixDouble &xy,int npoints);
   static void       MLPSetSparseDataset(CMLPTrainer &s,CSparseMatrix &xy,int npoints);
   static void       MLPSetDecay(CMLPTrainer &s,double decay);
   static void       MLPSetCond(CMLPTrainer &s,double wstep,int maxits);
   static void       MLPSetAlgoBatch(CMLPTrainer &s);
   static void       MLPTrainNetwork(CMLPTrainer &s,CMultilayerPerceptronShell &network,int nrestarts,CMLPReportShell &rep);
   static void       MLPStartTraining(CMLPTrainer &s,CMultilayerPerceptronShell &network,bool randomstart);
   static bool       MLPContinueTraining(CMLPTrainer &s,CMultilayerPerceptronShell &network);
   //--- neural networks ensemble functions
   static void       MLPECreate0(const int nin,const int nout,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreate1(const int nin,int nhid,const int nout,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreate2(const int nin,const int nhid1,const int nhid2,const int nout,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateB0(const int nin,const int nout,const double b,const double d,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateB1(const int nin,int nhid,const int nout,const double b,const double d,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateB2(const int nin,const int nhid1,const int nhid2,const int nout,const double b,const double d,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateR0(const int nin,const int nout,const double a,const double b,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateR1(const int nin,int nhid,const int nout,const double a,const double b,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateR2(const int nin,const int nhid1,const int nhid2,const int nout,const double a,const double b,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateC0(const int nin,const int nout,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateC1(const int nin,int nhid,const int nout,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateC2(const int nin,const int nhid1,const int nhid2,const int nout,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPECreateFromNetwork(CMultilayerPerceptronShell &network,const int ensemblesize,CMLPEnsembleShell &ensemble);
   static void       MLPERandomize(CMLPEnsembleShell &ensemble);
   static void       MLPEProperties(CMLPEnsembleShell &ensemble,int &nin,int &nout);
   static bool       MLPEIsSoftMax(CMLPEnsembleShell &ensemble);
   static void       MLPEProcess(CMLPEnsembleShell &ensemble,double &x[],double &y[]);
   static void       MLPEProcessI(CMLPEnsembleShell &ensemble,double &x[],double &y[]);
   static double     MLPERelClsError(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,const int npoints);
   static double     MLPEAvgCE(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,const int npoints);
   static double     MLPERMSError(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,const int npoints);
   static double     MLPEAvgError(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,const int npoints);
   static double     MLPEAvgRelError(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,const int npoints);
   static void       MLPEBaggingLM(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,const int npoints,const double decay,const int restarts,int &info,CMLPReportShell &rep,CMLPCVReportShell &ooberrors);
   static void       MLPEBaggingLBFGS(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,const int npoints,const double decay,const int restarts,const double wstep,const int maxits,int &info,CMLPReportShell &rep,CMLPCVReportShell &ooberrors);
   static void       MLPETrainES(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,const int npoints,const double decay,const int restarts,int &info,CMLPReportShell &rep);
   static void       MLPTrainEnsembleES(CMLPTrainer &s,CMLPEnsembleShell &ensemble,int nrestarts,CMLPReportShell &rep);
   //--- principal components analysis
   static void       PCABuildBasis(CMatrixDouble &x,const int npoints,const int nvars,int &info,double &s2[],CMatrixDouble &v);
   static void       PCABuildBasis(CMatrixDouble &x,const int npoints,const int nvars,int &info,CRowDouble &s2,CMatrixDouble &v);
   static void       PCATruncatedSubspace(CMatrixDouble &x,int npoints,int nvars,int nneeded,double eps,int maxits,CRowDouble &s2,CMatrixDouble &v);
   static void       PCATruncatedSubspaceSparse(CSparseMatrix &x,int npoints,int nvars,int nneeded,double eps,int maxits,CRowDouble &s2,CMatrixDouble &v);
   //--- functions of package diffequations
   static void       ODESolverRKCK(double &y[],const int n,double &x[],const int m,const double eps,const double h,CODESolverStateShell &state);
   static void       ODESolverRKCK(double &y[],double &x[],const double eps,const double h,CODESolverStateShell &state);
   static bool       ODESolverIteration(CODESolverStateShell &state);
   static void       ODESolverSolve(CODESolverStateShell &state,CNDimensional_ODE_RP &diff,CObject &obj);
   static void       ODESolverResults(CODESolverStateShell &state,int &m,double &xtbl[],CMatrixDouble &ytbl,CODESolverReportShell &rep);
   //--- filters
   static void       FilterSMA(CRowDouble &x,int n,int k);
   static void       FilterSMA(CRowDouble &x,int k);
   static void       FilterEMA(CRowDouble &x,int n,double alpha);
   static void       FilterEMA(CRowDouble &x,double alpha);
   static void       FilterLRMA(CRowDouble &x,int n,int k);
   static void       FilterLRMA(CRowDouble &x,int k);
   //--- SSA models
   static void       SSACreate(CSSAModel &s);
   static void       SSASetWindow(CSSAModel &s,int windowwidth);
   static void       SSASetSeed(CSSAModel &s,int seed);
   static void       SSASetPowerUpLength(CSSAModel &s,int pwlen);
   static void       SSASetMemoryLimit(CSSAModel &s,int memlimit);
   static void       SSAAddSequence(CSSAModel &s,CRowDouble &x,int n);
   static void       SSAAddSequence(CSSAModel &s,CRowDouble &x);
   static void       SSAAppendPointAndUpdate(CSSAModel &s,double x,double updateits);
   static void       SSAAppendSequenceAndUpdate(CSSAModel &s,CRowDouble &x,int nticks,double updateits);
   static void       SSAAppendSequenceAndUpdate(CSSAModel &s,CRowDouble &x,double updateits);
   static void       SSASetAlgoPrecomputed(CSSAModel &s,CMatrixDouble &a,int windowwidth,int nbasis);
   static void       SSASetAlgoPrecomputed(CSSAModel &s,CMatrixDouble &a);
   static void       SSASetAlgoTopKDirect(CSSAModel &s,int topk);
   static void       SSASetAlgoTopKRealtime(CSSAModel &s,int topk);
   static void       SSAClearData(CSSAModel &s);
   static void       SSAGetBasis(CSSAModel &s,CMatrixDouble &a,CRowDouble &sv,int &windowwidth,int &nbasis);
   static void       SSAGetLRR(CSSAModel &s,CRowDouble &a,int &windowwidth);
   static void       SSAAnalyzeLastWindow(CSSAModel &s,CRowDouble &trend,CRowDouble &noise,int &nticks);
   static void       SSAAnalyzeLast(CSSAModel &s,int nticks,CRowDouble &trend,CRowDouble &noise);
   static void       SSAAnalyzeSequence(CSSAModel &s,CRowDouble &data,int nticks,CRowDouble &trend,CRowDouble &noise);
   static void       SSAAnalyzeSequence(CSSAModel &s,CRowDouble &data,CRowDouble &trend,CRowDouble &noise);
   static void       SSAForecastLast(CSSAModel &s,int nticks,CRowDouble &trend);
   static void       SSAForecastSequence(CSSAModel &s,CRowDouble &data,int datalen,int forecastlen,bool applysmoothing,CRowDouble &trend);
   static void       SSAForecastSequence(CSSAModel &s,CRowDouble &data,int forecastlen,CRowDouble &trend);
   static void       SSAForecastAvgLast(CSSAModel &s,int m,int nticks,CRowDouble &trend);
   static void       SSAForecastAvgSequence(CSSAModel &s,CRowDouble &data,int datalen,int m,int forecastlen,bool applysmoothing,CRowDouble &trend);
   static void       SSAForecastAvgSequence(CSSAModel &s,CRowDouble &data,int m,int forecastlen,CRowDouble &trend);
   //--- KNN models
   static void       KNNSerialize(CKNNModel &obj,string &s_out);
   static void       KNNUnserialize(const string s_in,CKNNModel &obj);
   static void       KNNCreateBuffer(CKNNModel &model,CKNNBuffer &buf);
   static void       KNNBuilderCreate(CKNNBuilder &s);
   static void       KNNBuilderSetDatasetReg(CKNNBuilder &s,CMatrixDouble &xy,int npoints,int nvars,int nout);
   static void       KNNBuilderSetDatasetCLS(CKNNBuilder &s,CMatrixDouble &xy,int npoints,int nvars,int nclasses);
   static void       KNNBuilderSetNorm(CKNNBuilder &s,int nrmtype);
   static void       KNNBuilderBuildKNNModel(CKNNBuilder &s,int k,double eps,CKNNModel &model,CKNNReport &rep);
   static void       KNNRewriteKEps(CKNNModel &model,int k,double eps);
   static void       KNNProcess(CKNNModel &model,CRowDouble &x,CRowDouble &y);
   static double     KNNProcess0(CKNNModel &model,CRowDouble &x);
   static int        KNNClassify(CKNNModel &model,CRowDouble &x);
   static void       KNNProcessI(CKNNModel &model,CRowDouble &x,CRowDouble &y);
   static void       KNNTsProcess(CKNNModel &model,CKNNBuffer &buf,CRowDouble &x,CRowDouble &y);
   static double     KNNRelClsError(CKNNModel &model,CMatrixDouble &xy,int npoints);
   static double     KNNAvgCE(CKNNModel &model,CMatrixDouble &xy,int npoints);
   static double     KNNRMSError(CKNNModel &model,CMatrixDouble &xy,int npoints);
   static double     KNNAvgError(CKNNModel &model,CMatrixDouble &xy,int npoints);
   static double     KNNAvgRelError(CKNNModel &model,CMatrixDouble &xy,int npoints);
   static void       KNNAllErrors(CKNNModel &model,CMatrixDouble &xy,int npoints,CKNNReport &rep);
   //--- functions of package fasttransforms
   //--- fast Fourier transform
   static void       FFTC1D(complex &a[],const int n);
   static void       FFTC1D(complex &a[]);
   static void       FFTC1DInv(complex &a[],const int n);
   static void       FFTC1DInv(complex &a[]);
   static void       FFTR1D(double &a[],const int n,complex &f[]);
   static void       FFTR1D(double &a[],complex &f[]);
   static void       FFTR1DInv(complex &f[],const int n,double &a[]);
   static void       FFTR1DInv(complex &f[],double &a[]);
   static void       FFTC1D(CRowComplex &a,const int n);
   static void       FFTC1D(CRowComplex &a);
   static void       FFTC1DInv(CRowComplex &a,const int n);
   static void       FFTC1DInv(CRowComplex &a);
   static void       FFTR1D(CRowDouble &a,const int n,CRowComplex &f);
   static void       FFTR1D(CRowDouble &a,CRowComplex &f);
   static void       FFTR1DInv(CRowComplex &f,const int n,CRowDouble &a);
   static void       FFTR1DInv(CRowComplex &f,CRowDouble &a);
   //--- convolution
   static void       ConvC1D(complex &a[],const int m,complex &b[],const int n,complex &r[]);
   static void       ConvC1DInv(complex &a[],const int m,complex &b[],const int n,complex &r[]);
   static void       ConvC1DCircular(complex &s[],const int m,complex &r[],const int n,complex &c[]);
   static void       ConvC1DCircularInv(complex &a[],const int m,complex &b[],const int n,complex &r[]);
   static void       ConvR1D(double &a[],const int m,double &b[],const int n,double &r[]);
   static void       ConvR1DInv(double &a[],const int m,double &b[],const int n,double &r[]);
   static void       ConvR1DCircular(double &s[],const int m,double &r[],const int n,double &c[]);
   static void       ConvR1DCircularInv(double &a[],const int m,double &b[],const int n,double &r[]);
   static void       CorrC1D(complex &signal[],const int n,complex &pattern[],const int m,complex &r[]);
   static void       CorrC1DCircular(complex &signal[],const int m,complex &pattern[],const int n,complex &c[]);
   static void       CorrR1D(double &signal[],const int n,double &pattern[],const int m,double &r[]);
   static void       CorrR1DCircular(double &signal[],const int m,double &pattern[],const int n,double &c[]);
   //--- fast Hartley transform
   static void       FHTR1D(double &a[],const int n);
   static void       FHTR1DInv(double &a[],const int n);
   //--- functions of package integration
   //--- Gauss quadrature formula
   static void       GQGenerateRec(double &alpha[],double &beta[],const double mu0,const int n,int &info,double &x[],double &w[]);
   static void       GQGenerateGaussLobattoRec(double &alpha[],double &beta[],const double mu0,const double a,const double b,const int n,int &info,double &x[],double &w[]);
   static void       GQGenerateGaussRadauRec(double &alpha[],double &beta[],const double mu0,const double a,const int n,int &info,double &x[],double &w[]);
   static void       GQGenerateGaussLegendre(const int n,int &info,double &x[],double &w[]);
   static void       GQGenerateGaussJacobi(const int n,const double alpha,const double beta,int &info,double &x[],double &w[]);
   static void       GQGenerateGaussLaguerre(const int n,const double alpha,int &info,double &x[],double &w[]);
   static void       GQGenerateGaussHermite(const int n,int &info,double &x[],double &w[]);
   //--- Gauss-Kronrod quadrature formula
   static void       GKQGenerateRec(double &alpha[],double &beta[],const double mu0,const int n,int &info,double &x[],double &wkronrod[],double &wgauss[]);
   static void       GKQGenerateGaussLegendre(const int n,int &info,double &x[],double &wkronrod[],double &wgauss[]);
   static void       GKQGenerateGaussJacobi(const int n,const double alpha,const double beta,int &info,double &x[],double &wkronrod[],double &wgauss[]);
   static void       GKQLegendreCalc(const int n,int &info,double &x[],double &wkronrod[],double &wgauss[]);
   static void       GKQLegendreTbl(const int n,double &x[],double &wkronrod[],double &wgauss[],double &eps);
   //--- auto Gauss-Kronrod
   static void       AutoGKSmooth(const double a,const double b,CAutoGKStateShell &state);
   static void       AutoGKSmoothW(const double a,const double b,double xwidth,CAutoGKStateShell &state);
   static void       AutoGKSingular(const double a,const double b,const double alpha,const double beta,CAutoGKStateShell &state);
   static bool       AutoGKIteration(CAutoGKStateShell &state);
   static void       AutoGKIntegrate(CAutoGKStateShell &state,CIntegrator1_Func &func,CObject &obj);
   static void       AutoGKResults(CAutoGKStateShell &state,double &v,CAutoGKReportShell &rep);
   //--- functions of package interpolation
   //--- inverse distance weighting interpolation
   static void       IDWSerialize(CIDWModelShell &obj,string &s_out);
   static void       IDWUnserialize(string s_in,CIDWModelShell &obj);
   static void       IDWCreateCalcBuffer(CIDWModelShell &s,CIDWCalcBuffer &buf);
   static void       IDWBuilderCreate(int nx,int ny,CIDWBuilder &state);
   static void       IDWBuilderSetNLayers(CIDWBuilder &state,int nlayers);
   static void       IDWBuilderSetPoints(CIDWBuilder &state,CMatrixDouble &xy,int n);
   static void       IDWBuilderSetPoints(CIDWBuilder &state,CMatrixDouble &xy);
   static void       IDWBuilderSetAlgoMSTAB(CIDWBuilder &state,double srad);
   static void       IDWBuilderSetAlgoTextBookShepard(CIDWBuilder &state,double p);
   static void       IDWBuilderSetAlgoTextBookModShepard(CIDWBuilder &state,double r);
   static void       IDWBuilderSetUserTerm(CIDWBuilder &state,double v);
   static void       IDWBuilderSetConstTerm(CIDWBuilder &state);
   static void       IDWBuilderSetZeroTerm(CIDWBuilder &state);
   static double     IDWCalc1(CIDWModelShell &s,double x0);
   static double     IDWCalc2(CIDWModelShell &s,double x0,double x1);
   static double     IDWCalc3(CIDWModelShell &s,double x0,double x1,double x2);
   static void       IDWCalc(CIDWModelShell &s,CRowDouble &x,CRowDouble &y);
   static void       IDWCalcBuf(CIDWModelShell &s,CRowDouble &x,CRowDouble &y);
   static void       IDWTsCalcBuf(CIDWModelShell &s,CIDWCalcBuffer &buf,CRowDouble &x,CRowDouble &y);
   static void       IDWFit(CIDWBuilder &state,CIDWModelShell &model,CIDWReport &rep);
   //--- rational interpolation
   static double     BarycentricCalc(CBarycentricInterpolantShell &b,const double t);
   static void       BarycentricDiff1(CBarycentricInterpolantShell &b,const double t,double &f,double &df);
   static void       BarycentricDiff2(CBarycentricInterpolantShell &b,const double t,double &f,double &df,double &d2f);
   static void       BarycentricLinTransX(CBarycentricInterpolantShell &b,const double ca,const double cb);
   static void       BarycentricLinTransY(CBarycentricInterpolantShell &b,const double ca,const double cb);
   static void       BarycentricUnpack(CBarycentricInterpolantShell &b,int &n,double &x[],double &y[],double &w[]);
   static void       BarycentricBuildXYW(double &x[],double &y[],double &w[],const int n,CBarycentricInterpolantShell &b);
   static void       BarycentricBuildFloaterHormann(double &x[],double &y[],const int n,const int d,CBarycentricInterpolantShell &b);
   //--- polynomial interpolant
   static void       PolynomialBar2Cheb(CBarycentricInterpolantShell &p,const double a,const double b,double &t[]);
   static void       PolynomialCheb2Bar(double &t[],const int n,const double a,const double b,CBarycentricInterpolantShell &p);
   static void       PolynomialCheb2Bar(double &t[],const double a,const double b,CBarycentricInterpolantShell &p);
   static void       PolynomialBar2Pow(CBarycentricInterpolantShell &p,const double c,const double s,double &a[]);
   static void       PolynomialBar2Pow(CBarycentricInterpolantShell &p,double &a[]);
   static void       PolynomialPow2Bar(double &a[],const int n,const double c,const double s,CBarycentricInterpolantShell &p);
   static void       PolynomialPow2Bar(double &a[],CBarycentricInterpolantShell &p);
   static void       PolynomialBuild(double &x[],double &y[],const int n,CBarycentricInterpolantShell &p);
   static void       PolynomialBuild(double &x[],double &y[],CBarycentricInterpolantShell &p);
   static void       PolynomialBuildEqDist(const double a,const double b,double &y[],const int n,CBarycentricInterpolantShell &p);
   static void       PolynomialBuildEqDist(const double a,const double b,double &y[],CBarycentricInterpolantShell &p);
   static void       PolynomialBuildCheb1(const double a,const double b,double &y[],const int n,CBarycentricInterpolantShell &p);
   static void       PolynomialBuildCheb1(const double a,const double b,double &y[],CBarycentricInterpolantShell &p);
   static void       PolynomialBuildCheb2(const double a,const double b,double &y[],const int n,CBarycentricInterpolantShell &p);
   static void       PolynomialBuildCheb2(const double a,const double b,double &y[],CBarycentricInterpolantShell &p);
   static double     PolynomialCalcEqDist(const double a,const double b,double &f[],const int n,const double t);
   static double     PolynomialCalcEqDist(const double a,const double b,double &f[],const double t);
   static double     PolynomialCalcCheb1(const double a,const double b,double &f[],const int n,const double t);
   static double     PolynomialCalcCheb1(const double a,const double b,double &f[],const double t);
   static double     PolynomialCalcCheb2(const double a,const double b,double &f[],const int n,const double t);
   static double     PolynomialCalcCheb2(const double a,const double b,double &f[],const double t);
   //--- 1-dimensional spline interpolation
   static void       Spline1DBuildLinear(double &x[],double &y[],const int n,CSpline1DInterpolantShell &c);
   static void       Spline1DBuildLinear(double &x[],double &y[],CSpline1DInterpolantShell &c);
   static void       Spline1DBuildCubic(double &x[],double &y[],const int n,const int boundltype,const double boundl,const int boundrtype,const double boundr,CSpline1DInterpolantShell &c);
   static void       Spline1DBuildCubic(double &x[],double &y[],CSpline1DInterpolantShell &c);
   static void       Spline1DGridDiffCubic(double &x[],double &y[],const int n,const int boundltype,const double boundl,const int boundrtype,const double boundr,double &d[]);
   static void       Spline1DGridDiffCubic(double &x[],double &y[],double &d[]);
   static void       Spline1DGridDiff2Cubic(double &x[],double &y[],const int n,const int boundltype,const double boundl,const int boundrtype,const double boundr,double &d1[],double &d2[]);
   static void       Spline1DGridDiff2Cubic(double &x[],double &y[],double &d1[],double &d2[]);
   static void       Spline1DConvCubic(double &x[],double &y[],const int n,const int boundltype,const double boundl,const int boundrtype,const double boundr,double &x2[],int n2,double &y2[]);
   static void       Spline1DConvCubic(double &x[],double &y[],double &x2[],double &y2[]);
   static void       Spline1DConvDiffCubic(double &x[],double &y[],const int n,const int boundltype,const double boundl,const int boundrtype,const double boundr,double &x2[],int n2,double &y2[],double &d2[]);
   static void       Spline1DConvDiffCubic(double &x[],double &y[],double &x2[],double &y2[],double &d2[]);
   static void       Spline1DConvDiff2Cubic(double &x[],double &y[],const int n,const int boundltype,const double boundl,const int boundrtype,const double boundr,double &x2[],const int n2,double &y2[],double &d2[],double &dd2[]);
   static void       Spline1DConvDiff2Cubic(double &x[],double &y[],double &x2[],double &y2[],double &d2[],double &dd2[]);
   static void       Spline1DBuildCatmullRom(double &x[],double &y[],const int n,const int boundtype,const double tension,CSpline1DInterpolantShell &c);
   static void       Spline1DBuildCatmullRom(double &x[],double &y[],CSpline1DInterpolantShell &c);
   static void       Spline1DBuildHermite(double &x[],double &y[],double &d[],const int n,CSpline1DInterpolantShell &c);
   static void       Spline1DBuildHermite(double &x[],double &y[],double &d[],CSpline1DInterpolantShell &c);
   static void       Spline1DBuildAkima(double &x[],double &y[],const int n,CSpline1DInterpolantShell &c);
   static void       Spline1DBuildAkima(double &x[],double &y[],CSpline1DInterpolantShell &c);
   static double     Spline1DCalc(CSpline1DInterpolantShell &c,const double x);
   static void       Spline1DDiff(CSpline1DInterpolantShell &c,const double x,double &s,double &ds,double &d2s);
   static void       Spline1DUnpack(CSpline1DInterpolantShell &c,int &n,CMatrixDouble &tbl);
   static void       Spline1DLinTransX(CSpline1DInterpolantShell &c,const double a,const double b);
   static void       Spline1DLinTransY(CSpline1DInterpolantShell &c,const double a,const double b);
   static double     Spline1DIntegrate(CSpline1DInterpolantShell &c,const double x);
   static void       Spline1DFit(double &x[],double &y[],int n,int m,double lambdans,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFit(double &x[],double &y[],int m,double lambdans,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DBuildMonotone(double &x[],double &y[],int n,CSpline1DInterpolantShell &c);
   static void       Spline1DBuildMonotone(double &x[],double &y[],CSpline1DInterpolantShell &c);
   static void       Spline1DBuildMonotone(CRowDouble &x,CRowDouble &y,CSpline1DInterpolantShell &c);
   //--- least squares fitting
   static void       PolynomialFit(double &x[],double &y[],const int n,const int m,int &info,CBarycentricInterpolantShell &p,CPolynomialFitReportShell &rep);
   static void       PolynomialFit(double &x[],double &y[],const int m,int &info,CBarycentricInterpolantShell &p,CPolynomialFitReportShell &rep);
   static void       PolynomialFitWC(double &x[],double &y[],double &w[],const int n,double &xc[],double &yc[],int &dc[],const int k,const int m,int &info,CBarycentricInterpolantShell &p,CPolynomialFitReportShell &rep);
   static void       PolynomialFitWC(double &x[],double &y[],double &w[],double &xc[],double &yc[],int &dc[],const int m,int &info,CBarycentricInterpolantShell &p,CPolynomialFitReportShell &rep);
   static void       BarycentricFitFloaterHormannWC(double &x[],double &y[],double &w[],const int n,double &xc[],double &yc[],int &dc[],const int k,const int m,int &info,CBarycentricInterpolantShell &b,CBarycentricFitReportShell &rep);
   static void       BarycentricFitFloaterHormann(double &x[],double &y[],const int n,const int m,int &info,CBarycentricInterpolantShell &b,CBarycentricFitReportShell &rep);
   static void       Spline1DFitPenalized(double &x[],double &y[],const int n,const int m,const double rho,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitPenalized(double &x[],double &y[],const int m,const double rho,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitPenalizedW(double &x[],double &y[],double &w[],const int n,const int m,const double rho,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitPenalizedW(double &x[],double &y[],double &w[],const int m,const double rho,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitCubicWC(double &x[],double &y[],double &w[],const int n,double &xc[],double &yc[],int &dc[],const int k,const int m,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitCubicWC(double &x[],double &y[],double &w[],double &xc[],double &yc[],int &dc[],const int m,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitHermiteWC(double &x[],double &y[],double &w[],const int n,double &xc[],double &yc[],int &dc[],const int k,const int m,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitHermiteWC(double &x[],double &y[],double &w[],double &xc[],double &yc[],int &dc[],const int m,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitCubic(double &x[],double &y[],const int n,const int m,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitCubic(double &x[],double &y[],const int m,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitHermite(double &x[],double &y[],const int n,const int m,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       Spline1DFitHermite(double &x[],double &y[],const int m,int &info,CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep);
   static void       LSFitLinearW(double &y[],double &w[],CMatrixDouble &fmatrix,const int n,const int m,int &info,double &c[],CLSFitReportShell &rep);
   static void       LSFitLinearW(double &y[],double &w[],CMatrixDouble &fmatrix,int &info,double &c[],CLSFitReportShell &rep);
   static void       LSFitLinearWC(double &y[],double &w[],CMatrixDouble &fmatrix,CMatrixDouble &cmatrix,const int n,const int m,const int k,int &info,double &c[],CLSFitReportShell &rep);
   static void       LSFitLinearWC(double &y[],double &w[],CMatrixDouble &fmatrix,CMatrixDouble &cmatrix,int &info,double &c[],CLSFitReportShell &rep);
   static void       LSFitLinear(double &y[],CMatrixDouble &fmatrix,const int n,const int m,int &info,double &c[],CLSFitReportShell &rep);
   static void       LSFitLinear(double &y[],CMatrixDouble &fmatrix,int &info,double &c[],CLSFitReportShell &rep);
   static void       LSFitLinearC(double &y[],CMatrixDouble &fmatrix,CMatrixDouble &cmatrix,const int n,const int m,const int k,int &info,double &c[],CLSFitReportShell &rep);
   static void       LSFitLinearC(double &y[],CMatrixDouble &fmatrix,CMatrixDouble &cmatrix,int &info,double &c[],CLSFitReportShell &rep);
   static void       LSFitCreateWF(CMatrixDouble &x,double &y[],double &w[],double &c[],const int n,const int m,const int k,const double diffstep,CLSFitStateShell &state);
   static void       LSFitCreateWF(CMatrixDouble &x,double &y[],double &w[],double &c[],const double diffstep,CLSFitStateShell &state);
   static void       LSFitCreateF(CMatrixDouble &x,double &y[],double &c[],const int n,const int m,const int k,const double diffstep,CLSFitStateShell &state);
   static void       LSFitCreateF(CMatrixDouble &x,double &y[],double &c[],const double diffstep,CLSFitStateShell &state);
   static void       LSFitCreateWFG(CMatrixDouble &x,double &y[],double &w[],double &c[],const int n,const int m,const int k,const bool cheapfg,CLSFitStateShell &state);
   static void       LSFitCreateWFG(CMatrixDouble &x,double &y[],double &w[],double &c[],const bool cheapfg,CLSFitStateShell &state);
   static void       LSFitCreateFG(CMatrixDouble &x,double &y[],double &c[],const int n,const int m,const int k,const bool cheapfg,CLSFitStateShell &state);
   static void       LSFitCreateFG(CMatrixDouble &x,double &y[],double &c[],const bool cheapfg,CLSFitStateShell &state);
   static void       LSFitCreateWFGH(CMatrixDouble &x,double &y[],double &w[],double &c[],const int n,const int m,const int k,CLSFitStateShell &state);
   static void       LSFitCreateWFGH(CMatrixDouble &x,double &y[],double &w[],double &c[],CLSFitStateShell &state);
   static void       LSFitCreateFGH(CMatrixDouble &x,double &y[],double &c[],const int n,const int m,const int k,CLSFitStateShell &state);
   static void       LSFitCreateFGH(CMatrixDouble &x,double &y[],double &c[],CLSFitStateShell &state);
   static void       LSFitSetCond(CLSFitStateShell &state,const double epsx,const int maxits);
   static void       LSFitSetStpMax(CLSFitStateShell &state,const double stpmax);
   static void       LSFitSetXRep(CLSFitStateShell &state,const bool needxrep);
   static void       LSFitSetScale(CLSFitStateShell &state,double &s[]);
   static void       LSFitSetBC(CLSFitStateShell &state,double &bndl[],double &bndu[]);
   static bool       LSFitIteration(CLSFitStateShell &state);
   static void       LSFitFit(CLSFitStateShell &state,CNDimensional_PFunc &func,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       LSFitFit(CLSFitStateShell &state,CNDimensional_PFunc &func,CNDimensional_PGrad &grad,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       LSFitFit(CLSFitStateShell &state,CNDimensional_PFunc &func,CNDimensional_PGrad &grad,CNDimensional_PHess &hess,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       LSFitResults(CLSFitStateShell &state,int &info,double &c[],CLSFitReportShell &rep);
   static double     LogisticCalc4(double x,double a,double b,double c,double d);
   static double     LogisticCalc5(double x,double a,double b,double c,double d,double g);
   static void       LogisticFit4(CRowDouble &x,CRowDouble &y,int n,double &a,double &b,double &c,double &d,CLSFitReportShell &rep);
   static void       LogisticFit4ec(CRowDouble &x,CRowDouble &y,int n,double cnstrleft,double cnstrright,double &a,double &b,double &c,double &d,CLSFitReportShell &rep);
   static void       LogisticFit5(CRowDouble &x,CRowDouble &y,int n,double &a,double &b,double &c,double &d,double &g,CLSFitReportShell &rep);
   static void       LogisticFit5ec(CRowDouble &x,CRowDouble &y,int n,double cnstrleft,double cnstrright,double &a,double &b,double &c,double &d,double &g,CLSFitReportShell &rep);
   static void       LogisticFit45x(CRowDouble &x,CRowDouble &y,int n,double cnstrleft,double cnstrright,bool is4pl,double lambdav,double epsx,int rscnt,double &a,double &b,double &c,double &d,double &g,CLSFitReportShell &rep);
   //--- least squares (LS) circle
   static void       FitSphereLS(CMatrixDouble &xy,int npoints,int nx,CRowDouble &cx,double &r);
   static void       FitSphereMC(CMatrixDouble &xy,int npoints,int nx,CRowDouble &cx,double &rhi);
   static void       FitSphereMI(CMatrixDouble &xy,int npoints,int nx,CRowDouble &cx,double &rlo);
   static void       FitSphereMZ(CMatrixDouble &xy,int npoints,int nx,CRowDouble &cx,double &rlo,double &rhi);
   static void       FitSphereX(CMatrixDouble &xy,int npoints,int nx,int problemtype,double epsx,int aulits,double penalty,CRowDouble &cx,double &rlo,double &rhi);
   //--- parametric spline
   static void       PSpline2Build(CMatrixDouble &xy,const int n,const int st,const int pt,CPSpline2InterpolantShell &p);
   static void       PSpline3Build(CMatrixDouble &xy,const int n,const int st,const int pt,CPSpline3InterpolantShell &p);
   static void       PSpline2BuildPeriodic(CMatrixDouble &xy,const int n,const int st,const int pt,CPSpline2InterpolantShell &p);
   static void       PSpline3BuildPeriodic(CMatrixDouble &xy,const int n,const int st,const int pt,CPSpline3InterpolantShell &p);
   static void       PSpline2ParameterValues(CPSpline2InterpolantShell &p,int &n,double &t[]);
   static void       PSpline3ParameterValues(CPSpline3InterpolantShell &p,int &n,double &t[]);
   static void       PSpline2Calc(CPSpline2InterpolantShell &p,const double t,double &x,double &y);
   static void       PSpline3Calc(CPSpline3InterpolantShell &p,const double t,double &x,double &y,double &z);
   static void       PSpline2Tangent(CPSpline2InterpolantShell &p,const double t,double &x,double &y);
   static void       PSpline3Tangent(CPSpline3InterpolantShell &p,const double t,double &x,double &y,double &z);
   static void       PSpline2Diff(CPSpline2InterpolantShell &p,const double t,double &x,double &dx,double &y,double &dy);
   static void       PSpline3Diff(CPSpline3InterpolantShell &p,const double t,double &x,double &dx,double &y,double &dy,double &z,double &dz);
   static void       PSpline2Diff2(CPSpline2InterpolantShell &p,const double t,double &x,double &dx,double &d2x,double &y,double &dy,double &d2y);
   static void       PSpline3Diff2(CPSpline3InterpolantShell &p,const double t,double &x,double &dx,double &d2x,double &y,double &dy,double &d2y,double &z,double &dz,double &d2z);
   static double     PSpline2ArcLength(CPSpline2InterpolantShell &p,const double a,const double b);
   static double     PSpline3ArcLength(CPSpline3InterpolantShell &p,const double a,const double b);
   static void       ParametricRDPFixed(CMatrixDouble &x,int n,int d,int stopm,double stopeps,CMatrixDouble &x2,int &idx2[],int &nsections);
   //--- 2-dimensional spline interpolation
   static void       Spline2DSerialize(CSpline2DInterpolantShell &obj,string &s_out);
   static void       Spline2DUnserialize(string s_in,CSpline2DInterpolantShell &obj);
   static double     Spline2DCalc(CSpline2DInterpolantShell &c,const double x,const double y);
   static void       Spline2DDiff(CSpline2DInterpolantShell &c,const double x,const double y,double &f,double &fx,double &fy,double &fxy);
   static void       Spline2DCalcVBuf(CSpline2DInterpolantShell &c,double x,double y,CRowDouble &f);
   static double     Spline2DCalcVi(CSpline2DInterpolantShell &c,double x,double y,int i);
   static void       Spline2DCalcV(CSpline2DInterpolantShell &c,double x,double y,CRowDouble &f);
   static void       Spline2DDiffVi(CSpline2DInterpolantShell &c,double x,double y,int i,double &f,double &fx,double &fy,double &fxy);
   static void       Spline2DLinTransXY(CSpline2DInterpolantShell &c,const double ax,const double bx,const double ay,const double by);
   static void       Spline2DLinTransF(CSpline2DInterpolantShell &c,const double a,const double b);
   static void       Spline2DCopy(CSpline2DInterpolantShell &c,CSpline2DInterpolantShell &cc);
   static void       Spline2DResampleBicubic(CMatrixDouble &a,const int oldheight,const int oldwidth,CMatrixDouble &b,const int newheight,const int newwidth);
   static void       Spline2DResampleBilinear(CMatrixDouble &a,const int oldheight,const int oldwidth,CMatrixDouble &b,const int newheight,const int newwidth);
   static void       Spline2DBuildBilinearV(CRowDouble &x,int n,CRowDouble &y,int m,CRowDouble &f,int d,CSpline2DInterpolantShell &c);
   static void       Spline2DBuildBicubicV(CRowDouble &x,int n,CRowDouble &y,int m,CRowDouble &f,int d,CSpline2DInterpolantShell &c);
   static void       Spline2DUnpackV(CSpline2DInterpolantShell &c,int &m,int &n,int &d,CMatrixDouble &tbl);
   static void       Spline2DBuildBilinear(double &x[],double &y[],CMatrixDouble &f,const int m,const int n,CSpline2DInterpolantShell &c);
   static void       Spline2DBuildBicubic(double &x[],double &y[],CMatrixDouble &f,const int m,const int n,CSpline2DInterpolantShell &c);
   static void       Spline2DUnpack(CSpline2DInterpolantShell &c,int &m,int &n,CMatrixDouble &tbl);
   static void       Spline2DBuilderCreate(int d,CSpline2DBuilder &state);
   static void       Spline2DBuilderSetUserTerm(CSpline2DBuilder &state,double v);
   static void       Spline2DBuilderSetLinTerm(CSpline2DBuilder &state);
   static void       Spline2DBuilderSetConstTerm(CSpline2DBuilder &state);
   static void       Spline2DBuilderSetZeroTerm(CSpline2DBuilder &state);
   static void       Spline2DBuilderSetPoints(CSpline2DBuilder &state,CMatrixDouble &xy,int n);
   static void       Spline2DBuilderSetAreaAuto(CSpline2DBuilder &state);
   static void       Spline2DBuilderSetArea(CSpline2DBuilder &state,double xa,double xb,double ya,double yb);
   static void       Spline2DBuilderSetGrid(CSpline2DBuilder &state,int kx,int ky);
   static void       Spline2DBuilderSetAlgoFastDDM(CSpline2DBuilder &state,int nlayers,double lambdav);
   static void       Spline2DBuilderSetAlgoBlockLLS(CSpline2DBuilder &state,double lambdans);
   static void       Spline2DBuilderSetAlgoNaiveLLS(CSpline2DBuilder &state,double lambdans);
   static void       Spline2DFit(CSpline2DBuilder &state,CSpline2DInterpolantShell &s,CSpline2DFitReport &rep);
   //--- 3-dimensional spline interpolation
   static double     Spline3DCalc(CSpline3DInterpolant &c,double x,double y,double z);
   static void       Spline3DLinTransXYZ(CSpline3DInterpolant &c,double ax,double bx,double ay,double by,double az,double bz);
   static void       Spline3DLinTransF(CSpline3DInterpolant &c,double a,double b);
   static void       Spline3DResampleTrilinear(CRowDouble &a,int oldzcount,int oldycount,int oldxcount,int newzcount,int newycount,int newxcount,CRowDouble &b);
   static void       Spline3DBuildTrilinearV(CRowDouble &x,int n,CRowDouble &y,int m,CRowDouble &z,int l,CRowDouble &f,int d,CSpline3DInterpolant &c);
   static void       Spline3DCalcVBuf(CSpline3DInterpolant &c,double x,double y,double z,CRowDouble &f);
   static void       Spline3DCalcV(CSpline3DInterpolant &c,double x,double y,double z,CRowDouble &f);
   static void       Spline3DUnpackV(CSpline3DInterpolant &c,int &n,int &m,int &l,int &d,int &stype,CMatrixDouble &tbl);
   //--- RBF model
   static void       RBFSerialize(CRBFModel &obj,string &s_out);
   static void       RBFUnserialize(string s_in,CRBFModel &obj);
   static void       RBFCreate(int nx,int ny,CRBFModel &s);
   static void       RBFCreateCalcBuffer(CRBFModel &s,CRBFCalcBuffer &buf);
   static void       RBFSetPoints(CRBFModel &s,CMatrixDouble &xy,int n);
   static void       RBFSetPoints(CRBFModel &s,CMatrixDouble &xy);
   static void       RBFSetPointsAndScales(CRBFModel &r,CMatrixDouble &xy,int n,CRowDouble &s);
   static void       RBFSetPointsAndScales(CRBFModel &r,CMatrixDouble &xy,CRowDouble &s);
   static void       RBFSetAlgoQNN(CRBFModel &s,double q=1.0,double z=5.0);
   static void       RBFSetAlgoMultilayer(CRBFModel &s,double rbase,int nlayers,double lambdav=0.01);
   static void       RBFSetAlgoHierarchical(CRBFModel &s,double rbase,int nlayers,double lambdans);
   static void       RBFSetAlgoThinPlateSpline(CRBFModel &s,double lambdav=0.0);
   static void       RBFSetAlgoMultiQuadricManual(CRBFModel &s,double alpha,double lambdav=0.0);
   static void       RBFSetAlgoMultiQuadricAuto(CRBFModel &s,double lambdav=0.0);
   static void       RBFSetAlgoBiharmonic(CRBFModel &s,double lambdav=0.0);
   static void       RBFSetLinTerm(CRBFModel &s);
   static void       RBFSetConstTerm(CRBFModel &s);
   static void       RBFSetZeroTerm(CRBFModel &s);
   static void       RBFSetV2BF(CRBFModel &s,int bf);
   static void       RBFSetV2Its(CRBFModel &s,int maxits);
   static void       RBFSetV2SupportR(CRBFModel &s,double r);
   static void       RBFBuildModel(CRBFModel &s,CRBFReport &rep);
   static double     RBFCalc1(CRBFModel &s,double x0);
   static double     RBFCalc2(CRBFModel &s,double x0,double x1);
   static double     RBFCalc3(CRBFModel &s,double x0,double x1,double x2);
   static void       RBFDiff1(CRBFModel &s,double x0,double &y,double &dy0);
   static void       RBFDiff2(CRBFModel &s,double x0,double x1,double &y,double &dy0,double &dy1);
   static void       RBFDiff3(CRBFModel &s,double x0,double x1,double x2,double &y,double &dy0,double &dy1,double &dy2);
   static void       RBFCalc(CRBFModel &s,CRowDouble &x,CRowDouble &y);
   static void       RBFDiff(CRBFModel &s,CRowDouble &x,CRowDouble &y,CRowDouble &dy);
   static void       RBFHess(CRBFModel &s,CRowDouble &x,CRowDouble &y,CRowDouble &dy,CRowDouble &d2y);
   static void       RBFCalcBuf(CRBFModel &s,CRowDouble &x,CRowDouble &y);
   static void       RBFDiffBuf(CRBFModel &s,CRowDouble &x,CRowDouble &y,CRowDouble &dy);
   static void       RBFHessBuf(CRBFModel &s,CRowDouble &x,CRowDouble &y,CRowDouble &dy,CRowDouble &d2y);
   static void       RBFTSCalcBuf(CRBFModel &s,CRBFCalcBuffer &buf,CRowDouble &x,CRowDouble &y);
   static void       RBFTSDiffBuf(CRBFModel &s,CRBFCalcBuffer &buf,CRowDouble &x,CRowDouble &y,CRowDouble &dy);
   static void       RBFTSHessBuf(CRBFModel &s,CRBFCalcBuffer &buf,CRowDouble &x,CRowDouble &y,CRowDouble &dy,CRowDouble &d2y);
   static void       RBFGridCalc2(CRBFModel &s,CRowDouble &x0,int n0,CRowDouble &x1,int n1,CMatrixDouble &y);
   static void       RBFGridCalc2V(CRBFModel &s,CRowDouble &x0,int n0,CRowDouble &x1,int n1,CRowDouble &y);
   static void       RBFGridCalc2VSubset(CRBFModel &s,CRowDouble &x0,int n0,CRowDouble &x1,int n1,bool &flagy[],CRowDouble &y);
   static void       RBFGridCalc3V(CRBFModel &s,CRowDouble &x0,int n0,CRowDouble &x1,int n1,CRowDouble &x2,int n2,CRowDouble &y);
   static void       RBFGridCalc3VSubset(CRBFModel &s,CRowDouble &x0,int n0,CRowDouble &x1,int n1,CRowDouble &x2,int n2,bool &flagy[],CRowDouble &y);
   static void       RBFUnpack(CRBFModel &s,int &nx,int &ny,CMatrixDouble &xwr,int &nc,CMatrixDouble &v,int &modelversion);
   static int        RBFGetModelVersion(CRBFModel &s);
   static double     RBFPeekProgress(CRBFModel &s);
   static void       RBFRequestTermination(CRBFModel &s);
   //--- functions of package linalg
   //--- working with matrix forms
   static void       CMatrixTranspose(const int m,const int n,CMatrixComplex &a,const int ia,const int ja,CMatrixComplex &b,const int ib,const int jb);
   static void       RMatrixTranspose(const int m,const int n,CMatrixDouble &a,const int ia,const int ja,CMatrixDouble &b,const int ib,const int jb);
   static void       RMatrixEnforceSymmetricity(CMatrixDouble &a,int n,bool IsUpper);
   static void       CMatrixCopy(const int m,const int n,CMatrixComplex &a,const int ia,const int ja,CMatrixComplex &b,const int ib,const int jb);
   static void       RVectorCopy(int n,CRowDouble &a,int ia,CRowDouble &b,int ib);
   static void       RMatrixCopy(const int m,const int n,CMatrixDouble &a,const int ia,const int ja,CMatrixDouble &b,const int ib,const int jb);
   static void       RMatrixGenCopy(int m,int n,double alpha,CMatrixDouble &a,int ia,int ja,double beta,CMatrixDouble &b,int ib,int jb);
   static void       RMatrixGer(int m,int n,CMatrixDouble &a,int ia,int ja,double alpha,CRowDouble &u,int iu,CRowDouble &v,int iv);
   static void       CMatrixRank1(const int m,const int n,CMatrixComplex &a,const int ia,const int ja,complex &u[],const int iu,complex &v[],const int iv);
   static void       RMatrixRank1(const int m,const int n,CMatrixDouble &a,const int ia,const int ja,double &u[],const int iu,double &v[],const int iv);
   static void       RMatrixGemVect(int m,int n,double alpha,CMatrixDouble &a,int ia,int ja,int opa,CRowDouble &x,int ix,double beta,CRowDouble &y,int iy);
   static void       CMatrixMVect(const int m,const int n,CMatrixComplex &a,const int ia,const int ja,const int opa,complex &x[],const int ix,complex &y[],const int iy);
   static void       RMatrixMVect(const int m,const int n,CMatrixDouble &a,const int ia,const int ja,const int opa,double &x[],const int ix,double &y[],const int iy);
   static void       RMatrixSymVect(int n,double alpha,CMatrixDouble &a,int ia,int ja,bool IsUpper,CRowDouble &x,int ix,double beta,CRowDouble &y,int iy);
   static double     RMatrixSyvMVect(int n,CMatrixDouble &a,int ia,int ja,bool IsUpper,CRowDouble &x,int ix,CRowDouble &tmp);
   static void       RMatrixTrsVect(int n,CMatrixDouble &a,int ia,int ja,bool IsUpper,bool IsUnit,int OpType,CRowDouble &x,int ix);
   static void       CMatrixRightTrsM(const int m,const int n,CMatrixComplex &a,const int i1,const int j1,const bool IsUpper,const bool IsUnit,const int OpType,CMatrixComplex &x,const int i2,const int j2);
   static void       CMatrixLeftTrsM(const int m,const int n,CMatrixComplex &a,const int i1,const int j1,const bool IsUpper,const bool IsUnit,const int OpType,CMatrixComplex &x,const int i2,const int j2);
   static void       RMatrixRightTrsM(const int m,const int n,CMatrixDouble &a,const int i1,const int j1,const bool IsUpper,const bool IsUnit,const int OpType,CMatrixDouble &x,const int i2,const int j2);
   static void       RMatrixLeftTrsM(const int m,const int n,CMatrixDouble &a,const int i1,const int j1,const bool IsUpper,const bool IsUnit,const int OpType,CMatrixDouble &x,const int i2,const int j2);
   static void       CMatrixSyrk(const int n,const int k,const double alpha,CMatrixComplex &a,const int ia,const int ja,const int optypea,const double beta,CMatrixComplex &c,const int ic,const int jc,const bool IsUpper);
   static void       RMatrixSyrk(const int n,const int k,const double alpha,CMatrixDouble &a,const int ia,const int ja,const int optypea,const double beta,CMatrixDouble &c,const int ic,const int jc,const bool IsUpper);
   static void       CMatrixGemm(const int m,const int n,const int k,complex alpha,CMatrixComplex &a,const int ia,const int ja,const int optypea,CMatrixComplex &b,const int ib,const int jb,const int optypeb,complex beta,CMatrixComplex &c,const int ic,const int jc);
   static void       RMatrixGemm(const int m,const int n,const int k,const double alpha,CMatrixDouble &a,const int ia,const int ja,const int optypea,CMatrixDouble &b,const int ib,const int jb,const int optypeb,const double beta,CMatrixDouble &c,const int ic,const int jc);
   //--- orthogonal factorizations
   static void       RMatrixQR(CMatrixDouble &a,const int m,const int n,double &tau[]);
   static void       RMatrixLQ(CMatrixDouble &a,const int m,const int n,double &tau[]);
   static void       CMatrixQR(CMatrixComplex &a,const int m,const int n,complex &tau[]);
   static void       CMatrixLQ(CMatrixComplex &a,const int m,const int n,complex &tau[]);
   static void       RMatrixQRUnpackQ(CMatrixDouble &a,const int m,const int n,double &tau[],const int qcolumns,CMatrixDouble &q);
   static void       RMatrixQRUnpackR(CMatrixDouble &a,const int m,const int n,CMatrixDouble &r);
   static void       RMatrixLQUnpackQ(CMatrixDouble &a,const int m,const int n,double &tau[],const int qrows,CMatrixDouble &q);
   static void       RMatrixLQUnpackL(CMatrixDouble &a,const int m,const int n,CMatrixDouble &l);
   static void       CMatrixQRUnpackQ(CMatrixComplex &a,const int m,const int n,complex &tau[],const int qcolumns,CMatrixComplex &q);
   static void       CMatrixQRUnpackR(CMatrixComplex &a,const int m,const int n,CMatrixComplex &r);
   static void       CMatrixLQUnpackQ(CMatrixComplex &a,const int m,const int n,complex &tau[],const int qrows,CMatrixComplex &q);
   static void       CMatrixLQUnpackL(CMatrixComplex &a,const int m,const int n,CMatrixComplex &l);
   static void       RMatrixBD(CMatrixDouble &a,const int m,const int n,double &tauq[],double &taup[]);
   static void       RMatrixBDUnpackQ(CMatrixDouble &qp,const int m,const int n,double &tauq[],const int qcolumns,CMatrixDouble &q);
   static void       RMatrixBDMultiplyByQ(CMatrixDouble &qp,const int m,const int n,double &tauq[],CMatrixDouble &z,const int zrows,const int zcolumns,const bool fromtheright,const bool dotranspose);
   static void       RMatrixBDUnpackPT(CMatrixDouble &qp,const int m,const int n,double &taup[],const int ptrows,CMatrixDouble &pt);
   static void       RMatrixBDMultiplyByP(CMatrixDouble &qp,const int m,const int n,double &taup[],CMatrixDouble &z,const int zrows,const int zcolumns,const bool fromtheright,const bool dotranspose);
   static void       RMatrixBDUnpackDiagonals(CMatrixDouble &b,const int m,const int n,bool &IsUpper,double &d[],double &e[]);
   static void       RMatrixHessenberg(CMatrixDouble &a,const int n,double &tau[]);
   static void       RMatrixHessenbergUnpackQ(CMatrixDouble &a,const int n,double &tau[],CMatrixDouble &q);
   static void       RMatrixHessenbergUnpackH(CMatrixDouble &a,const int n,CMatrixDouble &h);
   static void       SMatrixTD(CMatrixDouble &a,const int n,const bool IsUpper,double &tau[],double &d[],double &e[]);
   static void       SMatrixTDUnpackQ(CMatrixDouble &a,const int n,const bool IsUpper,double &tau[],CMatrixDouble &q);
   static void       HMatrixTD(CMatrixComplex &a,const int n,const bool IsUpper,complex &tau[],double &d[],double &e[]);
   static void       HMatrixTDUnpackQ(CMatrixComplex &a,const int n,const bool IsUpper,complex &tau[],CMatrixComplex &q);
   //--- eigenvalues and eigenvectors
   static void       EigSubSpaceCreate(int n,int k,CEigSubSpaceState &state);
   static void       EigSubSpaceCreateBuf(int n,int k,CEigSubSpaceState &state);
   static void       EigSubSpaceSetCond(CEigSubSpaceState &state,double eps,int maxits);
   static void       EigSubSpaceSetWarmStart(CEigSubSpaceState &state,bool usewarmstart);
   static void       EigSubSpaceOOCStart(CEigSubSpaceState &state,int mtype);
   static bool       EigSubSpaceOOCContinue(CEigSubSpaceState &state);
   static void       EigSubSpaceOOCGetRequestInfo(CEigSubSpaceState &state,int &requesttype,int &requestsize);
   static void       EigSubSpaceOOCGetRequestData(CEigSubSpaceState &state,CMatrixDouble &x);
   static void       EigSubSpaceOOCSendResult(CEigSubSpaceState &state,CMatrixDouble &ax);
   static void       EigSubSpaceOOCStop(CEigSubSpaceState &state,CRowDouble &w,CMatrixDouble &z,CEigSubSpaceReport &rep);
   static void       EigSubSpaceSolveDenses(CEigSubSpaceState &state,CMatrixDouble &a,bool IsUpper,CRowDouble &w,CMatrixDouble &z,CEigSubSpaceReport &rep);
   static void       EigSubSpaceSolveSparses(CEigSubSpaceState &state,CSparseMatrix &a,bool IsUpper,CRowDouble &w,CMatrixDouble &z,CEigSubSpaceReport &rep);
   static bool       SMatrixEVD(CMatrixDouble &a,const int n,int zneeded,const bool IsUpper,double &d[],CMatrixDouble &z);
   static bool       SMatrixEVDR(CMatrixDouble &a,const int n,int zneeded,const bool IsUpper,double b1,double b2,int &m,double &w[],CMatrixDouble &z);
   static bool       SMatrixEVDI(CMatrixDouble &a,const int n,int zneeded,const bool IsUpper,const int i1,const int i2,double &w[],CMatrixDouble &z);
   static bool       HMatrixEVD(CMatrixComplex &a,const int n,const int zneeded,const bool IsUpper,double &d[],CMatrixComplex &z);
   static bool       HMatrixEVDR(CMatrixComplex &a,const int n,const int zneeded,const bool IsUpper,double b1,double b2,int &m,double &w[],CMatrixComplex &z);
   static bool       HMatrixEVDI(CMatrixComplex &a,const int n,const int zneeded,const bool IsUpper,const int i1,const int i2,double &w[],CMatrixComplex &z);
   static bool       SMatrixTdEVD(double &d[],double &e[],const int n,const int zneeded,CMatrixDouble &z);
   static bool       SMatrixTdEVDR(double &d[],double &e[],const int n,const int zneeded,const double a,const double b,int &m,CMatrixDouble &z);
   static bool       SMatrixTdEVDI(double &d[],double &e[],const int n,const int zneeded,const int i1,const int i2,CMatrixDouble &z);
   static bool       RMatrixEVD(CMatrixDouble &a,const int n,const int vneeded,double &wr[],double &wi[],CMatrixDouble &vl,CMatrixDouble &vr);
   //--- random matrix generation
   static void       RMatrixRndOrthogonal(const int n,CMatrixDouble &a);
   static void       RMatrixRndCond(const int n,const double c,CMatrixDouble &a);
   static void       CMatrixRndOrthogonal(const int n,CMatrixComplex &a);
   static void       CMatrixRndCond(const int n,const double c,CMatrixComplex &a);
   static void       SMatrixRndCond(const int n,const double c,CMatrixDouble &a);
   static void       SPDMatrixRndCond(const int n,const double c,CMatrixDouble &a);
   static void       HMatrixRndCond(const int n,const double c,CMatrixComplex &a);
   static void       HPDMatrixRndCond(const int n,const double c,CMatrixComplex &a);
   static void       RMatrixRndOrthogonalFromTheRight(CMatrixDouble &a,const int m,const int n);
   static void       RMatrixRndOrthogonalFromTheLeft(CMatrixDouble &a,const int m,const int n);
   static void       CMatrixRndOrthogonalFromTheRight(CMatrixComplex &a,const int m,const int n);
   static void       CMatrixRndOrthogonalFromTheLeft(CMatrixComplex &a,const int m,const int n);
   static void       SMatrixRndMultiply(CMatrixDouble &a,const int n);
   static void       HMatrixRndMultiply(CMatrixComplex &a,const int n);
   //--- sparse matrix
   static void       SparseSerialize(CSparseMatrix &obj,string &s_out);
   static void       SparseUunserialize(string s_in,CSparseMatrix &obj);
   static void       SparseCreate(int m,int n,int k,CSparseMatrix &s);
   static void       SparseCreate(int m,int n,CSparseMatrix &s);
   static void       SparseCreateBuf(int m,int n,int k,CSparseMatrix &s);
   static void       SparseCreateBuf(int m,int n,CSparseMatrix &s);
   static void       SparseCreateCRS(int m,int n,CRowInt &ner,CSparseMatrix &s);
   static void       SparseCreateCRS(int m,int n,int &ner[],CSparseMatrix &s);
   static void       SparseCreateCRSBuf(int m,int n,CRowInt &ner,CSparseMatrix &s);
   static void       SparseCreateCRSBuf(int m,int n,int &ner[],CSparseMatrix &s);
   static void       SparseCreateSKS(int m,int n,CRowInt &d,CRowInt &u,CSparseMatrix &s);
   static void       SparseCreateSKS(int m,int n,int &d[],int &u[],CSparseMatrix &s);
   static void       SparseCreateSKSBuf(int m,int n,CRowInt &d,CRowInt &u,CSparseMatrix &s);
   static void       SparseCreateSKSBuf(int m,int n,int &d[],int &u[],CSparseMatrix &s);
   static void       SparseCreateSKSBand(int m,int n,int bw,CSparseMatrix &s);
   static void       SparseCreateSKSBandBuf(int m,int n,int bw,CSparseMatrix &s);
   static void       SparseCopy(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseCopyBuf(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseSwap(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseAdd(CSparseMatrix &s,int i,int j,double v);
   static void       SparseSet(CSparseMatrix &s,int i,int j,double v);
   static double     SparseGet(CSparseMatrix &s,int i,int j);
   static bool       SparseExists(CSparseMatrix &s,int i,int j);
   static double     SparseGetDiagonal(CSparseMatrix &s,int i);
   static void       SparseMV(CSparseMatrix &s,CRowDouble &x,CRowDouble &y);
   static void       SparseMTV(CSparseMatrix &s,CRowDouble &x,CRowDouble &y);
   static void       SparseGemV(CSparseMatrix &s,double alpha,int ops,CRowDouble &x,int ix,double beta,CRowDouble &y,int iy);
   static void       SparseMV2(CSparseMatrix &s,CRowDouble &x,CRowDouble &y0,CRowDouble &y1);
   static void       SparseSMV(CSparseMatrix &s,bool IsUpper,CRowDouble &x,CRowDouble &y);
   static double     SparseVSMV(CSparseMatrix &s,bool IsUpper,CRowDouble &x);
   static void       SparseMM(CSparseMatrix &s,CMatrixDouble &a,int k,CMatrixDouble &b);
   static void       SparseMTM(CSparseMatrix &s,CMatrixDouble &a,int k,CMatrixDouble &b);
   static void       SparseMM2(CSparseMatrix &s,CMatrixDouble &a,int k,CMatrixDouble &b0,CMatrixDouble &b1);
   static void       SparseSMM(CSparseMatrix &s,bool IsUpper,CMatrixDouble &a,int k,CMatrixDouble &b);
   static void       SparseTRMV(CSparseMatrix &s,bool IsUpper,bool IsUnit,int OpType,CRowDouble &x,CRowDouble &y);
   static void       SparseTRSV(CSparseMatrix &s,bool IsUpper,bool IsUnit,int OpType,CRowDouble &x);
   static void       SparseSymmPermTbl(CSparseMatrix &a,bool IsUpper,CRowInt &p,CSparseMatrix &b);
   static void       SparseSymmPermTblBuf(CSparseMatrix &a,bool IsUpper,CRowInt &p,CSparseMatrix &b);
   static void       SparseResizeMatrix(CSparseMatrix &s);
   static bool       SparseEnumerate(CSparseMatrix &s,int &t0,int &t1,int &i,int &j,double &v);
   static bool       SparseRewriteExisting(CSparseMatrix &s,int i,int j,double v);
   static void       SparseGetRow(CSparseMatrix &s,int i,CRowDouble &irow);
   static void       SparseGetCompressedRow(CSparseMatrix &s,int i,CRowInt &colidx,CRowDouble &vals,int &nzcnt);
   static void       SparseTransposeSKS(CSparseMatrix &s);
   static void       SparseTransposeCRS(CSparseMatrix &s);
   static void       SparseCopyTransposeCRS(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseCopyTransposeCRSBuf(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseConvertTo(CSparseMatrix &s0,int fmt);
   static void       SparseCopyToBuf(CSparseMatrix &s0,int fmt,CSparseMatrix &s1);
   static void       SparseConvertToHash(CSparseMatrix &s);
   static void       SparseCopyToHash(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseCopyToHashBuf(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseConvertToCRS(CSparseMatrix &s);
   static void       SparseCopyToCRS(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseCopyToCRSBuf(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseConvertToSKS(CSparseMatrix &s);
   static void       SparseCopyToSKS(CSparseMatrix &s0,CSparseMatrix &s1);
   static void       SparseCopyToSKSBuf(CSparseMatrix &s0,CSparseMatrix &s1);
   static int        SparseGetMatrixType(CSparseMatrix &s);
   static bool       SparseIsHash(CSparseMatrix &s);
   static bool       SparseIsCRS(CSparseMatrix &s);
   static bool       SparseIsSKS(CSparseMatrix &s);
   static void       SparseFree(CSparseMatrix &s);
   static int        SparseGetNRows(CSparseMatrix &s);
   static int        SparseGetNCols(CSparseMatrix &s);
   static int        SparseGetUpperCount(CSparseMatrix &s);
   static int        SparseGetLowerCount(CSparseMatrix &s);
   //--- triangular factorizations
   static void       RMatrixLU(CMatrixDouble &a,const int m,const int n,int &pivots[]);
   static void       CMatrixLU(CMatrixComplex &a,const int m,const int n,int &pivots[]);
   static bool       HPDMatrixCholesky(CMatrixComplex &a,const int n,const bool IsUpper);
   static bool       SPDMatrixCholesky(CMatrixDouble &a,const int n,const bool IsUpper);
   static void       SPDMatrixCholeskyUpdateAdd1(CMatrixDouble &a,int n,bool IsUpper,CRowDouble &u);
   static void       SPDMatrixCholeskyUpdateFix(CMatrixDouble &a,int n,bool IsUpper,bool &fix[]);
   static void       SPDMatrixCholeskyUpdateAdd1Buf(CMatrixDouble &a,int n,bool IsUpper,CRowDouble &u,CRowDouble &bufr);
   static void       SPDMatrixCholeskyUpdateFixBuf(CMatrixDouble &a,int n,bool IsUpper,bool &fix[],CRowDouble &bufr);
   static bool       SparseLU(CSparseMatrix &a,int pivottype,CRowInt &p,CRowInt &q);
   static bool       SparseCholeskySkyLine(CSparseMatrix &a,int n,bool IsUpper);
   static bool       SparseCholesky(CSparseMatrix &a,bool IsUpper);
   static bool       SparseCholeskyP(CSparseMatrix &a,bool IsUpper,CRowInt &p);
   static bool       sparsecholeskyanalyze(CSparseMatrix &a,bool IsUpper,int facttype,int permtype,CSparseDecompositionAnalysis &analysis);
   static bool       SparseCholeskyFactorize(CSparseDecompositionAnalysis &analysis,bool needupper,CSparseMatrix &a,CRowDouble &d,CRowInt &p);
   static void       SparseCholeskyReload(CSparseDecompositionAnalysis &analysis,CSparseMatrix &a,bool IsUpper);
   //--- estimate of the condition numbers
   static double     RMatrixRCond1(CMatrixDouble &a,const int n);
   static double     RMatrixRCondInf(CMatrixDouble &a,const int n);
   static double     SPDMatrixRCond(CMatrixDouble &a,const int n,const bool IsUpper);
   static double     RMatrixTrRCond1(CMatrixDouble &a,const int n,const bool IsUpper,const bool IsUnit);
   static double     RMatrixTrRCondInf(CMatrixDouble &a,const int n,const bool IsUpper,const bool IsUnit);
   static double     HPDMatrixRCond(CMatrixComplex &a,const int n,const bool IsUpper);
   static double     CMatrixRCond1(CMatrixComplex &a,const int n);
   static double     CMatrixRCondInf(CMatrixComplex &a,const int n);
   static double     RMatrixLURCond1(CMatrixDouble &lua,const int n);
   static double     RMatrixLURCondInf(CMatrixDouble &lua,const int n);
   static double     SPDMatrixCholeskyRCond(CMatrixDouble &a,const int n,const bool IsUpper);
   static double     HPDMatrixCholeskyRCond(CMatrixComplex &a,const int n,const bool IsUpper);
   static double     CMatrixLURCond1(CMatrixComplex &lua,const int n);
   static double     CMatrixLURCondInf(CMatrixComplex &lua,const int n);
   static double     CMatrixTrRCond1(CMatrixComplex &a,const int n,const bool IsUpper,const bool IsUnit);
   static double     CMatrixTrRCondInf(CMatrixComplex &a,const int n,const bool IsUpper,const bool IsUnit);
   //--- norm estimator
   static void       NormEstimatorCreate(int m,int n,int nstart,int nits,CNormEstimatorState &state);
   static void       NormEstimatorSetSeed(CNormEstimatorState &state,int seedval);
   static void       NormEstimatorEstimateSparse(CNormEstimatorState &state,CSparseMatrix &a);
   static void       NormEstimatorResults(CNormEstimatorState &state,double &nrm);
   //--- matrix inversion
   static void       RMatrixLUInverse(CMatrixDouble &a,int &pivots[],const int n,int &info,CMatInvReportShell &rep);
   static void       RMatrixLUInverse(CMatrixDouble &a,int &pivots[],int &info,CMatInvReportShell &rep);
   static void       RMatrixInverse(CMatrixDouble &a,const int n,int &info,CMatInvReportShell &rep);
   static void       RMatrixInverse(CMatrixDouble &a,int &info,CMatInvReportShell &rep);
   static void       CMatrixLUInverse(CMatrixComplex &a,int &pivots[],const int n,int &info,CMatInvReportShell &rep);
   static void       CMatrixLUInverse(CMatrixComplex &a,int &pivots[],int &info,CMatInvReportShell &rep);
   static void       CMatrixInverse(CMatrixComplex &a,const int n,int &info,CMatInvReportShell &rep);
   static void       CMatrixInverse(CMatrixComplex &a,int &info,CMatInvReportShell &rep);
   static void       SPDMatrixCholeskyInverse(CMatrixDouble &a,const int n,const bool IsUpper,int &info,CMatInvReportShell &rep);
   static void       SPDMatrixCholeskyInverse(CMatrixDouble &a,int &info,CMatInvReportShell &rep);
   static void       SPDMatrixInverse(CMatrixDouble &a,const int n,const bool IsUpper,int &info,CMatInvReportShell &rep);
   static void       SPDMatrixInverse(CMatrixDouble &a,int &info,CMatInvReportShell &rep);
   static void       HPDMatrixCholeskyInverse(CMatrixComplex &a,const int n,const bool IsUpper,int &info,CMatInvReportShell &rep);
   static void       HPDMatrixCholeskyInverse(CMatrixComplex &a,int &info,CMatInvReportShell &rep);
   static void       HPDMatrixInverse(CMatrixComplex &a,const int n,const bool IsUpper,int &info,CMatInvReportShell &rep);
   static void       HPDMatrixInverse(CMatrixComplex &a,int &info,CMatInvReportShell &rep);
   static void       RMatrixTrInverse(CMatrixDouble &a,const int n,const bool IsUpper,const bool IsUnit,int &info,CMatInvReportShell &rep);
   static void       RMatrixTrInverse(CMatrixDouble &a,const bool IsUpper,int &info,CMatInvReportShell &rep);
   static void       CMatrixTrInverse(CMatrixComplex &a,const int n,const bool IsUpper,const bool IsUnit,int &info,CMatInvReportShell &rep);
   static void       CMatrixTrInverse(CMatrixComplex &a,const bool IsUpper,int &info,CMatInvReportShell &rep);
   //--- singular value decomposition of a bidiagonal matrix
   static bool       RMatrixBdSVD(double &d[],double &e[],const int n,const bool IsUpper,bool isfractionalaccuracyrequired,CMatrixDouble &u,const int nru,CMatrixDouble &c,const int ncc,CMatrixDouble &vt,const int ncvt);
   //--- singular value decomposition
   static bool       RMatrixSVD(CMatrixDouble &a,const int m,const int n,const int uneeded,const int vtneeded,const int additionalmemory,double &w[],CMatrixDouble &u,CMatrixDouble &vt);
   //--- calculation determinant of the matrix
   static double     RMatrixLUDet(CMatrixDouble &a,int &pivots[],const int n);
   static double     RMatrixLUDet(CMatrixDouble &a,int &pivots[]);
   static double     RMatrixDet(CMatrixDouble &a,const int n);
   static double     RMatrixDet(CMatrixDouble &a);
   static complex    CMatrixLUDet(CMatrixComplex &a,int &pivots[],const int n);
   static complex    CMatrixLUDet(CMatrixComplex &a,int &pivots[]);
   static complex    CMatrixDet(CMatrixComplex &a,const int n);
   static complex    CMatrixDet(CMatrixComplex &a);
   static double     SPDMatrixCholeskyDet(CMatrixDouble &a,const int n);
   static double     SPDMatrixCholeskyDet(CMatrixDouble &a);
   static double     SPDMatrixDet(CMatrixDouble &a,const int n,const bool IsUpper);
   static double     SPDMatrixDet(CMatrixDouble &a);
   //--- generalized symmetric positive definite eigenproblem
   static bool       SMatrixGEVD(CMatrixDouble &a,const int n,const bool isuppera,CMatrixDouble &b,const bool isupperb,const int zneeded,const int problemtype,double &d[],CMatrixDouble &z);
   static bool       SMatrixGEVDReduce(CMatrixDouble &a,const int n,const bool isuppera,CMatrixDouble &b,const bool isupperb,const int problemtype,CMatrixDouble &r,bool &isupperr);
   //--- update of the inverse matrix by the Sherman-Morrison formula
   static void       RMatrixInvUpdateSimple(CMatrixDouble &inva,const int n,const int updrow,const int updcolumn,const double updval);
   static void       RMatrixInvUpdateRow(CMatrixDouble &inva,const int n,const int updrow,double &v[]);
   static void       RMatrixInvUpdateColumn(CMatrixDouble &inva,const int n,const int updcolumn,double &u[]);
   static void       RMatrixInvUpdateUV(CMatrixDouble &inva,const int n,double &u[],double &v[]);
   //--- Schur decomposition
   static bool       RMatrixSchur(CMatrixDouble &a,const int n,CMatrixDouble &s);
   //--- functions of package optimization
   //--- conjugate gradient method
   static void       MinCGCreate(const int n,double &x[],CMinCGStateShell &state);
   static void       MinCGCreate(double &x[],CMinCGStateShell &state);
   static void       MinCGCreateF(const int n,double &x[],double diffstep,CMinCGStateShell &state);
   static void       MinCGCreateF(double &x[],double diffstep,CMinCGStateShell &state);
   static void       MinCGSetCond(CMinCGStateShell &state,double epsg,double epsf,double epsx,int maxits);
   static void       MinCGSetScale(CMinCGStateShell &state,double &s[]);
   static void       MinCGSetXRep(CMinCGStateShell &state,bool needxrep);
   static void       MinCGSetCGType(CMinCGStateShell &state,int cgtype);
   static void       MinCGSetStpMax(CMinCGStateShell &state,double stpmax);
   static void       MinCGSuggestStep(CMinCGStateShell &state,double stp);
   static void       MinCGSetPrecDefault(CMinCGStateShell &state);
   static void       MinCGSetPrecDiag(CMinCGStateShell &state,double &d[]);
   static void       MinCGSetPrecScale(CMinCGStateShell &state);
   static bool       MinCGIteration(CMinCGStateShell &state);
   static void       MinCGOptimize(CMinCGStateShell &state,CNDimensional_Func &func,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinCGOptimize(CMinCGStateShell &state,CNDimensional_Grad &grad,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinCGResults(CMinCGStateShell &state,double &x[],CMinCGReportShell &rep);
   static void       MinCGResultsBuf(CMinCGStateShell &state,double &x[],CMinCGReportShell &rep);
   static void       MinCGRestartFrom(CMinCGStateShell &state,double &x[]);
   static void       MinLBFGSRequestTermination(CMinLBFGSStateShell &state);
   //--- bound constrained optimization with additional linear equality and inequality constraints
   static void       MinBLEICCreate(const int n,double &x[],CMinBLEICStateShell &state);
   static void       MinBLEICCreate(double &x[],CMinBLEICStateShell &state);
   static void       MinBLEICCreateF(const int n,double &x[],double diffstep,CMinBLEICStateShell &state);
   static void       MinBLEICCreateF(double &x[],double diffstep,CMinBLEICStateShell &state);
   static void       MinBLEICSetBC(CMinBLEICStateShell &state,double &bndl[],double &bndu[]);
   static void       MinBLEICSetLC(CMinBLEICStateShell &state,CMatrixDouble &c,int &ct[],const int k);
   static void       MinBLEICSetLC(CMinBLEICStateShell &state,CMatrixDouble &c,int &ct[]);
   static void       MinBLEICSetInnerCond(CMinBLEICStateShell &state,const double epsg,const double epsf,const double epsx);
   static void       MinBLEICSetOuterCond(CMinBLEICStateShell &state,const double epsx,const double epsi);
   static void       MinBLEICSetScale(CMinBLEICStateShell &state,double &s[]);
   static void       MinBLEICSetPrecDefault(CMinBLEICStateShell &state);
   static void       MinBLEICSetPrecDiag(CMinBLEICStateShell &state,double &d[]);
   static void       MinBLEICSetPrecScale(CMinBLEICStateShell &state);
   static void       MinBLEICSetMaxIts(CMinBLEICStateShell &state,const int maxits);
   static void       MinBLEICSetXRep(CMinBLEICStateShell &state,bool needxrep);
   static void       MinBLEICSetStpMax(CMinBLEICStateShell &state,double stpmax);
   static bool       MinBLEICIteration(CMinBLEICStateShell &state);
   static void       MinBLEICOptimize(CMinBLEICStateShell &state,CNDimensional_Func &func,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinBLEICOptimize(CMinBLEICStateShell &state,CNDimensional_Grad &grad,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinBLEICOptGuardGradient(CMinBLEICStateShell &state,double teststep);
   static void       MinBLEICOptGuardSmoothness(CMinBLEICStateShell &state,int level=1);
   static void       MinBLEICOptGuardResults(CMinBLEICStateShell &state,COptGuardReport &rep);
   static void       MinBLEICOptGuardNonC1Test0Results(CMinBLEICStateShell &state,COptGuardNonC1Test0Report &strrep,COptGuardNonC1Test0Report &lngrep);
   static void       MinBLEICOptGuardNonC1Test1Results(CMinBLEICStateShell &state,COptGuardNonC1Test1Report &strrep,COptGuardNonC1Test1Report &lngrep);
   static void       MinBLEICResults(CMinBLEICStateShell &state,double &x[],CMinBLEICReportShell &rep);
   static void       MinBLEICResultsBuf(CMinBLEICStateShell &state,double &x[],CMinBLEICReportShell &rep);
   static void       MinBLEICRestartFrom(CMinBLEICStateShell &state,double &x[]);
   static void       MinBLEICRequestTermination(CMinBLEICStateShell &state);
   //--- limited memory BFGS method for large scale optimization
   static void       MinLBFGSCreate(const int n,const int m,double &x[],CMinLBFGSStateShell &state);
   static void       MinLBFGSCreate(const int m,double &x[],CMinLBFGSStateShell &state);
   static void       MinLBFGSCreateF(const int n,const int m,double &x[],const double diffstep,CMinLBFGSStateShell &state);
   static void       MinLBFGSCreateF(const int m,double &x[],const double diffstep,CMinLBFGSStateShell &state);
   static void       MinLBFGSSetCond(CMinLBFGSStateShell &state,const double epsg,const double epsf,const double epsx,const int maxits);
   static void       MinLBFGSSetXRep(CMinLBFGSStateShell &state,const bool needxrep);
   static void       MinLBFGSSetStpMax(CMinLBFGSStateShell &state,const double stpmax);
   static void       MinLBFGSSetScale(CMinLBFGSStateShell &state,double &s[]);
   static void       MinLBFGSSetPrecDefault(CMinLBFGSStateShell &state);
   static void       MinLBFGSSetPrecCholesky(CMinLBFGSStateShell &state,CMatrixDouble &p,const bool IsUpper);
   static void       MinLBFGSSetPrecDiag(CMinLBFGSStateShell &state,double &d[]);
   static void       MinLBFGSSetPrecScale(CMinLBFGSStateShell &state);
   static bool       MinLBFGSIteration(CMinLBFGSStateShell &state);
   static void       MinLBFGSOptimize(CMinLBFGSStateShell &state,CNDimensional_Func &func,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinLBFGSOptimize(CMinLBFGSStateShell &state,CNDimensional_Grad &grad,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinLBFGSResults(CMinLBFGSStateShell &state,double &x[],CMinLBFGSReportShell &rep);
   static void       MinLBFGSresultsbuf(CMinLBFGSStateShell &state,double &x[],CMinLBFGSReportShell &rep);
   static void       MinLBFGSRestartFrom(CMinLBFGSStateShell &state,double &x[]);
   //--- constrained quadratic programming
   static void       MinQPCreate(const int n,CMinQPStateShell &state);
   static void       MinQPSetLinearTerm(CMinQPStateShell &state,double &b[]);
   static void       MinQPSetQuadraticTerm(CMinQPStateShell &state,CMatrixDouble &a,const bool IsUpper);
   static void       MinQPSetQuadraticTerm(CMinQPStateShell &state,CMatrixDouble &a);
   static void       MinQPSetQuadraticTermSparse(CMinQPStateShell &state,CSparseMatrix &a,bool IsUpper);
   static void       MinQPSetStartingPoint(CMinQPStateShell &state,double &x[]);
   static void       MinQPSetOrigin(CMinQPStateShell &state,double &xorigin[]);
   static void       MinQPSetScale(CMinQPStateShell &state,CRowDouble &s);
   static void       MinQPSetScaleAutoDiag(CMinQPStateShell &state);
   static void       MinQPSetAlgoBLEIC(CMinQPStateShell &state,double epsg,double epsf,double epsx,int maxits);
   static void       MinQPSetAlgoDenseAUL(CMinQPStateShell &state,double epsx,double rho,int itscnt);
   static void       MinQPSetAlgoDenseIPM(CMinQPStateShell &state,double eps);
   static void       MinQPSetAlgoSparseIPM(CMinQPStateShell &state,double eps);
   static void       MinQPSetAlgoQuickQP(CMinQPStateShell &state,double epsg,double epsf,double epsx,int maxouterits,bool usenewton);
   static void       MinQPSetBCAll(CMinQPStateShell &state,double bndl,double bndu);
   static void       MinQPSetAlgoCholesky(CMinQPStateShell &state);
   static void       MinQPSetBC(CMinQPStateShell &state,double &bndl[],double &bndu[]);
   static void       MinQPSetBCI(CMinQPStateShell &state,int i,double bndl,double bndu);
   static void       MinQPSetLC(CMinQPStateShell &state,CMatrixDouble &c,CRowInt &ct,int k);
   static void       MinQPSetLC(CMinQPStateShell &state,CMatrixDouble &c,CRowInt &ct);
   static void       MinQPSetLCSparse(CMinQPStateShell &state,CSparseMatrix &c,CRowInt &ct,int k);
   static void       MinQPSetLCMixed(CMinQPStateShell &state,CSparseMatrix &sparsec,CRowInt &sparsect,int sparsek,CMatrixDouble &densec,CRowInt &densect,int densek);
   static void       MinQPSetLCMixedLegacy(CMinQPStateShell &state,CMatrixDouble &densec,CRowInt &densect,int densek,CSparseMatrix &sparsec,CRowInt &sparsect,int sparsek);
   static void       MinQPSetLC2Dense(CMinQPStateShell &state,CMatrixDouble &a,CRowDouble &al,CRowDouble &au,int k);
   static void       MinQPSetLC2Dense(CMinQPStateShell &state,CMatrixDouble &a,CRowDouble &al,CRowDouble &au);
   static void       MinQPSetLC2(CMinQPStateShell &state,CSparseMatrix &a,CRowDouble &al,CRowDouble &au,int k);
   static void       MinQPSetLC2Mixed(CMinQPStateShell &state,CSparseMatrix &sparsea,int ksparse,CMatrixDouble &densea,int kdense,CRowDouble &al,CRowDouble &au);
   static void       MinQPAddLC2Dense(CMinQPStateShell &state,CRowDouble &a,double al,double au);
   static void       MinQPAddLC2(CMinQPStateShell &state,CRowInt &idxa,CRowDouble &vala,int nnz,double al,double au);
   static void       MinQPAddLC2SparseFromDense(CMinQPStateShell &state,CRowDouble &da,double al,double au);
   static void       MinQPOptimize(CMinQPStateShell &state);
   static void       MinQPResults(CMinQPStateShell &state,double &x[],CMinQPReportShell &rep);
   static void       MinQPResultsBuf(CMinQPStateShell &state,double &x[],CMinQPReportShell &rep);
   //--- Levenberg-Marquardt method
   static void       MinLMCreateVJ(const int n,const int m,double &x[],CMinLMStateShell &state);
   static void       MinLMCreateVJ(const int m,double &x[],CMinLMStateShell &state);
   static void       MinLMCreateV(const int n,const int m,double &x[],double diffstep,CMinLMStateShell &state);
   static void       MinLMCreateV(const int m,double &x[],const double diffstep,CMinLMStateShell &state);
   static void       MinLMCreateFGH(const int n,double &x[],CMinLMStateShell &state);
   static void       MinLMCreateFGH(double &x[],CMinLMStateShell &state);
   static void       MinLMSetCond(CMinLMStateShell &state,const double epsx,const int maxits);
   static void       MinLMSetXRep(CMinLMStateShell &state,const bool needxrep);
   static void       MinLMSetStpMax(CMinLMStateShell &state,const double stpmax);
   static void       MinLMSetScale(CMinLMStateShell &state,double &s[]);
   static void       MinLMSetBC(CMinLMStateShell &state,double &bndl[],double &bndu[]);
   static void       MinLMSetAccType(CMinLMStateShell &state,const int acctype);
   static bool       MinLMIteration(CMinLMStateShell &state);
   static void       MinLMOptimize(CMinLMStateShell &state,CNDimensional_FVec &fvec,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinLMOptimize(CMinLMStateShell &state,CNDimensional_FVec &fvec,CNDimensional_Jac &jac,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinLMOptimize(CMinLMStateShell &state,CNDimensional_Func &func,CNDimensional_Grad &grad,CNDimensional_Hess &hess,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinLMOptimize(CMinLMStateShell &state,CNDimensional_Func &func,CNDimensional_Jac &jac,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinLMOptimize(CMinLMStateShell &state,CNDimensional_Func &func,CNDimensional_Grad &grad,CNDimensional_Jac &jac,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinLMResults(CMinLMStateShell &state,double &x[],CMinLMReportShell &rep);
   static void       MinLMResultsBuf(CMinLMStateShell &state,double &x[],CMinLMReportShell &rep);
   static void       MinLMRestartFrom(CMinLMStateShell &state,double &x[]);
   static void       MinLMCreateVGJ(const int n,const int m,double &x[],CMinLMStateShell &state);
   static void       MinLMCreateVGJ(const int m,double &x[],CMinLMStateShell &state);
   static void       MinLMCreateFGJ(const int n,const int m,double &x[],CMinLMStateShell &state);
   static void       MinLMCreateFGJ(const int m,double &x[],CMinLMStateShell &state);
   static void       MinLMCreateFJ(const int n,const int m,double &x[],CMinLMStateShell &state);
   static void       MinLMCreateFJ(const int m,double &x[],CMinLMStateShell &state);
   //--- linear programming
   static void       MinLPCreate(int n,CMinLPState &state);
   static void       MinLPSetAlgoDSS(CMinLPState &state,double eps);
   static void       MinLPSetAlgoIPM(CMinLPState &state,double eps=0);
   static void       MinLPSetCost(CMinLPState &state,CRowDouble &c);
   static void       MinLPSetScale(CMinLPState &state,CRowDouble &s);
   static void       MinLPSetBC(CMinLPState &state,CRowDouble &bndl,CRowDouble &bndu);
   static void       MinLPSetBCAll(CMinLPState &state,double bndl,double bndu);
   static void       MinLPSetBCi(CMinLPState &state,int i,double bndl,double bndu);
   static void       MinLPSetLC(CMinLPState &state,CMatrixDouble &a,CRowInt &ct,int k);
   static void       MinLPSetLC(CMinLPState &state,CMatrixDouble &a,CRowInt &ct);
   static void       MinLPSetLC2Dense(CMinLPState &state,CMatrixDouble &a,CRowDouble &al,CRowDouble &au,int k);
   static void       MinLPSetLC2Dense(CMinLPState &state,CMatrixDouble &a,CRowDouble &al,CRowDouble &au);
   static void       MinLPSetLC2(CMinLPState &state,CSparseMatrix &a,CRowDouble &al,CRowDouble &au,int k);
   static void       MinLPAddLC2Dense(CMinLPState &state,CRowDouble &a,double al,double au);
   static void       MinLPAddLC2(CMinLPState &state,CRowInt &idxa,CRowDouble &vala,int nnz,double al,double au);
   static void       MinLPOptimize(CMinLPState &state);
   static void       MinLPResults(CMinLPState &state,CRowDouble &x,CMinLPReport &rep);
   static void       MinLPResultsBuf(CMinLPState &state,CRowDouble &x,CMinLPReport &rep);
   //--- non-linear  constrained optimization
   static void       MinNLCCreate(int n,CRowDouble &x,CMinNLCState &state);
   static void       MinNLCCreate(CRowDouble &x,CMinNLCState &state);
   static void       MinNLCCreateF(int n,CRowDouble &x,double diffstep,CMinNLCState &state);
   static void       MinNLCCreateF(CRowDouble &x,double diffstep,CMinNLCState &state);
   static void       MinNLCSetBC(CMinNLCState &state,CRowDouble &bndl,CRowDouble &bndu);
   static void       MinNLCSetLC(CMinNLCState &state,CMatrixDouble &c,CRowInt &ct,int k);
   static void       MinNLCSetLC(CMinNLCState &state,CMatrixDouble &c,CRowInt &ct);
   static void       MinNLCSetNLC(CMinNLCState &state,int nlec,int nlic);
   static void       MinNLCSetCond(CMinNLCState &state,double epsx,int maxits);
   static void       MinNLCSetScale(CMinNLCState &state,CRowDouble &s);
   static void       MinNLCSetPrecInexact(CMinNLCState &state);
   static void       MinNLCSetPrecExactLowRank(CMinNLCState &state,int updatefreq);
   static void       MinNLCSetPrecExactRobust(CMinNLCState &state,int updatefreq);
   static void       MinNLCSetPrecNone(CMinNLCState &state);
   static void       MinNLCSetSTPMax(CMinNLCState &state,double stpmax);
   static void       MinNLCSetAlgoAUL(CMinNLCState &state,double rho,int itscnt);
   static void       MinNLCSetAlgoSLP(CMinNLCState &state);
   static void       MinNLCSetAlgoSQP(CMinNLCState &state);
   static void       MinNLCSetXRep(CMinNLCState &state,bool needxrep);
   static bool       MinNLCIteration(CMinNLCState &state);
   static void       MinNLCOptimize(CMinNLCState &state,CNDimensional_FVec &fvec,CNDimensional_Rep &rep,CObject &obj);
   static void       MinNLCOptimize(CMinNLCState &state,CNDimensional_Jac &jac,CNDimensional_Rep &rep,CObject &obj);
   static void       MinNLCOptGuardGradient(CMinNLCState &state,double teststep);
   static void       MinNLCOptGuardSmoothness(CMinNLCState &state,int level=1);
   static void       MinNLCOptGuardResults(CMinNLCState &state,COptGuardReport &rep);
   static void       MinNLCOptGuardNonC1Test0Results(CMinNLCState &state,COptGuardNonC1Test0Report &strrep,COptGuardNonC1Test0Report &lngrep);
   static void       MinNLCOptGuardNonC1Test1Results(CMinNLCState &state,COptGuardNonC1Test1Report &strrep,COptGuardNonC1Test1Report &lngrep);
   static void       MinNLCResults(CMinNLCState &state,CRowDouble &x,CMinNLCReport &rep);
   static void       MinNLCResultsBuf(CMinNLCState &state,CRowDouble &x,CMinNLCReport &rep);
   static void       MinNLCRequestTermination(CMinNLCState &state);
   static void       MinNLCRestartFrom(CMinNLCState &state,CRowDouble &x);
   //--- non-smooth non-convex optimization
   static void       MinNSCreate(int n,CRowDouble &x,CMinNSState &state);
   static void       MinNSCreate(CRowDouble &x,CMinNSState &state);
   static void       MinNSCreateF(int n,CRowDouble &x,double diffstep,CMinNSState &state);
   static void       MinNSCreateF(CRowDouble &x,double diffstep,CMinNSState &state);
   static void       MinNSSetBC(CMinNSState &state,CRowDouble &bndl,CRowDouble &bndu);
   static void       MinNSSetLC(CMinNSState &state,CMatrixDouble &c,CRowInt &ct,int k);
   static void       MinNSSetLC(CMinNSState &state,CMatrixDouble &c,CRowInt &ct);
   static void       MinNSSetNLC(CMinNSState &state,int nlec,int nlic);
   static void       MinNSSetCond(CMinNSState &state,double epsx,int maxits);
   static void       MinNSSetScale(CMinNSState &state,CRowDouble &s);
   static void       MinNSSetAlgoAGS(CMinNSState &state,double radius,double penalty);
   static void       MinNSSetXRep(CMinNSState &state,bool needxrep);
   static void       MinNSRequestTermination(CMinNSState &state);
   static bool       MinNSIteration(CMinNSState &state);
   static void       MinNSOptimize(CMinNSState &state,CNDimensional_FVec &fvec,CNDimensional_Rep &rep,CObject &obj);
   static void       MinNSOptimize(CMinNSState &state,CNDimensional_Jac &jac,CNDimensional_Rep &rep,CObject &obj);
   static void       MinNSResults(CMinNSState &state,CRowDouble &x,CMinNSReport &rep);
   static void       MinNSResultsBuf(CMinNSState &state,CRowDouble &x,CMinNSReport &rep);
   static void       MinNSRestartFrom(CMinNSState &state,CRowDouble &x);
   //---box constrained optimization
   static void       MinBCCreate(int n,CRowDouble &x,CMinBCState &state);
   static void       MinBCCreate(CRowDouble &x,CMinBCState &state);
   static void       MinBCCreateF(int n,CRowDouble &x,double diffstep,CMinBCState &state);
   static void       MinBCCreateF(CRowDouble &x,double diffstep,CMinBCState &state);
   static void       MinBCSetBC(CMinBCState &state,CRowDouble &bndl,CRowDouble &bndu);
   static void       MinBCSetCond(CMinBCState &state,double epsg,double epsf,double epsx,int maxits);
   static void       MinBCSetScale(CMinBCState &state,CRowDouble &s);
   static void       MinBCSetPrecDefault(CMinBCState &state);
   static void       MinBCSetPrecDiag(CMinBCState &state,CRowDouble &d);
   static void       MinBCSetPrecScale(CMinBCState &state);
   static void       MinBCSetXRep(CMinBCState &state,bool needxrep);
   static void       MinBCSetStpMax(CMinBCState &state,double stpmax);
   static bool       MinBCIteration(CMinBCState &state);
   static void       MinBCOptimize(CMinBCState &state,CNDimensional_Func &func,CNDimensional_Rep &rep,CObject &obj);
   static void       MinBCOptimize(CMinBCState &state,CNDimensional_Grad &grad,CNDimensional_Rep &rep,CObject &obj);
   static void       MinBCOptGuardGradient(CMinBCState &state,double teststep);
   static void       MinBCOptGuardSmoothness(CMinBCState &state,int level=1);
   static void       MinBCOptGuardResults(CMinBCState &state,COptGuardReport &rep);
   static void       MinBCOptGuardNonC1Test0Results(CMinBCState &state,COptGuardNonC1Test0Report &strrep,COptGuardNonC1Test0Report &lngrep);
   static void       MinBCOptGuardNonC1Test1Results(CMinBCState &state,COptGuardNonC1Test1Report &strrep,COptGuardNonC1Test1Report &lngrep);
   static void       MinBCResults(CMinBCState &state,CRowDouble &x,CMinBCReport &rep);
   static void       MinBCResultsBuf(CMinBCState &state,CRowDouble &x,CMinBCReport &rep);
   static void       MinBCRestartFrom(CMinBCState &state,CRowDouble &x);
   static void       MinBCRequestTermination(CMinBCState &state);
   //--- optimization
   static void       MinLBFGSSetDefaultPreconditioner(CMinLBFGSStateShell &state);
   static void       MinLBFGSSetCholeskyPreconditioner(CMinLBFGSStateShell &state,CMatrixDouble &p,bool IsUpper);
   static void       MinBLEICSetBarrierWidth(CMinBLEICStateShell &state,const double mu);
   static void       MinBLEICSetBarrierDecay(CMinBLEICStateShell &state,const double mudecay);
   static void       MinASACreate(const int n,double &x[],double &bndl[],double &bndu[],CMinASAStateShell &state);
   static void       MinASACreate(double &x[],double &bndl[],double &bndu[],CMinASAStateShell &state);
   static void       MinASASetCond(CMinASAStateShell &state,const double epsg,const double epsf,const double epsx,const int maxits);
   static void       MinASASetXRep(CMinASAStateShell &state,const bool needxrep);
   static void       MinASASetAlgorithm(CMinASAStateShell &state,const int algotype);
   static void       MinASASetStpMax(CMinASAStateShell &state,const double stpmax);
   static bool       MinASAIteration(CMinASAStateShell &state);
   static void       MinASAOptimize(CMinASAStateShell &state,CNDimensional_Grad &grad,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       MinASAResults(CMinASAStateShell &state,double &x[],CMinASAReportShell &rep);
   static void       MinASAResultsBuf(CMinASAStateShell &state,double &x[],CMinASAReportShell &rep);
   static void       MinASARestartFrom(CMinASAStateShell &state,double &x[],double &bndl[],double &bndu[]);
   //--- functions of package solvers
   //--- polynomial root finding
   static void       PolynomialSolve(CRowDouble &a,int n,CRowComplex &x,CPolynomialSolverReport &rep);
   //--- dense solver
   static void       RMatrixSolve(CMatrixDouble &a,const int n,double &b[],int &info,CDenseSolverReportShell &rep,double &x[]);
   static void       RMatrixSolveM(CMatrixDouble &a,const int n,CMatrixDouble &b,const int m,const bool rfs,int &info,CDenseSolverReportShell &rep,CMatrixDouble &x);
   static void       RMatrixLUSolve(CMatrixDouble &lua,int &p[],const int n,double &b[],int &info,CDenseSolverReportShell &rep,double &x[]);
   static void       RMatrixLUSolveM(CMatrixDouble &lua,int &p[],const int n,CMatrixDouble &b,const int m,int &info,CDenseSolverReportShell &rep,CMatrixDouble &x);
   static void       RMatrixMixedSolve(CMatrixDouble &a,CMatrixDouble &lua,int &p[],const int n,double &b[],int &info,CDenseSolverReportShell &rep,double &x[]);
   static void       RMatrixMixedSolveM(CMatrixDouble &a,CMatrixDouble &lua,int &p[],const int n,CMatrixDouble &b,const int m,int &info,CDenseSolverReportShell &rep,CMatrixDouble &x);
   static void       CMatrixSolveM(CMatrixComplex &a,const int n,CMatrixComplex &b,const int m,const bool rfs,int &info,CDenseSolverReportShell &rep,CMatrixComplex &x);
   static void       CMatrixSolve(CMatrixComplex &a,const int n,complex &b[],int &info,CDenseSolverReportShell &rep,complex &x[]);
   static void       CMatrixLUSolveM(CMatrixComplex &lua,int &p[],const int n,CMatrixComplex &b,const int m,int &info,CDenseSolverReportShell &rep,CMatrixComplex &x);
   static void       CMatrixLUSolve(CMatrixComplex &lua,int &p[],const int n,complex &b[],int &info,CDenseSolverReportShell &rep,complex &x[]);
   static void       CMatrixMixedSolveM(CMatrixComplex &a,CMatrixComplex &lua,int &p[],const int n,CMatrixComplex &b,const int m,int &info,CDenseSolverReportShell &rep,CMatrixComplex &x);
   static void       CMatrixMixedSolve(CMatrixComplex &a,CMatrixComplex &lua,int &p[],const int n,complex &b[],int &info,CDenseSolverReportShell &rep,complex &x[]);
   static void       SPDMatrixSolveM(CMatrixDouble &a,const int n,const bool IsUpper,CMatrixDouble &b,const int m,int &info,CDenseSolverReportShell &rep,CMatrixDouble &x);
   static void       SPDMatrixSolve(CMatrixDouble &a,const int n,const bool IsUpper,double &b[],int &info,CDenseSolverReportShell &rep,double &x[]);
   static void       SPDMatrixCholeskySolveM(CMatrixDouble &cha,const int n,const bool IsUpper,CMatrixDouble &b,const int m,int &info,CDenseSolverReportShell &rep,CMatrixDouble &x);
   static void       SPDMatrixCholeskySolve(CMatrixDouble &cha,const int n,const bool IsUpper,double &b[],int &info,CDenseSolverReportShell &rep,double &x[]);
   static void       HPDMatrixSolveM(CMatrixComplex &a,const int n,const bool IsUpper,CMatrixComplex &b,const int m,int &info,CDenseSolverReportShell &rep,CMatrixComplex &x);
   static void       HPDMatrixSolve(CMatrixComplex &a,const int n,const bool IsUpper,complex &b[],int &info,CDenseSolverReportShell &rep,complex &x[]);
   static void       HPDMatrixCholeskySolveM(CMatrixComplex &cha,const int n,const bool IsUpper,CMatrixComplex &b,const int m,int &info,CDenseSolverReportShell &rep,CMatrixComplex &x);
   static void       HPDMatrixCholeskySolve(CMatrixComplex &cha,const int n,const bool IsUpper,complex &b[],int &info,CDenseSolverReportShell &rep,complex &x[]);
   static void       RMatrixSolveLS(CMatrixDouble &a,const int nrows,const int ncols,double &b[],const double threshold,int &info,CDenseSolverLSReportShell &rep,double &x[]);
   //--- sparse linear solver
   static void       SparseSPDSolveSKS(CSparseMatrix &a,bool IsUpper,CRowDouble &b,CRowDouble &x,CSparseSolverReport &rep);
   static void       SparseSPDSolve(CSparseMatrix &a,bool IsUpper,CRowDouble &b,CRowDouble &x,CSparseSolverReport &rep);
   static void       SparseSPDCholeskySolve(CSparseMatrix &a,bool IsUpper,CRowDouble &b,CRowDouble &x,CSparseSolverReport &rep);
   static void       SparseSolve(CSparseMatrix &a,CRowDouble &b,CRowDouble &x,CSparseSolverReport &rep);
   static void       SparseLUSolve(CSparseMatrix &a,CRowInt &p,CRowInt &q,CRowDouble &b,CRowDouble &x,CSparseSolverReport &rep);
   //--- sparse symmetric linear solver
   static void       SparseSolveSymmetricGMRES(CSparseMatrix &a,bool IsUpper,CRowDouble &b,int k,double epsf,int maxits,CRowDouble &x,CSparseSolverReport &rep);
   static void       SparseSolveGMRES(CSparseMatrix &a,CRowDouble &b,int k,double epsf,int maxits,CRowDouble &x,CSparseSolverReport &rep);
   static void       SparseSolverCreate(int n,CSparseSolverState &state);
   static void       SparseSolverSetAlgoGMRES(CSparseSolverState &state,int k);
   static void       SparseSolverSetStartingPoint(CSparseSolverState &state,CRowDouble &x);
   static void       SparseSolverSetCond(CSparseSolverState &state,double epsf,int maxits);
   static void       SparseSolverSolveSymmetric(CSparseSolverState &state,CSparseMatrix &a,bool IsUpper,CRowDouble &b);
   static void       SparseSolverSolve(CSparseSolverState &state,CSparseMatrix &a,CRowDouble &b);
   static void       SparseSolverResults(CSparseSolverState &state,CRowDouble &x,CSparseSolverReport &rep);
   static void       SparseSolverSetXRep(CSparseSolverState &state,bool needxrep);
   static void       SparseSolverOOCStart(CSparseSolverState &state,CRowDouble &b);
   static bool       SparseSolverOOCContinue(CSparseSolverState &state);
   static void       SparseSolverOOCGetRequestInfo(CSparseSolverState &state,int &requesttype);
   static void       SparseSolverOOCGetRequestData(CSparseSolverState &state,CRowDouble &x);
   static void       SparseSolverOOCGetRequestData1(CSparseSolverState &state,double &v);
   static void       SparseSolverOOCSendResult(CSparseSolverState &state,CRowDouble &ax);
   static void       SparseSolverOOCStop(CSparseSolverState &state,CRowDouble &x,CSparseSolverReport &rep);
   static void       SparseSolverRequestTermination(CSparseSolverState &state);
   //--- linear CG Solver
   static void       LinCGCreate(int n,CLinCGState &state);
   static void       LinCGSetStartingPoint(CLinCGState &state,CRowDouble &x);
   static void       LinCGSetPrecUnit(CLinCGState &state);
   static void       LinCGSetPrecDiag(CLinCGState &state);
   static void       LinCGSetCond(CLinCGState &state,double epsf,int maxits);
   static void       LinCGSolveSparse(CLinCGState &state,CSparseMatrix &a,bool IsUpper,CRowDouble &b);
   static void       LinCGResult(CLinCGState &state,CRowDouble &x,CLinCGReport &rep);
   static void       LinCGSetRestartFreq(CLinCGState &state,int srf);
   static void       LinCGSetRUpdateFreq(CLinCGState &state,int freq);
   static void       LinCGSetXRep(CLinCGState &state,bool needxrep);
   //--- linear LSQR Solver
   static void       LinLSQRCreate(int m,int n,CLinLSQRState &state);
   static void       LinLSQRCreateBuf(int m,int n,CLinLSQRState &state);
   static void       LinLSQRSetPrecUnit(CLinLSQRState &state);
   static void       LinLSQRSetPrecDiag(CLinLSQRState &state);
   static void       LinLSQRSetLambdaI(CLinLSQRState &state,double lambdai);
   static void       LinLSQRSolveSparse(CLinLSQRState &state,CSparseMatrix &a,CRowDouble &b);
   static void       LinLSQRSetCond(CLinLSQRState &state,double epsa,double epsb,int maxits);
   static void       LinLSQRResults(CLinLSQRState &state,CRowDouble &x,CLinLSQRReport &rep);
   static void       LinLSQRSetXRep(CLinLSQRState &state,bool needxrep);
   static int        LinLSQRPeekIterationsCount(CLinLSQRState &s);
   static void       LinLSQRRequestTermination(CLinLSQRState &state);
   //--- solving systems of nonlinear equations
   static void       NlEqCreateLM(const int n,const int m,double &x[],CNlEqStateShell &state);
   static void       NlEqCreateLM(const int m,double &x[],CNlEqStateShell &state);
   static void       NlEqSetCond(CNlEqStateShell &state,const double epsf,const int maxits);
   static void       NlEqSetXRep(CNlEqStateShell &state,const bool needxrep);
   static void       NlEqSetStpMax(CNlEqStateShell &state,const double stpmax);
   static bool       NlEqIteration(CNlEqStateShell &state);
   static void       NlEqSolve(CNlEqStateShell &state,CNDimensional_Func &func,CNDimensional_Jac &jac,CNDimensional_Rep &rep,bool rep_status,CObject &obj);
   static void       NlEqResults(CNlEqStateShell &state,double &x[],CNlEqReportShell &rep);
   static void       NlEqResultsBuf(CNlEqStateShell &state,double &x[],CNlEqReportShell &rep);
   static void       NlEqRestartFrom(CNlEqStateShell &state,double &x[]);
   //--- functions of package specialfunctions
   //--- gamma function
   static double     GammaFunction(const double x);
   static double     LnGamma(const double x,double &sgngam);
   //--- normal distribution
   static double     ErrorFunction(const double x);
   static double     ErrorFunctionC(const double x);
   static double     NormalDistribution(const double x);
   static double     NormalPDF(const double x);
   static double     NormalCDF(const double x);
   static double     InvErF(const double e);
   static double     InvNormalDistribution(double y0);
   static double     InvNormalCDF(const double y0);
   static double     BivariateNormalPDF(const double x,const double y,const double rho);
   static double     BivariateNormalCDF(double x,double y,const double rho);
   
   //--- incomplete gamma function
   static double     IncompleteGamma(const double a,const double x);
   static double     IncompleteGammaC(const double a,const double x);
   static double     InvIncompleteGammaC(const double a,const double y0);
   //--- airy function
   static void       Airy(const double x,double &ai,double &aip,double &bi,double &bip);
   //--- Bessel function
   static double     BesselJ0(const double x);
   static double     BesselJ1(const double x);
   static double     BesselJN(const int n,const double x);
   static double     BesselY0(const double x);
   static double     BesselY1(const double x);
   static double     BesselYN(const int n,const double x);
   static double     BesselI0(const double x);
   static double     BesselI1(const double x);
   static double     BesselK0(const double x);
   static double     BesselK1(const double x);
   static double     BesselKN(const int nn,const double x);
   //--- beta function
   static double     Beta(const double a,const double b);
   static double     IncompleteBeta(const double a,const double b,const double x);
   static double     InvIncompleteBeta(const double a,const double b,double y);
   //--- binomial distribution
   static double     BinomialDistribution(const int k,const int n,const double p);
   static double     BinomialComplDistribution(const int k,const int n,const double p);
   static double     InvBinomialDistribution(const int k,const int n,const double y);
   //--- Chebyshev polynom
   static double     ChebyshevCalculate(int r,const int n,const double x);
   static double     ChebyshevSum(double &c[],const int r,const int n,const double x);
   static void       ChebyshevCoefficients(const int n,double &c[]);
   static void       FromChebyshev(double &a[],const int n,double &b[]);
   //--- chi-square distribution
   static double     ChiSquareDistribution(const double v,const double x);
   static double     ChiSquareComplDistribution(const double v,const double x);
   static double     InvChiSquareDistribution(const double v,const double y);
   //--- Dawson's Integral
   static double     DawsonIntegral(const double x);
   //--- elliptic integral
   static double     EllipticIntegralK(const double m);
   static double     EllipticIntegralKhighPrecision(const double m1);
   static double     IncompleteEllipticIntegralK(const double phi,const double m);
   static double     EllipticIntegralE(const double m);
   static double     IncompleteEllipticIntegralE(const double phi,const double m);
   //--- exponential integral
   static double     ExponentialIntegralEi(const double x);
   static double     ExponentialIntegralEn(const double x,const int n);
   //--- F distribution functions
   static double     FDistribution(const int a,const int b,const double x);
   static double     FComplDistribution(const int a,const int b,const double x);
   static double     InvFDistribution(const int a,const int b,const double y);
   //--- Fresnel integral
   static void       FresnelIntegral(const double x,double &c,double &s);
   //--- Hermite polynomial
   static double     HermiteCalculate(const int n,const double x);
   static double     HermiteSum(double &c[],const int n,const double x);
   static void       HermiteCoefficients(const int n,double &c[]);
   //--- Jacobian elliptic functions
   static void       JacobianEllipticFunctions(const double u,const double m,double &sn,double &cn,double &dn,double &ph);
   //--- Laguerre polynomial
   static double     LaguerreCalculate(const int n,const double x);
   static double     LaguerreSum(double &c[],const int n,const double x);
   static void       LaguerreCoefficients(const int n,double &c[]);
   //--- Legendre polynomial
   static double     LegendreCalculate(const int n,const double x);
   static double     LegendreSum(double &c[],const int n,const double x);
   static void       LegendreCoefficients(const int n,double &c[]);
   //--- Poisson distribution
   static double     PoissonDistribution(const int k,const double m);
   static double     PoissonComplDistribution(const int k,const double m);
   static double     InvPoissonDistribution(const int k,const double y);
   //--- psi function
   static double     Psi(const double x);
   //--- Student's t distribution
   static double     StudenttDistribution(const int k,const double t);
   static double     InvStudenttDistribution(const int k,const double p);
   //--- trigonometric integrals
   static void       SineCosineIntegrals(const double x,double &si,double &ci);
   static void       HyperbolicSineCosineIntegrals(const double x,double &shi,double &chi);
   //--- functions of package statistics
   //--- basic statistics methods
   static void       SampleMoments(const double &x[],const int n,double &mean,double &variance,double &skewness,double &kurtosis);
   static void       SampleMoments(const double &x[],double &mean,double &variance,double &skewness,double &kurtosis);
   static double     SampleMean(CRowDouble &x,int n);
   static double     SampleMean(CRowDouble &x);
   static double     SampleVariance(CRowDouble &x,int n);
   static double     SampleVariance(CRowDouble &x);
   static double     SampleSkewness(CRowDouble &x,int n);
   static double     SampleSkewness(CRowDouble &x);
   static double     SampleKurtosis(CRowDouble &x,int n);
   static double     SampleKurtosis(CRowDouble &x);
   static void       SampleAdev(const double &x[],const int n,double &adev);
   static void       SampleAdev(const double &x[],double &adev);
   static void       SampleMedian(const double &x[],const int n,double &median);
   static void       SampleMedian(const double &x[],double &median);
   static void       SamplePercentile(const double &x[],const int n,const double p,double &v);
   static void       SamplePercentile(const double &x[],const double p,double &v);
   static double     Cov2(const double &x[],const double &y[],const int n);
   static double     Cov2(const double &x[],const double &y[]);
   static double     PearsonCorr2(const double &x[],const double &y[],const int n);
   static double     PearsonCorr2(const double &x[],const double &y[]);
   static double     SpearmanCorr2(const double &x[],const double &y[],const int n);
   static double     SpearmanCorr2(const double &x[],const double &y[]);
   static void       CovM(const CMatrixDouble &x,const int n,const int m,CMatrixDouble &c);
   static void       CovM(const CMatrixDouble &x,CMatrixDouble &c);
   static void       PearsonCorrM(const CMatrixDouble &x,const int n,const int m,CMatrixDouble &c);
   static void       PearsonCorrM(CMatrixDouble &x,CMatrixDouble &c);
   static void       SpearmanCorrM(const CMatrixDouble &x,const int n,const int m,CMatrixDouble &c);
   static void       SpearmanCorrM(const CMatrixDouble &x,CMatrixDouble &c);
   static void       CovM2(const CMatrixDouble &x,const CMatrixDouble &y,const int n,const int m1,const int m2,CMatrixDouble &c);
   static void       CovM2(const CMatrixDouble &x,const CMatrixDouble &y,CMatrixDouble &c);
   static void       PearsonCorrM2(const CMatrixDouble &x,const CMatrixDouble &y,const int n,const int m1,const int m2,CMatrixDouble &c);
   static void       PearsonCorrM2(const CMatrixDouble &x,const CMatrixDouble &y,CMatrixDouble &c);
   static void       SpearmanCorrM2(const CMatrixDouble &x,const CMatrixDouble &y,const int n,const int m1,const int m2,CMatrixDouble &c);
   static void       SpearmanCorrM2(const CMatrixDouble &x,const CMatrixDouble &y,CMatrixDouble &c);
   static void       RankData(CMatrixDouble &xy,int npoints,int nfeatures);
   static void       RankData(CMatrixDouble &xy);
   static void       RankDataCentered(CMatrixDouble &xy,int npoints,int nfeatures);
   static void       RankDataCentered(CMatrixDouble &xy);
   //--- correlation tests
   static void       PearsonCorrelationSignificance(const double r,const int n,double &bothTails,double &leftTail,double &rightTail);
   static void       SpearmanRankCorrelationSignificance(const double r,const int n,double &bothTails,double &leftTail,double &rightTail);
   //--- Jarque-Bera test
   static void       JarqueBeraTest(const double &x[],const int n,double &p);
   //--- Mann-Whitney U-test
   static void       MannWhitneyUTest(const double &x[],const int n,const double &y[],const int m,double &bothTails,double &leftTail,double &rightTail);
   //--- sign test
   static void       OneSampleSignTest(const double &x[],const int n,const double median,double &bothTails,double &leftTail,double &rightTail);
   //--- Student Tests
   static void       StudentTest1(const double &x[],const int n,const double mean,double &bothTails,double &leftTail,double &rightTail);
   static void       StudentTest2(const double &x[],const int n,const double &y[],const int m,double &bothTails,double &leftTail,double &rightTail);
   static void       UnequalVarianceTest(const double &x[],const int n,const double &y[],const int m,double &bothTails,double &leftTail,double &rightTail);
   //--- variance tests
   static void       FTest(const double &x[],const int n,const double &y[],const int m,double &bothTails,double &leftTail,double &rightTail);
   static void       OneSampleVarianceTest(double &x[],int n,double variance,double &bothTails,double &leftTail,double &rightTail);
   //--- Wilcoxon signed-rank test
   static void       WilcoxonSignedRankTest(const double &x[],const int n,const double e,double &bothTails,double &leftTail,double &rightTail);
  };
//+------------------------------------------------------------------+
//| HQRNDState initialization with random values which come from     |
//| standard RNG.                                                    |
//+------------------------------------------------------------------+
void CAlglib::HQRndRandomize(CHighQualityRandStateShell &state)
  {
   CHighQualityRand::HQRndRandomize(state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| HQRNDState initialization with seed values                       |
//+------------------------------------------------------------------+
void CAlglib::HQRndSeed(const int s1,const int s2,CHighQualityRandStateShell &state)
  {
   CHighQualityRand::HQRndSeed(s1,s2,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function generates random real number in (0,1),             |
//| not including interval boundaries                                |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
double CAlglib::HQRndUniformR(CHighQualityRandStateShell &state)
  {
   return(CHighQualityRand::HQRndUniformR(state.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| This function generates random integer number in [0, N)          |
//| 1. N must be less than HQRNDMax-1.                               |
//| 2. State structure must be initialized with HQRNDRandomize() or  |
//| HQRNDSeed()                                                      |
//+------------------------------------------------------------------+
int CAlglib::HQRndUniformI(CHighQualityRandStateShell &state,const int n)
  {
   return(CHighQualityRand::HQRndUniformI(state.GetInnerObj(),n));
  }
//+------------------------------------------------------------------+
//| Random number generator: normal numbers                          |
//| This function generates one random number from normal            |
//| distribution.                                                    |
//| Its performance is equal to that of HQRNDNormal2()               |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
double CAlglib::HQRndNormal(CHighQualityRandStateShell &state)
  {
   return(CHighQualityRand::HQRndNormal(state.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| Random number generator: vector with random entries (normal      |
//| distribution)                                                    |
//| This function generates N random numbers from normal             |
//| distribution.                                                    |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
void CAlglib::HQRndNormalV(CHighQualityRandStateShell &state,
                           int n,CRowDouble &x)
  {
   CHighQualityRand::HQRndNormalV(state.GetInnerObj(),n,x);
  }
//+------------------------------------------------------------------+
//| Random number generator: vector with random entries (normal      |
//| distribution)                                                    |
//| This function generates N random numbers from normal             |
//| distribution.                                                    |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
void CAlglib::HQRndNormalV(CHighQualityRandStateShell &state,
                           int n,vector<double> &x)
  {
   CRowDouble X=x;
   HQRndNormalV(state,n,X);
   x=X.ToVector();
  }
//+------------------------------------------------------------------+
//| Random number generator: matrix with random entries (normal      |
//| distribution)                                                    |
//| This function generates MxN random matrix.                       |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
void CAlglib::HQRndNormalM(CHighQualityRandStateShell &state,
                           int m,int n,CMatrixDouble &x)
  {
   CHighQualityRand::HQRndNormalM(state.GetInnerObj(),m,n,x);
  }
//+------------------------------------------------------------------+
//| Random number generator: matrix with random entries (normal      |
//| distribution)                                                    |
//| This function generates MxN random matrix.                       |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
void CAlglib::HQRndNormalM(CHighQualityRandStateShell &state,
                           int m,int n,matrix<double> &x)
  {
   CMatrixDouble X=x;
   HQRndNormalM(state,m,n,X);
   x=X.ToMatrix();
  }
//+------------------------------------------------------------------+
//| Random number generator: random X and Y such that X^2+Y^2=1      |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
void CAlglib::HQRndUnit2(CHighQualityRandStateShell &state,
                         double &x,double &y)
  {
//--- initialization
   x=0;
   y=0;
//--- function call
   CHighQualityRand::HQRndUnit2(state.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| Random number generator: normal numbers                          |
//| This function generates two independent random numbers from      |
//| normal distribution. Its performance is equal to that of         |
//| HQRNDNormal()                                                    |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
void CAlglib::HQRndNormal2(CHighQualityRandStateShell &state,
                           double &x1,double &x2)
  {
//--- initialization
   x1=0;
   x2=0;
//--- function call
   CHighQualityRand::HQRndNormal2(state.GetInnerObj(),x1,x2);
  }
//+------------------------------------------------------------------+
//| Random number generator: exponential distribution                |
//| State structure must be initialized with HQRNDRandomize() or     |
//| HQRNDSeed().                                                     |
//+------------------------------------------------------------------+
double CAlglib::HQRndExponential(CHighQualityRandStateShell &state,
                                 const double lambdav)
  {
   return(CHighQualityRand::HQRndExponential(state.GetInnerObj(),lambdav));
  }
//+------------------------------------------------------------------+
//| This function generates  random number from discrete distribution|
//| given by finite sample X.                                        |
//| INPUT PARAMETERS                                                 |
//|   State   -   high quality random number generator, must be      |
//|               initialized with HQRNDRandomize() or HQRNDSeed().  |
//|   X   -   finite sample                                          |
//|   N   -   number of elements to use, N>=1                        |
//| RESULT                                                           |
//|   this function returns one of the X[i] for random i=0..N-1      |
//+------------------------------------------------------------------+
double CAlglib::HQRndDiscrete(CHighQualityRandStateShell &state,
                              int n,CRowDouble &x)
  {
   return(CHighQualityRand::HQRndDiscrete(state.GetInnerObj(),n,x));
  }
//+------------------------------------------------------------------+
//| This function generates  random number from discrete distribution|
//| given by finite sample X.                                        |
//| INPUT PARAMETERS                                                 |
//|   State   -   high quality random number generator, must be      |
//|               initialized with HQRNDRandomize() or HQRNDSeed().  |
//|   X   -   finite sample                                          |
//|   N   -   number of elements to use, N>=1                        |
//| RESULT                                                           |
//|   this function returns one of the X[i] for random i=0..N-1      |
//+------------------------------------------------------------------+
double CAlglib::HQRndDiscrete(CHighQualityRandStateShell &state,
                              int n,vector<double> &x)
  {
   return(CHighQualityRand::HQRndDiscrete(state.GetInnerObj(),n,x));
  }
//+------------------------------------------------------------------+
//| This function generates random number from continuous            |
//| distribution  given by finite sample X.                          |
//| INPUT PARAMETERS                                                 |
//|   State   -   high quality random number generator, must be      |
//|               initialized with HQRNDRandomize() or HQRNDSeed().  |
//|   X       -   finite sample, array[N] (can be larger, in this    |
//|               case only leading N elements are used). THIS ARRAY |
//|               MUST BE SORTED BY ASCENDING.                       |
//|   N       -   number of elements to use, N>=1                    |
//| RESULT                                                           |
//|   this function returns random number from continuous            |
//|   distribution which tries to approximate X as mush as possible. |
//|   min(X)<=Result<=max(X).                                        |
//+------------------------------------------------------------------+
double CAlglib::HQRndContinuous(CHighQualityRandStateShell &state,
                                int n,CRowDouble &x)
  {
   return(CHighQualityRand::HQRndContinuous(state.GetInnerObj(),n,x));
  }
//+------------------------------------------------------------------+
//| This function generates random number from continuous            |
//| distribution  given by finite sample X.                          |
//| INPUT PARAMETERS                                                 |
//|   State   -   high quality random number generator, must be      |
//|               initialized with HQRNDRandomize() or HQRNDSeed().  |
//|   X       -   finite sample, array[N] (can be larger, in this    |
//|               case only leading N elements are used). THIS ARRAY |
//|               MUST BE SORTED BY ASCENDING.                       |
//|   N       -   number of elements to use, N>=1                    |
//| RESULT                                                           |
//|   this function returns random number from continuous            |
//|   distribution which tries to approximate X as mush as possible. |
//|   min(X)<=Result<=max(X).                                        |
//+------------------------------------------------------------------+
double CAlglib::HQRndContinuous(CHighQualityRandStateShell &state,
                                int n,vector<double> &x)
  {
   return(CHighQualityRand::HQRndContinuous(state.GetInnerObj(),n,x));
  }
//+------------------------------------------------------------------+
//| This function serializes data structure to string.               |
//| Important properties of s_out:                                   |
//| * it contains alphanumeric characters, dots, underscores, minus  |
//|   signs                                                          |
//| * these symbols are grouped into words, which are separated by   |
//|   spaces and Windows-style (CR+LF) newlines                      |
//| * although serializer uses spaces and CR+LF as separators, you   |
//|   can replace any separator character by arbitrary combination   |
//|   of spaces, tabs, Windows or Unix newlines. It allows flexible  |
//|   reformatting of the string in case you want to include it into |
//|   text or XML file. But you should not insert separators into the|
//|   middle of the "words" nor you should change case of letters.   |
//| * s_out can be freely moved between 32-bit and 64-bit systems,   |
//|   little and big endian machines, and so on. You can reference   |
//|   structure on 32-bit machine and unserialize it on 64-bit one   |
//|   (or vice versa), or reference it on SPARC and unserialize on   |
//|   x86. You can also reference it in C# version of ALGLIB and     |
//|   unserialize in C++ one, and vice versa.                        |
//+------------------------------------------------------------------+
void CAlglib::KDTreeSerialize(CKDTreeShell &obj,string &s_out)
  {
//--- create a variable
   CSerializer s;
//--- serialization start
   s.Alloc_Start();
//--- function call
   CNearestNeighbor::KDTreeAlloc(s,obj.GetInnerObj());
   s.SStart_Str();
//--- function call
   CNearestNeighbor::KDTreeSerialize(s,obj.GetInnerObj());
   s.Stop();
//--- get result
   s_out=s.Get_String();
  }
//+------------------------------------------------------------------+
//| This function unserializes data structure from string.           |
//+------------------------------------------------------------------+
void CAlglib::KDTreeUnserialize(string s_in,CKDTreeShell &obj)
  {
//--- object of class
   CSerializer s;
   s.UStart_Str(s_in);
//--- function call
   CNearestNeighbor::KDTreeUnserialize(s,obj.GetInnerObj());
   s.Stop();
  }
//+------------------------------------------------------------------+
//| KD-tree creation                                                 |
//| This subroutine creates KD-tree from set of X-values and optional|
//| Y-values                                                         |
//| INPUT PARAMETERS                                                 |
//|     XY      -   dataset,array[0..N-1, 0..NX+NY-1].               |
//|                 one row corresponds to one point.                |
//|                 first NX columns contain X-values, next NY (NY   |
//|                 may be zero)                                     |
//|                 columns may contain associated Y-values          |
//|     N       -   number of points, N>=1                           |
//|     NX      -   space dimension, NX>=1.                          |
//|     NY      -   number of optional Y-values, NY>=0.              |
//|     NormType-   norm type:                                       |
//|                 * 0 denotes infinity-norm                        |
//|                 * 1 denotes 1-norm                               |
//|                 * 2 denotes 2-norm (Euclidean norm)              |
//| OUTPUT PARAMETERS                                                |
//|     KDT     -   KD-tree                                          |
//| NOTES                                                            |
//| 1. KD-tree  creation  have O(N*logN) complexity and              |
//|    O(N*(2*NX+NY)) memory requirements.                           |
//| 2. Although KD-trees may be used with any combination of N and   |
//|    NX, they are more efficient than brute-force search only when |
//|    N >> 4^NX. So they are most useful in low-dimensional tasks   |
//|    (NX=2, NX=3). NX=1  is another inefficient case, because      |
//|    simple  binary  search  (without  additional structures) is   |
//|    much more efficient in such tasks than KD-trees.              |
//+------------------------------------------------------------------+
void CAlglib::KDTreeBuild(CMatrixDouble &xy,const int n,const int nx,
                          const int ny,const int normtype,CKDTreeShell &kdt)
  {
   CNearestNeighbor::KDTreeBuild(xy,n,nx,ny,normtype,kdt.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| KD-tree creation                                                 |
//| This subroutine creates KD-tree from set of X-values and optional|
//| Y-values                                                         |
//| INPUT PARAMETERS                                                 |
//|     XY      -   dataset,array[0..N-1, 0..NX+NY-1].               |
//|                 one row corresponds to one point.                |
//|                 first NX columns contain X-values, next NY (NY   |
//|                 may be zero)                                     |
//|                 columns may contain associated Y-values          |
//|     N       -   number of points, N>=1                           |
//|     NX      -   space dimension, NX>=1.                          |
//|     NY      -   number of optional Y-values, NY>=0.              |
//|     NormType-   norm type:                                       |
//|                 * 0 denotes infinity-norm                        |
//|                 * 1 denotes 1-norm                               |
//|                 * 2 denotes 2-norm (Euclidean norm)              |
//| OUTPUT PARAMETERS                                                |
//|     KDT     -   KD-tree                                          |
//| NOTES                                                            |
//| 1. KD-tree  creation  have O(N*logN) complexity and              |
//|    O(N*(2*NX+NY)) memory requirements.                           |
//| 2. Although KD-trees may be used with any combination of N and   |
//|    NX, they are more efficient than brute-force search only when |
//|    N >> 4^NX. So they are most useful in low-dimensional tasks   |
//|    (NX=2, NX=3). NX=1  is another inefficient case, because      |
//|    simple  binary  search  (without  additional structures) is   |
//|    much more efficient in such tasks than KD-trees.              |
//+------------------------------------------------------------------+
void CAlglib::KDTreeBuild(CMatrixDouble &xy,const int nx,const int ny,
                          const int normtype,CKDTreeShell &kdt)
  {
//--- create a variable
   int n=(int)CAp::Rows(xy);
//--- function call
   CNearestNeighbor::KDTreeBuild(xy,n,nx,ny,normtype,kdt.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| KD-tree creation                                                 |
//| This subroutine creates KD-tree from set of X-values, integer    |
//| tags and optional Y-values                                       |
//| INPUT PARAMETERS                                                 |
//|     XY      -   dataset,array[0..N-1, 0..NX+NY-1].               |
//|                 one row corresponds to one point.                |
//|                 first NX columns contain X-values, next NY (NY   |
//|                 may be zero)                                     |
//|                 columns may contain associated Y-values          |
//|     Tags    -   tags, array[0..N-1], contains integer tags       |
//|                 associated with points.                          |
//|     N       -   number of points, N>=1                           |
//|     NX      -   space dimension, NX>=1.                          |
//|     NY      -   number of optional Y-values, NY>=0.              |
//|     NormType-   norm type:                                       |
//|                 * 0 denotes infinity-norm                        |
//|                 * 1 denotes 1-norm                               |
//|                 * 2 denotes 2-norm (Euclidean norm)              |
//| OUTPUT PARAMETERS                                                |
//|     KDT     -   KD-tree                                          |
//| NOTES                                                            |
//| 1. KD-tree  creation  have O(N*logN) complexity and              |
//|    O(N*(2*NX+NY)) memory requirements.                           |
//| 2. Although KD-trees may be used with any combination of N and   |
//|    NX, they are more efficient than brute-force search only when |
//|    N >> 4^NX. So they are most useful in low-dimensional tasks   |
//|    (NX=2, NX=3). NX=1 is another inefficient case, because simple|
//|    binary search (without additional structures) is much more    |
//|    efficient in such tasks than KD-trees.                        |
//+------------------------------------------------------------------+
void CAlglib::KDTreeBuildTagged(CMatrixDouble &xy,int &tags[],
                                const int n,const int nx,
                                const int ny,const int normtype,
                                CKDTreeShell &kdt)
  {
   CNearestNeighbor::KDTreeBuildTagged(xy,tags,n,nx,ny,normtype,kdt.GetInnerObj());
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::KDTreeBuildTagged(CMatrixDouble &xy,CRowInt &tags,
                                const int n,const int nx,
                                const int ny,const int normtype,
                                CKDTreeShell &kdt)
  {
   CNearestNeighbor::KDTreeBuildTagged(xy,tags,n,nx,ny,normtype,kdt.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| KD-tree creation                                                 |
//| This subroutine creates KD-tree from set of X-values, integer    |
//| tags and optional Y-values                                       |
//| INPUT PARAMETERS                                                 |
//|     XY      -   dataset,array[0..N-1, 0..NX+NY-1].               |
//|                 one row corresponds to one point.                |
//|                 first NX columns contain X-values, next NY (NY   |
//|                 may be zero)                                     |
//|                 columns may contain associated Y-values          |
//|     Tags    -   tags, array[0..N-1], contains integer tags       |
//|                 associated with points.                          |
//|     N       -   number of points, N>=1                           |
//|     NX      -   space dimension, NX>=1.                          |
//|     NY      -   number of optional Y-values, NY>=0.              |
//|     NormType-   norm type:                                       |
//|                 * 0 denotes infinity-norm                        |
//|                 * 1 denotes 1-norm                               |
//|                 * 2 denotes 2-norm (Euclidean norm)              |
//| OUTPUT PARAMETERS                                                |
//|     KDT     -   KD-tree                                          |
//| NOTES                                                            |
//| 1. KD-tree  creation  have O(N*logN) complexity and              |
//|    O(N*(2*NX+NY)) memory requirements.                           |
//| 2. Although KD-trees may be used with any combination of N and   |
//|    NX, they are more efficient than brute-force search only when |
//|    N >> 4^NX. So they are most useful in low-dimensional tasks   |
//|    (NX=2, NX=3). NX=1 is another inefficient case, because simple|
//|    binary search (without additional structures) is much more    |
//|    efficient in such tasks than KD-trees.                        |
//+------------------------------------------------------------------+
void CAlglib::KDTreeBuildTagged(CMatrixDouble &xy,int &tags[],
                                const int nx,const int ny,
                                const int normtype,CKDTreeShell &kdt)
  {
//--- check
   if((CAp::Rows(xy)!=CAp::Len(tags)))
     {
      Print(__FUNCTION__+": looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=(int)CAp::Rows(xy);
//--- function call
   CNearestNeighbor::KDTreeBuildTagged(xy,tags,n,nx,ny,normtype,kdt.GetInnerObj());
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::KDTreeBuildTagged(CMatrixDouble &xy,CRowInt &tags,
                                const int nx,const int ny,
                                const int normtype,CKDTreeShell &kdt)
  {
//--- check
   if((CAp::Rows(xy)!=CAp::Len(tags)))
     {
      Print(__FUNCTION__+": looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=(int)CAp::Rows(xy);
//--- function call
   CNearestNeighbor::KDTreeBuildTagged(xy,tags,n,nx,ny,normtype,kdt.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| K-NN query: K nearest neighbors                                  |
//| INPUT PARAMETERS                                                 |
//|     KDT         -   KD-tree                                      |
//|     X           -   point, array[0..NX-1].                       |
//|     K           -   number of neighbors to return, K>=1          |
//|     SelfMatch   -   whether self-matches are allowed:            |
//|                     * if True, nearest neighbor may be the point |
//|                       itself (if it exists in original dataset)  |
//|                     * if False, then only points with non-zero   |
//|                       distance are returned                      |
//|                     * if not given, considered True              |
//| RESULT                                                           |
//|     number of actual neighbors found (either K or N, if K>N).    |
//| This  subroutine performs query and stores its result in the     |
//| internal structures of the KD-tree. You can use following        |
//| subroutines to obtain these results:                             |
//| * KDTreeQueryResultsX() to get X-values                          |
//| * KDTreeQueryResultsXY() to get X- and Y-values                  |
//| * KDTreeQueryResultsTags() to get tag values                     |
//| * KDTreeQueryResultsDistances() to get distances                 |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryKNN(CKDTreeShell &kdt,double &x[],
                            const int k,const bool selfmatch=true)
  {
   return(CNearestNeighbor::KDTreeQueryKNN(kdt.GetInnerObj(),x,k,selfmatch));
  }
//+------------------------------------------------------------------+
//| K-NN query: K nearest neighbors                                  |
//| INPUT PARAMETERS                                                 |
//|     KDT         -   KD-tree                                      |
//|     X           -   point, array[0..NX-1].                       |
//|     K           -   number of neighbors to return, K>=1          |
//|     SelfMatch   -   whether self-matches are allowed:            |
//|                     * if True, nearest neighbor may be the point |
//|                       itself (if it exists in original dataset)  |
//|                     * if False, then only points with non-zero   |
//|                       distance are returned                      |
//|                     * if not given, considered True              |
//| RESULT                                                           |
//|     number of actual neighbors found (either K or N, if K>N).    |
//| This  subroutine performs query and stores its result in the     |
//| internal structures of the KD-tree. You can use following        |
//| subroutines to obtain these results:                             |
//| * KDTreeQueryResultsX() to get X-values                          |
//| * KDTreeQueryResultsXY() to get X- and Y-values                  |
//| * KDTreeQueryResultsTags() to get tag values                     |
//| * KDTreeQueryResultsDistances() to get distances                 |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryKNN(CKDTreeShell &kdt,vector<double> &x,
                            const int k,const bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeQueryKNN(kdt.GetInnerObj(),X,k,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryKNN(CKDTreeShell &kdt,CRowDouble &x,
                            const int k,const bool selfmatch=true)
  {
   return(CNearestNeighbor::KDTreeQueryKNN(kdt.GetInnerObj(),x,k,selfmatch));
  }
//+------------------------------------------------------------------+
//| K-NN query: K nearest neighbors, using external thread-local     |
//| buffer.                                                          |
//| You can call this function from multiple threads for same kd-tree|
//| instance, assuming that different instances of buffer object are |
//| passed to different threads.                                     |
//| INPUT PARAMETERS                                                 |
//|   KDT   -  kd-tree                                               |
//|   Buf   -  request buffer object created for this particular     |
//|            instance of kd-tree structure with                    |
//|            KDTreeCreateRequestBuffer() function.                 |
//|   X     -  point, array[0..NX-1].                                |
//|   K     -  number of neighbors to return, K>=1                   |
//|   SelfMatch   -  whether self-matches are allowed:               |
//|                  * if True, nearest neighbor may be the point    |
//|                    itself (if it exists in original dataset)     |
//|                  * if False, then only points with non-zero      |
//|                    distance are returned                         |
//|                  * if not given, considered True                 |
//| RESULT                                                           |
//|   number of actual neighbors found (either K or N, if K>N).      |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the buffer object. You can use following  |
//| subroutines to obtain these results (pay attention to "buf" in   |
//| their names):                                                    |
//|   * KDTreeTsQueryResultsX() to get X-values                      |
//|   * KDTreeTsQueryResultsXY() to get X- and Y-values              |
//|   * KDTreeTsQueryResultsTags() to get tag values                 |
//|   * KDTreeTsQueryResultsDistances() to get distances             |
//| IMPORTANT: kd-tree buffer should be used only with KD-tree object|
//| which was used to initialize buffer. Any attempt to use biffer   |
//| with different object is dangerous - you may get integrity check |
//| failure (exception) because sizes of internal arrays do not fit  |
//| to dimensions of KD-tree structure.                              |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryKNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                              double &x[],const int k,const bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeTsQueryKNN(kdt.GetInnerObj(),buf.GetInnerObj(),X,k,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryKNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                              vector<double> &x,const int k,const bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeTsQueryKNN(kdt.GetInnerObj(),buf.GetInnerObj(),X,k,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryKNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                              CRowDouble &x,const int k,const bool selfmatch)
  {
   return(CNearestNeighbor::KDTreeTsQueryKNN(kdt.GetInnerObj(),buf.GetInnerObj(),x,k,selfmatch));
  }
//+------------------------------------------------------------------+
//| R-NN query: all points within R-sphere centered at X             |
//| INPUT PARAMETERS                                                 |
//|     KDT         -   KD-tree                                      |
//|     X           -   point, array[0..NX-1].                       |
//|     R           -   radius of sphere (in corresponding norm), R>0|
//|     SelfMatch   -   whether self-matches are allowed:            |
//|                     * if True, nearest neighbor may be the point |
//|                       itself (if it exists in original dataset)  |
//|                     * if False, then only points with non-zero   |
//|                       distance are returned                      |
//|                     * if not given, considered True              |
//| RESULT                                                           |
//|     number of neighbors found, >=0                               |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the KD-tree. You can use following        |
//| subroutines to obtain actual results:                            |
//| * KDTreeQueryResultsX() to get X-values                          |
//| * KDTreeQueryResultsXY() to get X- and Y-values                  |
//| * KDTreeQueryResultsTags() to get tag values                     |
//| * KDTreeQueryResultsDistances() to get distances                 |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryRNN(CKDTreeShell &kdt,double &x[],
                            const double r,const bool selfmatch)
  {
   return(CNearestNeighbor::KDTreeQueryRNN(kdt.GetInnerObj(),x,r,selfmatch));
  }
//+------------------------------------------------------------------+
//| R-NN query: all points within R-sphere centered at X             |
//| INPUT PARAMETERS                                                 |
//|     KDT         -   KD-tree                                      |
//|     X           -   point, array[0..NX-1].                       |
//|     R           -   radius of sphere (in corresponding norm), R>0|
//|     SelfMatch   -   whether self-matches are allowed:            |
//|                     * if True, nearest neighbor may be the point |
//|                       itself (if it exists in original dataset)  |
//|                     * if False, then only points with non-zero   |
//|                       distance are returned                      |
//|                     * if not given, considered True              |
//| RESULT                                                           |
//|     number of neighbors found, >=0                               |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the KD-tree. You can use following        |
//| subroutines to obtain actual results:                            |
//| * KDTreeQueryResultsX() to get X-values                          |
//| * KDTreeQueryResultsXY() to get X- and Y-values                  |
//| * KDTreeQueryResultsTags() to get tag values                     |
//| * KDTreeQueryResultsDistances() to get distances                 |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryRNN(CKDTreeShell &kdt,vector<double> &x,
                            const double r,const bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeQueryRNN(kdt.GetInnerObj(),X,r,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryRNN(CKDTreeShell &kdt,CRowDouble &x,
                            const double r,const bool selfmatch)
  {
   return(CNearestNeighbor::KDTreeQueryRNN(kdt.GetInnerObj(),x,r,selfmatch));
  }
//+------------------------------------------------------------------+
//| R-NN query: all points within R-sphere centered at X, no ordering|
//| by distance as undicated by "U" suffix (faster that ordered      |
//| query, for large queries - significantly faster).                |
//| IMPORTANT: this function can not be used in multithreaded code   |
//|            because it uses internal temporary buffer of kd-tree  |
//|            object, which can not be shared between multiple      |
//|            threads. If you want to perform parallel requests, use|
//|            function which uses external request buffer:          |
//|            KDTreeTsQueryRNN() ("Ts" stands for "thread-safe").   |
//| INPUT PARAMETERS                                                 |
//|   KDT   -  KD-tree                                               |
//|   X     -  point, array[0..NX-1].                                |
//|   R     -  radius of sphere (in corresponding norm), R>0         |
//|   SelfMatch   -  whether self-matches are allowed:               |
//|                  * if True, nearest neighbor may be the point    |
//|                    itself (if it exists in original dataset)     |
//|                  * if False, then only points with non-zero      |
//|                    distance are returned                         |
//|                  * if not given, considered True                 |
//| RESULT                                                           |
//|   number of neighbors found, >=0                                 |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the KD-tree. You can use following        |
//| subroutines to obtain actual results:                            |
//|   * KDTreeQueryResultsX() to get X-values                        |
//|   * KDTreeQueryResultsXY() to get X- and Y-values                |
//|   * KDTreeQueryResultsTags() to get tag values                   |
//|   * KDTreeQueryResultsDistances() to get distances               |
//| As indicated by "U" suffix, this function returns unordered      |
//| results.                                                         |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryRNNU(CKDTreeShell &kdt,double &x[],const double r,bool selfmatch)
  {
   return(CNearestNeighbor::KDTreeQueryRNNU(kdt.GetInnerObj(),x,r,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryRNNU(CKDTreeShell &kdt,vector<double> &x,const double r,bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeQueryRNNU(kdt.GetInnerObj(),X,r,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryRNNU(CKDTreeShell &kdt,CRowDouble &x,const double r,bool selfmatch)
  {
   return(CNearestNeighbor::KDTreeQueryRNNU(kdt.GetInnerObj(),x,r,selfmatch));
  }
//+------------------------------------------------------------------+
//| R-NN query: all points within  R-sphere  centered  at  X, using  |
//| external thread-local buffer, sorted by distance between point   |
//| and X (by ascending)                                             |
//| You can call this function from multiple threads for same kd-tree|
//| instance, assuming that different instances of buffer object are |
//| passed to different threads.                                     |
//| NOTE: it is also possible to perform undordered queries performed|
//| by means of KDTreeQueryRNNU() and KDTreeTsQueryRNNU() functions. |
//| Such queries are faster because we do not have to use heap       |
//| structure for sorting.                                           |
//| INPUT PARAMETERS                                                 |
//|   KDT   -  KD-tree                                               |
//|   Buf   -  request buffer object created for this particular     |
//|            instance of kd-tree structure with                    |
//|            KDTreeCreateRequestBuffer() function.                 |
//|   X     -  point, array[0..NX-1].                                |
//|   R     -  radius of sphere (in corresponding norm), R>0         |
//|   SelfMatch   -  whether self-matches are allowed:               |
//|            * if True, nearest neighbor may be the point itself   |
//|              (if it exists in original dataset)                  |
//|            * if False, then only points with non-zero distance   |
//|              are returned                                        |
//|            * if not given, considered True                       |
//| RESULT                                                           |
//|   number of neighbors found, >=0                                 |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the buffer object. You can use following  |
//| subroutines to obtain these results (pay attention to "buf" in   |
//| their names):                                                    |
//|   * KDTreeTsQueryResultsX() to get X-values                      |
//|   * KDTreeTsQueryResultsXY() to get X- and Y-values              |
//|   * KDTreeTsQueryResultsTags() to get tag values                 |
//|   * KDTreeTsQueryResultsDistances() to get distances             |
//| IMPORTANT: kd-tree buffer should be used only with KD-tree object|
//|            which was used to initialize buffer. Any attempt to   |
//|            use biffer with different object is dangerous - you   |
//|            may get integrity check failure (exception) because   |
//|            sizes of internal arrays do not fit to dimensions of  |
//|            KD-tree structure.                                    |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryRNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                              double &x[],const double r,bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeTsQueryRNN(kdt.GetInnerObj(),buf.GetInnerObj(),X,r,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryRNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                              vector<double> &x,const double r,bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeTsQueryRNN(kdt.GetInnerObj(),buf.GetInnerObj(),X,r,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryRNN(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                              CRowDouble &x,const double r,bool selfmatch)
  {
   return(CNearestNeighbor::KDTreeTsQueryRNN(kdt.GetInnerObj(),buf.GetInnerObj(),x,r,selfmatch));
  }
//+------------------------------------------------------------------+
//| R-NN query: all points within R-sphere centered at X, using      |
//| external thread-local buffer, no ordering by distance as         |
//| undicated by "U" suffix (faster that ordered query, for large    |
//| queries - significantly faster).                                 |
//| You can call this function from multiple threads for same kd-tree|
//| instance, assuming that different instances of buffer object are |
//| passed to different threads.                                     |
//| INPUT PARAMETERS                                                 |
//|   KDT   -  KD-tree                                               |
//|   Buf   -  request buffer object created for this particular     |
//|            instance of kd-tree structure with                    |
//|            KDTreeCreateRequestBuffer() function.                 |
//|   X     -  point, array[0..NX-1].                                |
//|   R     -  radius of sphere (in corresponding norm), R>0         |
//|   SelfMatch   -  whether self-matches are allowed:               |
//|               * if True, nearest neighbor may be the point itself|
//|                 (if it exists in original dataset)               |
//|               * if False, then only points with non-zero distance|
//|                 are returned                                     |
//|               * if not given, considered True                    |
//| RESULT                                                           |
//|   number of neighbors found, >=0                                 |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the buffer object. You can use following  |
//| subroutines to obtain these results (pay attention to "buf" in   |
//| their names):                                                    |
//|   * KDTreeTsQueryResultsX() to get X-values                      |
//|   * KDTreeTsQueryResultsXY() to get X- and Y-values              |
//|   * KDTreeTsQueryResultsTags() to get tag values                 |
//|   * KDTreeTsQueryResultsDistances() to get distances             |
//| As indicated by "U" suffix, this function returns unordered      |
//| results.                                                         |
//| IMPORTANT: kd-tree buffer should be used only with KD-tree object|
//|            which was used to initialize buffer. Any attempt to   |
//|            use biffer with different object is dangerous - you   |
//|            may get integrity check failure (exception) because   |
//|            sizes of internal arrays do not fit to dimensions of  |
//|            KD-tree structure.                                    |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryRNNU(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                               double &x[],const double r,const bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeTsQueryRNNU(kdt.GetInnerObj(),buf.GetInnerObj(),X,r,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryRNNU(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                               vector<double> &x,const double r,const bool selfmatch)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeTsQueryRNNU(kdt.GetInnerObj(),buf.GetInnerObj(),X,r,selfmatch));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryRNNU(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,
                               CRowDouble &x,const double r,const bool selfmatch)
  {
   return(CNearestNeighbor::KDTreeTsQueryRNNU(kdt.GetInnerObj(),buf.GetInnerObj(),x,r,selfmatch));
  }
//+------------------------------------------------------------------+
//| K-NN query: approximate K nearest neighbors                      |
//| INPUT PARAMETERS                                                 |
//|     KDT         -   KD-tree                                      |
//|     X           -   point, array[0..NX-1].                       |
//|     K           -   number of neighbors to return, K>=1          |
//|     SelfMatch   -   whether self-matches are allowed:            |
//|                     * if True, nearest neighbor may be the point |
//|                       itself (if it exists in original dataset)  |
//|                     * if False, then only points with non-zero   |
//|                       distance are returned                      |
//|                     * if not given, considered True              |
//|     Eps         -   approximation factor, Eps>=0. eps-approximate|
//|                     nearest neighbor is a neighbor whose distance|
//|                     from X is at most (1+eps) times distance of  |
//|                     true nearest neighbor.                       |
//| RESULT                                                           |
//|     number of actual neighbors found (either K or N, if K>N).    |
//| NOTES                                                            |
//|     significant performance gain may be achieved only when Eps is|
//|     on the order of magnitude of 1 or larger.                    |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the KD-tree. You can use following        |
//| these subroutines to  obtain results:                            |
//| * KDTreeQueryResultsX() to get X-values                          |
//| * KDTreeQueryResultsXY() to get X- and Y-values                  |
//| * KDTreeQueryResultsTags() to get tag values                     |
//| * KDTreeQueryResultsDistances() to get distances                 |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryAKNN(CKDTreeShell &kdt,double &x[],
                             const int k,const bool selfmatch,
                             const double eps)
  {
   return(CNearestNeighbor::KDTreeQueryAKNN(kdt.GetInnerObj(),x,k,selfmatch,eps));
  }
//+------------------------------------------------------------------+
//| K-NN query: approximate K nearest neighbors                      |
//| INPUT PARAMETERS                                                 |
//|     KDT         -   KD-tree                                      |
//|     X           -   point, array[0..NX-1].                       |
//|     K           -   number of neighbors to return, K>=1          |
//|     SelfMatch   -   whether self-matches are allowed:            |
//|                     * if True, nearest neighbor may be the point |
//|                       itself (if it exists in original dataset)  |
//|                     * if False, then only points with non-zero   |
//|                       distance are returned                      |
//|                     * if not given, considered True              |
//|     Eps         -   approximation factor, Eps>=0. eps-approximate|
//|                     nearest neighbor is a neighbor whose distance|
//|                     from X is at most (1+eps) times distance of  |
//|                     true nearest neighbor.                       |
//| RESULT                                                           |
//|     number of actual neighbors found (either K or N, if K>N).    |
//| NOTES                                                            |
//|     significant performance gain may be achieved only when Eps is|
//|     on the order of magnitude of 1 or larger.                    |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the KD-tree. You can use following        |
//| these subroutines to  obtain results:                            |
//| * KDTreeQueryResultsX() to get X-values                          |
//| * KDTreeQueryResultsXY() to get X- and Y-values                  |
//| * KDTreeQueryResultsTags() to get tag values                     |
//| * KDTreeQueryResultsDistances() to get distances                 |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryAKNN(CKDTreeShell &kdt,vector<double> &x,
                             const int k,const bool selfmatch,
                             const double eps=0)
  {
   CRowDouble X=x;
   return(CNearestNeighbor::KDTreeQueryAKNN(kdt.GetInnerObj(),X,k,selfmatch,eps));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryAKNN(CKDTreeShell &kdt,CRowDouble &x,
                             const int k,const bool selfmatch,
                             const double eps)
  {
   return(CNearestNeighbor::KDTreeQueryAKNN(kdt.GetInnerObj(),x,k,selfmatch,eps));
  }
//+------------------------------------------------------------------+
//| Box query: all points within user-specified box.                 |
//| IMPORTANT: this function can not be used in multithreaded code   |
//|            because it uses internal temporary buffer of kd-tree  |
//|            object, which can not be shared between multiple      |
//|            threads. If you want to perform parallel requests,    |
//|            use function which uses external request buffer:      |
//|            KDTreeTsQueryBox() ("Ts" stands for "thread-safe").   |
//| INPUT PARAMETERS                                                 |
//|   KDT      -  KD-tree                                            |
//|   BoxMin   -  lower bounds, array[0..NX-1].                      |
//|   BoxMax   -  upper bounds, array[0..NX-1].                      |
//| RESULT                                                           |
//|   number of actual neighbors found (in [0,N]).                   |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the KD-tree. You can use following        |
//| subroutines to obtain these results:                             |
//|   * KDTreeQueryResultsX() to get X-values                        |
//|   * KDTreeQueryResultsXY() to get X- and Y-values                |
//|   * KDTreeQueryResultsTags() to get tag values                   |
//|   * KDTreeQueryResultsDistances() returns zeros for this request |
//| NOTE: this particular query returns unordered results, because   |
//|       there is no meaningful way of ordering points. Furthermore,|
//|       no 'distance' is associated with points - it is either     |
//|       INSIDE or OUTSIDE (so request for distances will return    |
//|       zeros).                                                    |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryBox(CKDTreeShell &kdt,double &boxmin[],double &boxmax[])
  {
   return(CNearestNeighbor::KDTreeQueryBox(kdt.GetInnerObj(),boxmin,boxmax));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryBox(CKDTreeShell &kdt,vector<double> &boxmin,vector<double> &boxmax)
  {
   CRowDouble Min=boxmin;
   CRowDouble Max=boxmax;
   return(CNearestNeighbor::KDTreeQueryBox(kdt.GetInnerObj(),Min,Max));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeQueryBox(CKDTreeShell &kdt,CRowDouble &boxmin,CRowDouble &boxmax)
  {
   return(CNearestNeighbor::KDTreeQueryBox(kdt.GetInnerObj(),boxmin,boxmax));
  }
//+------------------------------------------------------------------+
//| Box query: all points within user-specified box, using           |
//| thread-local buffer.                                             |
//| You can call this function from multiple threads for same kd-tree|
//| instance, assuming that different instances of buffer object are |
//| passed to different threads.                                     |
//| INPUT PARAMETERS                                                 |
//|   KDT      -  KD-tree                                            |
//|   Buf      -  request buffer object created for this particular  |
//|               instance of kd-tree structure with                 |
//|               KDTreeCreateRequestBuffer() function.              |
//|   BoxMin   -  lower bounds, array[0..NX-1].                      |
//|   BoxMax   -  upper bounds, array[0..NX-1].                      |
//| RESULT                                                           |
//|   number of actual neighbors found (in [0,N]).                   |
//| This subroutine performs query and stores its result in the      |
//| internal structures of the buffer object. You can use following  |
//| subroutines to obtain these results (pay attention to "ts" in    |
//| their names):                                                    |
//|   * KDTreeTsQueryResultsX() to get X-values                      |
//|   * KDTreeTsQueryResultsXY() to get X- and Y-values              |
//|   * KDTreeTsQueryResultsTags() to get tag values                 |
//|   * KDTreeTsQueryResultsDistances() returns zeros for this query |
//| NOTE: this particular query returns unordered results, because   |
//|       there is no meaningful way of ordering points. Furthermore,|
//|       no 'distance' is associated with points - it is either     |
//|       INSIDE  or OUTSIDE (so request for distances will return   |
//|       zeros).                                                    |
//| IMPORTANT: kd-tree buffer should be used only with KD-tree object|
//|            which was used to initialize buffer. Any attempt to   |
//|            use biffer with different object is dangerous - you   |
//|            may  get  integrity  check failure (exception) because|
//|            sizes of internal arrays do not fit to dimensions of  |
//|            KD-tree structure.                                    |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryBox(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,double &boxmin[],double &boxmax[])
  {
   return(CNearestNeighbor::KDTreeTsQueryBox(kdt.GetInnerObj(),buf.GetInnerObj(),boxmin,boxmax));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryBox(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,CRowDouble &boxmin,CRowDouble &boxmax)
  {
   return(CNearestNeighbor::KDTreeTsQueryBox(kdt.GetInnerObj(),buf.GetInnerObj(),boxmin,boxmax));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::KDTreeTsQueryBox(CKDTreeShell &kdt,CKDTreeRequestBufferShell &buf,vector<double> &boxmin,vector<double> &boxmax)
  {
   CRowDouble Min=boxmin;
   CRowDouble Max=boxmax;
   return(CNearestNeighbor::KDTreeTsQueryBox(kdt.GetInnerObj(),buf.GetInnerObj(),Min,Max));
  }
//+------------------------------------------------------------------+
//| X-values from last query                                         |
//| INPUT PARAMETERS                                                 |
//|     KDT     -   KD-tree                                          |
//|     X       -   possibly pre-allocated buffer. If X is too small |
//|                 to store result, it is resized. If size(X) is    |
//|                 enough to store result, it is left unchanged.    |
//| OUTPUT PARAMETERS                                                |
//|     X       -   rows are filled with X-values                    |
//| NOTES                                                            |
//| 1. points are ordered by distance from the query point (first =  |
//|    closest)                                                      |
//| 2. if  XY is larger than required to store result, only leading  |
//|    part will be overwritten; trailing part will be left          |
//|    unchanged. So if on input XY = [[A,B],[C,D]], and result is   |
//|    [1,2], then on exit we will get XY = [[1,2],[C,D]]. This is   |
//|    done purposely to increase performance; if you want function  |
//|    to resize array according to result size, use function with   |
//| same name and suffix 'I'.                                        |
//| SEE ALSO                                                         |
//| * KDTreeQueryResultsXY()            X- and Y-values              |
//| * KDTreeQueryResultsTags()          tag values                   |
//| * KDTreeQueryResultsDistances()     distances                    |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsX(CKDTreeShell &kdt,CMatrixDouble &x)
  {
   CNearestNeighbor::KDTreeQueryResultsX(kdt.GetInnerObj(),x);
  }
//+------------------------------------------------------------------+
//| X- and Y-values from last query                                  |
//| INPUT PARAMETERS                                                 |
//|     KDT     -   KD-tree                                          |
//|     XY      -   possibly pre-allocated buffer. If XY is too small|
//|                 to store result, it is resized. If size(XY) is   |
//|                 enough to store result, it is left unchanged.    |
//| OUTPUT PARAMETERS                                                |
//|     XY      -   rows are filled with points: first NX columns    |
//|                 with X-values, next NY columns - with Y-values.  |
//| NOTES                                                            |
//| 1. points are ordered by distance from the query point (first =  |
//|    closest)                                                      |
//| 2. if  XY is larger than required to store result, only leading  |
//|    part will be overwritten; trailing part will be left          |
//|    unchanged. So if on input XY = [[A,B],[C,D]], and result is   |
//|    [1,2], then on exit we will get XY = [[1,2],[C,D]]. This is   |
//|    done purposely to increase performance; if you want function  |
//|    to resize array according to result size, use function with   |
//|    same name and suffix 'I'.                                     |
//| SEE ALSO                                                         |
//| * KDTreeQueryResultsX()             X-values                     |
//| * KDTreeQueryResultsTags()          tag values                   |
//| * KDTreeQueryResultsDistances()     distances                    |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsXY(CKDTreeShell &kdt,CMatrixDouble &xy)
  {
   CNearestNeighbor::KDTreeQueryResultsXY(kdt.GetInnerObj(),xy);
  }
//+------------------------------------------------------------------+
//| Tags from last query                                             |
//| INPUT PARAMETERS                                                 |
//|     KDT     -   KD-tree                                          |
//|     Tags    -   possibly pre-allocated buffer. If X is too small |
//|                 to store result, it is resized. If size(X) is    |
//|                 enough to store result, it is left unchanged.    |
//| OUTPUT PARAMETERS                                                |
//|     Tags    -   filled with tags associated with points,         |
//|                 or, when no tags were supplied, with zeros       |
//| NOTES                                                            |
//| 1. points are ordered by distance from the query point (first    |
//|    = closest)                                                    |
//| 2. if  XY is larger than required to store result, only leading  |
//|    part will be overwritten; trailing part will be left          |
//|    unchanged. So if on input XY = [[A,B],[C,D]], and result is   |
//|    [1,2],then on exit we will get XY = [[1,2], [C,D]]. This is   |
//|    done purposely to increase performance; if you want function  |
//|    to resize array according to result size, use function with   |
//|    same name and suffix 'I'.                                     |
//| SEE ALSO                                                         |
//| * KDTreeQueryResultsX()             X-values                     |
//| * KDTreeQueryResultsXY()            X- and Y-values              |
//| * KDTreeQueryResultsDistances()     distances                    |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsTags(CKDTreeShell &kdt,int &tags[])
  {
   CNearestNeighbor::KDTreeQueryResultsTags(kdt.GetInnerObj(),tags);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsTags(CKDTreeShell &kdt,CRowInt &tags)
  {
   CNearestNeighbor::KDTreeQueryResultsTags(kdt.GetInnerObj(),tags);
  }
//+------------------------------------------------------------------+
//| Distances from last query                                        |
//| INPUT PARAMETERS                                                 |
//|     KDT     -   KD-tree                                          |
//|     R       -   possibly pre-allocated buffer. If X is too small |
//|                 to store result, it is resized. If size(X) is    |
//|                 enough to store result, it is left unchanged.    |
//| OUTPUT PARAMETERS                                                |
//|     R       -   filled with distances (in corresponding norm)    |
//| NOTES                                                            |
//| 1. points are ordered by distance from the query point (first    |
//|    = closest)                                                    |
//| 2. if  XY is larger than required to store result, only leading  |
//|    part will be overwritten; trailing part will be left          |
//|    unchanged. So if on input XY = [[A,B], [C,D]],and result is   |
//|    [1,2], then on exit we will get XY = [[1,2], C,D]]. This is   |
//|    done purposely to increase performance; if you want function  |
//|    to resize array according to result size, use function with   |
//|    same name and suffix 'I'.                                     |
//| SEE ALSO                                                         |
//| * KDTreeQueryResultsX()             X-values                     |
//| * KDTreeQueryResultsXY()            X- and Y-values              |
//| * KDTreeQueryResultsTags()          tag values                   |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsDistances(CKDTreeShell &kdt,double &r[])
  {
   CNearestNeighbor::KDTreeQueryResultsDistances(kdt.GetInnerObj(),r);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsDistances(CKDTreeShell &kdt,CRowDouble &r)
  {
   CNearestNeighbor::KDTreeQueryResultsDistances(kdt.GetInnerObj(),r);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsDistances(CKDTreeShell &kdt,vector<double> &r)
  {
//--- create variables
   CRowDouble R;
//--- function call
   CNearestNeighbor::KDTreeQueryResultsDistances(kdt.GetInnerObj(),R);
//---copy result
   r=R.ToVector();
  }
//+------------------------------------------------------------------+
//| X-values from last query; 'interactive' variant for languages    |
//| like Python which support constructs like "X =                   |
//| KDTreeQueryResultsXI(KDT)" and interactive mode of interpreter.  |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart, but |
//| it is more convenient when you call it from command line.        |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsXI(CKDTreeShell &kdt,CMatrixDouble &x)
  {
   CNearestNeighbor::KDTreeQueryResultsXI(kdt.GetInnerObj(),x);
  }
//+------------------------------------------------------------------+
//| XY-values from last query; 'interactive' variant for languages   |
//| like Python which support constructs like "XY =                  |
//| KDTreeQueryResultsXYI(KDT)" and interactive mode of interpreter. |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart, but |
//| it is more convenient when you call it from command line.        |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsXYI(CKDTreeShell &kdt,CMatrixDouble &xy)
  {
   CNearestNeighbor::KDTreeQueryResultsXYI(kdt.GetInnerObj(),xy);
  }
//+------------------------------------------------------------------+
//| Tags from last query; 'interactive' variant for languages like   |
//| Python which  support  constructs  like "Tags =                  |
//| KDTreeQueryResultsTagsI(KDT)" and interactive mode of            |
//| interpreter.                                                     |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart, but |
//| it is more convenient when you call it from command line.        |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsTagsI(CKDTreeShell &kdt,int &tags[])
  {
   CNearestNeighbor::KDTreeQueryResultsTagsI(kdt.GetInnerObj(),tags);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsTagsI(CKDTreeShell &kdt,CRowInt &tags)
  {
   CNearestNeighbor::KDTreeQueryResultsTagsI(kdt.GetInnerObj(),tags);
  }
//+------------------------------------------------------------------+
//| Distances from last query; 'interactive' variant for languages   |
//| like Python which support constructs like "R =                   |
//| KDTreeQueryResultsDistancesI(KDT)" and interactive mode of       |
//| interpreter.                                                     |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart, but |
//| it is more convenient when you call it from command line.        |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsDistancesI(CKDTreeShell &kdt,double &r[])
  {
   CNearestNeighbor::KDTreeQueryResultsDistancesI(kdt.GetInnerObj(),r);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsDistancesI(CKDTreeShell &kdt,vector<double> &r)
  {
   CRowDouble R;
   CNearestNeighbor::KDTreeQueryResultsDistancesI(kdt.GetInnerObj(),R);
   r=R.ToVector();
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::KDTreeQueryResultsDistancesI(CKDTreeShell &kdt,CRowDouble &r)
  {
   CNearestNeighbor::KDTreeQueryResultsDistancesI(kdt.GetInnerObj(),r);
  }
//+------------------------------------------------------------------+
//| Optimal binary classification                                    |
//| Algorithms finds optimal (=with minimal cross-entropy) binary    |
//| partition.                                                       |
//| Internal subroutine.                                             |
//| INPUT PARAMETERS:                                                |
//|     A       -   array[0..N-1], variable                          |
//|     C       -   array[0..N-1], class numbers (0 or 1).           |
//|     N       -   array size                                       |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   completetion code:                               |
//|                 * -3, all values of A[] are same (partition is   |
//|                   impossible)                                    |
//|                 * -2, one of C[] is incorrect (<0, >1)           |
//|                 * -1, incorrect pararemets were passed (N<=0).   |
//|                 *  1, OK                                         |
//|     Threshold-  partiton boundary. Left part contains values     |
//|                 which are strictly less than Threshold. Right    |
//|                 part contains values which are greater than or   |
//|                 equal to Threshold.                              |
//|     PAL, PBL-   probabilities P(0|v<Threshold) and               |
//|                 P(1|v<Threshold)                                 |
//|     PAR, PBR-   probabilities P(0|v>=Threshold) and              |
//|                 P(1|v>=Threshold)                                |
//|     CVE     -   cross-validation estimate of cross-entropy       |
//+------------------------------------------------------------------+
void CAlglib::DSOptimalSplit2(double &a[],int &c[],const int n,
                              int &info,double &threshold,
                              double &pal,double &pbl,double &par,
                              double &pbr,double &cve)
  {
//--- initialization
   info=0;
   threshold=0;
   pal=0;
   pbl=0;
   par=0;
   pbr=0;
   cve=0;
//--- function call
   CBdSS::DSOptimalSplit2(a,c,n,info,threshold,pal,pbl,par,pbr,cve);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::DSOptimalSplit2(CRowDouble &a,CRowInt &c,const int n,
                              int &info,double &threshold,
                              double &pal,double &pbl,double &par,
                              double &pbr,double &cve)
  {
//--- initialization
   info=0;
   threshold=0;
   pal=0;
   pbl=0;
   par=0;
   pbr=0;
   cve=0;
//--- function call
   CBdSS::DSOptimalSplit2(a,c,n,info,threshold,pal,pbl,par,pbr,cve);
  }
//+------------------------------------------------------------------+
//| Optimal partition, internal subroutine. Fast version.            |
//| Accepts:                                                         |
//|     A       array[0..N-1]       array of attributes array[0..N-1]|
//|     C       array[0..N-1]       array of class labels            |
//|     TiesBuf array[0..N]         temporaries (ties)               |
//|     CntBuf  array[0..2*NC-1]    temporaries (counts)             |
//|     Alpha                       centering factor (0<=alpha<=1,   |
//|                                 recommended value - 0.05)        |
//|     BufR    array[0..N-1]       temporaries                      |
//|     BufI    array[0..N-1]       temporaries                      |
//| Output:                                                          |
//|     Info    error code (">0"=OK, "<0"=bad)                       |
//|     RMS     training set RMS error                               |
//|     CVRMS   leave-one-out RMS error                              |
//| Note:                                                            |
//|     content of all arrays is changed by subroutine;              |
//|     it doesn't allocate temporaries.                             |
//+------------------------------------------------------------------+
void CAlglib::DSOptimalSplit2Fast(double &a[],int &c[],int &tiesbuf[],
                                  int &cntbuf[],double &bufr[],
                                  int &bufi[],const int n,
                                  const int nc,const double alpha,
                                  int &info,double &threshold,
                                  double &rms,double &cvrms)
  {
//--- initialization
   info=0;
   threshold=0;
   rms=0;
   cvrms=0;
//--- function call
   CBdSS::DSOptimalSplit2Fast(a,c,tiesbuf,cntbuf,bufr,bufi,n,nc,alpha,info,threshold,rms,cvrms);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::DSOptimalSplit2Fast(CRowDouble &a,CRowInt &c,CRowInt &tiesbuf,
                                  CRowInt &cntbuf,CRowDouble &bufr,
                                  CRowInt &bufi,const int n,
                                  const int nc,const double alpha,
                                  int &info,double &threshold,
                                  double &rms,double &cvrms)
  {
//--- initialization
   info=0;
   threshold=0;
   rms=0;
   cvrms=0;
//--- function call
   CBdSS::DSOptimalSplit2Fast(a,c,tiesbuf,cntbuf,bufr,bufi,n,nc,alpha,info,threshold,rms,cvrms);
  }
//+------------------------------------------------------------------+
//| This function serializes data structure to string.               |
//| Important properties of s_out:                                   |
//| * it contains alphanumeric characters, dots, underscores, minus  |
//|   signs                                                          |
//| * these symbols are grouped into words, which are separated by   |
//|   spaces and Windows-style (CR+LF) newlines                      |
//| * although  serializer  uses  spaces and CR+LF as separators, you|
//|   can replace any separator character by arbitrary combination of|
//|   spaces, tabs, Windows or Unix newlines. It allows flexible     |
//|   reformatting of the string in case you want to include it into |
//|   text or XML file. But you should not insert separators into the|
//|   middle of the "words" nor you should change case of letters.   |
//| * s_out can be freely moved between 32-bit and 64-bit systems,   |
//|   little and big endian machines, and so on. You can reference   |
//|   structure on 32-bit machine and unserialize it on 64-bit one   |
//|   (or vice versa), or reference it on SPARC and unserialize on   |
//|   x86. You can also reference it in C# version of ALGLIB and     |
//|   unserialize in C++ one, and vice versa.                        |
//+------------------------------------------------------------------+
void CAlglib::DFSerialize(CDecisionForestShell &obj,string &s_out)
  {
   CSerializer s;
//--- serialization start
   s.Alloc_Start();
//--- function call
   CDForest::DFAlloc(s,obj.GetInnerObj());
//--- serialization
   s.SStart_Str();
//--- function call
   CDForest::DFSerialize(s,obj.GetInnerObj());
//--- stop
   s.Stop();
//--- change value
   s_out=s.Get_String();
  }
//+------------------------------------------------------------------+
//| This function unserializes data structure from string.           |
//+------------------------------------------------------------------+
void CAlglib::DFUnserialize(const string s_in,CDecisionForestShell &obj)
  {
   CSerializer s;
//--- unserialization
   s.UStart_Str(s_in);
//--- function call
   CDForest::DFUnserialize(s,obj.GetInnerObj());
//--- stop
   s.Stop();
  }
//+------------------------------------------------------------------+
//| This function creates buffer structure which can be used to      |
//| perform parallel inference requests.                             |
//| DF subpackage provides two sets of computing functions - ones    |
//| which use internal buffer of DF model (these functions are       |
//| single-threaded because they use same buffer, which can not      |
//| shared between threads), and ones which use external buffer.     |
//| This function is used to initialize external buffer.             |
//| INPUT PARAMETERS:                                                |
//|   Model    -  DF model which is associated with newly created    |
//|               buffer                                             |
//| OUTPUT PARAMETERS:                                               |
//|   Buf      -  external buffer.                                   |
//| IMPORTANT: buffer object should be used only with model which was|
//|            used to initialize buffer. Any attempt to use buffer  |
//|            with different object is dangerous - you may get      |
//|            integrity check failure (exception) because sizes of  |
//|            internal arrays do not fit to dimensions of the model |
//|            structure.                                            |
//+------------------------------------------------------------------+
void CAlglib::DFCreateBuffer(CDecisionForestShell &model,
                             CDecisionForestBuffer &buf)
  {
   CDForest::DFCreateBuffer(model.GetInnerObj(),buf);
  }
//+------------------------------------------------------------------+
//| This subroutine creates CDecisionForestBuilder object which is   |
//| used to train decision forests.                                  |
//| By default, new builder stores empty dataset and some reasonable |
//| default settings. At the very least, you should specify dataset  |
//| prior to building decision forest. You can also tweak settings of|
//| the forest construction algorithm (recommended, although default |
//| setting should work well).                                       |
//| Following actions are mandatory:                                 |
//|   * calling DFBuilderSetDataset() to specify dataset             |
//|   * calling DFBuilderBuildRandomForest() to build decision forest|
//|     using current dataset and default settings                   |
//| Additionally, you may call:                                      |
//|   * DFBuilderSetRndVars() or DFBuilderSetRndVarsRatio() to       |
//|     specify number of variables randomly chosen for each split   |
//|   * DFBuilderSetSubsampleRatio() to specify fraction of the      |
//|     dataset randomly subsampled to build each tree               |
//|   * DFBuilderSetSeed() to control random seed chosen for tree    |
//|     construction                                                 |
//| INPUT PARAMETERS:                                                |
//|   none                                                           |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder                            |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderCreate(CDecisionForestBuilder &s)
  {
   CDForest::DFBuilderCreate(s);
  }
//+------------------------------------------------------------------+
//| This subroutine adds dense dataset to the internal storage of the|
//| builder object. Specifying your dataset in the dense format means|
//| that the dense version of the forest construction algorithm will |
//| be invoked.                                                      |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//|   XY       -  array[NPoints,NVars+1] (minimum size; actual size  |
//|               can be larger, only leading part is used anyway),  |
//|               dataset:                                           |
//|               * first NVars elements of each row store values of |
//|                 the independent variables                        |
//|               * last column store class number(in 0...NClasses-1)|
//|                 or real value of the dependent variable          |
//|   NPoints  -  number of rows in the dataset, NPoints>=1          |
//|   NVars    -  number of independent variables, NVars>=1          |
//|   NClasses -  indicates type of the problem being solved:        |
//|               * NClasses>=2 means that classification problem is |
//|                 solved (last column of the dataset stores class  |
//|                 number)                                          |
//|               * NClasses=1  means  that  regression  problem  is |
//|                 solved  (last  column  of  the  dataset  stores  |
//|                 variable value)                                  |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder                            |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetDataset(CDecisionForestBuilder &s,CMatrixDouble &xy,
                                  int npoints,int nvars,int nclasses)
  {
   CDForest::DFBuilderSetDataset(s,xy,npoints,nvars,nclasses);
  }
//+------------------------------------------------------------------+
//| This function sets number of variables (in [1,NVars] range) used |
//| by decision forest construction algorithm.                       |
//| The default option is to use roughly sqrt(NVars) variables.      |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//|   RndVars  -  number of randomly selected variables; values      |
//|               outside of [1,NVars] range are silently clipped.   |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder                            |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetRndVars(CDecisionForestBuilder &s,int rndvars)
  {
   CDForest::DFBuilderSetRndVars(s,rndvars);
  }
//+------------------------------------------------------------------+
//| This function sets number of variables used by decision forest   |
//| construction algorithm as a fraction of total variable count     |
//| (0,1) range.                                                     |
//| The default option is to use roughly sqrt(NVars) variables.      |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//|   F        -  round(NVars*F) variables are selected              |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder                            |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetRndVarsRatio(CDecisionForestBuilder &s,double f)
  {
   CDForest::DFBuilderSetRndVarsRatio(s,f);
  }
//+------------------------------------------------------------------+
//| This function tells decision forest builder to automatically     |
//| choose number of variables used by decision forest construction  |
//| algorithm. Roughly sqrt(NVars) variables will be used.           |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder                            |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetRndVarsAuto(CDecisionForestBuilder &s)
  {
   CDForest::DFBuilderSetRndVarsAuto(s);
  }
//+------------------------------------------------------------------+
//| This function sets size of dataset subsample generated the       |
//| decision forest construction algorithm. Size is specified as a   |
//| fraction of  total  dataset size.                                |
//| The default option is to use 50% of the dataset for training,    |
//| 50% for the OOB estimates. You can decrease fraction F down to   |
//| 10%, 1% or  even  below in order to reduce overfitting.          |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//|   F        -  fraction of the dataset to use, in (0,1] range.    |
//|               Values outside of this range will  be  silently    |
//|               clipped. At least one element is always selected   |
//|               for the training set.                              |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder                            |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetSubsampleRatio(CDecisionForestBuilder &s,double f)
  {
   CDForest::DFBuilderSetSubsampleRatio(s,f);
  }
//+------------------------------------------------------------------+
//| This  function  sets  seed  used  by  internal  RNG  for  random |
//| subsampling and random selection of variable subsets.            |
//| By  default  random  seed  is  used, i.e. every time you build   |
//| decision forest, we seed generator with new value obtained from  |
//| system-wide  RNG.  Thus, decision  forest  builder  returns      |
//| non-deterministic results. You  can  change  such  behavior  by  |
//| specyfing fixed positive seed value.                             |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//|   SeedVal  -  seed value:                                        |
//|               * positive values are used for seeding RNG with    |
//|                 fixed seed, i.e. subsequent runs on same data    |
//|                 will return same decision forests                |
//|               * non-positive seed means that random seed is used |
//|                 for every run of builder, i.e. subsequent  runs  |
//|                 on same datasets will return slightly different  |
//|                 decision forests                                 |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder, see                       |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetSeed(CDecisionForestBuilder &s,int seedval)
  {
   CDForest::DFBuilderSetSeed(s,seedval);
  }
//+------------------------------------------------------------------+
//| This function sets random decision forest construction algorithm.|
//| As for now, only one decision forest construction algorithm is   |
//| supported-a dense "baseline" RDF algorithm.                    |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//|   AlgoType -  algorithm type:                                    |
//|               * 0 = baseline dense RDF                           |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder, see                       |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetRDFAlgo(CDecisionForestBuilder &s,int algotype)
  {
   CDForest::DFBuilderSetRDFAlgo(s,algotype);
  }
//+------------------------------------------------------------------+
//| This function sets split selection algorithm used by decision    |
//| forest classifier. You may choose several algorithms, with       |
//| different speed and quality of the results.                      |
//| INPUT PARAMETERS:                                                |
//|   S              -  decision forest builder object               |
//|   SplitStrength  -  split type:                                  |
//|            * 0 = split at the random position, fastest one       |
//|            * 1 = split at the middle of the range                |
//|            * 2 = strong split at the best point of the range     |
//|                  (default)                                       |
//| OUTPUT PARAMETERS:                                               |
//|   S              -  decision forest builder, see                 |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetRDFSplitStrength(CDecisionForestBuilder &s,
                                           int splitstrength)
  {
   CDForest::DFBuilderSetRDFSplitStrength(s,splitstrength);
  }
//+------------------------------------------------------------------+
//| This function tells decision forest construction algorithm to use|
//| Gini impurity based variable importance estimation (also known as|
//| MDI).                                                            |
//| This version of importance estimation algorithm analyzes mean    |
//| decrease in impurity (MDI) on training sample during splits. The |
//| result is divided by impurity at the root node in order to       |
//| produce estimate in [0,1] range.                                 |
//| Such estimates are fast to calculate and beautifully normalized  |
//| (sum to one) but have following downsides:                       |
//|   * They ALWAYS sum to 1.0, even if output is completely         |
//|     unpredictable. I.e. MDI allows to order variables by         |
//|     importance, but does not  tell us about "absolute"           |
//|     importances of variables                                     |
//|   * there exist some bias towards continuous and high-cardinality|
//|     categorical variables                                        |
//| NOTE: informally speaking, MDA (permutation importance) rating   |
//|       answers the question  "what  part  of  the  model          |
//|       predictive power is ruined by permuting k-th variable?"    |
//|       while MDI tells us "what part of the model predictive power|
//|       was achieved due to usage of k-th variable".               |
//| Thus, MDA rates each variable independently at "0 to 1" scale    |
//| while MDI (and OOB-MDI too) tends to divide "unit amount of      |
//| importance" between several important variables.                 |
//| If all variables are equally important, they will have same      |
//| MDI/OOB-MDI rating, equal (for OOB-MDI: roughly equal)  to       |
//| 1/NVars. However, roughly  same  picture  will  be  produced for |
//| the "all variables provide information no one is critical"       |
//| situation and for the "all variables are critical, drop any one, |
//| everything is ruined" situation.                                 |
//| Contrary to that, MDA will rate critical variable as ~1.0        |
//| important, and important but non-critical variable will have less|
//| than unit rating.                                                |
//| NOTE: quite an often MDA and MDI return same results. It         |
//|       generally happens on problems with low test set error      |
//|       (a few percents at most) and large enough training set     |
//|       to avoid overfitting.                                      |
//| The difference between MDA, MDI and OOB-MDI becomes important    |
//| only on "hard" tasks with high test set error and/or small       |
//| training set.                                                    |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder object. Next call to the   |
//|               forest construction function will produce:         |
//|         * importance estimates in rep.varimportances field       |
//|         * variable ranks in rep.topvars field                    |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetImportanceTrnGini(CDecisionForestBuilder &s)
  {
   CDForest::DFBuilderSetImportanceTrnGini(s);
  }
//+------------------------------------------------------------------+
//| This function tells decision forest construction algorithm to use|
//| out-of-bag version of Gini variable importance estimation (also  |
//| known as OOB-MDI).                                               |
//| This version of importance estimation algorithm analyzes mean    |
//| decrease in impurity (MDI) on out-of-bag sample during splits.   |
//| The result is divided by impurity at the root node in order to   |
//| produce estimate in [0,1] range.                                 |
//| Such estimates are fast to calculate and resistant to overfitting|
//| issues (thanks to the out-of-bag estimates used). However, OOB   |
//| Gini rating has following downsides:                             |
//|         * there exist some bias towards continuous and           |
//|           high-cardinality categorical variables                 |
//|         * Gini rating allows us to order variables by importance,|
//|           but it is hard to define importance of the variable by |
//|           itself.                                                |
//| NOTE: informally speaking, MDA (permutation importance) rating   |
//|       answers the question "what part of the model predictive    |
//|       power is ruined by permuting k-th variable?" while MDI     |
//|       tells us "what part of the model predictive power was      |
//|       achieved due to usage of k-th variable".                   |
//| Thus, MDA rates each variable independently at "0 to 1" scale    |
//| while MDI (and OOB-MDI too) tends to divide "unit amount of      |
//| importance" between several important variables.                 |
//| If all variables are equally important, they will have same      |
//| MDI/OOB-MDI rating, equal (for OOB-MDI: roughly equal) to        |
//| 1/NVars. However, roughly same picture will be produced for the  |
//| "all variables provide information no one is critical" situation |
//| and for the "all variables are critical, drop any one, everything|
//| is ruined" situation.                                            |
//| Contrary to that, MDA will rate critical variable as ~1.0        |
//| important, and important but non-critical variable will have less|
//| than unit rating.                                                |
//| NOTE: quite an often MDA and MDI return same results. It         |
//|       generally happens on problems with low test set error      |
//|       (a few percents at most) and large enough training set to  |
//|       avoid overfitting.                                         |
//| The difference between MDA, MDI and OOB-MDI becomes important    |
//| only on "hard" tasks with high test set error and/or small       |
//| training set.                                                    |
//| INPUT PARAMETERS:                                                |
//|   S           -  decision forest builder object                  |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  decision forest builder object. Next call to the|
//|                  forest construction function will produce:      |
//|            * importance estimates in rep.varimportances field    |
//|            * variable ranks in rep.topvars field                 |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetImportanceOOBGini(CDecisionForestBuilder &s)
  {
   CDForest::DFBuilderSetImportanceOOBGini(s);
  }
//+------------------------------------------------------------------+
//| This function tells decision forest construction algorithm to use|
//| permutation variable importance estimator (also known as MDA).   |
//| This version of importance estimation algorithm analyzes mean    |
//| increase in out-of-bag sum of squared residuals after random     |
//| permutation of J-th variable. The result is divided by error     |
//| computed with all variables being perturbed in order to produce  |
//| R-squared-like estimate in [0,1] range.                          |
//| Such estimate is slower to calculate than Gini-based rating      |
//| because it needs multiple inference runs for each of variables   |
//| being studied.                                                   |
//| MDA rating has following benefits over Gini-based ones:          |
//|      * no bias towards specific variable types                   |
//|     *ability to directly evaluate "absolute" importance of some|
//|        variable at "0 to 1" scale (contrary to Gini-based rating,|
//|        which returns comparative importances).                   |
//| NOTE: informally speaking, MDA (permutation importance) rating   |
//|       answers the question "what part of the model predictive    |
//|       power is ruined by permuting k-th variable?" while MDI     |
//|       tells us "what part of the model predictive power was      |
//|       achieved due to usage of k-th variable".                   |
//| Thus, MDA rates each variable independently at "0 to 1" scale    |
//| while MDI (and OOB-MDI too) tends to divide "unit amount  of     |
//| importance" between several important variables.                 |
//| If all variables are equally important, they will have same      |
//| MDI/OOB-MDI rating, equal (for OOB-MDI: roughly equal)  to       |
//| 1/NVars. However, roughly same picture will be produced forthe   |
//| "all variables provide information no one is critical" situation |
//| and for the "all variables are critical, drop any one, everything|
//| is ruined" situation.                                            |
//| Contrary to that, MDA will rate critical variable as ~1.0        |
//| important, and important but non-critical variable will have less|
//| than unit rating.                                                |
//| NOTE: quite an often MDA and MDI return same results. It         |
//|       generally happens on problems with low test set error      |
//|       (a few percents at most) and large enough training set     |
//|       to avoid overfitting.                                      |
//| The difference between MDA, MDI and OOB-MDI becomes important    |
//| only on "hard" tasks with high test set error and/or small       |
//| training set.                                                    |
//| INPUT PARAMETERS:                                                |
//|   S           -  decision forest builder object                  |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  decision forest builder object. Next call to    |
//|                  the forest construction function will produce:  |
//|            * importance estimates in rep.varimportances field    |
//|            * variable ranks in rep.topvars field                 |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetImportancePermutation(CDecisionForestBuilder &s)
  {
   CDForest::DFBuilderSetImportancePermutation(s);
  }
//+------------------------------------------------------------------+
//| This function tells decision forest construction algorithm to    |
//| skip variable importance estimation.                             |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  decision forest builder object. Next call to the   |
//|               forest construction function will result in forest |
//|               being built without variable importance estimation.|
//+------------------------------------------------------------------+
void CAlglib::DFBuilderSetImportanceNone(CDecisionForestBuilder &s)
  {
   CDForest::DFBuilderSetImportanceNone(s);
  }
//+------------------------------------------------------------------+
//| This function is an alias for dfbuilderpeekprogress(), left in   |
//| ALGLIB for backward compatibility reasons.                       |
//+------------------------------------------------------------------+
double CAlglib::DFBuilderGetProgress(CDecisionForestBuilder &s)
  {
   return(CDForest::DFBuilderGetProgress(s));
  }
//+------------------------------------------------------------------+
//| This function is used to peek into decision forest construction  |
//| process from some other thread and get current progress indicator|
//| It returns value in [0,1].                                       |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object used to build forest|
//|               in some other thread                               |
//| RESULT:                                                          |
//|   progress value, in [0,1]                                       |
//+------------------------------------------------------------------+
double CAlglib::DFBuilderPeekProgress(CDecisionForestBuilder &s)
  {
   return(CDForest::DFBuilderPeekProgress(s));
  }
//+------------------------------------------------------------------+
//| This subroutine builds decision forest according to current      |
//| settings using dataset internally stored in the builder object.  |
//| Dense algorithm is used.                                         |
//| NOTE: this function uses dense algorithm for forest construction |
//|       independently from the dataset format (dense or sparse).   |
//| NOTE: forest built with this function is stored in-memory using  |
//|       64-bit data structures for offsets/indexes/split values. It|
//|       is possible to convert forest into more memory-efficient   |
//|       compressed binary representation. Depending on the problem |
//|       properties, 3.7x-5.7x compression factors are possible.    |
//| The downsides of compression are (a) slight reduction in  the    |
//| model accuracy and (b) ~1.5x reduction in the inference speed    |
//| (due to increased complexity of the storage format).             |
//| See comments on DFBinaryCompression() for more info.             |
//| Default settings are used by the algorithm; you can tweak them   |
//| with the help of the following functions:                        |
//|   * DFBuilderSetRFactor()    - to control a fraction of the      |
//|                                dataset used for subsampling      |
//|   * DFBuilderSetRandomVars() - to control number of variables    |
//|                                randomly chosen for decision rule |
//|                                creation                          |
//| INPUT PARAMETERS:                                                |
//|   S        -  decision forest builder object                     |
//|   NTrees   -  NTrees>=1, number of trees to train                |
//| OUTPUT PARAMETERS:                                               |
//|   D        -  decision forest. You can compress this forest to   |
//|               more compact 16-bit representation with            |
//|               DFBinaryCompression()                              |
//|   Rep      -  report, see below for information on its fields.   |
//| == report information produced by forest construction function = |
//| Decision forest training report includes following information:  |
//|      * training set errors                                       |
//|      * out-of-bag estimates of errors                            |
//|      * variable importance ratings                               |
//| Following fields are used to store information:                  |
//|   * training set errors are stored in rep.RelCLSError, rep.AvgCE,|
//|     rep.RMSError, rep.AvgError and rep.AvgRelError               |
//|   * out-of-bag estimates of errors are stored in                 |
//|     rep.oobrelclserror, rep.oobavgce, rep.oobrmserror,           |
//|     rep.oobavgerror and rep.oobavgrelerror                       |
//| Variable importance reports, if requested by                     |
//| DFBuilderSetImportanceGini(), DFBuilderSetImportanceTrnGini() or |
//| DFBuilderSetImportancePermutation() call, are stored in:         |
//|   * rep.varimportances field stores importance ratings           |
//|   * rep.topvars stores variable indexes ordered from the most    |
//|     important to less important ones                             |
//| You can find more information about report fields in:            |
//|   * comments on CDFReport structure                              |
//|   * comments on DFBuilderSetImportanceGini function              |
//|   * comments on DFBuilderSetImportanceTrnGini function           |
//|   * comments on DFBuilderDetImportancePermutation function       |
//+------------------------------------------------------------------+
void CAlglib::DFBuilderBuildRandomForest(CDecisionForestBuilder &s,int ntrees,
                                         CDecisionForestShell &df,
                                         CDFReportShell &rep)
  {
   CDForest::DFBuilderBuildRandomForest(s,ntrees,df.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function performs binary compression of the decision forest.|
//| Original decision forest produced by the forest builder is stored|
//| using 64-bit representation for all numbers - offsets, variable  |
//| indexes, split points.                                           |
//| It is possible to significantly reduce model size by means of:   |
//|   * using compressed dynamic encoding for integers (offsets and  |
//|     variable indexes), which uses just 1 byte to store small ints|
//|     (less than 128), just 2 bytes for larger values (less than   |
//|     128^2) and so on                                             |
//|   * storing floating point numbers using 8-bit exponent and      |
//|     16-bit mantissa                                              |
//| As result, model needs significantly less memory (compression    |
//| factor depends on  variable and class counts). In particular:    |
//|   * NVars<128 and NClasses<128 result in 4.4x-5.7x model size    |
//|     reduction                                                    |
//|   * NVars<16384 and NClasses<128 result in 3.7x-4.5x model size  |
//|     reduction                                                    |
//| Such storage format performs lossless compression of all integers|
//| but compression of floating point values (split values) is lossy,|
//| with roughly 0.01% relative error introduced during rounding.    |
//| Thus, we recommend you to re-evaluate model accuracy after       |
//| compression.                                                     |
//| Another downside of compression is ~1.5x reduction in the        |
//| inference speed due to necessity of dynamic decompression of the |
//| compressed model.                                                |
//| INPUT PARAMETERS:                                                |
//|   DF       -  decision forest built by forest builder            |
//| OUTPUT PARAMETERS:                                               |
//|   DF       -  replaced by compressed forest                      |
//| RESULT:                                                          |
//| compression factor (in-RAM size of the compressed model vs than  |
//| of the uncompressed one), positive number larger than 1.0        |
//+------------------------------------------------------------------+
double CAlglib::DFBinaryCompression(CDecisionForestShell &df)
  {
   return(CDForest::DFBinaryCompression(df.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| Procesing                                                        |
//| INPUT PARAMETERS:                                                |
//|     DF      -   decision forest model                            |
//|     X       -   input vector,  array[0..NVars-1].                |
//| OUTPUT PARAMETERS:                                               |
//|     Y       -   result. Regression estimate when solving         |
//|                 regression task, vector of posterior             |
//|                 probabilities for classification task.           |
//| See also DFProcessI.                                             |
//+------------------------------------------------------------------+
void CAlglib::DFProcess(CDecisionForestShell &df,double &x[],
                        double &y[])
  {
   CDForest::DFProcess(df.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| 'interactive' variant of DFProcess for languages like Python     |
//| which support constructs like "Y = DFProcessI(DF,X)" and         |
//| interactive mode of interpreter                                  |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart, but |
//| it is more convenient when you call it from command line.        |
//+------------------------------------------------------------------+
void CAlglib::DFProcessI(CDecisionForestShell &df,
                         double &x[],double &y[])
  {
   CDForest::DFProcessI(df.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//|  This function returns first component of the inferred vector    |
//| (i.e. one with index #0).                                        |
//| It is a convenience wrapper for dfprocess() intended for either: |
//|      * 1-dimensional regression problems                         |
//|      * 2-class classification problems                           |
//| In the former case this function returns inference result as     |
//| scalar, which is definitely more convenient that wrapping it as  |
//| vector. In the latter case it returns probability of object      |
//| belonging to class #0.                                           |
//| If you call it for anything different from two cases above, it   |
//| will work as defined, i.e. return y[0], although it is of less   |
//| use in such cases.                                               |
//| INPUT PARAMETERS:                                                |
//|   Model    -  DF model                                           |
//|   X        -  input vector,  array[0..NVars-1].                  |
//| RESULT:                                                          |
//|   Y[0]                                                           |
//+------------------------------------------------------------------+
double CAlglib::DFProcess0(CDecisionForestShell &model,double &X[])
  {
   CRowDouble x=X;
   return(CDForest::DFProcess0(model.GetInnerObj(),x));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CAlglib::DFProcess0(CDecisionForestShell &model,CRowDouble &x)
  {
   return(CDForest::DFProcess0(model.GetInnerObj(),x));
  }
//+------------------------------------------------------------------+
//| This function returns most probable class number for an input X. |
//| It is same as calling DFProcess(model,x,y), then determining     |
//| i=ArgMax(y[i]) and returning i.                                  |
//| A class number in [0,NOut) range in returned for classification  |
//| problems, -1 is returned when this function is called for        |
//| regression problems.                                             |
//| INPUT PARAMETERS:                                                |
//|   Model    -  decision forest model                              |
//|   X        -  input vector,  array[0..NVars-1].                  |
//| RESULT:                                                          |
//|   class number, -1 for regression tasks                          |
//+------------------------------------------------------------------+
int CAlglib::DFClassify(CDecisionForestShell &model,double &X[])
  {
   CRowDouble x=X;
   return(CDForest::DFClassify(model.GetInnerObj(),x));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CAlglib::DFClassify(CDecisionForestShell &model,CRowDouble &x)
  {
   return(CDForest::DFClassify(model.GetInnerObj(),x));
  }
//+------------------------------------------------------------------+
//| Relative classification error on the test set                    |
//| INPUT PARAMETERS:                                                |
//|     DF      -   decision forest model                            |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     percent of incorrectly classified cases.                     |
//|     Zero if model solves regression task.                        |
//+------------------------------------------------------------------+
double CAlglib::DFRelClsError(CDecisionForestShell &df,CMatrixDouble &xy,
                              const int npoints)
  {
   return(CDForest::DFRelClsError(df.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average cross-entropy (in bits per element) on the test set      |
//| INPUT PARAMETERS:                                                |
//|     DF      -   decision forest model                            |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     CrossEntropy/(NPoints*LN(2)).                                |
//|     Zero if model solves regression task.                        |
//+------------------------------------------------------------------+
double CAlglib::DFAvgCE(CDecisionForestShell &df,CMatrixDouble &xy,
                        const int npoints)
  {
   return(CDForest::DFAvgCE(df.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| RMS error on the test set                                        |
//| INPUT PARAMETERS:                                                |
//|     DF      -   decision forest model                            |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     root mean square error.                                      |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task,RMS error means error when estimating    |
//|     posterior probabilities.                                     |
//+------------------------------------------------------------------+
double CAlglib::DFRMSError(CDecisionForestShell &df,CMatrixDouble &xy,
                           const int npoints)
  {
   return(CDForest::DFRMSError(df.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average error on the test set                                    |
//| INPUT PARAMETERS:                                                |
//|     DF      -   decision forest model                            |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task, it means average error when estimating  |
//|     posterior probabilities.                                     |
//+------------------------------------------------------------------+
double CAlglib::DFAvgError(CDecisionForestShell &df,CMatrixDouble &xy,
                           const int npoints)
  {
   return(CDForest::DFAvgError(df.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average relative error on the test set                           |
//| INPUT PARAMETERS:                                                |
//|     DF      -   decision forest model                            |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task, it means average relative error when    |
//|     estimating posterior probability of belonging to the correct |
//|     class.                                                       |
//+------------------------------------------------------------------+
double CAlglib::DFAvgRelError(CDecisionForestShell &df,CMatrixDouble &xy,
                              const int npoints)
  {
   return(CDForest::DFAvgRelError(df.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| This subroutine builds random decision forest.                   |
//| ---- DEPRECATED VERSION! USE DECISION FOREST BUILDER OBJECT ---- |
//+------------------------------------------------------------------+
void CAlglib::DFBuildRandomDecisionForest(CMatrixDouble &xy,const int npoints,
                                          const int nvars,const int nclasses,
                                          const int ntrees,const double r,
                                          int &info,CDecisionForestShell &df,
                                          CDFReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CDForest::DFBuildRandomDecisionForest(xy,npoints,nvars,nclasses,ntrees,r,info,df.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds random decision forest.                   |
//| ---- DEPRECATED VERSION! USE DECISION FOREST BUILDER OBJECT ---- |
//+------------------------------------------------------------------+
void CAlglib::DFBuildRandomDecisionForestX1(CMatrixDouble &xy,
                                            const int npoints,
                                            const int nvars,
                                            const int nclasses,
                                            const int ntrees,
                                            int nrndvars,
                                            const double r,
                                            int &info,
                                            CDecisionForestShell &df,
                                            CDFReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CDForest::DFBuildRandomDecisionForestX1(xy,npoints,nvars,nclasses,ntrees,nrndvars,r,info,df.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function initializes clusterizer object. Newly initialized  |
//| object is empty, i.e. it does not contain dataset. You should    |
//| use it as follows:                                               |
//|   1. creation                                                    |
//|   2. dataset is added with ClusterizerSetPoints()                |
//|   3. additional parameters are set                               |
//|   3. clusterization is performed with one of the clustering      |
//|      functions                                                   |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerCreate(CClusterizerState &s)
  {
   CClustering::ClusterizerCreate(s);
  }
//+------------------------------------------------------------------+
//| This function adds dataset to the clusterizer structure.         |
//| This function overrides all previous calls of                    |
//| ClusterizerSetPoints() or ClusterizerSetDistances().             |
//| INPUT PARAMETERS:                                                |
//|   S        -  clusterizer state, initialized by                  |
//|               ClusterizerCreate()                                |
//|   XY       -  array[NPoints,NFeatures], dataset                  |
//|   NPoints  -  number of points, >=0                              |
//|   NFeatures-  number of features, >=1                            |
//|   DistType -  distance function:                                 |
//|               *  0    Chebyshev distance  (L-inf norm)           |
//|               *  1    city block distance (L1 norm)              |
//|               *  2    Euclidean distance  (L2 norm), non-squared |
//|               * 10    Pearson correlation:                       |
//|                       dist(a,b) = 1-corr(a,b)                    |
//|               * 11    Absolute Pearson correlation:              |
//|                       dist(a,b) = 1-|corr(a,b)|                  |
//|               * 12    Uncentered Pearson correlation (cosine of  |
//|                       the angle): dist(a,b) = a'*b/(|a|*|b|)     |
//|               * 13    Absolute uncentered Pearson correlation    |
//|                       dist(a,b) = |a'*b|/(|a|*|b|)               |
//|               * 20    Spearman rank correlation:                 |
//|                       dist(a,b) = 1-rankcorr(a,b)                |
//|               * 21    Absolute Spearman rank correlation         |
//|                       dist(a,b) = 1-|rankcorr(a,b)|              |
//| NOTE 1: different distance functions have different performance  |
//|         penalty:                                                 |
//|         * Euclidean or Pearson correlation distances are         |
//|           the fastest ones                                       |
//|         * Spearman correlation distance function is a bit slower |
//|         * city block and Chebyshev distances are order           |
//|           of magnitude slower                                    |
//|         The reason behing difference in performance is that      |
//|         correlation-based distance functions are computed using  |
//|         optimized linear algebra kernels, while Chebyshev and    |
//|         city block distance functions are computed using simple  |
//|         nested loops with two branches at each iteration.        |
//| NOTE 2: different clustering algorithms have different           |
//|         limitations:                                             |
//|         * agglomerative hierarchical clustering algorithms may   |
//|           be used with any kind of distance metric               |
//|         * k-means++ clustering algorithm may be used only with   |
//|           Euclidean distance function                            |
//|         Thus, list of specific clustering algorithms you may use |
//|         depends on distance function you specify when you set    |
//|         your dataset.                                            |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSetPoints(CClusterizerState &s,CMatrixDouble &xy,
                                   int npoints,int nfeatures,int disttype)
  {
   CClustering::ClusterizerSetPoints(s,xy,npoints,nfeatures,disttype);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSetPoints(CClusterizerState &s,CMatrixDouble &xy,int disttype)
  {
//--- create variables
   int npoints=CAp::Rows(xy);
   int nfeatures=CAp::Cols(xy);
//--- function call
   CClustering::ClusterizerSetPoints(s,xy,npoints,nfeatures,disttype);
  }
//+------------------------------------------------------------------+
//| This function adds dataset given by distance matrix to the       |
//| clusterizer structure. It is important that dataset is not given |
//| explicitly - only distance matrix is given.                      |
//| This function overrides all previous calls  of                   |
//| ClusterizerSetPoints() or ClusterizerSetDistances().             |
//| INPUT PARAMETERS:                                                |
//|   S        -  clusterizer state, initialized by                  |
//|               ClusterizerCreate()                                |
//|   D        -  array[NPoints,NPoints], distance matrix given by   |
//|               its upper or lower triangle (main diagonal is      |
//|               ignored because its entries are expected to        |
//|               be zero).                                          |
//|   NPoints  -  number of points                                   |
//|   IsUpper  -  whether upper or lower triangle of D is given.     |
//| NOTE 1: different clustering algorithms have different           |
//|         limitations:                                             |
//|         * agglomerative hierarchical clustering algorithms may   |
//|           be used with any kind of distance metric, including    |
//|           one which is given by distance matrix                  |
//|         * k-means++ clustering algorithm may be used only with   |
//|           Euclidean distance function and explicitly given       |
//|           points - it can not be used with dataset given by      |
//|           distance matrix. Thus, if you call this function, you  |
//|           will be unable to use k-means clustering algorithm     |
//|           to process your problem.                               |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSetDistances(CClusterizerState &s,CMatrixDouble &d,int npoints,bool IsUpper)
  {
   CClustering::ClusterizerSetDistances(s,d,npoints,IsUpper);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSetDistances(CClusterizerState &s,CMatrixDouble &d,bool IsUpper)
  {
//--- check
   if(!CAp::Assert(CAp::Rows(d)==CAp::Cols(d),"Error while calling 'ClusterizerSetDistances': looks like one of arguments has wrong size"))
      return;
//--- initialization
   int npoints=CAp::Rows(d);
//--- function call
   CClustering::ClusterizerSetDistances(s,d,npoints,IsUpper);
  }
//+------------------------------------------------------------------+
//| This function sets agglomerative hierarchical clustering         |
//| algorithm                                                        |
//| INPUT PARAMETERS:                                                |
//|   S     -  clusterizer state, initialized by ClusterizerCreate() |
//|   Algo  -  algorithm type:                                       |
//|            * 0     complete linkage(default algorithm)           |
//|            * 1     single linkage                                |
//|            * 2     unweighted average linkage                    |
//|            * 3     weighted average linkage                      |
//|            * 4     Ward's method                                 |
//| NOTE: Ward's method works correctly only with Euclidean distance,|
//|       that's why algorithm will return negative termination      |
//|       code(failure) for any other distance type.                 |
//| It is possible, however, to use this method with user - supplied |
//| distance matrix. It is your responsibility to pass one which was |
//| calculated with Euclidean distance function.                     |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSetAHCAlgo(CClusterizerState &s,int algo)
  {
   CClustering::ClusterizerSetAHCAlgo(s,algo);
  }
//+------------------------------------------------------------------+
//| This  function  sets k-means properties:                         |
//|      number  of  restarts and maximum                            |
//|      number of iterations per one run.                           |
//| INPUT PARAMETERS:                                                |
//|   S        -  clusterizer state, initialized by                  |
//|               ClusterizerCreate()                                |
//|   Restarts -  restarts count, >= 1.                              |
//|               k-means++ algorithm performs several restarts      |
//|               and chooses best set of centers(one with minimum   |
//|               squared distance).                                 |
//|   MaxIts   -  maximum number of k-means iterations performed     |
//|               during one run. >= 0, zero value means that        |
//|               algorithm performs unlimited number of iterations. |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSetKMeansLimits(CClusterizerState &s,int restarts,int maxits)
  {
   CClustering::ClusterizerSetKMeansLimits(s,restarts,maxits);
  }
//+------------------------------------------------------------------+
//| This function sets k-means initialization algorithm. Several     |
//| different algorithms can be chosen, including k-means++.         |
//| INPUT PARAMETERS:                                                |
//|   S        -  clusterizer state, initialized by                  |
//|               ClusterizerCreate()                                |
//|   InitAlgo -  initialization algorithm:                          |
//|      * 0  automatic selection(different  versions  of  ALGLIB    |
//|           may select different algorithms)                       |
//|      * 1  random initialization                                  |
//|      * 2  k-means++ initialization(best  quality  of  initial    |
//|           centers, but long  non-parallelizable initialization   |
//|           phase with bad cache locality)                         |
//|     *3  "fast-greedy" algorithm with efficient, easy to        |
//|           parallelize initialization. Quality of initial centers |
//|           is somewhat worse than that of k-means++. This         |
//|           algorithm is a default one in the current version of   |
//|           ALGLIB.                                                |
//|     *-1 "debug" algorithm which always selects first K rows    |
//|           of dataset; this algorithm is used for debug purposes  |
//|           only. Do not use it in the industrial code!            |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSetKMeansInit(CClusterizerState &s,int initalgo)
  {
   CClustering::ClusterizerSetKMeansInit(s,initalgo);
  }
//+------------------------------------------------------------------+
//| This function sets seed which is used to initialize internal RNG.|
//| By default, deterministic seed is used - same for each run of    |
//| clusterizer. If you specify non-deterministic seed value, then   |
//| some algorithms which depend on random initialization(in current |
//| version : k-means) may return slightly different results after   |
//| each run.                                                        |
//| INPUT PARAMETERS:                                                |
//|   S        -  clusterizer state, initialized by                  |
//|               ClusterizerCreate()                                |
//|   Seed     -  seed:                                              |
//|               * positive values = use deterministic seed for each|
//|                 run of algorithms which depend on random         |
//|                 initialization                                   |
//|               * zero or negative values = use non-deterministic  |
//|                 seed                                             |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSetSeed(CClusterizerState &s,int seed)
  {
   CClustering::ClusterizerSetSeed(s,seed);
  }
//+------------------------------------------------------------------+
//| This function performs agglomerative hierarchical clustering     |
//| NOTE: Agglomerative hierarchical clustering algorithm has two    |
//|       phases: distance matrix calculation and clustering itself. |
//| INPUT PARAMETERS:                                                |
//|   S        -  clusterizer state, initialized by                  |
//|               ClusterizerCreate()                                |
//| OUTPUT PARAMETERS:                                               |
//|   Rep      -  clustering results; see description of AHCReport   |
//|               structure for more information.                    |
//| NOTE 1: hierarchical clustering algorithms require large amounts |
//|         of memory. In particular, this implementation needs      |
//|         sizeof(double) *NPoints^2 bytes, which are used to store |
//|         distance matrix. In case we work with user - supplied    |
//|         matrix, this amount is multiplied by 2 (we have to store |
//|         original matrix and to work with its copy).              |
//|         For example, problem with 10000 points would require 800M|
//|         of RAM, even when working in a 1-dimensional space.      |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerRunAHC(CClusterizerState &s,CAHCReport &rep)
  {
   CClustering::ClusterizerRunAHC(s,rep);
  }
//+------------------------------------------------------------------+
//| This function performs clustering by k-means++ algorithm.        |
//| You may change algorithm properties by calling:                  |
//|      * ClusterizerSetKMeansLimits() to change number of restarts |
//|        or iterations                                             |
//|      * ClusterizerSetKMeansInit() to change initialization       |
//|        algorithm                                                 |
//| By default, one restart and unlimited number of iterations are   |
//| used. Initialization algorithm is chosen automatically.          |
//| NOTE: k-means clustering algorithm has two phases: selection of  |
//|       initial centers and clustering itself.                     |
//| INPUT PARAMETERS:                                                |
//|   S        -  clusterizer state, initialized by                  |
//|               ClusterizerCreate()                                |
//|   K        -  number of clusters, K >= 0.                        |
//|               K can be zero only when algorithm is called for    |
//|               empty dataset, in this case completion code is set |
//|               to success(+1).                                    |
//|               If K = 0 and dataset size is non-zero, we can not  |
//|               meaningfully assign points to some center(there are|
//|               no centers because K = 0) and return -3 as         |
//|               completion code (failure).                         |
//| OUTPUT PARAMETERS:                                               |
//|   Rep      -  clustering results; see description of KMeansReport|
//|               structure for more information.                    |
//| NOTE 1: k-means clustering can be performed only for datasets    |
//|         with Euclidean distance function. Algorithm will return  |
//|         negative completion code in Rep.TerminationType in case  |
//|         dataset was added to clusterizer with DistType other     |
//|         than Euclidean (or dataset was specified by distance     |
//|         matrix instead of explicitly given points).              |
//| NOTE 2: by default, k-means uses non-deterministic seed to       |
//|         initialize RNG which is used to select initial centers.  |
//|         As result, each run of algorithm may return different    |
//|         values. If you  need  deterministic behavior, use        |
//|         ClusterizerSetSeed() function.                           |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerRunKMeans(CClusterizerState &s,int k,CKmeansReport &rep)
  {
   CClustering::ClusterizerRunKMeans(s,k,rep);
  }
//+------------------------------------------------------------------+
//| This function returns distance matrix for dataset                |
//| INPUT PARAMETERS:                                                |
//|   XY       -  array[NPoints, NFeatures], dataset                 |
//|   NPoints  -  number of points, >= 0                             |
//|   NFeatures-  number of features, >= 1                           |
//|   DistType -  distance function:                                 |
//|               *  0    Chebyshev distance(L - inf norm)           |
//|               *  1    city block distance(L1 norm)               |
//|               *  2    Euclidean distance(L2 norm, non - squared) |
//|               * 10    Pearson correlation:                       |
//|                       dist(a, b) = 1 - corr(a, b)                |
//|               * 11    Absolute Pearson correlation:              |
//|                       dist(a, b) = 1 - |corr(a, b)|              |
//|               * 12    Uncentered Pearson correlation(cosine of   |
//|                       the angle): dist(a, b) = a'*b/(|a|*|b|)    |
//|               * 13    Absolute uncentered Pearson correlation    |
//|                       dist(a, b) = |a'*b|/(|a|*|b|)              |
//|               * 20    Spearman rank correlation:                 |
//|                       dist(a, b) = 1 - rankcorr(a, b)            |
//|               * 21    Absolute Spearman rank correlation         |
//|                       dist(a, b) = 1 - |rankcorr(a, b)|          |
//| OUTPUT PARAMETERS:                                               |
//|   D        -  array[NPoints, NPoints], distance matrix (full     |
//|               matrix is returned, with lower and upper triangles)|
//| NOTE: different distance functions have different performance    |
//|       penalty:                                                   |
//|      * Euclidean or Pearson correlation distances are the fastest|
//|        ones                                                      |
//|      * Spearman correlation distance function is a bit slower    |
//|      * city block and Chebyshev distances are order of magnitude |
//|        slower                                                    |
//| The reason behing difference in performance is that correlation -|
//| based distance functions are computed using optimized linear     |
//| algebra kernels, while Chebyshev and city block distance         |
//| functions are computed using simple nested loops with two        |
//| branches at each iteration.                                      |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerGetDistances(CMatrixDouble &xy,int npoints,
                                      int nfeatures,int disttype,
                                      CMatrixDouble &d)
  {
   d.Resize(0,0);
   CClustering::ClusterizerGetDistances(xy,npoints,nfeatures,disttype,d);
  }
//+------------------------------------------------------------------+
//| This function takes as input clusterization report Rep, desired  |
//| clusters count K, and builds top K clusters from hierarchical    |
//| clusterization tree.                                             |
//| It returns assignment of points to clusters(array of cluster     |
//| indexes).                                                        |
//| INPUT PARAMETERS:                                                |
//|   Rep     -  report from ClusterizerRunAHC() performed on XY     |
//|   K       -   desired number of clusters, 1 <= K <= NPoints.     |
//|               K can be zero only when NPoints = 0.               |
//| OUTPUT PARAMETERS:                                               |
//|   CIdx    -   array[NPoints], I-th element contains cluster      |
//|               index(from 0 to K-1) for I-th point of the dataset.|
//|   CZ      -   array[K]. This array allows  to  convert  cluster  |
//|               indexes returned by this function to indexes used  |
//|               by Rep.Z. J-th cluster returned by this function   |
//|               corresponds to CZ[J]-th cluster stored in          |
//|               Rep.Z/PZ/PM. It is guaranteed that CZ[I] < CZ[I+1].|
//| NOTE: K clusters built by this subroutine are assumed to have no |
//|       hierarchy. Although they were obtained by manipulation with|
//|       top K nodes of dendrogram(i.e. hierarchical decomposition  |
//|       of dataset), this function does not return information     |
//|       about hierarchy. Each of the clusters stand on its own.    |
//| NOTE: Cluster indexes returned by this function does not         |
//|       correspond to indexes returned in Rep.Z/PZ/PM. Either you  |
//|       work with hierarchical representation of the dataset       |
//|       (dendrogram), or you work with "flat" representation       |
//|       returned by this function. Each of representations has its |
//|       own clusters indexing system(former uses [0,2*NPoints-2]), |
//|       while latter uses [0..K-1]), although it is possible to    |
//|       perform conversion from one system to another by means of  |
//|       CZ array, returned by this function, which allows you to   |
//|       convert indexes stored in CIdx to the numeration system    |
//|       used by Rep.Z.                                             |
//| NOTE: this subroutine is optimized for moderate values of K.     |
//|       Say, for K=5 it will perform many times faster than for    |
//|       K=100. Its worst - case performance is O(N*K), although in |
//|       average case it perform better (up to O(N*log(K))).        |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerGetKClusters(CAHCReport &rep,int k,CRowInt &cidx,CRowInt &cz)
  {
   cidx.Resize(0);
   cz.Resize(0);
   CClustering::ClusterizerGetKClusters(rep,k,cidx,cz);
  }
//+------------------------------------------------------------------+
//| This function accepts AHC report Rep, desired minimum            |
//| intercluster distance and returns top clusters from hierarchical |
//| clusterization tree which are separated by distance R or HIGHER. |
//| It returns assignment of points to clusters (array of cluster    |
//| indexes).                                                        |
//| There is one more function with similar name -                   |
//| ClusterizerSeparatedByCorr, which returns clusters with          |
//| intercluster correlation equal to R or LOWER (note: higher for   |
//| distance, lower for correlation).                                |
//| INPUT PARAMETERS:                                                |
//|   Rep      -  report from ClusterizerRunAHC() performed on XY    |
//|   R        -   desired minimum intercluster distance, R >= 0     |
//| OUTPUT PARAMETERS:                                               |
//|   K        -  number of clusters, 1 <= K <= NPoints              |
//|   CIdx     -  array[NPoints], I-th element contains cluster      |
//|               index (from 0 to K-1) for I-th point of the dataset|
//|   CZ       -  array[K]. This array allows to convert cluster     |
//|               indexes returned by this function to indexes used  |
//|               by Rep.Z. J-th cluster returned by this function   |
//|               corresponds to CZ[J]-th cluster stored in          |
//|               Rep.Z/PZ/PM. It is guaranteed that CZ[I] < CZ[I+1].|
//| NOTE: K clusters built by this subroutine are assumed to have no |
//|       hierarchy. Although they were obtained by manipulation with|
//|       top K nodes of dendrogram (i.e. hierarchical decomposition |
//|       of dataset), this function does not return information     |
//|       about hierarchy. Each of the clusters stand on its own.    |
//| NOTE: Cluster indexes returned by this function does not         |
//|       correspond to indexes returned in Rep.Z/PZ/PM. Either you  |
//|       work with hierarchical representation of the dataset       |
//|       (dendrogram), or you work with "flat" representation       |
//|       returned by this function. Each of representations has its |
//|       own clusters indexing system (former uses [0,2*NPoints-2]),|
//|       while latter uses [0..K-1]), although it is possible to    |
//|       perform conversion from one system to another by means of  |
//|       CZ array, returned by this function, which allows you to   |
//|       convert indexes stored in CIdx to the numeration system    |
//|       used by Rep.Z.                                             |
//| NOTE: this subroutine is optimized for moderate values of K. Say,|
//|       for K=5 it will perform many times faster than for K=100.  |
//|       Its worst - case performance is O(N*K), although in average|
//|       case it perform better (up to O(N*log(K))).                |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSeparatedByDist(CAHCReport &rep,double r,int &k,
                                         CRowInt &cidx,CRowInt &cz)
  {
   k=0;
   cidx.Resize(0);
   cz.Resize(0);
   CClustering::ClusterizerSeparatedByDist(rep,r,k,cidx,cz);
  }
//+------------------------------------------------------------------+
//| This function accepts AHC report Rep, desired maximum            |
//| intercluster correlation and returns top clusters from           |
//| hierarchical clusterization tree which are separated by          |
//| correlation R or LOWER.                                          |
//| It returns assignment of points to clusters(array of cluster     |
//| indexes).                                                        |
//| There is one more function with similar name -                   |
//| ClusterizerSeparatedByDist, which returns clusters with          |
//| intercluster distance equal to R or HIGHER (note: higher for     |
//| distance, lower for correlation).                                |
//| INPUT PARAMETERS:                                                |
//|   Rep      -  report from ClusterizerRunAHC() performed on XY    |
//|   R        -  desired maximum intercluster correlation, -1<=R<=+1|
//| OUTPUT PARAMETERS:                                               |
//|   K        -  number of clusters, 1 <= K <= NPoints              |
//|   CIdx     -  array[NPoints], I-th element contains cluster index|
//|               (from 0 to K-1) for I-th point of the dataset.     |
//|   CZ       -  array[K]. This array allows to convert cluster     |
//|               indexes returned by this function to indexes used  |
//|               by Rep.Z. J-th cluster returned by this function   |
//|               corresponds to CZ[J]-th cluster stored in          |
//|               Rep.Z/PZ/PM. It is guaranteed that CZ[I] < CZ[I+1].|
//| NOTE: K clusters built by this subroutine are assumed to have no |
//|       hierarchy. Although they were obtained by manipulation with|
//|       top K nodes of dendrogram (i.e. hierarchical decomposition |
//|       of dataset), this function does not return information     |
//|       about hierarchy. Each of the clusters stand on its own.    |
//| NOTE: Cluster indexes returned by this function does not         |
//|       correspond to indexes returned in Rep.Z/PZ/PM. Either you  |
//|       work with hierarchical representation of the dataset       |
//|       (dendrogram), or you work with "flat" representation       |
//|       returned by this function. Each of representations has its |
//|       own clusters indexing system (former uses [0,2*NPoints-2]),|
//|       while latter uses [0..K-1]), although it is possible to    |
//|       perform conversion from one system to another by means of  |
//|       CZ array, returned by this function, which allows you to   |
//|       convert indexes stored in CIdx to the numeration system    |
//|       used by Rep.Z.                                             |
//| NOTE: this subroutine is optimized for moderate values of K. Say,|
//|       for K=5 it will perform many times faster than for K=100.  |
//|       Its worst - case performance is O(N*K), although in average|
//|       case it perform better (up to O(N*log(K))).                |
//+------------------------------------------------------------------+
void CAlglib::ClusterizerSeparatedByCorr(CAHCReport &rep,double r,
                                         int &k,CRowInt &cidx,CRowInt &cz)
  {
   k=0;
   cidx.Resize(0);
   cz.Resize(0);
   CClustering::ClusterizerSeparatedByCorr(rep,r,k,cidx,cz);
  }
//+------------------------------------------------------------------+
//| k-means++ clusterization                                         |
//| Backward compatibility function, we recommend to use CLUSTERING  |
//| subpackage as better replacement.                                |
//| INPUT PARAMETERS:                                                |
//|     XY          -   dataset, array [0..NPoints-1,0..NVars-1].    |
//|     NPoints     -   dataset size, NPoints>=K                     |
//|     NVars       -   number of variables, NVars>=1                |
//|     K           -   desired number of clusters, K>=1             |
//|     Restarts    -   number of restarts, Restarts>=1              |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code:                                 |
//|                     * -3, if task is degenerate (number of       |
//|                           distinct points is less than K)        |
//|                     * -1, if incorrect                           |
//|                           NPoints/NFeatures/K/Restarts was passed|
//|                     *  1, if subroutine finished successfully    |
//|     C           -   array[0..NVars-1,0..K-1].matrix whose columns|
//|                     store cluster's centers                      |
//|     XYC         -   array[NPoints], which contains cluster       |
//|                     indexes                                      |
//+------------------------------------------------------------------+
void CAlglib::KMeansGenerate(CMatrixDouble &xy,const int npoints,
                             const int nvars,const int k,
                             const int restarts,int &info,
                             CMatrixDouble &c,int &xyc[])
  {
//--- initialization
   info=0;
//--- function call
   CKMeans::KMeansGenerate(xy,npoints,nvars,k,restarts,info,c,xyc);
  }
//+------------------------------------------------------------------+
//| Multiclass Fisher LDA                                            |
//| Subroutine finds coefficients of linear combination which        |
//| optimally separates training set on classes.                     |
//| INPUT PARAMETERS:                                                |
//|     XY          -   training set, array[0..NPoints-1,0..NVars].  |
//|                     First NVars columns store values of          |
//|                     independent variables, next column stores    |
//|                     number of class (from 0 to NClasses-1) which |
//|                     dataset element belongs to. Fractional values|
//|                     are rounded to nearest integer.              |
//|     NPoints     -   training set size, NPoints>=0                |
//|     NVars       -   number of independent variables, NVars>=1    |
//|     NClasses    -   number of classes, NClasses>=2               |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code:                                 |
//|                     * -4, if internal EVD subroutine hasn't      |
//|                           converged                              |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NClasses-1].            |
//|                     * -1, if incorrect parameters was passed     |
//|                           (NPoints<0, NVars<1, NClasses<2)       |
//|                     *  1, if task has been solved                |
//|                     *  2, if there was a multicollinearity in    |
//|                           training set, but task has been solved.|
//|     W           -   linear combination coefficients,             |
//|                     array[0..NVars-1]                            |
//+------------------------------------------------------------------+
void CAlglib::FisherLDA(CMatrixDouble &xy,const int npoints,
                        const int nvars,const int nclasses,
                        int &info,double &w[])
  {
//--- initialization
   info=0;
//--- function call
   CLDA::FisherLDA(xy,npoints,nvars,nclasses,info,w);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FisherLDA(CMatrixDouble &xy,const int npoints,
                        const int nvars,const int nclasses,
                        int &info,CRowDouble &w)
  {
//--- initialization
   info=0;
//--- function call
   CLDA::FisherLDA(xy,npoints,nvars,nclasses,info,w);
  }
//+------------------------------------------------------------------+
//| N-dimensional multiclass Fisher LDA                              |
//| Subroutine finds coefficients of linear combinations which       |
//| optimally separates                                              |
//| training set on classes. It returns N-dimensional basis whose    |
//| vector are sorted                                                |
//| by quality of training set separation (in descending order).     |
//| INPUT PARAMETERS:                                                |
//|     XY          -   training set, array[0..NPoints-1,0..NVars].  |
//|                     First NVars columns store values of          |
//|                     independent variables, next column stores    |
//|                     number of class (from 0 to NClasses-1) which |
//|                     dataset element belongs to. Fractional values|
//|                     are rounded to nearest integer.              |
//|     NPoints     -   training set size, NPoints>=0                |
//|     NVars       -   number of independent variables, NVars>=1    |
//|     NClasses    -   number of classes, NClasses>=2               |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code:                                 |
//|                     * -4, if internal EVD subroutine hasn't      |
//|                           converged                              |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NClasses-1].            |
//|                     * -1, if incorrect parameters was passed     |
//|                           (NPoints<0, NVars<1, NClasses<2)       |
//|                     *  1, if task has been solved                |
//|                     *  2, if there was a multicollinearity in    |
//|                           training set, but task has been solved.|
//|     W           -   basis, array[0..NVars-1,0..NVars-1]          |
//|                     columns of matrix stores basis vectors,      |
//|                     sorted by quality of training set separation |
//|                     (in descending order)                        |
//+------------------------------------------------------------------+
void CAlglib::FisherLDAN(CMatrixDouble &xy,const int npoints,
                         const int nvars,const int nclasses,
                         int &info,CMatrixDouble &w)
  {
//--- initialization
   info=0;
//--- function call
   CLDA::FisherLDAN(xy,npoints,nvars,nclasses,info,w);
  }
//+------------------------------------------------------------------+
//| Linear regression                                                |
//| Subroutine builds model:                                         |
//|     Y = A(0)*X[0] + ... + A(N-1)*X[N-1] + A(N)                   |
//| and model found in ALGLIB format, covariation matrix, training   |
//| set errors (rms, average, average relative) and leave-one-out    |
//| cross-validation estimate of the generalization error. CV        |
//| estimate calculated using fast algorithm with O(NPoints*NVars)   |
//| complexity.                                                      |
//| When  covariation  matrix  is  calculated  standard deviations of|
//| function values are assumed to be equal to RMS error on the      |
//| training set.                                                    |
//| INPUT PARAMETERS:                                                |
//|     XY          -   training set, array [0..NPoints-1,0..NVars]: |
//|                     * NVars columns - independent variables      |
//|                     * last column - dependent variable           |
//|     NPoints     -   training set size, NPoints>NVars+1           |
//|     NVars       -   number of independent variables              |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code:                                 |
//|                     * -255, in case of unknown internal error    |
//|                     * -4, if internal SVD subroutine haven't     |
//|                           converged                              |
//|                     * -1, if incorrect parameters was passed     |
//|                           (NPoints<NVars+2, NVars<1).            |
//|                     *  1, if subroutine successfully finished    |
//|     LM          -   linear model in the ALGLIB format. Use       |
//|                     subroutines of this unit to work with the    |
//|                     model.                                       |
//|     AR          -   additional results                           |
//+------------------------------------------------------------------+
void CAlglib::LRBuild(CMatrixDouble &xy,const int npoints,const int nvars,
                      int &info,CLinearModelShell &lm,CLRReportShell &ar)
  {
//--- initialization
   info=0;
//--- function call
   CLinReg::LRBuild(xy,npoints,nvars,info,lm.GetInnerObj(),ar.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Linear regression                                                |
//| Variant of LRBuild which uses vector of standatd deviations      |
//| (errors in function values).                                     |
//| INPUT PARAMETERS:                                                |
//|     XY          -   training set, array [0..NPoints-1,0..NVars]: |
//|                     * NVars columns - independent variables      |
//|                     * last column - dependent variable           |
//|     S           -   standard deviations (errors in function      |
//|                     values) array[0..NPoints-1], S[i]>0.         |
//|     NPoints     -   training set size, NPoints>NVars+1           |
//|     NVars       -   number of independent variables              |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code:                                 |
//|                     * -255, in case of unknown internal error    |
//|                     * -4, if internal SVD subroutine haven't     |
//|                     converged                                    |
//|                     * -1, if incorrect parameters was passed     |
//|                     (NPoints<NVars+2, NVars<1).                  |
//|                     * -2, if S[I]<=0                             |
//|                     *  1, if subroutine successfully finished    |
//|     LM          -   linear model in the ALGLIB format. Use       |
//|                     subroutines of this unit to work with the    |
//|                     model.                                       |
//|     AR          -   additional results                           |
//+------------------------------------------------------------------+
void CAlglib::LRBuildS(CMatrixDouble &xy,double &s[],const int npoints,
                       const int nvars,int &info,CLinearModelShell &lm,
                       CLRReportShell &ar)
  {
//--- initialization
   info=0;
//--- function call
   CLinReg::LRBuildS(xy,s,npoints,nvars,info,lm.GetInnerObj(),ar.GetInnerObj());
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::LRBuildS(CMatrixDouble &xy,CRowDouble &s,const int npoints,
                       const int nvars,int &info,CLinearModelShell &lm,
                       CLRReportShell &ar)
  {
//--- initialization
   info=0;
//--- function call
   CLinReg::LRBuildS(xy,s,npoints,nvars,info,lm.GetInnerObj(),ar.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like LRBuildS, but builds model                                  |
//|     Y=A(0)*X[0] + ... + A(N-1)*X[N-1]                            |
//| i.m_e. with zero constant term.                                  |
//+------------------------------------------------------------------+
void CAlglib::LRBuildZS(CMatrixDouble &xy,double &s[],const int npoints,
                        const int nvars,int &info,CLinearModelShell &lm,
                        CLRReportShell &ar)
  {
//--- initialization
   info=0;
//--- function call
   CLinReg::LRBuildZS(xy,s,npoints,nvars,info,lm.GetInnerObj(),ar.GetInnerObj());
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::LRBuildZS(CMatrixDouble &xy,CRowDouble &s,const int npoints,
                        const int nvars,int &info,CLinearModelShell &lm,
                        CLRReportShell &ar)
  {
//--- initialization
   info=0;
//--- function call
   CLinReg::LRBuildZS(xy,s,npoints,nvars,info,lm.GetInnerObj(),ar.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like LRBuild but builds model                                    |
//|     Y=A(0)*X[0] + ... + A(N-1)*X[N-1]                            |
//| i.m_e. with zero constant term.                                  |
//+------------------------------------------------------------------+
void CAlglib::LRBuildZ(CMatrixDouble &xy,const int npoints,
                       const int nvars,int &info,CLinearModelShell &lm,
                       CLRReportShell &ar)
  {
//--- initialization
   info=0;
//--- function call
   CLinReg::LRBuildZ(xy,npoints,nvars,info,lm.GetInnerObj(),ar.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Unpacks coefficients of linear model.                            |
//| INPUT PARAMETERS:                                                |
//|     LM          -   linear model in ALGLIB format                |
//| OUTPUT PARAMETERS:                                               |
//|     V           -   coefficients,array[0..NVars]                 |
//|                     constant term (intercept) is stored in the   |
//|                     V[NVars].                                    |
//|     NVars       -   number of independent variables (one less    |
//|                     than number of coefficients)                 |
//+------------------------------------------------------------------+
void CAlglib::LRUnpack(CLinearModelShell &lm,double &v[],int &nvars)
  {
//--- initialization
   nvars=0;
//--- function call
   CLinReg::LRUnpack(lm.GetInnerObj(),v,nvars);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::LRUnpack(CLinearModelShell &lm,CRowDouble &v,int &nvars)
  {
//--- initialization
   nvars=0;
//--- function call
   CLinReg::LRUnpack(lm.GetInnerObj(),v,nvars);
  }
//+------------------------------------------------------------------+
//| "Packs" coefficients and creates linear model in ALGLIB format   |
//| (LRUnpack reversed).                                             |
//| INPUT PARAMETERS:                                                |
//|     V           -   coefficients, array[0..NVars]                |
//|     NVars       -   number of independent variables              |
//| OUTPUT PAREMETERS:                                               |
//|     LM          -   linear model.                                |
//+------------------------------------------------------------------+
void CAlglib::LRPack(double &v[],const int nvars,CLinearModelShell &lm)
  {
   CLinReg::LRPack(v,nvars,lm.GetInnerObj());
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::LRPack(CRowDouble &v,const int nvars,CLinearModelShell &lm)
  {
   CLinReg::LRPack(v,nvars,lm.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Procesing                                                        |
//| INPUT PARAMETERS:                                                |
//|     LM      -   linear model                                     |
//|     X       -   input vector, array[0..NVars-1].                 |
//| Result:                                                          |
//|     value of linear model regression estimate                    |
//+------------------------------------------------------------------+
double CAlglib::LRProcess(CLinearModelShell &lm,double &x[])
  {
   return(CLinReg::LRProcess(lm.GetInnerObj(),x));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CAlglib::LRProcess(CLinearModelShell &lm,CRowDouble &x)
  {
   return(CLinReg::LRProcess(lm.GetInnerObj(),x));
  }
//+------------------------------------------------------------------+
//| RMS error on the test set                                        |
//| INPUT PARAMETERS:                                                |
//|     LM      -   linear model                                     |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     root mean square error.                                      |
//+------------------------------------------------------------------+
double CAlglib::LRRMSError(CLinearModelShell &lm,CMatrixDouble &xy,
                           const int npoints)
  {
   return(CLinReg::LRRMSError(lm.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average error on the test set                                    |
//| INPUT PARAMETERS:                                                |
//|     LM      -   linear model                                     |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     average error.                                               |
//+------------------------------------------------------------------+
double CAlglib::LRAvgError(CLinearModelShell &lm,CMatrixDouble &xy,
                           const int npoints)
  {
   return(CLinReg::LRAvgError(lm.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| RMS error on the test set                                        |
//| INPUT PARAMETERS:                                                |
//|     LM      -   linear model                                     |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     average relative error.                                      |
//+------------------------------------------------------------------+
double CAlglib::LRAvgRelError(CLinearModelShell &lm,CMatrixDouble &xy,
                              const int npoints)
  {
   return(CLinReg::LRAvgRelError(lm.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| This function serializes data structure to string.               |
//| Important properties of s_out:                                   |
//| * it contains alphanumeric characters, dots, underscores, minus  |
//|   signs                                                          |
//| * these symbols are grouped into words, which are separated by   |
//|   spaces and Windows-style (CR+LF) newlines                      |
//| * although  serializer  uses  spaces and CR+LF as separators, you|
//|   can replace any separator character by arbitrary combination of|
//|   spaces, tabs, Windows or Unix newlines. It allows flexible     |
//|   reformatting of the string in case you want to include it into |
//|   text or XML file. But you should not insert separators into the|
//|   middle of the "words" nor you should change case of letters.   |
//| * s_out can be freely moved between 32-bit and 64-bit systems,   |
//|   little and big endian machines, and so on. You can reference   |
//|   structure on 32-bit machine and unserialize it on 64-bit one   |
//|   (or vice versa), or reference it on SPARC and unserialize on   |
//|   x86. You can also reference it in C# version of ALGLIB and     |
//|   unserialize in C++ one, and vice versa.                        |
//+------------------------------------------------------------------+
void CAlglib::MLPSerialize(CMultilayerPerceptronShell &obj,string &s_out)
  {
//--- create a variable
   CSerializer s;
//--- serialization start
   s.Alloc_Start();
//--- function call
   CMLPBase::MLPAlloc(s,obj.GetInnerObj());
//--- serialization
   s.SStart_Str();
//--- function call
   CMLPBase::MLPSerialize(s,obj.GetInnerObj());
//--- stop
   s.Stop();
//--- change value
   s_out=s.Get_String();
  }
//+------------------------------------------------------------------+
//| This function unserializes data structure from string.           |
//+------------------------------------------------------------------+
void CAlglib::MLPUnserialize(const string s_in,CMultilayerPerceptronShell &obj)
  {
//--- create a variable
   CSerializer s;
//--- unserialization
   s.UStart_Str(s_in);
//--- function call
   CMLPBase::MLPUnserialize(s,obj.GetInnerObj());
//--- stop
   s.Stop();
  }
//+------------------------------------------------------------------+
//| Creates  neural  network  with  NIn  inputs,  NOut outputs,      |
//| without hidden layers, with linear output layer. Network weights |
//| are filled with small random values.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPCreate0(const int nin,const int nout,
                         CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreate0(nin,nout,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Same as MLPCreate0, but with one hidden layer (NHid neurons) with|
//| non-linear activation function. Output layer is linear.          |
//+------------------------------------------------------------------+
void CAlglib::MLPCreate1(const int nin,int nhid,const int nout,
                         CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreate1(nin,nhid,nout,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Same as MLPCreate0,but with two hidden layers (NHid1 and NHid2   |
//| neurons) with non-linear activation function. Output layer is    |
//| linear.                                                          |
//|  $ALL                                                            |
//+------------------------------------------------------------------+
void CAlglib::MLPCreate2(const int nin,const int nhid1,const int nhid2,
                         const int nout,CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreate2(nin,nhid1,nhid2,nout,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Creates neural network with NIn inputs, NOut outputs, without    |
//| hidden layers with non-linear output layer. Network weights are  |
//| filled with small random values.                                 |
//| Activation function of the output layer takes values:            |
//|     (B, +INF), if D>=0                                           |
//| or                                                               |
//|     (-INF, B), if D<0.                                           |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateB0(const int nin,const int nout,const double b,
                          const double d,CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateB0(nin,nout,b,d,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Same as MLPCreateB0 but with non-linear hidden layer.            |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateB1(const int nin,int nhid,const int nout,
                          const double b,const double d,
                          CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateB1(nin,nhid,nout,b,d,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Same as MLPCreateB0 but with two non-linear hidden layers.       |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateB2(const int nin,const int nhid1,const int nhid2,
                          const int nout,const double b,const double d,
                          CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateB2(nin,nhid1,nhid2,nout,b,d,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Creates  neural  network  with  NIn  inputs,  NOut outputs,      |
//| without hidden layers with non-linear output layer. Network      |
//| weights are filled with small random values. Activation function |
//| of the output layer takes values [A,B].                          |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateR0(const int nin,const int nout,double a,
                          const double b,CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateR0(nin,nout,a,b,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Same as MLPCreateR0,but with non-linear hidden layer.            |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateR1(const int nin,int nhid,const int nout,
                          const double a,const double b,
                          CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateR1(nin,nhid,nout,a,b,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Same as MLPCreateR0,but with two non-linear hidden layers.       |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateR2(const int nin,const int nhid1,const int nhid2,
                          const int nout,const double a,const double b,
                          CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateR2(nin,nhid1,nhid2,nout,a,b,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Creates classifier network with NIn inputs and NOut possible     |
//| classes.                                                         |
//| Network contains no hidden layers and linear output layer with   |
//| SOFTMAX-normalization (so outputs sums up to 1.0 and converge to |
//| posterior probabilities).                                        |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateC0(const int nin,const int nout,
                          CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateC0(nin,nout,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Same as MLPCreateC0,but with one non-linear hidden layer.        |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateC1(const int nin,int nhid,const int nout,
                          CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateC1(nin,nhid,nout,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Same as MLPCreateC0, but with two non-linear hidden layers.      |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateC2(const int nin,const int nhid1,const int nhid2,
                          const int nout,CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPCreateC2(nin,nhid1,nhid2,nout,network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Randomization of neural network weights                          |
//+------------------------------------------------------------------+
void CAlglib::MLPRandomize(CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPRandomize(network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Randomization of neural network weights and standartisator       |
//+------------------------------------------------------------------+
void CAlglib::MLPRandomizeFull(CMultilayerPerceptronShell &network)
  {
   CMLPBase::MLPRandomizeFull(network.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Internal subroutine.                                             |
//+------------------------------------------------------------------+
void CAlglib::MLPInitPreprocessor(CMultilayerPerceptronShell &network,
                                  CMatrixDouble &xy,int ssize)
  {
   CMLPBase::MLPInitPreprocessor(network.GetInnerObj(),xy,ssize);
  }
//+------------------------------------------------------------------+
//| Returns information about initialized network: number of inputs, |
//| outputs, weights.                                                |
//+------------------------------------------------------------------+
void CAlglib::MLPProperties(CMultilayerPerceptronShell &network,
                            int &nin,int &nout,int &wcount)
  {
//--- initialization
   nin=0;
   nout=0;
   wcount=0;
//--- function call
   CMLPBase::MLPProperties(network.GetInnerObj(),nin,nout,wcount);
  }
//+------------------------------------------------------------------+
//| Returns number of inputs.                                        |
//+------------------------------------------------------------------+
int CAlglib::MLPGetInputsCount(CMultilayerPerceptronShell &network)
  {
   return(CMLPBase::MLPGetInputsCount(network.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| Returns number of outputs.                                       |
//+------------------------------------------------------------------+
int CAlglib::MLPGetOutputsCount(CMultilayerPerceptronShell &network)
  {
   return(CMLPBase::MLPGetOutputsCount(network.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| Returns number of weights.                                       |
//+------------------------------------------------------------------+
int CAlglib::MLPGetWeightsCount(CMultilayerPerceptronShell &network)
  {
   return(CMLPBase::MLPGetWeightsCount(network.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| Tells whether network is SOFTMAX-normalized (i.m_e. classifier)  |
//| or not.                                                          |
//+------------------------------------------------------------------+
bool CAlglib::MLPIsSoftMax(CMultilayerPerceptronShell &network)
  {
   return(CMLPBase::MLPIsSoftMax(network.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| This function returns total number of layers (including input,   |
//| hidden and output layers).                                       |
//+------------------------------------------------------------------+
int CAlglib::MLPGetLayersCount(CMultilayerPerceptronShell &network)
  {
   return(CMLPBase::MLPGetLayersCount(network.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| This function returns size of K-th layer.                        |
//| K=0 corresponds to input layer, K=CNT-1 corresponds to output    |
//| layer.                                                           |
//| Size of the output layer is always equal to the number of        |
//| outputs, although when we have softmax-normalized network, last  |
//| neuron doesn't have any connections - it is just zero.           |
//+------------------------------------------------------------------+
int CAlglib::MLPGetLayerSize(CMultilayerPerceptronShell &network,
                             const int k)
  {
   return(CMLPBase::MLPGetLayerSize(network.GetInnerObj(),k));
  }
//+------------------------------------------------------------------+
//| This function returns offset/scaling coefficients for I-th input |
//| of the network.                                                  |
//| INPUT PARAMETERS:                                                |
//|     Network     -   network                                      |
//|     I           -   input index                                  |
//| OUTPUT PARAMETERS:                                               |
//|     Mean        -   mean term                                    |
//|     Sigma       -   sigma term,guaranteed to be nonzero.         |
//| I-th input is passed through linear transformation               |
//|     IN[i]=(IN[i]-Mean)/Sigma                                     |
//| before feeding to the network                                    |
//+------------------------------------------------------------------+
void CAlglib::MLPGetInputScaling(CMultilayerPerceptronShell &network,
                                 const int i,double &mean,double &sigma)
  {
//--- initialization
   mean=0;
   sigma=0;
//--- function call
   CMLPBase::MLPGetInputScaling(network.GetInnerObj(),i,mean,sigma);
  }
//+------------------------------------------------------------------+
//| This function returns offset/scaling coefficients for I-th output|
//| of the network.                                                  |
//| INPUT PARAMETERS:                                                |
//|     Network     -   network                                      |
//|     I           -   input index                                  |
//| OUTPUT PARAMETERS:                                               |
//|     Mean        -   mean term                                    |
//|     Sigma       -   sigma term, guaranteed to be nonzero.        |
//| I-th output is passed through linear transformation              |
//|     OUT[i] = OUT[i]*Sigma+Mean                                   |
//| before returning it to user. In case we have SOFTMAX-normalized  |
//| network, we return (Mean,Sigma)=(0.0,1.0).                       |
//+------------------------------------------------------------------+
void CAlglib::MLPGetOutputScaling(CMultilayerPerceptronShell &network,
                                  const int i,double &mean,double &sigma)
  {
//--- initialization
   mean=0;
   sigma=0;
//--- function call
   CMLPBase::MLPGetOutputScaling(network.GetInnerObj(),i,mean,sigma);
  }
//+------------------------------------------------------------------+
//| This function returns information about Ith neuron of Kth layer  |
//| INPUT PARAMETERS:                                                |
//|     Network     -   network                                      |
//|     K           -   layer index                                  |
//|     I           -   neuron index (within layer)                  |
//| OUTPUT PARAMETERS:                                               |
//|     FKind       -   activation function type (used by            |
//|                     MLPActivationFunction()) this value is zero  |
//|                     for input or linear neurons                  |
//|     Threshold   -   also called offset, bias                     |
//|                     zero for input neurons                       |
//| NOTE: this function throws exception if layer or neuron with     |
//| given index do not exists.                                       |
//+------------------------------------------------------------------+
void CAlglib::MLPGetNeuronInfo(CMultilayerPerceptronShell &network,
                               const int k,const int i,int &fkind,
                               double &threshold)
  {
//--- initialization
   fkind=0;
   threshold=0;
//--- function call
   CMLPBase::MLPGetNeuronInfo(network.GetInnerObj(),k,i,fkind,threshold);
  }
//+------------------------------------------------------------------+
//| This function returns information about connection from I0-th    |
//| neuron of K0-th layer to I1-th neuron of K1-th layer.            |
//| INPUT PARAMETERS:                                                |
//|     Network     -   network                                      |
//|     K0          -   layer index                                  |
//|     I0          -   neuron index (within layer)                  |
//|     K1          -   layer index                                  |
//|     I1          -   neuron index (within layer)                  |
//| RESULT:                                                          |
//|     connection weight (zero for non-existent connections)        |
//| This function:                                                   |
//| 1. throws exception if layer or neuron with given index do not   |
//|    exists.                                                       |
//| 2. returns zero if neurons exist, but there is no connection     |
//|    between them                                                  |
//+------------------------------------------------------------------+
double CAlglib::MLPGetWeight(CMultilayerPerceptronShell &network,
                             const int k0,const int i0,const int k1,
                             const int i1)
  {
   return(CMLPBase::MLPGetWeight(network.GetInnerObj(),k0,i0,k1,i1));
  }
//+------------------------------------------------------------------+
//| This function sets offset/scaling coefficients for I-th input of |
//| the network.                                                     |
//| INPUT PARAMETERS:                                                |
//|     Network     -   network                                      |
//|     I           -   input index                                  |
//|     Mean        -   mean term                                    |
//|     Sigma       -   sigma term (if zero,will be replaced by 1.0) |
//| NTE: I-th input is passed through linear transformation          |
//|     IN[i]=(IN[i]-Mean)/Sigma                                     |
//| before feeding to the network. This function sets Mean and Sigma.|
//+------------------------------------------------------------------+
void CAlglib::MLPSetInputScaling(CMultilayerPerceptronShell &network,
                                 const int i,const double mean,
                                 const double sigma)
  {
   CMLPBase::MLPSetInputScaling(network.GetInnerObj(),i,mean,sigma);
  }
//+------------------------------------------------------------------+
//| This function sets offset/scaling coefficients for I-th output of|
//| the network.                                                     |
//| INPUT PARAMETERS:                                                |
//|     Network     -   network                                      |
//|     I           -   input index                                  |
//|     Mean        -   mean term                                    |
//|     Sigma       -   sigma term (if zero, will be replaced by 1.0)|
//| OUTPUT PARAMETERS:                                               |
//| NOTE: I-th output is passed through linear transformation        |
//|     OUT[i] = OUT[i]*Sigma+Mean                                   |
//| before returning it to user. This function sets Sigma/Mean. In   |
//| case we have SOFTMAX-normalized network, you can not set (Sigma, |
//| Mean) to anything other than(0.0,1.0) - this function will throw |
//| exception.                                                       |
//+------------------------------------------------------------------+
void CAlglib::MLPSetOutputScaling(CMultilayerPerceptronShell &network,
                                  const int i,const double mean,
                                  const double sigma)
  {
   CMLPBase::MLPSetOutputScaling(network.GetInnerObj(),i,mean,sigma);
  }
//+------------------------------------------------------------------+
//| This function modifies information about Ith neuron of Kth layer |
//| INPUT PARAMETERS:                                                |
//|     Network     -   network                                      |
//|     K           -   layer index                                  |
//|     I           -   neuron index (within layer)                  |
//|     FKind       -   activation function type (used by            |
//|                     MLPActivationFunction()) this value must be  |
//|                     zero for input neurons (you can not set      |
//|                     activation function for input neurons)       |
//|     Threshold   -   also called offset, bias                     |
//|                     this value must be zero for input neurons    |
//|                     (you can not set threshold for input neurons)|
//| NOTES:                                                           |
//| 1. this function throws exception if layer or neuron with given  |
//|    index do not exists.                                          |
//| 2. this function also throws exception when you try to set       |
//|    non-linear activation function for input neurons (any kind    |
//|    of network) or for output neurons of classifier network.      |
//| 3. this function throws exception when you try to set non-zero   |
//|    threshold for input neurons (any kind of network).            |
//+------------------------------------------------------------------+
void CAlglib::MLPSetNeuronInfo(CMultilayerPerceptronShell &network,
                               const int k,const int i,int fkind,
                               double threshold)
  {
   CMLPBase::MLPSetNeuronInfo(network.GetInnerObj(),k,i,fkind,threshold);
  }
//+------------------------------------------------------------------+
//| This function modifies information about connection from I0-th   |
//| neuron of K0-th layer to I1-th neuron of K1-th layer.            |
//| INPUT PARAMETERS:                                                |
//|     Network     -   network                                      |
//|     K0          -   layer index                                  |
//|     I0          -   neuron index (within layer)                  |
//|     K1          -   layer index                                  |
//|     I1          -   neuron index (within layer)                  |
//|     W           -   connection weight (must be zero for          |
//|                     non-existent connections)                    |
//| This function:                                                   |
//| 1. throws exception if layer or neuron with given index do not   |
//|    exists.                                                       |
//| 2. throws exception if you try to set non-zero weight for        |
//|    non-existent connection                                       |
//+------------------------------------------------------------------+
void CAlglib::MLPSetWeight(CMultilayerPerceptronShell &network,
                           const int k0,const int i0,const int k1,
                           const int i1,const double w)
  {
   CMLPBase::MLPSetWeight(network.GetInnerObj(),k0,i0,k1,i1,w);
  }
//+------------------------------------------------------------------+
//| Neural network activation function                               |
//| INPUT PARAMETERS:                                                |
//|     NET         -   neuron input                                 |
//|     K           -   function index (zero for linear function)    |
//| OUTPUT PARAMETERS:                                               |
//|     F           -   function                                     |
//|     DF          -   its derivative                               |
//|     D2F         -   its second derivative                        |
//+------------------------------------------------------------------+
void CAlglib::MLPActivationFunction(const double net,const int k,
                                    double &f,double &df,double &d2f)
  {
//--- initialization
   f=0;
   df=0;
   d2f=0;
//--- function call
   CMLPBase::MLPActivationFunction(net,k,f,df,d2f);
  }
//+------------------------------------------------------------------+
//| Procesing                                                        |
//| INPUT PARAMETERS:                                                |
//|     Network -   neural network                                   |
//|     X       -   input vector,  array[0..NIn-1].                  |
//| OUTPUT PARAMETERS:                                               |
//|     Y       -   result. Regression estimate when solving         |
//|                 regression task, vector of posterior             |
//|                 probabilities for classification task.           |
//| See also MLPProcessI                                             |
//+------------------------------------------------------------------+
void CAlglib::MLPProcess(CMultilayerPerceptronShell &network,
                         double &x[],double &y[])
  {
   CMLPBase::MLPProcess(network.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| 'interactive' variant of MLPProcess for languages like Python    |
//| which support constructs like "Y = MLPProcess(NN,X)" and         |
//| interactive mode of the interpreter                              |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart,     |
//| but it is more convenient when you call it from command line.    |
//+------------------------------------------------------------------+
void CAlglib::MLPProcessI(CMultilayerPerceptronShell &network,
                          double &x[],double &y[])
  {
   CMLPBase::MLPProcessI(network.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| Error function for neural network,internal subroutine.           |
//+------------------------------------------------------------------+
double CAlglib::MLPError(CMultilayerPerceptronShell &network,
                         CMatrixDouble &xy,const int ssize)
  {
   return(CMLPBase::MLPError(network.GetInnerObj(),xy,ssize));
  }
//+------------------------------------------------------------------+
//| Error of the neural network on dataset given by sparse matrix.   |
//| INPUT PARAMETERS:                                                |
//|   Network     -  neural network                                  |
//|   XY          -  training set, see below for information on the  |
//|                  training set format. This function checks       |
//|                  correctness of the dataset (no NANs/INFs, class |
//|                  numbers are correct) and throws exception when  |
//|                  incorrect dataset is passed. Sparse matrix must |
//|                  use CRS format for storage.                     |
//|   NPoints     -  points count, >=0                               |
//| RESULT:                                                          |
//|   sum-of-squares error, SUM(sqr(y[i]-desired_y[i])/2)            |
//| DATASET FORMAT:                                                  |
//| This function uses two different dataset formats - one for       |
//| regression networks, another one for classification networks.    |
//| For regression networks with NIn inputs and NOut outputs         |
//| following dataset format is used:                                |
//|   * dataset is given by NPoints*(NIn+NOut) matrix                |
//|   * each row corresponds to one example                          |
//|   * first NIn columns are inputs, next NOut columns are outputs  |
//| For classification networks with NIn inputs and NClasses clases  |
//| following dataset format is used:                                |
//|   * dataset is given by NPoints*(NIn+1) matrix                   |
//|   * each row corresponds to one example                          |
//|   * first NIn columns are inputs, last column stores class number|
//|     (from 0 to NClasses-1).                                      |
//+------------------------------------------------------------------+
double CAlglib::MLPErrorSparse(CMultilayerPerceptronShell &network,
                               CSparseMatrix &xy,int npoints)
  {
   return(CMLPBase::MLPErrorSparse(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Natural error function for neural network,internal subroutine.   |
//+------------------------------------------------------------------+
double CAlglib::MLPErrorN(CMultilayerPerceptronShell &network,
                          CMatrixDouble &xy,const int ssize)
  {
   return(CMLPBase::MLPErrorN(network.GetInnerObj(),xy,ssize));
  }
//+------------------------------------------------------------------+
//| Classification error                                             |
//+------------------------------------------------------------------+
int CAlglib::MLPClsError(CMultilayerPerceptronShell &network,
                         CMatrixDouble &xy,const int ssize)
  {
   return(CMLPBase::MLPClsError(network.GetInnerObj(),xy,ssize));
  }
//+------------------------------------------------------------------+
//| Relative classification error on the test set                    |
//| INPUT PARAMETERS:                                                |
//|     Network -   network                                          |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     percent of incorrectly classified cases. Works both for      |
//|     classifier networks and general purpose networks used as     |
//|     classifiers.                                                 |
//+------------------------------------------------------------------+
double CAlglib::MLPRelClsError(CMultilayerPerceptronShell &network,
                               CMatrixDouble &xy,const int npoints)
  {
   return(CMLPBase::MLPRelClsError(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Relative classification error on the test set given by sparse    |
//| matrix.                                                          |
//| INPUT PARAMETERS:                                                |
//|   Network     -  neural network;                                 |
//|   XY          -  training set, see below for information on the  |
//|                  training set format. Sparse matrix must use CRS |
//|                  format for storage.                             |
//|   NPoints     -  points count, >=0.                              |
//| RESULT:                                                          |
//|   Percent of incorrectly classified cases. Works both for        |
//|   classifier networks and general purpose networks used as       |
//|   classifiers.                                                   |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//|   For regression networks with NIn inputs and NOut outputs       |
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+NOut) matrix             |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, next NOut columns are       |
//|        outputs                                                   |
//|   For classification networks with NIn inputs and NClasses clases|
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+1) matrix                |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, last column stores class    |
//|        number (from 0 to NClasses-1).                            |
//+------------------------------------------------------------------+
double CAlglib::MLPRelClsErrorSparse(CMultilayerPerceptronShell &network,
                                     CSparseMatrix &xy,int npoints)
  {
   return(CMLPBase::MLPRelClsErrorSparse(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average cross-entropy (in bits per element) on the test set      |
//| INPUT PARAMETERS:                                                |
//|     Network -   neural network                                   |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     CrossEntropy/(NPoints*LN(2)).                                |
//|     Zero if network solves regression task.                      |
//+------------------------------------------------------------------+
double CAlglib::MLPAvgCE(CMultilayerPerceptronShell &network,
                         CMatrixDouble &xy,const int npoints)
  {
   return(CMLPBase::MLPAvgCE(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average cross-entropy (in bits per element) on the test set given|
//| by sparse matrix.                                                |
//| INPUT PARAMETERS:                                                |
//|   Network     -  neural network;                                 |
//|   XY          -  training set, see below for information on the  |
//|                  training set format. This function checks       |
//|                  correctness of the dataset (no NANs/INFs, class |
//|                  numbers are correct) and throws exception when  |
//|                  incorrect dataset is passed. Sparse matrix must |
//|                  use CRS format for storage.                     |
//|   NPoints     -  points count, >=0.                              |
//| RESULT:                                                          |
//|   CrossEntropy/(NPoints*LN(2)).                                  |
//|   Zero if network solves regression task.                        |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//|   For regression networks with NIn inputs and NOut outputs       |
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+NOut) matrix             |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, next NOut columns are       |
//|        outputs                                                   |
//|   For classification networks with NIn inputs and NClasses clases|
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+1) matrix                |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, last column stores class    |
//|        number (from 0 to NClasses-1).                            |
//+------------------------------------------------------------------+
double CAlglib::MLPAvgCESparse(CMultilayerPerceptronShell &network,
                               CSparseMatrix &xy,int npoints)
  {
   return(CMLPBase::MLPAvgCESparse(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| RMS error on the test set                                        |
//| INPUT PARAMETERS:                                                |
//|     Network -   neural network                                   |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     root mean square error.                                      |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task,RMS error means error when estimating    |
//|     posterior probabilities.                                     |
//+------------------------------------------------------------------+
double CAlglib::MLPRMSError(CMultilayerPerceptronShell &network,
                            CMatrixDouble &xy,const int npoints)
  {
   return(CMLPBase::MLPRMSError(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| RMS error on the test set given by sparse matrix.                |
//| INPUT PARAMETERS:                                                |
//|   Network     -  neural network;                                 |
//|   XY          -  training set, see below for information on the  |
//|                  training set format. This function checks       |
//|                  correctness of the dataset (no NANs/INFs, class |
//|                  numbers are correct) and throws exception when  |
//|                  incorrect dataset is passed. Sparse matrix must |
//|                  use CRS format for storage.                     |
//|   NPoints     -  points count, >=0.                              |
//| RESULT:                                                          |
//|   Root mean square error. Its meaning for regression task is     |
//|   obvious. As for classification task, RMS error means error when|
//|   estimating posterior probabilities.                            |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//|   For regression networks with NIn inputs and NOut outputs       |
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn + NOut) matrix           |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, next NOut columns are       |
//|        outputs                                                   |
//|   For classification networks with NIn inputs and NClasses       |
//|   clases following dataset format is used:                       |
//|      * dataset is given by NPoints*(NIn + 1) matrix              |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, last column stores class    |
//|        number(from 0 to NClasses - 1).                           |
//+------------------------------------------------------------------+
double CAlglib::MLPRMSErrorSparse(CMultilayerPerceptronShell &network,
                                  CSparseMatrix &xy,int npoints)
  {
   return(CMLPBase::MLPRMSErrorSparse(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average error on the test set                                    |
//| INPUT PARAMETERS:                                                |
//|     Network -   neural network                                   |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task,it means average error when estimating   |
//|     posterior probabilities.                                     |
//+------------------------------------------------------------------+
double CAlglib::MLPAvgError(CMultilayerPerceptronShell &network,
                            CMatrixDouble &xy,const int npoints)
  {
   return(CMLPBase::MLPAvgError(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average absolute error on the test set given by sparse matrix.   |
//| INPUT PARAMETERS:                                                |
//|   Network     -  neural network;                                 |
//|   XY          -  training set, see below for information on the  |
//|                  training set format. This function checks       |
//|                  correctness of the dataset (no NANs/INFs, class |
//|                  numbers are correct) and throws exception when  |
//|                  incorrect dataset is passed. Sparse matrix must |
//|                  use CRS format for storage.                     |
//|   NPoints     -  points count, >=0.                              |
//| RESULT:                                                          |
//|   Its meaning for regression task is obvious. As for             |
//|   classification task, it means average error when estimating    |
//|   posterior probabilities.                                       |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//|   For regression networks with NIn inputs and NOut outputs       |
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+NOut) matrix             |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, next NOut columns are       |
//|        outputs                                                   |
//|   For classification networks with NIn inputs and NClasses clases|
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+1) matrix                |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, last column stores class    |
//|        number (from 0 to NClasses-1).                            |
//+------------------------------------------------------------------+
double CAlglib::MLPAvgErrorSparse(CMultilayerPerceptronShell &network,
                                  CSparseMatrix &xy,int npoints)
  {
   return(CMLPBase::MLPAvgErrorSparse(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average relative error on the test set                           |
//| INPUT PARAMETERS:                                                |
//|     Network -   neural network                                   |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task, it means average relative error when    |
//|     estimating posterior probability of belonging to the correct |
//|     class.                                                       |
//+------------------------------------------------------------------+
double CAlglib::MLPAvgRelError(CMultilayerPerceptronShell &network,
                               CMatrixDouble &xy,const int npoints)
  {
   return(CMLPBase::MLPAvgRelError(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average relative error on the test set given by sparse matrix.   |
//| INPUT PARAMETERS:                                                |
//|   Network     -  neural network;                                 |
//|   XY          -  training set, see below for information on the  |
//|                  training set format. This function checks       |
//|                  correctness of the dataset (no NANs/INFs, class |
//|                  numbers are correct) and throws exception when  |
//|                  incorrect dataset is passed. Sparse matrix must |
//|                  use CRS format for storage.                     |
//|   NPoints     -  points count, >=0.                              |
//| RESULT:                                                          |
//|   Its meaning for regression task is obvious. As for             |
//|   classification task, it means average relative error when      |
//|   estimating posterior probability of belonging to the correct   |
//|   class.                                                         |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//|   For regression networks with NIn inputs and NOut outputs       |
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+NOut) matrix             |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, next NOut columns are       |
//|        outputs                                                   |
//|   For classification networks with NIn inputs and NClasses clases|
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+1) matrix                |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, last column stores class    |
//|        number (from 0 to NClasses-1).                            |
//+------------------------------------------------------------------+
double CAlglib::MLPAvgRelErrorSparse(CMultilayerPerceptronShell &network,
                                     CSparseMatrix &xy,int npoints)
  {
   return(CMLPBase::MLPAvgRelErrorSparse(network.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Gradient calculation                                             |
//| INPUT PARAMETERS:                                                |
//|     Network -   network initialized with one of the network      |
//|                 creation funcs                                   |
//|     X       -   input vector, length of array must be at least   |
//|                 NIn                                              |
//|     DesiredY-   desired outputs, length of array must be at least|
//|                 NOut                                             |
//|     Grad    -   possibly preallocated array. If size of array is |
//|                 smaller than WCount, it will be reallocated. It  |
//|                 is recommended to reuse previously allocated     |
//|                 array to reduce allocation overhead.             |
//| OUTPUT PARAMETERS:                                               |
//|     E       -   error function, SUM(sqr(y[i]-desiredy[i])/2,i)   |
//|     Grad    -   gradient of E with respect to weights of network,|
//|                 array[WCount]                                    |
//+------------------------------------------------------------------+
void CAlglib::MLPGrad(CMultilayerPerceptronShell &network,double &x[],
                      double &desiredy[],double &e,double &grad[])
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGrad(network.GetInnerObj(),x,desiredy,e,grad);
  }
//+------------------------------------------------------------------+
//| Gradient calculation (natural error function is used)            |
//| INPUT PARAMETERS:                                                |
//|     Network -   network initialized with one of the network      |
//|                 creation funcs                                   |
//|     X       -   input vector, length of array must be at least   |
//|                 NIn                                              |
//|     DesiredY-   desired outputs, length of array must be at least|
//|                 NOut                                             |
//|     Grad    -   possibly preallocated array. If size of array is |
//|                 smaller than WCount, it will be reallocated. It  |
//|                 is recommended to reuse previously allocated     |
//|                 array to reduce allocation overhead.             |
//| OUTPUT PARAMETERS:                                               |
//|     E       -   error function, sum-of-squares for regression    |
//|                 networks, cross-entropy for classification       |
//|                 networks.                                        |
//|     Grad    -   gradient of E with respect to weights of network,|
//|                 array[WCount]                                    |
//+------------------------------------------------------------------+
void CAlglib::MLPGradN(CMultilayerPerceptronShell &network,double &x[],
                       double &desiredy[],double &e,double &grad[])
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGradN(network.GetInnerObj(),x,desiredy,e,grad);
  }
//+------------------------------------------------------------------+
//| Batch gradient calculation for a set of inputs/outputs           |
//| INPUT PARAMETERS:                                                |
//|     Network -   network initialized with one of the network      |
//|                 creation funcs                                   |
//|     XY      -   set of inputs/outputs; one sample = one row;     |
//|                 first NIn columns contain inputs,                |
//|                 next NOut columns - desired outputs.             |
//|     SSize   -   number of elements in XY                         |
//|     Grad    -   possibly preallocated array. If size of array is |
//|                 smaller than WCount, it will be reallocated. It  |
//|                 is recommended to reuse previously allocated     |
//|                 array to reduce allocation overhead.             |
//| OUTPUT PARAMETERS:                                               |
//|     E       -   error function, SUM(sqr(y[i]-desiredy[i])/2,i)   |
//|     Grad    -   gradient of E with respect to weights of network,|
//|                 array[WCount]                                    |
//+------------------------------------------------------------------+
void CAlglib::MLPGradBatch(CMultilayerPerceptronShell &network,
                           CMatrixDouble &xy,const int ssize,
                           double &e,double &grad[])
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGradBatch(network.GetInnerObj(),xy,ssize,e,grad);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MLPGradBatch(CMultilayerPerceptronShell &network,
                           CMatrixDouble &xy,const int ssize,
                           double &e,CRowDouble &grad)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGradBatch(network.GetInnerObj(),xy,ssize,e,grad);
  }
//+------------------------------------------------------------------+
//| Batch gradient calculation for a set of inputs/outputs given by  |
//| sparse matrices                                                  |
//| INPUT PARAMETERS:                                                |
//|   Network  -  network initialized with one of the network        |
//|               creation funcs                                     |
//|   XY       -  original dataset in sparse format; one sample = one|
//|               row:                                               |
//|               * MATRIX MUST BE STORED IN CRS FORMAT              |
//|               * first NIn columns contain inputs.                |
//|               * for regression problem, next NOut columns store  |
//|                 desired outputs.                                 |
//|               * for classification problem, next column (just    |
//|                 one!) stores class number.                       |
//|   SSize    -  number of elements in XY                           |
//|   Grad     -  possibly preallocated array. If size of array is   |
//|               smaller than WCount, it will be reallocated. It is |
//|               recommended to reuse previously allocated array to |
//|               reduce allocation overhead.                        |
//| OUTPUT PARAMETERS:                                               |
//|   E        -  error function, SUM(sqr(y[i]-desiredy[i])/2,i)     |
//|   Grad     -  gradient of E with respect to weights of network,  |
//|               array[WCount]                                      |
//+------------------------------------------------------------------+
void CAlglib::MLPGradBatchSparse(CMultilayerPerceptronShell &network,
                                 CSparseMatrix &xy,int ssize,double &e,
                                 CRowDouble &grad)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGradBatchSparse(network.GetInnerObj(),xy,ssize,e,grad);
  }
//+------------------------------------------------------------------+
//| Batch gradient calculation for a subset of dataset               |
//| INPUT PARAMETERS:                                                |
//|   Network  -  network initialized with one of the network        |
//|               creation funcs                                     |
//|   XY       -  original dataset in dense format; one sample = one |
//|               row:                                               |
//|               * first NIn columns contain inputs,                |
//|               * for regression problem, next NOut columns store  |
//|                 desired outputs.                                 |
//|               * for classification problem, next column (just    |
//|                 one!) stores class number.                       |
//|   SetSize  -  real size of XY, SetSize>=0;                       |
//|   Idx      -  subset of SubsetSize elements, array[SubsetSize]:  |
//|               * Idx[I] stores row index in the original dataset  |
//|                 which is given by XY. Gradient is calculated with|
//|                 respect to rows whose indexes are stored in Idx[]|
//|               * Idx[]  must store correct indexes; this function |
//|                 throws an exception in case incorrect index (less|
//|                 than 0 or larger than rows(XY)) is given         |
//|               * Idx[] may store indexes in any order and even    |
//|                 with repetitions.                                |
//|   SubsetSize- number of elements in Idx[] array:                 |
//|               * positive value means that subset given by Idx[]  |
//|                 is processed                                     |
//|               * zero value results in zero gradient              |
//|               * negative value means that full dataset is        |
//|                 processed                                        |
//|   Grad     -  possibly preallocated array. If size of array is   |
//|               smaller than WCount, it will be reallocated. It is |
//|               recommended to reuse previously allocated array to |
//|               reduce allocation overhead.                        |
//| OUTPUT PARAMETERS:                                               |
//|   E        -  error function, SUM(sqr(y[i]-desiredy[i])/2,i)     |
//|   Grad     -  gradient of E with respect to weights of network,  |
//|               array[WCount]                                      |
//+------------------------------------------------------------------+
void CAlglib::MLPGradBatchSubset(CMultilayerPerceptronShell &network,
                                 CMatrixDouble &xy,int setsize,
                                 CRowInt &idx,int subsetsize,
                                 double &e,CRowDouble &grad)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGradBatchSubset(network.GetInnerObj(),xy,setsize,idx,subsetsize,e,grad);
  }
//+------------------------------------------------------------------+
//| Batch gradient calculation for a set of inputs/outputs for a     |
//| subset of dataset given by set of indexes.                       |
//| INPUT PARAMETERS:                                                |
//|   Network  -  network initialized with one of the network        |
//|               creation funcs                                     |
//|   XY       -  original dataset in sparse format; one sample = one|
//|               row:                                               |
//|               * MATRIX MUST BE STORED IN CRS FORMAT              |
//|               * first NIn columns contain inputs,                |
//|               * for regression problem, next NOut columns store  |
//|                 desired outputs.                                 |
//|               * for classification problem, next column (just    |
//|                 one!) stores class number.                       |
//|   SetSize  -  real size of XY, SetSize>=0;                       |
//|   Idx      -  subset of SubsetSize elements, array[SubsetSize]:  |
//|               * Idx[I] stores row index in the original dataset  |
//|                 which is given by XY. Gradient is calculated with|
//|                 respect to rows whose indexes are stored in Idx[]|
//|               * Idx[] must store correct indexes; this function  |
//|                 throws an exception in case incorrect index (less|
//|                 than 0 or larger than rows(XY)) is given         |
//|               * Idx[] may store indexes in any order and even    |
//|                 with repetitions.                                |
//|   SubsetSize- number of elements in Idx[] array:                 |
//|               * positive value means that subset given by Idx[]  |
//|                 is processed                                     |
//|               * zero value results in zero gradient              |
//|               * negative value means that full dataset is        |
//|                 processed                                        |
//|   Grad     -  possibly preallocated array. If size of array is   |
//|               smaller than WCount, it will be reallocated. It is |
//|               recommended to reuse previously allocated array to |
//|               reduce allocation overhead.                        |
//| OUTPUT PARAMETERS:                                               |
//|   E        -  error function, SUM(sqr(y[i]-desiredy[i])/2,i)     |
//|   Grad     -  gradient of E with respect to weights of network,  |
//|               array[WCount]                                      |
//| NOTE: when SubsetSize<0 is used full dataset by call             |
//|       MLPGradBatchSparse function.                               |
//+------------------------------------------------------------------+
void CAlglib::MLPGradBatchSparseSubset(CMultilayerPerceptronShell &network,
                                       CSparseMatrix &xy,int setsize,
                                       CRowInt &idx,int subsetsize,
                                       double &e,CRowDouble &grad)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGradBatchSparseSubset(network.GetInnerObj(),xy,setsize,idx,subsetsize,e,grad);
  }
//+------------------------------------------------------------------+
//| Batch gradient calculation for a set of inputs/outputs           |
//| (natural error function is used)                                 |
//| INPUT PARAMETERS:                                                |
//|     Network -   network initialized with one of the network      |
//|                 creation funcs                                   |
//|     XY      -   set of inputs/outputs; one sample=one row;       |
//|                 first NIn columns contain inputs,                |
//|                 next NOut columns - desired outputs.             |
//|     SSize   -   number of elements in XY                         |
//|     Grad    -   possibly preallocated array. If size of array is |
//|                 smaller than WCount, it will be reallocated. It  |
//|                 is recommended to reuse previously allocated     |
//|                 array to reduce allocation overhead.             |
//| OUTPUT PARAMETERS:                                               |
//|     E       -   error function, sum-of-squares for regression    |
//|                 networks, cross-entropy for classification       |
//|                 networks.                                        |
//|     Grad    -   gradient of E with respect to weights of network,|
//|                 array[WCount]                                    |
//+------------------------------------------------------------------+
void CAlglib::MLPGradNBatch(CMultilayerPerceptronShell &network,
                            CMatrixDouble &xy,const int ssize,
                            double &e,double &grad[])
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGradNBatch(network.GetInnerObj(),xy,ssize,e,grad);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MLPGradNBatch(CMultilayerPerceptronShell &network,
                            CMatrixDouble &xy,const int ssize,
                            double &e,CRowDouble &grad)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPGradNBatch(network.GetInnerObj(),xy,ssize,e,grad);
  }
//+------------------------------------------------------------------+
//| Batch Hessian calculation (natural error function) using         |
//| R-algorithm. Internal subroutine.                                |
//|      Hessian calculation based on R-algorithm described in       |
//|      "Fast Exact Multiplication by the Hessian",                 |
//|      B. A. Pearlmutter,                                          |
//|      Neural Computation, 1994.                                   |
//+------------------------------------------------------------------+
void CAlglib::MLPHessianNBatch(CMultilayerPerceptronShell &network,
                               CMatrixDouble &xy,const int ssize,
                               double &e,double &grad[],
                               CMatrixDouble &h)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPHessianNBatch(network.GetInnerObj(),xy,ssize,e,grad,h);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MLPHessianNBatch(CMultilayerPerceptronShell &network,
                               CMatrixDouble &xy,const int ssize,
                               double &e,CRowDouble &grad,
                               CMatrixDouble &h)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPHessianNBatch(network.GetInnerObj(),xy,ssize,e,grad,h);
  }
//+------------------------------------------------------------------+
//| Batch Hessian calculation using R-algorithm.                     |
//| Internal subroutine.                                             |
//|      Hessian calculation based on R-algorithm described in       |
//|      "Fast Exact Multiplication by the Hessian",                 |
//|      B. A. Pearlmutter,                                          |
//|      Neural Computation, 1994.                                   |
//+------------------------------------------------------------------+
void CAlglib::MLPHessianBatch(CMultilayerPerceptronShell &network,
                              CMatrixDouble &xy,const int ssize,
                              double &e,double &grad[],CMatrixDouble &h)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPHessianBatch(network.GetInnerObj(),xy,ssize,e,grad,h);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MLPHessianBatch(CMultilayerPerceptronShell &network,
                              CMatrixDouble &xy,const int ssize,
                              double &e,CRowDouble &grad,CMatrixDouble &h)
  {
//--- initialization
   e=0;
//--- function call
   CMLPBase::MLPHessianBatch(network.GetInnerObj(),xy,ssize,e,grad,h);
  }
//+------------------------------------------------------------------+
//| Calculation of all types of errors on subset of dataset.         |
//| INPUT PARAMETERS:                                                |
//|   Network  -  network initialized with one of the network        |
//|               creation funcs                                     |
//|   XY       -  original dataset; one sample = one row;            |
//|               first NIn columns contain inputs, next NOut        |
//|               columns - desired outputs.                         |
//|   SetSize  -  real size of XY, SetSize>=0;                       |
//|   Subset   -  subset of SubsetSize elements, array[SubsetSize];  |
//|   SubsetSize- number of elements in Subset[] array:              |
//|               * if SubsetSize>0, rows of XY with indices         |
//|                 Subset[0]......Subset[SubsetSize-1] are processed|
//|               * if SubsetSize=0, zeros are returned              |
//|               * if SubsetSize<0, entire dataset is  processed;   |
//|                 Subset[] array is ignored in this case.          |
//| OUTPUT PARAMETERS:                                               |
//|   Rep      -  it contains all type of errors.                    |
//+------------------------------------------------------------------+
void CAlglib::MLPAllErrorsSubset(CMultilayerPerceptronShell &network,
                                 CMatrixDouble &xy,int setsize,
                                 CRowInt &subset,int subsetsize,
                                 CModelErrors &rep)
  {
   CMLPBase::MLPAllErrorsSubset(network.GetInnerObj(),xy,setsize,subset,subsetsize,rep);
  }
//+------------------------------------------------------------------+
//| Calculation of all types of errors on subset of dataset.         |
//| INPUT PARAMETERS:                                                |
//|   Network  -  network initialized with one of the network        |
//|               creation funcs                                     |
//|   XY       -  original dataset given by sparse matrix;           |
//|               one sample = one row; first NIn columns contain    |
//|               inputs, next NOut columns - desired outputs.       |
//|   SetSize  -  real size of XY, SetSize>=0;                       |
//|   Subset   -  subset of SubsetSize elements, array[SubsetSize];  |
//|   SubsetSize- number of elements in Subset[] array:              |
//|               * if SubsetSize>0, rows of XY with indices         |
//|                 Subset[0]......Subset[SubsetSize-1] are processed|
//|               * if SubsetSize=0, zeros are returned              |
//|               * if SubsetSize<0, entire dataset is processed;    |
//|                 Subset[] array is ignored in this case.          |
//| OUTPUT PARAMETERS:                                               |
//|   Rep      -  it contains all type of errors.                    |
//+------------------------------------------------------------------+
void CAlglib::MLPAllErrorsSparseSubset(CMultilayerPerceptronShell &network,
                                       CSparseMatrix &xy,int setsize,
                                       CRowInt &subset,int subsetsize,
                                       CModelErrors &rep)
  {
   CMLPBase::MLPAllErrorsSparseSubset(network.GetInnerObj(),xy,setsize,subset,subsetsize,rep);
  }
//+------------------------------------------------------------------+
//| Error of the neural network on subset of dataset.                |
//| INPUT PARAMETERS:                                                |
//|   Network  -  neural network;                                    |
//|   XY       -  training set, see below for information on the     |
//|               training set format;                               |
//|   SetSize  -  real size of XY, SetSize>=0;                       |
//|   Subset   -  subset of SubsetSize elements, array[SubsetSize];  |
//|   SubsetSize- number of elements in Subset[] array:              |
//|               * if SubsetSize>0, rows of XY with indices         |
//|                 Subset[0]......Subset[SubsetSize-1] are processed|
//|               * if SubsetSize=0, zeros are returned              |
//|               * if SubsetSize<0, entire dataset is  processed;   |
//|                 Subset[] array is ignored in this case.          |
//| RESULT:                                                          |
//|   sum-of-squares error, SUM(sqr(y[i]-desired_y[i])/2)            |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//|   For regression networks with NIn inputs and NOut outputs       |
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+NOut) matrix             |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, next NOut columns are       |
//|        outputs                                                   |
//|   For classification networks with NIn inputs and NClasses clases|
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+1) matrix                |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, last column stores class    |
//|        number (from 0 to NClasses-1).                            |
//+------------------------------------------------------------------+
double CAlglib::MLPErrorSubset(CMultilayerPerceptronShell &network,
                               CMatrixDouble &xy,int setsize,
                               CRowInt &subset,int subsetsize)
  {
   return(CMLPBase::MLPErrorSubset(network.GetInnerObj(),xy,setsize,subset,subsetsize));
  }
//+------------------------------------------------------------------+
//| Error of the neural network on subset of sparse dataset.         |
//| INPUT PARAMETERS:                                                |
//|   Network  -  neural network;                                    |
//|   XY       -  training set, see below for information on the     |
//|               training set format. This function checks          |
//|               correctness of the dataset (no NANs/INFs, class    |
//|               numbers are correct) and throws exception when     |
//|               incorrect dataset is passed. Sparse matrix must use|
//|               CRS format for storage.                            |
//|   SetSize  -  real size of XY, SetSize>=0;  it is used when      |
//|               SubsetSize<0;                                      |
//|   Subset   -  subset of SubsetSize elements, array[SubsetSize];  |
//|   SubsetSize- number of elements in Subset[] array:              |
//|               * if SubsetSize>0, rows of XY with indices         |
//|                 Subset[0]......Subset[SubsetSize-1] are processed|
//|               * if SubsetSize=0, zeros are returned              |
//|               * if SubsetSize<0, entire dataset is processed;    |
//|                 Subset[] array is ignored in this case.          |
//| RESULT:                                                          |
//|   sum-of-squares error, SUM(sqr(y[i]-desired_y[i])/2)            |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//|   For regression networks with NIn inputs and NOut outputs       |
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+NOut) matrix             |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, next NOut columns are       |
//|        outputs                                                   |
//|   For classification networks with NIn inputs and NClasses clases|
//|   following dataset format is used:                              |
//|      * dataset is given by NPoints*(NIn+1) matrix                |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, last column stores class    |
//|        number (from 0 to NClasses-1).                            |
//+------------------------------------------------------------------+
double CAlglib::MLPErrorSparseSubset(CMultilayerPerceptronShell &network,
                                     CSparseMatrix &xy,int setsize,
                                     CRowInt &subset,int subsetsize)
  {
   return(CMLPBase::MLPErrorSparseSubset(network.GetInnerObj(),xy,setsize,subset,subsetsize));
  }
//+------------------------------------------------------------------+
//| This subroutine trains logit model.                              |
//| INPUT PARAMETERS:                                                |
//|     XY          -   training set, array[0..NPoints-1,0..NVars]   |
//|                     First NVars columns store values of          |
//|                     independent variables, next column stores    |
//|                     number of class (from 0 to NClasses-1) which |
//|                     dataset element belongs to. Fractional values|
//|                     are rounded to nearest integer.              |
//|     NPoints     -   training set size, NPoints>=1                |
//|     NVars       -   number of independent variables, NVars>=1    |
//|     NClasses    -   number of classes, NClasses>=2               |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code:                                 |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NClasses-1].            |
//|                     * -1, if incorrect parameters was passed     |
//|                           (NPoints<NVars+2, NVars<1, NClasses<2).|
//|                     *  1, if task has been solved                |
//|     LM          -   model built                                  |
//|     Rep         -   training report                              |
//+------------------------------------------------------------------+
void CAlglib::MNLTrainH(CMatrixDouble &xy,const int npoints,
                        const int nvars,const int nclasses,
                        int &info,CLogitModelShell &lm,
                        CMNLReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLogit::MNLTrainH(xy,npoints,nvars,nclasses,info,lm.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Procesing                                                        |
//| INPUT PARAMETERS:                                                |
//|     LM      -   logit model, passed by non-constant reference    |
//|                 (some fields of structure are used as temporaries|
//|                 when calculating model output).                  |
//|     X       -   input vector,  array[0..NVars-1].                |
//|     Y       -   (possibly) preallocated buffer; if size of Y is  |
//|                 less than NClasses, it will be reallocated.If it |
//|                 is large enough, it is NOT reallocated, so we    |
//|                 can save some time on reallocation.              |
//| OUTPUT PARAMETERS:                                               |
//|     Y       -   result, array[0..NClasses-1]                     |
//|                 Vector of posterior probabilities for            |
//|                 classification task.                             |
//+------------------------------------------------------------------+
void CAlglib::MNLProcess(CLogitModelShell &lm,double &x[],double &y[])
  {
   CLogit::MNLProcess(lm.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MNLProcess(CLogitModelShell &lm,CRowDouble &x,CRowDouble &y)
  {
   CLogit::MNLProcess(lm.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| 'interactive' variant of MNLProcess for languages like Python    |
//| which support constructs like "Y=MNLProcess(LM,X)" and           |
//| interactive mode of the interpreter                              |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart,     |
//| but it is  more  convenient when you call it from command line.  |
//+------------------------------------------------------------------+
void CAlglib::MNLProcessI(CLogitModelShell &lm,double &x[],double &y[])
  {
   CLogit::MNLProcessI(lm.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MNLProcessI(CLogitModelShell &lm,CRowDouble &x,CRowDouble &y)
  {
   CLogit::MNLProcessI(lm.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| Unpacks coefficients of logit model. Logit model have form:      |
//|     P(class=i) = S(i) / (S(0) + S(1) + ... +S(M-1))              |
//|     S(i) = Exp(A[i,0]*X[0] + ... + A[i,N-1]*X[N-1] + A[i,N]),    |
//|            when i<M-1                                            |
//|     S(M-1) = 1                                                   |
//| INPUT PARAMETERS:                                                |
//|     LM          -   logit model in ALGLIB format                 |
//| OUTPUT PARAMETERS:                                               |
//|     V           -   coefficients, array[0..NClasses-2,0..NVars]  |
//|     NVars       -   number of independent variables              |
//|     NClasses    -   number of classes                            |
//+------------------------------------------------------------------+
void CAlglib::MNLUnpack(CLogitModelShell &lm,CMatrixDouble &a,
                        int &nvars,int &nclasses)
  {
//--- initialization
   nvars=0;
   nclasses=0;
//--- function call
   CLogit::MNLUnpack(lm.GetInnerObj(),a,nvars,nclasses);
  }
//+------------------------------------------------------------------+
//| "Packs" coefficients and creates logit model in ALGLIB format    |
//| (MNLUnpack reversed).                                            |
//| INPUT PARAMETERS:                                                |
//|     A           -   model (see MNLUnpack)                        |
//|     NVars       -   number of independent variables              |
//|     NClasses    -   number of classes                            |
//| OUTPUT PARAMETERS:                                               |
//|     LM          -   logit model.                                 |
//+------------------------------------------------------------------+
void CAlglib::MNLPack(CMatrixDouble &a,const int nvars,
                      const int nclasses,CLogitModelShell &lm)
  {
   CLogit::MNLPack(a,nvars,nclasses,lm.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Average cross-entropy (in bits per element) on the test set      |
//| INPUT PARAMETERS:                                                |
//|     LM      -   logit model                                      |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     CrossEntropy/(NPoints*ln(2)).                                |
//+------------------------------------------------------------------+
double CAlglib::MNLAvgCE(CLogitModelShell &lm,CMatrixDouble &xy,
                         const int npoints)
  {
   return(CLogit::MNLAvgCE(lm.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Relative classification error on the test set                    |
//| INPUT PARAMETERS:                                                |
//|     LM      -   logit model                                      |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     percent of incorrectly classified cases.                     |
//+------------------------------------------------------------------+
double CAlglib::MNLRelClsError(CLogitModelShell &lm,CMatrixDouble &xy,
                               const int npoints)
  {
   return(CLogit::MNLRelClsError(lm.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| RMS error on the test set                                        |
//| INPUT PARAMETERS:                                                |
//|     LM      -   logit model                                      |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     root mean square error (error when estimating posterior      |
//|     probabilities).                                              |
//+------------------------------------------------------------------+
double CAlglib::MNLRMSError(CLogitModelShell &lm,CMatrixDouble &xy,
                            const int npoints)
  {
   return(CLogit::MNLRMSError(lm.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average error on the test set                                    |
//| INPUT PARAMETERS:                                                |
//|     LM      -   logit model                                      |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     average error (error when estimating posterior               |
//|     probabilities).                                              |
//+------------------------------------------------------------------+
double CAlglib::MNLAvgError(CLogitModelShell &lm,CMatrixDouble &xy,
                            const int npoints)
  {
   return(CLogit::MNLAvgError(lm.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average relative error on the test set                           |
//| INPUT PARAMETERS:                                                |
//|     LM      -   logit model                                      |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     average relative error (error when estimating posterior      |
//|     probabilities).                                              |
//+------------------------------------------------------------------+
double CAlglib::MNLAvgRelError(CLogitModelShell &lm,CMatrixDouble &xy,
                               const int ssize)
  {
   return(CLogit::MNLAvgRelError(lm.GetInnerObj(),xy,ssize));
  }
//+------------------------------------------------------------------+
//| Classification error on test set = MNLRelClsError*NPoints        |
//+------------------------------------------------------------------+
int CAlglib::MNLClsError(CLogitModelShell &lm,CMatrixDouble &xy,
                         const int npoints)
  {
   return(CLogit::MNLClsError(lm.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| DESCRIPTION:                                                     |
//| This function creates MCPD (Markov Chains for Population Data)   |
//| solver.                                                          |
//| This solver can be used to find transition matrix P for          |
//| N-dimensional prediction problem where transition from X[i] to   |
//|     X[i+1] is modelled as X[i+1] = P*X[i]                        |
//| where X[i] and X[i+1] are N-dimensional population vectors       |
//| (components of each X are non-negative), and P is a N*N          |
//| transition matrix (elements of   are non-negative, each column   |
//| sums to 1.0).                                                    |
//| Such models arise when when:                                     |
//| * there is some population of individuals                        |
//| * individuals can have different states                          |
//| * individuals can transit from one state to another              |
//| * population size is constant, i.e. there is no new individuals  |
//|   and no one leaves population                                   |
//| * you want to model transitions of individuals from one state    |
//|   into another                                                   |
//| USAGE:                                                           |
//| Here we give very brief outline of the MCPD. We strongly         |
//| recommend you to read examples in the ALGLIB Reference Manual    |
//| and to read ALGLIB User Guide on data analysis which is          |
//| available at http://www.alglib.net/dataanalysis/                 |
//| 1. User initializes algorithm state with MCPDCreate() call       |
//| 2. User adds one or more tracks -  sequences of states which     |
//|    describe evolution of a system being modelled from different  |
//|    starting conditions                                           |
//| 3. User may add optional boundary, equality and/or linear        |
//|    constraints on the coefficients of P by calling one of the    |
//|    following functions:                                          |
//|    * MCPDSetEC() to set equality constraints                     |
//|    * MCPDSetBC() to set bound constraints                        |
//|    * MCPDSetLC() to set linear constraints                       |
//| 4. Optionally, user may set custom weights for prediction errors |
//|    (by default, algorithm assigns non-equal, automatically chosen|
//|    weights for errors in the prediction of different components  |
//|    of X). It can be done with a call of                          |
//|    MCPDSetPredictionWeights() function.                          |
//| 5. User calls MCPDSolve() function which takes algorithm state   |
//|    and pointer (delegate, etc.) to callback function which       |
//|    calculates F/G.                                               |
//| 6. User calls MCPDResults() to get solution                      |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>=1                          |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure stores algorithm state                 |
//+------------------------------------------------------------------+
void CAlglib::MCPDCreate(const int n,CMCPDStateShell &s)
  {
   CMarkovCPD::MCPDCreate(n,s.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| DESCRIPTION:                                                     |
//| This function is a specialized version of MCPDCreate() function, |
//| and we recommend  you  to read comments for this function for    |
//| general information about MCPD solver.                           |
//| This function creates MCPD (Markov Chains for Population Data)   |
//| solver for "Entry-state" model, i.e. model where transition from |
//| X[i] to X[i+1] is modelled as                                    |
//|     X[i+1] = P*X[i]                                              |
//| where                                                            |
//|     X[i] and X[i+1] are N-dimensional state vectors              |
//|     P is a N*N transition matrix                                 |
//| and  one  selected component of X[] is called "entry" state and  |
//| is treated in a special way:                                     |
//|     system state always transits from "entry" state to some      |
//|     another state                                                |
//|     system state can not transit from any state into "entry"     |
//|     state                                                        |
//| Such conditions basically mean that row of P which corresponds to|
//| "entry" state is zero.                                           |
//| Such models arise when:                                          |
//| * there is some population of individuals                        |
//| * individuals can have different states                          |
//| * individuals can transit from one state to another              |
//| * population size is NOT constant -  at every moment of time     |
//|   there is some (unpredictable) amount of "new" individuals,     |
//|   which can transit into one of the states at the next turn, but |
//|   still no one leaves population                                 |
//| * you want to model transitions of individuals from one state    |
//|   into another                                                   |
//|*but you do NOT want to predict amount of "new" individuals     |
//|   because it does not depends on individuals already present     |
//|   (hence system can not transit INTO entry state - it can only   |
//|   transit FROM it).                                              |
//| This model is discussed in more details in the ALGLIB User Guide |
//| (see http://www.alglib.net/dataanalysis/ for more data).         |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>=2                          |
//|     EntryState- index of entry state, in 0..N-1                  |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure stores algorithm state                 |
//+------------------------------------------------------------------+
void CAlglib::MCPDCreateEntry(const int n,const int entrystate,
                              CMCPDStateShell &s)
  {
   CMarkovCPD::MCPDCreateEntry(n,entrystate,s.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| DESCRIPTION:                                                     |
//| This function is a specialized version of MCPDCreate() function, |
//| and we recommend  you  to read comments for this function for    |
//| general information about MCPD solver.                           |
//| This function creates MCPD (Markov Chains for Population Data)   |
//| solver for "Exit-state" model, i.e. model where transition from  |
//| X[i] to X[i+1] is modelled as                                    |
//|     X[i+1] = P*X[i]                                              |
//| where                                                            |
//|     X[i] and X[i+1] are N-dimensional state vectors              |
//|     P is a N*N transition matrix                                 |
//| and  one  selected component of X[] is called "exit" state and   |
//| is treated in a special way:                                     |
//|     system state can transit from any state into "exit" state    |
//|     system state can not transit from "exit" state into any other|
//|     state transition operator discards "exit" state (makes it    |
//|     zero at each turn)                                           |
//| Such conditions basically mean that column of P which            |
//| corresponds to "exit" state is zero. Multiplication by such P    |
//| may decrease sum of vector components.                           |
//| Such models arise when:                                          |
//| * there is some population of individuals                        |
//| * individuals can have different states                          |
//| * individuals can transit from one state to another              |
//| * population size is NOT constant - individuals can move into    |
//|   "exit" state and leave population at the next turn, but there  |
//|   are no new individuals                                         |
//| * amount of individuals which leave population can be predicted  |
//| * you want to model transitions of individuals from one state    |
//|   into another (including transitions into the "exit" state)     |
//| This model is discussed in more details in the ALGLIB User Guide |
//| (see http://www.alglib.net/dataanalysis/ for more data).         |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>=2                          |
//|     ExitState-  index of exit state, in 0..N-1                   |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure stores algorithm state                 |
//+------------------------------------------------------------------+
void CAlglib::MCPDCreateExit(const int n,const int exitstate,
                             CMCPDStateShell &s)
  {
   CMarkovCPD::MCPDCreateExit(n,exitstate,s.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| DESCRIPTION:                                                     |
//| This function is a specialized version of MCPDCreate() function, |
//| and we recommend you to read comments for this function for      |
//| general information about MCPD solver.                           |
//| This function creates MCPD (Markov Chains for Population Data)   |
//| solver for "Entry-Exit-states" model, i.e. model where transition|
//| from X[i] to X[i+1] is modelled as                               |
//|     X[i+1] = P*X[i]                                              |
//| where                                                            |
//|     X[i] and X[i+1] are N-dimensional state vectors              |
//|     P is a N*N transition matrix                                 |
//| one selected component of X[] is called "entry" state and is a   |
//| treated in special way:                                          |
//|     system state always transits from "entry" state to some      |
//|     another state                                                |
//|     system state can not transit from any state into "entry"     |
//|     state                                                        |
//| and another one component of X[] is called "exit" state and is   |
//| treated in a special way too:                                    |
//|     system state can transit from any state into "exit" state    |
//|     system state can not transit from "exit" state into any other|
//|     state transition operator discards "exit" state (makes it    |
//|     zero at each turn)                                           |
//| Such conditions basically mean that:                             |
//|     row of P which corresponds to "entry" state is zero          |
//|     column of P which corresponds to "exit" state is zero        |
//| Multiplication by such P may decrease sum of vector components.  |
//| Such models arise when:                                          |
//| * there is some population of individuals                        |
//| * individuals can have different states                          |
//| * individuals can transit from one state to another              |
//| * population size is NOT constant                                |
//| * at every moment of time there is some (unpredictable) amount   |
//|   of "new" individuals, which can transit into one of the states |
//|   at the next turn                                               |
//|*some individuals can move (predictably) into "exit" state      |
//|   and leave population at the next turn                          |
//| * you want to model transitions of individuals from one state    |
//|   into another, including transitions from the "entry" state and |
//|   into the "exit" state.                                         |
//|*but you do NOT want to predict amount of "new" individuals     |
//|   because it does not depends on individuals already present     |
//|   (hence system can not transit INTO entry state - it can only   |
//|   transit FROM it).                                              |
//| This model is discussed  in  more  details  in  the ALGLIB User  |
//| Guide (see http://www.alglib.net/dataanalysis/ for more data).   |
//| INPUT PARAMETERS:                                                |
//|     N       -   problem dimension, N>=2                          |
//|     EntryState- index of entry state, in 0..N-1                  |
//|     ExitState-  index of exit state, in 0..N-1                   |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure stores algorithm state                 |
//+------------------------------------------------------------------+
void CAlglib::MCPDCreateEntryExit(const int n,const int entrystate,
                                  const int exitstate,CMCPDStateShell &s)
  {
   CMarkovCPD::MCPDCreateEntryExit(n,entrystate,exitstate,s.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function is used to add a track - sequence of system states |
//| at the different moments of its evolution.                       |
//| You may add one or several tracks to the MCPD solver. In case you|
//| have several tracks, they won't overwrite each other. For        |
//| example, if you pass two tracks, A1-A2-A3 (system at t=A+1, t=A+2|
//| and t=A+3) and B1-B2-B3, then solver will try to model           |
//| transitions from t=A+1 to t=A+2, t=A+2 to t=A+3, t=B+1 to t=B+2, |
//| t=B+2 to t=B+3. But it WONT mix these two tracks - i.e. it wont  |
//| try to model transition from t=A+3 to t=B+1.                     |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     XY      -   track, array[K, N]:                               |
//|                 * I-th row is a state at t=I                     |
//|                 * elements of XY must be non-negative (exception |
//|                   will be thrown on negative elements)           |
//|     K       -   number of points in a track                      |
//|                 * if given, only leading K rows of XY are used   |
//|                 * if not given, automatically determined from    |
//|                   size of XY                                     |
//| NOTES:                                                           |
//| 1. Track may contain either proportional or population data:     |
//|    * with proportional data all rows of XY must sum to 1.0, i.e. |
//|      we have proportions instead of absolute population values   |
//|    * with population data rows of XY contain population counts   |
//|      and generally do not sum to 1.0 (although they still must be|
//|      non-negative)                                               |
//+------------------------------------------------------------------+
void CAlglib::MCPDAddTrack(CMCPDStateShell &s,CMatrixDouble &xy,
                           const int k)
  {
   CMarkovCPD::MCPDAddTrack(s.GetInnerObj(),xy,k);
  }
//+------------------------------------------------------------------+
//| This function is used to add a track - sequence of system states |
//| at the different moments of its evolution.                       |
//| You may add one or several tracks to the MCPD solver. In case you|
//| have several tracks, they won't overwrite each other. For        |
//| example, if you pass two tracks, A1-A2-A3 (system at t=A+1, t=A+2|
//| and t=A+3) and B1-B2-B3, then solver will try to model           |
//| transitions from t=A+1 to t=A+2, t=A+2 to t=A+3, t=B+1 to t=B+2, |
//| t=B+2 to t=B+3. But it WONT mix these two tracks - i.e. it wont  |
//| try to model transition from t=A+3 to t=B+1.                     |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     XY      -   track, array[K,N]:                               |
//|                 * I-th row is a state at t=I                     |
//|                 * elements of XY must be non-negative (exception |
//|                   will be thrown on negative elements)           |
//|     K       -   number of points in a track                      |
//|                 * if given, only leading K rows of XY are used   |
//|                 * if not given, automatically determined from    |
//|                   size of XY                                     |
//| NOTES:                                                           |
//| 1. Track may contain either proportional or population data:     |
//|    * with proportional data all rows of XY must sum to 1.0, i.e. |
//|      we have proportions instead of absolute population values   |
//|    * with population data rows of XY contain population counts   |
//|      and generally do not sum to 1.0 (although they still must be|
//|      non-negative)                                               |
//+------------------------------------------------------------------+
void CAlglib::MCPDAddTrack(CMCPDStateShell &s,CMatrixDouble &xy)
  {
//--- initialization
   int k=(int)CAp::Rows(xy);
//--- function call
   CMarkovCPD::MCPDAddTrack(s.GetInnerObj(),xy,k);
  }
//+------------------------------------------------------------------+
//| This function is used to add equality constraints on the elements|
//| of the transition matrix P.                                      |
//| MCPD solver has four types of constraints which can be placed    |
//| on P:                                                            |
//| * user-specified equality constraints (optional)                 |
//| * user-specified bound constraints (optional)                    |
//| * user-specified general linear constraints (optional)           |
//| * basic constraints (always present):                            |
//|   * non-negativity: P[i,j]>=0                                    |
//|   * consistency: every column of P sums to 1.0                   |
//| Final constraints which are passed to the underlying optimizer   |
//| are calculated as intersection of all present constraints. For   |
//| example, you may specify boundary constraint on P[0,0] and       |
//| equality one:                                                    |
//|     0.1<=P[0,0]<=0.9                                             |
//|     P[0,0]=0.5                                                   |
//| Such combination of constraints will be silently reduced to their|
//| intersection, which is P[0,0]=0.5.                               |
//| This function can be used to place equality constraints on       |
//| arbitrary subset of elements of P. Set of constraints is         |
//| specified by EC, which may contain either NAN's or finite numbers|
//| from [0,1]. NAN denotes absence of constraint, finite number     |
//| denotes equality constraint on specific element of P.            |
//| You can also use MCPDAddEC() function which allows to ADD        |
//| equality constraint for one element of P without changing        |
//| constraints for other elements.                                  |
//| These functions (MCPDSetEC and MCPDAddEC) interact as follows:   |
//| * there is internal matrix of equality constraints which is      |
//|   stored in the MCPD solver                                      |
//| * MCPDSetEC() replaces this matrix by another one (SET)          |
//| * MCPDAddEC() modifies one element of this matrix and leaves     |
//|   other ones unchanged (ADD)                                     |
//| * thus MCPDAddEC() call preserves all modifications done by      |
//|   previous calls, while MCPDSetEC() completely discards all      |
//|   changes done to the equality constraints.                      |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     EC      -   equality constraints, array[N,N]. Elements of EC |
//|                 can be either NAN's or finite numbers from [0,1].|
//|                 NAN denotes absence of constraints, while finite |
//|                 value denotes equality constraint on the         |
//|                 corresponding element of P.                      |
//| NOTES:                                                           |
//| 1. infinite values of EC will lead to exception being thrown.    |
//| Values less than 0.0 or greater than 1.0 will lead to error code |
//| being returned after call to MCPDSolve().                        |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetEC(CMCPDStateShell &s,CMatrixDouble &ec)
  {
   CMarkovCPD::MCPDSetEC(s.GetInnerObj(),ec);
  }
//+------------------------------------------------------------------+
//| This function is used to add equality constraints on the elements|
//| of the transition matrix P.                                      |
//| MCPD solver has four types of constraints which can be placed    |
//| on P:                                                            |
//| * user-specified equality constraints (optional)                 |
//| * user-specified bound constraints (optional)                    |
//| * user-specified general linear constraints (optional)           |
//| * basic constraints (always present):                            |
//|   * non-negativity: P[i,j]>=0                                    |
//|   * consistency: every column of P sums to 1.0                   |
//| Final constraints which are passed to the underlying optimizer   |
//| are calculated as intersection of all present constraints. For   |
//| example, you may specify boundary constraint on P[0,0] and       |
//| equality one:                                                    |
//|     0.1<=P[0,0]<=0.9                                             |
//|     P[0,0]=0.5                                                   |
//| Such combination of constraints will be silently reduced to their|
//| intersection, which is P[0,0]=0.5.                               |
//| This function can be used to ADD equality constraint for one     |
//| element of P without changing constraints for other elements.    |
//| You can also use MCPDSetEC() function which allows you to specify|
//| arbitrary set of equality constraints in one call.               |
//| These functions (MCPDSetEC and MCPDAddEC) interact as follows:   |
//| * there is internal matrix of equality constraints which is      |
//|   stored in the MCPD solver                                      |
//| * MCPDSetEC() replaces this matrix by another one (SET)          |
//| * MCPDAddEC() modifies one element of this matrix and leaves     |
//|   other ones unchanged (ADD)                                     |
//| * thus MCPDAddEC() call preserves all modifications done by      |
//|   previous calls, while MCPDSetEC() completely discards all      |
//|   changes done to the equality constraints.                      |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     I       -   row index of element being constrained           |
//|     J       -   column index of element being constrained        |
//|     C       -   value (constraint for P[I,J]). Can be either NAN |
//|                 (no constraint) or finite value from [0,1].      |
//| NOTES:                                                           |
//| 1. infinite values of C will lead to exception being thrown.     |
//| Values less than 0.0 or greater than 1.0 will lead to error code |
//| being returned after call to MCPDSolve().                        |
//+------------------------------------------------------------------+
void CAlglib::MCPDAddEC(CMCPDStateShell &s,const int i,const int j,
                        const double c)
  {
   CMarkovCPD::MCPDAddEC(s.GetInnerObj(),i,j,c);
  }
//+------------------------------------------------------------------+
//| This function is used to add bound constraints on the elements   |
//| of the transition matrix P.                                      |
//| MCPD solver has four types of constraints which can be placed    |
//| on P:                                                            |
//| * user-specified equality constraints (optional)                 |
//| * user-specified bound constraints (optional)                    |
//| * user-specified general linear constraints (optional)           |
//| * basic constraints (always present):                            |
//|   * non-negativity: P[i,j]>=0                                    |
//|   * consistency: every column of P sums to 1.0                   |
//| Final constraints which are passed to the underlying optimizer   |
//| are calculated as intersection of all present constraints. For   |
//| example, you may specify boundary constraint on P[0,0] and       |
//| equality one:                                                    |
//|     0.1<=P[0,0]<=0.9                                             |
//|     P[0,0]=0.5                                                   |
//| Such combination of constraints will be silently reduced to their|
//| intersection, which is P[0,0]=0.5.                               |
//| This function can be used to place bound constraints on arbitrary|
//| subset of elements of P. Set of constraints is specified by      |
//| BndL/BndU matrices, which may contain arbitrary combination of   |
//| finite numbers or infinities (like -INF<x<=0.5 or 0.1<=x<+INF).  |
//| You can also use MCPDAddBC() function which allows to ADD bound  |
//| constraint for one element of P without changing constraints for |
//| other elements.                                                  |
//| These functions (MCPDSetBC and MCPDAddBC) interact as follows:   |
//| * there is internal matrix of bound constraints which is stored  |
//|   in the MCPD solver                                             |
//| * MCPDSetBC() replaces this matrix by another one (SET)          |
//| * MCPDAddBC() modifies one element of this matrix and leaves     |
//|   other ones unchanged (ADD)                                     |
//| * thus MCPDAddBC() call preserves all modifications done by      |
//|   previous calls, while MCPDSetBC() completely discards all      |
//|   changes done to the equality constraints.                      |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     BndL    -   lower bounds constraints, array[N,N]. Elements of|
//|                 BndL can be finite numbers or -INF.              |
//|     BndU    -   upper bounds constraints, array[N,N]. Elements of|
//|                 BndU can be finite numbers or +INF.              |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetBC(CMCPDStateShell &s,CMatrixDouble &bndl,
                        CMatrixDouble &bndu)
  {
   CMarkovCPD::MCPDSetBC(s.GetInnerObj(),bndl,bndu);
  }
//+------------------------------------------------------------------+
//| This function is used to add bound constraints on the elements   |
//| of the transition matrix P.                                      |
//| MCPD solver has four types of constraints which can be placed    |
//| on P:                                                            |
//| * user-specified equality constraints (optional)                 |
//| * user-specified bound constraints (optional)                    |
//| * user-specified general linear constraints (optional)           |
//| * basic constraints (always present):                            |
//|   * non-negativity: P[i,j]>=0                                    |
//|   * consistency: every column of P sums to 1.0                   |
//| Final constraints which are passed to the underlying optimizer   |
//| are calculated as intersection of all present constraints. For   |
//| example, you may specify boundary constraint on P[0,0] and       |
//| equality one:                                                    |
//|     0.1<=P[0,0]<=0.9                                             |
//|     P[0,0]=0.5                                                   |
//| Such combination of constraints will be silently reduced to their|
//| intersection, which is P[0,0]=0.5.                               |
//| This function can be used to ADD bound constraint for one element|
//| of P without changing constraints for other elements.            |
//| You can also use MCPDSetBC() function which allows to place bound|
//| constraints on arbitrary subset of elements of P. Set of         |
//| constraints is specified  by  BndL/BndU matrices, which may      |
//| contain arbitrary combination of finite numbers or infinities    |
//| (like -INF<x<=0.5 or 0.1<=x<+INF).                               |
//| These functions (MCPDSetBC and MCPDAddBC) interact as follows:   |
//| * there is internal matrix of bound constraints which is stored  |
//|   in the MCPD solver                                             |
//| * MCPDSetBC() replaces this matrix by another one (SET)          |
//| * MCPDAddBC() modifies one element of this matrix and leaves     |
//|   other ones unchanged (ADD)                                     |
//| * thus MCPDAddBC() call preserves all modifications done by      |
//|   previous calls, while MCPDSetBC() completely discards all      |
//|   changes done to the equality constraints.                      |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     I       -   row index of element being constrained           |
//|     J       -   column index of element being constrained        |
//|     BndL    -   lower bound                                      |
//|     BndU    -   upper bound                                      |
//+------------------------------------------------------------------+
void CAlglib::MCPDAddBC(CMCPDStateShell &s,const int i,const int j,
                        const double bndl,const double bndu)
  {
   CMarkovCPD::MCPDAddBC(s.GetInnerObj(),i,j,bndl,bndu);
  }
//+------------------------------------------------------------------+
//| This function is used to set linear equality/inequality          |
//| constraints on the elements of the transition matrix P.          |
//| This function can be used to set one or several general linear   |
//| constraints on the elements of P. Two types of constraints are   |
//| supported:                                                       |
//| * equality constraints                                           |
//| * inequality constraints (both less-or-equal and                 |
//|   greater-or-equal)                                              |
//| Coefficients of constraints are specified by matrix C (one of the|
//| parameters). One row of C corresponds to one constraint.         |
//| Because transition matrix P has N*N elements, we need N*N columns|
//| to store all coefficients  (they  are  stored row by row), and   |
//| one more column to store right part - hence C has N*N+1 columns. |
//| Constraint kind is stored in the CT array.                       |
//| Thus, I-th linear constraint is                                  |
//|     P[0,0]*C[I,0] + P[0,1]*C[I,1] + .. + P[0,N-1]*C[I,N-1] +     |
//|         + P[1,0]*C[I,N] + P[1,1]*C[I,N+1] + ... +                |
//|         + P[N-1,N-1]*C[I,N*N-1]  ?=?  C[I,N*N]                   |
//| where ?=? can be either "=" (CT[i]=0), "<=" (CT[i]<0) or ">="    |
//| (CT[i]>0).                                                       |
//| Your constraint may involve only some subset of P (less than N*N |
//| elements).                                                       |
//| For example it can be something like                             |
//|     P[0,0] + P[0,1] = 0.5                                        |
//| In this case you still should pass matrix  with N*N+1 columns,   |
//| but all its elements (except for C[0,0], C[0,1] and C[0,N*N-1])  |
//| will be zero.                                                    |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     C       -   array[K,N*N+1] - coefficients of constraints     |
//|                 (see above for complete description)             |
//|     CT      -   array[K] - constraint types                      |
//|                 (see above for complete description)             |
//|     K       -   number of equality/inequality constraints, K>=0: |
//|                 * if given, only leading K elements of C/CT are  |
//|                   used                                           |
//|                 * if not given, automatically determined from    |
//|                   sizes of C/CT                                  |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetLC(CMCPDStateShell &s,CMatrixDouble &c,
                        int &ct[],const int k)
  {
   CMarkovCPD::MCPDSetLC(s.GetInnerObj(),c,ct,k);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetLC(CMCPDStateShell &s,CMatrixDouble &c,
                        CRowInt &ct,const int k)
  {
   CMarkovCPD::MCPDSetLC(s.GetInnerObj(),c,ct,k);
  }
//+------------------------------------------------------------------+
//| This function is used to set linear equality/inequality          |
//| constraints on the elements of the transition matrix P.          |
//| This function can be used to set one or several general linear   |
//| constraints on the elements of P. Two types of constraints are   |
//| supported:                                                       |
//| * equality constraints                                           |
//| * inequality constraints (both less-or-equal and                 |
//|   greater-or-equal)                                              |
//| Coefficients of constraints are specified by matrix C (one of the|
//| parameters). One row of C corresponds to one constraint.         |
//| Because transition matrix P has N*N elements, we need N*N columns|
//| to store all coefficients  (they  are  stored row by row), and   |
//| one more column to store right part - hence C has N*N+1 columns. |
//| Constraint kind is stored in the CT array.                       |
//| Thus, I-th linear constraint is                                  |
//|     P[0,0]*C[I,0] + P[0,1]*C[I,1] + .. + P[0,N-1]*C[I,N-1] +     |
//|         + P[1,0]*C[I,N] + P[1,1]*C[I,N+1] + ... +                |
//|         + P[N-1,N-1]*C[I,N*N-1]  ?=?  C[I,N*N]                   |
//| where ?=? can be either "=" (CT[i]=0), "<=" (CT[i]<0) or ">="    |
//| (CT[i]>0).                                                       |
//| Your constraint may involve only some subset of P (less than N*N |
//| elements).                                                       |
//| For example it can be something like                             |
//|     P[0,0] + P[0,1] = 0.5                                        |
//| In this case you still should pass matrix  with N*N+1 columns,   |
//| but all its elements (except for C[0,0], C[0,1] and C[0,N*N-1])  |
//| will be zero.                                                    |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     C       -   array[K,N*N+1] - coefficients of constraints     |
//|                 (see above for complete description)             |
//|     CT      -   array[K] - constraint types                      |
//|                 (see above for complete description)             |
//|     K       -   number of equality/inequality constraints, K>=0: |
//|                 * if given, only leading K elements of C/CT are  |
//|                   used                                           |
//|                 * if not given, automatically determined from    |
//|                   sizes of C/CT                                  |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetLC(CMCPDStateShell &s,CMatrixDouble &c,int &ct[])
  {
//--- check
   if((CAp::Rows(c)!=CAp::Len(ct)))
     {
      Print("Error while calling " + __FUNCTION__ + ": looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int k=(int)CAp::Rows(c);
//--- function call
   CMarkovCPD::MCPDSetLC(s.GetInnerObj(),c,ct,k);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetLC(CMCPDStateShell &s,CMatrixDouble &c,CRowInt &ct)
  {
//--- check
   if((CAp::Rows(c)!=CAp::Len(ct)))
     {
      Print("Error while calling " + __FUNCTION__ + ": looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int k=(int)CAp::Rows(c);
//--- function call
   CMarkovCPD::MCPDSetLC(s.GetInnerObj(),c,ct,k);
  }
//+------------------------------------------------------------------+
//| This function allows to tune amount of Tikhonov regularization   |
//| being applied to your problem.                                   |
//| By default, regularizing term is equal to r*||P-prior_P||^2,     |
//| where r is a small non-zero value,  P is transition matrix,      |
//| prior_P is identity matrix, ||X||^2 is a sum of squared elements |
//| of X.                                                            |
//| This function allows you to change coefficient r. You can also   |
//| change prior values with MCPDSetPrior() function.                |
//| INPUT PARAMETERS:                                                |
//|     S      -   solver                                            |
//|     V      -   regularization  coefficient, finite non-negative  |
//|                value. It is not recommended to specify zero      |
//|                value unless you are pretty sure that you want it.|
//+------------------------------------------------------------------+
void CAlglib::MCPDSetTikhonovRegularizer(CMCPDStateShell &s,
                                         const double v)
  {
   CMarkovCPD::MCPDSetTikhonovRegularizer(s.GetInnerObj(),v);
  }
//+------------------------------------------------------------------+
//| This function allows to set prior values used for regularization |
//| of your problem.                                                 |
//| By default, regularizing term is equal to r*||P-prior_P||^2,     |
//| where r is a small non-zero value,  P is transition matrix,      |
//| prior_P is identity matrix, ||X||^2 is a sum of squared elements |
//| of X.                                                            |
//| This function allows you to change prior values prior_P. You can |
//| also change r with MCPDSetTikhonovRegularizer() function.        |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     PP      -   array[N,N], matrix of prior values:              |
//|                 1. elements must be real numbers from [0,1]      |
//|                 2. columns must sum to 1.0.                      |
//|                 First property is checked (exception is thrown   |
//|                 otherwise), while second one is not              |
//|                 checked/enforced.                                |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetPrior(CMCPDStateShell &s,CMatrixDouble &pp)
  {
   CMarkovCPD::MCPDSetPrior(s.GetInnerObj(),pp);
  }
//+------------------------------------------------------------------+
//| This function is used to change prediction weights               |
//| MCPD solver scales prediction errors as follows                  |
//|     Error(P) = ||W*(y-P*x)||^2                                   |
//| where                                                            |
//|     x is a system state at time t                                |
//|     y is a system state at time t+1                              |
//|     P is a transition matrix                                     |
//|     W is a diagonal scaling matrix                               |
//| By default, weights are chosen in order to minimize relative     |
//| prediction error instead of absolute one. For example, if one    |
//| component of state is about 0.5 in magnitude and another one is  |
//| about 0.05, then algorithm will make corresponding weights equal |
//| to 2.0 and 20.0.                                                 |
//| INPUT PARAMETERS:                                                |
//|     S       -   solver                                           |
//|     PW      -   array[N], weights:                               |
//|                 * must be non-negative values (exception will be |
//|                 thrown otherwise)                                |
//|                 * zero values will be replaced by automatically  |
//|                 chosen values                                    |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetPredictionWeights(CMCPDStateShell &s,
                                       double &pw[])
  {
   CMarkovCPD::MCPDSetPredictionWeights(s.GetInnerObj(),pw);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::MCPDSetPredictionWeights(CMCPDStateShell &s,
                                       CRowDouble &pw)
  {
   CMarkovCPD::MCPDSetPredictionWeights(s.GetInnerObj(),pw);
  }
//+------------------------------------------------------------------+
//| This function is used to start solution of the MCPD problem.     |
//| After return from this function, you can use MCPDResults() to get|
//| solution and completion code.                                    |
//+------------------------------------------------------------------+
void CAlglib::MCPDSolve(CMCPDStateShell &s)
  {
   CMarkovCPD::MCPDSolve(s.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| MCPD results                                                     |
//| INPUT PARAMETERS:                                                |
//|     State   -   algorithm state                                  |
//| OUTPUT PARAMETERS:                                               |
//|     P      -   array[N,N], transition matrix                     |
//|     Rep    -   optimization report. You should check Rep.        |
//|                TerminationType in order to distinguish successful|
//|                termination from unsuccessful one. Speaking short,|
//|                positive values denote success, negative ones are |
//|                failures. More information about fields of this   |
//|                structure  can befound in the comments on         |
//|                MCPDReport datatype.                              |
//+------------------------------------------------------------------+
void CAlglib::MCPDResults(CMCPDStateShell &s,CMatrixDouble &p,
                          CMCPDReportShell &rep)
  {
   CMarkovCPD::MCPDResults(s.GetInnerObj(),p,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Neural network training using modified Levenberg-Marquardt with  |
//| exact Hessian calculation and regularization. Subroutine trains  |
//| neural network with restarts from random positions. Algorithm is |
//| well suited for small                                            |
//| and medium scale problems (hundreds of weights).                 |
//| INPUT PARAMETERS:                                                |
//|     Network     -   neural network with initialized geometry     |
//|     XY          -   training set                                 |
//|     NPoints     -   training set size                            |
//|     Decay       -   weight decay constant, >=0.001               |
//|                     Decay term 'Decay*||Weights||^2' is added to |
//|                     error function.                              |
//|                     If you don't know what Decay to choose, use  |
//|                     0.001.                                       |
//|     Restarts    -   number of restarts from random position, >0. |
//|                     If you don't know what Restarts to choose,   |
//|                     use 2.                                       |
//| OUTPUT PARAMETERS:                                               |
//|     Network     -   trained neural network.                      |
//|     Info        -   return code:                                 |
//|                     * -9, if internal matrix inverse subroutine  |
//|                           failed                                 |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NOut-1].                |
//|                     * -1, if wrong parameters specified          |
//|                           (NPoints<0, Restarts<1).               |
//|                     *  2, if task has been solved.               |
//|     Rep         -   training report                              |
//+------------------------------------------------------------------+
void CAlglib::MLPTrainLM(CMultilayerPerceptronShell &network,
                         CMatrixDouble &xy,const int npoints,
                         const double decay,const int restarts,
                         int &info,CMLPReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CMLPTrain::MLPTrainLM(network.GetInnerObj(),xy,npoints,decay,restarts,info,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Neural network training using L-BFGS algorithm with              |
//| regularization. Subroutine trains neural network with restarts   |
//| from random positions. Algorithm is well suited for problems of  |
//| any dimensionality (memory requirements and step complexity are  |
//| linear by weights number).                                       |
//| INPUT PARAMETERS:                                                |
//|     Network    -   neural network with initialized geometry      |
//|     XY         -   training set                                  |
//|     NPoints    -   training set size                             |
//|     Decay      -   weight decay constant, >=0.001                |
//|                    Decay term 'Decay*||Weights||^2' is added to  |
//|                    error function.                               |
//|                    If you don't know what Decay to choose, use   |
//|                    0.001.                                        |
//|     Restarts   -   number of restarts from random position, >0.  |
//|                    If you don't know what Restarts to choose,    |
//|                    use 2.                                        |
//|     WStep      -   stopping criterion. Algorithm stops if step   |
//|                    size is less than WStep. Recommended          |
//|                    value - 0.01. Zero step size means stopping   |
//|                    after MaxIts iterations.                      |
//|     MaxIts     -   stopping criterion. Algorithm stops after     |
//|                    MaxIts iterations (NOT gradient calculations).|
//|                    Zero MaxIts means stopping when step is       |
//|                    sufficiently small.                           |
//| OUTPUT PARAMETERS:                                               |
//|     Network     -   trained neural network.                      |
//|     Info        -   return code:                                 |
//|                     * -8, if both WStep=0 and MaxIts=0           |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NOut-1].                |
//|                     * -1, if wrong parameters specified          |
//|                           (NPoints<0, Restarts<1).               |
//|                     *  2, if task has been solved.               |
//|     Rep         -   training report                              |
//+------------------------------------------------------------------+
void CAlglib::MLPTrainLBFGS(CMultilayerPerceptronShell &network,
                            CMatrixDouble &xy,const int npoints,
                            const double decay,const int restarts,
                            const double wstep,int maxits,
                            int &info,CMLPReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CMLPTrain::MLPTrainLBFGS(network.GetInnerObj(),xy,npoints,decay,restarts,wstep,maxits,info,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Neural network training using early stopping (base algorithm -   |
//| L-BFGS with regularization).                                     |
//| INPUT PARAMETERS:                                                |
//|     Network     -   neural network with initialized geometry     |
//|     TrnXY       -   training set                                 |
//|     TrnSize     -   training set size                            |
//|     ValXY       -   validation set                               |
//|     ValSize     -   validation set size                          |
//|     Decay       -   weight decay constant, >=0.001               |
//|                     Decay term 'Decay*||Weights||^2' is added to |
//|                     error function.                              |
//|                     If you don't know what Decay to choose, use  |
//|                     0.001.                                       |
//|     Restarts    -   number of restarts from random position, >0. |
//|                     If you don't know what Restarts to choose,   |
//|                     use 2.                                       |
//| OUTPUT PARAMETERS:                                               |
//|     Network     -   trained neural network.                      |
//|     Info        -   return code:                                 |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NOut-1].                |
//|                     * -1, if wrong parameters specified          |
//|                           (NPoints<0, Restarts<1, ...).          |
//|                     *  2, task has been solved, stopping         |
//|                           criterion met - sufficiently small     |
//|                           step size. Not expected (we use EARLY  |
//|                           stopping) but possible and not an error|
//|                     *  6, task has been solved, stopping         |
//|                           criterion  met - increasing of         |
//|                           validation set error.                  |
//|     Rep         -   training report                              |
//| NOTE:                                                            |
//| Algorithm stops if validation set error increases for a long     |
//| enough or step size is small enought (there are task where       |
//| validation set may decrease for eternity). In any case solution  |
//| returned corresponds to the minimum of validation set error.     |
//+------------------------------------------------------------------+
void CAlglib::MLPTrainES(CMultilayerPerceptronShell &network,
                         CMatrixDouble &trnxy,const int trnsize,
                         CMatrixDouble &valxy,const int valsize,
                         const double decay,const int restarts,
                         int &info,CMLPReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CMLPTrain::MLPTrainES(network.GetInnerObj(),trnxy,trnsize,valxy,valsize,decay,restarts,info,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Cross-validation estimate of generalization error.               |
//| Base algorithm - L-BFGS.                                         |
//| INPUT PARAMETERS:                                                |
//|     Network     -   neural network with initialized geometry.    |
//|                     Network is not changed during                |
//|                     cross-validation - it is used only as a      |
//|                     representative of its architecture.          |
//|     XY          -   training set.                                |
//|     SSize       -   training set size                            |
//|     Decay       -   weight  decay, same as in MLPTrainLBFGS      |
//|     Restarts    -   number of restarts, >0.                      |
//|                     restarts are counted for each partition      |
//|                     separately, so total number of restarts will |
//|                     be Restarts*FoldsCount.                      |
//|     WStep       -   stopping criterion, same as in MLPTrainLBFGS |
//|     MaxIts      -   stopping criterion, same as in MLPTrainLBFGS |
//|     FoldsCount  -   number of folds in k-fold cross-validation,  |
//|                     2<=FoldsCount<=SSize.                        |
//|                     recommended value: 10.                       |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code, same as in MLPTrainLBFGS        |
//|     Rep         -   report, same as in MLPTrainLM/MLPTrainLBFGS  |
//|     CVRep       -   generalization error estimates               |
//+------------------------------------------------------------------+
void CAlglib::MLPKFoldCVLBFGS(CMultilayerPerceptronShell &network,
                              CMatrixDouble &xy,const int npoints,
                              const double decay,const int restarts,
                              const double wstep,const int maxits,
                              const int foldscount,int &info,
                              CMLPReportShell &rep,CMLPCVReportShell &cvrep)
  {
//--- initialization
   info=0;
//--- function call
   CMLPTrain::MLPKFoldCVLBFGS(network.GetInnerObj(),xy,npoints,decay,restarts,wstep,maxits,foldscount,info,rep.GetInnerObj(),cvrep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Cross-validation estimate of generalization error.               |
//| Base algorithm - Levenberg-Marquardt.                            |
//| INPUT PARAMETERS:                                                |
//|     Network     -   neural network with initialized geometry.    |
//|                     Network is not changed during                |
//|                     cross-validation - it is used only as a      |
//|                     representative of its architecture.          |
//|     XY          -   training set.                                |
//|     SSize       -   training set size                            |
//|     Decay       -   weight  decay, same as in MLPTrainLBFGS      |
//|     Restarts    -   number of restarts, >0.                      |
//|                     restarts are counted for each partition      |
//|                     separately, so total number of restarts will |
//|                     be Restarts*FoldsCount.                      |
//|     FoldsCount  -   number of folds in k-fold cross-validation,  |
//|                     2<=FoldsCount<=SSize.                        |
//|                     recommended value: 10.                       |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code, same as in MLPTrainLBFGS        |
//|     Rep         -   report, same as in MLPTrainLM/MLPTrainLBFGS  |
//|     CVRep       -   generalization error estimates               |
//+------------------------------------------------------------------+
void CAlglib::MLPKFoldCVLM(CMultilayerPerceptronShell &network,
                           CMatrixDouble &xy,const int npoints,
                           const double decay,const int restarts,
                           const int foldscount,int &info,
                           CMLPReportShell &rep,CMLPCVReportShell &cvrep)
  {
//--- initialization
   info=0;
//--- function call
   CMLPTrain::MLPKFoldCVLM(network.GetInnerObj(),xy,npoints,decay,restarts,foldscount,info,rep.GetInnerObj(),cvrep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Creation of the network trainer object for regression networks   |
//| INPUT PARAMETERS:                                                |
//|   NIn         -  number of inputs, NIn>=1                        |
//|   NOut        -  number of outputs, NOut>=1                      |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  neural network trainer object.                  |
//| This structure can be used to train any regression network with  |
//| NIn inputs and NOut outputs.                                     |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateTrainer(int nin,int nout,CMLPTrainer &s)
  {
   CMLPTrain::MLPCreateTrainer(nin,nout,s);
  }
//+------------------------------------------------------------------+
//| Creation of the network trainer object for classification        |
//| networks                                                         |
//| INPUT PARAMETERS:                                                |
//|   NIn         -  number of inputs, NIn>=1                        |
//|   NClasses    -  number of classes, NClasses>=2                  |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  neural network trainer object.                  |
//| This structure can be used to train any classification network   |
//| with NIn inputs and NOut outputs.                                |
//+------------------------------------------------------------------+
void CAlglib::MLPCreateTrainerCls(int nin,int nclasses,CMLPTrainer &s)
  {
   CMLPTrain::MLPCreateTrainerCls(nin,nclasses,s);
  }
//+------------------------------------------------------------------+
//| This function sets "current dataset" of the trainer object to one|
//| passed by user.                                                  |
//| INPUT PARAMETERS:                                                |
//|   S           -  trainer object                                  |
//|   XY          -  training set, see below for information on the  |
//|                  training set format. This function checks       |
//|                  correctness of the dataset (no NANs/INFs, class |
//|                  numbers are correct) and throws exception when  |
//|                  incorrect dataset is passed.                    |
//|   NPoints     -  points count, >=0.                              |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//| For regression networks with NIn inputs and NOut outputs         |
//| following dataset format is used:                                |
//|      * dataset is given by NPoints*(NIn+NOut) matrix             |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, next NOut columns are       |
//|        outputs                                                   |
//| For classification networks with NIn inputs and NClasses clases  |
//| following datasetformat is used:                                 |
//|      * dataset is given by NPoints*(NIn+1) matrix                |
//|      * each row corresponds to one example                       |
//|      * first NIn columns are inputs, last column stores class    |
//|        number (from 0 to NClasses-1).                            |
//+------------------------------------------------------------------+
void CAlglib::MLPSetDataset(CMLPTrainer &s,CMatrixDouble &xy,int npoints)
  {
   CMLPTrain::MLPSetDataset(s,xy,npoints);
  }
//+------------------------------------------------------------------+
//| This function sets "current dataset" of the trainer object to one|
//| passed by user (sparse matrix is used to store dataset).         |
//| INPUT PARAMETERS:                                                |
//|   S        -  trainer object                                     |
//|   XY       -  training set, see below for information on the     |
//|               training set format. This function checks          |
//|               correctness of the dataset (no NANs/INFs, class    |
//|               numbers are correct) and throws exception when     |
//|               incorrect dataset is passed. Any sparse storage    |
//|               format can be used: Hash-table, CRS...             |
//|   NPoints  -  points count, >=0                                  |
//| DATASET FORMAT:                                                  |
//|   This function uses two different dataset formats - one for     |
//|   regression networks, another one for classification networks.  |
//| For regression networks with NIn inputs and NOut outputs         |
//| following dataset format is used:                                |
//|   * dataset is given by NPoints*(NIn+NOut) matrix                |
//|   * each row corresponds to one example                          |
//|   * first NIn columns are inputs, next NOut columns are outputs  |
//| For classification networks with NIn inputs and NClasses clases  |
//| following datasetformat is used:                                 |
//|   * dataset is given by NPoints*(NIn+1) matrix                   |
//|   * each row corresponds to one example                          |
//|   * first NIn columns are inputs, last column stores class number|
//|     (from 0 to NClasses-1).                                      |
//+------------------------------------------------------------------+
void CAlglib::MLPSetSparseDataset(CMLPTrainer &s,CSparseMatrix &xy,int npoints)
  {
   CMLPTrain::MLPSetSparseDataset(s,xy,npoints);
  }
//+------------------------------------------------------------------+
//| This function sets weight decay coefficient which is used for    |
//| training.                                                        |
//| INPUT PARAMETERS:                                                |
//|   S        -  trainer object                                     |
//|   Decay    -  weight decay coefficient, >=0. Weight decay term   |
//|               'Decay*||Weights||^2' is added to error function.  |
//|               If you don't know what Decay to choose, use 1.0E-3.|
//|               Weight decay can be set to zero, in this case      |
//|               network is trained without weight decay.           |
//| NOTE: by default network uses some small nonzero value for weight|
//| decay.                                                           |
//+------------------------------------------------------------------+
void CAlglib::MLPSetDecay(CMLPTrainer &s,double decay)
  {
   CMLPTrain::MLPSetDecay(s,decay);
  }
//+------------------------------------------------------------------+
//| This function sets stopping criteria for the optimizer.          |
//| INPUT PARAMETERS:                                                |
//|   S        -  trainer object                                     |
//|   WStep    -  stopping criterion. Algorithm stops if step size is|
//|               less than WStep. Recommended value - 0.01. Zero    |
//|               step size means stopping after MaxIts iterations.  |
//|               WStep>=0.                                          |
//|   MaxIts   -  stopping criterion. Algorithm stops after MaxIts   |
//|               epochs (full passes over entire dataset). Zero     |
//|               MaxIts means stopping when step is sufficiently    |
//|               small. MaxIts>=0.                                  |
//| NOTE: by default, WStep=0.005 and MaxIts=0 are used. These values|
//|       are also used when MLPSetCond() is called with WStep=0 and |
//|       MaxIts=0.                                                  |
//| NOTE: these stopping criteria are used for all kinds of neural   |
//|       training-from "conventional" networks to early stopping  |
//|       ensembles. When used for "conventional" networks, they are |
//|       used as the only stopping criteria. When combined with     |
//|       early stopping, they used as ADDITIONAL stopping criteria  |
//|       which can terminate early stopping algorithm.              |
//+------------------------------------------------------------------+
void CAlglib::MLPSetCond(CMLPTrainer &s,double wstep,int maxits)
  {
   CMLPTrain::MLPSetCond(s,wstep,maxits);
  }
//+------------------------------------------------------------------+
//| This function sets training algorithm: batch training using      |
//| L-BFGS will be used.                                             |
//| This algorithm:                                                  |
//|   * the most robust for small-scale problems, but may be too slow|
//|     for large scale ones.                                        |
//|   * perfoms full pass through the dataset before performing step |
//|   * uses conditions specified by MLPSetCond() for stopping       |
//|   * is default one used by trainer object                        |
//| INPUT PARAMETERS:                                                |
//|   S        -  trainer object                                     |
//+------------------------------------------------------------------+
void CAlglib::MLPSetAlgoBatch(CMLPTrainer &s)
  {
   CMLPTrain::MLPSetAlgoBatch(s);
  }
//+------------------------------------------------------------------+
//| This function trains neural network passed to this function,     |
//| using current dataset (one which was passed to MLPSetDataset()   |
//| or MLPSetSparseDataset()) and current training settings. Training|
//| from NRestarts random starting positions is performed, best      |
//| network is chosen.                                               |
//| Training is performed using current training algorithm.          |
//| INPUT PARAMETERS:                                                |
//|   S        -  trainer object                                     |
//|   Network  -  neural network. It must have same number of inputs |
//|               and output/classes as was specified during creation|
//|               of the trainer object.                             |
//|   NRestarts-  number of restarts, >=0:                           |
//|               * NRestarts>0 means that specified number of random|
//|                 restarts are performed, best network is chosen   |
//|                 after training                                   |
//|               * NRestarts=0 means that current state of the      |
//|                 network is used for training.                    |
//| OUTPUT PARAMETERS:                                               |
//|   Network  -  trained network                                    |
//| NOTE: when no dataset was specified with MLPSetDataset /         |
//|       SetSparseDataset(), network is filled by zero values. Same |
//|       behavior for functions MLPStartTraining and                |
//|       MLPContinueTraining.                                       |
//| NOTE: this method uses sum-of-squares error function for training|
//+------------------------------------------------------------------+
void CAlglib::MLPTrainNetwork(CMLPTrainer &s,
                              CMultilayerPerceptronShell &network,
                              int nrestarts,CMLPReportShell &rep)
  {
   CMLPTrain::MLPTrainNetwork(s,network.GetInnerObj(),nrestarts,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| IMPORTANT: this is an "expert" version of the MLPTrain() function|
//|            We do not recommend you to use it unless you are      |
//|            pretty sure that you need ability to monitor training |
//|            progress.                                             |
//| This function performs step-by-step training of the neural       |
//| network. Here "step-by-step" means that training starts with     |
//| MLPStartTraining() call, and then user subsequently calls        |
//| MLPContinueTraining() to perform one more iteration of the       |
//| training.                                                        |
//| After call to this function trainer object remembers network and |
//| is ready to train it. However, no training is performed until    |
//| first call to MLPContinueTraining() function. Subsequent calls   |
//| to MLPContinueTraining() will advance training progress one      |
//| iteration further.                                               |
//| EXAMPLE:                                                         |
//|   >                                                              |
//|   > ...initialize network and trainer object....                 |
//|   >                                                              |
//|   > MLPStartTraining(Trainer, Network, True)                     |
//|   > while MLPContinueTraining(Trainer, Network) do               |
//|   >     ...visualize training progress...                        |
//|   >                                                              |
//| INPUT PARAMETERS:                                                |
//|   S        -  trainer object                                     |
//|   Network  -  neural network. It must have same number of inputs |
//|               and output/classes as was specified during creation|
//|               of the trainer object.                             |
//|   RandomStart -  randomize network before training or not:       |
//|               * True means that network is randomized and its    |
//|                 initial state (one which was passed to the       |
//|                 trainer object) is lost.                         |
//|               * False means that training is started from the    |
//|                 current state of the network                     |
//| OUTPUT PARAMETERS:                                               |
//|   Network  -  neural network which is ready to training (weights |
//|               are initialized, preprocessor is initialized using |
//|               current training set)                              |
//| NOTE: this method uses sum-of-squares error function for training|
//| NOTE: it is expected that trainer object settings are NOT changed|
//|       during step-by-step training, i.e. no one changes stopping |
//|       criteria or training set during training. It is possible   |
//|       and there is no defense against such actions, but algorithm|
//|       behavior in such cases is undefined and can be             |
//|       unpredictable.                                             |
//+------------------------------------------------------------------+
void CAlglib::MLPStartTraining(CMLPTrainer &s,
                               CMultilayerPerceptronShell &network,
                               bool randomstart)
  {
   CMLPTrain::MLPStartTraining(s,network.GetInnerObj(),randomstart);
  }
//+------------------------------------------------------------------+
//| IMPORTANT: this is an "expert" version of the MLPTrain() function|
//|            We do not recommend you to use it unless you are      |
//|            pretty sure that you need ability to monitor training |
//|            progress.                                             |
//| This function performs step-by-step training of the neural       |
//| network. Here "step-by-step" means that training starts with     |
//| MLPStartTraining() call, and then user subsequently calls        |
//| MLPContinueTraining() to perform one more iteration of the       |
//| training.                                                        |
//| This function performs one more iteration of the training and    |
//| returns either True (training continues) or False (training      |
//| stopped). In case True was returned, Network weights are updated |
//| according to the  current  state of the optimization progress.   |
//| In case False was returned, no additional updates is performed   |
//| (previous update of the network weights moved us to the final    |
//| point, and no additional updates is needed).                     |
//| EXAMPLE:                                                         |
//|   >                                                              |
//|   > [initialize network and trainer object]                      |
//|   >                                                              |
//|   > MLPStartTraining(Trainer, Network, True)                     |
//|   > while MLPContinueTraining(Trainer, Network) do               |
//|   >     [visualize training progress]                            |
//|   >                                                              |
//| INPUT PARAMETERS:                                                |
//|   S        -  trainer object                                     |
//|   Network  -  neural network structure, which is used to store   |
//|               current state of the training process.             |
//| OUTPUT PARAMETERS:                                               |
//|   Network  -  weights of the neural network are rewritten by the |
//|               current approximation.                             |
//| NOTE: this method uses sum-of-squares error function for training|
//| NOTE: it is expected that trainer object settings are NOT changed|
//|       during step-by-step training, i.e. no one changes stopping |
//|       criteria or training set during training. It is possible   |
//|       and there is no defense against such actions, but algorithm|
//|       behavior in such cases is undefined and can be             |
//|       unpredictable.                                             |
//| NOTE: It is expected that Network is the same one which was      |
//|       passed to MLPStartTraining() function. However, THIS       |
//|       function checks only following:                            |
//|         * that number of network inputs is consistent with       |
//|           trainer object settings                                |
//|         * that number of network outputs / classes is consistent |
//|           with trainer object settings                           |
//|         * that number of network weights is the same as number of|
//|           weights in the network passed to MLPStartTraining()    |
//|           function Exception is thrown when these conditions are |
//|           violated.                                              |
//| It is also expected that you do not change state of the network  |
//| on your own - the only party who has right to change network     |
//| during its training is a trainer object. Any attempt to interfere|
//| with trainer may lead to unpredictable results.                  |
//+------------------------------------------------------------------+
bool CAlglib::MLPContinueTraining(CMLPTrainer &s,
                                  CMultilayerPerceptronShell &network)
  {
   return(CMLPTrain::MLPContinueTraining(s,network.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| This function trains neural network ensemble passed to this      |
//| function using current dataset and early stopping training       |
//| algorithm. Each early stopping round performs NRestarts random   |
//| restarts (thus, EnsembleSize*NRestarts training rounds is        |
//| performed in total).                                             |
//| INPUT PARAMETERS:                                                |
//|   S           -  trainer object;                                 |
//|   Ensemble    -  neural network ensemble. It must have same      |
//|                  number of inputs and outputs/classes as was     |
//|                  specified during creation of the trainer object.|
//|   NRestarts   -  number of restarts, >=0:                        |
//|                  * NRestarts>0 means that specified number of    |
//|                    random restarts are performed during each ES  |
//|                    round;                                        |
//|                  * NRestarts=0 is silently replaced by 1.        |
//| OUTPUT PARAMETERS:                                               |
//|   Ensemble    -  trained ensemble;                               |
//|   Rep         -  it contains all type of errors.                 |
//| NOTE: this training method uses BOTH early stopping and weight   |
//|       decay! So, you should select weight decay before starting  |
//|       training just as you select it before training             |
//|       "conventional" networks.                                   |
//| NOTE: when no dataset was specified with MLPSetDataset /         |
//|       SetSparseDataset(), or single-point dataset was passed,    |
//|       ensemble is filled by zero values.                         |
//| NOTE: this method uses sum-of-squares error function for training|
//+------------------------------------------------------------------+
void CAlglib::MLPTrainEnsembleES(CMLPTrainer &s,
                                 CMLPEnsembleShell &ensemble,
                                 int nrestarts,
                                 CMLPReportShell &rep)
  {
   CMLPTrain::MLPTrainEnsembleES(s,ensemble.GetInnerObj(),nrestarts,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreate0, but for ensembles.                              |
//+------------------------------------------------------------------+
void CAlglib::MLPECreate0(const int nin,const int nout,const int ensemblesize,
                          CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreate0(nin,nout,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreate1, but for ensembles.                              |
//+------------------------------------------------------------------+
void CAlglib::MLPECreate1(const int nin,int nhid,const int nout,
                          const int ensemblesize,
                          CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreate1(nin,nhid,nout,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreate2, but for ensembles.                              |
//+------------------------------------------------------------------+
void CAlglib::MLPECreate2(const int nin,const int nhid1,const int nhid2,
                          const int nout,const int ensemblesize,
                          CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreate2(nin,nhid1,nhid2,nout,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateB0, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateB0(const int nin,const int nout,const double b,
                           const double d,const int ensemblesize,
                           CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateB0(nin,nout,b,d,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateB1, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateB1(const int nin,int nhid,const int nout,
                           const double b,const double d,const int ensemblesize,
                           CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateB1(nin,nhid,nout,b,d,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateB2, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateB2(const int nin,const int nhid1,const int nhid2,
                           const int nout,const double b,const double d,
                           const int ensemblesize,CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateB2(nin,nhid1,nhid2,nout,b,d,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateR0, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateR0(const int nin,const int nout,const double a,
                           const double b,const int ensemblesize,
                           CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateR0(nin,nout,a,b,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateR1, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateR1(const int nin,int nhid,const int nout,
                           const double a,const double b,const int ensemblesize,
                           CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateR1(nin,nhid,nout,a,b,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateR2, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateR2(const int nin,const int nhid1,const int nhid2,
                           const int nout,const double a,const double b,
                           const int ensemblesize,CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateR2(nin,nhid1,nhid2,nout,a,b,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateC0, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateC0(const int nin,const int nout,
                           const int ensemblesize,CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateC0(nin,nout,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateC1, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateC1(const int nin,int nhid,const int nout,
                           const int ensemblesize,CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateC1(nin,nhid,nout,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Like MLPCreateC2, but for ensembles.                             |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateC2(const int nin,const int nhid1,const int nhid2,
                           const int nout,const int ensemblesize,
                           CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateC2(nin,nhid1,nhid2,nout,ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Creates ensemble from network. Only network geometry is copied.  |
//+------------------------------------------------------------------+
void CAlglib::MLPECreateFromNetwork(CMultilayerPerceptronShell &network,
                                    const int ensemblesize,
                                    CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPECreateFromNetwork(network.GetInnerObj(),ensemblesize,ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Randomization of MLP ensemble                                    |
//+------------------------------------------------------------------+
void CAlglib::MLPERandomize(CMLPEnsembleShell &ensemble)
  {
   CMLPE::MLPERandomize(ensemble.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Return ensemble properties (number of inputs and outputs).       |
//+------------------------------------------------------------------+
void CAlglib::MLPEProperties(CMLPEnsembleShell &ensemble,
                             int &nin,int &nout)
  {
//--- initialization
   nin=0;
   nout=0;
//--- function call
   CMLPE::MLPEProperties(ensemble.GetInnerObj(),nin,nout);
  }
//+------------------------------------------------------------------+
//| Return normalization type (whether ensemble is SOFTMAX-normalized|
//| or not).                                                         |
//+------------------------------------------------------------------+
bool CAlglib::MLPEIsSoftMax(CMLPEnsembleShell &ensemble)
  {
   return(CMLPE::MLPEIsSoftMax(ensemble.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| Procesing                                                        |
//| INPUT PARAMETERS:                                                |
//|     Ensemble-   neural networks ensemble                         |
//|     X       -   input vector,  array[0..NIn-1].                  |
//|     Y       -   (possibly) preallocated buffer; if size of Y is  |
//|                 less than NOut, it will be reallocated. If it is |
//|                 large enough, it is NOT reallocated, so we can   |
//|                 save some time on reallocation.                  |
//| OUTPUT PARAMETERS:                                               |
//|     Y       -   result. Regression estimate when solving         |
//|                 regression task, vector of posterior             |
//|                 probabilities for classification task.           |
//+------------------------------------------------------------------+
void CAlglib::MLPEProcess(CMLPEnsembleShell &ensemble,
                          double &x[],double &y[])
  {
   CMLPE::MLPEProcess(ensemble.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| 'interactive' variant of MLPEProcess for languages like Python   |
//| which support constructs like "Y = MLPEProcess(LM,X)" and        |
//| interactive mode of the interpreter                              |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart, but |
//| it is more convenient when you call it from command line.        |
//+------------------------------------------------------------------+
void CAlglib::MLPEProcessI(CMLPEnsembleShell &ensemble,
                           double &x[],double &y[])
  {
   CMLPE::MLPEProcessI(ensemble.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| Relative classification error on the test set                    |
//| INPUT PARAMETERS:                                                |
//|     Ensemble-   ensemble                                         |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     percent of incorrectly classified cases.                     |
//|     Works both for classifier betwork and for regression networks|
//|     which are used as classifiers.                               |
//+------------------------------------------------------------------+
double CAlglib::MLPERelClsError(CMLPEnsembleShell &ensemble,
                                CMatrixDouble &xy,const int npoints)
  {
   return(CMLPE::MLPERelClsError(ensemble.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average cross-entropy (in bits per element) on the test set      |
//| INPUT PARAMETERS:                                                |
//|     Ensemble-   ensemble                                         |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     CrossEntropy/(NPoints*LN(2)).                                |
//|     Zero if ensemble solves regression task.                     |
//+------------------------------------------------------------------+
double CAlglib::MLPEAvgCE(CMLPEnsembleShell &ensemble,
                          CMatrixDouble &xy,const int npoints)
  {
   return(CMLPE::MLPEAvgCE(ensemble.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| RMS error on the test set                                        |
//| INPUT PARAMETERS:                                                |
//|     Ensemble-   ensemble                                         |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     root mean square error.                                      |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task RMS error means error when estimating    |
//|     posterior probabilities.                                     |
//+------------------------------------------------------------------+
double CAlglib::MLPERMSError(CMLPEnsembleShell &ensemble,
                             CMatrixDouble &xy,const int npoints)
  {
   return(CMLPE::MLPERMSError(ensemble.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average error on the test set                                    |
//| INPUT PARAMETERS:                                                |
//|     Ensemble-   ensemble                                         |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task it means average error when estimating   |
//|     posterior probabilities.                                     |
//+------------------------------------------------------------------+
double CAlglib::MLPEAvgError(CMLPEnsembleShell &ensemble,
                             CMatrixDouble &xy,const int npoints)
  {
   return(CMLPE::MLPEAvgError(ensemble.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average relative error on the test set                           |
//| INPUT PARAMETERS:                                                |
//|     Ensemble-   ensemble                                         |
//|     XY      -   test set                                         |
//|     NPoints -   test set size                                    |
//| RESULT:                                                          |
//|     Its meaning for regression task is obvious. As for           |
//|     classification task it means average relative error when     |
//|     estimating posterior probabilities.                          |
//+------------------------------------------------------------------+
double CAlglib::MLPEAvgRelError(CMLPEnsembleShell &ensemble,
                                CMatrixDouble &xy,const int npoints)
  {
   return(CMLPE::MLPEAvgRelError(ensemble.GetInnerObj(),xy,npoints));
  }
//+------------------------------------------------------------------+
//| Training neural networks ensemble using  bootstrap  aggregating  |
//| (bagging).                                                       |
//| Modified Levenberg-Marquardt algorithm is used as base training  |
//| method.                                                          |
//| INPUT PARAMETERS:                                                |
//|     Ensemble    -   model with initialized geometry              |
//|     XY          -   training set                                 |
//|     NPoints     -   training set size                            |
//|     Decay       -   weight decay coefficient, >=0.001            |
//|     Restarts    -   restarts, >0.                                |
//| OUTPUT PARAMETERS:                                               |
//|     Ensemble    -   trained model                                |
//|     Info        -   return code:                                 |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NClasses-1].            |
//|                     * -1, if incorrect parameters was passed     |
//|                           (NPoints<0, Restarts<1).               |
//|                     *  2, if task has been solved.               |
//|     Rep         -   training report.                             |
//|     OOBErrors   -   out-of-bag generalization error estimate     |
//+------------------------------------------------------------------+
void CAlglib::MLPEBaggingLM(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,
                            const int npoints,const double decay,
                            const int restarts,int &info,
                            CMLPReportShell &rep,CMLPCVReportShell &ooberrors)
  {
//--- initialization
   info=0;
//--- function call
   CMLPTrain::MLPEBaggingLM(ensemble.GetInnerObj(),xy,npoints,decay,restarts,info,rep.GetInnerObj(),ooberrors.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Training neural networks ensemble using  bootstrap aggregating   |
//| (bagging). L-BFGS algorithm is used as base training method.     |
//| INPUT PARAMETERS:                                                |
//|     Ensemble    -   model with initialized geometry              |
//|     XY          -   training set                                 |
//|     NPoints     -   training set size                            |
//|     Decay       -   weight decay coefficient, >=0.001            |
//|     Restarts    -   restarts, >0.                                |
//|     WStep       -   stopping criterion, same as in MLPTrainLBFGS |
//|     MaxIts      -   stopping criterion, same as in MLPTrainLBFGS |
//| OUTPUT PARAMETERS:                                               |
//|     Ensemble    -   trained model                                |
//|     Info        -   return code:                                 |
//|                     * -8, if both WStep=0 and MaxIts=0           |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NClasses-1].            |
//|                     * -1, if incorrect parameters was passed     |
//|                           (NPoints<0, Restarts<1).               |
//|                     *  2, if task has been solved.               |
//|     Rep         -   training report.                             |
//|     OOBErrors   -   out-of-bag generalization error estimate     |
//+------------------------------------------------------------------+
void CAlglib::MLPEBaggingLBFGS(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,
                               const int npoints,const double decay,
                               const int restarts,const double wstep,
                               const int maxits,int &info,
                               CMLPReportShell &rep,
                               CMLPCVReportShell &ooberrors)
  {
//--- initialization
   info=0;
//--- function call
   CMLPTrain::MLPEBaggingLBFGS(ensemble.GetInnerObj(),xy,npoints,decay,restarts,wstep,maxits,info,rep.GetInnerObj(),ooberrors.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Training neural networks ensemble using early stopping.          |
//| INPUT PARAMETERS:                                                |
//|     Ensemble    -   model with initialized geometry              |
//|     XY          -   training set                                 |
//|     NPoints     -   training set size                            |
//|     Decay       -   weight decay coefficient, >=0.001            |
//|     Restarts    -   restarts, >0.                                |
//| OUTPUT PARAMETERS:                                               |
//|     Ensemble    -   trained model                                |
//|     Info        -   return code:                                 |
//|                     * -2, if there is a point with class number  |
//|                           outside of [0..NClasses-1].            |
//|                     * -1, if incorrect parameters was passed     |
//|                           (NPoints<0, Restarts<1).               |
//|                     *  6, if task has been solved.               |
//|     Rep         -   training report.                             |
//|     OOBErrors   -   out-of-bag generalization error estimate     |
//+------------------------------------------------------------------+
void CAlglib::MLPETrainES(CMLPEnsembleShell &ensemble,CMatrixDouble &xy,
                          const int npoints,const double decay,
                          const int restarts,int &info,
                          CMLPReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CMLPTrain::MLPETrainES(ensemble.GetInnerObj(),xy,npoints,decay,restarts,info,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Principal components analysis                                    |
//| Subroutine builds orthogonal basis where first axis corresponds  |
//| to direction with maximum variance, second axis maximizes        |
//| variance in subspace orthogonal to first axis and so on.         |
//| It should be noted that, unlike LDA, PCA does not use class      |
//| labels.                                                          |
//| INPUT PARAMETERS:                                                |
//|     X           -   dataset, array[0..NPoints-1,0..NVars-1].     |
//|                     matrix contains ONLY INDEPENDENT VARIABLES.  |
//|     NPoints     -   dataset size, NPoints>=0                     |
//|     NVars       -   number of independent variables, NVars>=1    |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   return code:                                 |
//|                     * -4, if SVD subroutine haven't converged    |
//|                     * -1, if wrong parameters has been passed    |
//|                           (NPoints<0, NVars<1)                   |
//|                     *  1, if task is solved                      |
//|     S2          -   array[0..NVars-1]. variance values           |
//|                     corresponding to basis vectors.              |
//|     V           -   array[0..NVars-1,0..NVars-1]                 |
//|                     matrix, whose columns store basis vectors.   |
//+------------------------------------------------------------------+
void CAlglib::PCABuildBasis(CMatrixDouble &x,const int npoints,
                            const int nvars,int &info,double &s2[],
                            CMatrixDouble &v)
  {
//--- initialization
   info=0;
//--- function call
   CPCAnalysis::PCABuildBasis(x,npoints,nvars,info,s2,v);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::PCABuildBasis(CMatrixDouble &x,const int npoints,
                            const int nvars,int &info,CRowDouble &s2,
                            CMatrixDouble &v)
  {
//--- initialization
   info=0;
//--- function call
   CPCAnalysis::PCABuildBasis(x,npoints,nvars,info,s2,v);
  }
//+------------------------------------------------------------------+
//| Principal components analysis                                    |
//| This function performs truncated PCA, i.e. returns just a few    |
//| most important directions.                                       |
//| Internally it uses iterative eigensolver which is very efficient |
//| when only a minor fraction of full basis is required. Thus, if   |
//| you need full basis, it is better to use pcabuildbasis() function|
//| It should be noted that, unlike LDA, PCA does not use class      |
//| labels.                                                          |
//| INPUT PARAMETERS:                                                |
//|   X        -  dataset, array[0..NPoints-1,0..NVars-1] matrix     |
//|               contains ONLY INDEPENDENT VARIABLES.               |
//|   NPoints  -  dataset size, NPoints>=0                           |
//|   NVars    -  number of independent variables, NVars>=1          |
//|   NNeeded  -  number of requested components, in [1,NVars] range;|
//|               this function is efficient only for NNeeded<<NVars.|
//|   Eps      -  desired precision of vectors returned; underlying  |
//|               solver will stop iterations as soon as absolute    |
//|               error in corresponding singular values reduces to  |
//|               roughly eps*MAX(lambda[]), with lambda[] being     |
//|               array of eigen values.                             |
//|               Zero value means that algorithm performs number of |
//|               iterations specified by maxits parameter, without  |
//|               paying attention to precision.                     |
//|   MaxIts   -  number of iterations performed by subspace         |
//|               iteration method. Zero value means that no limit on|
//|               iteration count is placed (eps-based stopping      |
//|               condition is used).                                |
//| OUTPUT PARAMETERS:                                               |
//|   S2       -  array[NNeeded]. Variance values corresponding to   |
//|               basis vectors.                                     |
//|   V        -  array[NVars,NNeeded] matrix, whose columns store   |
//|               basis vectors.                                     |
//| NOTE: passing eps=0 and maxits=0 results in small eps being      |
//|      selected as stopping condition. Exact value of automatically|
//|      selected eps is version-dependent.                          |
//+------------------------------------------------------------------+
void CAlglib::PCATruncatedSubspace(CMatrixDouble &x,int npoints,int nvars,
                                   int nneeded,double eps,int maxits,
                                   CRowDouble &s2,CMatrixDouble &v)
  {
   s2.Resize(0);
   v.Resize(0,0);
//--- function call
   CPCAnalysis::PCATruncatedSubSpace(x,npoints,nvars,nneeded,eps,maxits,s2,v);
  }
//+------------------------------------------------------------------+
//| Sparse truncated principal components analysis                   |
//| This function performs sparse truncated PCA, i.e. returns just a |
//| few most important principal components for a sparse input X.    |
//| Internally it uses iterative eigensolver which is very efficient |
//| when only a minor fraction of full basis is required.            |
//| It should be noted that, unlike LDA, PCA does not use class      |
//| labels.                                                          |
//| INPUT PARAMETERS:                                                |
//|   X        -  sparse dataset, sparse npoints*nvars matrix. It is |
//|               recommended to use CRS sparse storage format;      |
//|               non-CRS input will be internally converted to CRS. |
//|               Matrix contains ONLY INDEPENDENT VARIABLES,  and   |
//|               must be EXACTLY npoints*nvars.                     |
//|   NPoints  -  dataset size, NPoints>=0                           |
//|   NVars    -  number of independent variables, NVars>=1          |
//|   NNeeded  -  number of requested components, in [1,NVars] range;|
//|               this function is efficient only for NNeeded<<NVars.|
//|   Eps      -  desired precision of vectors returned; underlying  |
//|               solver will stop iterations as soon as absolute    |
//|               error in corresponding singular values reduces  to |
//|               roughly eps*MAX(lambda[]), with lambda[] being     |
//|               array of eigen values.                             |
//|               Zero value means that algorithm performs number of |
//|               iterations specified by maxits parameter, without  |
//|               paying attention to precision.                     |
//|   MaxIts   -  number of iterations performed by subspace         |
//|               iteration method. Zero value means that no limit on|
//|               iteration count is placed (eps-based stopping      |
//|               condition is used).                                |
//| OUTPUT PARAMETERS:                                               |
//|   S2       -  array[NNeeded]. Variance values corresponding to   |
//|               basis vectors.                                     |
//|   V        -  array[NVars,NNeeded] matrix, whose columns store   |
//|               basis vectors.                                     |
//| NOTE: passing eps=0 and maxits=0 results in small eps being      |
//|       selected as a stopping condition. Exact value of           |
//|       automatically selected eps is version-dependent.           |
//| NOTE: zero MaxIts is silently replaced by some reasonable value  |
//|       which prevents eternal loops (possible when inputs are     |
//|       degenerate and too stringent stopping criteria are         |
//|       specified). In current version it is 50+2*NVars.           |
//+------------------------------------------------------------------+
void CAlglib::PCATruncatedSubspaceSparse(CSparseMatrix &x,int npoints,
                                         int nvars,int nneeded,double eps,
                                         int maxits,CRowDouble &s2,
                                         CMatrixDouble &v)
  {
   s2.Resize(0);
   v.Resize(0,0);
//--- function call
   CPCAnalysis::PCATruncatedSubSpaceSparse(x,npoints,nvars,nneeded,eps,maxits,s2,v);
  }
//+------------------------------------------------------------------+
//| Cash-Karp adaptive ODE solver.                                   |
//| This subroutine solves ODE  Y'=f(Y,x) with initial conditions    |
//| Y(xs)=Ys (here Y may be single variable or vector of N variables)|
//| INPUT PARAMETERS:                                                |
//|     Y       -   initial conditions, array[0..N-1].               |
//|                 contains values of Y[] at X[0]                   |
//|     N       -   system size                                      |
//|     X       -   points at which Y should be tabulated,           |
//|                 array[0..M-1] integrations starts at X[0], ends  |
//|                 at X[M-1], intermediate values at X[i] are       |
//|                 returned too.                                    |
//|                 SHOULD BE ORDERED BY ASCENDING OR BY DESCENDING!!|
//|     M       -   number of intermediate points + first point +    |
//|                 last point:                                      |
//|                 * M>2 means that you need both Y(X[M-1]) and M-2 |
//|                   values at intermediate points                  |
//|                 * M=2 means that you want just to integrate from |
//|                   X[0] to X[1] and don't interested in           |
//|                   intermediate values.                           |
//|                 * M=1 means that you don't want to integrate :)  |
//|                   it is degenerate case, but it will be handled  |
//|                   correctly.                                     |
//|                 * M<1 means error                                |
//|     Eps     -   tolerance (absolute/relative error on each step  |
//|                 will be less than Eps). When passing:            |
//|                 * Eps>0, it means desired ABSOLUTE error         |
//|                 * Eps<0, it means desired RELATIVE error.        |
//|                   Relative errors are calculated with respect to |
//|                   maximum values of Y seen so far. Be careful to |
//|                   use this criterion when starting from Y[] that |
//|                   are close to zero.                             |
//|     H       -   initial step lenth, it will be adjusted          |
//|                 automatically after the first step. If H=0, step |
//|                 will be selected automatically (usualy it will   |
//|                 be equal to 0.001 of min(x[i]-x[j])).            |
//| OUTPUT PARAMETERS                                                |
//|     State   -   structure which stores algorithm state between   |
//|                 subsequent calls of OdeSolverIteration. Used     |
//|                 for reverse communication. This structure should |
//|                 be passed to the OdeSolverIteration subroutine.  |
//| SEE ALSO                                                         |
//|     AutoGKSmoothW, AutoGKSingular, AutoGKIteration, AutoGKResults|
//+------------------------------------------------------------------+
void CAlglib::ODESolverRKCK(double &y[],const int n,double &x[],
                            const int m,const double eps,const double h,
                            CODESolverStateShell &state)
  {
   CODESolver::ODESolverRKCK(y,n,x,m,eps,h,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Cash-Karp adaptive ODE solver.                                   |
//| This subroutine solves ODE  Y'=f(Y,x) with initial conditions    |
//| Y(xs)=Ys (here Y may be single variable or vector of N variables)|
//| INPUT PARAMETERS:                                                |
//|     Y       -   initial conditions, array[0..N-1].               |
//|                 contains values of Y[] at X[0]                   |
//|     N       -   system size                                      |
//|     X       -   points at which Y should be tabulated,           |
//|                 array[0..M-1] integrations starts at X[0], ends  |
//|                 at X[M-1], intermediate values at X[i] are       |
//|                 returned too.                                    |
//|                 SHOULD BE ORDERED BY ASCENDING OR BY DESCENDING!!|
//|     M       -   number of intermediate points + first point +    |
//|                 last point:                                      |
//|                 * M>2 means that you need both Y(X[M-1]) and M-2 |
//|                   values at intermediate points                  |
//|                 * M=2 means that you want just to integrate from |
//|                   X[0] to X[1] and don't interested in           |
//|                   intermediate values.                           |
//|                 * M=1 means that you don't want to integrate :)  |
//|                   it is degenerate case, but it will be handled  |
//|                   correctly.                                     |
//|                 * M<1 means error                                |
//|     Eps     -   tolerance (absolute/relative error on each step  |
//|                 will be less than Eps). When passing:            |
//|                 * Eps>0, it means desired ABSOLUTE error         |
//|                 * Eps<0, it means desired RELATIVE error.        |
//|                   Relative errors are calculated with respect to |
//|                   maximum values of Y seen so far. Be careful to |
//|                   use this criterion when starting from Y[] that |
//|                   are close to zero.                             |
//|     H       -   initial step lenth, it will be adjusted          |
//|                 automatically after the first step. If H=0, step |
//|                 will be selected automatically (usualy it will   |
//|                 be equal to 0.001 of min(x[i]-x[j])).            |
//| OUTPUT PARAMETERS                                                |
//|     State   -   structure which stores algorithm state between   |
//|                 subsequent calls of OdeSolverIteration. Used     |
//|                 for reverse communication. This structure should |
//|                 be passed to the OdeSolverIteration subroutine.  |
//| SEE ALSO                                                         |
//|     AutoGKSmoothW, AutoGKSingular, AutoGKIteration, AutoGKResults|
//+------------------------------------------------------------------+
void CAlglib::ODESolverRKCK(double &y[],double &x[],const double eps,
                            const double h,CODESolverStateShell &state)
  {
//--- initialization
   int n=CAp::Len(y);
   int m=CAp::Len(x);
//--- function call
   CODESolver::ODESolverRKCK(y,n,x,m,eps,h,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function provides reverse communication interface           |
//| Reverse communication interface is not documented or recommended |
//| to use.                                                          |
//| See below for functions which provide better documented API      |
//+------------------------------------------------------------------+
bool CAlglib::ODESolverIteration(CODESolverStateShell &state)
  {
   return(CODESolver::ODESolverIteration(state.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| This function is used to launcn iterations of ODE solver         |
//| It accepts following parameters:                                 |
//|     diff    -   callback which calculates dy/dx for given y and x|
//|     obj     -   optional object which is passed to diff; can be  |
//|                 NULL                                             |
//+------------------------------------------------------------------+
void CAlglib::ODESolverSolve(CODESolverStateShell &state,
                             CNDimensional_ODE_RP &diff,
                             CObject &obj)
  {
//--- cycle
   while(CAlglib::ODESolverIteration(state))
     {
      //--- check
      if(state.GetNeedDY())
        {
         diff.ODE_RP(state.GetInnerObj().m_y,state.GetInnerObj().m_x,state.GetInnerObj().m_dy,obj);
         //--- next iteration
         continue;
        }
      Print("ALGLIB: unexpected error in 'odesolversolve'");
      CAp::exception_happened=true;
      break;
     }
  }
//+------------------------------------------------------------------+
//| ODE solver results                                               |
//| Called after OdeSolverIteration returned False.                  |
//| INPUT PARAMETERS:                                                |
//|     State   -   algorithm state (used by OdeSolverIteration).    |
//| OUTPUT PARAMETERS:                                               |
//|     M       -   number of tabulated values, M>=1                 |
//|     XTbl    -   array[0..M-1], values of X                       |
//|     YTbl    -   array[0..M-1,0..N-1], values of Y in X[i]        |
//|     Rep     -   solver report:                                   |
//|                 * Rep.TerminationType completetion code:         |
//|                     * -2    X is not ordered  by                 |
//|                             ascending/descending or there are    |
//|                             non-distinct X[], i.e.  X[i]=X[i+1]  |
//|                     * -1    incorrect parameters were specified  |
//|                     *  1    task has been solved                 |
//|                 * Rep.NFEV contains number of function           |
//|                   calculations                                   |
//+------------------------------------------------------------------+
void CAlglib::ODESolverResults(CODESolverStateShell &state,int &m,
                               double &xtbl[],CMatrixDouble &ytbl,
                               CODESolverReportShell &rep)
  {
//--- initialization
   m=0;
//--- function call
   CODESolver::ODESolverResults(state.GetInnerObj(),m,xtbl,ytbl,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Filters: simple moving averages (unsymmetric).                   |
//| This filter replaces array by results of SMA(K) filter. SMA(K)   |
//| is defined as filter which averages at most K previous points    |
//| (previous - not points AROUND central point) - or less, in case  |
//| of the first K-1 points.                                         |
//| INPUT PARAMETERS:                                                |
//|   X           -  array[N], array to process. It can be larger    |
//|                  than N, in this case only first N points are    |
//|                  processed.                                      |
//|   N           -  points count, N>=0                              |
//|   K           -  K>=1 (K can be larger than N, such cases will   |
//|                  be correctly handled). Window width. K=1        |
//|                  corresponds to identity transformation (nothing |
//|                  changes).                                       |
//| OUTPUT PARAMETERS:                                               |
//|   X           -  array, whose first N elements were processed    |
//|                  with SMA(K)                                     |
//| NOTE 1: this function uses efficient in-place algorithm which    |
//|         does not allocate temporary arrays.                      |
//| NOTE 2: this algorithm makes only one pass through array and     |
//|         uses running sum to speed-up calculation of the averages.|
//|         Additional measures are taken to ensure that running sum |
//|         on a long sequence of zero elements will be correctly    |
//|         reset to zero even in the presence of round-off error.   |
//| NOTE 3: this is unsymmetric version of the algorithm, which does |
//|         NOT averages points after the current one. Only          |
//|         X[i], X[i-1], ... are used when calculating new value    |
//|         of X[i]. We should also note that this algorithm uses    |
//|         BOTH previous points and  current  one,  i.e. new value  |
//|         of X[i] depends on BOTH previous point and X[i] itself.  |
//+------------------------------------------------------------------+
void CAlglib::FilterSMA(CRowDouble &x,int n,int k)
  {
   CFilters::FilterSMA(x,n,k);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FilterSMA(CRowDouble &x,int k)
  {
   int n=CAp::Len(x);
   CFilters::FilterSMA(x,n,k);
  }
//+------------------------------------------------------------------+
//| Filters: exponential moving averages.                            |
//| This filter replaces array by results of EMA(alpha) filter.      |
//| EMA(alpha) is defined as filter which replaces X[] by S[]:       |
//|      S[0] = X[0]                                                 |
//|      S[t] = alpha * X[t] + (1 - alpha) * S[t - 1]                |
//| INPUT PARAMETERS:                                                |
//|   X           -  array[N], array to process. It can be larger    |
//|                  than N, in this case only first N points are    |
//|                  processed.                                      |
//|   N           -  points count, N >= 0                            |
//|   alpha       -  0 < alpha <= 1, smoothing parameter.            |
//| OUTPUT PARAMETERS:                                               |
//|   X           -  array, whose first N elements were processed    |
//|                  with EMA(alpha)                                 |
//| NOTE 1: this function uses efficient in-place algorithm which    |
//|         does not allocate temporary arrays.                      |
//| NOTE 2: this algorithm uses BOTH previous points and current one,|
//|         i.e. new value of X[i] depends on BOTH previous point and|
//|         X[i] itself.                                             |
//| NOTE 3: technical analytis users quite often work with  EMA      |
//|         coefficient expressed in DAYS instead of fractions. If   |
//|         you want to calculate EMA(N), where N is a number of     |
//|         days, you can use alpha = 2 / (N + 1).                   |
//+------------------------------------------------------------------+
void CAlglib::FilterEMA(CRowDouble &x,int n,double alpha)
  {
   CFilters::FilterEMA(x,n,alpha);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FilterEMA(CRowDouble &x,double alpha)
  {
   int n=CAp::Len(x);
   CFilters::FilterEMA(x,n,alpha);
  }
//+------------------------------------------------------------------+
//| Filters: linear regression moving averages.                      |
//| This filter replaces array by results of LRMA(K) filter.         |
//| LRMA(K) is defined as filter which, for each data point, builds  |
//| linear regression model using K prevous points (point itself is  |
//| included in these K points) and calculates value of this linear  |
//| model at the point in question.                                  |
//| INPUT PARAMETERS:                                                |
//|   X           -  array[N], array to process. It can be larger    |
//|                  than N, in this case only first N points are    |
//|                  processed.                                      |
//|   N           -  points count, N >= 0                            |
//|   K           -  K >= 1(K can be larger than N, such cases will  |
//|                  be correctly handled). Window width. K = 1      |
//|                  corresponds to identity transformation(nothing  |
//|                  changes).                                       |
//| OUTPUT PARAMETERS:                                               |
//|   X           -  array, whose first N elements were processed    |
//|                  with SMA(K)                                     |
//| NOTE 1: this function uses efficient in-place algorithm which    |
//|         does not allocate temporary arrays.                      |
//| NOTE 2: this algorithm makes only one pass through array and     |
//|         uses running sum to speed-up calculation of the averages.|
//|         Additional measures are taken to ensure that running sum |
//|         on a long sequence of zero elements will be correctly    |
//|         reset to zero even in the presence of round - off error. |
//| NOTE 3: this is unsymmetric version of the algorithm, which does |
//|         NOT averages points after the current one. Only          |
//|         X[i], X[i - 1], ... are used when calculating new value  |
//|         of X[i]. We should also note that this algorithm uses    |
//|         BOTH previous points and current one, i.e. new value of  |
//|         X[i] depends on BOTH previous point and X[i] itself.     |
//+------------------------------------------------------------------+
void CAlglib::FilterLRMA(CRowDouble &x,int n,int k)
  {
   CFilters::FilterLRMA(x,n,k);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FilterLRMA(CRowDouble &x,int k)
  {
   int n=CAp::Len(x);
//--- function call
   CFilters::FilterLRMA(x,n,k);
  }
//+------------------------------------------------------------------+
//| This function creates SSA model object. Right after creation     |
//| model is in "dummy" mode - you can add data, but analyzing /     |
//| prediction will return just zeros (it assumes that basis is      |
//| empty).                                                          |
//| HOW TO USE SSA MODEL:                                            |
//|   1. create model with SSACreate()                               |
//|   2. add data with one/many SSAAddSequence() calls               |
//|   3. choose SSA algorithm with one of SSASetAlgo...() functions: |
//|      * SSASetAlgoTopKDirect() for direct one-run analysis        |
//|      * SSASetAlgoTopKRealtime() for algorithm optimized for many |
//|         subsequent runs with warm-start capabilities             |
//|      * SSASetAlgoPrecomputed() for user-supplied basis           |
//|   4. set window width with SSASetWindow()                        |
//|   5. perform one of the analysis-related activities:             |
//|      a) call SSAGetBasis() to get basis                          |
//|      b) call SSAAnalyzeLast() SSAAnalyzeSequence() or            |
//|         SSAAnalyzeLastWindow() to perform analysis (trend/noise  |
//|         separation)                                              |
//|      c) call one of the forecasting functions (SSAForecastLast() |
//|         or SSAForecastSequence()) to perform prediction;         |
//|         alternatively, you can extract linear recurrence         |
//|         coefficients with SSAGetLRR().                           |
//| SSA analysis will be performed during first call to analysis -   |
//| related function. SSA model is smart enough to track all changes |
//| in the dataset and model settings, to cache previously computed  |
//| basis and to re-evaluate basis only when necessary.              |
//| Additionally, if your setting involves constant stream  of       |
//| incoming data, you can perform quick update already calculated   |
//| model with one of the incremental append-and-update  functions:  |
//| SSAAppendPointAndUpdate() or SSAAppendSequenceAndUpdate().       |
//| NOTE: steps (2), (3), (4) can be performed in arbitrary order.   |
//| INPUT PARAMETERS:                                                |
//|      none                                                        |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  structure which stores model state              |
//+------------------------------------------------------------------+
void CAlglib::SSACreate(CSSAModel &s)
  {
   CSSA::SSACreate(s);
  }
//+------------------------------------------------------------------+
//| This function sets window width for SSA model. You should call it|
//| before analysis phase. Default window width is 1 (not for real   |
//| use).                                                            |
//| Special notes:                                                   |
//|   * this function call can be performed at any moment before     |
//|     first call to analysis-related functions                     |
//|   * changing window width invalidates internally stored basis;   |
//|     if you change window width AFTER you call analysis-related   |
//|     function, next analysis phase will require re-calculation of |
//|     the basis according to current algorithm.                    |
//|   * calling this function with exactly same window width as      |
//|     current one has no effect                                    |
//|   * if you specify window width larger than any data sequence    |
//|     stored in the model, analysis will return zero basis.        |
//| INPUT PARAMETERS:                                                |
//|   S           -   SSA model created with SSACreate()             |
//|   WindowWidth -   >=1, new window width                          |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  SSA model, updated                              |
//+------------------------------------------------------------------+
void CAlglib::SSASetWindow(CSSAModel &s,int windowwidth)
  {
   CSSA::SSASetWindow(s,windowwidth);
  }
//+------------------------------------------------------------------+
//| This function sets seed which is used to initialize internal RNG |
//| when we make pseudorandom decisions on model updates.            |
//| By default, deterministic seed is used - which results in same   |
//| sequence of pseudorandom decisions every time you run SSA model. |
//| If you specify non-deterministic seed value, then SSA model may  |
//| return slightly different results after each run.                |
//| This function can be useful when you have several SSA models     |
//| updated with SSAAppendPointAndUpdate() called with 0<UpdateIts<1 |
//| (fractional value) and due to performance limitations want them  |
//| to perform updates at different moments.                         |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model                                       |
//|   Seed        -  seed:                                           |
//|                  * positive values = use deterministic seed for  |
//|                    each run of algorithms which depend on random |
//|                    initialization                                |
//|                  * zero or negative values=use non-deterministic |
//|                    seed                                          |
//+------------------------------------------------------------------+
void CAlglib::SSASetSeed(CSSAModel &s,int seed)
  {
   CSSA::SSASetSeed(s,seed);
  }
//+------------------------------------------------------------------+
//| This function sets length of power-up cycle for real-time        |
//| algorithm.                                                       |
//| By default, this algorithm performs costly O(N*WindowWidth^2)    |
//| init phase followed by full run of truncated EVD. However, if you|
//| are ready to live with a bit lower-quality basis during first few|
//| iterations, you can split this O(N*WindowWidth^2) initialization |
//| between several subsequent append-and-update rounds. It results  |
//| in better latency of the algorithm.                              |
//| This function invalidates basis/solver, next analysis call will  |
//| result in full recalculation of everything.                      |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model                                       |
//|   PWLen       -  length of the power-up stage:                   |
//|                  * 0 means that no power-up is requested         |
//|                  * 1 is the same as 0                            |
//|                  * >1 means that delayed power-up is performed   |
//+------------------------------------------------------------------+
void CAlglib::SSASetPowerUpLength(CSSAModel &s,int pwlen)
  {
   CSSA::SSASetPowerUpLength(s,pwlen);
  }
//+------------------------------------------------------------------+
//| This function sets memory limit of SSA analysis.                 |
//| Straightforward SSA with sequence length T and window width W    |
//| needs O(T*W) memory. It is possible to reduce memory consumption |
//| by splitting task into smaller chunks.                           |
//| Thus function allows you to specify approximate memory limit     |
//| (measured in double precision numbers used for buffers). Actual  |
//| memory consumption will be comparable to the number specified by |
//| you.                                                             |
//| Default memory limit is 50.000.000 (400Mbytes) in current        |
//| version.                                                         |
//| INPUT PARAMETERS:                                                |
//|   S        -  SSA model                                          |
//|   MemLimit -  memory limit, >=0. Zero value means no limit.      |
//+------------------------------------------------------------------+
void CAlglib::SSASetMemoryLimit(CSSAModel &s,int memlimit)
  {
   CSSA::SSASetMemoryLimit(s,memlimit);
  }
//+------------------------------------------------------------------+
//| This function adds data sequence to SSA model. Only single-      |
//| dimensional sequences are supported.                             |
//| What is a sequences? Following definitions/requirements apply:   |
//|   * a sequence is an array of values measured in subsequent,     |
//|     equally separated time moments (ticks).                      |
//|   * you may have many sequences in your dataset; say, one        |
//|     sequence may correspond to one trading session.              |
//|   * sequence length should be larger than current window length  |
//|     (shorter sequences will be ignored during analysis).         |
//|   * analysis is performed within a sequence; different sequences |
//|     are NOT stacked together to produce one large contiguous     |
//|     stream of data.                                              |
//|   * analysis is performed for all sequences at once, i.e. same   |
//|     set of basis vectors is computed for all sequences           |
//| INCREMENTAL ANALYSIS                                             |
//| This function is non intended for incremental updates of         |
//| previously found SSA basis. Calling it invalidates all previous  |
//| analysis results (basis is reset and will be recalculated from   |
//| zero during next analysis).                                      |
//| If you want to perform incremental/real-time SSA, consider using |
//| following functions:                                             |
//|   * SSAAppendPointAndUpdate() for appending one point            |
//|   * SSAAppendSequenceAndUpdate() for appending new sequence      |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model created with SSACreate()              |
//|   X           -  array[N], data, can be larger (additional values|
//|                  are ignored)                                    |
//|   N           -  data length, can be automatically determined    |
//|                  from the array length. N>=0.                    |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  SSA model, updated                              |
//| NOTE: you can clear dataset with SSAClearData()                  |
//+------------------------------------------------------------------+
void CAlglib::SSAAddSequence(CSSAModel &s,CRowDouble &x,int n)
  {
   CSSA::SSAAddSequence(s,x,n);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::SSAAddSequence(CSSAModel &s,CRowDouble &x)
  {
   int n=CAp::Len(x);
//--- function call
   CSSA::SSAAddSequence(s,x,n);
  }
//+------------------------------------------------------------------+
//| This function appends single point to last data sequence stored  |
//| in the SSA model and tries to update model in the incremental    |
//| manner (if possible with current algorithm).                     |
//| If you want to add more than one point at once:                  |
//|   * if you want to add M points to the same sequence, perform    |
//|     M-1 calls with UpdateIts parameter set to 0.0, and last call |
//|     with non-zero UpdateIts.                                     |
//|   * if you want to add new sequence, use                         |
//|     SSAAppendSequenceAndUpdate()                                 |
//| Running time of this function does NOT depend on dataset size,   |
//| only on window width and number of singular vectors. Depending   |
//| on algorithm being used, incremental update has complexity:      |
//|   * for top-K real time   -  O(UpdateIts*K*Width^2), with        |
//|                              fractional UpdateIts                |
//|   * for top-K direct      -  O(Width^3) for any non-zero         |
//|                              UpdateIts                           |
//|   * for precomputed basis -  O(1), no update is performed        |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model created with SSACreate()              |
//|   X           -  new point                                       |
//|   UpdateIts   -  >=0, floating point(!) value, desired update    |
//|                  frequency:                                      |
//|            * zero value means that point is stored, but no update|
//|              is performed                                        |
//|            * integer part of the value means that specified      |
//|              number of iterations is always performed            |
//|            * fractional part of the value means that one         |
//|              iteration is performed with this probability.       |
//| Recommended value: 0<UpdateIts<=1. Values larger than 1 are VERY |
//| seldom needed. If your dataset changes slowly, you can set it to |
//| 0.1 and skip 90% of updates.                                     |
//| In any case, no information is lost even with zero value of      |
//| UpdateIts! It will be incorporated into model, sooner or later.  |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  SSA model, updated                              |
//| NOTE: this function uses internal RNG to handle fractional values|
//|       of UpdateIts. By default it is initialized with fixed seed |
//|       during initial calculation of basis. Thus subsequent calls |
//|       to this function will result in the same sequence of       |
//|       pseudorandom decisions.                                    |
//| However, if you have several SSA models which are calculated     |
//| simultaneously, and if you want to reduce computational          |
//| bottlenecks by performing random updates at random moments, then |
//| fixed seed is not an option - all updates will fire at same      |
//| moments.                                                         |
//| You may change it with SSASetSeed() function.                    |
//| NOTE: this function throws an exception if called for empty      |
//|       dataset (there is no "last" sequence to modify).           |
//+------------------------------------------------------------------+
void CAlglib::SSAAppendPointAndUpdate(CSSAModel &s,double x,double updateits)
  {
   CSSA::SSAAppendPointAndUpdate(s,x,updateits);
  }
//+------------------------------------------------------------------+
//| This function appends new sequence to dataset stored in the SSA  |
//| model and tries to update model in the incremental manner (if    |
//| possible with current algorithm).                                |
//| Notes:                                                           |
//|   * if you want to add M sequences at once, perform M-1 calls    |
//|     with UpdateIts parameter set to 0.0, and last call with      |
//|     non-zero UpdateIts.                                          |
//|   * if you want to add just one point, use                       |
//|     SSAAppendPointAndUpdate()                                    |
//| Running time of this function does NOT depend on dataset size,   |
//| only on sequence length, window width and number of singular     |
//| vectors. Depending on algorithm being used, incremental update   |
//| has complexity:                                                  |
//|   * for top-K real time   -  O(UpdateIts*K*Width^2+              |
//|                                (NTicks-Width)*Width^2)           |
//|   * for top-K direct      -  O(Width^3+(NTicks-Width)*Width^2)   |
//|   * for precomputed basis -  O(1), no update is performed        |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model created with SSACreate()              |
//|   X           -  new sequence, array[NTicks] or larget           |
//|   NTicks      -  >=1, number of ticks in the sequence            |
//|   UpdateIts   -  >=0, floating point(!) value, desired update    |
//|                  frequency:                                      |
//|               * zero value means that point is stored, but no    |
//|                 update is performed                              |
//|               * integer part of the value means  that  specified |
//|                 number of iterations is always performed         |
//|               * fractional part of the value means that one      |
//|                 iteration is performed with this probability.    |
//| Recommended value: 0<UpdateIts<=1. Values larger than 1 are VERY |
//| seldom needed. If your dataset changes slowly, you can set it to |
//| 0.1 and skip 90% of updates.                                     |
//| In any case, no information is lost even with zero value of      |
//| UpdateIts! It will be incorporated into model, sooner or later.  |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  SSA model, updated                              |
//| NOTE: this function uses internal RNG to handle fractional values|
//|       of UpdateIts. By default it is initialized with fixed seed |
//|       during initial calculation of basis. Thus subsequent calls |
//|       to this function will result in the same sequence of       |
//|       pseudorandom decisions.                                    |
//| However, if you have several SSA models which are calculated     |
//| simultaneously, and if you want to reduce computational          |
//| bottlenecks by performing random updates at random moments, then |
//| fixed seed is not an option - all updates will fire at same      |
//| moments.                                                         |
//| You may change it with SSASetSeed() function.                    |
//+------------------------------------------------------------------+
void CAlglib::SSAAppendSequenceAndUpdate(CSSAModel &s,CRowDouble &x,
                                         int nticks,double updateits)
  {
   CSSA::SSAAppendSequenceAndUpdate(s,x,nticks,updateits);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::SSAAppendSequenceAndUpdate(CSSAModel &s,CRowDouble &x,
                                         double updateits)
  {
   int nticks=CAp::Len(x);
//--- function call
   CSSA::SSAAppendSequenceAndUpdate(s,x,nticks,updateits);
  }
//+------------------------------------------------------------------+
//| This function sets SSA algorithm to "precomputed vectors"        |
//| algorithm.                                                       |
//| This algorithm uses precomputed set of orthonormal(orthogonal    |
//| AND normalized) basis vectors supplied by user. Thus, basis      |
//| calculation phase is not performed - we already have our basis - |
//| and only analysis/forecasting phase requires actual calculations.|
//| This algorithm may handle "append" requests which add just one/  |
//| few ticks to the end of the last sequence in O(1) time.          |
//| NOTE: this algorithm accepts both basis and window width, because|
//|       these two parameters are naturally aligned. Calling this   |
//|       function sets window width; if you call SSASetWindow()     |
//|       with other window width, then during analysis stage        |
//|       algorithm will detect conflict and reset to zero basis.    |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model                                       |
//|   A           -  array[WindowWidth, NBasis], orthonormalized     |
//|                  basis; this function does NOT control           |
//|                  orthogonality and does NOT perform any kind of  |
//|                  renormalization. It is your responsibility to   |
//|                  provide it with correct basis.                  |
//|   WindowWidth -  window width, >= 1                              |
//|   NBasis      -  number of basis vectors,                        |
//|                  1 <= NBasis <= WindowWidth                      |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  updated model                                   |
//| NOTE: calling this function invalidates basis in all cases.      |
//+------------------------------------------------------------------+
void CAlglib::SSASetAlgoPrecomputed(CSSAModel &s,CMatrixDouble &a,int windowwidth,int nbasis)
  {
   CSSA::SSASetAlgoPrecomputed(s,a,windowwidth,nbasis);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::SSASetAlgoPrecomputed(CSSAModel &s,CMatrixDouble &a)
  {
   int windowwidth=CAp::Rows(a);
   int nbasis=CAp::Cols(a);
//--- function call
   CSSA::SSASetAlgoPrecomputed(s,a,windowwidth,nbasis);
  }
//+------------------------------------------------------------------+
//| This function sets SSA algorithm to "direct top-K" algorithm.    |
//| "Direct top-K" algorithm performs full SVD of the N*WINDOW       |
//| trajectory matrix (hence its name - direct solver is used), then |
//| extracts top K components. Overall running time is               |
//| O(N * WINDOW ^ 2), where N is a number of ticks in the dataset,  |
//| WINDOW is window width.                                          |
//| This algorithm may handle "append" requests which add just one / |
//| few ticks to the end of the last sequence in O(WINDOW ^ 3) time, |
//| which is ~N/WINDOW times faster than re-computing everything from|
//| scratch.                                                         |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model                                       |
//|   TopK        -  number of components to analyze; TopK >= 1.     |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  updated model                                   |
//| NOTE: TopK>WindowWidth is silently decreased to WindowWidth      |
//|       during analysis phase                                      |
//| NOTE: calling this function invalidates basis, except for the    |
//|       situation when this algorithm was already set with same    |
//|       parameters.                                                |
//+------------------------------------------------------------------+
void CAlglib::SSASetAlgoTopKDirect(CSSAModel &s,int topk)
  {
   CSSA::SSASetAlgoTopKDirect(s,topk);
  }
//+------------------------------------------------------------------+
//| This function sets SSA algorithm to "top-K real time algorithm". |
//| This algo extracts K components with largest singular values.    |
//| It is real-time version of top-K algorithm which is optimized for|
//| incremental processing and fast start-up. Internally it uses     |
//| subspace eigensolver for truncated SVD. It results in ability to |
//| perform quick updates of the basis when only a few points /      |
//| sequences is added to dataset.                                   |
//| Performance profile of the algorithm is given below:             |
//|   * O(K * WindowWidth ^ 2) running time for incremental update   |
//|     of the dataset with one of the "append-and-update" functions |
//|     (SSAAppendPointAndUpdate() or SSAAppendSequenceAndUpdate()). |
//|   * O(N * WindowWidth ^ 2) running time for initial basis        |
//|     evaluation(N = size  of dataset)                             |
//|   * ability to split costly initialization across several        |
//|     incremental updates of the basis(so called "Power-Up"        |
//|     functionality, activated by SSASetPowerUpLength() function)  |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model                                       |
//|   TopK        -  number of components to analyze; TopK >= 1.     |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  updated model                                   |
//| NOTE: this algorithm is optimized for large-scale tasks with     |
//|       large datasets. On toy problems with just  5-10 points it  |
//|       can return basis which is slightly different from that     |
//|       returned by direct algorithm (SSASetAlgoTopKDirect()       |
//|       function). However, the difference becomes negligible as   |
//|       dataset grows.                                             |
//| NOTE: TopK > WindowWidth is silently decreased to WindowWidth    |
//|       during analysis phase                                      |
//| NOTE: calling this function invalidates basis, except for the    |
//|       situation when this algorithm was already set with same    |
//|       parameters.                                                |
//+------------------------------------------------------------------+
void CAlglib::SSASetAlgoTopKRealtime(CSSAModel &s,int topk)
  {
   CSSA::SSASetAlgoTopKRealtime(s,topk);
  }

//+------------------------------------------------------------------+
//| This function clears all data stored in the model and invalidates|
//| all basis components found so far.                               |
//| INPUT PARAMETERS:                                                |
//|   S        -  SSA model created with SSACreate()                 |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  SSA model, updated                                 |
//+------------------------------------------------------------------+
void CAlglib::SSAClearData(CSSAModel &s)
  {
   CSSA::SSAClearData(s);
  }
//+------------------------------------------------------------------+
//| This function executes SSA on internally stored dataset and      |
//| returns basis found by current method.                           |
//| INPUT PARAMETERS:                                                |
//|   S        -  SSA model                                          |
//| OUTPUT PARAMETERS:                                               |
//|   A        -  array[WindowWidth, NBasis], basis; vectors are     |
//|               stored in matrix columns, by descreasing variance  |
//|   SV       -  array[NBasis]:                                     |
//|            * zeros - for model initialized with                  |
//|              SSASetAlgoPrecomputed()                             |
//|            * singular values - for other algorithms              |
//|   WindowWidth -  current window                                  |
//|   NBasis   -  basis size                                         |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|   * first call performs full run of SSA; basis is stored in the  |
//|     cache                                                        |
//|   * subsequent calls reuse previously cached basis               |
//|   * if you call any function which changes model properties      |
//|     (window length, algorithm, dataset), internal basis will be  |
//|     invalidated.                                                 |
//|   * the only calls which do NOT invalidate basis are listed      |
//|     below:                                                       |
//|      a) SSASetWindow() with same window length                   |
//|      b) SSAAppendPointAndUpdate()                                |
//|      c) SSAAppendSequenceAndUpdate()                             |
//|      d) SSASetAlgoTopK...() with exactly same K Calling these    |
//|         functions will result in reuse of previously found basis.|
//| HANDLING OF DEGENERATE CASES                                     |
//| Calling this function in degenerate cases(no data or all data are|
//| shorter than window size; no algorithm is specified) returns     |
//| basis with just one zero vector.                                 |
//+------------------------------------------------------------------+
void CAlglib::SSAGetBasis(CSSAModel &s,CMatrixDouble &a,CRowDouble &sv,int &windowwidth,int &nbasis)
  {
   a.Resize(0,0);
   sv.Resize(0);
   windowwidth=0;
   nbasis=0;
//--- function call
   CSSA::SSAGetBasis(s,a,sv,windowwidth,nbasis);
  }
//+------------------------------------------------------------------+
//| This function returns linear recurrence relation(LRR)            |
//| coefficients found by current SSA algorithm.                     |
//| INPUT PARAMETERS:                                                |
//|   S        -  SSA model                                          |
//| OUTPUT PARAMETERS:                                               |
//|   A        -  array[WindowWidth - 1]. Coefficients of the linear |
//|               recurrence of the form:                            |
//|               X[W - 1] = X[W - 2] * A[W - 2] +                   |
//|                          X[W - 3] * A[W - 3] + ... + X[0] * A[0].|
//|               Empty array for WindowWidth = 1.                   |
//|   WindowWidth -  current window width                            |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|   * first call performs full run of SSA; basis is stored in the  |
//|     cache                                                        |
//|   * subsequent calls reuse previously cached basis               |
//|   * if you call any function which changes model properties      |
//|     (window length, algorithm, dataset), internal basis will be  |
//|     invalidated.                                                 |
//|   * the only calls which do NOT invalidate basis are listed      |
//|     below:                                                       |
//|      a) SSASetWindow() with same window length                   |
//|      b) SSAAppendPointAndUpdate()                                |
//|      c) SSAAppendSequenceAndUpdate()                             |
//|      d) SSASetAlgoTopK...() with exactly same K                  |
//| Calling these functions will result in reuse of previously found |
//| basis.                                                           |
//| HANDLING OF DEGENERATE CASES                                     |
//| Calling this function in degenerate cases (no data or all data   |
//| are shorter than window size; no algorithm is specified) returns |
//| zeros.                                                           |
//+------------------------------------------------------------------+
void CAlglib::SSAGetLRR(CSSAModel &s,CRowDouble &a,int &windowwidth)
  {
   a.Resize(0);
   windowwidth=0;
//--- function call
   CSSA::SSAGetLRR(s,a,windowwidth);
  }
//+------------------------------------------------------------------+
//| This function executes SSA on internally stored dataset and      |
//| returns analysis for the last window of the last sequence. Such  |
//| analysis is an lightweight alternative for full scale            |
//| reconstruction (see below).                                      |
//| Typical use case for this function is real-time setting, when    |
//| you are interested in quick-and-dirty (very quick and very dirty)|
//| processing of just a few last ticks of the trend.                |
//| IMPORTANT: full scale SSA involves analysis of the ENTIRE        |
//|            dataset, with reconstruction being done for all       |
//|            positions of sliding window with subsequent           |
//|            hankelization (diagonal averaging) of the resulting   |
//|            matrix.                                               |
//| Such analysis requires O((DataLen - Window)*Window*NBasis) FLOPs |
//| and can be quite costly. However, it has nice noise - canceling  |
//| effects due to averaging.                                        |
//| This function performs REDUCED analysis of the last window. It   |
//| is much faster - just O(Window*NBasis), but its results are      |
//| DIFFERENT from that of SSAAnalyzeLast(). In particular, first few|
//| points of the trend are much more prone to noise.                |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model                                       |
//| OUTPUT PARAMETERS:                                               |
//|   Trend       -  array[WindowSize], reconstructed trend line     |
//|   Noise       -  array[WindowSize], the rest of the signal; it   |
//|                  holds that ActualData = Trend + Noise.          |
//|   NTicks      -  current WindowSize                              |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|   * first call performs full run of SSA; basis is stored in the  |
//|     cache                                                        |
//|   * subsequent calls reuse previously cached basis               |
//|   * if you call any function which changes model properties      |
//|     (window length, algorithm, dataset), internal basis will     |
//|     be invalidated.                                              |
//|   * the only calls which do NOT invalidate basis are listed      |
//|     below:                                                       |
//|         a) SSASetWindow() with same window length                |
//|         b) SSAAppendPointAndUpdate()                             |
//|         c) SSAAppendSequenceAndUpdate()                          |
//|         d) SSASetAlgoTopK...() with exactly same K               |
//| Calling these functions will result in reuse of previously found |
//| basis.                                                           |
//| In any case, only basis is reused. Reconstruction is performed   |
//| from scratch every time you call this function.                  |
//| HANDLING OF DEGENERATE CASES                                     |
//| Following degenerate cases may happen:                           |
//|   * dataset is empty(no analysis can be done)                    |
//|   * all sequences are shorter than the window length, no analysis|
//|     can be done                                                  |
//|   * no algorithm is specified(no analysis can be done)           |
//|   * last sequence is shorter than the window length (analysis    |
//|     can be done, but we can not perform reconstruction on the    |
//|     last sequence)                                               |
//| Calling this function in degenerate cases returns following      |
//| result:                                                          |
//|   * in any case, WindowWidth ticks is returned                   |
//|   * trend is assumed to be zero                                  |
//|   * noise is initialized by the last sequence; if last sequence  |
//|     is shorter than the window size, it is moved to the end of   |
//|     the array, and the beginning of the noise array is filled by |
//|     zeros                                                        |
//| No analysis is performed in degenerate cases (we immediately     |
//| return dummy values, no basis is constructed).                   |
//+------------------------------------------------------------------+
void CAlglib::SSAAnalyzeLastWindow(CSSAModel &s,CRowDouble &trend,CRowDouble &noise,int &nticks)
  {
   trend.Resize(0);
   noise.Resize(0);
   nticks=0;
//--- function call
   CSSA::SSAAnalyzeLastWindow(s,trend,noise,nticks);
  }
//+------------------------------------------------------------------+
//| This function:                                                   |
//|   * builds SSA basis using internally stored(entire) dataset     |
//|   * returns reconstruction for the last NTicks of the last       |
//|     sequence                                                     |
//| If you want to analyze some other sequence, use                  |
//| SSAAnalyzeSequence().                                            |
//| Reconstruction phase involves  generation of NTicks-WindowWidth  |
//| sliding windows, their decomposition using empirical orthogonal  |
//| functions found by SSA, followed by averaging of each data point |
//| across several overlapping windows. Thus, every point in the     |
//| output trend is reconstructed using up to WindowWidth overlapping|
//| windows(WindowWidth windows exactly in the inner points, just one|
//| window at the extremal points).                                  |
//| IMPORTANT: due to averaging this function returns different      |
//|            results for different values of NTicks. It is expected|
//|            and not a bug.                                        |
//| For example:                                                     |
//|   * Trend[NTicks - 1] is always same because it is not averaged  |
//|     in any case(same applies to Trend[0]).                       |
//|   * Trend[NTicks - 2] has different values fo NTicks=WindowWidth |
//|     and NTicks=WindowWidth+1 because former case means that no   |
//|     averaging is performed, and latter case means that averaging |
//|     using two sliding windows is performed. Larger values of     |
//|     NTicks produce same results as NTicks = WindowWidth + 1.     |
//|   * ...and so on...                                              |
//| PERFORMANCE: this function has                                   |
//|         O((NTicks - WindowWidth) * WindowWidth*NBasis)           |
//| running time. If you work in time-constrained setting and have   |
//| to analyze just a few last ticks, choosing NTicks equal to       |
//| WindowWidth + SmoothingLen, with SmoothingLen = 1...WindowWidth  |
//| will result in good compromise between noise cancellation and    |
//| analysis speed.                                                  |
//| INPUT PARAMETERS:                                                |
//|   S        -  SSA model                                          |
//|   NTicks   -  number of ticks to analyze, Nticks >= 1.           |
//|            * special case of NTicks<=WindowWidth  is  handled    |
//|              by analyzing last window and returning NTicks       |
//|              last ticks.                                         |
//|            * special case NTicks>LastSequenceLen is handled by   |
//|              prepending result with NTicks-LastSequenceLen zeros.|
//| OUTPUT PARAMETERS:                                               |
//|   Trend    -  array[NTicks], reconstructed trend line            |
//|   Noise    -  array[NTicks], the rest of the signal; it holds    |
//|               that ActualData = Trend + Noise.                   |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|      * first call performs full run of SSA; basis is stored in   |
//|        the cache                                                 |
//|      * subsequent calls reuse previously cached basis            |
//|      * if you call any function which changes model properties   |
//|        (window length, algorithm, dataset), internal basis will  |
//|        be invalidated.                                           |
//|      * the only calls which do NOT invalidate basis are listed   |
//|        below:                                                    |
//|            a) SSASetWindow() with same window length             |
//|            b) SSAAppendPointAndUpdate()                          |
//|            c) SSAAppendSequenceAndUpdate()                       |
//|            d) SSASetAlgoTopK...() with exactly same K            |
//| Calling these functions will result in reuse of previously found |
//| basis.                                                           |
//| In any case, only basis is reused. Reconstruction is performed   |
//| from scratch every time you call this function.                  |
//| HANDLING OF DEGENERATE CASES                                     |
//| Following degenerate cases may happen:                           |
//|      * dataset is empty(no analysis can be done)                 |
//|      * all sequences are shorter than the window length, no      |
//|        analysis can be done                                      |
//|      * no algorithm is specified(no analysis can be done)        |
//|      * last sequence is shorter than the window length(analysis  |
//|        can be done, but we can not perform reconstruction on the |
//|        last sequence)                                            |
//| Calling this function in degenerate cases returns following      |
//| result:                                                          |
//|      * in any case, NTicks ticks is returned                     |
//|      * trend is assumed to be zero                               |
//|      * noise is initialized by the last sequence; if last        |
//|        sequence is shorter than the window size, it is moved     |
//|        to the end of the array, and the beginning of the noise   |
//|        array is filled by zeros                                  |
//| No analysis is performed in degenerate cases(we immediately      |
//| return dummy values, no basis is constructed).                   |
//+------------------------------------------------------------------+
void CAlglib::SSAAnalyzeLast(CSSAModel &s,int nticks,CRowDouble &trend,CRowDouble &noise)
  {
   trend.Resize(0);
   noise.Resize(0);
//--- function call
   CSSA::SSAAnalyzeLast(s,nticks,trend,noise);
  }
//+------------------------------------------------------------------+
//| This function:                                                   |
//|      * builds SSA basis using internally stored(entire) dataset  |
//|      * returns reconstruction for the sequence being passed to   |
//|        this function                                             |
//| If you want to analyze last sequence stored in the model, use    |
//| SSAAnalyzeLast().                                                |
//| Reconstruction phase involves generation of NTicks-WindowWidth   |
//| sliding windows, their decomposition using empirical orthogonal  |
//| functions found by SSA, followed by averaging of each data point |
//| across several overlapping windows. Thus, every point in the     |
//| output trend is reconstructed using up to WindowWidth overlapping|
//| windows(WindowWidth windows exactly in the inner points, just one|
//| window at the extremal points).                                  |
//| PERFORMANCE: this function has                                   |
//|         O((NTicks - WindowWidth)*WindowWidth*NBasis)             |
//| running time. If you work in time-constrained setting and have   |
//| to analyze just a few last ticks, choosing NTicks equal to       |
//| WindowWidth + SmoothingLen, with SmoothingLen = 1...WindowWidth  |
//| will result in good compromise between noise cancellation and    |
//| analysis speed.                                                  |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model                                       |
//|   Data        -  array[NTicks], can be larger(only NTicks leading|
//|                  elements will be used)                          |
//|   NTicks      -  number of ticks to analyze, Nticks >= 1.        |
//|               * special case of NTicks<WindowWidth is handled by |
//|                 returning zeros as trend, and signal as noise    |
//| OUTPUT PARAMETERS:                                               |
//|   Trend       -  array[NTicks], reconstructed trend line         |
//|   Noise       -  array[NTicks], the rest of the signal; it holds |
//|                  that ActualData = Trend + Noise.                |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|      * first call performs full run of SSA; basis is stored in   |
//|        the cache                                                 |
//|      * subsequent calls reuse previously cached basis            |
//|      * if you call any function which changes model properties   |
//|        (window  length, algorithm, dataset), internal basis will |
//|        be invalidated.                                           |
//|      * the only calls which do NOT invalidate basis are listed   |
//|        below:                                                    |
//|         a) SSASetWindow() with same window length                |
//|         b) SSAAppendPointAndUpdate()                             |
//|         c) SSAAppendSequenceAndUpdate()                          |
//|         d) SSASetAlgoTopK...() with exactly same K               |
//| Calling these functions will result in reuse of previously found |
//| basis.                                                           |
//| In any case, only basis is reused. Reconstruction is performed   |
//| from scratch every time you call this function.                  |
//| HANDLING OF DEGENERATE CASES                                     |
//| Following degenerate cases may happen:                           |
//|      * dataset is empty(no analysis can be done)                 |
//|      * all sequences are shorter than the window length, no      |
//|        analysis can be done                                      |
//|      * no algorithm is specified(no analysis can be done)        |
//|      * sequence being passed is shorter than the window length   |
//| Calling this function in degenerate cases returns following      |
//| result:                                                          |
//|      * in any case, NTicks ticks is returned                     |
//|      * trend is assumed to be zero                               |
//|      * noise is initialized by the sequence.                     |
//| No analysis is performed in degenerate cases(we immediately      |
//| return dummy values, no basis is constructed).                   |
//+------------------------------------------------------------------+
void CAlglib::SSAAnalyzeSequence(CSSAModel &s,CRowDouble &data,int nticks,CRowDouble &trend,CRowDouble &noise)
  {
   trend.Resize(0);
   noise.Resize(0);
//--- function call
   CSSA::SSAAnalyzeSequence(s,data,nticks,trend,noise);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::SSAAnalyzeSequence(CSSAModel &s,CRowDouble &data,CRowDouble &trend,CRowDouble &noise)
  {
   trend.Resize(0);
   noise.Resize(0);
   int nticks=CAp::Len(data);
//--- function call
   CSSA::SSAAnalyzeSequence(s,data,nticks,trend,noise);
  }

//+------------------------------------------------------------------+
//| This function builds SSA basis and performs forecasting for a    |
//| specified number of ticks, returning value of trend.             |
//| Forecast is performed as follows:                                |
//|      * SSA trend extraction is applied to last WindowWidth       |
//|        elements of the internally stored dataset; this step is   |
//|        basically a noise reduction.                              |
//|      * linear recurrence relation is applied to extracted trend  |
//| This function has following running time:                        |
//|      * O(NBasis*WindowWidth) for trend extraction phase (always  |
//|                              performed)                          |
//|      * O(WindowWidth*NTicks) for forecast phase                  |
//| NOTE: noise reduction is ALWAYS applied by this algorithm; if you|
//|       want to apply recurrence relation to raw unprocessed data, |
//|       use another function - SSAForecastSequence() which allows  |
//|       to turn on and off noise reduction phase.                  |
//| NOTE: this algorithm performs prediction using only one-last -   |
//|       sliding window. Predictions produced by such approach are  |
//|       smooth continuations of the reconstructed trend line, but  |
//|       they can be easily corrupted by noise. If you need noise - |
//|       resistant prediction, use SSAForecastAvgLast() function,   |
//|       which averages predictions built using several sliding     |
//|       windows.                                                   |
//| INPUT PARAMETERS:                                                |
//|   S        -  SSA model                                          |
//|   NTicks   -  number of ticks to forecast, NTicks >= 1           |
//| OUTPUT PARAMETERS:                                               |
//|   Trend    -  array[NTicks], predicted trend line                |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|      * first call performs full run of SSA; basis is stored in   |
//|        the cache                                                 |
//|      * subsequent calls reuse previously cached basis            |
//|      * if you call any function which changes model properties   |
//|        (window length, algorithm, dataset), internal basis will  |
//|        be invalidated.                                           |
//|      * the only calls which do NOT invalidate basis are listed   |
//|        below:                                                    |
//|         a) SSASetWindow() with same window length                |
//|         b) SSAAppendPointAndUpdate()                             |
//|         c) SSAAppendSequenceAndUpdate()                          |
//|         d) SSASetAlgoTopK...() with exactly same K               |
//| Calling these functions will result in reuse of previously found |
//| basis.                                                           |
//| HANDLING OF DEGENERATE CASES                                     |
//| Following degenerate cases may happen:                           |
//|      * dataset is empty(no analysis can be done)                 |
//|      * all sequences are shorter than the window length, no      |
//|        analysis can be done                                      |
//|      * no algorithm is specified(no analysis can be done)        |
//|      * last sequence is shorter than the WindowWidth(analysis can|
//|         be done, but we can not perform forecasting on the last  |
//|         sequence)                                                |
//|      * window lentgh is 1(impossible to use for forecasting)     |
//|      * SSA analysis algorithm is configured to extract basis     |
//|        whose size is equal to window length(impossible to use for|
//|        forecasting; only basis whose size is less than window    |
//|        length can be used).                                      |
//| Calling this function in degenerate cases returns following      |
//| result:                                                          |
//|      * NTicks copies of the last value is returned for non-empty |
//|        task with large enough dataset, but with overcomplete     |
//|        basis (window width = 1 or basis size is equal to window  |
//|        width)                                                    |
//|      * zero trend with length = NTicks is returned for empty task|
//| No analysis is performed in degenerate cases (we immediately     |
//| return dummy values, no basis is ever constructed).              |
//+------------------------------------------------------------------+
void CAlglib::SSAForecastLast(CSSAModel &s,int nticks,CRowDouble &trend)
  {
   trend.Resize(0);
//--- function call
   CSSA::SSAForecastLast(s,nticks,trend);
  }
//+------------------------------------------------------------------+
//| This function builds SSA basis and performs forecasting for  a   |
//| user - specified sequence, returning value of trend.             |
//| Forecasting is done in two stages:                               |
//|      * first,  we  extract  trend from the WindowWidth last      |
//|        elements of the sequence. This stage is optional, you can |
//|        turn it off if you pass data which are already processed  |
//|        with SSA. Of course, you can turn it off even for raw     |
//|        data, but it is not recommended - noise suppression is    |
//|        very important for correct prediction.                    |
//|      * then, we apply LRR for last WindowWidth - 1 elements of   |
//|        the extracted trend.                                      |
//| This function has following running time:                        |
//|      * O(NBasis*WindowWidth)    for trend extraction phase       |
//|      * O(WindowWidth*NTicks)    for forecast phase               |
//| NOTE: this algorithm performs prediction using only one-last -   |
//|       sliding window. Predictions produced by such approach are  |
//|       smooth continuations of the reconstructed trend line, but  |
//|       they can be easily corrupted by noise. If you need noise - |
//|       resistant prediction, use SSAForecastAvgSequence()         |
//|       function, which averages predictions built using several   |
//|       sliding windows.                                           |
//| INPUT PARAMETERS:                                                |
//|   S        -  SSA model                                          |
//|   Data     -  array[NTicks], data to forecast                    |
//|   DataLen  -  number of ticks in the data, DataLen >= 1          |
//|   ForecastLen -  number of ticks to predict, ForecastLen >= 1    |
//|   ApplySmoothing - whether to apply smoothing trend extraction or|
//|               not; if you do not know what to specify, pass True.|
//| OUTPUT PARAMETERS:                                               |
//|   Trend    -  array[ForecastLen], forecasted trend               |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|      * first call performs full run of SSA; basis is stored in   |
//|        the cache                                                 |
//|      * subsequent calls reuse previously cached basis            |
//|      * if you call any function which changes model properties   |
//|        (window length, algorithm, dataset), internal basis will  |
//|        be invalidated.                                           |
//|      * the only calls which do NOT invalidate basis are listed   |
//|        below:                                                    |
//|         a) SSASetWindow() with same window length                |
//|         b) SSAAppendPointAndUpdate()                             |
//|         c) SSAAppendSequenceAndUpdate()                          |
//|         d) SSASetAlgoTopK...() with exactly same K               |
//| Calling these functions will result in reuse of previously found |
//| basis.                                                           |
//| HANDLING OF DEGENERATE CASES                                     |
//| Following degenerate cases may happen:                           |
//|      * dataset is empty(no analysis can be done)                 |
//|      * all sequences are shorter than the window length, no      |
//|        analysis can be done                                      |
//|      * no algorithm is specified(no analysis can be done)        |
//|      * data sequence is shorter than the WindowWidth(analysis can|
//|        be done, but we can not perform forecasting on the last   |
//|        sequence)                                                 |
//|      * window lentgh is 1(impossible to use for forecasting)     |
//|      * SSA analysis algorithm is configured to extract basis     |
//|        whose size is equal to window length (impossible to use   |
//|        for forecasting; only basis whose size is less than window|
//|        length can be used).                                      |
//| Calling this function in degenerate cases returns following      |
//| result:                                                          |
//|      * ForecastLen copies of the last value is returned for      |
//|        non-empty task with large enough dataset, but with        |
//|        overcomplete basis (window width = 1 or basis size is     |
//|        equal to window width)                                    |
//|      * zero trend with length = ForecastLen is returned for empty|
//|        task                                                      |
//| No analysis is performed in degenerate cases (we immediately     |
//| return dummy values, no basis is ever constructed).              |
//+------------------------------------------------------------------+
void CAlglib::SSAForecastSequence(CSSAModel &s,CRowDouble &data,
                                  int datalen,int forecastlen,
                                  bool applysmoothing,CRowDouble &trend)
  {
   trend.Resize(0);
//--- function call
   CSSA::SSAForecastSequence(s,data,datalen,forecastlen,applysmoothing,trend);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::SSAForecastSequence(CSSAModel &s,CRowDouble &data,int forecastlen,CRowDouble &trend)
  {
   trend.Resize(0);
   int datalen=CAp::Len(data);
   bool applysmoothing=true;
//--- function call
   CSSA::SSAForecastSequence(s,data,datalen,forecastlen,applysmoothing,trend);
  }
//+------------------------------------------------------------------+
//| This function builds SSA basis and performs forecasting for a    |
//| specified number of ticks, returning value of trend.             |
//| Forecast is performed as follows:                                |
//|      * SSA  trend extraction is applied to last M sliding windows|
//|        of the internally stored dataset                          |
//|      * for each of M sliding windows, M predictions are built    |
//|      * average value of M predictions is returned                |
//| This function has following running time:                        |
//|      * O(NBasis*WindowWidth*M)  for trend extraction phase       |
//|                                 (always performed)               |
//|      * O(WindowWidth*NTicks*M)  for forecast phase               |
//| NOTE: noise reduction is ALWAYS applied by this algorithm; if you|
//|       want to apply recurrence relation to raw unprocessed data, |
//|       use another function - SSAForecastSequence() which allows  |
//|       to turn on and off noise reduction phase.                  |
//| NOTE: combination of several predictions results in lesser       |
//|       sensitivity to noise, but it may produce undesirable       |
//|       discontinuities between last point of the trend and first  |
//|       point of the prediction. The reason is that last point of  |
//|       the trend is usually corrupted by noise, but average value |
//|       of several predictions is less sensitive to noise, thus    |
//|       discontinuity appears. It is not a bug.                    |
//| INPUT PARAMETERS:                                                |
//|   S        -  SSA model                                          |
//|   M        -  number of sliding windows to combine, M >= 1. If   |
//|               your dataset has less than M sliding windows, this |
//|               parameter will be silently reduced.                |
//|   NTicks   -  number of ticks to forecast, NTicks >= 1           |
//| OUTPUT PARAMETERS:                                               |
//|   Trend    -  array[NTicks], predicted trend line                |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|      * first call performs full run of SSA; basis is stored in   |
//|        the cache                                                 |
//|      * subsequent calls reuse previously cached basis            |
//|      * if you call any function which changes model properties   |
//|        (window length, algorithm, dataset), internal basis will  |
//|        be invalidated.                                           |
//|      * the only calls which do NOT invalidate basis are listed   |
//|        below:                                                    |
//|            a) SSASetWindow() with same window length             |
//|            b) SSAAppendPointAndUpdate()                          |
//|            c) SSAAppendSequenceAndUpdate()                       |
//|            d) SSASetAlgoTopK...() with exactly same K            |
//| Calling these functions will result in reuse of previously found |
//| basis.                                                           |
//| HANDLING OF DEGENERATE CASES                                     |
//| Following degenerate cases may happen:                           |
//|      * dataset is empty(no analysis can be done)                 |
//|      * all sequences are shorter than the window length, no      |
//|        analysis can be done                                      |
//|      * no algorithm is specified(no analysis can be done)        |
//|      * last sequence is shorter than the WindowWidth(analysis can|
//|        be done, but we can not perform forecasting on the last   |
//|        sequence)                                                 |
//|      * window lentgh is 1(impossible to use for forecasting)     |
//|      * SSA analysis algorithm is configured to extract basis     |
//|        whose size is equal to window length(impossible to use for|
//|        forecasting; only basis whose size is less than window    |
//|        length can be used).                                      |
//| Calling this function in degenerate cases returns following      |
//| result:                                                          |
//|   * NTicks copies of the last value is returned for non-empty    |
//|     task with large enough dataset, but with overcomplete basis  |
//|     (window width = 1 or basis size is equal to window width)    |
//|   * zero trend with length = NTicks is returned for empty task   |
//| No analysis is performed in degenerate cases (we immediately     |
//| return dummy values, no basis is ever constructed).              |
//+------------------------------------------------------------------+
void CAlglib::SSAForecastAvgLast(CSSAModel &s,int m,int nticks,CRowDouble &trend)
  {
   trend.Resize(0);
//--- function call
   CSSA::SSAForecastAvgLast(s,m,nticks,trend);
  }
//+------------------------------------------------------------------+
//| This function builds SSA basis and performs forecasting for a    |
//| user - specified sequence, returning value of trend.             |
//| Forecasting is done in two stages:                               |
//|   * first, we extract trend from M last sliding windows of the   |
//|     sequence. This stage is optional, you can turn it off i you  |
//|     pass data which are already processed with SSA. Of course,   |
//|     you can turn it off even for raw data, but it is not         |
//|     recommended - noise suppression is very important for correct|
//|     prediction.                                                  |
//|   * then, we apply LRR independently for M sliding windows       |
//|   * average of M predictions is returned                         |
//| This function has following running time:                        |
//|   * O(NBasis*WindowWidth*M)     for trend extraction phase       |
//|   * O(WindowWidth*NTicks*M)     for forecast phase               |
//| NOTE: combination of several predictions results in lesser       |
//|       sensitivity to noise, but it may produce undesirable       |
//|       discontinuities between last point of the trend and first  |
//|       point of the prediction. The reason is that last point of  |
//|       the trend is usually corrupted by noise, but average value |
//|       of several predictions is less sensitive to noise, thus    |
//|       discontinuity appears. It is not a bug.                    |
//| INPUT PARAMETERS:                                                |
//|   S           -  SSA model                                       |
//|   Data        -  array[NTicks], data to forecast                 |
//|   DataLen     -  number of ticks in the data, DataLen >= 1       |
//|   M           -  number of sliding windows to combine, M >= 1.   |
//|                  If your dataset has less than M sliding windows,|
//|                  this parameter will be silently reduced.        |
//|   ForecastLen -  number of ticks to predict, ForecastLen >= 1    |
//|   ApplySmoothing - whether to apply smoothing trend extraction   |
//|                  or not. if you do not know what to specify,     |
//|                  pass true.                                      |
//| OUTPUT PARAMETERS:                                               |
//|   Trend       -  array[ForecastLen], forecasted trend            |
//| CACHING / REUSE OF THE BASIS                                     |
//| Caching / reuse of previous results is performed:                |
//|   * first call performs full run of SSA; basis is stored in      |
//|     the cache                                                    |
//|   * subsequent calls reuse previously cached basis               |
//|   * if you call any function which changes model properties      |
//|     (window length, algorithm, dataset), internal basis will be  |
//|     invalidated.                                                 |
//|   * the only calls which do NOT invalidate basis are listed      |
//|     below:                                                       |
//|      a) SSASetWindow() with same window length                   |
//|      b) SSAAppendPointAndUpdate()                                |
//|      c) SSAAppendSequenceAndUpdate()                             |
//|      d) SSASetAlgoTopK...() with exactly same K                  |
//| Calling these functions will result in reuse of previously found |
//| basis.                                                           |
//| HANDLING OF DEGENERATE CASES                                     |
//| Following degenerate cases may happen:                           |
//|   * dataset is empty(no analysis can be done)                    |
//|   * all sequences are shorter than the window length, no analysis|
//|     can be done                                                  |
//|   * no algorithm is specified(no analysis can be done)           |
//|   * data sequence is shorter than the WindowWidth (analysis can  |
//|     be done, but we can not perform forecasting on the last      |
//|     sequence)                                                    |
//|   * window lentgh is 1 (impossible to use for forecasting)       |
//|   * SSA analysis algorithm is configured to extract basis whose  |
//|     size is equal to window length (impossible to use for        |
//|     forecasting; only basis whose size is less than window length|
//|     can be used).                                                |
//| Calling this function in degenerate cases returns following      |
//| result:                                                          |
//|   * ForecastLen copies of the last value is returned for         |
//|     non-empty task with large enough dataset, but with           |
//|     overcomplete basis (window width = 1 or basis size           |
//|     is equal to window width)                                    |
//|   * zero trend with length = ForecastLen is returned for         |
//|     empty task                                                   |
//| No analysis is performed in degenerate case s(we immediately     |
//| return dummy values, no basis is ever constructed).              |
//+------------------------------------------------------------------+
void CAlglib::SSAForecastAvgSequence(CSSAModel &s,CRowDouble &data,
                                     int datalen,int m,int forecastlen,
                                     bool applysmoothing,CRowDouble &trend)
  {
   trend.Resize(0);
//--- function call
   CSSA::SSAForecastAvgSequence(s,data,datalen,m,forecastlen,applysmoothing,trend);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::SSAForecastAvgSequence(CSSAModel &s,CRowDouble &data,
                                     int m,int forecastlen,
                                     CRowDouble &trend)
  {
   trend.Resize(0);
   int datalen=CAp::Len(data);
   bool applysmoothing=true;
//--- function call
   CSSA::SSAForecastAvgSequence(s,data,datalen,m,forecastlen,applysmoothing,trend);
  }
//+------------------------------------------------------------------+
//| This function serializes data structure to string.               |
//| Important properties of s_out:                                   |
//|   * it contains alphanumeric characters, dots, underscores, minus|
//|     signs                                                        |
//|   * these symbols are grouped into words, which are separated by |
//|     spaces and Windows-style (CR+LF) newlines                    |
//|   * although  serializer  uses  spaces and CR+LF as separators,  |
//|     you can replace any separator character by arbitrary         |
//|     combination of spaces, tabs, Windows or Unix newlines. It    |
//|     allows flexible reformatting of the string in case you want  |
//|     to include it into text or XML file. But you should not      |
//|     insert separators into the middle of the "words" nor you     |
//|     should change case of letters.                               |
//|   * s_out can be freely moved between 32-bit and 64-bit systems, |
//|     little and big endian machines, and so on. You can serialize |
//|     structure on 32-bit machine and unserialize it on 64-bit one |
//|     (or vice versa), or serialize it on SPARC and unserialize on |
//|     x86. You can also serialize  it  in  C# version of ALGLIB and|
//|     unserialize in C++ one, and vice versa.                      |
//+------------------------------------------------------------------+
void CAlglib::KNNSerialize(CKNNModel &obj,string &s_out)
  {
//--- create a variable
   CSerializer s;
//--- serialization start
   s.Alloc_Start();
//--- function call
   CKNN::KNNAlloc(s,obj);
//--- serialization
   s.SStart_Str();
   CKNN::KNNSerialize(s,obj);
   s.Stop();
   s_out=s.Get_String();
  }
//+------------------------------------------------------------------+
//| This function unserializes data structure from string.           |
//+------------------------------------------------------------------+
void CAlglib::KNNUnserialize(const string s_in,CKNNModel &obj)
  {
   CSerializer s;
   s.UStart_Str(s_in);
   CKNN::KNNUnserialize(s,obj);
   s.Stop();
  }
//+------------------------------------------------------------------+
//| This function creates buffer structure which can be used to      |
//| perform parallel KNN requests.                                   |
//| KNN subpackage provides two sets of computing functions - ones   |
//| which use internal buffer of KNN model (these  functions are     |
//| single-threaded because they use same buffer, which can not      |
//| shared between threads), and ones which use external buffer.     |
//| This function is used to initialize external buffer.             |
//| INPUT PARAMETERS                                                 |
//|   Model       -  KNN model which is associated with newly created|
//|                  buffer                                          |
//| OUTPUT PARAMETERS                                                |
//|   Buf         -  external buffer.                                |
//| IMPORTANT: buffer object should be used only with model which was|
//|            used to initialize buffer. Any attempt to  use  buffer|
//|            with different object is dangerous - you  may   get   |
//|            integrity check failure (exception) because sizes of  |
//|            internal arrays do not fit to dimensions of the model |
//|            structure.                                            |
//+------------------------------------------------------------------+
void CAlglib::KNNCreateBuffer(CKNNModel &model,CKNNBuffer &buf)
  {
   CKNN::KNNCreateBuffer(model,buf);
  }
//+------------------------------------------------------------------+
//| This subroutine creates KNNBuilder object which is used to train |
//| KNN models.                                                      |
//| By default, new builder stores empty dataset and some reasonable |
//| default settings. At the very least, you should specify dataset  |
//| prior to building KNN model. You can also tweak settings of the  |
//| model construction algorithm (recommended, although default      |
//| settings should work well).                                      |
//| Following actions are mandatory:                                 |
//|   * calling knnbuildersetdataset() to specify dataset            |
//|   * calling KNNBuilderBuildKNNModel() to build KNN model using   |
//|     current dataset and default settings                         |
//| Additionally, you may call:                                      |
//|   * KNNBuilderSetNorm() to change norm being used                |
//| INPUT PARAMETERS:                                                |
//|   none                                                           |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  KNN builder                                     |
//+------------------------------------------------------------------+
void CAlglib::KNNBuilderCreate(CKNNBuilder &s)
  {
   CKNN::KNNBuilderCreate(s);
  }
//+------------------------------------------------------------------+
//| Specifies regression problem (one or more continuous output      |
//| variables are predicted). There also exists "classification"     |
//| version of this function.                                        |
//| This subroutine adds dense dataset to the internal storage of the|
//| builder object. Specifying your dataset in the dense format means|
//| that the dense version of the KNN construction algorithm will be |
//| invoked.                                                         |
//| INPUT PARAMETERS:                                                |
//|   S           -  KNN builder object                              |
//|   XY          -  array[NPoints,NVars+NOut] (note: actual size can|
//|                  be larger, only leading part is used anyway),   |
//|                  dataset:                                        |
//|                  * first NVars elements of each row store values |
//|                    of the independent variables                  |
//|                  * next NOut elements store  values  of  the     |
//|                    dependent variables                           |
//|   NPoints     -  number of rows in the dataset, NPoints>=1       |
//|   NVars       -  number of independent variables, NVars>=1       |
//|   NOut        -  number of dependent variables, NOut>=1          |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  KNN builder                                     |
//+------------------------------------------------------------------+
void CAlglib::KNNBuilderSetDatasetReg(CKNNBuilder &s,CMatrixDouble &xy,
                                      int npoints,int nvars,int nout)
  {
   CKNN::KNNBuilderSetDatasetReg(s,xy,npoints,nvars,nout);
  }
//+------------------------------------------------------------------+
//| Specifies classification problem (two or more classes are        |
//| predicted). There also exists "regression" version of this       |
//| function.                                                        |
//| This subroutine adds dense dataset to the internal storage of the|
//| builder object. Specifying your dataset in the dense format means|
//| that the dense version of the KNN construction algorithm will be |
//| invoked.                                                         |
//| INPUT PARAMETERS:                                                |
//|   S        -  KNN builder object                                 |
//|   XY       -  array[NPoints, NVars + 1] (note: actual size can be|
//|               larger, only leading part is used anyway), dataset:|
//|               * first NVars elements of each row store values of |
//|                 the independent variables                        |
//|               * next element stores class index, in [0, NClasses)|
//|   NPoints  -  number of rows in the dataset, NPoints >= 1        |
//|   NVars    -  number of independent variables, NVars >= 1        |
//|   NClasses -  number of classes, NClasses >= 2                   |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  KNN builder                                        |
//+------------------------------------------------------------------+
void CAlglib::KNNBuilderSetDatasetCLS(CKNNBuilder &s,CMatrixDouble &xy,
                                      int npoints,int nvars,int nclasses)
  {
   CKNN::KNNBuilderSetDatasetCLS(s,xy,npoints,nvars,nclasses);
  }
//+------------------------------------------------------------------+
//| This function sets norm type used for neighbor search.           |
//| INPUT PARAMETERS:                                                |
//|   S           -  decision forest builder object                  |
//|   NormType    -  norm type:                                      |
//|                  * 0      inf-norm                               |
//|                  * 1      1-norm                                 |
//|                  * 2      Euclidean norm(default)                |
//| OUTPUT PARAMETERS:                                               |
//|   S           -  decision forest builder                         |
//+------------------------------------------------------------------+
void CAlglib::KNNBuilderSetNorm(CKNNBuilder &s,int nrmtype)
  {
   CKNN::KNNBuilderSetNorm(s,nrmtype);
  }
//+------------------------------------------------------------------+
//| This subroutine builds KNN model according to current settings,  |
//| using dataset internally stored in the builder object.           |
//| The model being built performs inference using Eps-approximate   |
//| K nearest neighbors search algorithm, with:                      |
//|     *K=1,  Eps=0 corresponding to the "nearest neighbor    |
//|                        algorithm"                                |
//|     *K>1,  Eps=0 corresponding to the "K nearest neighbors |
//|                        algorithm"                                |
//|     *K>=1, Eps>0 corresponding to "approximate nearest     |
//|                        neighbors algorithm"                      |
//| An approximate KNN is a good option for high-dimensional datasets|
//| (exact KNN works slowly when dimensions count grows).            |
//| An ALGLIB implementation of kd-trees is used to perform k-nn     |
//| searches.                                                        |
//| INPUT PARAMETERS:                                                |
//|   S           -  KNN builder object                              |
//|   K           -  number of neighbors to search for, K >= 1       |
//|   Eps         -  approximation factor:                           |
//|               * Eps = 0 means that exact kNN search is performed |
//|               * Eps > 0 means that(1 + Eps) - approximate search |
//|                         is performed                             |
//| OUTPUT PARAMETERS:                                               |
//|   Model       -  KNN model                                       |
//|   Rep         -  report                                          |
//+------------------------------------------------------------------+
void CAlglib::KNNBuilderBuildKNNModel(CKNNBuilder &s,int k,double eps,
                                      CKNNModel &model,CKNNReport &rep)
  {
   CKNN::KNNBuilderBuildKNNModel(s,k,eps,model,rep);
  }
//+------------------------------------------------------------------+
//| Changing search settings of KNN model.                           |
//| K and EPS parameters of KNN(AKNN) search are specified  during   |
//| model construction. However, plain KNN algorithm with Euclidean  |
//| distance allows you to change them at any moment.                |
//| NOTE: future versions of KNN model may support advanced versions |
//|       of KNN, such as NCA or LMNN. It is possible that such      |
//|       algorithms won't allow you to change search settings on the|
//|       fly. If you call this function for an algorithm which does |
//|       not support on-the-fly changes, it  will throw an exception|
//| INPUT PARAMETERS:                                                |
//|   Model       -  KNN model                                       |
//|   K           -  K >= 1, neighbors count                         |
//|   EPS         -  accuracy of the EPS-approximate NN search. Set  |
//|                  to 0.0, if you want to perform "classic" KNN    |
//|                  search. Specify larger values if you need to    |
//|                  speed-up high-dimensional KNN queries.          |
//| OUTPUT PARAMETERS:                                               |
//|      nothing on success, exception on failure                    |
//+------------------------------------------------------------------+
void CAlglib::KNNRewriteKEps(CKNNModel &model,int k,double eps)
  {
   CKNN::KNNRewriteKEps(model,k,eps);
  }
//+------------------------------------------------------------------+
//| Inference using KNN model.                                       |
//| See also KNNProcess0(), KNNProcessI() and KNNClassify() for      |
//| options with a bit more convenient interface.                    |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   X        -  input vector,  array[0..NVars - 1].                |
//|   Y        -  possible preallocated buffer. Reused if long enough|
//| OUTPUT PARAMETERS:                                               |
//|   Y        -  result. Regression estimate when solving regression|
//|               task, vector of posterior probabilities for        |
//|               classification task.                               |
//+------------------------------------------------------------------+
void CAlglib::KNNProcess(CKNNModel &model,CRowDouble &x,CRowDouble &y)
  {
   CKNN::KNNProcess(model,x,y);
  }
//+------------------------------------------------------------------+
//| This function returns first component of the inferred vector     |
//| (i.e.one with index #0).                                         |
//| It is a convenience wrapper for KNNProcess() intended for either:|
//|      * 1 - dimensional regression problems                       |
//|      * 2 - class classification problems                         |
//| In the former case this function returns inference result as     |
//| scalar, which is definitely more convenient that wrapping it as  |
//| vector. In the latter case it returns probability of object      |
//| belonging to class #0.                                           |
//| If you call it for anything different from two cases above, it   |
//| will work as defined, i.e. return y[0], although it is of less   |
//| use in such cases.                                               |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   X        -  input vector,  array[0..NVars - 1].                |
//| RESULT:                                                          |
//|   Y[0]                                                           |
//+------------------------------------------------------------------+
double CAlglib::KNNProcess0(CKNNModel &model,CRowDouble &x)
  {
   return(CKNN::KNNProcess0(model,x));
  }
//+------------------------------------------------------------------+
//| This function returns most probable class number for an  input X.|
//| It is same as calling KNNProcess(model, x, y), then determining  |
//| i = argmax(y[i]) and returning i.                                |
//| A class number in [0, NOut) range in returned for classification |
//| problems, -1 is returned when this function is called for        |
//| regression problems.                                             |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   X        -  input vector,  array[0..NVars - 1].                |
//| RESULT:                                                          |
//|   class number, -1 for regression tasks                          |
//+------------------------------------------------------------------+
int CAlglib::KNNClassify(CKNNModel &model,CRowDouble &x)
  {
   return(CKNN::KNNClassify(model,x));
  }
//+------------------------------------------------------------------+
//| 'interactive' variant of KNNProcess() which support constructs   |
//| like "y = KNNProcessI(model,x)" and interactive mode of the      |
//| interpreter.                                                     |
//| This function allocates new array on each call, so it is         |
//| significantly slower than its 'non-interactive' counterpart, but |
//| it is more convenient when you call it from command line.        |
//+------------------------------------------------------------------+
void CAlglib::KNNProcessI(CKNNModel &model,CRowDouble &x,CRowDouble &y)
  {
   y.Resize(0);
//--- function call
   CKNN::KNNProcessI(model,x,y);
  }
//+------------------------------------------------------------------+
//| Thread - safe procesing using external buffer for temporaries.   |
//| This function is thread-safe(i.e. you can use same KNN model from|
//| multiple threads) as long as you use different buffer objects for|
//| different threads.                                               |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   Buf      -  buffer object, must be allocated specifically for  |
//|               this model with KNNCreateBuffer().                 |
//|   X        -  input vector,  array[NVars]                        |
//| OUTPUT PARAMETERS:                                               |
//|   Y        -  result, array[NOut]. Regression estimate when      |
//|               solving regression task, vector of posterior       |
//|               probabilities for a classification task.           |
//+------------------------------------------------------------------+
void CAlglib::KNNTsProcess(CKNNModel &model,CKNNBuffer &buf,
                           CRowDouble &x,CRowDouble &y)
  {
   CKNN::KNNTsProcess(model,buf,x,y);
  }
//+------------------------------------------------------------------+
//| Relative classification error on the test set                    |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   XY       -  test set                                           |
//|   NPoints  -  test set size                                      |
//| RESULT:                                                          |
//|   percent of incorrectly classified cases.                       |
//|   Zero if model solves regression task.                          |
//| NOTE: if you need several different kinds of error metrics, it is|
//|       better to use KNNAllErrors() which computes all error      |
//|       metric with just one pass over dataset.                    |
//+------------------------------------------------------------------+
double CAlglib::KNNRelClsError(CKNNModel &model,CMatrixDouble &xy,
                               int npoints)
  {
   return(CKNN::KNNRelClsError(model,xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average cross-entropy (in bits per element) on the test set      |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   XY       -  test set                                           |
//|   NPoints  -  test set size                                      |
//| RESULT:                                                          |
//|   CrossEntropy / NPoints.                                        |
//|   Zero if model solves regression task.                          |
//| NOTE: the cross-entropy metric is too unstable when used         |
//|       to evaluate KNN models (such models can report exactly     |
//|       zero probabilities), so we do not recommend using it.      |
//| NOTE:  if you need several different kinds of error metrics, it  |
//|        is better to use KNNAllErrors() which computes all error  |
//|        metric with just one pass over dataset.                   |
//+------------------------------------------------------------------+
double CAlglib::KNNAvgCE(CKNNModel &model,CMatrixDouble &xy,int npoints)
  {
   return(CKNN::KNNAvgCE(model,xy,npoints));
  }
//+------------------------------------------------------------------+
//| RMS error on the test set.                                       |
//| Its meaning for regression task is obvious. As for classification|
//| problems, RMS error means error when estimating posterior        |
//| probabilities.                                                   |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   XY       -  test set                                           |
//|   NPoints  -  test set size                                      |
//| RESULT:                                                          |
//|   root mean square error.                                        |
//| NOTE: if you need several different kinds of error metrics, it   |
//|       is better to use KNNAllErrors() which computes all error   |
//|       metric with just one pass over dataset.                    |
//+------------------------------------------------------------------+
double CAlglib::KNNRMSError(CKNNModel &model,CMatrixDouble &xy,
                            int npoints)
  {
   return(CKNN::KNNRMSError(model,xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average error on the test set                                    |
//| Its meaning for regression task is obvious. As for classification|
//| problems, average error means error when estimating posterior    |
//| probabilities.                                                   |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   XY       -  test set                                           |
//|   NPoints  -  test set size                                      |
//| RESULT:                                                          |
//|   average error                                                  |
//| NOTE: if you need several different kinds of error metrics, it   |
//|       is better to use KNNAllErrors() which computes all error   |
//|       metric with just one pass over dataset.                    |
//+------------------------------------------------------------------+
double CAlglib::KNNAvgError(CKNNModel &model,CMatrixDouble &xy,int npoints)
  {
   return(CKNN::KNNAvgError(model,xy,npoints));
  }
//+------------------------------------------------------------------+
//| Average relative error on the test set                           |
//| Its meaning for regression task is obvious. As for classification|
//| problems, average relative error means error when estimating     |
//| posterior probabilities.                                         |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   XY       -  test set                                           |
//|   NPoints  -  test set size                                      |
//| RESULT:                                                          |
//|   average relative error                                         |
//| NOTE: if you need several different kinds of error metrics, it   |
//|       is better to use KNNAllErrors() which computes all error   |
//|       metric with just one pass over dataset.                    |
//+------------------------------------------------------------------+
double CAlglib::KNNAvgRelError(CKNNModel &model,CMatrixDouble &xy,int npoints)
  {
   return(CKNN::KNNAvgRelError(model,xy,npoints));
  }
//+------------------------------------------------------------------+
//| Calculates all kinds of errors for the model in one call.        |
//| INPUT PARAMETERS:                                                |
//|   Model    -  KNN model                                          |
//|   XY       -  test set:                                          |
//|               * one row per point                                |
//|               * first NVars columns store independent variables  |
//|               * depending on problem type:                       |
//|                  * next column stores class number               |
//|                    in [0, NClasses) - for classification         |
//|                    problems                                      |
//|                  * next NOut columns store dependent             |
//|                    variables - for regression problems           |
//|   NPoints  -  test set size, NPoints >= 0                        |
//| OUTPUT PARAMETERS:                                               |
//|   Rep      -  following fields are loaded with errors for both   |
//|               regression and classification models:              |
//|               * rep.RMSError - RMS error for the output          |
//|               * rep.AvgError - average error                     |
//|               * rep.AvgRelError - average relative error         |
//|            following fields are set only for classification      |
//|            models, zero for regression ones:                     |
//|               * relclserror   - relative classification error,   |
//|                                 in [0, 1]                        |
//|               * avgce - average cross-entropy in bits per        |
//|                         dataset entry                            |
//| NOTE: the cross-entropy metric is too unstable when used to      |
//|       evaluate KNN models (such models can report exactly zero   |
//|       probabilities), so we do not recommend using it.           |
//+------------------------------------------------------------------+
void CAlglib::KNNAllErrors(CKNNModel &model,CMatrixDouble &xy,
                           int npoints,CKNNReport &rep)
  {
   CKNN::KNNAllErrors(model,xy,npoints,rep);
  }
//+------------------------------------------------------------------+
//| 1-dimensional complex FFT.                                       |
//| Array size N may be arbitrary number (composite or prime).       |
//| Composite N's are handled with cache-oblivious variation of a    |
//| Cooley-Tukey algorithm. Small prime-factors are transformed using|
//| hard coded codelets (similar to FFTW codelets, but without       |
//| low-level optimization), large prime-factors are handled with    |
//| Bluestein's algorithm.                                           |
//| Fastests transforms are for smooth N's (prime factors are 2, 3,  |
//| 5 only), most fast for powers of 2. When N have prime factors    |
//| larger than these, but orders of magnitude smaller than N,       |
//| computations will be about 4 times slower than for nearby highly |
//| composite N's. When N itself is prime, speed will be 6 times     |
//| lower.                                                           |
//| Algorithm has O(N*logN) complexity for any N (composite or       |
//| prime).                                                          |
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..N-1] - complex function to be transformed   |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     A   -   DFT of a input array, array[0..N-1]                  |
//|             A_out[j] = SUM(A_in[k]*exp(-2*pi*sqrt(-1)*j*k/N),    |
//|             k = 0..N-1)                                          |
//+------------------------------------------------------------------+
void CAlglib::FFTC1D(complex &a[],const int n)
  {
   CFastFourierTransform::FFTC1D(a,n);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FFTC1D(CRowComplex &a,const int n)
  {
   CFastFourierTransform::FFTC1D(a,n);
  }
//+------------------------------------------------------------------+
//| 1-dimensional complex FFT.                                       |
//| Array size N may be arbitrary number (composite or prime).       |
//| Composite N's are handled with cache-oblivious variation of a    |
//| Cooley-Tukey algorithm. Small prime-factors are transformed using|
//| hard coded codelets (similar to FFTW codelets, but without       |
//| low-level optimization), large prime-factors are handled with    |
//| Bluestein's algorithm.                                           |
//| Fastests transforms are for smooth N's (prime factors are 2, 3,  |
//| 5 only), most fast for powers of 2. When N have prime factors    |
//| larger than these, but orders of magnitude smaller than N,       |
//| computations will be about 4 times slower than for nearby highly |
//| composite N's. When N itself is prime, speed will be 6 times     |
//| lower.                                                           |
//| Algorithm has O(N*logN) complexity for any N (composite or       |
//| prime).                                                          |
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..N-1] - complex function to be transformed   |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     A   -   DFT of a input array, array[0..N-1]                  |
//|             A_out[j] = SUM(A_in[k]*exp(-2*pi*sqrt(-1)*j*k/N),    |
//|             k = 0..N-1)                                          |
//+------------------------------------------------------------------+
void CAlglib::FFTC1D(complex &a[])
  {
//--- initialization
   int n=CAp::Len(a);
//--- function call
   CFastFourierTransform::FFTC1D(a,n);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FFTC1D(CRowComplex &a)
  {
//--- initialization
   int n=CAp::Len(a);
//--- function call
   CFastFourierTransform::FFTC1D(a,n);
  }
//+------------------------------------------------------------------+
//| 1-dimensional complex inverse FFT.                               |
//| Array size N may be arbitrary number (composite or prime).       |
//| Algorithm has O(N*logN) complexity for any N (composite or prime)|
//| See FFTC1D() description for more information about algorithm    |
//| performance.                                                     |
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..N-1] - complex array to be transformed      |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     A   -   inverse DFT of a input array, array[0..N-1]          |
//|             A_out[j] = SUM(A_in[k]/N*exp(+2*pi*sqrt(-1)*j*k/N),  |
//|             k = 0..N-1)                                          |
//+------------------------------------------------------------------+
void CAlglib::FFTC1DInv(complex &a[],const int n)
  {
   CFastFourierTransform::FFTC1DInv(a,n);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FFTC1DInv(CRowComplex &a,const int n)
  {
   CFastFourierTransform::FFTC1DInv(a,n);
  }
//+------------------------------------------------------------------+
//| 1-dimensional complex inverse FFT.                               |
//| Array size N may be arbitrary number (composite or prime).       |
//| Algorithm has O(N*logN) complexity for any N (composite or prime)|
//| See FFTC1D() description for more information about algorithm    |
//| performance.                                                     |
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..N-1] - complex array to be transformed      |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     A   -   inverse DFT of a input array, array[0..N-1]          |
//|             A_out[j] = SUM(A_in[k]/N*exp(+2*pi*sqrt(-1)*j*k/N),  |
//|             k = 0..N-1)                                          |
//+------------------------------------------------------------------+
void CAlglib::FFTC1DInv(complex &a[])
  {
//--- initialization
   int n=CAp::Len(a);
//--- function call
   CFastFourierTransform::FFTC1DInv(a,n);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FFTC1DInv(CRowComplex &a)
  {
//--- initialization
   int n=CAp::Len(a);
//--- function call
   CFastFourierTransform::FFTC1DInv(a,n);
  }
//+------------------------------------------------------------------+
//| 1-dimensional real FFT.                                          |
//| Algorithm has O(N*logN) complexity for any N (composite or       |
//| prime).                                                          |
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..N-1] - real function to be transformed      |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     F   -   DFT of a input array, array[0..N-1]                  |
//|             F[j] = SUM(A[k]*exp(-2*pi*sqrt(-1)*j*k/N),           |
//|             k = 0..N-1)                                          |
//| NOTE:                                                            |
//|     F[] satisfies symmetry property F[k] = conj(F[N-k]), so just |
//| one half of array is usually needed. But for convinience         |
//| subroutine returns full complex array (with frequencies above    |
//| N/2), so its result may be used by other FFT-related subroutines.|
//+------------------------------------------------------------------+
void CAlglib::FFTR1D(double &a[],const int n,complex &f[])
  {
   CFastFourierTransform::FFTR1D(a,n,f);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FFTR1D(CRowDouble &a,const int n,CRowComplex &f)
  {
   CFastFourierTransform::FFTR1D(a,n,f);
  }
//+------------------------------------------------------------------+
//| 1-dimensional real FFT.                                          |
//| Algorithm has O(N*logN) complexity for any N (composite or       |
//| prime).                                                          |
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..N-1] - real function to be transformed      |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     F   -   DFT of a input array, array[0..N-1]                  |
//|             F[j] = SUM(A[k]*exp(-2*pi*sqrt(-1)*j*k/N),           |
//|             k = 0..N-1)                                          |
//| NOTE:                                                            |
//|     F[] satisfies symmetry property F[k] = conj(F[N-k]), so just |
//| one half of array is usually needed. But for convinience         |
//| subroutine returns full complex array (with frequencies above    |
//| N/2), so its result may be used by other FFT-related subroutines.|
//+------------------------------------------------------------------+
void CAlglib::FFTR1D(double &a[],complex &f[])
  {
//--- initialization
   int n=CAp::Len(a);
//--- function call
   CFastFourierTransform::FFTR1D(a,n,f);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FFTR1D(CRowDouble &a,CRowComplex &f)
  {
//--- initialization
   int n=CAp::Len(a);
//--- function call
   CFastFourierTransform::FFTR1D(a,n,f);
  }
//+------------------------------------------------------------------+
//| 1-dimensional real inverse FFT.                                  |
//| Algorithm has O(N*logN) complexity for any N (composite or       |
//| prime).                                                          |
//| INPUT PARAMETERS                                                 |
//|     F   -   array[0..floor(N/2)] - frequencies from forward real |
//|             FFT                                                  |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     A   -   inverse DFT of a input array, array[0..N-1]          |
//| NOTE:                                                            |
//|     F[] should satisfy symmetry property F[k] = conj(F[N-k]),    |
//| so just one half of frequencies array is needed - elements from 0|
//| to floor(N/2). F[0] is ALWAYS real. If N is even F[floor(N/2)] is|
//| real too. If N is odd, then F[floor(N/2)] has no special         |
//| properties.                                                      |
//| Relying on properties noted above, FFTR1DInv subroutine uses only|
//| elements from 0th to floor(N/2)-th. It ignores imaginary part of |
//| F[0], and in case N is even it ignores imaginary part of         |
//| F[floor(N/2)] too.                                               |
//| When you call this function using full arguments list -          |
//| "FFTR1DInv(F,N,A)"                                               |
//| - you can pass either either frequencies array with N elements or|
//| reduced array with roughly N/2 elements - subroutine will        |
//| successfully transform both.                                     |
//| If you call this function using reduced arguments list -         |
//| "FFTR1DInv(F,A)" - you must pass FULL array with N elements      |
//| (although higher N/2 are still not used) because array size is   |
//| used to automatically determine FFT length                       |
//+------------------------------------------------------------------+
void CAlglib::FFTR1DInv(complex &f[],const int n,double &a[])
  {
   CFastFourierTransform::FFTR1DInv(f,n,a);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FFTR1DInv(CRowComplex &f,const int n,CRowDouble &a)
  {
   CFastFourierTransform::FFTR1DInv(f,n,a);
  }
//+------------------------------------------------------------------+
//| 1-dimensional real inverse FFT.                                  |
//| Algorithm has O(N*logN) complexity for any N (composite or       |
//| prime).                                                          |
//| INPUT PARAMETERS                                                 |
//|     F   -   array[0..floor(N/2)] - frequencies from forward real |
//|             FFT                                                  |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     A   -   inverse DFT of a input array, array[0..N-1]          |
//| NOTE:                                                            |
//|     F[] should satisfy symmetry property F[k] = conj(F[N-k]),    |
//| so just one half of frequencies array is needed - elements from 0|
//| to floor(N/2). F[0] is ALWAYS real. If N is even F[floor(N/2)] is|
//| real too. If N is odd, then F[floor(N/2)] has no special         |
//| properties.                                                      |
//| Relying on properties noted above, FFTR1DInv subroutine uses only|
//| elements from 0th to floor(N/2)-th. It ignores imaginary part of |
//| F[0], and in case N is even it ignores imaginary part of         |
//| F[floor(N/2)] too.                                               |
//| When you call this function using full arguments list -          |
//| "FFTR1DInv(F,N,A)"                                               |
//| - you can pass either either frequencies array with N elements or|
//| reduced array with roughly N/2 elements - subroutine will        |
//| successfully transform both.                                     |
//| If you call this function using reduced arguments list -         |
//| "FFTR1DInv(F,A)" - you must pass FULL array with N elements      |
//| (although higher N/2 are still not used) because array size is   |
//| used to automatically determine FFT length                       |
//+------------------------------------------------------------------+
void CAlglib::FFTR1DInv(complex &f[],double &a[])
  {
//--- initialization
   int n=CAp::Len(f);
//--- function call
   CFastFourierTransform::FFTR1DInv(f,n,a);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::FFTR1DInv(CRowComplex &f,CRowDouble &a)
  {
//--- initialization
   int n=CAp::Len(f);
//--- function call
   CFastFourierTransform::FFTR1DInv(f,n,a);
  }
//+------------------------------------------------------------------+
//| 1-dimensional complex convolution.                               |
//| For given A/B returns conv(A,B) (non-circular). Subroutine can   |
//| automatically choose between three implementations:              |
//| straightforward O(M*N) formula for very small N (or M),          |
//| significantly larger than min(M,N), but O(M*N) algorithm is too  |
//| slow, and general FFT-based formula for cases where two previois |
//| algorithms are too slow.                                         |
//| Algorithm has max(M,N)*log(max(M,N)) complexity for any M/N.     |
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..M-1] - complex function to be transformed   |
//|     M   -   problem size                                         |
//|     B   -   array[0..N-1] - complex function to be transformed   |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     R   -   convolution: A*B. array[0..N+M-2].                   |
//| NOTE:                                                            |
//|     It is assumed that A is zero at T<0, B is zero too. If one or|
//| both functions have non-zero values at negative T's, you can     |
//| still use this subroutine - just shift its result                |
//| correspondingly.                                                 |
//+------------------------------------------------------------------+
void CAlglib::ConvC1D(complex &a[],const int m,complex &b[],
                      const int n,complex &r[])
  {
   CConv::ConvC1D(a,m,b,n,r);
  }
//+------------------------------------------------------------------+
//| 1-dimensional complex non-circular deconvolution (inverse of     |
//| ConvC1D()).                                                      |
//| Algorithm has M*log(M)) complexity for any M (composite or prime)|
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..M-1] - convolved signal, A = conv(R, B)     |
//|     M   -   convolved signal length                              |
//|     B   -   array[0..N-1] - response                             |
//|     N   -   response length, N<=M                                |
//| OUTPUT PARAMETERS                                                |
//|     R   -   deconvolved signal. array[0..M-N].                   |
//| NOTE:                                                            |
//|     deconvolution is unstable process and may result in division |
//| by zero (if your response function is degenerate, i.e. has zero  |
//| Fourier coefficient).                                            |
//| NOTE:                                                            |
//|     It is assumed that A is zero at T<0, B is zero too. If one   |
//| or both functions have non-zero values at negative T's, you can  |
//| still use this subroutine - just shift its result correspondingly|
//+------------------------------------------------------------------+
void CAlglib::ConvC1DInv(complex &a[],const int m,complex &b[],
                         const int n,complex &r[])
  {
   CConv::ConvC1DInv(a,m,b,n,r);
  }
//+------------------------------------------------------------------+
//| 1-dimensional circular complex convolution.                      |
//| For given S/R returns conv(S,R) (circular). Algorithm has        |
//| linearithmic complexity for any M/N.                             |
//| IMPORTANT:  normal convolution is commutative, i.e. it is        |
//| symmetric - conv(A,B)=conv(B,A). Cyclic convolution IS NOT. One  |
//| function - S - is a signal, periodic function, and another - R - |
//| is a response, non-periodic function with limited length.        |
//| INPUT PARAMETERS                                                 |
//|     S   -   array[0..M-1] - complex periodic signal              |
//|     M   -   problem size                                         |
//|     B   -   array[0..N-1] - complex non-periodic response        |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     R   -   convolution: A*B. array[0..M-1].                     |
//| NOTE:                                                            |
//|     It is assumed that B is zero at T<0. If it has non-zero      |
//| values at negative T's, you can still use this subroutine - just |
//| shift its result correspondingly.                                |
//+------------------------------------------------------------------+
void CAlglib::ConvC1DCircular(complex &s[],const int m,complex &r[],
                              const int n,complex &c[])
  {
   CConv::ConvC1DCircular(s,m,r,n,c);
  }
//+------------------------------------------------------------------+
//| 1-dimensional circular complex deconvolution (inverse of         |
//| ConvC1DCircular()).                                              |
//| Algorithm has M*log(M)) complexity for any M (composite or prime)|
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..M-1] - convolved periodic signal,           |
//|             A = conv(R, B)                                       |
//|     M   -   convolved signal length                              |
//|     B   -   array[0..N-1] - non-periodic response                |
//|     N   -   response length                                      |
//| OUTPUT PARAMETERS                                                |
//|     R   -   deconvolved signal. array[0..M-1].                   |
//| NOTE:                                                            |
//|     deconvolution is unstable process and may result in division |
//| by zero (if your response function is degenerate, i.e. has zero  |
//| Fourier coefficient).                                            |
//| NOTE:                                                            |
//|     It is assumed that B is zero at T<0. If it has non-zero      |
//| values at negative T's, you can still use this subroutine - just |
//| shift its result correspondingly.                                |
//+------------------------------------------------------------------+
void CAlglib::ConvC1DCircularInv(complex &a[],const int m,complex &b[],
                                 const int n,complex &r[])
  {
   CConv::ConvC1DCircularInv(a,m,b,n,r);
  }
//+------------------------------------------------------------------+
//| 1-dimensional real convolution.                                  |
//| Analogous to ConvC1D(), see ConvC1D() comments for more details. |
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..M-1] - real function to be transformed      |
//|     M   -   problem size                                         |
//|     B   -   array[0..N-1] - real function to be transformed      |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     R   -   convolution: A*B. array[0..N+M-2].                   |
//| NOTE:                                                            |
//|     It is assumed that A is zero at T<0, B is zero too. If one   |
//| or both functions have non-zero values at negative T's, you can  |
//| still use this subroutine - just shift its result correspondingly|
//+------------------------------------------------------------------+
void CAlglib::ConvR1D(double &a[],const int m,double &b[],
                      const int n,double &r[])
  {
   CConv::ConvR1D(a,m,b,n,r);
  }
//+------------------------------------------------------------------+
//| 1-dimensional real deconvolution (inverse of ConvC1D()).         |
//| Algorithm has M*log(M)) complexity for any M (composite or prime)|
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..M-1] - convolved signal, A = conv(R, B)     |
//|     M   -   convolved signal length                              |
//|     B   -   array[0..N-1] - response                             |
//|     N   -   response length, N<=M                                |
//| OUTPUT PARAMETERS                                                |
//|     R   -   deconvolved signal. array[0..M-N].                   |
//| NOTE:                                                            |
//|     deconvolution is unstable process and may result in division |
//| by zero (if your response function is degenerate, i.e. has zero  |
//| Fourier coefficient).                                            |
//| NOTE:                                                            |
//|     It is assumed that A is zero at T<0, B is zero too. If one or|
//| both functions have non-zero values at negative T's, you can     |
//| still use this subroutine - just shift its result correspondingly|
//+------------------------------------------------------------------+
void CAlglib::ConvR1DInv(double &a[],const int m,double &b[],
                         const int n,double &r[])
  {
   CConv::ConvR1DInv(a,m,b,n,r);
  }
//+------------------------------------------------------------------+
//| 1-dimensional circular real convolution.                         |
//| Analogous to ConvC1DCircular(), see ConvC1DCircular() comments   |
//| for more details.                                                |
//| INPUT PARAMETERS                                                 |
//|     S   -   array[0..M-1] - real signal                          |
//|     M   -   problem size                                         |
//|     B   -   array[0..N-1] - real response                        |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     R   -   convolution: A*B. array[0..M-1].                     |
//| NOTE:                                                            |
//|     It is assumed that B is zero at T<0. If it has non-zero      |
//| values at negative T's, you can still use this subroutine - just |
//| shift its result correspondingly.                                |
//+------------------------------------------------------------------+
void CAlglib::ConvR1DCircular(double &s[],const int m,double &r[],
                              const int n,double &c[])
  {
   CConv::ConvR1DCircular(s,m,r,n,c);
  }
//+------------------------------------------------------------------+
//| 1-dimensional complex deconvolution (inverse of ConvC1D()).      |
//| Algorithm has M*log(M)) complexity for any M (composite or prime)|
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..M-1] - convolved signal, A = conv(R, B)     |
//|     M   -   convolved signal length                              |
//|     B   -   array[0..N-1] - response                             |
//|     N   -   response length                                      |
//| OUTPUT PARAMETERS                                                |
//|     R   -   deconvolved signal. array[0..M-N].                   |
//| NOTE:                                                            |
//|     deconvolution is unstable process and may result in division |
//| by zero (if your response function is degenerate, i.e. has zero  |
//| Fourier coefficient).                                            |
//| NOTE:                                                            |
//|     It is assumed that B is zero at T<0. If it has non-zero      |
//| values at negative T's, you can still use this subroutine - just |
//| shift its result correspondingly.                                |
//+------------------------------------------------------------------+
void CAlglib::ConvR1DCircularInv(double &a[],const int m,double &b[],
                                 const int n,double &r[])
  {
   CConv::ConvR1DCircularInv(a,m,b,n,r);
  }
//+------------------------------------------------------------------+
//| 1-dimensional complex cross-correlation.                         |
//| For given Pattern/Signal returns corr(Pattern,Signal)            |
//| (non-circular).                                                  |
//| Correlation is calculated using reduction to convolution.        |
//| Algorithm with max(N,N)*log(max(N,N)) complexity is used (see    |
//| ConvC1D() for more info about performance).                      |
//| IMPORTANT:                                                       |
//|     for historical reasons subroutine accepts its parameters in  |
//|     reversed order: CorrC1D(Signal, Pattern) = Pattern x Signal  |
//|     (using traditional definition of cross-correlation, denoting |
//|     cross-correlation as "x").                                   |
//| INPUT PARAMETERS                                                 |
//|     Signal  -   array[0..N-1] - complex function to be           |
//|                 transformed, signal containing pattern           |
//|     N       -   problem size                                     |
//|     Pattern -   array[0..M-1] - complex function to be           |
//|                 transformed, pattern to search withing signal    |
//|     M       -   problem size                                     |
//| OUTPUT PARAMETERS                                                |
//|     R       -   cross-correlation, array[0..N+M-2]:              |
//|                 * positive lags are stored in R[0..N-1],         |
//|                   R[i] = sum(conj(pattern[j])*signal[i+j]        |
//|                 * negative lags are stored in R[N..N+M-2],       |
//|                   R[N+M-1-i] = sum(conj(pattern[j])*signal[-i+j] |
//| NOTE:                                                            |
//|     It is assumed that pattern domain is [0..M-1]. If Pattern is |
//| non-zero on [-K..M-1], you can still use this subroutine, just   |
//| shift result by K.                                               |
//+------------------------------------------------------------------+
void CAlglib::CorrC1D(complex &signal[],const int n,complex &pattern[],
                      const int m,complex &r[])
  {
   CCorr::CorrC1D(signal,n,pattern,m,r);
  }
//+------------------------------------------------------------------+
//| 1-dimensional circular complex cross-correlation.                |
//| For given Pattern/Signal returns corr(Pattern,Signal) (circular).|
//| Algorithm has linearithmic complexity for any M/N.               |
//| IMPORTANT:                                                       |
//|     for historical reasons subroutine accepts its parameters in  |
//|     reversed order: CorrC1DCircular(Signal, Pattern) = Pattern x |
//|     Signal (using traditional definition of cross-correlation,   |
//|     denoting cross-correlation as "x").                          |
//| INPUT PARAMETERS                                                 |
//|     Signal  -   array[0..N-1] - complex function to be           |
//|                 transformed, periodic signal containing pattern  |
//|     N       -   problem size                                     |
//|     Pattern -   array[0..M-1] - complex function to be           |
//|                 transformed, non-periodic pattern to search      |
//|                 withing signal                                   |
//|     M       -   problem size                                     |
//| OUTPUT PARAMETERS                                                |
//|     R   -   convolution: A*B. array[0..M-1].                     |
//+------------------------------------------------------------------+
void CAlglib::CorrC1DCircular(complex &signal[],const int m,
                              complex &pattern[],const int n,
                              complex &c[])
  {
   CCorr::CorrC1DCircular(signal,m,pattern,n,c);
  }
//+------------------------------------------------------------------+
//| 1-dimensional real cross-correlation.                            |
//| For given Pattern/Signal returns corr(Pattern,Signal)            |
//| (non-circular).                                                  |
//| Correlation is calculated using reduction to convolution.        |
//| Algorithm with max(N,N)*log(max(N,N)) complexity is used (see    |
//| ConvC1D() for more info about performance).                      |
//| IMPORTANT:                                                       |
//|     for  historical reasons subroutine accepts its parameters in |
//|     reversed order: CorrR1D(Signal, Pattern) = Pattern x Signal  |
//|     (using  traditional definition of cross-correlation, denoting|
//|     cross-correlation as "x").                                   |
//| INPUT PARAMETERS                                                 |
//|     Signal  -   array[0..N-1] - real function to be transformed, |
//|                 signal containing pattern                        |
//|     N       -   problem size                                     |
//|     Pattern -   array[0..M-1] - real function to be transformed, |
//|                 pattern to search withing signal                 |
//|     M       -   problem size                                     |
//| OUTPUT PARAMETERS                                                |
//|     R       -   cross-correlation, array[0..N+M-2]:              |
//|                 * positive lags are stored in R[0..N-1],         |
//|                   R[i] = sum(pattern[j]*signal[i+j]              |
//|                 * negative lags are stored in R[N..N+M-2],       |
//|                   R[N+M-1-i] = sum(pattern[j]*signal[-i+j]       |
//| NOTE:                                                            |
//|     It is assumed that pattern domain is [0..M-1]. If Pattern is |
//| non-zero on [-K..M-1],  you can still use this subroutine, just  |
//| shift result by K.                                               |
//+------------------------------------------------------------------+
void CAlglib::CorrR1D(double &signal[],const int n,double &pattern[],
                      const int m,double &r[])
  {
   CCorr::CorrR1D(signal,n,pattern,m,r);
  }
//+------------------------------------------------------------------+
//| 1-dimensional circular real cross-correlation.                   |
//| For given Pattern/Signal returns corr(Pattern,Signal) (circular).|
//| Algorithm has linearithmic complexity for any M/N.               |
//| IMPORTANT:                                                       |
//|     for  historical reasons subroutine accepts its parameters in |
//|     reversed order: CorrR1DCircular(Signal, Pattern) = Pattern x |
//|     Signal (using traditional definition of cross-correlation,   |
//|     denoting cross-correlation as "x").                          |
//| INPUT PARAMETERS                                                 |
//|     Signal  -   array[0..N-1] - real function to be transformed, |
//|                 periodic signal containing pattern               |
//|     N       -   problem size                                     |
//|     Pattern -   array[0..M-1] - real function to be transformed, |
//|                 non-periodic pattern to search withing signal    |
//|     M       -   problem size                                     |
//| OUTPUT PARAMETERS                                                |
//|     R   -   convolution: A*B. array[0..M-1].                     |
//+------------------------------------------------------------------+
void CAlglib::CorrR1DCircular(double &signal[],const int m,
                              double &pattern[],const int n,
                              double &c[])
  {
   CCorr::CorrR1DCircular(signal,m,pattern,n,c);
  }
//+------------------------------------------------------------------+
//| 1-dimensional Fast Hartley Transform.                            |
//| Algorithm has O(N*logN) complexity for any N (composite or prime)|
//| INPUT PARAMETERS                                                 |
//|     A  -  array[0..N-1] - real function to be transformed        |
//|     N  -  problem size                                           |
//| OUTPUT PARAMETERS                                                |
//|     A  -  FHT of a input array, array[0..N-1],                   |
//|           A_out[k]=sum(A_in[j]*(cos(2*pi*j*k/N)+sin(2*pi*j*k/N)),|
//|           j=0..N-1)                                              |
//+------------------------------------------------------------------+
void CAlglib::FHTR1D(double &a[],const int n)
  {
   CFastHartleyTransform::FHTR1D(a,n);
  }
//+------------------------------------------------------------------+
//| 1-dimensional inverse FHT.                                       |
//| Algorithm has O(N*logN) complexity for any N (composite or prime)|
//| INPUT PARAMETERS                                                 |
//|     A   -   array[0..N-1] - complex array to be transformed      |
//|     N   -   problem size                                         |
//| OUTPUT PARAMETERS                                                |
//|     A   -   inverse FHT of a input array, array[0..N-1]          |
//+------------------------------------------------------------------+
void CAlglib::FHTR1DInv(double &a[],const int n)
  {
   CFastHartleyTransform::FHTR1DInv(a,n);
  }
//+------------------------------------------------------------------+
//| Computation of nodes and weights for a Gauss quadrature formula  |
//| The algorithm generates the N-point Gauss quadrature formula     |
//| with weight function given by coefficients alpha and beta of a   |
//| recurrence relation which generates a system of orthogonal       |
//| polynomials:                                                     |
//| P-1(x)   =  0                                                    |
//| P0(x)    =  1                                                    |
//| Pn+1(x)  =  (x-alpha(n))*Pn(x)  -  beta(n)*Pn-1(x)               |
//| and zeroth moment Mu0                                            |
//| Mu0 = integral(W(x)dx,a,b)                                       |
//| INPUT PARAMETERS:                                                |
//|     Alpha   ?   array[0..N-1], alpha coefficients                |
//|     Beta    ?   array[0..N-1], beta coefficients                 |
//|                 Zero-indexed element is not used and may be      |
//|                 arbitrary. Beta[I]>0.                            |
//|     Mu0     ?   zeroth moment of the weight function.            |
//|     N       ?   number of nodes of the quadrature formula, N>=1  |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -3    internal eigenproblem solver hasn't      |
//|                         converged                                |
//|                 * -2    Beta[i]<=0                               |
//|                 * -1    incorrect N was passed                   |
//|                 *  1    OK                                       |
//|     X       -   array[0..N-1] - array of quadrature nodes,       |
//|                 in ascending order.                              |
//|     W       -   array[0..N-1] - array of quadrature weights.     |
//+------------------------------------------------------------------+
void CAlglib::GQGenerateRec(double &alpha[],double &beta[],
                            const double mu0,const int n,
                            int &info,double &x[],double &w[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussQ::GQGenerateRec(alpha,beta,mu0,n,info,x,w);
  }
//+------------------------------------------------------------------+
//| Computation of nodes and weights for a Gauss-Lobatto quadrature  |
//| formula                                                          |
//| The algorithm generates the N-point Gauss-Lobatto quadrature     |
//| formula with weight function given by coefficients alpha and beta|
//| of a recurrence which generates a system of orthogonal           |
//| polynomials.                                                     |
//| P-1(x)   =  0                                                    |
//| P0(x)    =  1                                                    |
//| Pn+1(x)  =  (x-alpha(n))*Pn(x)  -  beta(n)*Pn-1(x)               |
//| and zeroth moment Mu0                                            |
//| Mu0 = integral(W(x)dx,a,b)                                       |
//| INPUT PARAMETERS:                                                |
//|     Alpha   ?   array[0..N-2], alpha coefficients                |
//|     Beta    ?   array[0..N-2], beta coefficients.                |
//|                 Zero-indexed element is not used, may be         |
//|                 arbitrary. Beta[I]>0                             |
//|     Mu0     ?   zeroth moment of the weighting function.         |
//|     A       ?   left boundary of the integration interval.       |
//|     B       ?   right boundary of the integration interval.      |
//|     N       ?   number of nodes of the quadrature formula, N>=3  |
//|                 (including the left and right boundary nodes).   |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -3    internal eigenproblem solver hasn't      |
//|                         converged                                |
//|                 * -2    Beta[i]<=0                               |
//|                 * -1    incorrect N was passed                   |
//|                 *  1    OK                                       |
//|     X       -   array[0..N-1] - array of quadrature nodes,       |
//|                 in ascending order.                              |
//|     W       -   array[0..N-1] - array of quadrature weights.     |
//+------------------------------------------------------------------+
void CAlglib::GQGenerateGaussLobattoRec(double &alpha[],double &beta[],
                                        const double mu0,const double a,
                                        const double b,const int n,
                                        int &info,double &x[],double &w[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussQ::GQGenerateGaussLobattoRec(alpha,beta,mu0,a,b,n,info,x,w);
  }
//+------------------------------------------------------------------+
//| Computation of nodes and weights for a Gauss-Radau quadrature    |
//| formula                                                          |
//| The algorithm generates the N-point Gauss-Radau  quadrature      |
//| formula with weight function given by the coefficients alpha and |
//| beta of a recurrence which generates a system of orthogonal      |
//| polynomials.                                                     |
//| P-1(x)   =  0                                                    |
//| P0(x)    =  1                                                    |
//| Pn+1(x)  =  (x-alpha(n))*Pn(x)  -  beta(n)*Pn-1(x)               |
//| and zeroth moment Mu0                                            |
//| Mu0 = integral(W(x)dx,a,b)                                       |
//| INPUT PARAMETERS:                                                |
//|     Alpha   ?   array[0..N-2], alpha coefficients.               |
//|     Beta    ?   array[0..N-1], beta coefficients                 |
//|                 Zero-indexed element is not used.                |
//|                 Beta[I]>0                                        |
//|     Mu0     ?   zeroth moment of the weighting function.         |
//|     A       ?   left boundary of the integration interval.       |
//|     N       ?   number of nodes of the quadrature formula, N>=2  |
//|                 (including the left boundary node).              |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -3    internal eigenproblem solver hasn't      |
//|                         converged                                |
//|                 * -2    Beta[i]<=0                               |
//|                 * -1    incorrect N was passed                   |
//|                 *  1    OK                                       |
//|     X       -   array[0..N-1] - array of quadrature nodes,       |
//|                 in ascending order.                              |
//|     W       -   array[0..N-1] - array of quadrature weights.     |
//+------------------------------------------------------------------+
void CAlglib::GQGenerateGaussRadauRec(double &alpha[],double &beta[],
                                      const double mu0,const double a,
                                      const int n,int &info,
                                      double &x[],double &w[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussQ::GQGenerateGaussRadauRec(alpha,beta,mu0,a,n,info,x,w);
  }
//+------------------------------------------------------------------+
//| Returns nodes/weights for Gauss-Legendre quadrature on [-1,1]    |
//| with N nodes.                                                    |
//| INPUT PARAMETERS:                                                |
//|     N           -   number of nodes, >=1                         |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   error code:                                  |
//|                     * -4    an error was detected when           |
//|                             calculating weights/nodes. N is too  |
//|                             large to obtain weights/nodes with   |
//|                             high enough accuracy. Try to use     |
//|                             multiple precision version.          |
//|                     * -3    internal eigenproblem solver hasn't  |
//|                             converged                            |
//|                     * -1    incorrect N was passed               |
//|                     * +1    OK                                   |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     in ascending order.                          |
//|     W           -   array[0..N-1] - array of quadrature weights. |
//+------------------------------------------------------------------+
void CAlglib::GQGenerateGaussLegendre(const int n,int &info,
                                      double &x[],double &w[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussQ::GQGenerateGaussLegendre(n,info,x,w);
  }
//+------------------------------------------------------------------+
//| Returns  nodes/weights  for  Gauss-Jacobi quadrature on [-1,1]   |
//| with weight function W(x)=Power(1-x,Alpha)*Power(1+x,Beta).      |
//| INPUT PARAMETERS:                                                |
//|     N           -   number of nodes, >=1                         |
//|     Alpha       -   power-law coefficient, Alpha>-1              |
//|     Beta        -   power-law coefficient, Beta>-1               |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   error code:                                  |
//|                     * -4    an error was detected when           |
//|                             calculating weights/nodes. Alpha or  |
//|                             Beta are too close to -1 to obtain   |
//|                             weights/nodes with high enough       |
//|                             accuracy, or, may be, N is too large.|
//|                             Try to use multiple precision version|
//|                     * -3    internal eigenproblem solver hasn't  |
//|                             converged                            |
//|                     * -1    incorrect N/Alpha/Beta was passed    |
//|                     * +1    OK                                   |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     in ascending order.                          |
//|     W           -   array[0..N-1] - array of quadrature weights. |
//+------------------------------------------------------------------+
void CAlglib::GQGenerateGaussJacobi(const int n,const double alpha,
                                    const double beta,int &info,
                                    double &x[],double &w[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussQ::GQGenerateGaussJacobi(n,alpha,beta,info,x,w);
  }
//+------------------------------------------------------------------+
//| Returns nodes/weights for Gauss-Laguerre quadrature on [0,+inf)  |
//| with weight function W(x)=Power(x,Alpha)*Exp(-x)                 |
//| INPUT PARAMETERS:                                                |
//|     N           -   number of nodes, >=1                         |
//|     Alpha       -   power-law coefficient, Alpha>-1              |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   error code:                                  |
//|                     * -4    an error was detected when           |
//|                             calculating weights/nodes. Alpha is  |
//|                             too close to -1 to obtain            |
//|                             weights/nodes with high enough       |
//|                             accuracy or, may  be, N is too large.|
//|                             Try  to  use multiple precision      |
//|                             version.                             |
//|                     * -3    internal eigenproblem solver hasn't  |
//|                             converged                            |
//|                     * -1    incorrect N/Alpha was passed         |
//|                     * +1    OK                                   |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     in ascending order.                          |
//|     W           -   array[0..N-1] - array of quadrature weights. |
//+------------------------------------------------------------------+
void CAlglib::GQGenerateGaussLaguerre(const int n,const double alpha,
                                      int &info,double &x[],double &w[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussQ::GQGenerateGaussLaguerre(n,alpha,info,x,w);
  }
//+------------------------------------------------------------------+
//| Returns  nodes/weights  for  Gauss-Hermite  quadrature on        |
//| (-inf,+inf) with weight function W(x)=Exp(-x*x)                  |
//| INPUT PARAMETERS:                                                |
//|     N           -   number of nodes, >=1                         |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   error code:                                  |
//|                     * -4    an error was detected when           |
//|                             calculating weights/nodes. May be, N |
//|                             is too large. Try to use multiple    |
//|                             precision version.                   |
//|                     * -3    internal eigenproblem solver hasn't  |
//|                             converged                            |
//|                     * -1    incorrect N/Alpha was passed         |
//|                     * +1    OK                                   |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     in ascending order.                          |
//|     W           -   array[0..N-1] - array of quadrature weights. |
//+------------------------------------------------------------------+
void CAlglib::GQGenerateGaussHermite(const int n,int &info,
                                     double &x[],double &w[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussQ::GQGenerateGaussHermite(n,info,x,w);
  }
//+------------------------------------------------------------------+
//| Computation of nodes and weights of a Gauss-Kronrod quadrature   |
//| formula                                                          |
//| The algorithm generates the N-point Gauss-Kronrod quadrature     |
//| formula with weight function given by coefficients alpha and beta|
//| of a recurrence relation which generates a system of orthogonal  |
//| polynomials:                                                     |
//|     P-1(x)   =  0                                                |
//|     P0(x)    =  1                                                |
//|     Pn+1(x)  =  (x-alpha(n))*Pn(x)  -  beta(n)*Pn-1(x)           |
//| and zero moment Mu0                                              |
//|     Mu0 = integral(W(x)dx,a,b)                                   |
//| INPUT PARAMETERS:                                                |
//|     Alpha       ?   alpha coefficients, array[0..floor(3*K/2)].  |
//|     Beta        ?   beta coefficients,  array[0..ceil(3*K/2)].   |
//|                     Beta[0] is not used and may be arbitrary.    |
//|                     Beta[I]>0.                                   |
//|     Mu0         ?   zeroth moment of the weight function.        |
//|     N           ?   number of nodes of the Gauss-Kronrod         |
//|                     quadrature formula,                          |
//|                     N >= 3,                                      |
//|                     N =  2*K+1.                                  |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   error code:                                  |
//|                     * -5    no real and positive Gauss-Kronrod   |
//|                             formula can be created for such a    |
//|                             weight function with a given number  |
//|                             of nodes.                            |
//|                     * -4    N is too large, task may be ill      |
//|                             conditioned - x[i]=x[i+1] found.     |
//|                     * -3    internal eigenproblem solver hasn't  |
//|                             converged                            |
//|                     * -2    Beta[i]<=0                           |
//|                     * -1    incorrect N was passed               |
//|                     * +1    OK                                   |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     in ascending order.                          |
//|     WKronrod    -   array[0..N-1] - Kronrod weights              |
//|     WGauss      -   array[0..N-1] - Gauss weights (interleaved   |
//|                     with zeros corresponding to extended Kronrod |
//|                     nodes).                                      |
//+------------------------------------------------------------------+
void CAlglib::GKQGenerateRec(double &alpha[],double &beta[],
                             const double mu0,const int n,
                             int &info,double &x[],
                             double &wkronrod[],double &wgauss[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussKronrodQ::GKQGenerateRec(alpha,beta,mu0,n,info,x,wkronrod,wgauss);
  }
//+------------------------------------------------------------------+
//| Returns Gauss and Gauss-Kronrod nodes/weights for Gauss-Legendre |
//| quadrature with N points.                                        |
//| GKQLegendreCalc (calculation) or GKQLegendreTbl (precomputed     |
//| table) is used depending on machine precision and number of      |
//| nodes.                                                           |
//| INPUT PARAMETERS:                                                |
//|     N           -   number of Kronrod nodes, must be odd number, |
//|                     >=3.                                         |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   error code:                                  |
//|                     * -4    an error was detected when           |
//|                             calculating weights/nodes. N is too  |
//|                             obtain large to weights/nodes with   |
//|                             high enough accuracy. Try to use     |
//|                             multiple precision version.          |
//|                     * -3    internal eigenproblem solver hasn't  |
//|                             converged                            |
//|                     * -1    incorrect N was passed               |
//|                     * +1    OK                                   |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     ordered in ascending order.                  |
//|     WKronrod    -   array[0..N-1] - Kronrod weights              |
//|     WGauss      -   array[0..N-1] - Gauss weights (interleaved   |
//|                     with zeros corresponding to extended Kronrod |
//|                     nodes).                                      |
//+------------------------------------------------------------------+
void CAlglib::GKQGenerateGaussLegendre(const int n,int &info,
                                       double &x[],double &wkronrod[],
                                       double &wgauss[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussKronrodQ::GKQGenerateGaussLegendre(n,info,x,wkronrod,wgauss);
  }
//+------------------------------------------------------------------+
//| Returns Gauss and Gauss-Kronrod nodes/weights for Gauss-Jacobi   |
//| quadrature on [-1,1] with weight function                        |
//|     W(x)=Power(1-x,Alpha)*Power(1+x,Beta).                       |
//| INPUT PARAMETERS:                                                |
//|     N           -   number of Kronrod nodes, must be odd number, |
//|                     >=3.                                         |
//|     Alpha       -   power-law coefficient, Alpha>-1              |
//|     Beta        -   power-law coefficient, Beta>-1               |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   error code:                                  |
//|                     * -5    no real and positive Gauss-Kronrod   |
//|                             formula can be created for such a    |
//|                             weight function with a given number  |
//|                             of nodes.                            |
//|                     * -4    an  error was detected when          |
//|                             calculating weights/nodes. Alpha or  |
//|                             Beta are too close to -1 to obtain   |
//|                             weights/nodes with high enough       |
//|                             accuracy, or, may be, N is too large.|
//|                             Try to use multiple precision version|
//|                     * -3    internal eigenproblem solver hasn't  |
//|                             converged                            |
//|                     * -1    incorrect N was passed               |
//|                     * +1    OK                                   |
//|                     * +2    OK, but quadrature rule have exterior|
//|                             nodes, x[0]<-1 or x[n-1]>+1          |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     ordered in ascending order.                  |
//|     WKronrod    -   array[0..N-1] - Kronrod weights              |
//|     WGauss      -   array[0..N-1] - Gauss weights (interleaved   |
//|                     with zeros corresponding to extended Kronrod |
//|                     nodes).                                      |
//+------------------------------------------------------------------+
void CAlglib::GKQGenerateGaussJacobi(const int n,const double alpha,
                                     const double beta,int &info,
                                     double &x[],double &wkronrod[],
                                     double &wgauss[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussKronrodQ::GKQGenerateGaussJacobi(n,alpha,beta,info,x,wkronrod,wgauss);
  }
//+------------------------------------------------------------------+
//| Returns Gauss and Gauss-Kronrod nodes for quadrature with N      |
//| points.                                                          |
//| Reduction to tridiagonal eigenproblem is used.                   |
//| INPUT PARAMETERS:                                                |
//|     N           -   number of Kronrod nodes, must be odd number, |
//|                     >=3.                                         |
//| OUTPUT PARAMETERS:                                               |
//|     Info        -   error code:                                  |
//|                     * -4    an error was detected when           |
//|                             calculating weights/nodes. N is too  |
//|                             large to obtain weights/nodes with   |
//|                             high enough accuracy.                |
//|                             Try to use multiple precision        |
//|                             version.                             |
//|                     * -3    internal eigenproblem solver hasn't  |
//|                             converged                            |
//|                     * -1    incorrect N was passed               |
//|                     * +1    OK                                   |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     ordered in ascending order.                  |
//|     WKronrod    -   array[0..N-1] - Kronrod weights              |
//|     WGauss      -   array[0..N-1] - Gauss weights (interleaved   |
//|                     with zeros corresponding to extended Kronrod |
//|                     nodes).                                      |
//+------------------------------------------------------------------+
void CAlglib::GKQLegendreCalc(const int n,int &info,double &x[],
                              double &wkronrod[],double &wgauss[])
  {
//--- initialization
   info=0;
//--- function call
   CGaussKronrodQ::GKQLegendreCalc(n,info,x,wkronrod,wgauss);
  }
//+------------------------------------------------------------------+
//| Returns Gauss and Gauss-Kronrod nodes for quadrature with N      |
//| points using pre-calculated table. Nodes/weights were computed   |
//| with accuracy up to 1.0E-32 (if MPFR version of ALGLIB is used). |
//| In standard double precision accuracy reduces to something about |
//| 2.0E-16 (depending  on your compiler's handling of long floating |
//| point constants).                                                |
//| INPUT PARAMETERS:                                                |
//|     N           -   number of Kronrod nodes.                     |
//|                     N can be 15, 21, 31, 41, 51, 61.             |
//| OUTPUT PARAMETERS:                                               |
//|     X           -   array[0..N-1] - array of quadrature nodes,   |
//|                     ordered in ascending order.                  |
//|     WKronrod    -   array[0..N-1] - Kronrod weights              |
//|     WGauss      -   array[0..N-1] - Gauss weights (interleaved   |
//|                     with zeros corresponding to extended Kronrod |
//|                     nodes).                                      |
//+------------------------------------------------------------------+
void CAlglib::GKQLegendreTbl(const int n,double &x[],double &wkronrod[],
                             double &wgauss[],double &eps)
  {
//--- initialization
   eps=0;
//--- function call
   CGaussKronrodQ::GKQLegendreTbl(n,x,wkronrod,wgauss,eps);
  }
//+------------------------------------------------------------------+
//| Integration of a smooth function F(x) on a finite interval [a,b].|
//| Fast-convergent algorithm based on a Gauss-Kronrod formula is    |
//| used. Result is calculated with accuracy close to the machine    |
//| precision.                                                       |
//| Algorithm works well only with smooth integrands. It may be used |
//| with continuous non-smooth integrands, but with less performance.|
//| It should never be used with integrands which have integrable    |
//| singularities at lower or upper limits - algorithm may crash.    |
//| Use AutoGKSingular in such cases.                                |
//| INPUT PARAMETERS:                                                |
//|     A, B    -   interval boundaries (A<B, A=B or A>B)            |
//| OUTPUT PARAMETERS                                                |
//|     State   -   structure which stores algorithm state           |
//| SEE ALSO                                                         |
//|     AutoGKSmoothW, AutoGKSingular, AutoGKResults.                |
//+------------------------------------------------------------------+
void CAlglib::AutoGKSmooth(const double a,const double b,
                           CAutoGKStateShell &state)
  {
   CAutoGK::AutoGKSmooth(a,b,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Integration of a smooth function F(x) on a finite interval [a,b].|
//| This subroutine is same as AutoGKSmooth(), but it guarantees that|
//| interval [a,b] is partitioned into subintervals which have width |
//| at most XWidth.                                                  |
//| Subroutine can be used when integrating nearly-constant function |
//| with narrow "bumps" (about XWidth wide). If "bumps" are too      |
//| narrow, AutoGKSmooth subroutine can overlook them.               |
//| INPUT PARAMETERS:                                                |
//|     A, B    -   interval boundaries (A<B, A=B or A>B)            |
//| OUTPUT PARAMETERS                                                |
//|     State   -   structure which stores algorithm state           |
//| SEE ALSO                                                         |
//|     AutoGKSmooth, AutoGKSingular, AutoGKResults.                 |
//+------------------------------------------------------------------+
void CAlglib::AutoGKSmoothW(const double a,const double b,
                            double xwidth,CAutoGKStateShell &state)
  {
   CAutoGK::AutoGKSmoothW(a,b,xwidth,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Integration on a finite interval [A,B].                          |
//| Integrand have integrable singularities at A/B.                  |
//| F(X) must diverge as "(x-A)^alpha" at A, as "(B-x)^beta" at B,   |
//| with known alpha/beta (alpha>-1, beta>-1). If alpha/beta are not |
//| known, estimates from below can be used (but these estimates     |
//| should be greater than -1 too).                                  |
//| One of alpha/beta variables (or even both alpha/beta) may be     |
//| equal to 0, which means than function F(x) is non-singular at    |
//| A/B. Anyway (singular at bounds or not), function F(x) is        |
//| supposed to be continuous on (A,B).                              |
//| Fast-convergent algorithm based on a Gauss-Kronrod formula is    |
//| used. Result is calculated with accuracy close to the machine    |
//| precision.                                                       |
//| INPUT PARAMETERS:                                                |
//|     A, B    -   interval boundaries (A<B, A=B or A>B)            |
//|     Alpha   -   power-law coefficient of the F(x) at A,          |
//|                 Alpha>-1                                         |
//|     Beta    -   power-law coefficient of the F(x) at B,          |
//|                 Beta>-1                                          |
//| OUTPUT PARAMETERS                                                |
//|     State   -   structure which stores algorithm state           |
//| SEE ALSO                                                         |
//|     AutoGKSmooth, AutoGKSmoothW, AutoGKResults.                  |
//+------------------------------------------------------------------+
void CAlglib::AutoGKSingular(const double a,const double b,const double alpha,
                             const double beta,CAutoGKStateShell &state)
  {
   CAutoGK::AutoGKSingular(a,b,alpha,beta,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function provides reverse communication interface           |
//| Reverse communication interface is not documented or recommended |
//| to use.                                                          |
//| See below for functions which provide better documented API      |
//+------------------------------------------------------------------+
bool CAlglib::AutoGKIteration(CAutoGKStateShell &state)
  {
   return(CAutoGK::AutoGKIteration(state.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| This function is used to launcn iterations of ODE solver         |
//| It accepts following parameters:                                 |
//|     diff    -   callback which calculates dy/dx for given y and x|
//|     obj     -   optional object which is passed to diff; can be  |
//|                 NULL                                             |
//+------------------------------------------------------------------+
void CAlglib::AutoGKIntegrate(CAutoGKStateShell &state,
                              CIntegrator1_Func &func,
                              CObject &obj)
  {
//--- cycle
   while(CAlglib::AutoGKIteration(state))
     {
      //--- check
      if(state.GetNeedF())
        {
         func.Int_Func(state.GetX(),state.GetXMinusA(),state.GetBMinusX(),state.GetInnerObj().m_f,obj);
         //--- next iteration
         continue;
        }
      Print("ALGLIB: unexpected error in 'autogksolve'");
      CAp::exception_happened=true;
      break;
     }
  }
//+------------------------------------------------------------------+
//| Adaptive integration results                                     |
//| Called after AutoGKIteration returned False.                     |
//| Input parameters:                                                |
//|     State   -   algorithm state (used by AutoGKIteration).       |
//| Output parameters:                                               |
//|     V       -   integral(f(x)dx,a,b)                             |
//|     Rep     -   optimization report (see AutoGKReport            |
//|                 description)                                     |
//+------------------------------------------------------------------+
void CAlglib::AutoGKResults(CAutoGKStateShell &state,double &v,
                            CAutoGKReportShell &rep)
  {
//--- initialization
   v=0;
//--- function call
   CAutoGK::AutoGKResults(state.GetInnerObj(),v,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function serializes data structure to string.               |
//| Important properties of s_out:                                   |
//|   * it contains alphanumeric characters, dots, underscores, minus|
//|     signs                                                        |
//|   * these symbols are grouped into words, which are separated by |
//|     spaces and Windows-style (CR+LF) newlines                    |
//|   * although  serializer  uses  spaces and CR+LF as separators,  |
//|     you can  replace any separator character by arbitrary        |
//|     combination of spaces, tabs, Windows or Unix newlines. It    |
//|     allows flexible reformatting  of the  string  in  case you   |
//|     want to include it into text or XML file. But you should not |
//|     insert separators into the middle of the "words" nor you     |
//|     should change case of letters.                               |
//|   * s_out can be freely moved between 32-bit and 64-bit systems, |
//|     little and big endian machines, and so on. You can serialize |
//|     structure on 32-bit machine and unserialize it on 64-bit one |
//|     (or vice versa), or serialize it on  SPARC  and  unserialize |
//|     on x86. You can also serialize it in C# version of ALGLIB and|
//|     unserialize in C++ one, and vice versa.                      |
//+------------------------------------------------------------------+
void CAlglib::IDWSerialize(CIDWModelShell &obj,string &s_out)
  {
//--- create a variable
   CSerializer s;
//--- serialization start
   s.Alloc_Start();
//--- function call
   CIDWInt::IDWAlloc(s,obj.GetInnerObj());
//--- serialization
   s.SStart_Str();
   CIDWInt::IDWSerialize(s,obj.GetInnerObj());
   s.Stop();
   s_out=s.Get_String();
  }
//+------------------------------------------------------------------+
//| This function unserializes data structure from string.           |
//+------------------------------------------------------------------+
void CAlglib::IDWUnserialize(string s_in,CIDWModelShell &obj)
  {
   CSerializer s;
   s.UStart_Str(s_in);
   CIDWInt::IDWUnserialize(s,obj.GetInnerObj());
   s.Stop();
  }
//+------------------------------------------------------------------+
//| This function creates buffer structure which can be used to      |
//| perform parallel IDW model evaluations (with one IDW model       |
//| instance being used from multiple threads, as long as different  |
//| threads use different instances of buffer).                      |
//| This buffer object can be used with IDWTsCalcBuf() function (here|
//| "ts" stands for "thread-safe", "buf" is a suffix which denotes   |
//| function which reuses previously allocated output space).        |
//| How to use it:                                                   |
//|   * create IDW model structure or load it from file              |
//|   * call IDWCreateCalcBuffer(), once per thread working with IDW |
//|     model (you should call this function only AFTER model        |
//|     initialization, see below for more information)              |
//|   * call IDWTsCalcBuf() from different threads, with each thread |
//|     working with its own copy of buffer object.                  |
//| INPUT PARAMETERS:                                                |
//|   S        -  IDW model                                          |
//| OUTPUT PARAMETERS:                                               |
//|   Buf      -  external buffer.                                   |
//| IMPORTANT: buffer object should be used only with IDW model      |
//|            object which was used to initialize buffer. Any       |
//|            attempt to use buffer with different object is        |
//|            dangerous - you may get memory violation error because|
//|            sizes of internal arrays do not fit to dimensions of  |
//|            the IDW structure.                                    |
//| IMPORTANT: you should call this function only for model which was|
//|            built with model builder (or unserialized from file). |
//|            Sizes of some internal structures are determined only |
//|            after model is built, so buffer object created before |
//|            model construction stage will be useless (and any     |
//|            attempt to use it will result in exception).          |
//+------------------------------------------------------------------+
void CAlglib::IDWCreateCalcBuffer(CIDWModelShell &s,CIDWCalcBuffer &buf)
  {
   CIDWInt::IDWCreateCalcBuffer(s.GetInnerObj(),buf);
  }
//+------------------------------------------------------------------+
//| This subroutine creates builder object used to generate IDW model|
//| from irregularly sampled (scattered) dataset. Multidimensional   |
//| scalar/vector-valued are supported.                              |
//| Builder object is used to fit model to data as follows:          |
//|   * builder object is created with idwbuildercreate() function   |
//|   * dataset is added with IDWBuilderSetPoints() function         |
//|   * one of the modern IDW algorithms is chosen with either:      |
//|      * IDWBuilderSetAlgoMSTAB() - Multilayer STABilized algorithm|
//|                                   (interpolation).               |
//| Alternatively, one of the textbook algorithms can be chosen (not |
//| recommended):                                                    |
//|   * IDWBuilderSetAlgoTextBookShepard()   -  textbook Shepard     |
//|                                             algorithm            |
//|   * IDWBuilderSetAlgoTextBookModShepard()-  textbook modified    |
//|                                             Shepard algorithm    |
//|   * finally, model construction is performed with IDWFit()       |
//|     function.                                                    |
//| INPUT PARAMETERS:                                                |
//|   NX       -  dimensionality of the argument, NX>=1              |
//|   NY       -  dimensionality of the function being modeled,      |
//|               NY>=1; NY=1 corresponds to classic scalar function,|
//|               NY>=1 corresponds to vector-valued function.       |
//| OUTPUT PARAMETERS:                                               |
//|   State    -  builder object                                     |
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderCreate(int nx,int ny,CIDWBuilder &state)
  {
   CIDWInt::IDWBuilderCreate(nx,ny,state);
  }
//+------------------------------------------------------------------+
//| This function changes number of layers used by IDW-MSTAB         |
//| algorithm.                                                       |
//| The more layers you have, the finer details can be reproduced    |
//| with IDW model. The less layers you have, the less memory and CPU|
//| time is consumed by the model.                                   |
//| Memory consumption grows linearly with layers count, running time|
//| grows sub-linearly.                                              |
//| The default number of layers is 16, which allows you to reproduce|
//| details at distance down to SRad/65536. You will rarely need to  |
//| change it.                                                       |
//| INPUT PARAMETERS:                                                |
//|   State    -  builder object                                     |
//|   NLayers  -  NLayers>=1, the number of layers used by the model.|
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetNLayers(CIDWBuilder &state,int nlayers)
  {
   CIDWInt::IDWBuilderSetNLayers(state,nlayers);
  }
//+------------------------------------------------------------------+
//| This function adds dataset to the builder object.                |
//| This function overrides results of the previous calls, i.e.      |
//| multiple calls of this function will result in only the last set |
//| being added.                                                     |
//| INPUT PARAMETERS:                                                |
//|   State    -  builder object                                     |
//|   XY       -  points, array[N, NX+NY]. One row corresponds to one |
//|               point in the dataset. First NX elements are        |
//|               coordinates, next NY elements are function values. |
//|               Array may be larger than specified, in this case   |
//|               only leading [N,NX+NY] elements will be used.      |
//|   N        -  number of points in the dataset, N>=0.             |
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetPoints(CIDWBuilder &state,CMatrixDouble &xy,int n)
  {
   CIDWInt::IDWBuilderSetPoints(state,xy,n);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetPoints(CIDWBuilder &state,CMatrixDouble &xy)
  {
   int n=CAp::Rows(xy);
//--- function call
   CIDWInt::IDWBuilderSetPoints(state,xy,n);
  }
//+------------------------------------------------------------------+
//| This function sets IDW model construction algorithm to the       |
//| Multilayer Stabilized IDW method (IDW-MSTAB), a latest           |
//| incarnation of the inverse distance weighting interpolation which|
//| fixes shortcomings of the original and modified Shepard's        |
//| variants.                                                        |
//| The distinctive features of IDW-MSTAB are:                       |
//|   1) exact interpolation is pursued (as opposed to fitting and   |
//|      noise suppression)                                          |
//|   2) improved robustness when compared with that of other        |
//|      algorithms:                                                 |
//|      * MSTAB shows almost no strange fitting artifacts like      |
//|        ripples and sharp spikes (unlike N-dimensional splines    |
//|        and HRBFs)                                                |
//|      * MSTAB does not return function values far from the        |
//|        interval spanned by the dataset; say, if all your points  |
//|        have |f|<=1, you can be sure that model value won't       |
//|        deviate too much from [-1,+1]                             |
//|   3) good model construction time competing with that of HRBFs   |
//|      and bicubic splines                                         |
//|   4) ability to work with any number of dimensions, starting     |
//|      from NX=1                                                   |
//| The drawbacks of IDW-MSTAB (and all IDW algorithms in general)   |
//| are:                                                             |
//|   1) dependence of the model evaluation time on the search radius|
//|   2) bad extrapolation properties, models built by this method   |
//|      are usually conservative in their predictions               |
//| Thus, IDW-MSTAB is a good "default" option if you want to perform|
//| scattered multidimensional interpolation. Although it has its    |
//| drawbacks, it is easy to use and robust, which makes it a good   |
//| first step.                                                      |
//| INPUT PARAMETERS:                                                |
//|   State    -  builder object                                     |
//|   SRad     -  initial search radius, SRad>0 is required. A model |
//|               value is obtained by "smart" averaging of the      |
//|               dataset points within search radius.               |
//| NOTE 1: IDW interpolation can correctly handle ANY dataset,      |
//|         including datasets with non-distinct points. In case     |
//|         non-distinct points are found, an average value for this |
//|         point will be calculated.                                |
//| NOTE 2: the memory requirements for model storage are            |
//|         O(NPoints*NLayers). The model construction needs twice   |
//|         as much memory as model storage.                         |
//| NOTE 3: by default 16 IDW layers are built which is enough for   |
//|         most cases. You can change this parameter with           |
//|         IDWBuilderSetNLayers() method. Larger values may be      |
//|         necessary if you need to reproduce extrafine details at  |
//|         distances smaller than SRad/65536. Smaller value  may    |
//|         be necessary if you have to save memory and computing    |
//|         time, and ready to sacrifice some model quality.         |
//| ALGORITHM DESCRIPTION:                                           |
//|   ALGLIB implementation of IDW is somewhat similar to the        |
//|   modified Shepard's method (one with search radius R) but       |
//|   overcomes several of its drawbacks, namely:                    |
//|      1) a tendency to show stepwise behavior for uniform datasets|
//|      2) a tendency to show terrible interpolation properties for |
//|         highly nonuniform datasets which often arise in          |
//|         geospatial tasks (function values are densely sampled    |
//|         across multiple separated "tracks")                      |
//| IDW-MSTAB method performs several passes over dataset and builds |
//| a sequence of progressively refined IDW models (layers), which   |
//| starts from one with largest search radius SRad and continues    |
//| to smaller search radii until required number of layers is built.|
//| Highest layers reproduce global behavior of the target function  |
//| at larger distances whilst lower layers reproduce fine details at|
//| smaller distances.                                               |
//| Each layer is an IDW model built with following modifications:   |
//|   * weights go to zero when distance approach to the current     |
//|     search radius                                                |
//|   * an additional regularizing term is added to the distance:    |
//|     w=1/(d^2+lambda)                                             |
//|   * an additional fictional term with unit weight and zero       |
//|     function value is added in order to promote continuity       |
//|     properties at the isolated and boundary points               |
//| By default, 16 layers is built, which is enough for most cases.  |
//| You can change this parameter with IDWBuilderSetNLayers() method.|
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetAlgoMSTAB(CIDWBuilder &state,double srad)
  {
   CIDWInt::IDWBuilderSetAlgoMSTAB(state,srad);
  }
//+------------------------------------------------------------------+
//| This function sets IDW model construction algorithm to the       |
//| textbook Shepard's algorithm with custom (user-specified) power  |
//| parameter.                                                       |
//| IMPORTANT: we do NOT recommend using textbook IDW algorithms     |
//|            because they have terrible interpolation properties.  |
//|            Use MSTAB in all cases.                               |
//| INPUT PARAMETERS:                                                |
//|   State    -  builder object                                     |
//|   P        -  power parameter, P>0; good value to start with is  |
//|               2.0                                                |
//| NOTE 1: IDW interpolation can correctly handle ANY dataset,      |
//|         including datasets with non-distinct points. In case     |
//|         non-distinct points are found, an average value for this |
//|         point will be calculated.                                |
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetAlgoTextBookShepard(CIDWBuilder &state,double p)
  {
   CIDWInt::IDWBuilderSetAlgoTextBookShepard(state,p);
  }
//+------------------------------------------------------------------+
//| This function sets IDW model construction algorithm to the       |
//| 'textbook' modified Shepard's algorithm with user-specified      |
//| search radius.                                                   |
//| IMPORTANT: we do NOT recommend using textbook IDW algorithms     |
//|            because they have terrible interpolation properties.  |
//|            Use MSTAB in all cases.                               |
//| INPUT PARAMETERS:                                                |
//|   State    -  builder object                                     |
//|   R        -  search radius                                      |
//| NOTE 1: IDW interpolation can correctly handle ANY dataset,      |
//|         including datasets with non-distinct points. In case     |
//|         non-distinct points are found, an average value for this |
//|         point will be calculated.                                |
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetAlgoTextBookModShepard(CIDWBuilder &state,double r)
  {
   CIDWInt::IDWBuilderSetAlgoTextBookModShepard(state,r);
  }
//+------------------------------------------------------------------+
//| This function sets prior term (model value at infinity) as       |
//| user-specified value.                                            |
//| INPUT PARAMETERS:                                                |
//|   S        -  spline builder                                     |
//|   V        -  value for user-defined prior                       |
//| NOTE: for vector-valued models all components of the prior are   |
//|       set to same user-specified value                           |
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetUserTerm(CIDWBuilder &state,double v)
  {
   CIDWInt::IDWBuilderSetUserTerm(state,v);
  }
//+------------------------------------------------------------------+
//| This function sets constant prior term (model value at infinity).|
//| Constant prior term is determined as mean value over dataset.    |
//| INPUT PARAMETERS:                                                |
//|   S        -  spline builder                                     |
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetConstTerm(CIDWBuilder &state)
  {
   CIDWInt::IDWBuilderSetConstTerm(state);
  }
//+------------------------------------------------------------------+
//| This function sets zero prior term (model value at infinity).    |
//| INPUT PARAMETERS:                                                |
//|   S        -  spline builder                                     |
//+------------------------------------------------------------------+
void CAlglib::IDWBuilderSetZeroTerm(CIDWBuilder &state)
  {
   CIDWInt::IDWBuilderSetZeroTerm(state);
  }
//+------------------------------------------------------------------+
//| IDW interpolation: scalar target, 1-dimensional argument         |
//| NOTE: this function modifies internal temporaries of the IDW     |
//|       model, thus IT IS NOT THREAD-SAFE! If you want to perform  |
//|       parallel model evaluation from the multiple threads, use   |
//|       IDWTsCalcBuf() with per-thread buffer object.              |
//| INPUT PARAMETERS:                                                |
//|   S        -  IDW interpolant built with IDW builder             |
//|   X0       -  argument value                                     |
//| Result:                                                          |
//|   IDW interpolant S(X0)                                          |
//+------------------------------------------------------------------+
double CAlglib::IDWCalc1(CIDWModelShell &s,double x0)
  {
   return(CIDWInt::IDWCalc1(s.GetInnerObj(),x0));
  }
//+------------------------------------------------------------------+
//| IDW interpolation: scalar target, 2-dimensional argument         |
//| NOTE: this function modifies internal temporaries of the IDW     |
//|       model, thus IT IS NOT THREAD-SAFE! If you want to perform  |
//|       parallel model evaluation from the multiple threads, use   |
//|       IDWTsCalcBuf() with per- thread buffer object.             |
//| INPUT PARAMETERS:                                                |
//|   S        -  IDW interpolant built with IDW builder             |
//|   X0, X1   -  argument value                                     |
//| Result:                                                          |
//|   IDW interpolant S(X0,X1)                                       |
//+------------------------------------------------------------------+
double CAlglib::IDWCalc2(CIDWModelShell &s,double x0,double x1)
  {
   return(CIDWInt::IDWCalc2(s.GetInnerObj(),x0,x1));
  }
//+------------------------------------------------------------------+
//| IDW interpolation: scalar target, 3-dimensional argument         |
//| NOTE: this function modifies internal temporaries of the IDW     |
//|       model, thus IT IS NOT THREAD-SAFE! If you want to perform  |
//|       parallel model evaluation from the multiple threads, use   |
//|      IDWTsCalcBuf() with per- thread buffer object.              |
//| INPUT PARAMETERS:                                                |
//|   S        -  IDW interpolant built with IDW builder             |
//|   X0,X1,X2 -  argument value                                     |
//| Result:                                                          |
//|   IDW interpolant S(X0,X1,X2)                                    |
//+------------------------------------------------------------------+
double CAlglib::IDWCalc3(CIDWModelShell &s,double x0,double x1,double x2)
  {
   return(CIDWInt::IDWCalc3(s.GetInnerObj(),x0,x1,x2));
  }
//+------------------------------------------------------------------+
//| This function calculates values of the IDW model at the given    |
//| point.                                                           |
//| This is general function which can be used for arbitrary NX      |
//| (dimension of the space of arguments) and NY (dimension of the   |
//| function itself). However when you have NY=1 you may find more   |
//| convenient to use IDWCalc1(), IDWCalc2() or IDWCalc3().          |
//| NOTE: this function modifies internal temporaries of the IDW     |
//|       model, thus IT IS NOT THREAD-SAFE! If you want to perform  |
//|       parallel model evaluation from the multiple threads, use   |
//|       IDWTsCalcBuf() with per-thread buffer object.              |
//| INPUT PARAMETERS:                                                |
//|   S        -  IDW model                                          |
//|   X        -  coordinates, array[NX]. X may have more than NX    |
//|               elements, in this case only leading NX will be used|
//| OUTPUT PARAMETERS:                                               |
//|   Y        -  function value, array[NY]. Y is out-parameter and  |
//|               will be reallocated after call to this function. In|
//|               case you want to reuse previously allocated Y, you |
//|               may use IDWCalcBuf(), which reallocates Y only when|
//|               it is too small.                                   |
//+------------------------------------------------------------------+
void CAlglib::IDWCalc(CIDWModelShell &s,CRowDouble &x,CRowDouble &y)
  {
   CIDWInt::IDWCalc(s.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| This function calculates values of the IDW model at the given    |
//| point.                                                           |
//| Same as IDWCalc(), but does not reallocate Y when in is large    |
//| enough to store function values.                                 |
//| NOTE: this function modifies internal temporaries of the IDW     |
//|       model, thus IT IS NOT THREAD-SAFE! If you want to perform  |
//|       parallel model evaluation from the multiple threads, use   |
//|       IDWTsCalcBuf() with per-thread buffer object.              |
//| INPUT PARAMETERS:                                                |
//|   S        -  IDW model                                          |
//|   X        -  coordinates, array[NX]. X may have more than NX    |
//|               elements, in this case only leading NX will be used|
//|   Y        -  possibly preallocated array                        |
//| OUTPUT PARAMETERS:                                               |
//|   Y        -  function value, array[NY]. Y is not reallocated    |
//|               when it is larger than NY.                         |
//+------------------------------------------------------------------+
void CAlglib::IDWCalcBuf(CIDWModelShell &s,CRowDouble &x,CRowDouble &y)
  {
   CIDWInt::IDWCalcBuf(s.GetInnerObj(),x,y);
  }
//+------------------------------------------------------------------+
//| This function calculates values of the IDW model at the given    |
//| point, using external buffer object (internal temporaries of IDW |
//| model are not modified).                                         |
//| This function allows to use same IDW model object in different   |
//| threads, assuming that different  threads use different instances|
//| of the buffer structure.                                         |
//| INPUT PARAMETERS:                                                |
//|   S        -  IDW model, may be shared between different threads |
//|   Buf      -  buffer object created for this particular instance |
//|               of IDW model with IDWCreateCalcBuffer().           |
//|   X        -  coordinates, array[NX]. X may have more than NX    |
//|               elements, in this case only leading NX will be used|
//|   Y        -  possibly preallocated array                        |
//| OUTPUT PARAMETERS:                                               |
//|   Y        -  function value, array[NY]. Y is not reallocated    |
//|               when it is larger than NY.                         |
//+------------------------------------------------------------------+
void CAlglib::IDWTsCalcBuf(CIDWModelShell &s,CIDWCalcBuffer &buf,
                           CRowDouble &x,CRowDouble &y)
  {
   CIDWInt::IDWTsCalcBuf(s.GetInnerObj(),buf,x,y);
  }
//+------------------------------------------------------------------+
//| This function fits IDW model to the dataset using current IDW    |
//| construction algorithm. A model being built and fitting report   |
//| are returned.                                                    |
//| INPUT PARAMETERS:                                                |
//|   State    -  builder object                                     |
//| OUTPUT PARAMETERS:                                               |
//|   Model    -  an IDW model built with current algorithm          |
//|   Rep      -  model fitting report, fields of this structure     |
//|               contain information about average fitting errors.  |
//| NOTE: although IDW-MSTAB algorithm is an interpolation method,   |
//|       i.e. it tries to fit the model exactly, it can handle      |
//|       datasets with non-distinct points which can not be fit     |
//|       exactly; in such cases least-squares fitting is performed. |
//+------------------------------------------------------------------+
void CAlglib::IDWFit(CIDWBuilder &state,CIDWModelShell &model,CIDWReport &rep)
  {
   CIDWInt::IDWFit(state,model.GetInnerObj(),rep);
  }
//+------------------------------------------------------------------+
//| Rational interpolation using barycentric formula                 |
//| F(t)=SUM(i=0,n-1,w[i]*f[i]/(t-x[i])) / SUM(i=0,n-1,w[i]/(t-x[i]))|
//| Input parameters:                                                |
//|     B   -   barycentric interpolant built with one of model      |
//|             building subroutines.                                |
//|     T   -   interpolation point                                  |
//| Result:                                                          |
//|     barycentric interpolant F(t)                                 |
//+------------------------------------------------------------------+
double CAlglib::BarycentricCalc(CBarycentricInterpolantShell &b,
                                const double t)
  {
   return(CRatInt::BarycentricCalc(b.GetInnerObj(),t));
  }
//+------------------------------------------------------------------+
//| Differentiation of barycentric interpolant: first derivative.    |
//| Algorithm used in this subroutine is very robust and should not  |
//| fail until provided with values too close to MaxRealNumber       |
//| (usually  MaxRealNumber/N or greater will overflow).             |
//| INPUT PARAMETERS:                                                |
//|     B   -   barycentric interpolant built with one of model      |
//|             building subroutines.                                |
//|     T   -   interpolation point                                  |
//| OUTPUT PARAMETERS:                                               |
//|     F   -   barycentric interpolant at T                         |
//|     DF  -   first derivative                                     |
//+------------------------------------------------------------------+
void CAlglib::BarycentricDiff1(CBarycentricInterpolantShell &b,
                               const double t,double &f,double &df)
  {
//--- initialization
   f=0;
   df=0;
//--- function call
   CRatInt::BarycentricDiff1(b.GetInnerObj(),t,f,df);
  }
//+------------------------------------------------------------------+
//| Differentiation of barycentric interpolant: first/second         |
//| derivatives.                                                     |
//| INPUT PARAMETERS:                                                |
//|     B   -   barycentric interpolant built with one of model      |
//|             building subroutines.                                |
//|     T   -   interpolation point                                  |
//| OUTPUT PARAMETERS:                                               |
//|     F   -   barycentric interpolant at T                         |
//|     DF  -   first derivative                                     |
//|     D2F -   second derivative                                    |
//| NOTE: this algorithm may fail due to overflow/underflor if used  |
//| on data whose values are close to MaxRealNumber or MinRealNumber.|
//| Use more robust BarycentricDiff1() subroutine in such cases.     |
//+------------------------------------------------------------------+
void CAlglib::BarycentricDiff2(CBarycentricInterpolantShell &b,
                               const double t,double &f,double &df,
                               double &d2f)
  {
//--- initialization
   f=0;
   df=0;
   d2f=0;
//--- function call
   CRatInt::BarycentricDiff2(b.GetInnerObj(),t,f,df,d2f);
  }
//+------------------------------------------------------------------+
//| This subroutine performs linear transformation of the argument.  |
//| INPUT PARAMETERS:                                                |
//|     B       -   rational interpolant in barycentric form         |
//|     CA, CB  -   transformation coefficients: x = CA*t + CB       |
//| OUTPUT PARAMETERS:                                               |
//|     B       -   transformed interpolant with X replaced by T     |
//+------------------------------------------------------------------+
void CAlglib::BarycentricLinTransX(CBarycentricInterpolantShell &b,
                                   const double ca,const double cb)
  {
   CRatInt::BarycentricLinTransX(b.GetInnerObj(),ca,cb);
  }
//+------------------------------------------------------------------+
//| This subroutine performs linear transformation of the barycentric|
//| interpolant.                                                     |
//| INPUT PARAMETERS:                                                |
//|     B       -   rational interpolant in barycentric form         |
//|     CA, CB  -   transformation coefficients: B2(x) = CA*B(x) + CB|
//| OUTPUT PARAMETERS:                                               |
//|     B       -   transformed interpolant                          |
//+------------------------------------------------------------------+
void CAlglib::BarycentricLinTransY(CBarycentricInterpolantShell &b,
                                   const double ca,const double cb)
  {
   CRatInt::BarycentricLinTransY(b.GetInnerObj(),ca,cb);
  }
//+------------------------------------------------------------------+
//| Extracts X/Y/W arrays from rational interpolant                  |
//| INPUT PARAMETERS:                                                |
//|     B   -   barycentric interpolant                              |
//| OUTPUT PARAMETERS:                                               |
//|     N   -   nodes count, N>0                                     |
//|     X   -   interpolation nodes, array[0..N-1]                   |
//|     F   -   function values, array[0..N-1]                       |
//|     W   -   barycentric weights, array[0..N-1]                   |
//+------------------------------------------------------------------+
void CAlglib::BarycentricUnpack(CBarycentricInterpolantShell &b,
                                int &n,double &x[],double &y[],
                                double &w[])
  {
//--- initialization
   n=0;
//--- function call
   CRatInt::BarycentricUnpack(b.GetInnerObj(),n,x,y,w);
  }
//+------------------------------------------------------------------+
//| Rational interpolant from X/Y/W arrays                           |
//| F(t)=SUM(i=0,n-1,w[i]*f[i]/(t-x[i])) / SUM(i=0,n-1,w[i]/(t-x[i]))|
//| INPUT PARAMETERS:                                                |
//|     X   -   interpolation nodes, array[0..N-1]                   |
//|     F   -   function values, array[0..N-1]                       |
//|     W   -   barycentric weights, array[0..N-1]                   |
//|     N   -   nodes count, N>0                                     |
//| OUTPUT PARAMETERS:                                               |
//|     B   -   barycentric interpolant built from (X, Y, W)         |
//+------------------------------------------------------------------+
void CAlglib::BarycentricBuildXYW(double &x[],double &y[],double &w[],
                                  const int n,CBarycentricInterpolantShell &b)
  {
   CRatInt::BarycentricBuildXYW(x,y,w,n,b.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Rational interpolant without poles                               |
//| The subroutine constructs the rational interpolating function    |
//| without real poles (see 'Barycentric rational interpolation with |
//| no poles and high rates of approximation', Michael S. Floater.   |
//| and Kai Hormann, for more information on this subject).          |
//| Input parameters:                                                |
//|     X   -   interpolation nodes, array[0..N-1].                  |
//|     Y   -   function values, array[0..N-1].                      |
//|     N   -   number of nodes, N>0.                                |
//|     D   -   order of the interpolation scheme, 0 <= D <= N-1.    |
//|             D<0 will cause an error.                             |
//|             D>=N it will be replaced with D=N-1.                 |
//|             if you don't know what D to choose, use small value  |
//|             about 3-5.                                           |
//| Output parameters:                                               |
//|     B   -   barycentric interpolant.                             |
//| Note:                                                            |
//|     this algorithm always succeeds and calculates the weights    |
//|     with close to machine precision.                             |
//+------------------------------------------------------------------+
void CAlglib::BarycentricBuildFloaterHormann(double &x[],double &y[],
                                             const int n,const int d,
                                             CBarycentricInterpolantShell &b)
  {
   CRatInt::BarycentricBuildFloaterHormann(x,y,n,d,b.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Conversion from barycentric representation to Chebyshev basis.   |
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     P   -   polynomial in barycentric form                       |
//|     A,B -   base interval for Chebyshev polynomials (see below)  |
//|             A<>B                                                 |
//| OUTPUT PARAMETERS                                                |
//|     T   -   coefficients of Chebyshev representation;            |
//|             P(x) = sum { T[i]*Ti(2*(x-A)/(B-A)-1), i=0..N-1 },   |
//|             where Ti - I-th Chebyshev polynomial.                |
//| NOTES:                                                           |
//|     barycentric interpolant passed as P may be either polynomial |
//|     obtained from polynomial interpolation/ fitting or rational  |
//|     function which is NOT polynomial. We can't distinguish       |
//|     between these two cases, and this algorithm just tries to    |
//|     work assuming that P IS a polynomial. If not, algorithm will |
//|     return results, but they won't have any meaning.             |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBar2Cheb(CBarycentricInterpolantShell &p,
                                 const double a,const double b,
                                 double &t[])
  {
   CPolInt::PolynomialBar2Cheb(p.GetInnerObj(),a,b,t);
  }
//+------------------------------------------------------------------+
//| Conversion from Chebyshev basis to barycentric representation.   |
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     T   -   coefficients of Chebyshev representation;            |
//|             P(x) = sum { T[i]*Ti(2*(x-A)/(B-A)-1), i=0..N },     |
//|             where Ti - I-th Chebyshev polynomial.                |
//|     N   -   number of coefficients:                              |
//|             * if given, only leading N elements of T are used    |
//|             * if not given, automatically determined from size   |
//|               of T                                               |
//|     A,B -   base interval for Chebyshev polynomials (see above)  |
//|             A<B                                                  |
//| OUTPUT PARAMETERS                                                |
//|     P   -   polynomial in barycentric form                       |
//+------------------------------------------------------------------+
void CAlglib::PolynomialCheb2Bar(double &t[],const int n,const double a,
                                 const double b,
                                 CBarycentricInterpolantShell &p)
  {
   CPolInt::PolynomialCheb2Bar(t,n,a,b,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Conversion from Chebyshev basis to barycentric representation.   |
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     T   -   coefficients of Chebyshev representation;            |
//|             P(x) = sum { T[i]*Ti(2*(x-A)/(B-A)-1), i=0..N },     |
//|             where Ti - I-th Chebyshev polynomial.                |
//|     N   -   number of coefficients:                              |
//|             * if given, only leading N elements of T are used    |
//|             * if not given, automatically determined from size   |
//|               of T                                               |
//|     A,B -   base interval for Chebyshev polynomials (see above)  |
//|             A<B                                                  |
//| OUTPUT PARAMETERS                                                |
//|     P   -   polynomial in barycentric form                       |
//+------------------------------------------------------------------+
void CAlglib::PolynomialCheb2Bar(double &t[],const double a,
                                 const double b,
                                 CBarycentricInterpolantShell &p)
  {
//--- initialization
   int n=CAp::Len(t);
//--- function call
   CPolInt::PolynomialCheb2Bar(t,n,a,b,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Conversion from barycentric representation to power basis.       |
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     P   -   polynomial in barycentric form                       |
//|     C   -   offset (see below); 0.0 is used as default value.    |
//|     S   -   scale (see below); 1.0 is used as default value.     |
//|             S<>0.                                                |
//| OUTPUT PARAMETERS                                                |
//|     A   -   coefficients,                                        |
//|             P(x) = sum { A[i]*((X-C)/S)^i, i=0..N-1 }            |
//|     N   -   number of coefficients (polynomial degree plus 1)    |
//| NOTES:                                                           |
//| 1.  this function accepts offset and scale, which can be set to  |
//|     improve numerical properties of polynomial. For example, if  |
//|     P was obtained as result of interpolation on [-1,+1], you can|
//|     set C=0 and S=1 and represent P as sum of 1, x, x^2, x^3 and |
//|     so on. In most cases you it is exactly what you need.        |
//|     However, if your interpolation model was built on [999,1001],|
//|     you will see significant growth of numerical errors when     |
//|     using {1, x, x^2, x^3} as basis. Representing P as sum of 1, |
//|     (x-1000), (x-1000)^2, (x-1000)^3 will be better option. Such |
//|     representation can be  obtained  by  using 1000.0 as offset  |
//|     C and 1.0 as scale S.                                        |
//| 2.  power basis is ill-conditioned and tricks described above    |
//|     can't solve this problem completely. This function  will     |
//|     return coefficients in any case, but for N>8 they will become|
//|     unreliable. However, N's less than 5 are pretty safe.        |
//| 3.  barycentric interpolant passed as P may be either polynomial |
//|     obtained from polynomial interpolation/ fitting or rational  |
//|     function which is NOT polynomial. We can't distinguish       |
//|     between these two cases, and this algorithm just tries to    |
//|     work assuming that P IS a polynomial. If not, algorithm will |
//|     return results, but they won't have any meaning.             |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBar2Pow(CBarycentricInterpolantShell &p,
                                const double c,const double s,
                                double &a[])
  {
   CPolInt::PolynomialBar2Pow(p.GetInnerObj(),c,s,a);
  }
//+------------------------------------------------------------------+
//| Conversion from barycentric representation to power basis.       |
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     P   -   polynomial in barycentric form                       |
//|     C   -   offset (see below); 0.0 is used as default value.    |
//|     S   -   scale (see below); 1.0 is used as default value.     |
//|             S<>0.                                                |
//| OUTPUT PARAMETERS                                                |
//|     A   -   coefficients,                                        |
//|             P(x) = sum { A[i]*((X-C)/S)^i, i=0..N-1 }            |
//|     N   -   number of coefficients (polynomial degree plus 1)    |
//| NOTES:                                                           |
//| 1.  this function accepts offset and scale, which can be set to  |
//|     improve numerical properties of polynomial. For example, if  |
//|     P was obtained as result of interpolation on [-1,+1], you can|
//|     set C=0 and S=1 and represent P as sum of 1, x, x^2, x^3 and |
//|     so on. In most cases you it is exactly what you need.        |
//|     However, if your interpolation model was built on [999,1001],|
//|     you will see significant growth of numerical errors when     |
//|     using {1, x, x^2, x^3} as basis. Representing P as sum of 1, |
//|     (x-1000), (x-1000)^2, (x-1000)^3 will be better option. Such |
//|     representation can be  obtained  by  using 1000.0 as offset  |
//|     C and 1.0 as scale S.                                        |
//| 2.  power basis is ill-conditioned and tricks described above    |
//|     can't solve this problem completely. This function  will     |
//|     return coefficients in any case, but for N>8 they will become|
//|     unreliable. However, N's less than 5 are pretty safe.        |
//| 3.  barycentric interpolant passed as P may be either polynomial |
//|     obtained from polynomial interpolation/ fitting or rational  |
//|     function which is NOT polynomial. We can't distinguish       |
//|     between these two cases, and this algorithm just tries to    |
//|     work assuming that P IS a polynomial. If not, algorithm will |
//|     return results, but they won't have any meaning.             |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBar2Pow(CBarycentricInterpolantShell &p,
                                double &a[])
  {
//--- initialization
   double c=0;
   double s=1;
//--- function call
   CPolInt::PolynomialBar2Pow(p.GetInnerObj(),c,s,a);
  }
//+------------------------------------------------------------------+
//| Conversion from power basis to barycentric representation.       |
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     A   -   coefficients, P(x)=sum { A[i]*((X-C)/S)^i, i=0..N-1 }|
//|     N   -   number of coefficients (polynomial degree plus 1)    |
//|             * if given, only leading N elements of A are used    |
//|             * if not given, automatically determined from size   |
//|               of A                                               |
//|     C   -   offset (see below); 0.0 is used as default value.    |
//|     S   -   scale (see below); 1.0 is used as default value.     |
//|             S<>0.                                                |
//| OUTPUT PARAMETERS                                                |
//|     P   -   polynomial in barycentric form                       |
//| NOTES:                                                           |
//| 1.  this function accepts offset and scale, which can be set to  |
//|     improve numerical properties of polynomial. For example, if  |
//|     you interpolate on [-1,+1], you can set C=0 and S=1 and      |
//|     convert from sum of 1, x, x^2, x^3 and so on. In most cases  |
//|     you it is exactly what you need.                             |
//|     However, if your interpolation model was built on [999,1001],|
//|     you will see significant growth of numerical errors when     |
//|     using {1, x, x^2, x^3} as input basis. Converting from sum   |
//|     of 1, (x-1000), (x-1000)^2, (x-1000)^3 will be better option |
//|     (you have to specify 1000.0 as offset C and 1.0 as scale S). |
//| 2.  power basis is ill-conditioned and tricks described above    |
//|     can't solve this problem completely. This function will      |
//|     return barycentric model in any case, but for N>8 accuracy   |
//|     well degrade. However, N's less than 5 are pretty safe.      |
//+------------------------------------------------------------------+
void CAlglib::PolynomialPow2Bar(double &a[],const int n,const double c,
                                const double s,CBarycentricInterpolantShell &p)
  {
   CPolInt::PolynomialPow2Bar(a,n,c,s,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Conversion from power basis to barycentric representation.       |
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     A   -   coefficients, P(x)=sum { A[i]*((X-C)/S)^i, i=0..N-1 }|
//|     N   -   number of coefficients (polynomial degree plus 1)    |
//|             * if given, only leading N elements of A are used    |
//|             * if not given, automatically determined from size   |
//|               of A                                               |
//|     C   -   offset (see below); 0.0 is used as default value.    |
//|     S   -   scale (see below); 1.0 is used as default value.     |
//|             S<>0.                                                |
//| OUTPUT PARAMETERS                                                |
//|     P   -   polynomial in barycentric form                       |
//| NOTES:                                                           |
//| 1.  this function accepts offset and scale, which can be set to  |
//|     improve numerical properties of polynomial. For example, if  |
//|     you interpolate on [-1,+1], you can set C=0 and S=1 and      |
//|     convert from sum of 1, x, x^2, x^3 and so on. In most cases  |
//|     you it is exactly what you need.                             |
//|     However, if your interpolation model was built on [999,1001],|
//|     you will see significant growth of numerical errors when     |
//|     using {1, x, x^2, x^3} as input basis. Converting from sum   |
//|     of 1, (x-1000), (x-1000)^2, (x-1000)^3 will be better option |
//|     (you have to specify 1000.0 as offset C and 1.0 as scale S). |
//| 2.  power basis is ill-conditioned and tricks described above    |
//|     can't solve this problem completely. This function will      |
//|     return barycentric model in any case, but for N>8 accuracy   |
//|     well degrade. However, N's less than 5 are pretty safe.      |
//+------------------------------------------------------------------+
void CAlglib::PolynomialPow2Bar(double &a[],CBarycentricInterpolantShell &p)
  {
//--- initialization
   int    n=CAp::Len(a);
   double c=0;
   double s=1;
//--- function call
   CPolInt::PolynomialPow2Bar(a,n,c,s,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Lagrange intepolant: generation of the model on the general grid.|
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     X   -   abscissas, array[0..N-1]                             |
//|     Y   -   function values, array[0..N-1]                       |
//|     N   -   number of points, N>=1                               |
//| OUTPUT PARAMETERS                                                |
//|     P   -   barycentric model which represents Lagrange          |
//|             interpolant (see ratint unit info and                |
//|             BarycentricCalc() description for more information). |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBuild(double &x[],double &y[],const int n,
                              CBarycentricInterpolantShell &p)
  {
   CPolInt::PolynomialBuild(x,y,n,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Lagrange intepolant: generation of the model on the general grid.|
//| This function has O(N^2) complexity.                             |
//| INPUT PARAMETERS:                                                |
//|     X   -   abscissas, array[0..N-1]                             |
//|     Y   -   function values, array[0..N-1]                       |
//|     N   -   number of points, N>=1                               |
//| OUTPUT PARAMETERS                                                |
//|     P   -   barycentric model which represents Lagrange          |
//|             interpolant (see ratint unit info and                |
//|             BarycentricCalc() description for more information). |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBuild(double &x[],double &y[],
                              CBarycentricInterpolantShell &p)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'polynomialbuild': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=CAp::Len(x);
//--- function call
   CPolInt::PolynomialBuild(x,y,n,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Lagrange intepolant: generation of the model on equidistant grid.|
//| This function has O(N) complexity.                               |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     Y   -   function values at the nodes, array[0..N-1]          |
//|     N   -   number of points, N>=1                               |
//|             for N=1 a constant model is constructed.             |
//| OUTPUT PARAMETERS                                                |
//|     P   -   barycentric model which represents Lagrange          |
//|             interpolant (see ratint unit info and                |
//|             BarycentricCalc() description for more information). |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBuildEqDist(const double a,const double b,
                                    double &y[],const int n,
                                    CBarycentricInterpolantShell &p)
  {
   CPolInt::PolynomialBuildEqDist(a,b,y,n,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Lagrange intepolant: generation of the model on equidistant grid.|
//| This function has O(N) complexity.                               |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     Y   -   function values at the nodes, array[0..N-1]          |
//|     N   -   number of points, N>=1                               |
//|             for N=1 a constant model is constructed.             |
//| OUTPUT PARAMETERS                                                |
//|     P   -   barycentric model which represents Lagrange          |
//|             interpolant (see ratint unit info and                |
//|             BarycentricCalc() description for more information). |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBuildEqDist(const double a,const double b,
                                    double &y[],
                                    CBarycentricInterpolantShell &p)
  {
//--- initialization
   int n=CAp::Len(y);
//--- function call
   CPolInt::PolynomialBuildEqDist(a,b,y,n,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Lagrange intepolant on Chebyshev grid (first kind).              |
//| This function has O(N) complexity.                               |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     Y   -   function values at the nodes, array[0..N-1],         |
//|             Y[I] = Y(0.5*(B+A) + 0.5*(B-A)*Cos(PI*(2*i+1)/(2*n)))|
//|     N   -   number of points, N>=1                               |
//|             for N=1 a constant model is constructed.             |
//| OUTPUT PARAMETERS                                                |
//|     P   -   barycentric model which represents Lagrange          |
//|             interpolant (see ratint unit info and                |
//|             BarycentricCalc() description for more information). |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBuildCheb1(const double a,const double b,
                                   double &y[],const int n,
                                   CBarycentricInterpolantShell &p)
  {
   CPolInt::PolynomialBuildCheb1(a,b,y,n,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Lagrange intepolant on Chebyshev grid (first kind).              |
//| This function has O(N) complexity.                               |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     Y   -   function values at the nodes, array[0..N-1],         |
//|             Y[I] = Y(0.5*(B+A) + 0.5*(B-A)*Cos(PI*(2*i+1)/(2*n)))|
//|     N   -   number of points, N>=1                               |
//|             for N=1 a constant model is constructed.             |
//| OUTPUT PARAMETERS                                                |
//|     P   -   barycentric model which represents Lagrange          |
//|             interpolant (see ratint unit info and                |
//|             BarycentricCalc() description for more information). |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBuildCheb1(const double a,const double b,
                                   double &y[],
                                   CBarycentricInterpolantShell &p)
  {
//--- initialization
   int n=CAp::Len(y);
//--- function call
   CPolInt::PolynomialBuildCheb1(a,b,y,n,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Lagrange intepolant on Chebyshev grid (second kind).             |
//| This function has O(N) complexity.                               |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     Y   -   function values at the nodes, array[0..N-1],         |
//|             Y[I] = Y(0.5*(B+A) + 0.5*(B-A)*Cos(PI*i/(n-1)))      |
//|     N   -   number of points, N>=1                               |
//|             for N=1 a constant model is constructed.             |
//| OUTPUT PARAMETERS                                                |
//|     P   -   barycentric model which represents Lagrange          |
//|             interpolant (see ratint unit info and                |
//|             BarycentricCalc() description for more information). |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBuildCheb2(const double a,const double b,
                                   double &y[],const int n,
                                   CBarycentricInterpolantShell &p)
  {
   CPolInt::PolynomialBuildCheb2(a,b,y,n,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Lagrange intepolant on Chebyshev grid (second kind).             |
//| This function has O(N) complexity.                               |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     Y   -   function values at the nodes, array[0..N-1],         |
//|             Y[I] = Y(0.5*(B+A) + 0.5*(B-A)*Cos(PI*i/(n-1)))      |
//|     N   -   number of points, N>=1                               |
//|             for N=1 a constant model is constructed.             |
//| OUTPUT PARAMETERS                                                |
//|     P   -   barycentric model which represents Lagrange          |
//|             interpolant (see ratint unit info and                |
//|             BarycentricCalc() description for more information). |
//+------------------------------------------------------------------+
void CAlglib::PolynomialBuildCheb2(const double a,const double b,
                                   double &y[],
                                   CBarycentricInterpolantShell &p)
  {
//--- initialization
   int n=CAp::Len(y);
//--- function call
   CPolInt::PolynomialBuildCheb2(a,b,y,n,p.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Fast equidistant polynomial interpolation function with O(N)     |
//| complexity                                                       |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     F   -   function values, array[0..N-1]                       |
//|     N   -   number of points on equidistant grid, N>=1           |
//|             for N=1 a constant model is constructed.             |
//|     T   -   position where P(x) is calculated                    |
//| RESULT                                                           |
//|     value of the Lagrange interpolant at T                       |
//| IMPORTANT                                                        |
//|     this function provides fast interface which is not           |
//|     overflow-safe nor it is very precise.                        |
//|     the best option is to use PolynomialBuildEqDist() or         |
//|     BarycentricCalc() subroutines unless you are pretty sure that|
//|     your data will not result in overflow.                       |
//+------------------------------------------------------------------+
double CAlglib::PolynomialCalcEqDist(const double a,const double b,
                                     double &f[],const int n,
                                     const double t)
  {
   return(CPolInt::PolynomialCalcEqDist(a,b,f,n,t));
  }
//+------------------------------------------------------------------+
//| Fast equidistant polynomial interpolation function with O(N)     |
//| complexity                                                       |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     F   -   function values, array[0..N-1]                       |
//|     N   -   number of points on equidistant grid, N>=1           |
//|             for N=1 a constant model is constructed.             |
//|     T   -   position where P(x) is calculated                    |
//| RESULT                                                           |
//|     value of the Lagrange interpolant at T                       |
//| IMPORTANT                                                        |
//|     this function provides fast interface which is not           |
//|     overflow-safe nor it is very precise.                        |
//|     the best option is to use PolynomialBuildEqDist() or         |
//|     BarycentricCalc() subroutines unless you are pretty sure that|
//|     your data will not result in overflow.                       |
//+------------------------------------------------------------------+
double CAlglib::PolynomialCalcEqDist(const double a,const double b,
                                     double &f[],const double t)
  {
//--- initialization
   int n=CAp::Len(f);
//--- return result
   return(CPolInt::PolynomialCalcEqDist(a,b,f,n,t));
  }
//+------------------------------------------------------------------+
//| Fast polynomial interpolation function on Chebyshev points (first|
//| kind) with O(N) complexity.                                      |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     F   -   function values, array[0..N-1]                       |
//|     N   -   number of points on Chebyshev grid (first kind),     |
//|             X[i] = 0.5*(B+A) + 0.5*(B-A)*Cos(PI*(2*i+1)/(2*n))   |
//|             for N=1 a constant model is constructed.             |
//|     T   -   position where P(x) is calculated                    |
//| RESULT                                                           |
//|     value of the Lagrange interpolant at T                       |
//| IMPORTANT                                                        |
//|     this function provides fast interface which is not           |
//|     overflow-safe nor it is very precise                         |
//|     the best option is to use  PolIntBuildCheb1() or             |
//|     BarycentricCalc() subroutines unless you are pretty sure that|
//|     your data will not result in overflow.                       |
//+------------------------------------------------------------------+
double CAlglib::PolynomialCalcCheb1(const double a,const double b,
                                    double &f[],const int n,
                                    const double t)
  {
   return(CPolInt::PolynomialCalcCheb1(a,b,f,n,t));
  }
//+------------------------------------------------------------------+
//| Fast polynomial interpolation function on Chebyshev points (first|
//| kind) with O(N) complexity.                                      |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     F   -   function values, array[0..N-1]                       |
//|     N   -   number of points on Chebyshev grid (first kind),     |
//|             X[i] = 0.5*(B+A) + 0.5*(B-A)*Cos(PI*(2*i+1)/(2*n))   |
//|             for N=1 a constant model is constructed.             |
//|     T   -   position where P(x) is calculated                    |
//| RESULT                                                           |
//|     value of the Lagrange interpolant at T                       |
//| IMPORTANT                                                        |
//|     this function provides fast interface which is not           |
//|     overflow-safe nor it is very precise                         |
//|     the best option is to use  PolIntBuildCheb1() or             |
//|     BarycentricCalc() subroutines unless you are pretty sure that|
//|     your data will not result in overflow.                       |
//+------------------------------------------------------------------+
double CAlglib::PolynomialCalcCheb1(const double a,const double b,
                                    double &f[],const double t)
  {
//--- initialization
   int n=CAp::Len(f);
//--- return result
   return(CPolInt::PolynomialCalcCheb1(a,b,f,n,t));
  }
//+------------------------------------------------------------------+
//| Fast polynomial interpolation function on Chebyshev points       |
//| (second kind) with O(N) complexity.                              |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     F   -   function values, array[0..N-1]                       |
//|     N   -   number of points on Chebyshev grid (second kind),    |
//|             X[i] = 0.5*(B+A) + 0.5*(B-A)*Cos(PI*i/(n-1))         |
//|             for N=1 a constant model is constructed.             |
//|     T   -   position where P(x) is calculated                    |
//| RESULT                                                           |
//|     value of the Lagrange interpolant at T                       |
//| IMPORTANT                                                        |
//|     this function provides fast interface which is not           |
//|     overflow-safe nor it is very precise.                        |
//|     the best option is to use PolIntBuildCheb2() or              |
//|     BarycentricCalc() subroutines unless you are pretty sure that|
//|     your data will not result in overflow.                       |
//+------------------------------------------------------------------+
double CAlglib::PolynomialCalcCheb2(const double a,const double b,
                                    double &f[],const int n,
                                    const double t)
  {
   return(CPolInt::PolynomialCalcCheb2(a,b,f,n,t));
  }
//+------------------------------------------------------------------+
//| Fast polynomial interpolation function on Chebyshev points       |
//| (second kind) with O(N) complexity.                              |
//| INPUT PARAMETERS:                                                |
//|     A   -   left boundary of [A,B]                               |
//|     B   -   right boundary of [A,B]                              |
//|     F   -   function values, array[0..N-1]                       |
//|     N   -   number of points on Chebyshev grid (second kind),    |
//|             X[i] = 0.5*(B+A) + 0.5*(B-A)*Cos(PI*i/(n-1))         |
//|             for N=1 a constant model is constructed.             |
//|     T   -   position where P(x) is calculated                    |
//| RESULT                                                           |
//|     value of the Lagrange interpolant at T                       |
//| IMPORTANT                                                        |
//|     this function provides fast interface which is not           |
//|     overflow-safe nor it is very precise.                        |
//|     the best option is to use PolIntBuildCheb2() or              |
//|     BarycentricCalc() subroutines unless you are pretty sure that|
//|     your data will not result in overflow.                       |
//+------------------------------------------------------------------+
double CAlglib::PolynomialCalcCheb2(const double a,const double b,
                                    double &f[],const double t)
  {
//--- initialization
   int n=CAp::Len(f);
//--- return result
   return(CPolInt::PolynomialCalcCheb2(a,b,f,n,t));
  }
//+------------------------------------------------------------------+
//| This subroutine builds linear spline interpolant                 |
//| INPUT PARAMETERS:                                                |
//|     X   -   spline nodes, array[0..N-1]                          |
//|     Y   -   function values, array[0..N-1]                       |
//|     N   -   points count (optional):                             |
//|             * N>=2                                               |
//|             * if given, only first N points are used to build    |
//|               spline                                             |
//|             * if not given, automatically detected from X/Y      |
//|               sizes (len(X) must be equal to len(Y))             |
//| OUTPUT PARAMETERS:                                               |
//|     C   -   spline interpolant                                   |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildLinear(double &x[],double &y[],const int n,
                                  CSpline1DInterpolantShell &c)
  {
   CSpline1D::Spline1DBuildLinear(x,y,n,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds linear spline interpolant                 |
//| INPUT PARAMETERS:                                                |
//|     X   -   spline nodes, array[0..N-1]                          |
//|     Y   -   function values, array[0..N-1]                       |
//|     N   -   points count (optional):                             |
//|             * N>=2                                               |
//|             * if given, only first N points are used to build    |
//|               spline                                             |
//|             * if not given, automatically detected from X/Y      |
//|               sizes (len(X) must be equal to len(Y))             |
//| OUTPUT PARAMETERS:                                               |
//|     C   -   spline interpolant                                   |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildLinear(double &x[],double &y[],
                                  CSpline1DInterpolantShell &c)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dbuildlinear': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=CAp::Len(x);
//--- function call
   CSpline1D::Spline1DBuildLinear(x,y,n,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds cubic spline interpolant.                 |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes, array[0..N-1].                 |
//|     Y           -   function values, array[0..N-1].              |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used to  |
//|                       build spline                               |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//| OUTPUT PARAMETERS:                                               |
//|     C           -   spline interpolant                           |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|             Y[first]).                                           |
//|     *  0, which  corresponds to the parabolically terminated     |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildCubic(double &x[],double &y[],const int n,
                                 const int boundltype,const double boundl,
                                 const int boundrtype,const double boundr,
                                 CSpline1DInterpolantShell &c)
  {
   CSpline1D::Spline1DBuildCubic(x,y,n,boundltype,boundl,boundrtype,boundr,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds cubic spline interpolant.                 |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes, array[0..N-1].                 |
//|     Y           -   function values, array[0..N-1].              |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used to  |
//|                       build spline                               |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//| OUTPUT PARAMETERS:                                               |
//|     C           -   spline interpolant                           |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|             Y[first]).                                           |
//|     *  0, which  corresponds to the parabolically terminated     |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildCubic(double &x[],double &y[],
                                 CSpline1DInterpolantShell &c)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dbuildcubic': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int    n=CAp::Len(x);
   int    boundltype=0;
   double boundl=0;
   int    boundrtype=0;
   double boundr=0;
//--- function call
   CSpline1D::Spline1DBuildCubic(x,y,n,boundltype,boundl,boundrtype,boundr,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at nodes x[], it calculates and returns table of |
//| function derivatives  d[] (calculated at the same nodes x[]).    |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed  by sequence of Spline1DDiff() calls, but it can be     |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes                                 |
//|     Y           -   function values                              |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used     |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//| OUTPUT PARAMETERS:                                               |
//|     D           -   derivative values at X[]                     |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array. Derivative values are correctly reordered on     |
//| return, so D[I] is always equal to S'(X[I]) independently of     |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|           Y[first]).                                             |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DGridDiffCubic(double &x[],double &y[],
                                    const int n,const int boundltype,
                                    const double boundl,const int boundrtype,
                                    const double boundr,double &d[])
  {
   CSpline1D::Spline1DGridDiffCubic(x,y,n,boundltype,boundl,boundrtype,boundr,d);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at nodes x[], it calculates and returns table of |
//| function derivatives  d[] (calculated at the same nodes x[]).    |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed  by sequence of Spline1DDiff() calls, but it can be     |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes                                 |
//|     Y           -   function values                              |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used     |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//| OUTPUT PARAMETERS:                                               |
//|     D           -   derivative values at X[]                     |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array. Derivative values are correctly reordered on     |
//| return, so D[I] is always equal to S'(X[I]) independently of     |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|           Y[first]).                                             |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DGridDiffCubic(double &x[],double &y[],
                                    double &d[])
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dgriddiffcubic': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int    n=CAp::Len(x);
   int    boundltype=0;
   double boundl=0;
   int    boundrtype=0;
   double boundr=0;
//--- function call
   CSpline1D::Spline1DGridDiffCubic(x,y,n,boundltype,boundl,boundrtype,boundr,d);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at nodes x[], it calculates and returns tables of|
//| first and second function derivatives d1[] and d2[] (calculated  |
//| at the same nodes x[]).                                          |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed  by sequence of Spline1DDiff() calls, but it can be     |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes                                 |
//|     Y           -   function values                              |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used     |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//| OUTPUT PARAMETERS:                                               |
//|     D1          -   S' values at X[]                             |
//|     D2          -   S'' values at X[]                            |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array. Derivative values are correctly reordered on     |
//| return, so D[I] is always equal to S'(X[I]) independently of     |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|           Y[first]).                                             |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point].                                    |
//| However, this subroutine doesn't require you to specify equal    |
//| values for the first and last points - it automatically forces   |
//| them to be equal by copying Y[first_point] (corresponds to the   |
//| leftmost, minimal X[]) to Y[last_point]. However it is           |
//| recommended to pass consistent values of Y[], i.e. to make       |
//| Y[first_point]=Y[last_point].                                    |
//+------------------------------------------------------------------+
void CAlglib::Spline1DGridDiff2Cubic(double &x[],double &y[],
                                     const int n,const int boundltype,
                                     const double boundl,const int boundrtype,
                                     const double boundr,double &d1[],
                                     double &d2[])
  {
   CSpline1D::Spline1DGridDiff2Cubic(x,y,n,boundltype,boundl,boundrtype,boundr,d1,d2);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at nodes x[], it calculates and returns tables of|
//| first and second function derivatives d1[] and d2[] (calculated  |
//| at the same nodes x[]).                                          |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed  by sequence of Spline1DDiff() calls, but it can be     |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes                                 |
//|     Y           -   function values                              |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used     |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//| OUTPUT PARAMETERS:                                               |
//|     D1          -   S' values at X[]                             |
//|     D2          -   S'' values at X[]                            |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array. Derivative values are correctly reordered on     |
//| return, so D[I] is always equal to S'(X[I]) independently of     |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|           Y[first]).                                             |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point].                                    |
//| However, this subroutine doesn't require you to specify equal    |
//| values for the first and last points - it automatically forces   |
//| them to be equal by copying Y[first_point] (corresponds to the   |
//| leftmost, minimal X[]) to Y[last_point]. However it is           |
//| recommended to pass consistent values of Y[], i.e. to make       |
//| Y[first_point]=Y[last_point].                                    |
//+------------------------------------------------------------------+
void CAlglib::Spline1DGridDiff2Cubic(double &x[],double &y[],
                                     double &d1[],double &d2[])
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dgriddiff2cubic': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int    n=CAp::Len(x);
   int    boundltype=0;
   double boundl=0;
   int    boundrtype=0;
   double boundr=0;
//--- function call
   CSpline1D::Spline1DGridDiff2Cubic(x,y,n,boundltype,boundl,boundrtype,boundr,d1,d2);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at old nodes x[]  and new nodes  x2[], it        |
//| calculates and returns table of function values y2[] (calculated |
//| at x2[]).                                                        |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed  by sequence of Spline1DDiff() calls, but it can be     |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   old spline nodes                             |
//|     Y           -   function values                              |
//|     X2           -  new spline nodes                             |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points from X/Y are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//|     N2          -   new points count:                            |
//|                     * N2>=2                                      |
//|                     * if given, only first N2 points from X2 are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X2 size                                    |
//| OUTPUT PARAMETERS:                                               |
//|     F2          -   function values at X2[]                      |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller  may pass       |
//| unsorted array. Function  values  are correctly reordered on     |
//| return, so F2[I] is always equal to S(X2[I]) independently of    |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|           Y[first]).                                             |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DConvCubic(double &x[],double &y[],const int n,
                                const int boundltype,const double boundl,
                                const int boundrtype,const double boundr,
                                double &x2[],int n2,double &y2[])
  {
   CSpline1D::Spline1DConvCubic(x,y,n,boundltype,boundl,boundrtype,boundr,x2,n2,y2);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at old nodes x[]  and new nodes  x2[], it        |
//| calculates and returns table of function values y2[] (calculated |
//| at x2[]).                                                        |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed  by sequence of Spline1DDiff() calls, but it can be     |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   old spline nodes                             |
//|     Y           -   function values                              |
//|     X2           -  new spline nodes                             |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points from X/Y are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//|     N2          -   new points count:                            |
//|                     * N2>=2                                      |
//|                     * if given, only first N2 points from X2 are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X2 size                                    |
//| OUTPUT PARAMETERS:                                               |
//|     F2          -   function values at X2[]                      |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller  may pass       |
//| unsorted array. Function  values  are correctly reordered on     |
//| return, so F2[I] is always equal to S(X2[I]) independently of    |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|           Y[first]).                                             |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DConvCubic(double &x[],double &y[],
                                double &x2[],double &y2[])
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dconvcubic': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int    n=CAp::Len(x);
   int    boundltype=0;
   double boundl=0;
   int    boundrtype=0;
   double boundr=0;
   int    n2=CAp::Len(x2);
//--- function call
   CSpline1D::Spline1DConvCubic(x,y,n,boundltype,boundl,boundrtype,boundr,x2,n2,y2);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at old nodes x[] and new nodes x2[], it          |
//| calculates and returns table of function values y2[] and         |
//| derivatives d2[] (calculated at x2[]).                           |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed by sequence of Spline1DDiff() calls, but it can be      |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   old spline nodes                             |
//|     Y           -   function values                              |
//|     X2           -  new spline nodes                             |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points from X/Y are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//|     N2          -   new points count:                            |
//|                     * N2>=2                                      |
//|                     * if given, only first N2 points from X2 are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X2 size                                    |
//| OUTPUT PARAMETERS:                                               |
//|     F2          -   function values at X2[]                      |
//|     D2          -   first derivatives at X2[]                    |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array. Function  values  are correctly reordered on     |
//| return, so F2[I] is always equal to S(X2[I]) independently of    |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|             Y[first]).                                           |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DConvDiffCubic(double &x[],double &y[],
                                    const int n,const int boundltype,
                                    const double boundl,const int boundrtype,
                                    const double boundr,double &x2[],
                                    int n2,double &y2[],double &d2[])
  {
   CSpline1D::Spline1DConvDiffCubic(x,y,n,boundltype,boundl,boundrtype,boundr,x2,n2,y2,d2);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at old nodes x[] and new nodes x2[], it          |
//| calculates and returns table of function values y2[] and         |
//| derivatives d2[] (calculated at x2[]).                           |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed by sequence of Spline1DDiff() calls, but it can be      |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   old spline nodes                             |
//|     Y           -   function values                              |
//|     X2           -  new spline nodes                             |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points from X/Y are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//|     N2          -   new points count:                            |
//|                     * N2>=2                                      |
//|                     * if given, only first N2 points from X2 are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X2 size                                    |
//| OUTPUT PARAMETERS:                                               |
//|     F2          -   function values at X2[]                      |
//|     D2          -   first derivatives at X2[]                    |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array. Function  values  are correctly reordered on     |
//| return, so F2[I] is always equal to S(X2[I]) independently of    |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|             Y[first]).                                           |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DConvDiffCubic(double &x[],double &y[],
                                    double &x2[],double &y2[],
                                    double &d2[])
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dconvdiffcubic': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int    n=CAp::Len(x);
   int    boundltype=0;
   double boundl=0;
   int    boundrtype=0;
   double boundr=0;
   int    n2=CAp::Len(x2);
//--- function call
   CSpline1D::Spline1DConvDiffCubic(x,y,n,boundltype,boundl,boundrtype,boundr,x2,n2,y2,d2);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at old nodes x[] and new nodes  x2[], it         |
//| calculates and returns table of function values y2[], first and  |
//| second derivatives d2[] and dd2[] (calculated at x2[]).          |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed  by sequence of Spline1DDiff() calls, but it can be     |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   old spline nodes                             |
//|     Y           -   function values                              |
//|     X2           -  new spline nodes                             |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points from X/Y are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//|     N2          -   new points count:                            |
//|                     * N2>=2                                      |
//|                     * if given, only first N2 points from X2 are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X2 size                                    |
//| OUTPUT PARAMETERS:                                               |
//|     F2          -   function values at X2[]                      |
//|     D2          -   first derivatives at X2[]                    |
//|     DD2         -   second derivatives at X2[]                   |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array. Function values are correctly reordered on       |
//| return, so F2[I] is always equal to S(X2[I]) independently of    |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|             Y[first]).                                           |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DConvDiff2Cubic(double &x[],double &y[],
                                     const int n,const int boundltype,
                                     const double boundl,const int boundrtype,
                                     const double boundr,double &x2[],
                                     const int n2,double &y2[],
                                     double &d2[],double &dd2[])
  {
   CSpline1D::Spline1DConvDiff2Cubic(x,y,n,boundltype,boundl,boundrtype,boundr,x2,n2,y2,d2,dd2);
  }
//+------------------------------------------------------------------+
//| This function solves following problem: given table y[] of       |
//| function values at old nodes x[] and new nodes  x2[], it         |
//| calculates and returns table of function values y2[], first and  |
//| second derivatives d2[] and dd2[] (calculated at x2[]).          |
//| This function yields same result as Spline1DBuildCubic() call    |
//| followed  by sequence of Spline1DDiff() calls, but it can be     |
//| several times faster when called for ordered X[] and X2[].       |
//| INPUT PARAMETERS:                                                |
//|     X           -   old spline nodes                             |
//|     Y           -   function values                              |
//|     X2           -  new spline nodes                             |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points from X/Y are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundLType  -   boundary condition type for the left boundary|
//|     BoundL      -   left boundary condition (first or second     |
//|                     derivative, depending on the BoundLType)     |
//|     BoundRType  -   boundary condition type for the right        |
//|                     boundary                                     |
//|     BoundR      -   right boundary condition (first or second    |
//|                     derivative, depending on the BoundRType)     |
//|     N2          -   new points count:                            |
//|                     * N2>=2                                      |
//|                     * if given, only first N2 points from X2 are |
//|                       used                                       |
//|                     * if not given, automatically detected from  |
//|                       X2 size                                    |
//| OUTPUT PARAMETERS:                                               |
//|     F2          -   function values at X2[]                      |
//|     D2          -   first derivatives at X2[]                    |
//|     DD2         -   second derivatives at X2[]                   |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array. Function values are correctly reordered on       |
//| return, so F2[I] is always equal to S(X2[I]) independently of    |
//| points order.                                                    |
//| SETTING BOUNDARY VALUES:                                         |
//| The BoundLType/BoundRType parameters can have the following      |
//| values:                                                          |
//|     * -1, which corresonds to the periodic (cyclic) boundary     |
//|           conditions. In this case:                              |
//|           * both BoundLType and BoundRType must be equal to -1.  |
//|           * BoundL/BoundR are ignored                            |
//|           * Y[last] is ignored (it is assumed to be equal to     |
//|             Y[first]).                                           |
//|     *  0, which corresponds to the parabolically terminated      |
//|           spline (BoundL and/or BoundR are ignored).             |
//|     *  1, which corresponds to the first derivative boundary     |
//|           condition                                              |
//|     *  2, which corresponds to the second derivative boundary    |
//|           condition                                              |
//|     *  by default, BoundType=0 is used                           |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DConvDiff2Cubic(double &x[],double &y[],
                                     double &x2[],double &y2[],
                                     double &d2[],double &dd2[])
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dconvdiff2cubic': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=CAp::Len(x);
   int boundltype=0;
   double boundl=0;
   int    boundrtype=0;
   double boundr=0;
   int    n2=CAp::Len(x2);
//--- function call
   CSpline1D::Spline1DConvDiff2Cubic(x,y,n,boundltype,boundl,boundrtype,boundr,x2,n2,y2,d2,dd2);
  }
//+------------------------------------------------------------------+
//| This subroutine builds Catmull-Rom spline interpolant.           |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes, array[0..N-1].                 |
//|     Y           -   function values, array[0..N-1].              |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used to  |
//|                       build spline                               |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundType   -   boundary condition type:                     |
//|                     * -1 for periodic boundary condition         |
//|                     *  0 for parabolically terminated spline     |
//|                        (default)                                 |
//|     Tension     -   tension parameter:                           |
//|                     * tension=0   corresponds to classic         |
//|                       Catmull-Rom spline (default)               |
//|                     * 0<tension<1 corresponds to more general    |
//|                       form - cardinal spline                     |
//| OUTPUT PARAMETERS:                                               |
//|     C           -   spline interpolant                           |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildCatmullRom(double &x[],double &y[],
                                      const int n,const int boundtype,
                                      const double tension,
                                      CSpline1DInterpolantShell &c)
  {
   CSpline1D::Spline1DBuildCatmullRom(x,y,n,boundtype,tension,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds Catmull-Rom spline interpolant.           |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes, array[0..N-1].                 |
//|     Y           -   function values, array[0..N-1].              |
//| OPTIONAL PARAMETERS:                                             |
//|     N           -   points count:                                |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used to  |
//|                       build spline                               |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//|     BoundType   -   boundary condition type:                     |
//|                     * -1 for periodic boundary condition         |
//|                     *  0 for parabolically terminated spline     |
//|                        (default)                                 |
//|     Tension     -   tension parameter:                           |
//|                     * tension=0   corresponds to classic         |
//|                       Catmull-Rom spline (default)               |
//|                     * 0<tension<1 corresponds to more general    |
//|                       form - cardinal spline                     |
//| OUTPUT PARAMETERS:                                               |
//|     C           -   spline interpolant                           |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//| PROBLEMS WITH PERIODIC BOUNDARY CONDITIONS:                      |
//| Problems with periodic boundary conditions have                  |
//| Y[first_point]=Y[last_point]. However, this subroutine doesn't   |
//| require you to specify equal values for the first and last       |
//| points - it automatically forces them to be equal by copying     |
//| Y[first_point] (corresponds to the leftmost, minimal X[]) to     |
//| Y[last_point]. However it is recommended to pass consistent      |
//| values of Y[], i.e. to make Y[first_point]=Y[last_point].        |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildCatmullRom(double &x[],double &y[],
                                      CSpline1DInterpolantShell &c)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dbuildcatmullrom': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int    n=CAp::Len(x);
   int    boundtype=0;
   double tension=0;
//--- function call
   CSpline1D::Spline1DBuildCatmullRom(x,y,n,boundtype,tension,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds Hermite spline interpolant.               |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes, array[0..N-1]                  |
//|     Y           -   function values, array[0..N-1]               |
//|     D           -   derivatives, array[0..N-1]                   |
//|     N           -   points count (optional):                     |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used to  |
//|                       build spline                               |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//| OUTPUT PARAMETERS:                                               |
//|     C           -   spline interpolant.                          |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildHermite(double &x[],double &y[],double &d[],
                                   const int n,CSpline1DInterpolantShell &c)
  {
   CSpline1D::Spline1DBuildHermite(x,y,d,n,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds Hermite spline interpolant.               |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes, array[0..N-1]                  |
//|     Y           -   function values, array[0..N-1]               |
//|     D           -   derivatives, array[0..N-1]                   |
//|     N           -   points count (optional):                     |
//|                     * N>=2                                       |
//|                     * if given, only first N points are used to  |
//|                       build spline                               |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//| OUTPUT PARAMETERS:                                               |
//|     C           -   spline interpolant.                          |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildHermite(double &x[],double &y[],double &d[],
                                   CSpline1DInterpolantShell &c)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)) || (CAp::Len(x)!=CAp::Len(d)))
     {
      Print("Error while calling 'spline1dbuildhermite': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=CAp::Len(x);
//--- function call
   CSpline1D::Spline1DBuildHermite(x,y,d,n,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds Akima spline interpolant                  |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes, array[0..N-1]                  |
//|     Y           -   function values, array[0..N-1]               |
//|     N           -   points count (optional):                     |
//|                     * N>=5                                       |
//|                     * if given, only first N points are used to  |
//|                       build spline                               |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//| OUTPUT PARAMETERS:                                               |
//|     C           -   spline interpolant                           |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildAkima(double &x[],double &y[],const int n,
                                 CSpline1DInterpolantShell &c)
  {
   CSpline1D::Spline1DBuildAkima(x,y,n,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine builds Akima spline interpolant                  |
//| INPUT PARAMETERS:                                                |
//|     X           -   spline nodes, array[0..N-1]                  |
//|     Y           -   function values, array[0..N-1]               |
//|     N           -   points count (optional):                     |
//|                     * N>=5                                       |
//|                     * if given, only first N points are used to  |
//|                       build spline                               |
//|                     * if not given, automatically detected from  |
//|                       X/Y sizes (len(X) must be equal to len(Y)) |
//| OUTPUT PARAMETERS:                                               |
//|     C           -   spline interpolant                           |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildAkima(double &x[],double &y[],
                                 CSpline1DInterpolantShell &c)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dbuildakima': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=CAp::Len(x);
//--- function call
   CSpline1D::Spline1DBuildAkima(x,y,n,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This subroutine calculates the value of the spline at the given  |
//| point X.                                                         |
//| INPUT PARAMETERS:                                                |
//|     C   -   spline interpolant                                   |
//|     X   -   point                                                |
//| Result:                                                          |
//|     S(x)                                                         |
//+------------------------------------------------------------------+
double CAlglib::Spline1DCalc(CSpline1DInterpolantShell &c,const double x)
  {
   return(CSpline1D::Spline1DCalc(c.GetInnerObj(),x));
  }
//+------------------------------------------------------------------+
//| This subroutine differentiates the spline.                       |
//| INPUT PARAMETERS:                                                |
//|     C   -   spline interpolant.                                  |
//|     X   -   point                                                |
//| Result:                                                          |
//|     S   -   S(x)                                                 |
//|     DS  -   S'(x)                                                |
//|     D2S -   S''(x)                                               |
//+------------------------------------------------------------------+
void CAlglib::Spline1DDiff(CSpline1DInterpolantShell &c,const double x,
                           double &s,double &ds,double &d2s)
  {
//--- initialization
   s=0;
   ds=0;
   d2s=0;
//--- function call
   CSpline1D::Spline1DDiff(c.GetInnerObj(),x,s,ds,d2s);
  }
//+------------------------------------------------------------------+
//| This subroutine unpacks the spline into the coefficients table.  |
//| INPUT PARAMETERS:                                                |
//|     C   -   spline interpolant.                                  |
//|     X   -   point                                                |
//| Result:                                                          |
//|     Tbl -   coefficients table, unpacked format, array[0..N-2,   |
//|             0..5].                                               |
//|             For I = 0...N-2:                                     |
//|                 Tbl[I,0] = X[i]                                  |
//|                 Tbl[I,1] = X[i+1]                                |
//|                 Tbl[I,2] = C0                                    |
//|                 Tbl[I,3] = C1                                    |
//|                 Tbl[I,4] = C2                                    |
//|                 Tbl[I,5] = C3                                    |
//|             On [x[i], x[i+1]] spline is equals to:               |
//|                 S(x) = C0 + C1*t + C2*t^2 + C3*t^3               |
//|                 t = x-x[i]                                       |
//+------------------------------------------------------------------+
void CAlglib::Spline1DUnpack(CSpline1DInterpolantShell &c,int &n,
                             CMatrixDouble &tbl)
  {
//--- initialization
   n=0;
//--- function call
   CSpline1D::Spline1DUnpack(c.GetInnerObj(),n,tbl);
  }
//+------------------------------------------------------------------+
//| This subroutine performs linear transformation of the spline     |
//| argument.                                                        |
//| INPUT PARAMETERS:                                                |
//|     C   -   spline interpolant.                                  |
//|     A, B-   transformation coefficients: x = A*t + B             |
//| Result:                                                          |
//|     C   -   transformed spline                                   |
//+------------------------------------------------------------------+
void CAlglib::Spline1DLinTransX(CSpline1DInterpolantShell &c,
                                const double a,const double b)
  {
   CSpline1D::Spline1DLinTransX(c.GetInnerObj(),a,b);
  }
//+------------------------------------------------------------------+
//| This subroutine performs linear transformation of the spline.    |
//| INPUT PARAMETERS:                                                |
//|     C   -   spline interpolant.                                  |
//|     A,B-   transformation coefficients: S2(x)=A*S(x) + B         |
//| Result:                                                          |
//|     C   -   transformed spline                                   |
//+------------------------------------------------------------------+
void CAlglib::Spline1DLinTransY(CSpline1DInterpolantShell &c,
                                const double a,const double b)
  {
   CSpline1D::Spline1DLinTransY(c.GetInnerObj(),a,b);
  }
//+------------------------------------------------------------------+
//| This subroutine integrates the spline.                           |
//| INPUT PARAMETERS:                                                |
//|     C   -   spline interpolant.                                  |
//|     X   -   right bound of the integration interval [a, x],      |
//|             here 'a' denotes min(x[])                            |
//| Result:                                                          |
//|     integral(S(t)dt,a,x)                                         |
//+------------------------------------------------------------------+
double CAlglib::Spline1DIntegrate(CSpline1DInterpolantShell &c,
                                  const double x)
  {
   return(CSpline1D::Spline1DIntegrate(c.GetInnerObj(),x));
  }
//+------------------------------------------------------------------+
//| Fitting by smoothing (penalized) cubic spline.                   |
//| This function approximates N scattered points (some of X[] may   |
//| be equal to each other) by cubic spline with M  nodes  at        |
//| equidistant  grid  spanning interval [min(x,xc),max(x,xc)].      |
//| The problem is regularized by adding nonlinearity penalty to     |
//| usual  least squares penalty function:                           |
//|               MERIT_FUNC = F_LS + F_NL                           |
//| where F_LS is a least squares error  term,  and  F_NL  is  a     |
//| nonlinearity penalty which is roughly proportional to            |
//| LambdaNS*integral{ S''(x)^2*dx }. Algorithm applies automatic    |
//| renormalization of F_NL  which  makes  penalty term roughly      |
//| invariant to scaling of X[] and changes in M.                    |
//| This function is a new edition  of  penalized  regression  spline|
//| fitting, a fast and compact one which needs much less resources  |
//| that  its  previous version: just O(maxMN) memory and            |
//| O(maxMN*log(maxMN)) time.                                        |
//| NOTE: it is OK to run this function with both M<<N and M>>N; say,|
//|       it is possible to process 100 points with 1000-node spline.|
//| INPUT PARAMETERS:                                                |
//|   X        -  points, array[0..N-1].                             |
//|   Y        -  function values, array[0..N-1].                    |
//|   N        -  number of points (optional):                       |
//|               * N>0                                              |
//|               * if given, only first N elements of X/Y are       |
//|                 processed                                        |
//|               * if not given, automatically determined from      |
//|                 lengths                                          |
//|   M        -  number of basis functions ( = number_of_nodes),    |
//|               M>=4.                                              |
//|   LambdaNS -  LambdaNS>=0, regularization  constant  passed by   |
//|               user. It penalizes nonlinearity in the regression  |
//|               spline. Possible values to start from are 0.00001, |
//|               0.1, 1                                             |
//| OUTPUT PARAMETERS:                                               |
//|   S        -  spline interpolant.                                |
//|   Rep      -  Following fields are set:                          |
//|               * RMSError      rms error on the (X,Y).            |
//|               * AvgError      average error on the (X,Y).        |
//|               * AvgRelError   average relative error on the      |
//|                               non-zero Y                         |
//|               * MaxError      maximum error                      |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFit(double &x[],double &y[],int n,int m,
                          double lambdans,CSpline1DInterpolantShell &s,
                          CSpline1DFitReportShell &rep)
  {
   CSpline1D::Spline1DFit(x,y,n,m,lambdans,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFit(double &x[],double &y[],int m,double lambdans,
                          CSpline1DInterpolantShell &s,CSpline1DFitReportShell &rep)
  {
//--- check
   if(!CAp::Assert(CAp::Len(x)==CAp::Len(y),"Error while calling 'Spline1DFit': looks like one of arguments has wrong size"))
      return;
//--- initialization
   int n=CAp::Len(x);
//--- function call
   CSpline1D::Spline1DFit(x,y,n,m,lambdans,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| This function builds monotone cubic Hermite interpolant. This    |
//| interpolant is monotonic in [x(0),x(n-1)] and is constant outside|
//| of this interval.                                                |
//| In  case  y[]  form  non-monotonic  sequence,  interpolant  is   |
//| piecewise monotonic.  Say, for x=(0,1,2,3,4)  and  y=(0,1,2,1,0) |
//| interpolant  will monotonically grow at [0..2] and monotonically |
//| decrease at [2..4].                                              |
//| INPUT PARAMETERS:                                                |
//|   X        -  spline nodes, array[0..N-1]. Subroutine            |
//|               automatically sorts points, so caller may pass     |
//|               unsorted array.                                    |
//|   Y        -  function values, array[0..N-1]                     |
//|   N        -  the number of points(N>=2).                        |
//| OUTPUT PARAMETERS:                                               |
//|   C        -  spline interpolant.                                |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildMonotone(double &x[],double &y[],int n,CSpline1DInterpolantShell &c)
  {
   CSpline1D::Spline1DBuildMonotone(x,y,n,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildMonotone(CRowDouble &x,CRowDouble &y,CSpline1DInterpolantShell &c)
  {
//--- create variables
   double X[];
   double Y[];
//--- initialization
   x.ToArray(X);
   y.ToArray(Y);
//--- function call
   Spline1DBuildMonotone(X,Y,c);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DBuildMonotone(double &x[],double &y[],CSpline1DInterpolantShell &c)
  {
//--- check
   if(!CAp::Assert(CAp::Len(x)==CAp::Len(y),"Error while calling 'Spline1DBuildMonotone': looks like one of arguments has wrong size"))
      return;
//--- initialization
   int n=CAp::Len(x);
//--- function call
   CSpline1D::Spline1DBuildMonotone(x,y,n,c.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Fitting by polynomials in barycentric form. This function        |
//| provides simple unterface for unconstrained unweighted fitting.  |
//| See PolynomialFitWC() if you need constrained fitting.           |
//| Task is linear, so linear least squares solver is used.          |
//| Complexity of this computational scheme is O(N*M^2), mostly      |
//| dominated by least squares solver                                |
//| SEE ALSO:                                                        |
//|     PolynomialFitWC()                                            |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     N   -   number of points, N>0                                |
//|             * if given, only leading N elements of X/Y are used  |
//|             * if not given, automatically determined from sizes  |
//|               of X/Y                                             |
//|     M   -   number of basis functions (= polynomial_degree + 1), |
//|             M>=1                                                 |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearW() subroutine:         |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|     P   -   interpolant in barycentric form.                     |
//|     Rep -   report, same format as in LSFitLinearW() subroutine. |
//|             Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| NOTES:                                                           |
//|     you can convert P from barycentric form to the power or      |
//|     Chebyshev basis with PolynomialBar2Pow() or                  |
//|     PolynomialBar2Cheb() functions from POLINT subpackage.       |
//+------------------------------------------------------------------+
void CAlglib::PolynomialFit(double &x[],double &y[],const int n,
                            const int m,int &info,CBarycentricInterpolantShell &p,
                            CPolynomialFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::PolynomialFit(x,y,n,m,info,p.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Fitting by polynomials in barycentric form. This function        |
//| provides simple unterface for unconstrained unweighted fitting.  |
//| See PolynomialFitWC() if you need constrained fitting.           |
//| Task is linear, so linear least squares solver is used.          |
//| Complexity of this computational scheme is O(N*M^2), mostly      |
//| dominated by least squares solver                                |
//| SEE ALSO:                                                        |
//|     PolynomialFitWC()                                            |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     N   -   number of points, N>0                                |
//|             * if given, only leading N elements of X/Y are used  |
//|             * if not given, automatically determined from sizes  |
//|               of X/Y                                             |
//|     M   -   number of basis functions (= polynomial_degree + 1), |
//|             M>=1                                                 |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearW() subroutine:         |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|     P   -   interpolant in barycentric form.                     |
//|     Rep -   report, same format as in LSFitLinearW() subroutine. |
//|             Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| NOTES:                                                           |
//|     you can convert P from barycentric form to the power or      |
//|     Chebyshev basis with PolynomialBar2Pow() or                  |
//|     PolynomialBar2Cheb() functions from POLINT subpackage.       |
//+------------------------------------------------------------------+
void CAlglib::PolynomialFit(double &x[],double &y[],const int m,
                            int &info,CBarycentricInterpolantShell &p,
                            CPolynomialFitReportShell &rep)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'polynomialfit': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(x);
//--- function call
   CLSFit::PolynomialFit(x,y,n,m,info,p.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted  fitting by polynomials in barycentric form, with       |
//| constraints  on function values or first derivatives.            |
//| Small regularizing term is used when solving constrained tasks   |
//| (to improve stability).                                          |
//| Task is linear, so linear least squares solver is used.          |
//| Complexity of this computational scheme is O(N*M^2), mostly      |
//| dominated by least squares solver                                |
//| SEE ALSO:                                                        |
//|     PolynomialFit()                                              |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if you|
//|             don't want to solve weighted task.                   |
//|     N   -   number of points, N>0.                               |
//|             * if given, only leading N elements of X/Y/W are used|
//|             * if not given, automatically determined from sizes  |
//|               of X/Y/W                                           |
//|     XC  -   points where polynomial values/derivatives are       |
//|             constrained, array[0..K-1].                          |
//|     YC  -   values of constraints, array[0..K-1]                 |
//|     DC  -   array[0..K-1], types of constraints:                 |
//|             * DC[i]=0   means that P(XC[i])=YC[i]                |
//|             * DC[i]=1   means that P'(XC[i])=YC[i]               |
//|             SEE BELOW FOR IMPORTANT INFORMATION ON CONSTRAINTS   |
//|     K   -   number of constraints, 0<=K<M.                       |
//|             K=0 means no constraints (XC/YC/DC are not used in   |
//|             such cases)                                          |
//|     M   -   number of basis functions (= polynomial_degree + 1), |
//|             M>=1                                                 |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearW() subroutine:         |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|     P   -   interpolant in barycentric form.                     |
//|     Rep -   report, same format as in LSFitLinearW() subroutine. |
//|             Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| NOTES:                                                           |
//|     you can convert P from barycentric form to the power or      |
//|     Chebyshev basis with PolynomialBar2Pow() or                  |
//|     PolynomialBar2Cheb() functions from POLINT subpackage.       |
//| SETTING CONSTRAINTS - DANGERS AND OPPORTUNITIES:                 |
//| Setting constraints can lead to undesired results, like          |
//| ill-conditioned behavior, or inconsistency being detected.       |
//| From the other side, it allows us to improve quality of the fit. |
//| Here we summarize our experience with constrained regression     |
//| splines:                                                         |
//| * even simple constraints can be inconsistent, see Wikipedia     |
//|   article on this subject:                                       |
//|   http://en.wikipedia.org/wiki/Birkhoff_interpolation            |
//| * the greater is M (given fixed constraints), the more chances   |
//|   that constraints will be consistent                            |
//| * in the general case, consistency of constraints is NOT         |
//|   GUARANTEED.                                                    |
//| * in the one special cases, however, we can guarantee            |
//|   consistency. This case  is:  M>1  and constraints on the       |
//|   function values (NOT DERIVATIVES)                              |
//| Our final recommendation is to use constraints WHEN AND ONLY when|
//| you can't solve your task without them. Anything beyond special  |
//| cases given above is not guaranteed and may result in            |
//| inconsistency.                                                   |
//+------------------------------------------------------------------+
void CAlglib::PolynomialFitWC(double &x[],double &y[],double &w[],
                              const int n,double &xc[],double &yc[],
                              int &dc[],const int k,const int m,
                              int &info,CBarycentricInterpolantShell &p,
                              CPolynomialFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::PolynomialFitWC(x,y,w,n,xc,yc,dc,k,m,info,p.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted  fitting by polynomials in barycentric form, with       |
//| constraints  on function values or first derivatives.            |
//| Small regularizing term is used when solving constrained tasks   |
//| (to improve stability).                                          |
//| Task is linear, so linear least squares solver is used.          |
//| Complexity of this computational scheme is O(N*M^2), mostly      |
//| dominated by least squares solver                                |
//| SEE ALSO:                                                        |
//|     PolynomialFit()                                              |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if you|
//|             don't want to solve weighted task.                   |
//|     N   -   number of points, N>0.                               |
//|             * if given, only leading N elements of X/Y/W are used|
//|             * if not given, automatically determined from sizes  |
//|               of X/Y/W                                           |
//|     XC  -   points where polynomial values/derivatives are       |
//|             constrained, array[0..K-1].                          |
//|     YC  -   values of constraints, array[0..K-1]                 |
//|     DC  -   array[0..K-1], types of constraints:                 |
//|             * DC[i]=0   means that P(XC[i])=YC[i]                |
//|             * DC[i]=1   means that P'(XC[i])=YC[i]               |
//|             SEE BELOW FOR IMPORTANT INFORMATION ON CONSTRAINTS   |
//|     K   -   number of constraints, 0<=K<M.                       |
//|             K=0 means no constraints (XC/YC/DC are not used in   |
//|             such cases)                                          |
//|     M   -   number of basis functions (= polynomial_degree + 1), |
//|             M>=1                                                 |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearW() subroutine:         |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|     P   -   interpolant in barycentric form.                     |
//|     Rep -   report, same format as in LSFitLinearW() subroutine. |
//|             Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| NOTES:                                                           |
//|     you can convert P from barycentric form to the power or      |
//|     Chebyshev basis with PolynomialBar2Pow() or                  |
//|     PolynomialBar2Cheb() functions from POLINT subpackage.       |
//| SETTING CONSTRAINTS - DANGERS AND OPPORTUNITIES:                 |
//| Setting constraints can lead to undesired results, like          |
//| ill-conditioned behavior, or inconsistency being detected.       |
//| From the other side, it allows us to improve quality of the fit. |
//| Here we summarize our experience with constrained regression     |
//| splines:                                                         |
//| * even simple constraints can be inconsistent, see Wikipedia     |
//|   article on this subject:                                       |
//|   http://en.wikipedia.org/wiki/Birkhoff_interpolation            |
//| * the greater is M (given fixed constraints), the more chances   |
//|   that constraints will be consistent                            |
//| * in the general case, consistency of constraints is NOT         |
//|   GUARANTEED.                                                    |
//| * in the one special cases, however, we can guarantee            |
//|   consistency. This case  is:  M>1  and constraints on the       |
//|   function values (NOT DERIVATIVES)                              |
//| Our final recommendation is to use constraints WHEN AND ONLY when|
//| you can't solve your task without them. Anything beyond special  |
//| cases given above is not guaranteed and may result in            |
//| inconsistency.                                                   |
//+------------------------------------------------------------------+
void CAlglib::PolynomialFitWC(double &x[],double &y[],double &w[],
                              double &xc[],double &yc[],int &dc[],
                              const int m,int &info,CBarycentricInterpolantShell &p,
                              CPolynomialFitReportShell &rep)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)) || (CAp::Len(x)!=CAp::Len(w)))
     {
      Print("Error while calling 'polynomialfitwc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
   if((CAp::Len(xc)!=CAp::Len(yc)) || (CAp::Len(xc)!=CAp::Len(dc)))
     {
      Print("Error while calling 'polynomialfitwc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(x);
   int k=CAp::Len(xc);
//--- function call
   CLSFit::PolynomialFitWC(x,y,w,n,xc,yc,dc,k,m,info,p.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weghted rational least squares fitting using Floater-Hormann     |
//| rational functions with optimal D chosen from [0,9], with        |
//| constraints and individual weights.                              |
//| Equidistant grid with M node on [min(x),max(x)] is used to build |
//| basis functions. Different values of D are tried, optimal D      |
//| (least WEIGHTED root mean square error) is chosen. Task is       |
//| linear, so linear least squares solver is used. Complexity of    |
//| this computational scheme is O(N*M^2) (mostly dominated by the   |
//| least squares solver).                                           |
//| SEE ALSO                                                         |
//|*BarycentricFitFloaterHormann(), "lightweight" fitting without  |
//|   invididual weights and constraints.                            |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if    |
//|             you don't want to solve weighted task.               |
//|     N   -   number of points, N>0.                               |
//|     XC  -   points where function values/derivatives are         |
//|             constrained, array[0..K-1].                          |
//|     YC  -   values of constraints, array[0..K-1]                 |
//|     DC  -   array[0..K-1], types of constraints:                 |
//|             * DC[i]=0   means that S(XC[i])=YC[i]                |
//|             * DC[i]=1   means that S'(XC[i])=YC[i]               |
//|             SEE BELOW FOR IMPORTANT INFORMATION ON CONSTRAINTS   |
//|     K   -   number of constraints, 0<=K<M.                       |
//|             K=0 means no constraints (XC/YC/DC are not used in   |
//|             such cases)                                          |
//|     M   -   number of basis functions ( = number_of_nodes),      |
//|             M>=2.                                                |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearWC() subroutine.        |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|                         -1 means another errors in parameters    |
//|                            passed (N<=0, for example)            |
//|     B   -   barycentric interpolant.                             |
//|     Rep -   report, same format as in LSFitLinearWC() subroutine.|
//|             Following fields are set:                            |
//|             * DBest         best value of the D parameter        |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroutine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| SETTING CONSTRAINTS - DANGERS AND OPPORTUNITIES:                 |
//| Setting constraints can lead to undesired results, like          |
//| ill-conditioned behavior, or inconsistency being detected. From  |
//| the other side, it allows us to improve quality of the fit. Here |
//| we summarize our experience with constrained barycentric         |
//| interpolants:                                                    |
//| * excessive constraints can be inconsistent. Floater-Hormann     |
//|   basis functions aren't as flexible as splines (although they   |
//|   are very smooth).                                              |
//| * the more evenly constraints are spread across [min(x),max(x)], |
//|   the more chances that they will be consistent                  |
//| * the greater is M (given  fixed  constraints), the more chances |
//|   that constraints will be consistent                            |
//| * in the general case, consistency of constraints IS NOT         |
//|   GUARANTEED.                                                    |
//| * in the several special cases, however, we CAN guarantee        |
//|   consistency.                                                   |
//| * one of this cases is constraints on the function VALUES at the |
//|   interval boundaries. Note that consustency of the constraints  |
//|   on the function DERIVATIVES is NOT guaranteed (you can use in  |
//|   such cases cubic splines which are more flexible).             |
//| * another special case is ONE constraint on the function value   |
//|   (OR, but not AND, derivative) anywhere in the interval         |
//| Our final recommendation is to use constraints WHEN AND ONLY     |
//| WHEN you can't solve your task without them. Anything beyond     |
//| special cases given above is not guaranteed and may result in    |
//| inconsistency.                                                   |
//+------------------------------------------------------------------+
void CAlglib::BarycentricFitFloaterHormannWC(double &x[],double &y[],
                                             double &w[],const int n,
                                             double &xc[],double &yc[],
                                             int &dc[],const int k,
                                             const int m,int &info,
                                             CBarycentricInterpolantShell &b,
                                             CBarycentricFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::BarycentricFitFloaterHormannWC(x,y,w,n,xc,yc,dc,k,m,info,b.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Rational least squares fitting using Floater-Hormann rational    |
//| functions with optimal D chosen from [0,9].                      |
//| Equidistant grid with M node on [min(x),max(x)] is used to build |
//| basis functions. Different values of D are tried, optimal D      |
//| (least root mean square error) is chosen.  Task is linear, so    |
//| linear least squares solver is used. Complexity of this          |
//| computational scheme is O(N*M^2) (mostly dominated by the least  |
//| squares solver).                                                 |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     N   -   number of points, N>0.                               |
//|     M   -   number of basis functions ( = number_of_nodes), M>=2.|
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearWC() subroutine.        |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|     B   -   barycentric interpolant.                             |
//|     Rep -   report, same format as in LSFitLinearWC() subroutine.|
//|             Following fields are set:                            |
//|             * DBest         best value of the D parameter        |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//+------------------------------------------------------------------+
void CAlglib::BarycentricFitFloaterHormann(double &x[],double &y[],
                                           const int n,const int m,
                                           int &info,CBarycentricInterpolantShell &b,
                                           CBarycentricFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::BarycentricFitFloaterHormann(x,y,n,m,info,b.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Rational least squares fitting using Floater-Hormann rational    |
//| functions with optimal D chosen from [0,9].                      |
//| Equidistant grid with M node on [min(x),max(x)] is used to build |
//| basis functions. Different values of D are tried, optimal D      |
//| (least root mean square error) is chosen. Task is linear, so     |
//| linear least squares solver is used. Complexity of this          |
//| computational scheme is O(N*M^2) (mostly dominated by the least  |
//| squares solver).                                                 |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     N   -   number of points, N>0.                               |
//|     M   -   number of basis functions ( = number_of_nodes), M>=2.|
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearWC() subroutine.        |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|     B   -   barycentric interpolant.                             |
//|     Rep -   report, same format as in LSFitLinearWC() subroutine.|
//|             Following fields are set:                            |
//|             * DBest         best value of the D parameter        |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitPenalized(double &x[],double &y[],const int n,
                                   const int m,const double rho,int &info,
                                   CSpline1DInterpolantShell &s,
                                   CSpline1DFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CIntComp::Spline1DFitPenalized(x,y,n,m,rho,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Rational least squares fitting using Floater-Hormann rational    |
//| functions with optimal D chosen from [0,9].                      |
//| Equidistant grid with M node on [min(x),max(x)] is used to build |
//| basis functions. Different values of D are tried, optimal D      |
//| (least root mean square error) is chosen. Task is linear, so     |
//| linear least squares solver is used. Complexity of this          |
//| computational scheme is O(N*M^2) (mostly dominated by the least  |
//| squares solver).                                                 |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     N   -   number of points, N>0.                               |
//|     M   -   number of basis functions ( = number_of_nodes), M>=2.|
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearWC() subroutine.        |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|     B   -   barycentric interpolant.                             |
//|     Rep -   report, same format as in LSFitLinearWC() subroutine.|
//|             Following fields are set:                            |
//|             * DBest         best value of the D parameter        |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitPenalized(double &x[],double &y[],const int m,
                                   const double rho,int &info,
                                   CSpline1DInterpolantShell &s,
                                   CSpline1DFitReportShell &rep)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dfitpenalized': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(x);
//--- function call
   CIntComp::Spline1DFitPenalized(x,y,n,m,rho,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted fitting by penalized cubic spline.                      |
//| Equidistant grid with M nodes on [min(x,xc),max(x,xc)] is used to|
//| build basis functions. Basis functions are cubic splines with    |
//| natural boundary conditions. Problem is regularized by adding    |
//| non-linearity penalty to the usual least squares penalty         |
//| function:                                                        |
//|     S(x) = arg min { LS + P }, where                             |
//|     LS   = SUM { w[i]^2*(y[i] - S(x[i]))^2 } - least squares     |
//|            penalty                                               |
//|     P    = C*10^rho*integral{ S''(x)^2*dx } - non-linearity      |
//|            penalty                                               |
//|     rho  - tunable constant given by user                        |
//|     C    - automatically determined scale parameter,             |
//|            makes penalty invariant with respect to scaling of X, |
//|            Y, W.                                                 |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if    |
//|             you don't want to solve weighted problem.            |
//|     N   -   number of points (optional):                         |
//|             * N>0                                                |
//|             * if given, only first N elements of X/Y/W are       |
//|               processed                                          |
//|             * if not given, automatically determined from X/Y/W  |
//|               sizes                                              |
//|     M   -   number of basis functions ( = number_of_nodes), M>=4.|
//|     Rho -   regularization constant passed by user. It penalizes |
//|             nonlinearity in the regression spline. It is         |
//|             logarithmically scaled, i.e. actual value of         |
//|             regularization constant is calculated as 10^Rho. It  |
//|             is automatically scaled so that:                     |
//|             * Rho=2.0 corresponds to moderate amount of          |
//|               nonlinearity                                       |
//|             * generally, it should be somewhere in the           |
//|               [-8.0,+8.0]                                        |
//|             If you do not want to penalize nonlineary,           |
//|             pass small Rho. Values as low as -15 should work.    |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearWC() subroutine.        |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                            or Cholesky decomposition; problem    |
//|                            may be too ill-conditioned (very      |
//|                            rare)                                 |
//|     S   -   spline interpolant.                                  |
//|     Rep -   Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| NOTE 1: additional nodes are added to the spline outside of the  |
//| fitting interval to force linearity when x<min(x,xc) or          |
//| x>max(x,xc). It is done for consistency - we penalize            |
//| non-linearity at [min(x,xc),max(x,xc)], so it is natural to      |
//| force linearity outside of this interval.                        |
//| NOTE 2: function automatically sorts points, so caller may pass  |
//| unsorted array.                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitPenalizedW(double &x[],double &y[],
                                    double &w[],const int n,
                                    const int m,const double rho,
                                    int &info,CSpline1DInterpolantShell &s,
                                    CSpline1DFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CIntComp::Spline1DFitPenalizedW(x,y,w,n,m,rho,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted fitting by penalized cubic spline.                      |
//| Equidistant grid with M nodes on [min(x,xc),max(x,xc)] is used to|
//| build basis functions. Basis functions are cubic splines with    |
//| natural boundary conditions. Problem is regularized by adding    |
//| non-linearity penalty to the usual least squares penalty         |
//| function:                                                        |
//|     S(x) = arg min { LS + P }, where                             |
//|     LS   = SUM { w[i]^2*(y[i] - S(x[i]))^2 } - least squares     |
//|            penalty                                               |
//|     P    = C*10^rho*integral{ S''(x)^2*dx } - non-linearity      |
//|            penalty                                               |
//|     rho  - tunable constant given by user                        |
//|     C    - automatically determined scale parameter,             |
//|            makes penalty invariant with respect to scaling of X, |
//|            Y, W.                                                 |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if    |
//|             you don't want to solve weighted problem.            |
//|     N   -   number of points (optional):                         |
//|             * N>0                                                |
//|             * if given, only first N elements of X/Y/W are       |
//|               processed                                          |
//|             * if not given, automatically determined from X/Y/W  |
//|               sizes                                              |
//|     M   -   number of basis functions ( = number_of_nodes), M>=4.|
//|     Rho -   regularization constant passed by user. It penalizes |
//|             nonlinearity in the regression spline. It is         |
//|             logarithmically scaled, i.e. actual value of         |
//|             regularization constant is calculated as 10^Rho. It  |
//|             is automatically scaled so that:                     |
//|             * Rho=2.0 corresponds to moderate amount of          |
//|               nonlinearity                                       |
//|             * generally, it should be somewhere in the           |
//|               [-8.0,+8.0]                                        |
//|             If you do not want to penalize nonlineary,           |
//|             pass small Rho. Values as low as -15 should work.    |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearWC() subroutine.        |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                            or Cholesky decomposition; problem    |
//|                            may be too ill-conditioned (very      |
//|                            rare)                                 |
//|     S   -   spline interpolant.                                  |
//|     Rep -   Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| NOTE 1: additional nodes are added to the spline outside of the  |
//| fitting interval to force linearity when x<min(x,xc) or          |
//| x>max(x,xc). It is done for consistency - we penalize            |
//| non-linearity at [min(x,xc),max(x,xc)], so it is natural to      |
//| force linearity outside of this interval.                        |
//| NOTE 2: function automatically sorts points, so caller may pass  |
//| unsorted array.                                                  |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitPenalizedW(double &x[],double &y[],
                                    double &w[],const int m,
                                    const double rho,int &info,
                                    CSpline1DInterpolantShell &s,
                                    CSpline1DFitReportShell &rep)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)) || (CAp::Len(x)!=CAp::Len(w)))
     {
      Print("Error while calling 'spline1dfitpenalizedw': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(x);
//--- function call
   CIntComp::Spline1DFitPenalizedW(x,y,w,n,m,rho,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted fitting by cubic spline, with constraints on function   |
//| values or derivatives.                                           |
//| Equidistant grid with M-2 nodes on [min(x,xc),max(x,xc)] is used |
//| to build basis functions. Basis functions are cubic splines with |
//| continuous second derivatives and non-fixed first derivatives at |
//| interval ends. Small regularizing term is used when solving      |
//| constrained tasks (to improve stability).                        |
//| Task is linear, so linear least squares solver is used.          |
//| Complexity of this computational scheme is O(N*M^2), mostly      |
//| dominated by least squares solver                                |
//| SEE ALSO                                                         |
//|     Spline1DFitHermiteWC()  -   fitting by Hermite splines (more |
//|                                 flexible, less smooth)           |
//|     Spline1DFitCubic()     -  "lightweight" fitting by cubic   |
//|                                 splines, without invididual      |
//|                                 weights and constraints          |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if you|
//|             don't want to solve weighted task.                   |
//|     N   -   number of points (optional):                         |
//|             * N>0                                                |
//|             * if given, only first N elements of X/Y/W are       |
//|               processed                                          |
//|             * if not given, automatically determined from X/Y/W  |
//|               sizes                                              |
//|     XC  -   points where spline values/derivatives are           |
//|             constrained, array[0..K-1].                          |
//|     YC  -   values of constraints, array[0..K-1]                 |
//|     DC  -   array[0..K-1], types of constraints:                 |
//|             * DC[i]=0   means that S(XC[i])=YC[i]                |
//|             * DC[i]=1   means that S'(XC[i])=YC[i]               |
//|             SEE BELOW FOR IMPORTANT INFORMATION ON CONSTRAINTS   |
//|     K   -   number of constraints (optional):                    |
//|             * 0<=K<M.                                            |
//|             * K=0 means no constraints (XC/YC/DC are not used)   |
//|             * if given, only first K elements of XC/YC/DC are    |
//|               used                                               |
//|             * if not given, automatically determined from        |
//|               XC/YC/DC                                           |
//|     M   -   number of basis functions ( = number_of_nodes+2),    |
//|             M>=4.                                                |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearWC() subroutine.        |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|     S   -   spline interpolant.                                  |
//|     Rep -   report, same format as in LSFitLinearWC() subroutine.|
//|             Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//| SETTING CONSTRAINTS - DANGERS AND OPPORTUNITIES:                 |
//| Setting constraints can lead  to undesired  results, like        |
//| ill-conditioned behavior, or inconsistency being detected. From  |
//| the other side, it allows us to improve quality of the fit.      |
//| Here we summarize our experience with constrained regression     |
//| splines:                                                         |
//| * excessive constraints can be inconsistent. Splines are         |
//|   piecewise cubic functions, and it is easy to create an         |
//|   example, where large number of constraints  concentrated in    |
//|   small area will result in inconsistency. Just because spline   |
//|   is not flexible enough to satisfy all of them. And same        |
//|   constraints spread across the [min(x),max(x)] will be          |
//|   perfectly consistent.                                          |
//| * the more evenly constraints are spread across [min(x),max(x)], |
//|   the more chances that they will be consistent                  |
//| * the greater is M (given fixed constraints), the more chances   |
//|   that constraints will be consistent                            |
//| * in the general case, consistency of constraints IS NOT         |
//|   GUARANTEED.                                                    |
//| * in the several special cases, however, we CAN guarantee        |
//|   consistency.                                                   |
//| * one of this cases is constraints on the function values        |
//|   AND/OR its derivatives at the interval boundaries.             |
//| * another special case is ONE constraint on the function value   |
//|   (OR, but not AND, derivative) anywhere in the interval         |
//| Our final recommendation is to use constraints WHEN AND ONLY WHEN|
//| you can't solve your task without them. Anything beyond special  |
//| cases given above is not guaranteed and may result in            |
//| inconsistency.                                                   |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitCubicWC(double &x[],double &y[],double &w[],
                                 const int n,double &xc[],double &yc[],
                                 int &dc[],const int k,const int m,
                                 int &info,CSpline1DInterpolantShell &s,
                                 CSpline1DFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::Spline1DFitCubicWC(x,y,w,n,xc,yc,dc,k,m,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted fitting by cubic spline, with constraints on function   |
//| values or derivatives.                                           |
//| Equidistant grid with M-2 nodes on [min(x,xc),max(x,xc)] is used |
//| to build basis functions. Basis functions are cubic splines with |
//| continuous second derivatives and non-fixed first derivatives at |
//| interval ends. Small regularizing term is used when solving      |
//| constrained tasks (to improve stability).                        |
//| Task is linear, so linear least squares solver is used.          |
//| Complexity of this computational scheme is O(N*M^2), mostly      |
//| dominated by least squares solver                                |
//| SEE ALSO                                                         |
//|     Spline1DFitHermiteWC()  -   fitting by Hermite splines (more |
//|                                 flexible, less smooth)           |
//|     Spline1DFitCubic()     -  "lightweight" fitting by cubic   |
//|                                 splines, without invididual      |
//|                                 weights and constraints          |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if you|
//|             don't want to solve weighted task.                   |
//|     N   -   number of points (optional):                         |
//|             * N>0                                                |
//|             * if given, only first N elements of X/Y/W are       |
//|               processed                                          |
//|             * if not given, automatically determined from X/Y/W  |
//|               sizes                                              |
//|     XC  -   points where spline values/derivatives are           |
//|             constrained, array[0..K-1].                          |
//|     YC  -   values of constraints, array[0..K-1]                 |
//|     DC  -   array[0..K-1], types of constraints:                 |
//|             * DC[i]=0   means that S(XC[i])=YC[i]                |
//|             * DC[i]=1   means that S'(XC[i])=YC[i]               |
//|             SEE BELOW FOR IMPORTANT INFORMATION ON CONSTRAINTS   |
//|     K   -   number of constraints (optional):                    |
//|             * 0<=K<M.                                            |
//|             * K=0 means no constraints (XC/YC/DC are not used)   |
//|             * if given, only first K elements of XC/YC/DC are    |
//|               used                                               |
//|             * if not given, automatically determined from        |
//|               XC/YC/DC                                           |
//|     M   -   number of basis functions ( = number_of_nodes+2),    |
//|             M>=4.                                                |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearWC() subroutine.        |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|     S   -   spline interpolant.                                  |
//|     Rep -   report, same format as in LSFitLinearWC() subroutine.|
//|             Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| ORDER OF POINTS                                                  |
//| Subroutine automatically sorts points, so caller may pass        |
//| unsorted array.                                                  |
//| SETTING CONSTRAINTS - DANGERS AND OPPORTUNITIES:                 |
//| Setting constraints can lead  to undesired  results, like        |
//| ill-conditioned behavior, or inconsistency being detected. From  |
//| the other side, it allows us to improve quality of the fit.      |
//| Here we summarize our experience with constrained regression     |
//| splines:                                                         |
//| * excessive constraints can be inconsistent. Splines are         |
//|   piecewise cubic functions, and it is easy to create an         |
//|   example, where large number of constraints  concentrated in    |
//|   small area will result in inconsistency. Just because spline   |
//|   is not flexible enough to satisfy all of them. And same        |
//|   constraints spread across the [min(x),max(x)] will be          |
//|   perfectly consistent.                                          |
//| * the more evenly constraints are spread across [min(x), max(x)], |
//|   the more chances that they will be consistent                  |
//| * the greater is M (given fixed constraints), the more chances   |
//|   that constraints will be consistent                            |
//| * in the general case, consistency of constraints IS NOT         |
//|   GUARANTEED.                                                    |
//| * in the several special cases, however, we CAN guarantee        |
//|   consistency.                                                   |
//| * one of this cases is constraints on the function values        |
//|   AND/OR its derivatives at the interval boundaries.             |
//| * another special case is ONE constraint on the function value   |
//|   (OR, but not AND, derivative) anywhere in the interval         |
//| Our final recommendation is to use constraints WHEN AND ONLY WHEN|
//| you can't solve your task without them. Anything beyond special  |
//| cases given above is not guaranteed and may result in            |
//| inconsistency.                                                   |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitCubicWC(double &x[],double &y[],double &w[],
                                 double &xc[],double &yc[],int &dc[],
                                 const int m,int &info,CSpline1DInterpolantShell &s,
                                 CSpline1DFitReportShell &rep)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)) || (CAp::Len(x)!=CAp::Len(w)))
     {
      Print("Error while calling 'spline1dfitcubicwc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
   if((CAp::Len(xc)!=CAp::Len(yc)) || (CAp::Len(xc)!=CAp::Len(dc)))
     {
      Print("Error while calling 'spline1dfitcubicwc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info=0;
   int n=CAp::Len(x);
   int k=CAp::Len(xc);
//--- function call
   CLSFit::Spline1DFitCubicWC(x,y,w,n,xc,yc,dc,k,m,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted fitting by Hermite spline, with constraints on function |
//| values or first derivatives.                                     |
//| Equidistant grid with M nodes on [min(x,xc),max(x,xc)] is used to|
//| build basis functions. Basis functions are Hermite splines. Small|
//| regularizing term is used when solving constrained tasks (to     |
//| improve stability).                                              |
//| Task is linear, so linear least squares solver is used.          |
//| Complexity of this computational scheme is O(N*M^2), mostly      |
//| dominated by least squares solver                                |
//| SEE ALSO                                                         |
//|     Spline1DFitCubicWC()    -   fitting by Cubic splines (less   |
//|                                 flexible, more smooth)           |
//|     Spline1DFitHermite()   -  "lightweight" Hermite fitting,   |
//|                                 without invididual weights and   |
//|                                 constraints                      |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if    |
//|             you don't want to solve weighted task.               |
//|     N   -   number of points (optional):                         |
//|             * N>0                                                |
//|             * if given, only first N elements of X/Y/W are       |
//|               processed                                          |
//|             * if not given, automatically determined from X/Y/W  |
//|               sizes                                              |
//|     XC  -   points where spline values/derivatives are           |
//|             constrained, array[0..K-1].                          |
//|     YC  -   values of constraints, array[0..K-1]                 |
//|     DC  -   array[0..K-1], types of constraints:                 |
//|             * DC[i]=0   means that S(XC[i])=YC[i]                |
//|             * DC[i]=1   means that S'(XC[i])=YC[i]               |
//|             SEE BELOW FOR IMPORTANT INFORMATION ON CONSTRAINTS   |
//|     K   -   number of constraints (optional):                    |
//|             * 0<=K<M.                                            |
//|             * K=0 means no constraints (XC/YC/DC are not used)   |
//|             * if given, only first K elements of XC/YC/DC are    |
//|               used                                               |
//|             * if not given, automatically determined from        |
//|               XC/YC/DC                                           |
//|     M   -   number of basis functions (= 2 * number of nodes),   |
//|             M>=4,                                                |
//|             M IS EVEN!                                           |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearW() subroutine:         |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|                         -2 means odd M was passed (which is not  |
//|                            supported)                            |
//|                         -1 means another errors in parameters    |
//|                            passed (N<=0, for example)            |
//|     S   -   spline interpolant.                                  |
//|     Rep -   report, same format as in LSFitLinearW() subroutine. |
//|             Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| IMPORTANT:                                                       |
//|     this subroitine supports only even M's                       |
//| ORDER OF POINTS                                                  |
//| ubroutine automatically sorts points, so caller may pass         |
//| unsorted array.                                                  |
//| SETTING CONSTRAINTS - DANGERS AND OPPORTUNITIES:                 |
//| Setting constraints can lead to undesired results, like          |
//| ill-conditioned behavior, or inconsistency being detected. From  |
//| the other side, it allows us to improve quality of the fit. Here |
//| we summarize our experience  with constrained regression splines:|
//| * excessive constraints can be inconsistent. Splines are         |
//|   piecewise cubic functions, and it is easy to create an example,|
//|   where large number of constraints concentrated in small area   |
//|   will result in inconsistency. Just because spline is not       |
//|   flexible enough to satisfy all of them. And same constraints   |
//|   spread across the [min(x),max(x)] will be perfectly consistent.|
//| * the more evenly constraints are spread across [min(x),max(x)], |
//|   the more chances that they will be consistent                  |
//| * the greater is M (given  fixed  constraints), the more chances |
//|   that constraints will be consistent                            |
//| * in the general case, consistency of constraints is NOT         |
//|   GUARANTEED.                                                    |
//| * in the several special cases, however, we can guarantee        |
//|   consistency.                                                   |
//| * one of this cases is M>=4 and constraints on the function      |
//|   value (AND/OR its derivative) at the interval boundaries.      |
//| * another special case is M>=4 and ONE constraint on the         |
//|   function value (OR, BUT NOT AND, derivative) anywhere in       |
//|   [min(x),max(x)]                                                |
//| Our final recommendation is to use constraints WHEN AND ONLY when|
//| you can't solve your task without them. Anything beyond  special |
//| cases given above is not guaranteed and may result in            |
//| inconsistency.                                                   |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitHermiteWC(double &x[],double &y[],double &w[],
                                   const int n,double &xc[],double &yc[],
                                   int &dc[],const int k,const int m,
                                   int &info,CSpline1DInterpolantShell &s,
                                   CSpline1DFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::Spline1DFitHermiteWC(x,y,w,n,xc,yc,dc,k,m,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted fitting by Hermite spline, with constraints on function |
//| values or first derivatives.                                     |
//| Equidistant grid with M nodes on [min(x,xc),max(x,xc)] is used to|
//| build basis functions. Basis functions are Hermite splines. Small|
//| regularizing term is used when solving constrained tasks (to     |
//| improve stability).                                              |
//| Task is linear, so linear least squares solver is used.          |
//| Complexity of this computational scheme is O(N*M^2), mostly      |
//| dominated by least squares solver                                |
//| SEE ALSO                                                         |
//|     Spline1DFitCubicWC()    -   fitting by Cubic splines (less   |
//|                                 flexible, more smooth)           |
//|     Spline1DFitHermite()   -  "lightweight" Hermite fitting,   |
//|                                 without invididual weights and   |
//|                                 constraints                      |
//| INPUT PARAMETERS:                                                |
//|     X   -   points, array[0..N-1].                               |
//|     Y   -   function values, array[0..N-1].                      |
//|     W   -   weights, array[0..N-1]                               |
//|             Each summand in square sum of approximation          |
//|             deviations from given values is multiplied by the    |
//|             square of corresponding weight. Fill it by 1's if    |
//|             you don't want to solve weighted task.               |
//|     N   -   number of points (optional):                         |
//|             * N>0                                                |
//|             * if given, only first N elements of X/Y/W are       |
//|               processed                                          |
//|             * if not given, automatically determined from X/Y/W  |
//|               sizes                                              |
//|     XC  -   points where spline values/derivatives are           |
//|             constrained, array[0..K-1].                          |
//|     YC  -   values of constraints, array[0..K-1]                 |
//|     DC  -   array[0..K-1], types of constraints:                 |
//|             * DC[i]=0   means that S(XC[i])=YC[i]                |
//|             * DC[i]=1   means that S'(XC[i])=YC[i]               |
//|             SEE BELOW FOR IMPORTANT INFORMATION ON CONSTRAINTS   |
//|     K   -   number of constraints (optional):                    |
//|             * 0<=K<M.                                            |
//|             * K=0 means no constraints (XC/YC/DC are not used)   |
//|             * if given, only first K elements of XC/YC/DC are    |
//|               used                                               |
//|             * if not given, automatically determined from        |
//|               XC/YC/DC                                           |
//|     M   -   number of basis functions (= 2 * number of nodes),   |
//|             M>=4,                                                |
//|             M IS EVEN!                                           |
//| OUTPUT PARAMETERS:                                               |
//|     Info-   same format as in LSFitLinearW() subroutine:         |
//|             * Info>0    task is solved                           |
//|             * Info<=0   an error occured:                        |
//|                         -4 means inconvergence of internal SVD   |
//|                         -3 means inconsistent constraints        |
//|                         -2 means odd M was passed (which is not  |
//|                            supported)                            |
//|                         -1 means another errors in parameters    |
//|                            passed (N<=0, for example)            |
//|     S   -   spline interpolant.                                  |
//|     Rep -   report, same format as in LSFitLinearW() subroutine. |
//|             Following fields are set:                            |
//|             * RMSError      rms error on the (X,Y).              |
//|             * AvgError      average error on the (X,Y).          |
//|             * AvgRelError   average relative error on the        |
//|                             non-zero Y                           |
//|             * MaxError      maximum error                        |
//|                             NON-WEIGHTED ERRORS ARE CALCULATED   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//| IMPORTANT:                                                       |
//|     this subroitine supports only even M's                       |
//| ORDER OF POINTS                                                  |
//| ubroutine automatically sorts points, so caller may pass         |
//| unsorted array.                                                  |
//| SETTING CONSTRAINTS - DANGERS AND OPPORTUNITIES:                 |
//| Setting constraints can lead to undesired results, like          |
//| ill-conditioned behavior, or inconsistency being detected. From  |
//| the other side, it allows us to improve quality of the fit. Here |
//| we summarize our experience  with constrained regression splines:|
//| * excessive constraints can be inconsistent. Splines are         |
//|   piecewise cubic functions, and it is easy to create an example,|
//|   where large number of constraints concentrated in small area   |
//|   will result in inconsistency. Just because spline is not       |
//|   flexible enough to satisfy all of them. And same constraints   |
//|   spread across the [min(x),max(x)] will be perfectly consistent.|
//| * the more evenly constraints are spread across [min(x),max(x)], |
//|   the more chances that they will be consistent                  |
//| * the greater is M (given  fixed  constraints), the more chances |
//|   that constraints will be consistent                            |
//| * in the general case, consistency of constraints is NOT         |
//|   GUARANTEED.                                                    |
//| * in the several special cases, however, we can guarantee        |
//|   consistency.                                                   |
//| * one of this cases is M>=4 and constraints on the function      |
//|   value (AND/OR its derivative) at the interval boundaries.      |
//| * another special case is M>=4 and ONE constraint on the         |
//|   function value (OR, BUT NOT AND, derivative) anywhere in       |
//|   [min(x),max(x)]                                                |
//| Our final recommendation is to use constraints WHEN AND ONLY when|
//| you can't solve your task without them. Anything beyond  special |
//| cases given above is not guaranteed and may result in            |
//| inconsistency.                                                   |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitHermiteWC(double &x[],double &y[],double &w[],
                                   double &xc[],double &yc[],int &dc[],
                                   const int m,int &info,
                                   CSpline1DInterpolantShell &s,
                                   CSpline1DFitReportShell &rep)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)) || (CAp::Len(x)!=CAp::Len(w)))
     {
      Print("Error while calling 'spline1dfithermitewc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
   if((CAp::Len(xc)!=CAp::Len(yc)) || (CAp::Len(xc)!=CAp::Len(dc)))
     {
      Print("Error while calling 'spline1dfithermitewc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(x);
   int k=CAp::Len(xc);
//--- function call
   CLSFit::Spline1DFitHermiteWC(x,y,w,n,xc,yc,dc,k,m,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Least squares fitting by cubic spline.                           |
//| This subroutine is "lightweight" alternative for more complex    |
//| and feature - rich Spline1DFitCubicWC(). See Spline1DFitCubicWC()|
//| for more information about subroutine parameters (we don't       |
//| duplicate it here because of length)                             |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitCubic(double &x[],double &y[],const int n,
                               const int m,int &info,
                               CSpline1DInterpolantShell &s,
                               CSpline1DFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::Spline1DFitCubic(x,y,n,m,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Least squares fitting by cubic spline.                           |
//| This subroutine is "lightweight" alternative for more complex    |
//| and feature - rich Spline1DFitCubicWC(). See Spline1DFitCubicWC()|
//| for more information about subroutine parameters (we don't       |
//| duplicate it here because of length)                             |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitCubic(double &x[],double &y[],const int m,
                               int &info,CSpline1DInterpolantShell &s,
                               CSpline1DFitReportShell &rep)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dfitcubic': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(x);
//--- function call
   CLSFit::Spline1DFitCubic(x,y,n,m,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Least squares fitting by Hermite spline.                         |
//| This subroutine is "lightweight" alternative for more complex    |
//| and feature - rich Spline1DFitHermiteWC(). See                   |
//| Spline1DFitHermiteWC() description for more information about    |
//| subroutine parameters (we don't duplicate it here because of     |
//| length).                                                         |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitHermite(double &x[],double &y[],const int n,
                                 const int m,int &info,
                                 CSpline1DInterpolantShell &s,
                                 CSpline1DFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::Spline1DFitHermite(x,y,n,m,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Least squares fitting by Hermite spline.                         |
//| This subroutine is "lightweight" alternative for more complex    |
//| and feature - rich Spline1DFitHermiteWC(). See                   |
//| Spline1DFitHermiteWC() description for more information about    |
//| subroutine parameters (we don't duplicate it here because of     |
//| length).                                                         |
//+------------------------------------------------------------------+
void CAlglib::Spline1DFitHermite(double &x[],double &y[],const int m,
                                 int &info,CSpline1DInterpolantShell &s,
                                 CSpline1DFitReportShell &rep)
  {
//--- check
   if((CAp::Len(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'spline1dfithermite': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(x);
//--- function call
   CLSFit::Spline1DFitHermite(x,y,n,m,info,s.GetInnerObj(),rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted linear least squares fitting.                           |
//| QR decomposition is used to reduce task to MxM, then triangular  |
//| solver or SVD-based solver is used depending on condition number |
//| of the system. It allows to maximize speed and retain decent     |
//| accuracy.                                                        |
//| INPUT PARAMETERS:                                                |
//|     Y       -   array[0..N-1] Function values in N points.       |
//|     W       -   array[0..N-1] Weights corresponding to function  |
//|                 values. Each summand in square sum of            |
//|                 approximation deviations from given values is    |
//|                 multiplied by the square of corresponding weight.|
//|     FMatrix -   a table of basis functions values,               |
//|                 array[0..N-1, 0..M-1]. FMatrix[I, J] - value of  |
//|                 J-th basis function in I-th point.               |
//|     N       -   number of points used. N>=1.                     |
//|     M       -   number of basis functions, M>=1.                 |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -4    internal SVD decomposition subroutine    |
//|                         failed (very rare and for degenerate     |
//|                         systems only)                            |
//|                 * -1    incorrect N/M were specified             |
//|                 *  1    task is solved                           |
//|     C       -   decomposition coefficients, array[0..M-1]        |
//|     Rep     -   fitting report. Following fields are set:        |
//|                 * Rep.TaskRCond     reciprocal of condition      |
//|                                     number                       |
//|                 * RMSError          rms error on the (X,Y).      |
//|                 * AvgError          average error on the (X,Y).  |
//|                 * AvgRelError       average relative error on the|
//|                                     non-zero Y                   |
//|                 * MaxError          maximum error                |
//|                                     NON-WEIGHTED ERRORS ARE      |
//|                                     CALCULATED                   |
//+------------------------------------------------------------------+
void CAlglib::LSFitLinearW(double &y[],double &w[],CMatrixDouble &fmatrix,
                           const int n,const int m,int &info,
                           double &c[],CLSFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::LSFitLinearW(y,w,fmatrix,n,m,info,c,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted linear least squares fitting.                           |
//| QR decomposition is used to reduce task to MxM, then triangular  |
//| solver or SVD-based solver is used depending on condition number |
//| of the system. It allows to maximize speed and retain decent     |
//| accuracy.                                                        |
//| INPUT PARAMETERS:                                                |
//|     Y       -   array[0..N-1] Function values in N points.       |
//|     W       -   array[0..N-1] Weights corresponding to function  |
//|                 values. Each summand in square sum of            |
//|                 approximation deviations from given values is    |
//|                 multiplied by the square of corresponding weight.|
//|     FMatrix -   a table of basis functions values,               |
//|                 array[0..N-1, 0..M-1]. FMatrix[I, J] - value of  |
//|                 J-th basis function in I-th point.               |
//|     N       -   number of points used. N>=1.                     |
//|     M       -   number of basis functions, M>=1.                 |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -4    internal SVD decomposition subroutine    |
//|                         failed (very rare and for degenerate     |
//|                         systems only)                            |
//|                 * -1    incorrect N/M were specified             |
//|                 *  1    task is solved                           |
//|     C       -   decomposition coefficients, array[0..M-1]        |
//|     Rep     -   fitting report. Following fields are set:        |
//|                 * Rep.TaskRCond     reciprocal of condition      |
//|                                     number                       |
//|                 * RMSError          rms error on the (X,Y).      |
//|                 * AvgError          average error on the (X,Y).  |
//|                 * AvgRelError       average relative error on the|
//|                                     non-zero Y                   |
//|                 * MaxError          maximum error                |
//|                                     NON-WEIGHTED ERRORS ARE      |
//|                                     CALCULATED                   |
//+------------------------------------------------------------------+
void CAlglib::LSFitLinearW(double &y[],double &w[],CMatrixDouble &fmatrix,
                           int &info,double &c[],CLSFitReportShell &rep)
  {
//--- check
   if((CAp::Len(y)!=CAp::Len(w)) || (CAp::Len(y)!=CAp::Rows(fmatrix)))
     {
      Print("Error while calling 'lsfitlinearw': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(y);
   int m=(int)CAp::Cols(fmatrix);
//--- function call
   CLSFit::LSFitLinearW(y,w,fmatrix,n,m,info,c,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted constained linear least squares fitting.                |
//| This is variation of LSFitLinearW(), which searchs for           |
//| min|A*x=b| given that K additional constaints C*x=bc are         |
//| satisfied. It reduces original task to modified one: min|B*y-d|  |
//| WITHOUT constraints, then LSFitLinearW() is called.              |
//| INPUT PARAMETERS:                                                |
//|     Y       -   array[0..N-1] Function values in  N  points.     |
//|     W       -   array[0..N-1] Weights corresponding to function  |
//|                 values. Each summand in square sum of            |
//|                 approximation deviations from given values is    |
//|                 multiplied by the square of corresponding        |
//|                 weight.                                          |
//|     FMatrix -   a table of basis functions values,               |
//|                 array[0..N-1,0..M-1]. FMatrix[I,J] - value of    |
//|                 J-th basis function in I-th point.               |
//|     CMatrix -   a table of constaints, array[0..K-1,0..M].       |
//|                 I-th row of CMatrix corresponds to I-th linear   |
//|                 constraint: CMatrix[I,0]*C[0] + ... +            |
//|                 + CMatrix[I,M-1]*C[M-1] = CMatrix[I,M]           |
//|     N       -   number of points used. N>=1.                     |
//|     M       -   number of basis functions, M>=1.                 |
//|     K       -   number of constraints, 0 <= K < M                |
//|                 K=0 corresponds to absence of constraints.       |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -4    internal SVD decomposition subroutine    |
//|                         failed (very rare and for degenerate     |
//|                         systems only)                            |
//|                 * -3    either too many constraints (M or more), |
//|                         degenerate constraints (some constraints |
//|                         are repetead twice) or inconsistent      |
//|                         constraints were specified.              |
//|                 *  1    task is solved                           |
//|     C       -   decomposition coefficients, array[0..M-1]        |
//|     Rep     -   fitting report. Following fields are set:        |
//|                 * RMSError          rms error on the (X,Y).      |
//|                 * AvgError          average error on the (X,Y).  |
//|                 * AvgRelError       average relative error on the|
//|                                     non-zero Y                   |
//|                 * MaxError          maximum error                |
//|                                     NON-WEIGHTED ERRORS ARE      |
//|                                     CALCULATED                   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//+------------------------------------------------------------------+
void CAlglib::LSFitLinearWC(double &y[],double &w[],CMatrixDouble &fmatrix,
                            CMatrixDouble &cmatrix,const int n,
                            const int m,const int k,int &info,
                            double &c[],CLSFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::LSFitLinearWC(y,w,fmatrix,cmatrix,n,m,k,info,c,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted constained linear least squares fitting.                |
//| This is variation of LSFitLinearW(), which searchs for           |
//| min|A*x=b| given that K additional constaints C*x=bc are         |
//| satisfied. It reduces original task to modified one: min|B*y-d|  |
//| WITHOUT constraints, then LSFitLinearW() is called.              |
//| INPUT PARAMETERS:                                                |
//|     Y       -   array[0..N-1] Function values in  N  points.     |
//|     W       -   array[0..N-1] Weights corresponding to function  |
//|                 values. Each summand in square sum of            |
//|                 approximation deviations from given values is    |
//|                 multiplied by the square of corresponding        |
//|                 weight.                                          |
//|     FMatrix -   a table of basis functions values,               |
//|                 array[0..N-1,0..M-1]. FMatrix[I,J] - value of    |
//|                 J-th basis function in I-th point.               |
//|     CMatrix -   a table of constaints, array[0..K-1,0..M].       |
//|                 I-th row of CMatrix corresponds to I-th linear   |
//|                 constraint: CMatrix[I,0]*C[0] + ... +            |
//|                 + CMatrix[I,M-1]*C[M-1] = CMatrix[I,M]           |
//|     N       -   number of points used. N>=1.                     |
//|     M       -   number of basis functions, M>=1.                 |
//|     K       -   number of constraints, 0 <= K < M                |
//|                 K=0 corresponds to absence of constraints.       |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -4    internal SVD decomposition subroutine    |
//|                         failed (very rare and for degenerate     |
//|                         systems only)                            |
//|                 * -3    either too many constraints (M or more), |
//|                         degenerate constraints (some constraints |
//|                         are repetead twice) or inconsistent      |
//|                         constraints were specified.              |
//|                 *  1    task is solved                           |
//|     C       -   decomposition coefficients, array[0..M-1]        |
//|     Rep     -   fitting report. Following fields are set:        |
//|                 * RMSError          rms error on the (X,Y).      |
//|                 * AvgError          average error on the (X,Y).  |
//|                 * AvgRelError       average relative error on the|
//|                                     non-zero Y                   |
//|                 * MaxError          maximum error                |
//|                                     NON-WEIGHTED ERRORS ARE      |
//|                                     CALCULATED                   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//+------------------------------------------------------------------+
void CAlglib::LSFitLinearWC(double &y[],double &w[],CMatrixDouble &fmatrix,
                            CMatrixDouble &cmatrix,int &info,
                            double &c[],CLSFitReportShell &rep)
  {
//--- check
   if((CAp::Len(y)!=CAp::Len(w)) || (CAp::Len(y)!=CAp::Rows(fmatrix)))
     {
      Print("Error while calling 'lsfitlinearwc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
   if((CAp::Cols(fmatrix)!=CAp::Cols(cmatrix)-1))
     {
      Print("Error while calling 'lsfitlinearwc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(y);
   int m=(int)CAp::Cols(fmatrix);
   int k=(int)CAp::Rows(cmatrix);
//--- function call
   CLSFit::LSFitLinearWC(y,w,fmatrix,cmatrix,n,m,k,info,c,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Linear least squares fitting.                                    |
//| QR decomposition is used to reduce task to MxM, then triangular  |
//| solver or SVD-based solver is used depending on condition number |
//| of the system. It allows to maximize speed and retain decent     |
//| accuracy.                                                        |
//| INPUT PARAMETERS:                                                |
//|     Y       -   array[0..N-1] Function values in  N  points.     |
//|     FMatrix -   a table of basis functions values,               |
//|                 array[0..N-1, 0..M-1].                           |
//|                 FMatrix[I, J] - value of J-th basis function in  |
//|                 I-th point.                                      |
//|     N       -   number of points used. N>=1.                     |
//|     M       -   number of basis functions, M>=1.                 |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -4    internal SVD decomposition subroutine    |
//|                         failed (very rare and for degenerate     |
//|                         systems only)                            |
//|                 *  1    task is solved                           |
//|     C       -   decomposition coefficients, array[0..M-1]        |
//|     Rep     -   fitting report. Following fields are set:        |
//|                 * Rep.TaskRCond     reciprocal of condition      |
//|                                     number                       |
//|                 * RMSError          rms error on the (X,Y).      |
//|                 * AvgError          average error on the (X,Y).  |
//|                 * AvgRelError       average relative error on the|
//|                                     non-zero Y                   |
//|                 * MaxError          maximum error                |
//|                                     NON-WEIGHTED ERRORS ARE      |
//|                                     CALCULATED                   |
//+------------------------------------------------------------------+
void CAlglib::LSFitLinear(double &y[],CMatrixDouble &fmatrix,
                          const int n,const int m,int &info,
                          double &c[],CLSFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::LSFitLinear(y,fmatrix,n,m,info,c,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Linear least squares fitting.                                    |
//| QR decomposition is used to reduce task to MxM, then triangular  |
//| solver or SVD-based solver is used depending on condition number |
//| of the system. It allows to maximize speed and retain decent     |
//| accuracy.                                                        |
//| INPUT PARAMETERS:                                                |
//|     Y       -   array[0..N-1] Function values in  N  points.     |
//|     FMatrix -   a table of basis functions values,               |
//|                 array[0..N-1, 0..M-1].                           |
//|                 FMatrix[I, J] - value of J-th basis function in  |
//|                 I-th point.                                      |
//|     N       -   number of points used. N>=1.                     |
//|     M       -   number of basis functions, M>=1.                 |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -4    internal SVD decomposition subroutine    |
//|                         failed (very rare and for degenerate     |
//|                         systems only)                            |
//|                 *  1    task is solved                           |
//|     C       -   decomposition coefficients, array[0..M-1]        |
//|     Rep     -   fitting report. Following fields are set:        |
//|                 * Rep.TaskRCond     reciprocal of condition      |
//|                                     number                       |
//|                 * RMSError          rms error on the (X,Y).      |
//|                 * AvgError          average error on the (X,Y).  |
//|                 * AvgRelError       average relative error on the|
//|                                     non-zero Y                   |
//|                 * MaxError          maximum error                |
//|                                     NON-WEIGHTED ERRORS ARE      |
//|                                     CALCULATED                   |
//+------------------------------------------------------------------+
void CAlglib::LSFitLinear(double &y[],CMatrixDouble &fmatrix,
                          int &info,double &c[],CLSFitReportShell &rep)
  {
//--- check
   if((CAp::Len(y)!=CAp::Rows(fmatrix)))
     {
      Print("Error while calling 'lsfitlinear': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(y);
   int m=(int)CAp::Cols(fmatrix);
//--- function call
   CLSFit::LSFitLinear(y,fmatrix,n,m,info,c,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Constained linear least squares fitting.                         |
//| This is variation of LSFitLinear(), which searchs for min|A*x=b| |
//| given that K additional constaints C*x=bc are satisfied. It      |
//| reduces original task to modified one: min|B*y-d| WITHOUT        |
//| constraints, then LSFitLinear() is called.                       |
//| INPUT PARAMETERS:                                                |
//|     Y       -   array[0..N-1] Function values in N points.       |
//|     FMatrix -   a table of basis functions values,               |
//|                 array[0..N-1,0..M-1]. FMatrix[I,J] - value of    |
//|                 J-th basis function in I-th point.               |
//|     CMatrix -   a table of constaints, array[0..K-1,0..M].       |
//|                 I-th row of CMatrix corresponds to I-th linear   |
//|                 constraint: CMatrix[I,0]*C[0] + ... +            |
//|                 + CMatrix[I,M-1]*C[M-1] = CMatrix[I,M]           |
//|     N       -   number of points used. N>=1.                     |
//|     M       -   number of basis functions, M>=1.                 |
//|     K       -   number of constraints, 0 <= K < M                |
//|                 K=0 corresponds to absence of constraints.       |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -4    internal SVD decomposition subroutine    |
//|                         failed (very rare and for degenerate     |
//|                         systems only)                            |
//|                 * -3    either too many constraints (M or more), |
//|                         degenerate constraints (some constraints |
//|                         are repetead twice) or inconsistent      |
//|                         constraints were specified.              |
//|                 *  1    task is solved                           |
//|     C       -   decomposition coefficients, array[0..M-1]        |
//|     Rep     -   fitting report. Following fields are set:        |
//|                 * RMSError          rms error on the (X,Y).      |
//|                 * AvgError          average error on the (X,Y).  |
//|                 * AvgRelError       average relative error on the|
//|                                     non-zero Y                   |
//|                 * MaxError          maximum error                |
//|                                     NON-WEIGHTED ERRORS ARE      |
//|                                     CALCULATED                   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//+------------------------------------------------------------------+
void CAlglib::LSFitLinearC(double &y[],CMatrixDouble &fmatrix,
                           CMatrixDouble &cmatrix,const int n,
                           const int m,const int k,int &info,
                           double &c[],CLSFitReportShell &rep)
  {
//--- initialization
   info=0;
//--- function call
   CLSFit::LSFitLinearC(y,fmatrix,cmatrix,n,m,k,info,c,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Constained linear least squares fitting.                         |
//| This is variation of LSFitLinear(), which searchs for min|A*x=b| |
//| given that K additional constaints C*x=bc are satisfied. It      |
//| reduces original task to modified one: min|B*y-d| WITHOUT        |
//| constraints, then LSFitLinear() is called.                       |
//| INPUT PARAMETERS:                                                |
//|     Y       -   array[0..N-1] Function values in N points.       |
//|     FMatrix -   a table of basis functions values,               |
//|                 array[0..N-1,0..M-1]. FMatrix[I,J] - value of    |
//|                 J-th basis function in I-th point.               |
//|     CMatrix -   a table of constaints, array[0..K-1,0..M].       |
//|                 I-th row of CMatrix corresponds to I-th linear   |
//|                 constraint: CMatrix[I,0]*C[0] + ... +            |
//|                 + CMatrix[I,M-1]*C[M-1] = CMatrix[I,M]           |
//|     N       -   number of points used. N>=1.                     |
//|     M       -   number of basis functions, M>=1.                 |
//|     K       -   number of constraints, 0 <= K < M                |
//|                 K=0 corresponds to absence of constraints.       |
//| OUTPUT PARAMETERS:                                               |
//|     Info    -   error code:                                      |
//|                 * -4    internal SVD decomposition subroutine    |
//|                         failed (very rare and for degenerate     |
//|                         systems only)                            |
//|                 * -3    either too many constraints (M or more), |
//|                         degenerate constraints (some constraints |
//|                         are repetead twice) or inconsistent      |
//|                         constraints were specified.              |
//|                 *  1    task is solved                           |
//|     C       -   decomposition coefficients, array[0..M-1]        |
//|     Rep     -   fitting report. Following fields are set:        |
//|                 * RMSError          rms error on the (X,Y).      |
//|                 * AvgError          average error on the (X,Y).  |
//|                 * AvgRelError       average relative error on the|
//|                                     non-zero Y                   |
//|                 * MaxError          maximum error                |
//|                                     NON-WEIGHTED ERRORS ARE      |
//|                                     CALCULATED                   |
//| IMPORTANT:                                                       |
//|     this subroitine doesn't calculate task's condition number    |
//|     for K<>0.                                                    |
//+------------------------------------------------------------------+
void CAlglib::LSFitLinearC(double &y[],CMatrixDouble &fmatrix,
                           CMatrixDouble &cmatrix,int &info,
                           double &c[],CLSFitReportShell &rep)
  {
//--- check
   if((CAp::Len(y)!=CAp::Rows(fmatrix)))
     {
      Print("Error while calling 'lsfitlinearc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
   if((CAp::Cols(fmatrix)!=CAp::Cols(cmatrix)-1))
     {
      Print("Error while calling 'lsfitlinearc': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   info =0;
   int n=CAp::Len(y);
   int m=(int)CAp::Cols(fmatrix);
   int k=(int)CAp::Rows(cmatrix);
//--- function call
   CLSFit::LSFitLinearC(y,fmatrix,cmatrix,n,m,k,info,c,rep.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted nonlinear least squares fitting using function values   |
//| only.                                                            |
//| Combination of numerical differentiation and secant updates is   |
//| used to obtain function Jacobian.                                |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = (w[0]*(f(c,x[0])-y[0]))^2 + ... +                     |
//|     + (w[n-1]*(f(c,x[n-1])-y[n-1]))^2,                           |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * w is an N-dimensional vector of weight coefficients,       |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses only f(c,x[i]).                             |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     W       -   weights, array[0..N-1]                           |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//|     DiffStep-   numerical differentiation step;                  |
//|                 should not be very small or large;               |
//|                 large = loss of accuracy                         |
//|                 small = growth of round-off errors               |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateWF(CMatrixDouble &x,double &y[],double &w[],
                            double &c[],const int n,const int m,
                            const int k,const double diffstep,
                            CLSFitStateShell &state)
  {
   CLSFit::LSFitCreateWF(x,y,w,c,n,m,k,diffstep,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted nonlinear least squares fitting using function values   |
//| only.                                                            |
//| Combination of numerical differentiation and secant updates is   |
//| used to obtain function Jacobian.                                |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = (w[0]*(f(c,x[0])-y[0]))^2 + ... +                     |
//|     + (w[n-1]*(f(c,x[n-1])-y[n-1]))^2,                           |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * w is an N-dimensional vector of weight coefficients,       |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses only f(c,x[i]).                             |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     W       -   weights, array[0..N-1]                           |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//|     DiffStep-   numerical differentiation step;                  |
//|                 should not be very small or large;               |
//|                 large = loss of accuracy                         |
//|                 small = growth of round-off errors               |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateWF(CMatrixDouble &x,double &y[],double &w[],
                            double &c[],const double diffstep,
                            CLSFitStateShell &state)
  {
//--- check
   if((CAp::Rows(x)!=CAp::Len(y)) || (CAp::Rows(x)!=CAp::Len(w)))
     {
      Print("Error while calling 'lsfitcreatewf': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=(int)CAp::Rows(x);
   int m=(int)CAp::Cols(x);
   int k=CAp::Len(c);
//--- function call
   CLSFit::LSFitCreateWF(x,y,w,c,n,m,k,diffstep,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Nonlinear least squares fitting using function values only.      |
//| Combination of numerical differentiation and secant updates is   |
//| used to obtain function Jacobian.                                |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = (f(c,x[0])-y[0])^2 + ... + (f(c,x[n-1])-y[n-1])^2,    |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * w is an N-dimensional vector of weight coefficients,       |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses only f(c,x[i]).                             |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//|     DiffStep-   numerical differentiation step;                  |
//|                 should not be very small or large;               |
//|                 large = loss of accuracy                         |
//|                 small = growth of round-off errors               |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateF(CMatrixDouble &x,double &y[],double &c[],
                           const int n,const int m,const int k,
                           const double diffstep,CLSFitStateShell &state)
  {
   CLSFit::LSFitCreateF(x,y,c,n,m,k,diffstep,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Nonlinear least squares fitting using function values only.      |
//| Combination of numerical differentiation and secant updates is   |
//| used to obtain function Jacobian.                                |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = (f(c,x[0])-y[0])^2 + ... + (f(c,x[n-1])-y[n-1])^2,    |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * w is an N-dimensional vector of weight coefficients,       |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses only f(c,x[i]).                             |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//|     DiffStep-   numerical differentiation step;                  |
//|                 should not be very small or large;               |
//|                 large = loss of accuracy                         |
//|                 small = growth of round-off errors               |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateF(CMatrixDouble &x,double &y[],double &c[],
                           const double diffstep,CLSFitStateShell &state)
  {
//--- check
   if((CAp::Rows(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'lsfitcreatef': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=(int)CAp::Rows(x);
   int m=(int)CAp::Cols(x);
   int k=CAp::Len(c);
//--- function call
   CLSFit::LSFitCreateF(x,y,c,n,m,k,diffstep,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted nonlinear least squares fitting using gradient only.    |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = (w[0]*(f(c,x[0])-y[0]))^2 + ... +                     |
//|     + (w[n-1]*(f(c,x[n-1])-y[n-1]))^2,                           |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * w is an N-dimensional vector of weight coefficients,       |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses only f(c,x[i]) and its gradient.            |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     W       -   weights, array[0..N-1]                           |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//|     CheapFG -   boolean flag, which is:                          |
//|                 * True if both function and gradient calculation |
//|                        complexity are less than O(M^2). An       |
//|                        improved algorithm can be used which      |
//|                        corresponds to FGJ scheme from MINLM unit.|
//|                 * False otherwise.                               |
//|                        Standard Jacibian-bases                   |
//|                        Levenberg-Marquardt algo will be used (FJ |
//|                        scheme).                                  |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//| See also:                                                        |
//|     LSFitResults                                                 |
//|     LSFitCreateFG (fitting without weights)                      |
//|     LSFitCreateWFGH (fitting using Hessian)                      |
//|     LSFitCreateFGH (fitting using Hessian, without weights)      |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateWFG(CMatrixDouble &x,double &y[],double &w[],
                             double &c[],const int n,const int m,
                             const int k,const bool cheapfg,
                             CLSFitStateShell &state)
  {
   CLSFit::LSFitCreateWFG(x,y,w,c,n,m,k,cheapfg,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted nonlinear least squares fitting using gradient only.    |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = (w[0]*(f(c,x[0])-y[0]))^2 + ... +                     |
//|     + (w[n-1]*(f(c,x[n-1])-y[n-1]))^2,                           |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * w is an N-dimensional vector of weight coefficients,       |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses only f(c,x[i]) and its gradient.            |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     W       -   weights, array[0..N-1]                           |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//|     CheapFG -   boolean flag, which is:                          |
//|                 * True if both function and gradient calculation |
//|                        complexity are less than O(M^2). An       |
//|                        improved algorithm can be used which      |
//|                        corresponds to FGJ scheme from MINLM unit.|
//|                 * False otherwise.                               |
//|                        Standard Jacibian-bases                   |
//|                        Levenberg-Marquardt algo will be used (FJ |
//|                        scheme).                                  |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//| See also:                                                        |
//|     LSFitResults                                                 |
//|     LSFitCreateFG (fitting without weights)                      |
//|     LSFitCreateWFGH (fitting using Hessian)                      |
//|     LSFitCreateFGH (fitting using Hessian, without weights)      |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateWFG(CMatrixDouble &x,double &y[],double &w[],
                             double &c[],const bool cheapfg,
                             CLSFitStateShell &state)
  {
//--- check
   if((CAp::Rows(x)!=CAp::Len(y)) || (CAp::Rows(x)!=CAp::Len(w)))
     {
      Print("Error while calling 'lsfitcreatewfg': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=(int)CAp::Rows(x);
   int m=(int)CAp::Cols(x);
   int k=CAp::Len(c);
//--- function call
   CLSFit::LSFitCreateWFG(x,y,w,c,n,m,k,cheapfg,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Nonlinear least squares fitting using gradient only, without     |
//| individual weights.                                              |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = ((f(c,x[0])-y[0]))^2 + ... + ((f(c,x[n-1])-y[n-1]))^2,|
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses only f(c,x[i]) and its gradient.            |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//|     CheapFG -   boolean flag, which is:                          |
//|                 * True  if both function and gradient calculation|
//|                         complexity are less than O(M^2). An      |
//|                         improved algorithm can be used which     |
//|                         corresponds to FGJ scheme from MINLM     |
//|                         unit.                                    |
//|                 * False otherwise.                               |
//|                         Standard Jacibian-bases                  |
//|                         Levenberg-Marquardt algo will be used    |
//|                         (FJ scheme).                             |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateFG(CMatrixDouble &x,double &y[],double &c[],
                            const int n,const int m,const int k,
                            const bool cheapfg,CLSFitStateShell &state)
  {
   CLSFit::LSFitCreateFG(x,y,c,n,m,k,cheapfg,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Nonlinear least squares fitting using gradient only, without     |
//| individual weights.                                              |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = ((f(c,x[0])-y[0]))^2 + ... + ((f(c,x[n-1])-y[n-1]))^2,|
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses only f(c,x[i]) and its gradient.            |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//|     CheapFG -   boolean flag, which is:                          |
//|                 * True  if both function and gradient calculation|
//|                         complexity are less than O(M^2). An      |
//|                         improved algorithm can be used which     |
//|                         corresponds to FGJ scheme from MINLM     |
//|                         unit.                                    |
//|                 * False otherwise.                               |
//|                         Standard Jacibian-bases                  |
//|                         Levenberg-Marquardt algo will be used    |
//|                         (FJ scheme).                             |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateFG(CMatrixDouble &x,double &y[],double &c[],
                            const bool cheapfg,CLSFitStateShell &state)
  {
//--- check
   if((CAp::Rows(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'lsfitcreatefg': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=(int)CAp::Rows(x);
   int m=(int)CAp::Cols(x);
   int k=CAp::Len(c);
//--- function call
   CLSFit::LSFitCreateFG(x,y,c,n,m,k,cheapfg,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted nonlinear least squares fitting using gradient/Hessian. |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = (w[0]*(f(c,x[0])-y[0]))^2 + ... +                     |
//|     (w[n-1]*(f(c,x[n-1])-y[n-1]))^2,                             |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * w is an N-dimensional vector of weight coefficients,       |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|     vector,                                                      |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses f(c,x[i]), its gradient and its Hessian.    |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     W       -   weights, array[0..N-1]                           |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateWFGH(CMatrixDouble &x,double &y[],double &w[],
                              double &c[],const int n,const int m,
                              const int k,CLSFitStateShell &state)
  {
   CLSFit::LSFitCreateWFGH(x,y,w,c,n,m,k,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Weighted nonlinear least squares fitting using gradient/Hessian. |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = (w[0]*(f(c,x[0])-y[0]))^2 + ... +                     |
//|     (w[n-1]*(f(c,x[n-1])-y[n-1]))^2,                             |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * w is an N-dimensional vector of weight coefficients,       |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|     vector,                                                      |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses f(c,x[i]), its gradient and its Hessian.    |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     W       -   weights, array[0..N-1]                           |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateWFGH(CMatrixDouble &x,double &y[],double &w[],
                              double &c[],CLSFitStateShell &state)
  {
//--- check
   if((CAp::Rows(x)!=CAp::Len(y)) || (CAp::Rows(x)!=CAp::Len(w)))
     {
      Print("Error while calling 'lsfitcreatewfgh': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=(int)CAp::Rows(x);
   int m=(int)CAp::Cols(x);
   int k=CAp::Len(c);
//--- function call
   CLSFit::LSFitCreateWFGH(x,y,w,c,n,m,k,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Nonlinear least squares fitting using gradient/Hessian, without  |
//| individial weights.                                              |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = ((f(c,x[0])-y[0]))^2 + ... +                          |
//|     ((f(c,x[n-1])-y[n-1]))^2,                                    |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses f(c,x[i]), its gradient and its Hessian.    |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateFGH(CMatrixDouble &x,double &y[],double &c[],
                             const int n,const int m,const int k,
                             CLSFitStateShell &state)
  {
   CLSFit::LSFitCreateFGH(x,y,c,n,m,k,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Nonlinear least squares fitting using gradient/Hessian, without  |
//| individial weights.                                              |
//| Nonlinear task min(F(c)) is solved, where                        |
//|     F(c) = ((f(c,x[0])-y[0]))^2 + ... +                          |
//|     ((f(c,x[n-1])-y[n-1]))^2,                                    |
//|     * N is a number of points,                                   |
//|     * M is a dimension of a space points belong to,              |
//|     * K is a dimension of a space of parameters being fitted,    |
//|     * x is a set of N points, each of them is an M-dimensional   |
//|       vector,                                                    |
//|     * c is a K-dimensional vector of parameters being fitted     |
//| This subroutine uses f(c,x[i]), its gradient and its Hessian.    |
//| INPUT PARAMETERS:                                                |
//|     X       -   array[0..N-1,0..M-1], points (one row = one      |
//|                 point)                                           |
//|     Y       -   array[0..N-1], function values.                  |
//|     C       -   array[0..K-1], initial approximation to the      |
//|                 solution,                                        |
//|     N       -   number of points, N>1                            |
//|     M       -   dimension of space                               |
//|     K       -   number of parameters being fitted                |
//| OUTPUT PARAMETERS:                                               |
//|     State   -   structure which stores algorithm state           |
//+------------------------------------------------------------------+
void CAlglib::LSFitCreateFGH(CMatrixDouble &x,double &y[],double &c[],
                             CLSFitStateShell &state)
  {
//--- check
   if((CAp::Rows(x)!=CAp::Len(y)))
     {
      Print("Error while calling 'lsfitcreatefgh': looks like one of arguments has wrong size");
      CAp::exception_happened=true;
      return;
     }
//--- initialization
   int n=(int)CAp::Rows(x);
   int m=(int)CAp::Cols(x);
   int k=CAp::Len(c);
//--- function call
   CLSFit::LSFitCreateFGH(x,y,c,n,m,k,state.GetInnerObj());
  }
//+------------------------------------------------------------------+
//| Stopping conditions for nonlinear least squares fitting.         |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm state           |
//|     EpsF    -   stopping criterion. Algorithm stops if           |
//|                 |F(k+1)-F(k)| <= EpsF*max{|F(k)|, |F(k+1)|, 1}   |
//|     EpsX    -   >=0                                              |
//|                 The subroutine finishes its work if on k+1-th    |
//|                 iteration the condition |v|<=EpsX is fulfilled,  |
//|                 where:                                           |
//|                 * |.| means Euclidian norm                       |
//|                 * v - scaled step vector, v[i]=dx[i]/s[i]        |
//|                 * dx - ste pvector, dx=X(k+1)-X(k)               |
//|                 * s - scaling coefficients set by LSFitSetScale()|
//|     MaxIts  -   maximum number of iterations. If MaxIts=0, the   |
//|                 number of iterations is unlimited. Only          |
//|                 Levenberg-Marquardt iterations are counted       |
//|                 (L-BFGS/CG iterations are NOT counted because    |
//|                 their cost is very low compared to that of LM).  |
//| NOTE                                                             |
//| Passing EpsF=0, EpsX=0 and MaxIts=0 (simultaneously) will lead to|
//| automatic stopping criterion selection (according to the scheme  |
//| used by MINLM unit).                                             |
//+------------------------------------------------------------------+
void CAlglib::LSFitSetCond(CLSFitStateShell &state,
                           const double epsx,const int maxits)
  {
   CLSFit::LSFitSetCond(state.GetInnerObj(),epsx,maxits);
  }
//+------------------------------------------------------------------+
//| This function sets maximum step length                           |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm state           |
//|     StpMax  -   maximum step length, >=0. Set StpMax to 0.0, if  |
//|                 you don't want to limit step length.             |
//| Use this subroutine when you optimize target function which      |
//| contains exp() or other fast growing functions, and optimization |
//| algorithm makes too large steps which leads to overflow. This    |
//| function allows us to reject steps that are too large (and       |
//| therefore expose us to the possible overflow) without actually   |
//| calculating function value at the x+stp*d.                       |
//| NOTE: non-zero StpMax leads to moderate performance degradation  |
//| because intermediate step of preconditioned L-BFGS optimization  |
//| is incompatible with limits on step size.                        |
//+------------------------------------------------------------------+
void CAlglib::LSFitSetStpMax(CLSFitStateShell &state,const double stpmax)
  {
   CLSFit::LSFitSetStpMax(state.GetInnerObj(),stpmax);
  }
//+------------------------------------------------------------------+
//| This function turns on/off reporting.                            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure which stores algorithm state           |
//|     NeedXRep-   whether iteration reports are needed or not      |
//| When reports are needed, State.C (current parameters) and State. |
//| F (current value of fitting function) are reported.              |
//+------------------------------------------------------------------+
void CAlglib::LSFitSetXRep(CLSFitStateShell &state,const bool needxrep)
  {
   CLSFit::LSFitSetXRep(state.GetInnerObj(),needxrep);
  }
//+------------------------------------------------------------------+
//| This function sets scaling coefficients for underlying optimizer.|
//| ALGLIB optimizers use scaling matrices to test stopping          |
//| conditions (step size and gradient are scaled before comparison  |
//| with tolerances). Scale of the I-th variable is a translation    |
//| invariant measure of:                                            |
//| a) "how large" the variable is                                   |
//| b) how large the step should be to make significant changes in   |
//| the function                                                     |
//| Generally, scale is NOT considered to be a form of               |
//| preconditioner. But LM optimizer is unique in that it uses       |
//| scaling matrix both in the stopping condition tests and as       |
//| Marquardt damping factor.                                        |
//| Proper scaling is very important for the algorithm performance.  |
//| It is less important for the quality of results, but still has   |
//| some influence (it is easier to converge when variables are      |
//| properly scaled, so premature stopping is possible when very     |
//| badly scalled variables are combined with relaxed stopping       |
//| conditions).                                                     |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure stores algorithm state                 |
//|     S       -   array[N], non-zero scaling coefficients          |
//|                 S[i] may be negative, sign doesn't matter.       |
//+------------------------------------------------------------------+
void CAlglib::LSFitSetScale(CLSFitStateShell &state,double &s[])
  {
   CLSFit::LSFitSetScale(state.GetInnerObj(),s);
  }
//+------------------------------------------------------------------+
//| This function sets boundary constraints for underlying optimizer |
//| Boundary constraints are inactive by default (after initial      |
//| creation). They are preserved until explicitly turned off with   |
//| another SetBC() call.                                            |
//| INPUT PARAMETERS:                                                |
//|     State   -   structure stores algorithm state                 |
//|     BndL    -   lower bounds, array[K].                          |
//|                 If some (all) variables are unbounded, you may   |
//|                 specify very small number or -INF (latter is     |
//|                 recommended because it will allow solver to use  |
//|                 better algorithm).                               |
//|     BndU    -   upper bounds, array[K].                          |
//|                 If some (all) variables are unbounded, you may   |
//|                 specify very large number or +INF (latter is     |
//|                 recommended because it will allow solver to use  |
//|                 better algorithm).                               |
//| NOTE 1: it is possible to specify BndL[i]=BndU[i]. In this case  |
//| I-th variable will be "frozen" at X[i]=BndL[i]=BndU[i].          |
//| NOTE 2: unlike other constrained optimization algorithms, this   |
//| solver has following useful properties:                          |
//| * bound constraints are always satisfied exactly                 |
//| * function is evaluated only INSIDE area specified by bound      |
//|   constraints                                                    |
//+------------------------------------------------------------------+
void CAlglib::LSFitSetBC(CLSFitStateShell &state,double &bndl[],
                         double &bndu[])
  {
   CLSFit::LSFitSetBC(state.GetInnerObj(),bndl,bndu);
  }
//+------------------------------------------------------------------+
//| This function provides reverse communication interface           |
//| Reverse communication interface is not documented or recommended |
//| to use.                                                          |
//| See below for functions which provide better documented API      |
//+------------------------------------------------------------------+
bool CAlglib::LSFitIteration(CLSFitStateShell &state)
  {
   return(CLSFit::LSFitIteration(state.GetInnerObj()));
  }
//+------------------------------------------------------------------+
//| This family of functions is used to launcn iterations of         |
//| nonlinear fitter                                                 |
//| These functions accept following parameters:                     |
//|     func    -   callback which calculates function (or merit     |
//|                 function) value func at given point x            |
//|     grad    -   callback which calculates function (or merit     |
//|                 function) value func and gradient grad at given  |
//|                 point x                                          |
//|     hess    -   callback which calculates function (or merit     |
//|                 function) value func, gradient grad and Hessian  |
//|                 hess at given point x                            |
//|     rep     -   optional callback which is called after each     |
//|                 iteration can be null                            |
//|     obj     -   optional object which is passed to               |
//|                 func/grad/hess/jac/rep can be null               |
//| NOTES:                                                           |
//| 1. this algorithm is somewhat unusual because it works with      |
//|    parameterized function f(C,X), where X is a function argument |
//|    (we have many points which are characterized by different     |
//|    argument values), and C is a parameter to fit.                |
//|    For example, if we want to do linear fit by                   |
//|    f(c0,c1,x)=c0*x+c1, then x will be argument, and {c0,c1} will |
//|    be parameters.                                                |
//|    It is important to understand that this algorithm finds       |
//|    minimum in the space of function PARAMETERS (not arguments),  |
//|    so it needs derivatives of f() with respect to C, not X.      |
//|    In the example above it will need f=c0*x+c1 and               |
//|    {df/dc0,df/dc1}={x,1} instead of {df/dx}={c0}.                |
//| 2. Callback functions accept C as the first parameter, and X as  |
//|    the second                                                    |
//| 3. If state was created with LSFitCreateFG(), algorithm needs    |
//|    just function and its gradient, but if state wascreated with  |
//|    LSFitCreateFGH(), algorithm will need function, gradient and  |
//|    Hessian.                                                      |
//|    According to the said above, there ase several versions of    |
//|    this function, which accept different sets of callbacks.      |
//|    This flexibility opens way to subtle errors - you may create  |
//|    state with LSFitCreateFGH() (optimization using Hessian), but |
//|    call function which does not accept Hessian. So when algorithm|
//|    will request Hessian, there will be no callback to call. In   |
//|    this case exception will be thrown.        