//+------------------------------------------------------------------+
//|                                                   		TreeView |
//|                             Copyright 2006-2011, www.FXmaster.de |
//|                                                  www.FXmaster.de |
//+------------------------------------------------------------------+
#property copyright "Copyright 2006-2011, www.FXmaster.de"
#property link      "www.FXmaster.de"
#property version   "1.00"

#include "View.mqh"
#include "ScrollView.mqh"

#include "..\Tree.mqh"

#define BTN_CLEAR			0
#define BTN_COLLAPS		1
#define BTN_RESET			2
#define BTN_EXPAND		3
#define BTN_OPEN			4
#define BTN_SHOW			5
#define BTN_LABEL			6
#define BTN_CHECK			7
#define BTN_SCROLL		8
#define BTN_DESCRIPT	9
#define BTN_BREAK			10

//------------------------------------------------------------------	class CTreeView
class CTreeView: public CView
{
//--- basic functions
public:
	CTreeView(); // constructor
	~CTreeView(); // destructor
	virtual void Create(long chart, string name, int wnd, color clr, color bgclr, color selclr, int x, int y, int dx, int dy, int corn=0, int fontsize=8, string font="Arial");
	
//--- function of processing of state
public:
	CTreeCtrl	*m_tree; // pointer to the object of tree for displaying
	int m_sid; // last selected object (for highlighting)
	virtual int OnClick(string name); // processing the event of clicking on the object
	
//--- functions of displaying
public:
	int m_ndx, m_ndy; // size of margin from button for drawing
	int m_bdx, m_bdy; // size of button of nodes
	CScrollView m_scroll;
	bool m_bDesc; // show description against nodes
	
	virtual void Draw(); // refresh the view
	virtual void DrawTree(CNode *first, int xpos, int &ypos, int &up, int &dn); // redraw
	virtual void DrawNode(CNode *node, int x, int y); // draw node
	virtual void DeleteView(CNode *root=NULL, bool delparent=true); // delete all elements of the view starting from a specified one
};
//------------------------------------------------------------------	CTreeView
void CTreeView::CTreeView()
{
	m_bdx=12; m_bdy=12; // size of button of nodes
	m_ndx=m_bdx+17; m_ndy=14; // size of margins from the button
	m_sid=0; m_bDesc=false;
}
//------------------------------------------------------------------	~CTreeView
void CTreeView::~CTreeView() 
{
	ObjectDelete(m_chart, m_name);
	ObjectDelete(m_chart, m_name+".collaps");
	ObjectDelete(m_chart, m_name+".expand");
	ObjectDelete(m_chart, m_name+".clear");
	ObjectDelete(m_chart, m_name+".reset");
	ObjectDelete(m_chart, m_name+".desc");
	DeleteView();
 	m_scroll.DeleteView();
 }
//------------------------------------------------------------------	Create
void CTreeView::Create(long achart, string name, int wnd, color clr, color bgclr, color selclr, int x, int y, int dx, int dy, int corn=0, int fontsize=8, string font="Arial")
{
	CView::Create(achart, name,  wnd, clr, bgclr, selclr, x, y, dx, dy, corn, fontsize, font);
	
	x=m_x+m_dx-20; dx=16; 
	y=m_y+2*m_bdy; dy=m_dy-3*m_bdy;
	m_scroll.Create(m_chart, m_name+".scroll", m_wnd, m_selclr, m_bgclr, m_bgclr, x, y, dx, dy, m_corn, m_fontsize);
	m_scroll.m_dir=SV_VDIR;
	OnClick(m_name+string(m_sid)+".lbl");
}
//------------------------------------------------------------------	OnClick
int CTreeView::OnClick(string name) // name of object that has been clicked
{
	int start=StringLen(m_name); int len=StringFind(name, ".")-start;
	int id=(int)StringToInteger(StringSubstr(name, start, len));
	if (StringFind(name, ".clear")>=0) 
	{
		DeleteView(); m_tree.Clear(); Draw();
		ObjectSetInteger(m_chart, name, OBJPROP_STATE, false);
		return(BTN_CLEAR);
	}
	else if (StringFind(name, ".reset")>=0) 
	{
		m_tree.Reset(); Draw();
		ObjectSetInteger(m_chart, name, OBJPROP_STATE, false);
		return(BTN_RESET);
	}
	else if (StringFind(name, ".collaps")>=0) 
	{
		DeleteView(); m_tree.ExpandBy(TG_ALL, NULL, false); Draw();
		ObjectSetInteger(m_chart, name, OBJPROP_STATE, false);
		return(BTN_COLLAPS);
	}
	else if (StringFind(name, ".expand")>=0) 
	{
		m_tree.ExpandBy(TG_ALL, NULL, true); Draw();
		ObjectSetInteger(m_chart, name, OBJPROP_STATE, false);
		return(BTN_EXPAND);
	}
	else if (StringFind(name, ".desc")>=0) 
	{
		m_bDesc=!m_bDesc;
		return(BTN_DESCRIPT);
	}
	else if (StringFind(name, ".break")>=0) 
	{
		m_tree.m_bBreak=ObjectGetInteger(m_chart, name, OBJPROP_STATE);
		return(BTN_BREAK);
	}
	else if (StringFind(name, ".btn")>=0) // if the button
	{
		m_tree.ExpandIt(id, !m_tree.IsExpand(id)); // change state
		CNode *node=m_tree.FindNode(id); if (NIL(node)) return(0); // if there is an error of getting a node
		DeleteView(node); // delete of all objects of that node
		Draw(); // draw again
		return(BTN_SHOW);
	}
	else if (StringFind(name, ".lbl")>=0) // in case of click on an element of tree
	{
		m_tree.SelectBy(TG_ALL, NULL, false);// if there is no previous one, kill all
		m_tree.CheckBy(TG_ALL, NULL, false);
		if (m_sid>=0) // otherwise only one
		{
			ObjectSetInteger(m_chart, m_name+string(m_sid)+".lbl", OBJPROP_COLOR, m_clr);
			ObjectSetInteger(m_chart, m_name+string(m_sid)+".chk", OBJPROP_COLOR, m_clr);
			if (m_tree.IsCheck(m_sid)) m_tree.CheckIt(m_sid, false);
			if (m_tree.IsSelect(m_sid)) m_tree.SelectIt(m_sid, false);
		}
		// and select one new
		m_sid=id; // memorize
		ObjectSetInteger(m_chart, m_name+string(m_sid)+".lbl", OBJPROP_COLOR, m_selclr);
		ObjectSetInteger(m_chart, m_name+string(m_sid)+".chk", OBJPROP_COLOR, m_selclr);
		m_tree.CheckIt(m_sid, true);
		m_tree.SelectIt(m_sid, true);
		Draw(); // draw again
		return(BTN_LABEL);
	}
	else if (StringFind(name, ".chk")>=0) // in case of click on an element of choosing
	{
		m_tree.SelectBy(TG_ALL, NULL, false);// if there is no previous one, kill all
		m_tree.CheckBy(TG_ALL, NULL, false);
		if (m_sid>=0) if (m_tree.IsCheck(m_sid)) m_tree.CheckIt(m_sid, false); // changed state
		m_sid=id; // changed selection
		m_tree.CheckIt(m_sid, true); // changed state
		Draw(); // draw again
		return(BTN_CHECK);
	}
	else if (StringFind(name, ".scroll")>=0)
	{
		int s=m_scroll.OnClick(name); // passed further
		if (s!=0) { m_y0=(int)-m_scroll.m_value*m_ndy; Draw(); } // draw if there is an update
		return(BTN_SCROLL);
	}
	return(-1);
}
//------------------------------------------------------------------	Draw
void CTreeView::Draw()
{
	if (!IsOpenView()) return; // if a view is not created, exit

	SetRectLbl(m_chart, m_name+".rect", m_wnd, "", m_clr, m_bgclr, m_x, m_y, m_dx, m_dy, m_corn, m_fontsize, m_font, true);
	int ax=int(m_x+m_dx-1.5*m_bdx); int adx=int(1.2*m_bdx); int i=-1;

	i++; SetButton(m_chart, m_name+".collaps", 0, "-", m_bgclr, m_selclr, ax-i*adx, int(m_y+0.5*m_bdy), m_bdx, m_bdy, 0, 7, "Tahoma", false);
	i++; SetButton(m_chart, m_name+".expand", 0, "+", m_bgclr, m_selclr, ax-i*adx, int(m_y+0.5*m_bdy), m_bdx, m_bdy, 0, 7, "Tahoma", false);
	if (m_tree.m_mode==1)
	{
		i++; SetButton(m_chart, m_name+".clear", 0, "X", m_bgclr, m_selclr, ax-i*adx, int(m_y+0.5*m_bdy), m_bdx, m_bdy, 0, 7, "Tahoma", false);
		i++; SetButton(m_chart, m_name+".reset", 0, "0", m_bgclr, m_selclr, ax-i*adx, int(m_y+0.5*m_bdy), m_bdx, m_bdy, 0, 7, "Tahoma", false);
		i++; SetButton(m_chart, m_name+".desc", 0, "i", m_bgclr, clrDarkOliveGreen, ax-i*adx, int(m_y+0.5*m_bdy), m_bdx, m_bdy, 0, 7, "Tahoma", m_bDesc); 
		i++; SetButton(m_chart, m_name+".break", 0, "D", m_bgclr, clrRed, ax-i*adx, int(m_y+0.5*m_bdy), m_bdx, m_bdy, 0, 7, "Tahoma", m_tree.m_bBreak); 
	}

	int show=0, up=0, dn=0; 
	DrawTree(m_tree.m_root, 1, show, up, dn); // drew tree
	
	if (up<=0 && dn<=0) m_scroll.DeleteView(); // removed the scrollbar for redrawing
	else
	{
		m_scroll.m_max=up+dn; m_scroll.m_value=up;
		m_scroll.m_x=m_x+m_dx-20; m_scroll.m_dx=16; 
		m_scroll.m_y=m_y+2*m_bdy; m_scroll.m_dy=m_dy-3*m_bdy+8;
		m_scroll.Draw(); // drew the scrollbar
	}
}
//------------------------------------------------------------------	DrawTree
void CTreeView::DrawTree(CNode *root, int xpos, int &ypos, int &up, int &dn)
{
	if (xpos==1) { root=m_tree.m_root; ypos=0; } // if we start from the first element, take the root and the left position
	if (root==NULL) { root=m_tree.m_root; xpos=1; } // if the specified node is not the first one, take the root and upper position
	if (NIL(root)) return; // validity of pointer
	string name=m_name+string(root.m_id);

	if (!IsOpenView()) return; // if a view is not created, exit

	int x=m_x0+m_x+5+(xpos-1)*m_bdx;
	int y=m_y0+m_y+5+ypos*m_ndy;
	
	bool ishide=(y<=m_y || y>=m_y+m_dy-m_ndy);
	if (root.m_break && ishide)
	{
		while (y<m_y) { m_y0+=m_ndy; y=m_y0+m_y+5+ypos*m_ndy; }// moved down
		while (y>m_y+m_dy-m_ndy) { m_y0-=m_ndy; y=m_y0+m_y+5+ypos*m_ndy; }// moved up
		up=0; dn=0;
		DrawTree(m_tree.m_root, 1, ypos, up, dn); // drew the entire tree again
	}
	
	if (!ishide) DrawNode(root, x, y); // if it doesn't go out of visible area
	else 
	{
		ObjectDelete(m_chart, name+".lbl"); ObjectDelete(m_chart, name+".btn"); 
		ObjectDelete(m_chart, name+".chk"); ObjectDelete(m_chart, name+".dbg"); 
		if (y<m_y) up++; else dn++;
	}
	
	if (!root.m_expand) return; // skip if not expanded
	for (int i=0; i<ArraySize(root.m_next); i++)
	{
		if (NIL(root.m_next[i])) continue;
		ypos++; // increased position
		DrawTree(root.m_next[i], xpos+1, ypos, up, dn); // drew the next branch
	}
}
//------------------------------------------------------------------	DrawNode
void CTreeView::DrawNode(CNode *node, int x, int y)
{
	if (NIL(node)) return;
	string name=m_name+string(node.m_id);
	string txt=node.m_text; 
	StringTrimLeft(node.m_desc); // clear the string from the left
	if (m_bDesc && node.m_desc!="") txt+="  |  "+node.m_desc; // if we need to display description
	// determine color
	color clr=node.m_select ? m_selclr : m_clr;
	if (node.m_break) clr=clrRed;

	// display text for a node
	SetLabel(m_chart, name+".lbl", m_wnd, txt, clr, x+m_ndx, y-1, m_corn, m_fontsize, m_font);
	
	// mark of node
	string state;
	if (!node.m_break)
	{
		state=node.m_check ? CharToString(110) : CharToString(112);
		SetLabel(m_chart, name+".chk", m_wnd, state, clr, x+m_ndx-m_bdx, y, m_corn, m_fontsize, "Wingdings");
	}
	else ObjectDelete(m_chart, name+".chk");
	
	// button of node expanding
	state=node.m_expand ? "-" : "+";
	if (ArraySize(node.m_next)>0) 	
		SetButton(m_chart, name+".btn", m_wnd, state, m_clr, m_bgclr, x, y, m_bdx, m_bdy, m_corn, m_fontsize, m_font, node.m_expand);
	else ObjectDelete(m_chart, name+".btn");
	
	// debug button (its name belongs to the entire tree since it's only one)
	if (node.m_break) 
		SetButton(m_chart, name+".dbg", m_wnd, "*", clr, clr, x+m_ndx-m_bdx-2, y, m_bdx, m_bdy, m_corn, m_fontsize, m_font, false);
	else ObjectDelete(m_chart, name+".dbg");
}
//------------------------------------------------------------------	DeleteView
void CTreeView::DeleteView(CNode *root=NULL, bool delparent=true)
{
	CView::DeleteView(); // method of base class
	if (root==NULL) root=m_tree.m_root; if (NIL(root)) return;
	if(delparent) // delete node of the parent
	{
		string name=m_name+string(root.m_id);
		ObjectDelete(m_chart, name+".btn");
		ObjectDelete(m_chart, name+".lbl");
		ObjectDelete(m_chart, name+".chk");
	}
	for (int i=0; i<ArraySize(root.m_next); i++)
		if (!NIL(root.m_next[i])) DeleteView(root.m_next[i], true);
}