//+------------------------------------------------------------------+
//|                                                   	  TraceView |
//|                             Copyright 2006-2011, www.FXmaster.de |
//|                                                  www.FXmaster.de |
//+------------------------------------------------------------------+
#property copyright "Copyright 2006-2011, www.FXmaster.de"
#property link      "www.FXmaster.de"
#property version   "1.00"

#include "..\Trace.mqh"
#include "PropertyView.mqh"

//------------------------------------------------------------------	class CTreeView
class CTraceView: public CView
{
//--- basic functions
public:
	CTraceView() { }; // constructor
	~CTraceView() { Deinit(); } // destructor
	void Deinit(); // full deinitialization of view
	virtual void Create(long chart); // create and activate view

//--- function of processing of state
public:
	int m_hagent; // handler of indicator-agent used for sending messages
	CTraceCtrl* m_trace; // pointer to created tracer
	CTreeView* m_viewstack; // tree for displaying of stack
	CPropertyView* m_viewprop; // tree for displaying of node properties
	CTreeView* m_viewfile; // tree for displaying of stack with grouping by files
	CTreeView* m_viewclass; // tree for displaying of stack with grouping by classes
	void OnTimer(); // handler of timer
	void OnChartEvent(const int , const long& , const double& , const string& ); // handler of event

//--- functions of displaying
public:
	virtual void Draw(); // updated objects
	virtual void DeleteView(); // deleted view

	void UpdateInfo(CNode *node, bool bclear); // displaying window of detailed information
	string TimeSeparate(long time); // special function for transformation of time into string
};
//------------------------------------------------------------------	~CTraceView
void CTraceView::Deinit()
{ _IN("");
	EventKillTimer();
	delete m_viewclass;
	delete m_viewfile;
	delete m_viewprop;
	delete m_viewstack;
	ChartIndicatorDelete(m_chart, 0, "!TraceAgent");
	m_hagent=INVALID_HANDLE;
	ChartClose(m_chart);
}
//------------------------------------------------------------------	Create
void CTraceView::Create(long chart)
{ _IN("");
	m_chart=ChartOpen(Symbol(), Period()); // open the window of chart for working
	
	// attach there the agent for sending messages to the current window
	m_hagent=iCustom(Symbol(), Period(), "!TraceAgent", chart);
	if (m_hagent==INVALID_HANDLE) return;
	ChartIndicatorAdd(m_chart, 0, m_hagent);
	
	HideChart(m_chart, LightSteelBlue); // hide the chart view and elements
	int heigh=(int)ChartGetInteger(m_chart, CHART_HEIGHT_IN_PIXELS)-m_dy-5; // take chart parameters
	int width=(int)ChartGetInteger(m_chart, CHART_WIDTH_IN_PIXELS)-10; // take chart parameters
	// create four trees for displaying of tracing and its groups
	double k=1.7, kx=width/(k+2);
	int dh=int(heigh/2.5);
	int x=5, y=m_dy+5, dx=int(k*kx);
	// main stack
	m_viewstack=new CTreeView; m_viewstack.m_tree=m_trace.m_stack; // attach the graph to the view
	m_viewstack.Create(m_chart, "viewstack", 0, Black, White, DarkBlue, x, y, dx, heigh-dh-3, 0, 8, "Tahoma"); // create area
	m_viewstack.m_tree.SortBy(TG_TEXT, true);
	m_viewstack.Draw();
	// information about a node
	m_viewprop=new CPropertyView; m_viewprop.m_tree=m_trace.m_prop; // attach graph to the view
	m_viewprop.Create(m_chart, "viewprop", 0, Black, White, DarkBlue, x, y+heigh-dh, dx, dh, 0, 8, "Tahoma"); // create area
	m_viewprop.Draw();
	// grouping by classes
	x+=dx+3; dx=(int)kx;
	m_viewclass=new CTreeView; m_viewclass.m_tree=m_trace.m_class; // attach graph to the view
	m_viewclass.Create(m_chart, "viewclass", 0, Black, White, DarkBlue, x, y, dx, heigh, 0, 8, "Tahoma"); // create area
	m_viewclass.m_tree.SortBy(TG_TEXT, true);
	m_viewclass.Draw();
	// grouping by files
	x+=dx+3; dx=(int)kx;
	m_viewfile=new CTreeView; m_viewfile.m_tree=m_trace.m_file; // attach graph to the view
	m_viewfile.Create(m_chart, "viewfile", 0, Black, White, DarkBlue, x, y, dx, heigh, 0, 8, "Tahoma"); // create area
	m_viewfile.m_tree.SortBy(TG_TEXT, true);
	m_viewfile.Draw();
	
	EventSetTimer(2);
	ChartRedraw(m_chart);
}
//------------------------------------------------------------------	OnTimer
void CTraceView::OnTimer()
{ _IN("");
	_WATCH("time", TTS(TimeLocal()));
	Draw(); // redraw trees
}
//------------------------------------------------------------------	OnChartEvent
void CTraceView::OnChartEvent(const int aid, const long& lparam, const double& dparam, const string& sparam)
{ _IN("");
	if (aid<CHARTEVENT_CUSTOM) return; // working only with custom events
	int id=aid-CHARTEVENT_CUSTOM;
	_WATCH("Event", EnumToString(ENUM_CHART_EVENT(id)));
	
	// analyze a click on an object
	if (id==CHARTEVENT_OBJECT_CLICK)
	{ _IN2("");
		if (StringFind(sparam, "viewstack")>=0) // at the main stack
		{
			int btn=m_viewstack.OnClick(sparam); 
			if (btn==BTN_CLEAR) 
			{
				m_viewclass.DeleteView(); m_viewclass.m_tree.Clear(); m_viewclass.Draw();
				m_viewfile.DeleteView(); m_viewfile.m_tree.Clear(); m_viewfile.Draw();
				m_viewprop.DeleteView();
			}
			if (m_viewstack.m_sid<=0) return;
			CNode *node=m_viewstack.m_tree.FindNode(m_viewstack.m_sid);	if (NIL(node)) return;
			UpdateInfo(node, true);
			m_viewclass.m_tree.SelectBy(TG_ALL, NULL, false); m_viewclass.m_tree.CheckBy(TG_ALL, NULL, false);
			m_viewclass.m_tree.SelectBy(TG_CLASS|TG_FUNC, node, true); m_viewclass.m_tree.CheckBy(TG_CLASS|TG_FUNC, node, true);
			m_viewclass.m_sid=-1; m_viewclass.Draw();
			
			m_viewfile.m_tree.SelectBy(TG_ALL, NULL, false); m_viewfile.m_tree.CheckBy(TG_ALL, NULL, false); 
			m_viewfile.m_tree.SelectBy(TG_FILE|TG_FUNC, node, true); m_viewfile.m_tree.CheckBy(TG_FILE|TG_FUNC, node, true);
			m_viewfile.m_sid=-1; m_viewfile.Draw();
			ChartRedraw(m_chart);
		}
		if (StringFind(sparam, "viewclass")>=0) // at the tree of classes
		{
			int btn=m_viewclass.OnClick(sparam); 
			if (btn==BTN_CLEAR) 
			{	
				m_viewstack.m_tree.GroupBy(TG_CLASS, m_viewclass.m_tree); m_viewclass.m_tree.SortBy(TG_TEXT, true);
				m_viewclass.Draw();
			}
			m_viewprop.DeleteView();
			if (m_viewclass.m_sid<=0) return;
			CNode *node=m_viewclass.m_tree.FindNode(m_viewclass.m_sid);	if (NIL(node)) return;
			m_viewstack.m_tree.SelectBy(TG_ALL, NULL, false); m_viewstack.m_tree.CheckBy(TG_ALL, NULL, false);
			m_viewstack.m_tree.SelectBy(TG_CLASS|TG_FUNC, node, true); m_viewstack.m_tree.CheckBy(TG_CLASS|TG_FUNC, node, true);
			m_viewstack.m_sid=-1; m_viewstack.Draw();

			m_viewfile.m_tree.SelectBy(TG_ALL, NULL, false); m_viewfile.m_tree.CheckBy(TG_ALL, NULL, false);
			m_viewfile.m_tree.SelectBy(TG_FILE|TG_FUNC, node, true); m_viewfile.m_tree.CheckBy(TG_FILE|TG_FUNC, node, true);
			m_viewfile.m_sid=-1; m_viewfile.Draw();
			ChartRedraw(m_chart);
		}
		if (StringFind(sparam, "viewfile")>=0) // at the tree of files
		{
			int btn=m_viewfile.OnClick(sparam); 
			if (btn==BTN_CLEAR) 
			{ 
				m_viewstack.m_tree.GroupBy(TG_FILE, m_viewfile.m_tree); m_viewfile.m_tree.SortBy(TG_TEXT, true); 
				m_viewfile.Draw();
			}
			m_viewprop.DeleteView();
			if (m_viewfile.m_sid<=0) return;
			CNode *node=m_viewfile.m_tree.FindNode(m_viewfile.m_sid);	if (NIL(node)) return;
			m_viewstack.m_tree.SelectBy(TG_ALL, NULL, false); m_viewstack.m_tree.CheckBy(TG_ALL, NULL, false);
			m_viewstack.m_tree.SelectBy(TG_FILE|TG_FUNC, node, true); m_viewstack.m_tree.CheckBy(TG_FILE|TG_FUNC, node, true);
			m_viewstack.m_sid=-1; m_viewstack.Draw();

			m_viewclass.m_tree.SelectBy(TG_ALL, NULL, false); m_viewclass.m_tree.CheckBy(TG_ALL, NULL, false);
			m_viewclass.m_tree.SelectBy(TG_FILE|TG_FUNC, node, true); m_viewclass.m_tree.CheckBy(TG_FILE|TG_FUNC, node, true);
			m_viewclass.m_sid=-1; m_viewclass.Draw();
			ChartRedraw(m_chart);
		}
		if (StringFind(sparam, "viewprop")>=0) // at the tree of node properties
		{
			m_viewprop.OnClick(sparam); ChartRedraw(m_chart);
		}
	}
	if (id==CHARTEVENT_OBJECT_ENDEDIT) 
	{
		if (StringFind(sparam, "viewprop")>=0) // at the tree of node properties
		{
			if (StringFind(sparam, ".prop")>=0) 
			{
				int n=(int)StringToInteger(ObjectGetString(m_chart, sparam, OBJPROP_TEXT)); // get
				m_viewstack.m_tree.SetDataBy(TG_BRKUSE, m_viewstack.m_sid, n); // set
			}
		}
	}
	if (id==CHARTEVENT_CHART_CHANGE) 
	{
		Draw(); // size chart windows has changed
	}
}
//------------------------------------------------------------------	Draw
void CTraceView::Draw()
{ _IN("");
	int heigh=(int)ChartGetInteger(m_chart, CHART_HEIGHT_IN_PIXELS)-m_dy-5; // get parameters
	int dh=int(heigh/2.5);
	m_viewstack.m_dy=heigh-dh-3; m_viewprop.m_dy=dh; m_viewprop.m_y=m_viewstack.m_y+heigh-dh;
	m_viewclass.m_dy=heigh; m_viewfile.m_dy=heigh;

	int width=(int)ChartGetInteger(m_chart, CHART_WIDTH_IN_PIXELS)-10; // take chart parameters
	double k=1.7, kx=width/(k+2);
	int x=5, dx=int(k*kx); m_viewstack.m_dx=dx; m_viewprop.m_dx=dx; 
	x+=dx+3; dx=int(kx); m_viewclass.m_x=x; m_viewclass.m_dx=dx; 
	x+=dx+3; dx=int(kx); m_viewfile.m_x=x; m_viewfile.m_dx=dx;
	m_viewstack.Draw(); m_viewclass.Draw(); m_viewfile.Draw(); // redraw trees
	CNode *node=m_viewstack.m_tree.FindNode(m_viewstack.m_sid); // display information about the node
	UpdateInfo(node, false);
	ChartRedraw(m_chart);
}
//------------------------------------------------------------------	Draw
void CTraceView::DeleteView()
{ _IN("");
	m_viewstack.DeleteView(); m_viewprop.DeleteView();
	m_viewclass.DeleteView(); m_viewfile.DeleteView(); // deleted the view
}

//------------------------------------------------------------------	UpdateInfo
void CTraceView::UpdateInfo(CNode *node, bool bclear)
{
	if (NIL(node)) return;
	if (NIL(m_viewprop.m_tree.m_root)) return;
	
	//1. Copy several details of the node to the root node of the tree to make the processing of buttons easier
	CNode *root=m_viewprop.m_tree.m_root;
	root.m_text=node.m_text; // node name
	root.m_path=node.m_path; // file name
	root.m_line=node.m_line; // number of row in the file

	int id, id2;
	//2. Create tree by a node
	id=m_viewprop.m_tree.AddNode(0, "Run-time information");
	id2=m_viewprop.m_tree.AddNode(id, "Number of calls"); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, string(node.m_uses));
	id2=m_viewprop.m_tree.AddNode(id, "DebugBreak after"); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, node.m_brkuse<=0 ? "<dsbl>":string(node.m_brkuse)); 
	m_viewprop.m_tree.SetDataBy(TG_EDIT, id2, true);
	id2=m_viewprop.m_tree.AddNode(id, "Total time");  m_viewprop.m_tree.SetDataBy(TG_DESC, id2, TimeSeparate(node.m_tick));
	id2=m_viewprop.m_tree.AddNode(id, "Last entering"); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, TTS(node.m_last));
	id2=m_viewprop.m_tree.AddNode(id, "Additional"); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, node.m_desc);

	id=m_viewprop.m_tree.AddNode(0, "Position");
	id2=m_viewprop.m_tree.AddNode(id, "File"); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, node.m_file);
	id2=m_viewprop.m_tree.AddNode(id, "Row number"); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, string(node.m_line));
	id2=m_viewprop.m_tree.AddNode(id, "Class"); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, node.m_class);
	id2=m_viewprop.m_tree.AddNode(id, "Function"); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, node.m_func);
	id=m_viewprop.m_tree.AddNode(0, "Watch parameters");
	if (bclear)
	{
		CNode *w=m_viewprop.m_tree.FindNode(id); // get watch node
		m_viewprop.DeleteView(w, false); // delete its view
		m_viewprop.m_tree.Clear(w); // delete the array of sub-nodes
	}
	// drew watch
	int n=ArraySize(node.m_watch);
	for (int i=0; i<n; i++)
	{
		id2=m_viewprop.m_tree.AddNode(id, node.m_watch[i].m_name); m_viewprop.m_tree.SetDataBy(TG_DESC, id2, node.m_watch[i].m_val);
	}
	m_viewprop.Draw();
}
//------------------------------------------------------------------	TimeSeparate
string CTraceView::TimeSeparate(long time)
{
	long sec=1000; long min=sec*60; long hour=min*60; long day=hour*24;
	string txt="";
	long d=long(time/day); if (d>0) txt+=string(d)+" days ";
	long h=long(time/hour)-d*day; if (h>9) txt+=string(h)+":"; else txt+="0"+string(h)+":";
	long m=long(time/min)-(h*hour+d*day); if (m>9) txt+=string(m)+":"; else txt+="0"+string(m)+":";
	long s=long(time/sec)-(m*min+h*hour+d*day); if (s>9) txt+=string(s)+"."; else txt+="0"+string(s)+".";
	long ms=time-(s*sec+m*min+h*hour+d*day); txt+=string(ms);
	return(txt);
}

extern CTraceView* m_traceview; // the only instance for displaying
