//+------------------------------------------------------------------+
//|                                              log4mqlm_sample.mq4 |
//|                                    Copyright 2021, Jens Lippmann |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2021, Jens Lippmann"
#property link      "https://www.mql5.com/en/users/lippmaje"
#property version   "1.00"
#property strict

/*
    This is a usage example of Log4mql(mini).
    It will enable you to log messages in a standardized, efficient manner.

    Functions:
    ---
    PRINT        - Prints its message if log level is set at least to INFO (= normal logging behavior).
                   Supports up to 9 arguments - PRINT("1",2,etc) - if you need more, expand the templates.
    PRINTF       - Like PRINT but prints a formatted message. Same rules as known from PrintFormat apply.
    WARN         - Prints its message if log level is set at least to WARN.
    WARNF        - Like WARN but prints a formatted message.
    ERROR        - Prints its message if log level is at least to ERROR. Adds _LastError description if set.
    ERRORF       - Like ERROR but prints a formatted message.
    FATAL        - Like ERROR but indicates termination. You should call ExpertRemove() after this.
    FATALF       - Like FATAL but prints a formatted message.
    DEBUG,DEBUGF - Messages for DEBUG level.
    TRACE,TRACEF - Messages for TRACE (more verbose than DEBUG, for function entries, calculations etc).

    In case you need it:
    bool   L4mq.logged           - true if the last event was logged.
    string L4mq.LastMessage()    - Get the last message that has been logged.
    int    L4mq.LastLevel()      - Get the last severity level that has been set.
    int    L4mq.Level()          - Get the current severity level of the logger.
    string L4mq.LevelAsString()  - Get the severity level of the logger as string.
    void   L4mq.SetLevel(int)    - Set the severity level for the logger.
    ulong  L4mq.SeqNr()          - Get the sequence number of the last logged event.
    -
    string Log4mql::GetErrorDescription(int err)  - Get the description of an error code like _LastError.
    string Log4mql::GetUninitReason(int reason)   - Get the description of the uninit reason.
    string Log4mql::LoglevelToString(int level)   - Convert a log level to a descriptive string.

    Note:
      Use logging whenever possible. Proper logging is very helpful to track possible bugs.
*/

#include <log4mqlm.mqh>

//+------------------------------------------------------------------+
//| Inputs                                                           |
//|  add your inputs here                                            |
//+------------------------------------------------------------------+
sinput ENUM_LOG_LEVEL InpLoglevel=LOGLEVEL_INFO; // Log Level (Info = normal logging)

//+------------------------------------------------------------------+
//| Global variables                                                 |
//+------------------------------------------------------------------+
Log4mql L4mq(InpLoglevel);

//+------------------------------------------------------------------+
//| Some function to demonstrate logging                             |
//+------------------------------------------------------------------+
void SomeFunction(double bid,double ask,double stoploss)
  {
   PRINTF("bid:%g ask:%g",bid,ask);              // 2021.01.18 00:01:00  log4mqlm_sample EURUSD,M1: SomeFunction: bid:1.2082 ask:1.20828
   DEBUGF("sl:%g",stoploss);                     // 2021.01.18 00:01:00  log4mqlm_sample EURUSD,M1: SomeFunction(log4mqlm_sample.mq4:65): sl:1.20816
   if(MathAbs(bid-stoploss)<10*_Point)
      WARNF("stoploss too tight: %g",stoploss);  // 2021.01.18 00:01:00  log4mqlm_sample EURUSD,M1: WARN SomeFunction(log4mqlm_sample.mq4:67): stoploss too tight: 1.20816
  }

//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
   TRACE("initializing");
   PRINT(" Log level: ",L4mq.LevelAsString());
   return INIT_SUCCEEDED;
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
   TRACE("closing, reason: ",L4mq.GetUninitReason());
   string file="some file";
   int fh=FileOpen(file,FILE_READ);
   if(fh==INVALID_HANDLE)
      ERRORF("open '%s' failed",file);           // 2021.01.18 00:01:00  log4mqlm_sample EURUSD,M1: ERROR OnDeinit(log4mqlm_sample.mq4:88): open 'some file' failed,  error 5004 - Cannot open file
   TRACE("exit");
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
   MqlTick tick;
   SymbolInfoTick(Symbol(),tick);
   SomeFunction(tick.bid,tick.ask,tick.bid-4*_Point);
   ExpertRemove();
  }
//+------------------------------------------------------------------+
