//+------------------------------------------------------------------+
//|                                   itimeseries_nicholishen_v2.mqh |
//|                                                      nicholishen |
//|                                   www.reddit.com/u/nicholishenFX |
//+------------------------------------------------------------------+
#property copyright  "nicholishen"
#property link       "www.reddit.com/u/nicholishenFX"
#include <Arrays\ArrayObj.mqh>
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
enum ENUM_SERIES_MODE
  {
   MODE_OPEN   = 0,
   MODE_LOW    = 1,
   MODE_HIGH   = 2,
   MODE_CLOSE  = 3,
   MODE_VOLUME = 4,
   MODE_TIME   = 5
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CRatesArray : public CObject
  {
protected:
   string            m_symbol;
   ENUM_TIMEFRAMES   m_period;
   MqlRates          m_rates[];
   int               m_ratesSize;
   int               m_indexes[];
   int               m_totalIndex;
   int               m_trimExcess;
   datetime          m_lastWarn;
   int               m_lastBars;
   uint              m_lastTick;
   int               m_periodSec;
   int               m_periodSecAdj;

public:

   int               BarsTotal()const{  return m_lastBars;      } //                                   
   int Shift         (  datetime, bool          ); //
   double Open       (  const int               ); //
   double High       (  const int               ); //
   double Low        (  const int               ); //
   double Close      (  const int               ); //
   datetime Time     (  const int               ); //
   long              Volume(const int);
   int               Highest(ENUM_SERIES_MODE,
                             int,
                             int);
   int               Lowest(ENUM_SERIES_MODE,
                            int,
                            int);

                     CRatesArray(const string,
                                                   const ENUM_TIMEFRAMES);
   bool              IsMatching(const string,
                                const ENUM_TIMEFRAMES);
   bool              Refresh();
protected:
   int               TrimTime(datetime time);
   int               TrimToMinutes(int time);
   int               TrimToHours(int time);
   int               TrimToDays(int time);
  };
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
class CiTimeSeries : public CArrayObj
  {
protected:
   string            m_chartSymbol;
   ENUM_TIMEFRAMES   m_chartPeriod;
   bool              m_autoRefresh;
   ENUM_TIMEFRAMES   m_allowed[];
   bool              m_periodWarn;

   CRatesArray      *m_lastRefPointer;
   string            m_lastRefSymbol;
   ENUM_TIMEFRAMES   m_lastRefPeriod;

   static int        m_objectCount;

public:
                     CiTimeSeries();

   static int        ObjectCount() { return m_objectCount;}

   bool Init         (  string            symbol      = NULL,
                      ENUM_TIMEFRAMES   period      = PERIOD_CURRENT,
                      const bool        autoRefresh = true   );
   bool              Init(string            symbol,
                          ENUM_TIMEFRAMES   &period[],
                          const bool        autoRefresh = true   );
   bool InitAllPeriods (  string            symbol      = NULL,
                        const bool        autoRefresh = true   );
   void              AutoRefresh(const bool ref) { m_autoRefresh=ref;     }
   bool              AutoRefresh() const { return m_autoRefresh;          }
   // data access methods  
   int               Shift(string            symbol,
                           ENUM_TIMEFRAMES   period,
                           const datetime    time,
                           const bool        exact=false);
   int               Shift(const datetime    time,
                           const bool        exact=false);

   datetime          Time(string            symbol,
                          ENUM_TIMEFRAMES   period,
                          const int         shift);
   datetime          Time(const int         shift);

   int               BarsTotal(string            symbol,
                               ENUM_TIMEFRAMES   period);
   int               BarsTotal(void);

   double            Open(string            symbol,
                          ENUM_TIMEFRAMES   period,
                          const int         shift);
   double            Open(const int         shift);

   double            High(string            symbol,
                          ENUM_TIMEFRAMES   period,
                          const int         shift);
   double            High(const int         shift);

   double            Low(string            symbol,
                         ENUM_TIMEFRAMES   period,
                         const int         shift);
   double            Low(const int         shift);

   double            Close(string            symbol,
                           ENUM_TIMEFRAMES   period,
                           const int         shift);
   double            Close(const int         shift);

   long              Volume(string            symbol,
                            ENUM_TIMEFRAMES   period,
                            const int         shift);
   long              Volume(const int         shift);

   int               Highest(string            symbol,
                             ENUM_TIMEFRAMES   period,
                             ENUM_SERIES_MODE  mode  = MODE_HIGH,
                             int               count = WHOLE_ARRAY,
                             int               start = 0 );
   int Highest       (  ENUM_SERIES_MODE  mode  = MODE_HIGH,
                      int               count = WHOLE_ARRAY,
                      int               start = 0 );
   int               Lowest(string            symbol,
                            ENUM_TIMEFRAMES   period,
                            ENUM_SERIES_MODE  mode  = MODE_LOW,
                            int               count = WHOLE_ARRAY,
                            int               start = 0 );
   int Lowest        (  ENUM_SERIES_MODE  mode  = MODE_LOW,
                      int               count = WHOLE_ARRAY,
                      int               start = 0 );
   bool              Refresh();
   CRatesArray      *GetArrayObjPointer(string            symbol,
                                        ENUM_TIMEFRAMES   period);
protected:
   void              CorrectInput(string            &symbol,
                                  ENUM_TIMEFRAMES   &period);
   CRatesArray      *operator[](const int         index)
   const { return(CRatesArray*)this.At(index); }

  };
int CiTimeSeries::m_objectCount=0;
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+


CiTimeSeries::CiTimeSeries(void):m_periodWarn(false),m_autoRefresh(true)
  {
   m_chartPeriod = ::Period();
   m_chartSymbol = ::Symbol();
   ENUM_TIMEFRAMES allowed[]=
     {
      PERIOD_M1,PERIOD_M2,PERIOD_M3,PERIOD_M4,PERIOD_M5,PERIOD_M6,
      PERIOD_M10,PERIOD_M12,PERIOD_M15,PERIOD_M20,PERIOD_M30,
      PERIOD_H1,PERIOD_H2,PERIOD_H3,PERIOD_H4,PERIOD_H6,PERIOD_H8,PERIOD_H12,
      PERIOD_D1,PERIOD_W1,PERIOD_MN1 
     };
   ArrayCopy(m_allowed,allowed);
   m_objectCount++;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CiTimeSeries::Init(string symbol,ENUM_TIMEFRAMES &period[],const bool autoRefresh=true)
  {
   bool res=true;
   int total= ArraySize(period);
   for(int i=0;i<total;i++)
     {
      if(!Init(symbol,period[i],autoRefresh))
        {
         Print(__FUNCTION__," Failed to initialize.");
         res=false;
        }
     }
   return res;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CiTimeSeries::Init(string symbol=NULL,ENUM_TIMEFRAMES period=PERIOD_CURRENT,const bool autoRefresh=true)
  {
   CorrectInput(symbol,period);
   m_autoRefresh=autoRefresh;

   if(GetArrayObjPointer(symbol,period)==NULL)
      return false;
   else
      return true;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CiTimeSeries::InitAllPeriods(string symbol=NULL,const bool autoRefresh=true)
  {
   return Init(symbol,m_allowed,autoRefresh);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CRatesArray *CiTimeSeries::GetArrayObjPointer(string symbol,ENUM_TIMEFRAMES period)
  {
   if(symbol==NULL || period==PERIOD_CURRENT)
      CorrectInput(symbol,period);
   if(m_lastRefSymbol==symbol && m_lastRefPeriod==period && m_lastRefPointer!=NULL)
     {
      if(m_autoRefresh)
         m_lastRefPointer.Refresh();
      return m_lastRefPointer;
     }

   m_lastRefPointer=NULL;
   for(int i=0;i<Total();i++)
     {
      if(this[i].IsMatching(symbol,period))
        {
         if(m_autoRefresh)
            this[i].Refresh();
         m_lastRefPointer=this[i];
        }
     }
   if(m_lastRefPointer==NULL)
     {
      m_lastRefPointer=new CRatesArray(symbol,period);
      Add(m_lastRefPointer);
     }
   if(m_lastRefPointer!=NULL)
     {
      m_lastRefPeriod = period;
      m_lastRefSymbol = symbol;
      return m_lastRefPointer;
     }
   else
      return NULL;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CiTimeSeries::Refresh(void)
  {
   bool res = true;
   for(int i=0;i<Total();i++)
      if(!this[i].Refresh())
         res=false;
   return res;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void CiTimeSeries::CorrectInput(string &symbol,ENUM_TIMEFRAMES &period)
  {
   symbol = symbol == NULL ? m_chartSymbol : symbol;
   period = period == PERIOD_CURRENT ? m_chartPeriod : period;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CiTimeSeries::BarsTotal(void)
  {
   return BarsTotal(m_chartSymbol,m_chartPeriod);

  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CiTimeSeries::BarsTotal(string symbol,ENUM_TIMEFRAMES period)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.BarsTotal();
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
datetime CiTimeSeries::Time(const int shift)
  {
   return Time(m_chartSymbol,m_chartPeriod,shift);
  }
//+------------------------------------------------------------------+
datetime CiTimeSeries::Time(string symbol,ENUM_TIMEFRAMES period,const int shift)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.Time(shift);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CiTimeSeries::Open(const int shift)
  {
   return Open(m_chartSymbol,m_chartPeriod,shift);
  }
//+------------------------------------------------------------------+
double CiTimeSeries::Open(string symbol,ENUM_TIMEFRAMES period,const int shift)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.Open(shift);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CiTimeSeries::High(const int shift)
  {
   return High(m_chartSymbol,m_chartPeriod,shift);
  }
//+------------------------------------------------------------------+
double CiTimeSeries::High(string symbol,ENUM_TIMEFRAMES period,const int shift)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.High(shift);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CiTimeSeries::Low(const int shift)
  {
   return Low(m_chartSymbol,m_chartPeriod,shift);
  }
//+------------------------------------------------------------------+
double CiTimeSeries::Low(string symbol,ENUM_TIMEFRAMES period,const int shift)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.Low(shift);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CiTimeSeries::Close(const int shift)
  {
   return Close(m_chartSymbol,m_chartPeriod,shift);
  }
//+------------------------------------------------------------------+
double CiTimeSeries::Close(string symbol,ENUM_TIMEFRAMES period,const int shift)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.Close(shift);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
long CiTimeSeries::Volume(const int shift)
  {
   return Volume(m_chartSymbol,m_chartPeriod,shift);
  }
//+------------------------------------------------------------------+
long CiTimeSeries::Volume(string symbol,ENUM_TIMEFRAMES period,const int shift)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.Volume(shift);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+

int CiTimeSeries::Lowest(ENUM_SERIES_MODE mode=MODE_LOW,int count=WHOLE_ARRAY,int start=0)
  {
   return Lowest(m_chartSymbol,m_chartPeriod,mode,count,start);
  }
//+------------------------------------------------------------------+
int CiTimeSeries::Lowest(string symbol,ENUM_TIMEFRAMES period,ENUM_SERIES_MODE mode=MODE_LOW,int count=WHOLE_ARRAY,int start=0)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.Lowest(mode,count,start);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CiTimeSeries::Highest(ENUM_SERIES_MODE mode=MODE_HIGH,int count=WHOLE_ARRAY,int start=0)
  {
   return Highest(m_chartSymbol,m_chartPeriod,mode,count,start);
  }
//+------------------------------------------------------------------+
int CiTimeSeries::Highest(string symbol,ENUM_TIMEFRAMES period,ENUM_SERIES_MODE mode=MODE_HIGH,int count=WHOLE_ARRAY,int start=0)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.Highest(mode,count,start);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CiTimeSeries::Shift(const datetime time,const bool exact=false)
  {
   return Shift(m_chartSymbol,m_chartPeriod,time,exact);
  }
//+------------------------------------------------------------------+
int CiTimeSeries::Shift(string symbol,ENUM_TIMEFRAMES period,const datetime time,const bool exact=false)
  {
   CRatesArray *timeArr=GetArrayObjPointer(symbol,period);
   if(timeArr==NULL)
     {
      if(!m_periodWarn)
        {
         Print(__FUNCTION__,__LINE__," Error: Period not initialized,");
         m_periodWarn=true;
        }
      return -1;
     }
   return timeArr.Shift(time,exact);
  }
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~+
//+------------------------------------------------------------------+
//|    MQL4 global functions                                         |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CiTimeSeries *iTSeriesPointer()
  {
   static CiTimeSeries iBar;
//Print(iBar.ObjectCount());
   return &iBar;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool  iTimeSeriesInit(string symbol,ENUM_TIMEFRAMES &period[])
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Init(symbol,period);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool  iTimeSeriesInit(string symbol,ENUM_TIMEFRAMES period)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Init(symbol,period);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int iTimeSeriesObjectCount()
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.ObjectCount();
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int  iBars(string symbol,ENUM_TIMEFRAMES period)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.BarsTotal(symbol,period);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double iOpen(string symbol,ENUM_TIMEFRAMES period,const int index)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Open(symbol,period,index);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double iHigh(string symbol,ENUM_TIMEFRAMES period,const int index)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.High(symbol,period,index);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double iLow(string symbol,ENUM_TIMEFRAMES period,const int index)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Low(symbol,period,index);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double iClose(string symbol,ENUM_TIMEFRAMES period,const int index)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Close(symbol,period,index);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int   iHighest(string symbol,ENUM_TIMEFRAMES period,ENUM_SERIES_MODE mode,int count=WHOLE_ARRAY,int start=0)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Highest(symbol,period,mode,count,start);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int   iLowest(string symbol,ENUM_TIMEFRAMES period,ENUM_SERIES_MODE mode,int count=WHOLE_ARRAY,int start=0)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Lowest(symbol,period,mode,count,start);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
datetime iTime(string symbol,ENUM_TIMEFRAMES period,const int index)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Time(symbol,period,index);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
long  iVolume(string symbol,ENUM_TIMEFRAMES period,const int index)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Volume(symbol,period,index);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int   iBarShift(string symbol,ENUM_TIMEFRAMES period,datetime time,const bool exact=false)
  {
   CiTimeSeries *iBar=iTSeriesPointer();
   return iBar.Shift(symbol,period,time,exact);
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
CRatesArray::CRatesArray(const string            symbol,
                         const ENUM_TIMEFRAMES   period
                         ):
                         m_symbol(symbol),
                         m_period(period),
                         m_totalIndex(0)
  {
   ArraySetAsSeries(m_rates,true);
   Refresh();
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
datetime CRatesArray::Time(const int index)
  {
   if(index<0 || index>=m_ratesSize)
      return (datetime)0;
   return m_rates[index].time;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CRatesArray::Open(const int index)
  {
   if(index<0 || index>=m_ratesSize)
      return -1;
   return m_rates[index].open;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CRatesArray::High(const int index)
  {
   if(index<0 || index>=m_ratesSize)
      return -1;
   return m_rates[index].high;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CRatesArray::Low(const int index)
  {
   if(index<0 || index>=m_ratesSize)
      return -1;
   return m_rates[index].low;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
double CRatesArray::Close(const int index)
  {
   if(index<0 || index>=m_ratesSize)
      return -1;
   return m_rates[index].close;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
long CRatesArray::Volume(const int index)
  {
   if(index<0 || index>=m_ratesSize)
      return -1;
   return m_rates[index].tick_volume;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+

int CRatesArray::Shift(datetime time,bool exact)
  {
   int index=TrimTime(time);
   index-=m_trimExcess;
   if(index>=0 && index<m_totalIndex)
     {
      int res=m_indexes[index]-1;
      if(!exact && res<0)
        {
         bool goUp=true,goDn=true;
         int cnt=0;
         while(true)
           {
            if(!goDn && !goUp)
               break;
            cnt++;
            if(goDn)
              {
               index=TrimTime(time -(m_periodSecAdj*cnt));
               if(index<0)
                 {
                  goDn=false;
                 }
               else if(m_indexes[index-m_trimExcess]-1>=0)
                 {
                  res=m_indexes[index-m_trimExcess]-1;
                  break;
                 }
              }
            if(goUp)
              {
               index=TrimTime(time+(m_periodSecAdj*cnt));
               if(index>=m_totalIndex)
                 {
                  goUp=false;
                 }
               else if(m_indexes[index-m_trimExcess]-1>=0)
                 {
                  res=m_indexes[index-m_trimExcess]-1;
                  break;
                 }
              }
           }
        }
      return res;
     }
   else if(index>=m_totalIndex)
     {
      if(TimeCurrent()!=m_lastWarn)
        {
         Print(__FUNCTION__," Line:",__LINE__," iBarShift object should be refreshed");
         m_lastWarn=TimeCurrent();
        }
     }

   return -1;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CRatesArray::Refresh(void)
  {
   if(GetTickCount()-m_lastTick<2500)
      return true;
   else
      m_lastTick= GetTickCount();
   int totalBars= Bars(m_symbol,m_period);
   if(totalBars == m_lastBars)
      return true;
   else
      m_lastBars=totalBars;
//Clear arrays on subsequent refresh operations
/*
   if(ArraySize(m_rates)>0)
      ArrayFree(m_rates);
   if(m_totalIndex > 0)
   {
      ArrayFree(m_indexes);
      m_totalIndex = 0;
   }
   */
//Copy bar time to array  

   m_ratesSize=CopyRates(m_symbol,m_period,0,totalBars,m_rates);
   if(m_ratesSize!=totalBars)
     {
      Print(__FUNCTION__," Error in CopyRates: ",GetLastError());
      return false;
     }
//Beginning of time index

   int timeBegin=(int)m_rates[m_ratesSize-1].time;
//End of time index
   int timeEnd=(int)m_rates[0].time;
//Excess elements to trim
// + 1;

   m_periodSec=PeriodSeconds(m_period);
   m_periodSecAdj=m_periodSec<=PeriodSeconds(PERIOD_D1) ? m_periodSec : PeriodSeconds(PERIOD_D1);
//New index size = Total seconds + Period seconds - excess elements 
   int newSize    = (timeEnd + m_periodSec)/m_periodSecAdj;
   newSize       -= (m_trimExcess / m_periodSecAdj);
   newSize++;
   m_trimExcess   = timeBegin / m_periodSecAdj;
   m_totalIndex   = ArrayResize(m_indexes,newSize);

   if(m_totalIndex!=newSize)
     {
      Print(__FUNCTION__," Error Resizing Index array.");
      //return false;
     }

   int j=0;

   for(int i=0;i<m_ratesSize;i++)
     {
      j=0;
      while(j<m_periodSec)
        {
         int ind=TrimTime(m_rates[i].time+j)-m_trimExcess;
         m_indexes[ind]=i+1;
         j+=m_periodSecAdj;
        }
     }

   return true;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool CRatesArray::IsMatching(const string symbol,const ENUM_TIMEFRAMES period)
  {
   if(m_symbol==symbol && m_period==period)
      return true;
   else
      return false;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CRatesArray::Highest(ENUM_SERIES_MODE mode,int count=WHOLE_ARRAY,int start=0)
  {
   count = count == WHOLE_ARRAY ? BarsTotal() : count;
   count = start+count > BarsTotal() ? BarsTotal()-start : start+count;
   start = start < 0 ? 0 : start;

   double   highest     = DBL_MIN;
   datetime highesttime = (datetime)0;
   long     highestvol  = 0;
   int    index=-1;

   switch(mode)
     {
      case MODE_HIGH:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].high>highest)
              {
               highest = m_rates[i].high;
               index   = i;
              }
           }
         break;
      case MODE_OPEN:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].open>highest)
              {
               highest = m_rates[i].open;
               index   = i;
              }
           }
         break;
      case MODE_LOW:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].low>highest)
              {
               highest = m_rates[i].low;
               index   = i;
              }
           }
         break;
      case MODE_CLOSE:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].close>highest)
              {
               highest = m_rates[i].close;
               index   = i;
              }
           }
         break;
      case MODE_VOLUME :
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].tick_volume>highestvol)
              {
               highestvol=m_rates[i].tick_volume;
               index=i;
              }
           }
         break;
      case MODE_TIME:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].time>highesttime)
              {
               highesttime=m_rates[i].time;
               index=i;
              }
           }
         break;
      default:
         index=-1;
         break;
     }

   return index;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CRatesArray::Lowest(ENUM_SERIES_MODE mode,int count=WHOLE_ARRAY,int start=0)
  {
   count = count == WHOLE_ARRAY ? BarsTotal() : count;
   count = start+count > BarsTotal() ? BarsTotal()-start : start+count;
   start = start < 0 ? 0 : start;

   double   lowest     = DBL_MAX;
   datetime lowesttime = TimeCurrent();
   long     lowestvol  = LONG_MAX;
   int      index=-1;

   switch(mode)
     {
      case MODE_HIGH:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].high<lowest)
              {
               lowest=m_rates[i].high;
               index=i;
              }
           }
         break;
      case MODE_OPEN:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].open<lowest)
              {
               lowest=m_rates[i].open;
               index=i;
              }
           }
         break;
      case MODE_LOW:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].low<lowest)
              {
               lowest=m_rates[i].low;
               index=i;
              }
           }
         break;
      case MODE_CLOSE:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].close<lowest)
              {
               lowest=m_rates[i].close;
               index=i;
              }
           }
         break;
      case MODE_VOLUME :
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].tick_volume<lowestvol)
              {
               lowestvol=m_rates[i].tick_volume;
               index=i;
              }
           }
         break;
      case MODE_TIME:
         for(int i=start;i<count;i++)
           {
            if(m_rates[i].time<lowesttime)
              {
               lowesttime=m_rates[i].time;
               index=i;
              }
           }
         break;
      default:
         index=-1;
         break;
     }

   return index;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CRatesArray::TrimTime(datetime time)
  {
   int timeInt    = (int)time;
   int alt        = 0;
   int newTime    = 0;

   switch(m_period)
     {
      case PERIOD_M1 :
         newTime = TrimToMinutes(timeInt);
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M2 :
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 2);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M3 :
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 3);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M4 :
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 4);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M5 :
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 5);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M6 :
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 6);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M10 :
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 10);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M12 :
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 12);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M15:
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 15);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M20 :
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 20);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_M30:
         newTime=TrimToHours(timeInt);
         alt = (TrimToMinutes(timeInt)-newTime) / 60;
         alt = alt - (alt % 30);
         newTime+= 60*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_H1:
         newTime = TrimToHours(timeInt);
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_H2:
         newTime=TrimToDays(timeInt);
         alt = (TrimToHours(timeInt)-newTime) / 3600;
         alt = alt - (alt % 2);
         newTime+=3600*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_H3:
         newTime=TrimToDays(timeInt);
         alt = (TrimToHours(timeInt)-newTime) / 3600;
         alt = alt - (alt % 3);
         newTime+=3600*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_H4:
         newTime=TrimToDays(timeInt);
         alt = (TrimToHours(timeInt)-newTime) / 3600;
         alt = alt - (alt % 4);
         newTime+=3600*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_H6:
         newTime=TrimToDays(timeInt);
         alt = (TrimToHours(timeInt)-newTime) / 3600;
         alt = alt - (alt % 6);
         newTime+=3600*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_H8:
         newTime=TrimToDays(timeInt);
         alt = (TrimToHours(timeInt)-newTime) / 3600;
         alt = alt - (alt % 8);
         newTime+=3600*alt;
         newTime/= m_periodSecAdj;
         break;
      case PERIOD_H12:
         newTime=TrimToDays(timeInt);
         alt = (TrimToHours(timeInt)-newTime) / 3600;
         alt = alt - (alt % 12);
         newTime+=3600*alt;
         newTime/= m_periodSecAdj;
         break;
      default:
         newTime = TrimToDays(timeInt);
         newTime/= m_periodSecAdj;
         break;
     }
   return newTime;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CRatesArray::TrimToMinutes(int time)
  {
   return (time - (time % 60));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CRatesArray::TrimToHours(int time)
  {
   return (time - (time % 3600));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int CRatesArray::TrimToDays(int time)
  {
   return (time - (time % 86400));
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
